/* By Philipp Reisner 
 * 
 * Purpose: 
 * Provides http Protocoll.
 */

#include "../document.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>       
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <netdb.h>
#include <errno.h>
#include <signal.h>
#include <netinet/in_systm.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>


struct AddrRes
{
  struct Document* cont;
  struct Results* par;
  char *hostname;
  struct in_addr addr;
  struct AddrRes* next;
};

#define HOST_NAME_LEN 100

struct NameCacheEntry
{
  char name[HOST_NAME_LEN];
  struct in_addr addr;
  unsigned long lts;  /* logic clock */ 
  unsigned long uc;  /* ussage count */
};

#define NAME_CACHE_LEN 20

/* Yes, there are globals!! */
int http_fifo_req[2],http_fifo_res[2];
struct NameCacheEntry http_nameCache[NAME_CACHE_LEN];
unsigned long http_logicClock=0;
int http_nameCHits=0,http_nameCMisses=0,http_nameCFetches=0;
struct AddrRes* http_addrr;

/* Declarations */
void loadedUrl(struct Results* );
void gotAddress(struct Results* );
void calc_sel_n();


/* The following two functions maintainain an Cache for the hostname
   to ip_addreass translation */
void addToNameCache(char* hname, struct in_addr addr)
{
  int n,i,nv,vnv;

  if( strlen(hname) > HOST_NAME_LEN-1 ) return;

  http_logicClock++;

  http_nameCFetches++;

  dbg( DI, DBG_HTTP | 1, "adding %s",hname);
  
  for(i=0;i<NAME_CACHE_LEN;i++)
    {
      vnv = (http_nameCache[i].lts + http_nameCache[i].uc) / 2;
      if (vnv < nv || i == 0)
	{
	  nv = vnv;
	  n = i;
	}
      if(!strcmp(hname,http_nameCache[i].name)) 
	{
	  /* is already in the cache! */
	  http_nameCache[i].lts=http_logicClock;
	  http_nameCache[i].uc++;
	  return;
	}
    }
  http_nameCache[n].lts = http_logicClock;
  http_nameCache[n].uc = 1;
  strcpy(http_nameCache[n].name,hname);
  http_nameCache[n].addr = addr;
}

int checkNameCache(char* hname,struct in_addr* addr)
{
  int i;

  http_logicClock++;

  for(i=0;i<NAME_CACHE_LEN;i++)
    {
      if(!strcmp(hname,http_nameCache[i].name))
	{
	  /* This is a cache hit! */
	  http_nameCache[i].lts=http_logicClock;
	  http_nameCache[i].uc++;
	  *addr = http_nameCache[i].addr;
	  http_nameCHits++;
	  return 1;
	}
    }
  http_nameCMisses++;
  return 0; 
}

void errorUrl(int s,struct Results* par)
{
  par->status=s;
  if(par->text) free(par->text);
  par->text=NULL;
  par->size=0;
  loadedUrl(par);
}

char* fNameOfUrl(char* url)
{
  return strrchr(url,'/');
}

char* pfNameOfUrl(char* url)
{
  char* p;
  p=strchr(url,'/')+1;
  p=strchr(p,'/')+1;
  return strchr(p,'/');
}

/* The following Functions reflects the states of a connection */
void startConnection(struct Results* par)
{
  char* hostname;
  char* filename;
  int port = 80;
  char* p;
  int rr;

  hostname = malloc( strlen(par->url) +1 );
  
  for(rr=0;rr<4;rr++) par->url[rr] = tolower(par->url[rr]);
  
  if(strncmp(par->url,"http://",7)!=0)
    {
      filename=hostname;
      rr=0;
      if(strncmp(par->url,"file:/",6)==0) rr=5;
      if(strncmp(par->url,"file://",7)==0) rr=6;
      if(rr)
	{      /* Load from local file! */
	  strcpy(filename,&(par->url[rr]));
	  port=strlen(filename)-1;
	  if(filename[port]=='/') filename[port]=(char)0;
	  port = open(filename,O_RDONLY);
	  if(port == -1) 
	    {
	      free(hostname);
	      errorUrl(-17,par); 
	      return;
	    }
	  par->fd = port;
	  FD_SET(par->fd,&(context->http.rfdset));
	  if(par->fd > context->http.sel_n) context->http.sel_n = par->fd;
	  
	  par->size = lseek(par->fd,0,SEEK_END); /* Get size */
	  lseek(par->fd,0,SEEK_SET); /* return to beginning */
	  par->text = malloc((size_t)par->size+1);
	  par->left = par->size;
	  par->inpos = par->text;
	  par->status = 200;
	  par->istate = HI_WFDataS;
	  free(hostname);
	  return;
	}
      else 
	{
	  free(hostname);
	  errorUrl(-10,par);
	  return;
	}
    }
  
  if(context->pref.http_proxy[0]) strcpy(hostname,context->pref.http_proxy);
  else strcpy(hostname,&(par->url[7]));
  
  p = strchr(hostname,'/');
  if(p != NULL) *p = 0;
  
  p = strchr(hostname,':');
  if(p != NULL)
    {         /* Huh! There is a ':' in the URL! */
      *p=0;
      p++;
      port = strtol(p,(char** )NULL,10);
    }
  
  par->sa.sin_port = htons(port);
  par->sa.sin_family = AF_INET;

  if( (par->sa.sin_addr.s_addr=inet_addr(hostname)) != -1) 
    {
      /* We already know the address!!, so continue...*/
      gotAddress(par);
    }
  else
    {
      if( checkNameCache(hostname,(struct in_addr *)
			 &(par->sa.sin_addr.s_addr)) )
	{
	  gotAddress(par);
	}
      else
	{
	  /* So we have to perform a DNS - query!, because this usually
	     blocks the process, we deligate this task to a child process! */
	  rr = strlen(hostname)+1;

	  write(http_fifo_req[1],&rr,sizeof(int));
	  write(http_fifo_req[1],hostname,rr);
	  write(http_fifo_req[1],&par,sizeof(void *));
	  write(http_fifo_req[1],&context,sizeof(void *));

	  par->istate = HI_WFGetHostName;
	}
    }
  free(hostname);
}


void gotAddress(struct Results* par)
{
  struct protoent *tcpproto;
  int on=1;

  par->fd = socket(PF_INET, SOCK_STREAM, 0);
  
  if (par->fd < 0)
    errorUrl(-12,par);
  if ((tcpproto = getprotobyname("tcp")) != NULL)
    setsockopt(par->fd, tcpproto->p_proto, TCP_NODELAY, (char *)&on, 
	       sizeof(int));

  /* Set socket to non-blocking */
  if(fcntl(par->fd,F_SETFL,O_RDWR | O_NONBLOCK)<0)
    {
      dbg( DI, DBG_HTTP | 3, "fcntl() failed" );
      errorUrl(-99,par);
    }

  dbg( DI, DBG_HTTP | 3, "first call to connect() ");
  connect(par->fd, (struct sockaddr *)&(par->sa), sizeof(par->sa));
  if(errno != EINPROGRESS)
    {
      dbg( DI, DBG_HTTP | 3, "first connect() failed with %d",errno );
      errorUrl(-13,par);
      return;
    }

  FD_SET(par->fd,&(context->http.wfdset));
  if(par->fd > context->http.sel_n) context->http.sel_n = par->fd;
  par->istate=HI_WFConnect;
}


void gotConnect(struct Results* par)
{
  char* p;
  int l;
  char* buf;

  FD_CLR(par->fd,&(context->http.wfdset));
  if(par->fd == context->http.sel_n) calc_sel_n();

  if(connect(par->fd, (struct sockaddr *)&(par->sa), sizeof(par->sa)) < 0) 
    {
      if(errno != EISCONN)
	{
	  dbg( DI, DBG_HTTP | 3, "second connect() failed with %d",errno );
	  errorUrl(-13,par);
	  return;
	}
    }

  buf = malloc( strlen(par->url) +20 ); /* +20 = GET ... HTTP/1.0\n\n */

  FD_SET(par->fd,&(context->http.rfdset));
  if(par->fd > context->http.sel_n) context->http.sel_n = par->fd;
  
  strcpy(buf,"GET ");
  if(context->pref.http_proxy[0])
    {  /* This connection is to a dammned proxy */
      strcat(buf,par->url);
    }
  else
    {  /* This is a direct connection */
      p=pfNameOfUrl(par->url);
      if(p!=NULL) strcat(buf,p);
      else strcat(buf,"/");
    }
  strcat(buf," HTTP/1.1\n");

  l=strlen(buf);
  if(write(par->fd,buf,l)!=l) 
    dbg( DI, DBG_HTTP | 3, "write failed????" );

  strcpy(buf,"Host: ");
  strcat(buf,&(par->url[7]));
  p = strchr(buf,'/');
  if(p != NULL) *p = 0;
  strcat(buf,"\n\n");

  l=strlen(buf);
  if(write(par->fd,buf,l)!=l) 
    dbg( DI, DBG_HTTP | 3, "write failed????" );

  par->istate=HI_WFHeader;
  free(buf);
  return;
}



/* Usually we get the header in one peace, however sometime we only receive
   a fragment, so this state tries to get the missing information out of
   the _second_ fragment we get! */
void gotHeader2(struct Results* par)
{
  int rr;
  char wline[BSIZE+1];
  char* p;
  
  rr=read(par->fd,wline,BSIZE);
  
  if(rr<1) 
    {
      dbg( DI, DBG_HTTP | 3, "read failed????" );
      errorUrl(-99,par);
      return;
    }

  wline[rr]=0;  /* Terminating null char for C-functions */
  par->inpos=wline;

  par->inpos = strstr(wline,"Content-length: ");
  if(par->inpos==NULL) par->inpos = strstr(wline,"Content-Length: ");
  if(par->inpos) par->size = strtol(par->inpos + 16 ,(char** )NULL,10);
  else par->size=0;
  par->inpos = strstr(wline,"Content-Type: ");
  if(par->inpos)
    {
      p = strchr(par->inpos,'\n');
      if( p[-1] == '\r' ) p--;
      strncpy(par->mimetype,par->inpos,(size_t)(p - par->inpos));
    }
  par->inpos = strstr(wline,"\r\n\r\n") + 4;
  if(par->inpos==(char*)4) par->inpos = strstr(wline,"\n\n") + 2;  

  if(par->inpos==(char*)2) 
    {
      dbg( DI, DBG_HTTP | 1, "Malformed header!" );
      errorUrl(-99,par);
      return;
    }

  if(par->size)
    {
      par->text = (char* )malloc((size_t)(par->size+1));
      rr = rr - (par->inpos-wline);
      memcpy(par->text,par->inpos,rr);
      par->inpos = par->text + rr ;
      par->left = par->size - rr;
      par->istate=HI_WFDataS;

      if(par->left == 0)
	{
	  par->text[par->size]=0; /* Terminating null char */
	  loadedUrl(par); 
	}

      return;
    }

  /* At this point, we know that we have either a HTTP/0.9 answer or 
     a sizeless answer!    Thats is more or less the same for us.
     inpos ... should point to the begin of the data. 
     rr ...... holds the return value of the last read*/
  par->left = GSIZE;
  par->size = rr - (par->inpos - wline);
  par->text = malloc((size_t)GSIZE);
  if(par->size) memcpy(par->text,par->inpos, par->size);  
  par->left = par->left - par->size;
  par->inpos = par->text + par->size;
  par->status=200;
  par->istate=HI_WFDataSl;
}


/* Usually we get the whole header in one peace! ...*/
void gotHeader(struct Results* par)
{
  int rr;
  char wline[BSIZE+1];
  char* p;
  
  rr=read(par->fd,wline,BSIZE);
  
  if(rr<1) 
    {
      dbg( DI, DBG_HTTP | 3, "read failed????" );
      errorUrl(-99,par);
      return;
    }
  
  wline[rr]=0;  /* Terminating null char for C-functions */
  par->inpos=wline;
  if(strncmp(wline,"HTTP/",5)==0)
    { /* It is a HTTP/1.0, HTTP/1.1 or even higher server */
      par->inpos = &wline[9];
      par->status = strtol(par->inpos,(char** )NULL,10);
      par->inpos = strstr(wline,"Content-length: ");
      if(par->inpos==NULL) par->inpos = strstr(wline,"Content-Length: ");
      if(par->inpos) par->size = strtol(par->inpos + 16 ,(char** )NULL,10);
      else par->size=0;
      par->inpos = strstr(wline,"Content-Type: ");
      if(par->inpos)
	{
	  p = strchr(par->inpos,'\n');
	  if( p[-1] == '\r' ) p--;
	  strncpy(par->mimetype,par->inpos,(size_t)(p - par->inpos));
	}
      par->inpos = strstr(wline,"\r\n\r\n") + 4;
      if(par->inpos==(char*)4) par->inpos = strstr(wline,"\n\n") + 2;  

      /*
	if(par->status != 200) 
	{
	  errorUrl(par->status,par);
	  return;
	}
      */
      
      if(par->inpos==(char*)2) 
	{
	  par->istate = HI_WFHeader2;
	  dbg( DI, DBG_HTTP | 1, "Header is not complete!! ");
	  return;
	}

      if(par->size)
	{
	  par->text = (char* )malloc((size_t)(par->size+1));
	  rr = rr - (par->inpos-wline);
	  memcpy(par->text,par->inpos,rr);
	  par->inpos = par->text + rr ;
	  par->left = par->size - rr;
	  par->istate=HI_WFDataS;

	  if(par->left == 0)
	    {
	      par->text[par->size]=0; /* Terminating null char */
	      loadedUrl(par); 
	    }

	  return;
	}
    }
  /* At this point, we know that we have either a HTTP/0.9 answer or 
     a sizeless answer!    Thats is more or less the same for us.
     inpos ... should point to the begin of the data. 
     rr ...... holds the return value of the last read*/
  par->left = GSIZE;
  par->size = rr - (par->inpos - wline);
  par->text = malloc((size_t)GSIZE);
  if(par->size) memcpy(par->text,par->inpos, par->size);  
  par->left = par->left - par->size;
  par->inpos = par->text + par->size;
  par->status=200;
  par->istate=HI_WFDataSl;

}


void gotDataS(struct Results* par)
{
  int rr;

  rr = read(par->fd,par->inpos,par->left);

  if(rr<0)                /* case of error */
    {
      dbg( DI, DBG_HTTP | 3, "read failed????" );
      errorUrl(-16,par);
    }

  if(rr==0)               /* case of 'end of file' */
    {
      par->text[par->size]=0; /* Terminating null char */
      loadedUrl(par); 
    }
  
  if(rr>0)                /* normal case */
    {
      par->inpos = par->inpos + rr;
      par->left = par->left - rr;

      if(par->left == 0)
	{
	  par->text[par->size]=0; /* Terminating null char */
	  loadedUrl(par); 
	}

    }
}


void gotDataSl(struct Results* par)
{
  int rr;
  int osize;

  if(par->left == 0) 
    {                     /* More memory needed! */
      osize = par->inpos - par->text;
      par->left = osize;
      par->text=realloc(par->text,(size_t)osize*2); 
      if(par->text == NULL) 
	{
	  errorUrl(-14,par);
	  return;
	}
      par->inpos=par->text+par->left;
    }

  rr = read(par->fd,par->inpos,par->left);

  if(rr<0)                /* case of error */
    {
      dbg( DI, DBG_HTTP | 3, "read failed????" );
      errorUrl(-16,par);
    }

  if(rr==0)               /* case of 'end of file' */
    {
      par->text[par->size]=0; /* Terminating null char */
      loadedUrl(par); 
    }
  
  if(rr>0)                /* normal case */
    {
      par->inpos = par->inpos + rr;
      par->left = par->left - rr;
      par->size = par->size + rr;
    }
}


/* This function is forked off as second process. It communicates via
   two pipes with the main process. It's task is to resolve host names.
   Since the call to gethostbyname() blocks the calling process, it is
   not acceptable to make this system call from the main process, which
   is in charge of the user interface.! */
void getHostClient(void )
{
  int rr,len;
  char* hostname;
  struct hostent *hp;
  struct in_addr addr;
  void* uaddr1;
  void* uaddr2;


  dbg( DI, DBG_HTTP | 1, "** getHostClient created **");


  close(http_fifo_res[0]);
  close(http_fifo_req[1]);

  while(1)
    {
      rr = read(http_fifo_req[0],&len,sizeof(int));
      hostname = NULL;
      if(rr == sizeof(int))
	 {
	   hostname = malloc(len); /* len includes terminating 0 byte */
	   rr = read(http_fifo_req[0],hostname,len);
	   rr = read(http_fifo_req[0],&uaddr1,sizeof(void* ));
	   rr = read(http_fifo_req[0],&uaddr2,sizeof(void* ));
	 }

      if(rr==0) 
	{
	  close(http_fifo_res[1]);
	  close(http_fifo_req[0]);
	  dbg( DI, DBG_HTTP | 1, "** getHostClient dieing **");
	  exit(0);
	}

      if(rr>0)
	{
	  dbg( DI, DBG_HTTP | 3, "got Request for %s",hostname);
	  hp = gethostbyname(hostname);
	  if(hp) 
	      addr = *(struct in_addr *)(hp->h_addr_list[0]);
	  else
	      strcpy(hostname,"(error)");

	  len = strlen(hostname) + 1;

	  dbg( DI, DBG_HTTP | 3, "sending Addr for %s",hostname);
	  write(http_fifo_res[1],&len,sizeof(int));
	  write(http_fifo_res[1],hostname,len);
	  write(http_fifo_res[1],&uaddr1,sizeof(void* ));
	  write(http_fifo_res[1],&uaddr2,sizeof(void* ));
	  write(http_fifo_res[1],&addr,sizeof(struct in_addr));
	}
      if(hostname) free(hostname);
    }
}


int http_glob_init()
{  
  int i;

  http_addrr = NULL;

  for(i=0;i<NAME_CACHE_LEN;i++)
    {
      http_nameCache[i].lts=0;
      http_nameCache[i].uc=0;
    }
  
  
  if(pipe(http_fifo_req)) 
    {
      dbg( DI, DBG_HTTP | 1, "pipe() failed! ");
      return -1;
    }
  if(pipe(http_fifo_res)) 
    {
      dbg( DI, DBG_HTTP | 1, "pipe() failed! ");
      return -1;
    }
        
  /* Start it up! */
  switch(fork())
    {
    case 0:  getHostClient(); 
    case -1: return -1;
    }

  close(http_fifo_req[0]);
  close(http_fifo_res[1]);	

  return 0;
}  

void http_glob_finalize()
{

  close(http_fifo_req[1]);
  close(http_fifo_res[0]);	

  dbg( DI, DBG_HTTP | 1, "name Cache Statistics: ");
  dbg( DI, DBG_HTTP | 1, "misses = %d; hits = %d; fetches = %d",
       http_nameCMisses,http_nameCHits,http_nameCFetches);
}


int http_initialize()
{
  FD_ZERO(&context->http.rfdset);
  FD_ZERO(&context->http.wfdset);
  
  FD_SET(http_fifo_res[0],&context->http.rfdset);
  context->http.sel_n = http_fifo_res[0];

  context->http.threads=0;
  context->http.delayed=0;
  context->http.firstPost=NULL;
  context->http.lastPost=NULL;
  context->http.firstRun=NULL;
  context->http.firstReady=NULL;
  context->http.lastReady=NULL;

  return 0;
}


int http_finalize()
{
  http_abortAll();
 
  return 1;
}


struct Results* orderWithPriority(char* url,int pri)
{
  struct Results* par;

  http_schedule(0);

  par = malloc( sizeof(struct Results) + strlen(url) + 1 );
  par->url = (char *)par + sizeof(struct Results);
  strcpy(par->url,url);
  par->size=0;
  par->fd=0;
  par->text=NULL;

  dbg( DI, DBG_HTTP | 2, "%s @%p",par->url,par );

  if(context->http.threads < context->pref.http_threadsmax)
    {
      par->next=context->http.firstRun;
      context->http.firstRun=par;
      context->http.threads++;
      startConnection(par);
      return(par);
    }
  else
    {/* This request is postponed! */
      if(pri)
	{  /* High priority!, Goes to the beginning of the waiting queue */
	  par->next = context->http.firstPost;
	  context->http.firstPost = par;	  

	  if(context->http.lastPost == NULL)
	    context->http.lastPost = par;
	}
      else
	{  /* Low priority!, Goes to the end of the waiting queue */
	  par->next = NULL;
	  
	  if(context->http.lastPost)
	    context->http.lastPost->next = par;
	  else /* Leere Liste */
	    context->http.firstPost = par;

	  context->http.lastPost = par;
	}

      par->istate = HI_postponed;
      context->http.delayed++;
      return par;
    }
}

struct Results* http_orderQuick(char* url)
{
return orderWithPriority(url,1);
}

struct Results* http_order(char* url)
{
return orderWithPriority(url,0);
}


void startDelayed()
{
  struct Results* par;
  if(context->http.threads < context->pref.http_threadsmax)
    {
      if((par=context->http.firstPost)!=NULL)
	{
	  context->http.firstPost=par->next;
	  if(context->http.firstPost==NULL) context->http.lastPost=NULL;
	  context->http.delayed--;
	}
      if(par)
	{
	  par->next=context->http.firstRun;
	  context->http.firstRun=par;
	  context->http.threads++;
	  startConnection(par);
	}
    }
}


void calc_sel_n()
{
  struct Results* p;

  p=context->http.firstRun;

  context->http.sel_n = http_fifo_res[0];
  while(p)
    {
      if((FD_ISSET(p->fd,&(context->http.rfdset))  ||
	 FD_ISSET(p->fd,&(context->http.wfdset))) &&	 
	 (p->fd > context->http.sel_n))     context->http.sel_n = p->fd;
      p=p->next;
    }
}



void loadedUrl(struct Results* par)
{
  struct Results* p,** pl;

  dbg( DI, DBG_HTTP | 2, "%s st:%d si:%d",par->url,par->status,par->size );

  par->istate=HI_ready;

  if(par->fd) 
    {
      FD_CLR(par->fd,&(context->http.rfdset));
      if(par->fd == context->http.sel_n) calc_sel_n();
      close(par->fd);
    }

  p=context->http.firstRun;       /* take par out of the list of runnings */
  pl=&(context->http.firstRun);

  while(p != par)
    {
      pl=&(p->next);
      p=p->next;
    }
  *pl=p->next; /* p==par */

  par->next=NULL;              /* insert par into the list of readys */
  if(context->http.lastReady==NULL)
    {
      context->http.firstReady=par;
      context->http.lastReady=par;
    }
  else
    {
      context->http.lastReady->next=par;
      context->http.lastReady=par;
    }
  context->http.threads--;

  startDelayed();
}

/* 
Here we check for activity on all the connections...

   schedule_in_addr()
       Multiplexed the incomming ined_addresses from the DNS-lookup
       client to the according connections.

   gotResponseFromClient()
       Is invokod on anything from the DNS-lookup client.

   http_schedule()
       Here is the call to select(). And anny state-transfers of the 
       connections are triggered from here.
*/
void schedule_in_addr(struct AddrRes* a)
{
  struct Results* p;

  p=context->http.firstRun;
  while(p)
    {
      if( p == a->par ) 
	{
	  /* The Request, which is waiting for address this still 
	     exists, so continue processing it*/
	  if(!strcmp(a->hostname,"(error)")) 
	    errorUrl(-11,p); /*  indicates a resolv error */
	  else
	    {
	      p->sa.sin_addr.s_addr = a->addr.s_addr;
	      gotAddress(p);
	      break;
	    }
	}
      p=p->next;
    }
}


void gotResponseFromClient()
{
  struct AddrRes* addrr;
  int len;

  read(http_fifo_res[0],&len,sizeof(int));

  addrr = malloc( sizeof(struct AddrRes) + len );
  addrr->hostname = (char *)addrr + sizeof(struct AddrRes);

  read(http_fifo_res[0],addrr->hostname,len);
  read(http_fifo_res[0],&(addrr->par),sizeof(void *));
  read(http_fifo_res[0],&(addrr->cont),sizeof(void *));
  read(http_fifo_res[0],&(addrr->addr),sizeof(struct in_addr));
  
  dbg( DI, DBG_HTTP | 3, "got Address for %s",addrr->hostname);
  
  if(strcmp(addrr->hostname,"(error)")) 
    addToNameCache(addrr->hostname,addrr->addr);  
  
  if(addrr->cont == context) 
    {
      schedule_in_addr(addrr);
      free(addrr);
    }
  else
    {
      dbg( DI, DBG_HTTP | 2, "not in this context! %s",addrr->hostname);
      addrr->next = http_addrr;
      http_addrr = addrr;
    }
}


void http_schedule(int blocking)
{

  fd_set rsfd;
  fd_set wsfd;
  struct timeval tv;
  struct Results* p,* np;
  int rr;
  struct AddrRes* a, **la;

  /* 1. check the waiting results */

  a = http_addrr;
  la = &http_addrr;
  while(a)
    {
      if(a->cont == context)
	{
	  dbg( DI, DBG_HTTP | 2, "found the right context! %s",a->hostname);
	  schedule_in_addr(a);
	  *la = a->next;
	  free(a);
	}
      la=&(a->next);
      a=a->next;
    }

  /* 2. Check the open connections for activity */

  tv.tv_sec=0;
  tv.tv_usec=0;

  rsfd = context->http.rfdset;
  wsfd = context->http.wfdset;

  if(blocking) rr = select(context->http.sel_n+1,&rsfd,&wsfd,NULL,NULL);
  else         rr = select(context->http.sel_n+1,&rsfd,&wsfd,NULL,&tv);
  
  if( FD_ISSET(http_fifo_res[0],&rsfd) )
    gotResponseFromClient();

  if(rr>0)
    {
      p=context->http.firstRun;
      while(p)
	{
	  np=p->next;/*Because the got** functions can change p->next!!*/
	  if(FD_ISSET(p->fd,&rsfd))
	    switch(p->istate)
	      {
	      case HI_WFHeader:  gotHeader(p);  break;
	      case HI_WFHeader2: gotHeader2(p); break;
	      case HI_WFDataS:   gotDataS(p);   break;
	      case HI_WFDataSl:  gotDataSl(p);  break; 
	      default: dbg( DI, DBG_HTTP | 1, 
		    "Thread with unknown istate is in the running queue!");
	      }
	  if(FD_ISSET(p->fd,&wsfd))
	    switch(p->istate)
	      {
	      case HI_WFConnect: gotConnect(p); break;
	      default: dbg( DI, DBG_HTTP | 1, 
		    "Thread with unknown istate is in the running queue!");
	      }	      
	  p=np;       
	}
    }
}


struct Results* http_waitNext()
{
  struct Results* retval;

  retval = http_getNext();
  
  while(retval == NULL && context->http.threads > 0)
    {               
      dbg( DI, DBG_HTTP | 2, "calling http_schedule(blocking)");
      http_schedule(1);  /* Here comes the blocking call!! */
      retval = http_getNext();
    }
  return retval;
}


struct Results* http_getNext()
{
  struct Results* retval;

  http_schedule(0);

  if((retval=context->http.firstReady)!=NULL)
    {
      context->http.firstReady=retval->next;
      if(retval==context->http.lastReady)
	  context->http.lastReady=NULL;
    }

  if(retval)
    dbg( DI, DBG_HTTP | 2, "%s st:%d si:%d @%p",retval->url,retval->status,
	 retval->size,retval );
  return retval;
}


void http_free(struct Results* par)
{
  if(par->size != 0) free(par->text);
  free(par);
  return;
}

void http_abort(struct Results* par)
{
  struct Results* p,**lp;


  if(par->istate == HI_postponed)
    {
      p=context->http.firstPost;
      lp=&(context->http.firstPost);
      while(p!=par)
	{
	  lp=&(p->next);
	  p=p->next;
	}
      *lp=p->next;
      if(p->next==NULL) context->http.lastPost=NULL;
      context->http.delayed--;
      return;
    }
  
  if(par->istate == HI_ready)
    {
      p=context->http.firstReady;
      lp=&(context->http.firstReady);
      while(p!=par)
	{
	  lp=&(p->next);
	  p=p->next;
	}
      *lp=p->next;
      if(p->next==NULL) context->http.lastReady=NULL;
      http_free(par);
      return;
    }

  /* The Thread is in some sort of running state */
  
  p=context->http.firstRun;
  lp=&(context->http.firstRun);
  while(p!=par)
    {
      lp=&(p->next);
      p=p->next;
    }
  *lp=p->next;

  if(par->fd)
    {
     FD_CLR(par->fd,&(context->http.rfdset));
     if(par->fd == context->http.sel_n) calc_sel_n();
     close(par->fd);
    }
  
  http_free(par);  
  context->http.threads--;
  startDelayed();
}


int http_numberOfThreads()
{
  return context->http.threads;
}

int http_numberOfPostponed()
{
  return context->http.delayed;
}

void http_abortAll()
{
  struct Results* par;

  /* Free all postponed requests! */

  while(context->http.firstPost)
    {
      par = context->http.firstPost;
      context->http.firstPost = par->next;
      if(context->http.firstPost==NULL) context->http.lastPost = NULL;
      context->http.delayed--;
    
      http_free(par);
    }

  while(context->http.firstRun)
    http_abort(context->http.firstRun);

  /* Free all ready requests! */

  while(context->http.firstReady)
    {
      par = context->http.firstReady;
      context->http.firstReady = par->next;
      if(context->http.firstReady==NULL) context->http.lastReady = NULL;
    
      http_free(par);
    }
}



int http_finished()
{
return( context->http.threads==0 &&
	context->http.delayed==0 &&
	context->http.firstReady==NULL );
}
