/* html.c written-from-scratch (tm) by philipp richter */
#include <pwd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "../document.h"
#include "html_priv.h"

PL *createPicEntry( const char *url )
{
  PL *p;

  if( url == NULL )
    return NULL;

  p = malloc( sizeof( PL ) + strlen( url ) + 1);

  if( p == NULL )
    return NULL;

  p->url = (char *)p + sizeof( PL );
  strcpy( p->url, url);
  p->next = CH.piclist;
  CH.piclist = p;

  return p;
}

PL *picInMem( const char *url )
{
  PL *p;
  for( p = CH.piclist; p != NULL; p = p->next )
    if( strcasecmp( url, p->url ) == 0)
    {
      dbg( DI, DBG_HTML|3, "found picture %s in mem", url);
      return p;
    }

  dbg( DI, DBG_HTML|3, "didn't find picture %s in mem", url);
  return NULL;
}

PL *addPicEntry( const char *url )
{
  PL *p;

  if( picInMem( url ) )
    return NULL;

  if( ( p = createPicEntry( url ) ) == NULL )
    return NULL;

  if( ( p->address = http_order( (char *)makeOrderName( url ) ) ) == NULL )
    return FALSE;

  return p;
}

/*
  allocate memory for a new node and insert it into tree structure
  returns NULL if memory is exhausted
*/

TI *createNode( TI *father, const char *url )
{
  TI *n, *ll;
  NTI *cp, **npch, *fp;

  if( url == NULL )
    return NULL;

  /* allocate memory for node & url */
  n = malloc( sizeof( TI ) + strlen( url ) + 1);

  /* no memory ? that's no good */
  if( n == NULL )
    return NULL;

  /* initialize treeinfo */
  memset( n, 0, sizeof( TI ) );
  n->active = 1;
  n->doctype = Unknown;

  /* copy url to end of treeinfo and set urlpointer */
  n->url = (char *)n + sizeof( TI );
  strcpy( n->url, url );

  *(n->title) = '\0';

  layout_initTreeInfo( n );

  /* if there is no father then we are done */
  if( father == NULL )
  {
    dbg( DI, DBG_HTML | 3, "create root node with url %s succeded", url );
    return n;
  }

  n->active = father->active;
  n->depth = father->depth + 1;
  n->father = father;

  /* append at end of linear list*/
  for( ll = father; ll->nextList ; ll = ll->nextList ) ;
  ll->nextList = n;

  if( ( cp = malloc( sizeof( NTI ) ) ) == NULL)
  {
    free( n );
    return NULL;
  }
  if( ( fp = malloc( sizeof( NTI ) ) ) == NULL)
  {
    free( cp );
    free( n );
    return NULL;
  }

  /* add node to fathers childlist */
  /* "install" nexttreeinfo in childlist of father */
  /* this one goes to the beginning of the list (shorter) */
  /*  cp->tree = n;
  cp->next = father->nextTree;
  layout_initNextTreeInfo( cp );
  father->nextTree = cp;*/

  cp->tree = n;
  cp->next = NULL;
  layout_initNextTreeInfo( cp );
  for( npch = &(father->nextTree) ; *npch ; npch = &((*npch)->next) ) ;
  *npch = cp;
  
  /* and this one goes to the end to satisfy bernhard ;-) */
  /* ok, actually the end is the beginning ... */
  fp->tree = father;
  fp->next = NULL;
  layout_initNextTreeInfo( fp );
  n->nextFather = fp;
  /*  dbg( DI, DBG_HTML | 1, "Father '%s' added to '%s'", father->url, url ); */

  layout_nodeAdded( n );

  dbg( DI, DBG_HTML | 3, "create node with url %s succeded", url );
  return n;
}

/*
  adds node to tree structure if necessary and orderes http data
  if node already in memory then repair links to/from father-node...

  returns FALSE if node was not added
*/

int addNode( TI *father, const char *url )
{
  TI *n;
  NTI *nc, *nf, **npfa, **npch;
  int inlist;

  /* check if we will load this url*/
  if( !inexPassed( url ) )
    return FALSE;

  /* check if searchdepth is ok */
  if( ( father ) && ( ( father->depth - 1 ) >= CP.searchdepth ) )
    return FALSE;

  /* if we already have the node then make links to/from father instead of
     creating a new one */  
  if( ( n = nodeInMem( url ) ) && ( father ) )
  {
    /* check if we are already in childlist */
    inlist = FALSE;
    for( nc = father->nextTree; nc; nc = nc->next )
      if( nc->tree == n )
      {
	inlist = TRUE;
	break;
      }
    nc = NULL;
    if( !inlist )
    {
      /* insert node as new child of father */
/*      if( ( nc = malloc( sizeof( NTI ) ) ) == NULL )
	return FALSE;
      layout_initNextTreeInfo( nc );
      nc->tree = n;
      nc->next = father->nextTree;
      father->nextTree = nc; */

      /* append at end of childlist (needed?) not insert at beginning ? */
      if( ( nc = malloc( sizeof( NTI ) ) ) == NULL )
	return FALSE;
      nc->tree = n;
      nc->next = NULL;
      layout_initNextTreeInfo( nc );
      for( npch = &(father->nextTree) ; *npch ; npch = &((*npch)->next) ) ;
      *npch = nc;
    }

    /* check if we have the father already in our fatherlist */
    inlist = FALSE;
    for( nf = n->nextFather; nf; nf = nf->next )
      if( nf->tree == father )
      {
	inlist = TRUE;
	break;
      }

    if( !inlist )
    {
      /* _append_ (for bernhard) father in fatherlist of node */
      if( ( nf = malloc( sizeof( NTI ) ) ) == NULL )
      {
	free( nc );
	return FALSE;
      }
      nf->tree = father;
      nf->next = NULL;
      layout_initNextTreeInfo( nf );
      /* dbg( DI, DBG_HTML | 1, "Father '%s' added to '%s'", father->url, n->url ); */
      /* go to end of father list */
      for( npfa = &(n->nextFather) ; *npfa ; npfa = &((*npfa)->next) ) ;
      *npfa = nf;
    }

    return TRUE;
  }
  else
  {
    if( ( n = createNode( father, url ) ) == NULL )
      return FALSE;

    /* if we are the first node then install as root node */
    if( father == NULL )
      CH.root = n;

    /* order the document */
    if( ( n->address = http_order( (char *)makeOrderName( url ) ) ) == NULL )
      return FALSE;

    dbg( DI, DBG_HTML | 3, "node added: %s", url );

    return TRUE;
  }

  /* never reached */
  return FALSE;
}

/*
  deletes node from tree structure in memory 
  if loopNode is not NULL, the second of two identical nodes (node)  
  is deleted from WebTree datastructure in memory and the 
  pointer from its father in tree structure points to the first 
  identical nodes (and children too!) 
  Rewritten from scratch by Philipp Reisner
  mode = 0 (node is NEW)
  mode = 1 (loopnode is NEW)
*/

void deleteNode( TI *node, TI *loopNode, int mode )
{
  TI **lp,*p;
  NTI **ln,*n,*c,*d, *e, *n2,**ln2;
  int is_in;

  if( node == NULL ) return;

  if (mode==1) /* loopnode is new */
    {
      /* Remove 'loopNode' from linear list */
      lp = &(context->html.root);
      p = *lp;
      
      while(p)
	{
	  if( p == loopNode )
	    {
	      *lp = p->nextList;
	      break;
	    }
	  lp = &(p->nextList);
	  p = *lp;
	}
      
      /* replace node with loopNode */
      lp = &(context->html.root);
      p = *lp;
      
      while(p)
	{
	  if( p == node )
	    {
	      loopNode->nextList=p->nextList;
	      *lp = loopNode;
	      break;
	    }
	  lp = &(p->nextList);
	  p = *lp;
	}
    }
  else /* node is new */
    {
      /* Remove 'node' from linear list */
      lp = &(context->html.root);
      p = *lp;
      
      while(p)
	{
	  if( p == node )
	    {
	      *lp = p->nextList;
	      break;
	    }
	  lp = &(p->nextList);
	  p = *lp;
	}
    }


  /* In every parent of 'node', 'node' must be replaced by 'loopnode', or if
     reference to 'loopNode' is already there, it must be removed */
  
  for( c=node->nextFather ; c ; c=c->next )
  {
    p = c->tree;
    is_in=0;
    n = p->nextTree;

    while(n)
    {
      ln2 = &(n->next);
      n2 = *ln2;

      if(n->tree==node || n->tree==loopNode )
      {
	if(is_in) 
	{
	  if(ln) *ln = n->next;
	  ln2=ln;
	  layout_freeNextTreeInfo(n);
	  free(n);
	}
	else
	{
	  n->tree = loopNode;
	  is_in=1;
	}
      }
      
      ln = ln2;
      n = n2;
    }
  }

  /* Merge entries of "node's" parentlist into "loopNode's" parentlist,
     and free the not longer used parentlist entries of "node" */
  e=NULL;
  if (mode==1)
    {
      e=loopNode->nextFather;
      loopNode->nextFather=NULL;
    }

  c=node->nextFather;
  while(c)
  {
    n2=c->next;
    p = c->tree;
    is_in=0;
    for( n=loopNode->nextFather ; n ; n=n->next)
    {
      if(n->tree == p)
      {
	is_in=1;
	break;
      }
    }

    if(is_in==0)
    {
      c->next=NULL;
      if (loopNode->nextFather)
      {
	for (d=loopNode->nextFather; d->next; d=d->next);
	d->next=c;
      }
      else
      {
	loopNode->nextFather = c;
      }
    }
    else
    {
      /* layout_freeNextTreeInfo(c); - don't call it for entries in fatherlist */
      free(c);
    }
    c = n2;
  }

  if (e)
    {
      ln=&(loopNode->nextFather);
      while (*ln) ln=&((*ln)->next);
      while (e)
	{
	  c=e->next;
	  for (d=loopNode->nextFather; d; d=d->next)
	    if (d->tree==e->tree)
	      break;
	  if (!d)
	    {
	      *ln=e;
	      e->next=NULL;
	      ln=&(e->next);
	    }
	  e=c;
	}
    }


  /*
    
   */
  for( c=node->nextTree ; c ; c=c->next )
  {
    p = c->tree;
    is_in=0;
    n = p->nextFather;

    while(n)
    {
      ln2 = &(n->next);
      n2 = *ln2;

      if(n->tree==node || n->tree==loopNode )
      {
	if(is_in) 
	{
	  if(ln) *ln = n->next;
	  ln2=ln;
	  /*	  layout_freeNextTreeInfo(n);*/
	  free(n);
	}
	else
	{
	  n->tree = loopNode;
	  is_in=1;
	}
      }
      
      ln = ln2;
      n = n2;
    }
  }

  /*  for( c = node->nextTree; c; c = c->next)
  {
    is_in = 0;
    for( d = c->tree->nextFather; d ; d = d->next)
    {
      if( d->tree == node )
      {
	if( is_in )
	d->tree = loopNode;
    }
  }*/

  /* copy child-list from node to loopnode */
  /* DONT DO THIS THAT WAY: loopNode->nextTree = node->nextTree; */

  c=node->nextTree;
  while(c)
  {
    n2=c->next;
    p = c->tree;
    is_in=0;
    for( n=loopNode->nextTree ; n ; n=n->next)
    {
      if(n->tree == p)
      {
	is_in=1;
	break;
      }
    }

    if(is_in==0)
    {
      c->next=NULL;
      if (loopNode->nextTree)
      {
	for (d=loopNode->nextTree; d->next; d=d->next);
	d->next=c;
      }
      else
      {
	loopNode->nextTree = c;
      }
    }
    else
    {
      layout_freeNextTreeInfo(c);
      free(c);
    }
    c = n2;
  }


  loopNode->depth=node->depth;
  node->nextTree=NULL;


  /* resolv all recursive links in loopnode */
  ln=&(loopNode->nextTree);
  while (*ln)
    {
      if ((*ln)->tree==loopNode)
	{
	  c=*ln;
	  *ln=c->next;
	  layout_freeNextTreeInfo(c);
	  free(c);
	}
      else
	ln=&((*ln)->next);
    }

  ln=&(loopNode->nextFather);
  while (*ln)
    {
      if ((*ln)->tree==loopNode)
	{
	  c=*ln;
	  *ln=c->next;
	  free(c);
	}
      else
	ln=&((*ln)->next);
    }


  /* free actual node */
  /*  layout_freeTreeInfo( node );*/
  free( node );
  layout_recalcAll(); /* BKJ */
}

/*
  initializes HTML variables and datastructures
*/

void html_initialize()
{
  CH.root = NULL;
  CH.piclist = NULL;
  CH.extreq = NULL;
  CH.currentMirrorSize = 0;
  CH.mirrorpath[0] = '\0';
  /*  CH.urlbuffer[0] = '\0';*/

  dbg( DI, DBG_HTML | 2, "Initialized all data structures" );
}

/*
 free all memory occupied by html module
*/

void html_free()
{
  /* frees memory allocated by HTML */
  TI  *t, *u;
  NTI *m, *n;
  PL  *p, *q;

  html_stopLoading();

  for( t = CH.root; t ; )
  {
    u = t;
    t = t->nextList;

    /* free childlist */
    for( m = u->nextTree; m ; )
    {
      n = m; m = m->next;
      layout_freeNextTreeInfo( n );
      free( n );
    }

    /* free fatherlist */
    for( m = u->nextFather; m ; )
    {
      n = m; m = m->next;
      free( n );
    }

    /* and finaly free node itself */
    layout_freeTreeInfo( u );
    free( u );
  }
  CH.root = NULL;
  
  /* frees picture list pointed to by piclist */

  for( p = context->html.piclist; p ; )
  {
    q = p; p = p->next;
    free( q );
  }
  CH.piclist = NULL;
  CH.currentMirrorSize = 0;
  dbg(DI, DBG_HTML | 2, "Freed all data structures");
}

int html_buildWebTree()
{
  TI *n, *n2;
  R *r;
  ER **pe, *p;
  PL *pl;
  char *e, *f;
  FILE *fd;
  int dc;

  /* check if we are finished */
  if( http_finished() && CH.extreq == NULL )
  {
    view_setStatusLine( "Done" );
    view_htmlFinished();
    return FALSE;
  }

  if( (r = http_getNext()) == NULL  )
    return FALSE;

  /*
    look if this was an ext req, if yes call the callback and remove the entry 
    from the list
  */
  for( pe = &(CH.extreq) ; *pe ; pe = &((*pe)->next) )
    if( (*pe)->res == r )
    {
      p = *pe;
      p->callback( r->status, r->size, r->text, p->users_data );
      *pe = p->next;
      free( p );
      http_free( r );
      return TRUE;	/* for now we are finished */
    }

  /* ok, we are not in the ext req list, so search the linear list */

  for( n = CH.root; n ; n = n->nextList )
    if( n->address == r )
      break;

  if( n ) /* found, ok. now process the data */
  {
    /*
      http error
    */

    if( r->status < 200 || r->size == 0 )
    {
      e = htmlErrorString( r->status);
      view_setStatusLine( e );
      dbg( DI, DBG_HTML | 1, "%s: %s", n->url, e );
      n->doctype = Error;
      n->address = NULL;
      strncpy( n->title, e, NAME_MAXLEN - 1 );
      n->title[ NAME_MAXLEN ] = '\0';

      http_free( r );
      return TRUE;
    }
    else
      n->doctype = doctype( n->url, r->text );

    /*
      all went ok, we should have some data to parse...
    */
    
    n->loaded = TRUE;
    n->length = r->size;
    n->address = NULL;
    n->checksum = checksum( r->text, r->size );

    /*
      now search if the same page already exists in the tree structure and if
      yes, delete the first one. i.e. http://blabla , later comes a reference to
      http://blabla/index.html which is the same. so delete http://blabla and repair
      all connections from it
    */

    for( n2 = CH.root ; n2 ; n2 = n2->nextList )
      if( ( n != n2 ) && ( n->checksum == n2->checksum ) )
	break;

    /*
      TODO berlegen, ab es noch andere dinge in der struct gibt, auf die
      man aufpassen mu (wie z.b. title) je nachdem ob man den neuen oder
      den alten node lscht
    */

    /*
      n2 is now the node to be deleted
    */

    if( n2 && urlHasFilename( n2->url ) )
    {
      /*
	the original has a filename. so keep it and delete the new one
      */

      dbg( DI, DBG_HTML | 3, "DELETE NEW node %s, with old %s", n->url, n2->url );
      deleteNode( n, n2, 0 );
    }
    else if( n2 )
    {
      /*
	delete the original one
      */

      dbg( DI, DBG_HTML | 3, "DELETE old node %s, with %s", n2->url, n->url );
      strcpy( n->title, n2->title );
      deleteNode( n2, n, 1 );
    }
    else
    {
      /*
	action: parse htmlfile and place new orders according to the referrences 
	found in the doc.
      */

      dbg( DI, DBG_HTML | 3, "now parsing %s ...", n->url );

      /*
	if we mirror htmlfiles then generate path/file and redirect output of 
	scanner to the file
      */
      
      fd = NULL;
      if( CP.mirrorhttp )
      {
	/* TODO fopen vor genDirectories probieren -> weniger overhead */
	if( ( f = genFilename( n->url ) ) &&
	    ( dc = genDirectories( f ) ) &&
	    ( fd = fopen( f, "w" ) ) )
	  ;
	else
	{
	  if( f == NULL )
	    dbg( DI, DBG_HTML|1, "could not make filename of %s !?", n->url );
	  else if( (dc = FALSE) )
	    dbg( DI, DBG_HTML|1, "could not make directories for %s", f );
	  else
	    dbg( DI, DBG_HTML|1, "could not open file %s, errno = %d", f, errno);
	}
      }

      if( n->doctype == HTMLDoc )
	parseHtml( PARSER_MAKERELATIVE, fd, n, r->text, r->size);

      if( CP.mirrorhttp && fd )
      {
	fclose( fd );
	/*yyout = NULL;*/
      }

      layout_nodeAdded( n );
      view_nodeAdded( n );
    }
  }
  else
  {
    /* not found? ok but in the picture list we have to find it */
    for( pl = CH.piclist; pl ; pl = pl->next )
      if( ( pl->address == r ) )
	break;

    if( pl == NULL) /* that's bad. there must be something wrong */
    {
      dbg( DI, DBG_HTML | 1, "something went terribly wrong. http address not found..." );
      http_free( r );
      return FALSE;
    }

    /* process pic list ... */
    /* if we get here we will always mirror the pic */

    pl->address = NULL;

    /* an error? uh oh */
    if( r->status != 200 || r->size == 0 )
    {
      e = htmlErrorString( r->status);
      view_setStatusLine( e );
      dbg( DI, DBG_HTML | 1, "%s: %s", pl->url, e );
      http_free( r );
      return TRUE;
    }

    if( ( f = genFilename( pl->url ) ) &&
	( dc = genDirectories( f ) ) &&
	( fd = fopen( f, "w" ) ) )
      ;
    else
    {
      if( f == NULL )
	dbg( DI, DBG_HTML|1, "could not make filename of pic %s !?", pl->url );
      else if( (dc = FALSE) )
	dbg( DI, DBG_HTML|1, "could not make directories for pic %s", f );
      else
	dbg( DI, DBG_HTML|1, "could not open pic file %s, errno = %d", f, errno);
    }

    if( fd )
    {
      fwrite( r->text, sizeof( char ), r->size, fd );
      fclose( fd );
      dbg( DI, DBG_HTML|3, "wrote %d bytes of picture %s", r->size, pl->url );
    }
    
  }

  http_free( r );
  return TRUE;
}

struct TreeInfo *html_getRoot()
{
  return CH.root;
}

int html_startLoading( char *url )
{
  char *mp;

  html_free();
  layout_recalcAll();

  /* generate fully qualified directory from mirrorpath */
  if( CP.mirrorhttp || CP.mirrorpic || CP.mirrorother )
  {
    if( ( mp = expandMirrorDir( CP.mirrorpath ) ) )
    {
      strncpy( CH.mirrorpath, mp, FILE_MAXLEN - 1 );
      CH.mirrorpath[ FILE_MAXLEN ] = '\0';
    }
    else
    {
      dbg( DI, DBG_HTML|1, "wrong mirrordir. mirroring disabled!");
      CP.mirrorhttp = CP.mirrorpic = CP.mirrorother = FALSE;
    }
  } 

  if( addNode( NULL, url ) )
    return FALSE;

  return TRUE;
}

/*
  performs a complete stop - free all requests which are not needed 
  further - these are htmldocuments (WebTree structure) and 
  graphics (piclist)
*/

void html_stopLoading()
{
  ER *e, *e2;
  TI *n;
  PL *p;
  http_abortAll();

  /* set all http addresses to NULL */
  for( n = CH.root; n ; n = n->nextList )
    n->address = NULL;

  /* same for picture list */
  for( p = CH.piclist; p ; p = p->next )
    p->address = NULL;
    
  /* and same for external requests */
  for( e = CH.extreq ; e ; )
  {
    e2 = e;
    e = e->next;
    /* notify callback that request has abandoned */
    e2->callback( -23, 0, NULL, e->users_data );
    free( e2 );
  }
  CH.extreq = NULL;
}

void startOrStopRecursive( TI *node, int start, int first )
{
  NTI *cti, *fti;
  
  fprintf(stderr, "start- or stopping %s\n", node->url );

  /*
    go through all childs and stop or start
  */

  for( cti = node->nextTree; cti; cti = cti->next )
  {
    if( cti->layout.type & DIRECT )
      startOrStopRecursive( cti->tree, start, FALSE );
  }

  if( !first )
    if( start )
      node->active = 1;
    else
      node->active = 0;
  
  if( start )
  {
    if( node->loaded == 0 )
      node->address = http_order( (char *)makeOrderName( node->url ) );
  }
  else if( node->address )
  {
    http_abort( node->address );
    node->address = NULL;
  }
}

void html_stopLoadingNode( TI *node )
{
  startOrStopRecursive( node, 0, TRUE );
  layout_recalcAll();
}

void html_startLoadingNode( TI *node )
{
  startOrStopRecursive( node, 1, TRUE );
  layout_recalcAll();
}

void html_getUrl( char *url ,void( *callback )( int, int, char *, void * ), void *userd )
{
}

void html_substituteMirrorLocations( void )
{
}


