#if !defined(typedopt_h)
#define typedopt_h

#include "optc.h"
#include <map>
#include <string>
#include "stringcol.h"
#include "composed.h"
#include "excpt.h"
#include <assert.h>

template<class CValue>
class COptions : public COptionsCode
{
public:
    typedef CValue (*FCompute)(const COptions<CValue> *, const char *);

private:
    typedef std::map<std::string, CValue> TValueMap;
    TValueMap ValueMap;

public:
    COptions();

    void Add(const char *Key, const CValue &Value);
        // checks key is not duplicate
    void Add(const char *Key, FCompute Compute);
	// checks key is not duplicate

    CValue Get(const char *Key) const;
	// checks key exists;

    // returns 0 OK, 1 key is not in ValueMap
    int Override(const char *Section,
        const char *Key, 
	const CValue &Value);

    virtual std::string GetKey(const char *Section, 
	const char *Key) const = 0;

    // belongs into CUnoOptions<> (ancestor of COptions<>)
    void Add(const char *Section, const char *Key, const CValue &Value);
    void Add(const char *Section, const char *Key, FCompute Compute);
    void SetUnoClass(const TStringCol &List,
        const char *Attr[],
        int AttrCount);
    CValue GetUno(const std::string &Name, const char *Key) const;
    CValue GetMostSpecific(const std::string &Section, const char *Key) const;

    bool CanAdd(const std::string &Key);
    inline void CheckCanAdd(const std::string &Key);
};

template<class CValue>
COptions<CValue>::COptions()
{
}

template<class CValue>
void COptions<CValue>::Add(const char *Key, const CValue &Value)
{
    assert(Key != NULL);
    std::string KeyStr = Key;
    CheckCanAdd(KeyStr);
    ValueMap[KeyStr] = Value;
}

template<class CValue>
void COptions<CValue>::Add(const char *Key,
    CValue (*Compute)(const COptions<CValue> *, const char *))
{
    assert(Key != NULL);
    std::string KeyStr = Key;
    CheckCanAdd(KeyStr);
    FncMap[KeyStr] = reinterpret_cast<FGenCompute>(Compute);
}

template<class CValue>
CValue COptions<CValue>::Get(const char *Key) const
{
    std::string KeyStr = Key;

    TValueMap::const_iterator vi = ValueMap.find(KeyStr);
    if (vi != ValueMap.end())
	return vi->second;

    TFncMap::const_iterator fi = FncMap.find(KeyStr);
    if (fi == FncMap.end())
	throw UndefinedKeyException(KeyStr);

    const FCompute Compute = reinterpret_cast<FCompute>(fi->second);
    return (*Compute)(this, Key);
}

template<class CValue>
int COptions<CValue>::Override(const char *Section,
    const char *Key,
    const CValue &Value)
{
    std::string KeyStr = GetKey(Section, Key);

    TValueMap::iterator i = ValueMap.find(KeyStr);
    if (i == ValueMap.end())
	return 1;

    i->second = Value;
    return 0;
}

template<class CValue>
CValue COptions<CValue>::GetMostSpecific(const std::string &Section, 
    const char *Key) const
{
    if (Section.IsEmpty())
	return Get(Key);

    std::string Compo = GetLongOptionKey(Section, Key);
    if (!CanAdd(Compo))
	return Get(Compo.c_str());
    else
	return Get(Key);
}

template<class CValue>
CValue COptions<CValue>::GetUno(const std::string &Name,
    const char *Key) const
{
    if (Name.empty())
	return Get(Key);

    std::string Compo = GetLongOptionKey(Name, Key);
    return Get(Compo.c_str());
}

template<class CValue>
void COptions<CValue>::Add(const char *Section, 
    const char *Key, 
    const CValue &Value)
{
    if (!Section)
	Add(Key, Value);
    else
    {   std::string LongKey = GetLongOptionKey(std::string(Section),
	    Key);
	Add(LongKey.c_str(), Get(Key));
    }
}

template<class CValue>
void COptions<CValue>::Add(const char *Section, const char *Key,
    CValue (*Compute)(const COptions<CValue> *, const char *))
{
    if (!Section)
	Add(Key, Compute);
    else
    {   std::string LongKey = GetLongOptionKey(std::string(Section),
	    Key);  
	Add(LongKey.c_str(), Get(Key));
    }
}

template<class CValue>
bool COptions<CValue>::CanAdd(const std::string &Key)
{
    return ValueMap.find(Key) == ValueMap.end() &&
	FncMap.find(Key) == FncMap.end();
}

template<class CValue>
inline void COptions<CValue>::CheckCanAdd(const std::string &Key)
{
    if (!CanAdd(Key))
	throw DuplicateKeyException(Key);
}

template<class CValue>
void COptions<CValue>::SetUnoClass(const TStringCol &List,
     const char *Attr[],
     int AttrCount)
{
    for (TStringCol::const_iterator ni = List.begin();
	 ni != List.end();
	 ++ni)
    {   for (int ai = 0; ai < AttrCount; ai++)
	{   std::string Key = GetLongOptionKey(*ni, Attr[ai]);   
	    Add(Key.c_str(), Get(Attr[ai]));
	}
    }
}

#endif

