/*
    Requires the Qt widget libraries, available at no cost at
    http://www.troll.no

    Copyright (C) 1998 Stephen Hutton
                       shutton@acm.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* propdialog.cpp */

#include "propdialog.h"
#include "propdialog.moc"


PropDialog::PropDialog( QWidget* parent, 
							 class UrlItem* pUrlItem, const char* name )
			: QDialog( parent, name, 1 )
{
	m_pUrlItem = pUrlItem; 

   	// url address widgets										
  	QLabel* pUrlLabel = new QLabel( i18n( "URL Address:" ), this, "");
   pUrlLabel->setMinimumSize( pUrlLabel->sizeHint() );

	m_pUrlEdit = new QLineEdit( this, "" );
	m_pUrlEdit->setMinimumSize( m_pUrlEdit->sizeHint() );
	m_pUrlEdit->setMaxLength( URL_LEN - 1 );
   m_pUrlEdit->setFocus();

   // separator
	QFrame* pSep1 = new QFrame( this );
   pSep1->setFrameStyle( 0x0004 | 0x0030 );
   pSep1->setLineWidth( 2 ); 
   pSep1->setFixedHeight( 2 );
	  	
	// how often widgets		
	QLabel* pHowOftenLabel = new QLabel( i18n( "Check every:" ), this, "");
   pHowOftenLabel->setMinimumSize( pHowOftenLabel->sizeHint() );
   		   
	m_pHowOftenCombo = new QComboBox( FALSE, this, "" );
	m_pHowOftenCombo->insertStrList( (const char**)timechoices );
	m_pHowOftenCombo->setMinimumSize( m_pHowOftenCombo->sizeHint() );
	
   m_pCheckSkipTags = new QCheckBox( i18n("Ignore tags when making best guess"), this, "" );
   m_pCheckSkipTags->setMinimumSize( m_pCheckSkipTags->sizeHint() );
   m_pCheckSkipTags->setChecked( pUrlItem->SkippingTags() );

	// separator 2
   QFrame* pSep2 = new QFrame( this );
   pSep2->setFrameStyle( 0x0004 | 0x0030 );
   pSep2->setLineWidth( 2 );
   pSep2->setFixedHeight( 2 );  
		
	// last time widgets		
	QLabel* pLastTimeLabel = new QLabel( i18n( "Last time checked:" ), this, "");
	pLastTimeLabel->setMinimumSize( pLastTimeLabel->sizeHint() );

	m_pLastTimeValue = new QLabel( this, "" );
	m_pLastResultValue = new QLabel("", this, "");
	 
	// next time widgets
	QLabel* pNextTimeLabel = new QLabel( i18n( "Next scheduled check:" ), this, "");
   pNextTimeLabel->setMinimumSize( pNextTimeLabel->sizeHint() );
   
	m_pNextTimeValue = new QLabel( this, i18n("(Not yet checked)") );
  
   // browser choice radio buttons 
   m_pRadioMyBrowser = new QRadioButton( i18n( "Use this browser: " ), this, "");
   m_pBrowserPathEdit = new QLineEdit( this, "" );
   m_pBrowseButton = new QPushButton( i18n( "&Select..." ), this, "" );
   m_pRadioDefault = new QRadioButton( i18n( "Use default browser" ), this, "" );
   
   QSize button_size = m_pBrowseButton->sizeHint();
   m_pBrowseButton->setMinimumSize( button_size );
   m_pRadioDefault->setMinimumSize( m_pRadioDefault->sizeHint() );
   m_pRadioMyBrowser->setMinimumSize( m_pRadioMyBrowser->sizeHint() );
   m_pBrowserPathEdit->setMinimumSize( m_pBrowserPathEdit->sizeHint() );
   m_pBrowserPathEdit->setMaxLength( BROWSER_PATH_LEN - 1 );

   // separator 3
   QFrame* pSep3 = new QFrame( this );
   pSep3->setFrameStyle( 0x0004 | 0x0030 );
   pSep3->setLineWidth( 3 );
   pSep3->setFixedHeight( 3 );     	
	
	// ok & cancel buttons
	m_pOKButton = new QPushButton( i18n( "&OK" ), this, "ok" );
	m_pCancelButton = new QPushButton( i18n( "&Cancel" ), this, "cancel" );
	
   m_pOKButton->setMinimumSize(  button_size );  
   m_pCancelButton->setMinimumSize( button_size ); 
  
	// fill in the details for this url
	m_pUrlEdit->setText( pUrlItem->GetUrl() );

	if ( pUrlItem->GetLastTime() )
	{
	   QString lt = pUrlItem->GetLastTime();
	   int colon = lt.find( ':' );
	   m_pLastTimeValue->setText( lt.left(colon + 3) );
	}
	else 
	   m_pLastTimeValue->setText( "N/A" );
	
	m_pNextTimeValue->setText( ::StripTime( *(pUrlItem->GetNextTime()) ));

	m_pLastTimeValue->setMinimumSize( m_pLastTimeValue->sizeHint() );
   m_pNextTimeValue->setMinimumSize( m_pNextTimeValue->sizeHint() );

	// Fill in the combobox with the current value of HowOften	
	int i;
	for( i=0; strcmp (timechoices[i], pUrlItem->GetHowOften()); i++ );
	   m_pHowOftenCombo->setCurrentItem(i);

   // Find out if our result is a guess
   QString result_prefix = "(";
   if( pUrlItem->UsingBestGuess() )
       result_prefix += i18n( "BEST GUESS: " );

	// Fill in our last result
	switch( pUrlItem->GetResult() )
	{
		case NOT_YET_CHECKED:
			m_pLastResultValue->setText( i18n("") );
			break;
		case NO_CHANGE:
			m_pLastResultValue->setText(  result_prefix + i18n("Unchanged)") );
			break;
		case UPDATE_FOUND: 
			m_pLastResultValue->setText(  result_prefix + i18n("Update found!)") );
			break;
		case CHECK_NOT_NEEDED:
		   m_pLastResultValue->setText( i18n("(No check needed)") );
		   break;
		case CONNECT_FAILED:
			m_pLastResultValue->setText( i18n("(Connection failed)") );
			break;
	   case UPDATE_READ:
		   m_pLastResultValue->setText( i18n("(Update read)") );
		   break;
		case URL_SYNTAX_ERROR:
		   m_pLastResultValue->setText( i18n("(URL syntax error)") );
		   break;
		case URL_NOT_FOUND:
		   m_pLastResultValue->setText( i18n("(URL not found)") );
		   break;
		case URL_MOVED:
		   m_pLastResultValue->setText( i18n("(URL moved)") );
		   break; 
	}

	// check one of the radio buttons and fill in our browser path if needed
	if( pUrlItem->UsingDefaultBrowser() )
	{
	   // check the default browser button
	   m_pRadioDefault->setChecked( TRUE );
	    
      // disable the browser edit box and the "select" button
      m_pBrowserPathEdit->setEnabled( FALSE );
      m_pBrowseButton->setEnabled( FALSE );
   }
   else
   {
      // check the "use this browser button"
      m_pRadioMyBrowser->setChecked( TRUE );
      
      // fill in the current browser path for this url
      m_pBrowserPathEdit->setText( pUrlItem->GetBrowserPath() );
      
	}					
	// main layout for dialog
   QBoxLayout* pTopLayout = new QVBoxLayout( this, 10 );		
				
   // Layout the url address box & label
   QBoxLayout* pAddressLayout = new QHBoxLayout();
   pTopLayout->addLayout( pAddressLayout );
   
   pAddressLayout->addWidget( pUrlLabel );
   pAddressLayout->addWidget( m_pUrlEdit, 1 );
  	
	// Layout the HowOften row
	QBoxLayout* pHowOftenLayout = new QHBoxLayout();
	pTopLayout->addLayout( pHowOftenLayout );
	pHowOftenLayout->addWidget( pHowOftenLabel );
	pHowOftenLayout->addWidget( m_pHowOftenCombo );

	pTopLayout->addWidget( m_pCheckSkipTags );
   pTopLayout->addStretch( 1 ); 
   pTopLayout->addWidget( pSep1, 1 );
	
	// Layout the last time row
	QBoxLayout* pLastTimeLayout = new QHBoxLayout();
	pTopLayout->addLayout( pLastTimeLayout, 1 );
	pLastTimeLayout->addWidget( pLastTimeLabel );
	pLastTimeLayout->addStretch( 1 );
	pLastTimeLayout->addWidget( m_pLastTimeValue );
	
  	QBoxLayout* pResultLayout = new QHBoxLayout();
	pTopLayout->addLayout( pResultLayout );
	m_pLastResultValue->setMinimumSize( m_pLastResultValue->sizeHint() );
	pResultLayout->addStretch( 1 );
	pResultLayout->addWidget( m_pLastResultValue );

	// Layout the next time row   
	QBoxLayout* pNextTimeLayout = new QHBoxLayout();
   pTopLayout->addLayout( pNextTimeLayout, 1 );
	pNextTimeLayout->addWidget( pNextTimeLabel );
	pNextTimeLayout->addStretch( 1 );
	pNextTimeLayout->addWidget( m_pNextTimeValue );
  
   pTopLayout->addStretch( 2 ); 
   pTopLayout->addWidget( pSep2, 1 );

   // Rows for "use this browser" & "use default browser"
   QBoxLayout* pUseThisLayout = new QHBoxLayout();
   QBoxLayout* pUseDefaultLayout = new QHBoxLayout();
   pTopLayout->addLayout( pUseThisLayout );
   pTopLayout->addLayout( pUseDefaultLayout );
        
   pUseThisLayout->addWidget( m_pRadioMyBrowser );
   pUseThisLayout->addWidget( m_pBrowserPathEdit, 1 );
   
   pUseDefaultLayout->addWidget( m_pRadioDefault );
   pUseDefaultLayout->addStretch( 1 ); //spacing
   pUseDefaultLayout->addWidget( m_pBrowseButton );

   pTopLayout->addWidget( pSep3, 1 );
      
   // add a horizontal layout for OK & CANCEL
   QBoxLayout* pOkCancelLayout = new QHBoxLayout();
   pTopLayout->addLayout( pOkCancelLayout );
   pOkCancelLayout->addStretch(1);
   pOkCancelLayout->addWidget( m_pOKButton );
   pOkCancelLayout->addWidget( m_pCancelButton );
	
	pTopLayout->activate();
	
	QString caption;
	caption.sprintf( "KWebWatch: %s properties", pUrlItem->GetAlias() );
	this->setCaption( caption );
		
	connect( m_pOKButton, SIGNAL(clicked()), SLOT( ExitUrl() )); 
	connect( m_pCancelButton, SIGNAL(clicked()), SLOT(reject()) );	
   connect( m_pBrowseButton, SIGNAL(clicked()), SLOT( ShowFileDialog() ));
   connect( m_pRadioDefault, SIGNAL(clicked()), SLOT( UseDefaultBrowser() ));
   connect( m_pRadioMyBrowser, SIGNAL(clicked()), SLOT( UseThisBrowser() ));
}



void PropDialog::ExitUrl()
{
   // save any changes the user has made back to our object
   m_pUrlItem->SetUrl( TrimString( (char*)m_pUrlEdit->text()) );
   m_pUrlItem->ParseUrl();
  
   m_pUrlItem->SetHowOften( m_pHowOftenCombo->currentText() );
   m_pUrlItem->UpdateNextTime( );

	if( !m_pUrlItem->UsingDefaultBrowser() )
   	  m_pUrlItem->SetBrowserPath( m_pBrowserPathEdit->text() );
	  
	m_pUrlItem->SetSkipTags(  m_pCheckSkipTags->isChecked() );  
				
	// Write our changes to the dbm	
	m_pUrlItem->WriteToDisk();
	
	// Close the dialog
	this->hide();
}

void PropDialog::UseThisBrowser()
{
   m_pRadioDefault->setChecked( FALSE );
   m_pBrowserPathEdit->setEnabled( TRUE );
   m_pBrowseButton->setEnabled( TRUE );
   
   m_pUrlItem->SetUsingDefaultBrowser( FALSE );
}

void PropDialog::UseDefaultBrowser()
{
   m_pRadioMyBrowser->setChecked( FALSE );
   m_pBrowserPathEdit->setEnabled( FALSE );
   m_pBrowseButton->setEnabled( FALSE );

   // get our default browser path from the config file
   KConfig* pConfig = kapp->getConfig();
	
	pConfig->setGroup( "General Options" );
   QString path = pConfig->readEntry( "Default Browser", "" );

   m_pUrlItem->SetUsingDefaultBrowser( TRUE );
   m_pUrlItem->SetBrowserPath( path );   
}

void PropDialog::ShowFileDialog()
{
   m_pBrowserPathEdit->setText(KFileDialog::getOpenFileName() );
}

