/*

    Requires the Qt widget libraries, available at no cost at
    http://www.troll.no

    Copyright (C) 1998 Stephen Hutton
                       shutton@acm.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/   

/* mainwidget.cpp */

#include "mainwidget.h"
#include <qdatetm.h>
#include <qpainter.h>
#include <qtimer.h>
#include <kmsgbox.h>
#include <kapp.h>

#include "mainwidget.moc"

#define MAX_ARGS 100

QString g_data_file;
QString g_icon_dir;
QString g_pics_dir;

MainWidget::MainWidget( QWidget* parent, const char* name )
		 	 : KTopLevelWidget( name )

{
  	this->setCaption( APP_CAPTION );
	g_icon_dir = kapp->kde_icondir();
	g_pics_dir = kapp->kde_datadir() + "/kwebwatch/pics";
	        
   m_bCheckingUrls = FALSE;
   
	m_pPopUpMenu = new QPopupMenu();
	m_pMenuUrl = new QPopupMenu();
	m_pMenuView = new QPopupMenu();
	m_pMenuHelp = new QPopupMenu();

	// Right mouse popup menu
	m_pPopUpMenu->insertItem( i18n( "&Properties..." ), this, SLOT( ShowProperties() ) );
	m_pPopUpMenu->insertItem( i18n( "&Launch Browser" ), this, SLOT( LaunchSelected() ) );
	m_pPopUpMenu->insertItem( i18n( "&Check Now" ), this, SLOT( CheckSelectedNow() ) );
	m_pPopUpMenu->insertItem( i18n( "&Mark as Read" ), this, SLOT( MarkSelectedRead() ) );
	m_pPopUpMenu->insertSeparator();
	m_pPopUpMenu->insertItem( i18n( "&Delete" ), this, SLOT( Delete() ) );
		
	// Url Menu
	m_pMenuUrl->insertItem( i18n( "&New..." ), this, SLOT(AddUrl()) );
	m_pMenuUrl->insertItem( i18n( "&Properties..." ), this, SLOT( ShowProperties() ) );
	m_pMenuUrl->insertItem( i18n( "&Launch Browser" ), this, SLOT( LaunchSelected() ) );
	m_pMenuUrl->insertItem( i18n( "&Check Now" ), this, SLOT( CheckSelectedNow( ) ) );
	m_pMenuUrl->insertItem( i18n( "&Mark as Read" ), this, SLOT( MarkSelectedRead() ) );
	m_pMenuUrl->insertSeparator();
	m_pMenuUrl->insertItem( i18n( "&Delete" ), this, SLOT( Delete() ) );
	m_pMenuUrl->insertItem( i18n( "Check &All URLs Now" ), this, SLOT( CheckAllUrls() ) );
	m_pMenuUrl->insertSeparator();
	m_pMenuUrl->insertItem( i18n( "E&xit" ), kapp->getKApplication(), SLOT(quit()) );

   // View Menu
   m_pMenuView->insertItem( i18n( "&Dock to panel" ), this, SLOT( DockToPanel() ));
   m_pMenuView->insertSeparator();
   m_pMenuView->insertItem( i18n( "&Options..." ), this, SLOT( ShowOptions() ));
   
   // Help Menu
   m_pMenuHelp->insertItem( i18n( "&Contents" ), this, SLOT( HelpContents()), Key_F1);  
   m_pMenuHelp->insertItem( i18n( "&About KWebWatch..." ), this, SLOT( HelpAbout() ));
   
	m_pMenuBar = new QMenuBar( this );
	CHECK_PTR( m_pMenuBar );
	
	m_pMenuBar->insertItem( i18n( "&Url" ), m_pMenuUrl );
	m_pMenuBar->insertItem( i18n( "&View" ), m_pMenuView );
	m_pMenuBar->insertSeparator();
	m_pMenuBar->insertItem( i18n( "&Help" ), m_pMenuHelp );
	
   m_pLayout = new QVBoxLayout( this, 5 );
	m_pListBox = new MainListBox(this);
	m_pStatusText = new QLabel(this);

   m_pListBox->setFocus();
			
   // write the time to the status bar
   m_pStatusText->setText( ::StripTime( QDateTime::currentDateTime() ) );
   m_pStatusText->setFrameStyle( QFrame::Panel | QFrame::Sunken );
   
   m_pStatusText->setFixedHeight( m_pStatusText->sizeHint().height() );
   m_pStatusText->setAlignment( AlignVCenter | AlignLeft );
    
	LoadUrls(); // read everything in from disk
		
	m_pListBox->show();
	m_pLayout->setMenuBar( m_pMenuBar );
   m_pLayout->addWidget( m_pListBox );
  	m_pLayout->addWidget( m_pStatusText); 
	
	m_pLayout->activate();
	
	m_pDockWidget = new DockWidget( this, "");

   // Set up a timer that goes off every second
	QTimer *timer = new QTimer( this );    
	connect( timer, SIGNAL(timeout()), this,
	     SLOT( CheckUrlsIfTime()) ); 
	     	     
	timer->start( 1000, FALSE );  
	
   // Get initial values from config file
   KConfig* pConfig = kapp->getConfig();
   QString oldgroup = pConfig->group();									      							      
   
   // use our last geometry if it is there
   QRect* pDefaultGeom = new QRect( 0, 0, 210, 480 );  
   pConfig->setGroup( "General Options" );
   QRect geom = pConfig->readRectEntry( "MainWindow", pDefaultGeom );
   this->setGeometry( geom );
   delete pDefaultGeom;  
      
   // If there is no default browser, set it to KFM
   pConfig->setGroup( "General Options" );
   QString browser = pConfig->readEntry( "Default Browser", "" );
   if( browser == "" )
   {
      QString kfmclient_path = kapp->kde_bindir() + "/kfmclient openURL";
      pConfig->writeEntry( "Default Browser", kfmclient_path );
      
   }

   m_StartDocked = pConfig->readBoolEntry( "StartDocked", 0 );
   
   if ( m_StartDocked )
      DockToPanel();
   else
      this->show();

  	connect( m_pListBox, SIGNAL(RightMouse(QPoint)), this,
      		SLOT( ShowPopupMenu(QPoint)) );
	
	connect( m_pListBox, SIGNAL( selected(int) ), this,
	     	     SLOT( LaunchSelected() ) );	

	if( pConfig->readBoolEntry( "CheckUrlsAtStart", 0 ) )
	{
	   CheckAllUrls();
	}
		     	     
   pConfig->setGroup( oldgroup ); 							   	
}

MainWidget::~MainWidget()
{
	 KConfig* pConfig = kapp->getConfig();										      
																					      										      
	 pConfig->setGroup( "General Options" );
	 pConfig->writeEntry( "MainWindow", this->geometry() );
}


// Read all of our urls from disk;  Create a list of UrlItem objects
void MainWidget::LoadUrls()
{
	// create a urllist object
	m_pUrlList = new UrlList();
	
	// define a structure to retrieve the data from dbm
	struct url_data data;
	memset( &data, '\0', sizeof(data) );
	
	DBM*	dbm_ptr;
	datum key;
	datum retrieved_datum;
	
	QString local_data_dir = kapp->localkdedir() + "/share/apps/kwebwatch";
	mkdir( local_data_dir, S_IRUSR | S_IWUSR | S_IXUSR );
	
	g_data_file = local_data_dir + "/urls";
	
	dbm_ptr = dbm_open( g_data_file, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR );
	int list_box_position = 0;
	
	for( key = dbm_firstkey(dbm_ptr); key.dptr; key = dbm_nextkey(dbm_ptr) )
	{
		// read data from dbm to create a urlitem
		retrieved_datum = dbm_fetch( dbm_ptr, key );
		if ( retrieved_datum.dptr )
		{
			memcpy( &data, retrieved_datum.dptr, retrieved_datum.dsize );
	
			// translate the NextTime string into a QDateTime
			QString* pStr = new QString(data.next_time);
			QDateTime* pDateTime = GetDateTimeFromString( *pStr );
			
			// add the urlitem to our list
			UrlItem* pUrlItem =  new UrlItem( data.alias,
			                                  data.url,
			                                  data.how_often,
								                   data.use_default_browser,
										             data.browser_path,
												       data.best_guess,
												       data.skip_tags,
									                data.last_time,
											          pDateTime,
												       data.result,
								                   data.last_modified,
										             data.size );
			m_pUrlList->Add( pUrlItem ); 
								  
		   // add the item's alias to our listbox 
			QListBoxText* pListText = new QListBoxText( data.alias );
			m_pListBox->insertItem( pListText, list_box_position );
	    
	      UpdateListBox( pUrlItem, FALSE );
	      
			list_box_position++;
			memset( &data, '\0', sizeof( data ) );
		}
	}
	dbm_close( dbm_ptr );

}


// SLOTS:
	
void MainWidget::AddUrl()
{
	m_pNewDialog = new NewDialog( this, m_pUrlList, "New Url" );

	connect( m_pNewDialog, SIGNAL( NewListBoxItem(const char*) ),
			   this ,SLOT( AddItemToListBox(const char*) ) );

   connect( m_pNewDialog, SIGNAL( PaintNewLBItem( UrlItem*, bool ) ),
			   this ,SLOT( UpdateListBox( UrlItem*, bool ) ) );
  	
	QRect main_geometry = geometry();
	const int offset = 180;
	int dlg_x = main_geometry.x() + offset;
	int dlg_y = main_geometry.y() + offset;
	
	if( main_geometry.x() > offset )
		dlg_x = main_geometry.x() - offset -25;
	if( main_geometry.y() > offset )
		dlg_y = main_geometry.y() - offset;
		
	m_pNewDialog->setGeometry( dlg_x, dlg_y, 350, 300 );

   m_pNewDialog->resize( m_pNewDialog->sizeHint() );
			
	m_pNewDialog->show();
}


void MainWidget::AddItemToListBox( const char* item )
{
	m_pListBox->insertItem( item, -1 );
}

	
void MainWidget::Delete()
{
	char selected_text[ ALIAS_LEN ];
	
	// Get the current selection from the list box
	int index = m_pListBox->currentItem();
	
	if( index >= 0 )
	{
		strcpy( selected_text, m_pListBox->text(index) );
		
		// Delete it from the database
		struct url_data data;
	   memset( &data, '\0', sizeof(data) );
	
	   DBM*	dbm_ptr;
	   datum key;
		key.dptr = selected_text;
		key.dsize = strlen(selected_text);
		
	   dbm_ptr = dbm_open( g_data_file, O_RDWR, S_IRUSR | S_IWUSR  );
		dbm_delete( dbm_ptr, key );
		dbm_close( dbm_ptr );
		
		// Remove it from the URlList
	   UrlItem* pItemToRemove = m_pUrlList->Find( m_pListBox->text( index ) );
		m_pUrlList->Remove( pItemToRemove );
	
		// Delete it from the listbox	
		m_pListBox->removeItem(index);
	}
}

void MainWidget::ShowProperties()
{
	char *selected_text= new char[51];	
	
	// Get the current selection from the list box
	int index = m_pListBox->currentItem();
	
	if( index >= 0 )
	{
		selected_text = (char*)m_pListBox->text( index );
		UrlItem* pItemToDisplay = m_pUrlList->Find( m_pListBox->text( index ) );
	
		m_pPropDialog = new PropDialog( this, pItemToDisplay,	"" );
				         
		const int offset = 180;
	   QRect main_geometry = geometry();
		int dlg_x = main_geometry.x() + offset;
	   int dlg_y = main_geometry.y() + offset;
	
		if( main_geometry.x() > offset )
			dlg_x = main_geometry.x() - offset -25;
		if( main_geometry.y() > offset )
			dlg_y = main_geometry.y() - offset;
		
		m_pPropDialog->setGeometry( dlg_x, dlg_y, 350, 300 );

		m_pPropDialog->resize( m_pPropDialog->sizeHint() );
		m_pPropDialog->show();
		
	}
}


void MainWidget::ShowPopupMenu( QPoint point )
{
	QPoint GlobalPt = mapToGlobal( point );
	m_pPopUpMenu->popup( GlobalPt );
}

void MainWidget::CheckAllUrls()
{
   int last_result;
    if( !m_bCheckingUrls )
      {
	      for (int i = 0; i < m_pUrlList->Count(); i++ )
         {
	         UrlItem* pUrlItem = m_pUrlList->GetAt( i );
           
				last_result = pUrlItem->GetResult();
	         if( last_result != UPDATE_FOUND && last_result != CHECK_NOT_NEEDED )
			   {
				      if (!CheckNow( pUrlItem ))
					       CheckNow( pUrlItem );
			   }
			   else if ( last_result == UPDATE_FOUND )
			   {
			         pUrlItem->SetResult( CHECK_NOT_NEEDED );
				     				      
				      pUrlItem->UpdateLastTime();
		            pUrlItem->UpdateNextTime();
		            
 		            pUrlItem->WriteToDisk();
				      UpdateListBox( pUrlItem, TRUE );
 			   }
		     
	     } // end for
   	 }    

   
}

void MainWidget::CheckUrlsIfTime()
{
   QDateTime current_time = QDateTime::currentDateTime();
   int last_result;
   
   if( current_time.time().second() == 0 )
   {   
      // write the time to the status bar
      m_pStatusText->setText( ::StripTime( current_time ) );
      
       if( !m_bCheckingUrls )
      {
	      for (int i = 0; i < m_pUrlList->Count(); i++ )
         {
	         UrlItem* pUrlItem = m_pUrlList->GetAt( i );
            if( current_time > *(pUrlItem->GetNextTime()) )
			   {
			      last_result = pUrlItem->GetResult();
	            if( last_result != UPDATE_FOUND && last_result != CHECK_NOT_NEEDED )
				   {
				      if (!CheckNow( pUrlItem ))
					       CheckNow( pUrlItem );
			      }
			      else if ( last_result == UPDATE_FOUND )
			      {
			         pUrlItem->SetResult( CHECK_NOT_NEEDED );
				     				      
				      pUrlItem->UpdateLastTime();
		            pUrlItem->UpdateNextTime();
		            
 		            pUrlItem->WriteToDisk();
				      UpdateListBox( pUrlItem, TRUE );
 				   }
		     }
	     } // end for
   	 }    

      
   }
}



void MainWidget::CheckSelectedNow()
{
	char selected_text[51];	
	
	// Get the current selection from the list box
	int index = m_pListBox->currentItem();
	
	if( index >= 0 )
	{
		// find our matching item in memory
		strcpy( selected_text, m_pListBox->text( index ) );
		UrlItem* pItem = m_pUrlList->Find( selected_text );
	   
	   if (!CheckNow( pItem ))
	      CheckNow( pItem ); // retry if the URL has been moved
  	}
	
} 


bool MainWidget::CheckNow( UrlItem* pUrl )
{
	int fd;
	int n;
   int  result;
	char buff[MAX_HEADER + 1];
	char server_text[MAX_HEADER + 1];
	bool bRtn = TRUE;
	char status_text[300];
   QString addr;
   QString port;
  
   m_bCheckingUrls = TRUE;
   QApplication::setOverrideCursor( waitCursor );

   KConfig* pConfig = kapp->getConfig();
   pConfig->setGroup( "General Options" );

   bool proxy_flag = pConfig->readBoolEntry( "UseProxy", 0 );

   sprintf( status_text, i18n( "Connecting to %s..." ), pUrl->GetHost() );
	m_pStatusText->setText( status_text );
	
	if (!proxy_flag)
   {
	   fd = tcp_connect( pUrl->GetHost(), SERV );
	}
	else
   {
      pConfig->setGroup( "Proxy Support" );
      addr =  pConfig->readEntry( "Proxy Address", ""  );
      port =  pConfig->readEntry( "Proxy Port", ""  );

      fd = tcp_connect( addr, port );
   }
	if( fd < 0 )
	{
	   result = CONNECT_FAILED;
	}
	else
	{  
	   // Go get the HTML
	   m_pStatusText->setText( i18n( "Querying server..." ) );
	 	     			
	   if( !proxy_flag )// if not using proxy
	   {   		  	  
	      n = snprintf( buff, sizeof(buff), GET_CMD, pUrl->GetFname(),
				        pUrl->GetLastModified() );
		}
		else // else using proxy - prepend the host to our filename
		{
		   char file_to_get[URL_LEN];
         char proxy_address[PROXY_ADDR_LEN];

	      snprintf( proxy_address, sizeof( proxy_address ), "%s:%s",
			          (const char*)addr, 
				       (const char*)port );

	      snprintf( file_to_get, sizeof( file_to_get ), "%s%s",
			          pUrl->GetHost(), pUrl->GetFname() );

	      n = snprintf( buff, sizeof(buff), GET_CMD_PROX, file_to_get, 
				        pUrl->GetLastModified() );
      }
		
      warning(buff);
	   writen( fd, buff, n ); // send our get command over the socket
   	  
      m_pStatusText->setText( i18n( "Reading response..." ) );
      kapp->processEvents();
	          
      // read in MAXHEADER characters
      n = read(fd, buff, MAX_HEADER );
	   strcpy( server_text,  buff ); 
	   server_text[n] = '\0';
      int total_read = n;
	   
	   //testing
	   warning( server_text );
	   while( n != 0 && total_read != MAX_HEADER )
	   {
	      kapp->processEvents();
	      n = read(fd, buff, MAX_HEADER - total_read);
	      
	      buff[n] = '\0';
	      strcat( server_text, buff );
			   
	      total_read += n;
	   }
	
	   if( !total_read )
	   { 
	      result = CONNECT_FAILED;
	   }
	   else // continue if we read anything
	   {
	      // Find out what HTTP code the server returned
	      char* p = strstr( server_text, "HTTP/" );
	      if ( !p )
		       result = CONNECT_FAILED;
		   else
		   {
		      p += strlen( "HTTP/1.1 " );
		      int rc = atoi( p );
		      switch( rc )
		      {
			       case 304: // server returned not modified
				       result = NO_CHANGE;
				       break;
			       case 200: // server reports OK - see if it honors last modified requests
					    if ( pUrl->LookForLastModifiedDate( server_text ) )
					       result = UPDATE_FOUND;
					    else if ( pUrl->SkippingTags() ) // compare char count outside of tags 
						 {
   					       pUrl->SetBestGuess( TRUE );

						    if ( pUrl->LengthNotCountingTagsChanged( fd, server_text ) )
						       result = UPDATE_FOUND;
						    else
						       result = NO_CHANGE;
						 }
				       else  // compare character count including tags
				       {
					       pUrl->SetBestGuess( TRUE ); 
						    if ( pUrl->FileBodyLengthChanged( fd, server_text ) )
						       result = UPDATE_FOUND;
						    else
						       result = NO_CHANGE;
  			          }
				       break;
			       case 404:
				       result = URL_NOT_FOUND;
				       break;
				    case 302:  // if the url has moved,
					 case 301:  // get the new one and return false so we can retry  
				       result = URL_MOVED;
			          p = strstr( server_text, "Location: " );
				       if ( p )
					    {
					       // extract the new url
						    strtok( p, " " );
						    
						    char* new_url = strtok( NULL, "\r" );
						    pUrl->SetUrl( new_url );
						    pUrl->ParseUrl();
		             }
				       bRtn = FALSE;
				       break;
				    default:
					    result = UNKNOWN_ERROR;
					    char msg[300];
					    sprintf( msg, "KWebWatch: %s returned %d", pUrl->GetHost(), rc );
					    warning( msg );
					    break;
 		      } // end switch()
		   }
		   
     } // end if( total_read )   
     ::close(fd);
               
   } // end if( fd ) 

	pUrl->SetResult( result );
	pUrl->UpdateLastTime();
	pUrl->UpdateNextTime( );
   pUrl->WriteToDisk();
    
   UpdateListBox( pUrl, TRUE );

   // add a menuitem to the dockwidget object
   if( UPDATE_FOUND == result )
   {
      m_pDockWidget->UpdateMenu( pUrl, TRUE );
   }
           
   // write the time to the status bar
   m_pStatusText->setText( ::StripTime( QDateTime::currentDateTime() ) );
   
   QApplication::restoreOverrideCursor();
   m_bCheckingUrls = FALSE;
   
   return bRtn;
}

void MainWidget::UpdateListBox( UrlItem* pItem, bool checked_this_session )
{ 
	QColor color;
	int 		 position;
	QString item_text = pItem->GetAlias();
	int result;
	
	// find the position of this urlitem in the listbox
	int index = 0;
	int count = m_pListBox->count();
  
   while ( index < count && item_text != m_pListBox->text(index) )
	{ 
      index++;
	}
						
   QPixmap* pPix;
	
	result = pItem->GetResult();
	      
   if( checked_this_session )
   {
      if( result == URL_NOT_FOUND || result == CONNECT_FAILED || result == URL_MOVED )
	      pPix = new QPixmap( g_pics_dir + "/kww_red-bullet.xpm" );
	   else
         pPix = new QPixmap( g_pics_dir + "/kww_green-bullet.xpm" );
   }
   else
      pPix = new QPixmap( g_pics_dir + "/kww_grey-bullet.xpm" );

   position = -1; //end of list
   switch( result )
   {  
	      case NOT_YET_CHECKED:
	         color = NULL;
		      break;
	      case UPDATE_FOUND:	/* FALLTHRU */
		   case CHECK_NOT_NEEDED:
		      position = 0;
		    	 color = UPDATE_FOUND_COLOR;
			   pPix = new QPixmap( g_pics_dir+ "/kww_new.xpm" );
		      break;
	      case UPDATE_READ:
	         color = NULL;
		      break;
	      case NO_CHANGE:
		      color = NULL;
		      break;
		   case CONNECT_FAILED:
		      color = CONNECT_FAILED_COLOR;
		      break;
		   case NO_DATE_PROVIDED:
		      color = NO_DATE_PROVIDED_COLOR;
		      break;
		   case URL_SYNTAX_ERROR:
		      color = URL_SYNTAX_ERROR_COLOR;
		      break;
		   case URL_NOT_FOUND:
		      color = URL_NOT_FOUND_COLOR;
		      break;
		   case URL_MOVED:
		      color = URL_MOVED_COLOR;
		      break;
		   case UNKNOWN_ERROR:
		      color = UNKNOWN_ERROR_COLOR;
		      break;
	 }  
		  
    MyListBoxItem* pLBI = new MyListBoxItem( item_text, color, *pPix );
    
  	 if( count == 1 )
    {
	    m_pListBox->removeItem( 0 );
       m_pListBox->insertItem( pLBI, 0 );	 
    }
	 else
    {
       m_pListBox->removeItem( index );
		 m_pListBox->insertItem( pLBI, position );	     
	 }
  
}

UrlItem* MainWidget::GetSelectedUrlItem()
{
   UrlItem* pItem = NULL;
   char selected_text[51];

   // Get the current selection from the list box
	int index = m_pListBox->currentItem();
	
	if( index >= 0 )
	{
		// find our matching item in memory
		strcpy( selected_text, m_pListBox->text( index ) );
		pItem = m_pUrlList->Find( selected_text );
   }
   
   return pItem;
}

void MainWidget::LaunchSelected()
{
   UrlItem* pItem = NULL;
   pItem = GetSelectedUrlItem();
   
   if( pItem )
      LaunchBrowser( pItem );
}

void MainWidget::LaunchBrowser( UrlItem* pItem )
{
   char url[300];
   char* tmp = new char[300];
   	       
  	strcpy( url, pItem->GetUrl() );
	   
	char* pFind = strstr( url, "http://" );
	if (!pFind)
	   sprintf( url, "http://%s", pItem->GetUrl() );

	// parse the path into its arguments if there are any
	char* args[MAX_ARGS];
	char path[300];
	char* temp;
	strcpy( path,  pItem->GetBrowserPath() );
	temp = path;
		
	int j;
	for( j=0; j < MAX_ARGS; j++ )
	{
	   if( (args[j] = strtok(temp, " ")) == 0 )
	       break;
	
	   temp = NULL;
	}
	
	args[j++] = url;
	args[j] = NULL;
		
      
	pid_t pid = fork();
	   	      
	if ( 0 == pid )
	{
	   int rc = execvp( args[0], args );
	   if( -1 == rc )
	   {
	      warning("execv FAILED (could not launch browser)!" );
	    	 exit(-1);
      }
	}
		
	pItem->SetResult( UPDATE_READ );
	pItem->UpdateLastTime();
   pItem->UpdateNextTime( );
      
   pItem->WriteToDisk();
   
   // Remove the item from the dockwidget menu
   m_pDockWidget->UpdateMenu( pItem, FALSE );  
	    
  	UpdateListBox( pItem, TRUE );
 
	delete []tmp;
}


void MainWidget::ShowOptions()
{
	OptionsDialog* pDlg = new OptionsDialog();
					   
	connect( pDlg, SIGNAL( NewDefaultBrowser(const char*) ),
	    this, SLOT( ChangeDefaultBrowserPaths( const char*) ) );				   
					   
	const int offset = 100;
	QRect main_geometry = geometry();
	int dlg_x = main_geometry.x() + offset;
	int dlg_y = main_geometry.y() + offset;
	
	if( main_geometry.x() > offset )
		dlg_x = main_geometry.x() - offset -25;
	if( main_geometry.y() > offset )
		dlg_y = main_geometry.y() - offset;
		
	pDlg->setGeometry( dlg_x, dlg_y, 370, 160 );

   pDlg->resize( pDlg->sizeHint() );
	pDlg->show();
	
}

void MainWidget::ChangeDefaultBrowserPaths( const char* path )
{
    for (int i = 0; i < m_pUrlList->Count(); i++ )
    {
	    UrlItem* pUrlItem = m_pUrlList->GetAt( i );
       if( pUrlItem->UsingDefaultBrowser() )
          pUrlItem->SetBrowserPath( path );
	 }
}

void MainWidget::MarkSelectedRead()
{
   char selected_text[51];	
	
	// Get the current selection from the list box
	int index = m_pListBox->currentItem();
	
	if( index >= 0 )
	{
		// find our matching item in memory
		strcpy( selected_text, m_pListBox->text( index ) );
			 
		UrlItem* pItem = m_pUrlList->Find( selected_text );
	   
	   pItem->SetResult( UPDATE_READ );
     	pItem->UpdateLastTime();
      pItem->UpdateNextTime( ); 

	   pItem->WriteToDisk();
	   UpdateListBox( pItem, TRUE );
	   
	   // Remove the item from the dockwidget menu
      m_pDockWidget->UpdateMenu( pItem, FALSE );  
  	}
}

void MainWidget::DockToPanel()
{
   m_pDockWidget->Dock();
   this->hide();
}

void MainWidget::HelpAbout()
{
   QString msg = "KWebWatch 0.6\n\n";
   msg += "Copyright (c) 1998 Stephen Hutton <shutton@acm.org>\n\n";
   msg += "This program is free software; you can redistribute it\nand/or modify";
   msg += " it under the terms of the GNU General Public\nLicense as";
	msg += " published by";
   msg += " the Free Software Foundation; either\nversion 2 ";
   msg += "of the License, or (at your option) any later version.\n\n"; 
   msg += "The program is distributed in hopes that it will be useful,\nbut";
   msg += " WITHOUT ANY WARRANTY; without even the implied\nwarranty of";
   msg += " MERCHANTABILITY or FITNESS FOR A\nPARTICULAR PURPOSE. See the GNU";
   msg += " General Public\nLicense for more details.\n\n";
   
   msg += "You should have received a copy of the GNU General\nPublic License along";
   msg += " with this program.  If not, write\nto the Free Software Foundation, Inc.,";
   msg += " 675 Mass Ave,\nCambridge, MA 02139, USA.";
   
   QMessageBox::about( this, "About KWebWatch", msg ); 
 
} 

void MainWidget::HelpContents()
{
 //  kapp->invokeHTMLHelp(QString("kwebwatch/kwebwatch.html"),QString()); 
   kapp->invokeHTMLHelp("", "");
}
