/*
*/

#include "../config.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "aftplib.h"
#include "common.h"
#include "misc.h"
#include "ahttplib.h"
#include "logger.h"
#include "version.h"

char host_name[512],file_to_get[512],protocol[10];
int max_retr=MAX_RETRIES;
int wait_time=WAIT_INTERVAL;

void show_version() {
  printf("got_it v%s Written by Andy Roger, 1997-98, (C)\n",VERSION);
  exit(1);
}

void usage() {
  printf("usage: got_it [OPTIONS] URL\n");
  printf("where options are:\n");
  
#ifdef HAVE_GETOPT_H
  printf(" -V,--version\t\tshow version and exit; do nothing\n");
  printf(" -t#,--timeout=#\ttimeout (sec)\n");
  printf(" -r#,--retry=#\t\tmax retry number\n");
  printf(" -w#,--wait=#\t\twait interval between retry (sec)\n");
  printf(" -l<name>,--log=<name>\tname of log file ('-' mean write to stdout)\n");
  printf(" -o<name>,--out=<name>\tname of output file ('-' mean write to stdout)\n");
  printf(" -d<dir>,--dir=<dir>\tchange to <dir> before retrieve any files\n");
  printf(" -a=<xxx>,--auth=<xxx>\tspecify auth information (name:password)\n");
  printf(" -u=<agent>,\t\tspecify user-agent field for HTTP request\n");
  printf(" --uagent=<agent>\n");
#else
  printf(" -V\t\tshow version and exit; do nothing\n");
  printf(" -t#\t\ttimeout (sec)\n");
  printf(" -r#\t\tmax retry number\n");
  printf(" -w#\t\twait interval between retry (sec)\n");
  printf(" -l<name>\tname of log file ('-' mean write to stdout)\n");
  printf(" -o<name>\tname of output file ('-' mean write to stdout)\n");
  printf(" -d<dir>\tchange to <dir> before retrieve any files\n");
  printf(" -a=<xxx>\tspecify auth information (name:password)\n");
  printf(" -u=<agent>\tspecify user-agent field for HTTP request\n");
#endif

  exit(1);
}

char *read_options(int argc, char **argv) {
#ifdef HAVE_GETOPT_H
  static struct option long_options[] = {
    { "version", 0, 0, 'V' },
    { "timeout", 1, 0, 't' },
    { "retry",   1, 0, 'r' },
    { "wait",    1, 0, 'w' },
    { "log",     1, 0, 'l' },
    { "out",     1, 0, 'o' },
    { "dir",     1, 0, 'd' },
    { "auth",    1, 0, 'a' },
    { "uagent",  1, 0, 'u' },
    { 0,         0, 0, 0   }
  };
  int option_index=0;
#endif
  int c;
  
  while (1) {
#ifdef HAVE_GETOPT_H
    c=getopt_long(argc,argv,"Vt:r:w:l:o:d:a:u:",long_options,
                  &option_index);
#else
    c=getopt(argc,argv,"Vt:r:w:l:o:d:a:u:");
#endif
    if (c==-1) break;
    switch (c) {
      case 'V': 
        show_version();
        break;
        
      case 't':
        ftp_timeout=ahttp_timeout=atoi(optarg);
        break;
        
      case 'r':
        max_retr=atoi(optarg);
        break;
        
      case 'w':
        wait_time=atoi(optarg);
        break;
        
      case 'l': /* log file name */
        setlogfilename(optarg);
        break;
        
      case 'o':
        strcpy(ftp_outfilename,optarg);
        strcpy(http_outfilename,optarg);
        break;
        
      case 'd':
        change_to_dir(optarg);
        break;
        
      case 'a':
        strcpy(ftp_auth,optarg);
        strcpy(http_auth,optarg);
        break;
        
      case 'u':
        strcpy(http_uagent,optarg);
        break;
    }
  }
  if (optind<argc)
    return(strdup(argv[optind]));
  else
    return(NULL);
}

void exit_routine() {
  restore_start_dir();
  ftp_remove_temp();
}

int main (int argc, char **argv) {
  int i,port=0;
  int retry_counter=1;
  char *url=NULL;
  char buf[512];
  int prot_ftp=0,prot_http=0;
  
  if (atexit(exit_routine)==-1) {
    perror("Can't register atexit function");
    exit(1);
  }
  
  ftp_outfilename[0]=http_outfilename[0]='\0';
  ftp_auth[0]=http_auth[0]=http_uagent[0]='\0';
  if (argc<2) usage();
  url=read_options(argc,argv);
  if (!url) {
    printf("URL must be specified.\n");
    exit(1);
  }
  if (!parse_url(url,protocol,host_name,&port,file_to_get)) {
    printf("can't parse '%s'\n",url);
    exit(1);
  }
  if ((strcmp(protocol,"ftp")!=0) && (strcmp(protocol,"http")!=0)) {
    printf("only ftp and http protocols implemented\n");
    exit(1);
  }
  
  openlogfile(NULL);
  logit(LC_INFO,"Program started");
  sprintf(buf,"Get '%s' from '%s', port %i",file_to_get,host_name,port);
  logit(LC_INFO,buf);
  sprintf(buf,"Timeout is %i, max retries is %i, wait interval is %i",
          ftp_timeout,max_retr,wait_time);
  logit(LC_INFO,buf);
  if (ftp_outfilename[0]!='\0') {
    if (strcmp(ftp_outfilename,"-")==0)
      sprintf(buf,"Output to STDOUT");
    else
      sprintf(buf,"Output file is %s",ftp_outfilename);
    logit(LC_INFO,buf);
  }
  if (ftp_auth[0]!='\0') logit(LC_INFO,"Auth info defined");
  if (http_uagent[0]!='\0') {
    sprintf(buf,"User-agent is %s",http_uagent);
    logit(LC_INFO,buf);
  }
  
  prot_ftp=strcmp(protocol,"ftp")==0;
  prot_http=strcmp(protocol,"http")==0;
  
  while (retry_counter<max_retr) {
    sprintf(buf,"Attempt %i",retry_counter);
    logit(LC_INFO,buf);
    if (prot_ftp) {
      if ((i=ftp_gotit(host_name,port,file_to_get))==FTP_OK) {
        logit(LC_INFO,"All done");
        closelogfile();
        exit(0);
      }
      if (i==FTP_FATAL) break;
    }
    if (prot_http) {
      if ((i=http_gotit(host_name,port,file_to_get))==HTTP_OK) {
        logit(LC_INFO,"All done");
        closelogfile();
        exit(0);
      }
      if (i==HTTP_FATAL) break;
    }  
    retry_counter++;
    logit(LC_INFO,"Waiting...");
    waiting(wait_time);
  }
  
  logit(LC_FATAL,"Can't do it, sorry...");
  closelogfile();
  exit(2);
}
