from urlparse import urlparse
from os       import path, utime, stat, mkdir
from stat     import ST_MTIME
from regex    import search
from time     import mktime, gmtime, strftime, localtime
from string   import atoi, strip
from types    import *

from version  import *
from httplib_new import HTTP_NEW

SCHEME = 'http'

# Return format : [MsgType, Msg, ...]
# with MsgType :
NE = -2
NW = -1
N0 = 1
N1 = 3
N2 = 5

###############################################################################
## Parse URL
def scheme_from_url(url):
	return urlparse(url, SCHEME)[0]

def host_from_url(url):
	return urlparse(url)[1]

## delete the leading / if absolute path
def path_from_url(url):
	u = urlparse(url)[2]
	if len(u)!=0 and u[0]=='/': return u[1:]
	return u

###############################################################################
## Check HTTP error code
def check_error_code(errcode, isHeadRequest, isDated):
	if errcode==200 or errcode==302: return 1
	if not isHeadRequest and isDated and errcode==304 : return 1
	return 0

###############################################################################
## Determine the type of a file
def determine_type(filePath):
	for a in ["\.html", "\.htm", "\.shtml"]:
		if search(a, filePath)==(len(filePath)-len(a)+1):
			return "text/html"
	return "text/plain"

###############################################################################
## Check if the filePath is in the restricted directory
def in_restricted_directory(filePath, resDir):
	if path.isabs(filePath) and find(filePath, resDir)!=0:
		return 'outside restricted directory'

###############################################################################
## Check for numeric argument
def check_integer(val, default):
	if type(val)!=IntType: return default
	return val

def check_bounded_integer(val, min, max, default):
	if type(val)!=IntType: return default
	if val<min or val>max: return default
	return val

## Check for string argument
## empty string is None
def check_string(val):
	if type(val)!=StringType or val=='': return None
	return val

###############################################################################
## Split completely a path
def splitPath(p):
	tmp = [p]
	spath = []
	while tmp[0]:
		tmp = path.split(tmp[0])
		spath.append(tmp[1])
	return spath

###############################################################################
## Return a path relative to a base path
def relativePath(pth, basePath):
	p = splitPath(pth)
	bp = splitPath(basePath)
	l = min(len(p), len(bp))
	ap = []
	while bp and p:
		pi = len(p)-1
		bpi = len(bp)-1
		if p[pi]!=bp[bpi]: break
		del p[pi]
		del bp[bpi]
	for e in bp:
		ap.append('..')	
	p = p + ap
	np = ''
	for e in p:
		np = e + '/' + np
	return np

###############################################################################
## The following methods use the Return class
###############################################################################

class Return:
	def __init__(self):
		self.msgType = None
		self.value = None

	def error(self): return self.msgType==NE

class ReturnValue(Return):
	def __init__(self, value):
		self.msgType = None
		self.value = value

class ReturnMessage(Return):
	def __init__(self, msgType, msg):
		self.msgType = msgType
		self.msg = msg
		self.value = None

class ReturnAll(Return):
	def __init__(self, msgType, msg, value):
		self.msgType = msgType
		self.msg = msg
		self.value = value

class ReturnError(ReturnMessage):
	def __init__(self, msg):
		ReturnMessage.__init__(self, NE, msg)

## Send HTTP request
def send_request(host, isHeadRequest, filePath, date):
	try:
		h = HTTP_NEW(host)
		if isHeadRequest: s = 'HEAD'
		else: s = 'GET'
		h.send('%s /%s HTTP/1.0\r\n' %(s, filePath))
		h.send('Host: %s\r\n' %host)
		h.send('User-Agent: %s %s\r\n' %(APP_NAME, APP_VERSION))
		if date: h.send('If-Modified-Since: %s\r\n' %date)
		h.send('Accept: */*\r\n')
		h.send('\r\n')
	except: return ReturnError('cannot join the host (%s)' %host)
	return ReturnValue(h)

## Check if the file is already there
def check_file(fp):
	try:
		if path.exists(fp) and path.isfile(fp):
			return ReturnValue(localtime( stat(fp)[ST_MTIME] ))
		else: return Return()
	except: return ReturnError('cannot get time for file %s' %fp)

## Create a directory (if it not already exists)
def create_directory(dir):
	spath = splitPath(dir)
	tmp = ''
	try:
		for i in range(len(spath)-1, -1, -1):
			tmp = tmp + spath[i] + '/'
			if path.isdir(tmp): continue
			mkdir(tmp, 493)
		return Return()
	except: return ReturnError('cannot create directory %s' %tmp)

## Save a file in the given path (relative to the current directory)
## Eventually create the directory
## Return 0 if it failed
def save_file(filePath, data, t):
	# Split the file path in [path, fileName]
	list = path.split(filePath)
	fileName = list[1]
	fp = list[0]
		
	# Sanity check
	if path.isabs(fp):
		return ReturnError('path is absolute (begins with a /) : %s' %fp)
	r = create_directory(fp)
	if r.error(): return r

	# Save file
	fp = path.join(fp, fileName)
	try:
		f = open(fp, 'w')
		f.write(data)
		f.close()
	except IOError, msg:
		return ReturnError('cannot save file %s (%s)' %(fp, msg))

	# Set time
	try:
		if t: utime(filePath, (t, t))
	except: return ReturnMessage(NE, 'cannot set time for file %s' %fp)
		
	return Return()

## Check if length returned in header is not over limit
def check_length(header, maxLength):
	s = header.getheader('Content-length')
	if not s: return ReturnError('no "Content-length" field in HTTP answer')
	try: length = atoi(s)
	except: return ReturnError('invalid length in field "Content-length"')
	if maxLength and length > maxLength: return ReturnValue([0, length])
	return ReturnValue([1, length])

## Check if URL is "ok"
def check_url(url, param, host, resDir):
	if not url: return Return()
	u = urlparse(url, SCHEME)
	if u[0]!=SCHEME: msg = 'not "http" scheme'
	elif u[1]!=host and u[1]!='': msg = 'different host'
	else:
		msg = None
		# need to be dependent on the data type ... after HEAD request
		if resDir and (param=='href' or param=='longdesc'):
			msg = in_restricted_directory(u[2], resDir)
	return ReturnValue([msg, u])

