###############################################################################
###############################################################################
##                          mainwin.tcl
###############################################################################
###############################################################################
## Defines the main window of Getleft, the menus and such
###############################################################################
###############################################################################
## (c) 2001-2002 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the GPL v2
###############################################################################
###############################################################################

namespace eval iconNS {

################################################################################
# DefineIconImages
#     Prepares the images for the icons in the taskbar so that Tcl can use them
################################################################################
proc DefineIconImages {} {
    global dirGetleft
    variable iconImages

    set iconNS::iconImages(url)     [image create photo \
            -file [file join "$dirGetleft(images)" url.gif]]
    set iconNS::iconImages(start)   [image create photo \
            -file [file join "$dirGetleft(images)" start.gif]]
    set iconNS::iconImages(auto)    [image create photo \
            -file [file join "$dirGetleft(images)" auto.gif]]
    set iconNS::iconImages(pause)   [image create photo \
            -file [file join "$dirGetleft(images)" pause.gif]]
    set iconNS::iconImages(stop)    [image create photo \
            -file [file join "$dirGetleft(images)" stop.gif]]
    set iconNS::iconImages(conf)    [image create photo \
            -file [file join "$dirGetleft(images)" conf.gif]]
    set iconNS::iconImages(delay)   [image create photo \
            -file [file join "$dirGetleft(images)" delay.gif]]
    set iconNS::iconImages(log)     [image create photo \
            -file [file join "$dirGetleft(images)" log.gif]]
    set iconNS::iconImages(fileLog) [image create photo \
            -file [file join "$dirGetleft(images)" fileLog.gif]]
    set iconNS::iconImages(get)     [image create photo \
            -file [file join "$dirGetleft(images)" get.gif]]
    set iconNS::iconImages(help)    [image create photo \
            -file [file join "$dirGetleft(images)" help.gif]]

    set iconNS::iconImages(update)  [image create photo \
            -file [file join "$dirGetleft(images)" update.gif]]
    set iconNS::iconImages(queue)   [image create photo \
            -file [file join "$dirGetleft(images)" queue.gif]]
    set iconNS::iconImages(exit)    [image create photo \
            -file [file join "$dirGetleft(images)" exit.gif]]

    set iconNS::iconImages(clear)   [image create photo \
            -file [file join "$dirGetleft(images)" clear.gif]]

    return
}
    DefineIconImages
}

namespace eval mainWin {

################################################################################
# ActivateDownloadButtons
#    When invoked activates or disables the taskbar buttons depending on
#    whether we are downloading or not
################################################################################
proc ActivateDownloadButtons {} {
    variable listbox
    variable taskbar
    global   getleftState

    $mainWin::taskbar(start) configure -state disabled
    $mainWin::taskbar(auto)  configure -state disabled

    if {[$listbox size]} {
        if {$getleftState(downloading)==0} {
            set anchor [$listbox curselection]
            if {$anchor!=""} {
                $mainWin::taskbar(start) configure -state normal
            }
        }
        $mainWin::taskbar(auto)  configure -state normal
    }

    return
}

################################################################################
# ActivateTaskbarButtons
#    When invoked activates or disables the taskbar buttons depending on
#    whether we are downloading or not
################################################################################
proc ActivateTaskbarButtons {} {
    global getleftState labelDialogs labelMessages dirGetleft
    variable ctrFrameLabel
    variable tmpCtrFrameLabel
    variable taskbar

    if {$getleftState(downloading)==0} {
        $taskbar(pause) configure -state disabled
        $taskbar(stop)  configure -state disabled
        set ctrFrameLabel    $labelDialogs(ready)
        set tmpCtrFrameLabel $labelDialogs(ready)
    } else {
        $taskbar(pause) configure -state normal
        $taskbar(stop)  configure -state normal
        set ctrFrameLabel    "$labelMessages(downloading): $getleftState(url)"
        set tmpCtrFrameLabel "$labelMessages(downloading): $getleftState(url)"
    }

    ActivateDownloadButtons

    return
}

################################################################################
# DefineTaskbarButtons
#    Defines the buttons in the taskbar.
#
# Parameters
#    The frames for the buttons.
################################################################################
proc DefineTaskbarButtons {left right} {
    global   labelDialogs labelMenus getleftState
    variable taskbar

    set taskbar(url)   [button $left.url   -image $iconNS::iconImages(url)   \
            -relief flat -bd 1 -command EnterUrl::EnterUrl]
    ::BalloonHelp::set_balloon $taskbar(url) $labelDialogs(enterUrl)

    set taskbar(start) [button $left.start -image $iconNS::iconImages(start) \
            -relief flat -bd 1                                               \
            -command  {AutomaticDownload [mainWin::UrlListSelectedUrl]}]
    ::BalloonHelp::set_balloon $taskbar(start) $labelDialogs(start)

    set taskbar(auto) [button $left.auto -image $iconNS::iconImages(auto) \
            -relief flat -bd 1 -command mainWin::SetAutoDownload]
    ::BalloonHelp::set_balloon $taskbar(auto) $labelDialogs(auto)

    set taskbar(pause) [menubutton $left.pause -image $iconNS::iconImages(pause) \
            -bd 1 -menu $left.pause.menu]
    ::BalloonHelp::set_balloon $taskbar(pause) $labelDialogs(pause)
    set menus(pause) [menu $taskbar(pause).menu -tearoff 0 -fg black -postcommand "
            BalloonHelp::kill_balloon
    "]

    set taskbar(stop)  [menubutton $left.stop  -image $iconNS::iconImages(stop)  \
            -bd 1 -menu $left.stop.menu]
    ::BalloonHelp::set_balloon $taskbar(stop) $labelDialogs(stop)
    set menus(stop) [menu $taskbar(stop).menu -tearoff 0 -fg black -postcommand "
        event generate $taskbar(stop) <Leave>
    "]

    set taskbar(conf)  [button $left.conf  -image $iconNS::iconImages(conf)  \
            -relief flat -bd 1                                               \
            -command "::mainWin::UrlListChangeOptions"]
    ::BalloonHelp::set_balloon $taskbar(conf) $labelDialogs(options)

    set taskbar(delay) [button $left.delay -image $iconNS::iconImages(delay) \
            -relief flat -bd 1                                               \
            -command "::Delay::DelayedDownload"]
    if {$getleftState(delaySet)==1} {
        $taskbar(delay) configure -relief sunken
    }    
    ::BalloonHelp::set_balloon $taskbar(delay) $labelDialogs(progDelay)

    set taskbar(log)   [button $left.log   -image $iconNS::iconImages(log)  \
            -relief flat -bd 1                                              \
            -command "::getLog::GetLog"]
    ::BalloonHelp::set_balloon $taskbar(log) $labelDialogs(getLog)

    set taskbar(fileLog) [button $right.fileLog -image $iconNS::iconImages(fileLog) \
            -relief flat -bd 1 -command ::getLog::FileLog]
    ::BalloonHelp::set_balloon $taskbar(fileLog) $labelDialogs(fileLog)
    set taskbar(help)  [button $right.conf -image $iconNS::iconImages(help)  \
            -relief flat -bd 1 -command Ayuda::Manual]
    ::BalloonHelp::set_balloon $taskbar(help) $labelDialogs(help)

    set taskbar(get)   [button $right.get  -image $iconNS::iconImages(get)   \
            -relief flat -bd 1 -command Ayuda::About]
    ::BalloonHelp::set_balloon $taskbar(get) $labelDialogs(about)

    foreach {name path} [array get taskbar] {
        bindtags $path "[bindtags $path] taskIcon"
    }
    bind taskIcon <Enter> {
        %W configure -relief raised
        set mainWin::tmpCtrFrameLabel $mainWin::ctrFrameLabel
        set mainWin::ctrFrameLabel $BalloonHelp::Bulle(%W)
    }
    bind taskIcon <Leave> {
        if {(("%W"==$mainWin::taskbar(auto))&&($::getleftState(autoDown)==1))
                ||(("%W"==$mainWin::taskbar(delay))&&($::getleftState(delaySet)==1))} {
            %W configure -relief sunken
        } else {
            %W configure -relief flat
        }        
        set mainWin::ctrFrameLabel $mainWin::tmpCtrFrameLabel
    }
    foreach {name path} [array get menus] {
        bind $path <FocusOut> "
            $taskbar($name) configure -relief flat
            BalloonHelp::kill_balloon
        "
    }

    $menus(pause) add check -variable getleftOptions(pausePage)              \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopPage)
    $menus(pause) add check -variable getleftOptions(pauseFile)              \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopFile)

    $menus(stop) add check -variable getleftOptions(stopPage)                \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopPage)              \
            -command {
                if {$getleftState(autoDown)==1}    mainWin::SetAutoDownload
                if {$getleftState(delayedDown)==1} Delay::DelayedCancelDelay
            }
    $menus(stop) add check -variable getleftOptions(stopFile)                \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopFile)              \
            -command {
                if {$getleftState(autoDown)==1}    mainWin::SetAutoDownload
                if {$getleftState(delayedDown)==1} Delay::DelayedCancelDelay
            }
    $menus(stop) add command -label $labelMenus(stopNow) -command {
        if {$getleftState(delayedDown)==1} Delay::DelayedCancelDelay
        ::Ventana::CancelDownloading .
    }
    return
}

################################################################################
# NextToAutoDownload
#    When invoked this procedure will start downloading the next site in the
#    queue, the point here is that urls entered to get the site map are only
#    downloaded up to where the map is done and then left in the queue.
################################################################################
proc NextToAutoDownload {} {
    global getleftState urlsToDownload dirGetleft

    set found 0
    foreach url $getleftState(urlQueue) {
        array set optionsTemp $urlsToDownload($url,options)
        if {$optionsTemp(map)==0} {
            set found 1
            break    
        }
        set domain [lindex [HtmlParser::ParseUrl $url] 1]
        set domain [string tolower $domain]
        set fileName [file join $dirGetleft(conf) $domain.gdt]
        if {[catch {open "$fileName" r} handle]} {
            set found 1
            break
        }
        for {} {![eof $handle]} {} {
            set line [gets $handle]
            if {[regexp {\(mapLinks\)} "$line"]} {
                set found 1
                break
            }
        }
        close $handle
        if {$found} break
    }
    if {$found==1} {
        AutomaticDownload $url
    } else {
        SetAutoDownload
    }

    return
}

################################################################################
# SetAutoDownload
#   When the button in the taskbar is clicked, this procedure enables or
#   disables full atomatic download.
################################################################################
proc SetAutoDownload {} {
    global getleftState
    variable taskbar

    if {$getleftState(autoDown)==0} {
        set getleftState(autoDown) 1
        $taskbar(auto) configure -relief sunken
        if {($getleftState(downloading)==0)\
                &&([llength $getleftState(urlQueue)]>0)} {
            NextToAutoDownload
        }
    } else {
        set getleftState(autoDown) 0
        $taskbar(auto) configure -relief flat
    }

    return
}

################################################################################
# UrlListUrlIndex
#   Given an Url this procedure returns its index in the url list.
#
# Parameters
#   url: The url to search for.
#
# Returns
#   The index of the url, or -1 if the is none to return
################################################################################
proc UrlListUrlIndex {url} {
    variable listbox

    set rowNumber [$listbox size]
    for {set i 0} {$i<$rowNumber} {incr i} {
        set site [$listbox cellcget $i,0 -text]
        if {[string match $url $site]} {
            return $i
        }
    }
    return -1
}

################################################################################
# UrlListIncrResume
#   Given an Url this procedure increments the 'Resumed' counter in the
#   main window.
#
# Parameters
#   url: The url that is going to be resumed.
################################################################################
proc UrlListIncrResume {url} {
    variable listbox

    set index [UrlListUrlIndex $url]
    set i [$listbox cellcget $index,1 -text]
    $listbox cellconfigure $index,1 -text [incr i]

    return
}

################################################################################
# UrlListDeleteEntry
#   Deletes one of the url from the url list.
#
# Parameters
#   url: The url of the entry to delete.
################################################################################
proc UrlListDeleteEntry {url} {
    variable listbox
    global getleftState urlsToDownload dirGetleft

    set index [UrlListUrlIndex $url]
    if {$index==-1} return
    $listbox delete $index

    set index [lsearch $getleftState(urlQueue) $url]
    if {$index==-1} return
    set getleftState(urlQueue)                                             \
            [lreplace $getleftState(urlQueue) $index $index]

    getLog::UrlLogEnter $url $urlsToDownload($url,dir)                     \
        $urlsToDownload($url,options)

    catch {unset urlsToDownload($url)}
    catch {unset urlsToDownload($url,dir)}
    catch {unset urlsToDownload($url,resume)}
    catch {unset urlsToDownload($url,options)}

    set domain [string tolower [lindex [HtmlParser::ParseUrl $url] 1]]
    catch {file delete [file join $dirGetleft(conf) $domain.gdt]}

    SetResumeSitesMenu
    ActivateDownloadButtons

    return
}

################################################################################
# UrlListMoveEntry
#   Moves one of the entry one row up or down.
#
# Parameters
#   url: The url of the entry to delete.
#   tableIndex: The index of the url in the tablelist.
#   how: '-1' up, '1' down.
################################################################################
proc UrlListMoveEntry {url tableIndex how} {
    variable listbox
    global getleftState urlsToDownload

    set urlIndex [lsearch $getleftState(urlQueue) $url]
    if {$how==-1} {
        if {$tableIndex==0} {
            return
        }
    } else {
        set rowLimit [expr [$listbox size] - 1]
        if {$rowLimit==$tableIndex} {
            return
        }
    }
    $listbox delete $tableIndex
    set tableIndex [expr $tableIndex + $how]
    $listbox insert $tableIndex [list                            \
            $url $urlsToDownload($url,resume)                    \
            [file nativename $urlsToDownload($url,dir)]]
    set getleftState(urlQueue) [lreplace $getleftState(urlQueue) \
            $urlIndex $urlIndex]
    set urlIndex [expr $urlIndex + $how]
    set getleftState(urlQueue) [linsert $getleftState(urlQueue)  \
            $urlIndex $url]

    $listbox selection anchor $tableIndex

    return
}

################################################################################
# UrlListMoveTopBottom
#   Moves one of the urls to the top or the bottom of the list.
#
# Parameters
#   url: The url of the entry to delete.
#   where: Either 'top' or 'bottom', defaults to 'bottom'
################################################################################
proc UrlListMoveTopBottom {url {where bottom}} {
    variable listbox
    global getleftState urlsToDownload

    set urlIndex [lsearch $getleftState(urlQueue) $url]
    set tableIndex [UrlListUrlIndex $url]

    if {$where=="bottom"} {
        set where end
    } else {
        set where 0
    }

    set rowLimit [expr [$listbox size] - 1]
    if {($rowLimit==$tableIndex)&&($where=="end")} {
        return
    }
    if {($tableIndex==0)&&($where==0)} {
        return
    }

    $listbox delete $tableIndex
    $listbox insert $where [list                                 \
            $url $urlsToDownload($url,resume)                    \
            [file nativename $urlsToDownload($url,dir)]]

    set getleftState(urlQueue) [lreplace $getleftState(urlQueue) \
            $urlIndex $urlIndex]
    set getleftState(urlQueue) [linsert $getleftState(urlQueue)  \
            end $url]

    $listbox selection anchor $tableIndex

    return
}

################################################################################
# UrlListEnterEntry
#   Enters one site into the url list.
#
# Parameters
#   url: The url of the entry to enter.
#   resume: How many times it has been resumed so far.
#   dir: The directory where it is going to be saved.
################################################################################
proc UrlListEnterEntry {url resume dir} {
    variable listbox

    $listbox insert end [list $url $resume [file nativename $dir]]

    $listbox yview moveto 1

    ActivateDownloadButtons

    return
}

################################################################################
# UrlListSelectedUrl
#   Returns the url of the selected entry in the list.
################################################################################
proc UrlListSelectedUrl {} {
    variable listbox

    set anchor [$listbox curselection]
    if {[catch {$listbox cellcget $anchor,0 -text} url]} {
        return
    }

    return $url
}

###############################################################################
# UpdateUrlOptions
#    During a download, if the users changes the options by using the menus,
#    procedure is invoked to make sure, the new chosen options are saved for
#    the site.
###############################################################################
proc UpdateUrlOptions {} {
    global getleftState urlsToDownload downOptions

    if {$getleftState(downloading)==0} return

    set urlsToDownload($getleftState(url),options) [array get downOptions]

    return
}

################################################################################
# UrlListChangeOptions
#   Change the options for downloading an URL in the urlList.
#
# Parameters
#   url: The site we are talking about, defaults to "", which happens when
#        it is invoked from the main menu of the application.
################################################################################
proc UrlListChangeOptions {{url ""}} {
    global urlsToDownload getleftState

    if {$url==""} {
        ::Opciones::ChooseOptions ::downOptions
        UpdateUrlOptions
        return
    }

    array set ::downOptionsTemp $urlsToDownload($url,options)
    set done [::Opciones::ChooseOptions ::downOptionsTemp]
    if {$done==1} {
        set urlsToDownload($url,options) [array get ::downOptionsTemp]
        if {($getleftState(downloading)==1) \
                &&([string match $getleftState(url) $url])} {
            array set ::downOptions $urlsToDownload($url,options)
        }
    }
    return
}

################################################################################
# UrlListGetIndex
#   When the user clicks in the url list, this procedure takes care of selecting
#   the nearest entry and returns its index.
#
# Parameters
#   win: window where you clicked.
#   y: the coordinate where you did it
#
# Returns
#   The index of the url, or -1 if the is none to return
################################################################################
proc UrlListGetIndex {win y} {
    
    set ::tablelist::win [winfo parent $win]
    set ::tablelist::y [expr "$y + [winfo y $win]"]
    if {[winfo exists $::tablelist::win]} {
        set index [$::tablelist::win nearest $::tablelist::y]
	    if {$index!=-1} {
            if {[catch {tkListboxBeginSelect $::tablelist::win $index}]} {
                tk::ListboxBeginSelect $::tablelist::win $index
            }
	    }
	    return $index
    }
    return -1
}

################################################################################
# UrlListRightClick
#   This procedure is invoked when the user right-clicks in the multicolumn
#   list with the sites to download, it makes a context menu pop up with the
#   appropiate options
#
# Parameters
#   win: window where you right-clicked.
#   y,x: the coordinates where you did it for the tablelist widget.
#   X,Y: and for the pop up menu.
################################################################################
proc UrlListRightClick {win x y X Y} {
    variable listbox
    variable menus
    global getleftState urlsToDownload

    set index [UrlListGetIndex $win $y]
    if {$index==-1} return

    ActivateDownloadButtons

    $menus(rightClick) entryconfigure 0             \
            -command "AutomaticDownload  [mainWin::UrlListSelectedUrl]"
    $menus(rightClick) entryconfigure 2             \
            -command [list mainWin::UrlListChangeOptions [mainWin::UrlListSelectedUrl]]
    $menus(rightClick) entryconfigure 4             \
            -command "
                mainWin::ActivateDownloadButtons
                clipboard clear
                clipboard append [mainWin::UrlListSelectedUrl]
            "
    $menus(rightClick) entryconfigure 6             \
            -command "mainWin::UrlListMoveTopBottom [mainWin::UrlListSelectedUrl] top"
    $menus(rightClick) entryconfigure 7             \
            -command "mainWin::UrlListMoveEntry [mainWin::UrlListSelectedUrl] $index -1"
    $menus(rightClick) entryconfigure 8             \
            -command "mainWin::UrlListMoveEntry [mainWin::UrlListSelectedUrl] $index  1"
    $menus(rightClick) entryconfigure 9             \
            -command "mainWin::UrlListMoveTopBottom [mainWin::UrlListSelectedUrl] bottom"
    $menus(rightClick) entryconfigure 11            \
            -command "mainWin::UrlListDeleteEntry [mainWin::UrlListSelectedUrl]"

    if {$getleftState(downloading)==0} {
        $menus(rightClick) entryconfigure 0 -state normal
        $menus(rightClick) entryconfigure 9 -state normal
    } else {
        $menus(rightClick) entryconfigure 0 -state disable
	    set url [$listbox cellcget $index,0 -text]
        if {[string match $url $getleftState(url)]} {
            $menus(rightClick) entryconfigure 9 -state disable
        } else {
            $menus(rightClick) entryconfigure 9 -state normal
        }
    }
    tk_popup $menus(rightClick) $X $Y

    return
}

################################################################################
# UrlListDoubleClick
#   This procedure is invoked when the user double-clicks in the multicolumn
#   list with the sites to download, it will try to start downloading the site.
#
# Parameters
#   win: window where you double-clicked.
#   y: the coordinate where you did it
################################################################################
proc UrlListDoubleClick {win y} {
    variable listbox
    global getleftState

    if {$getleftState(downloading)!=0} return

    set index [UrlListGetIndex $win $y]
    if {$index==-1} return

    set url [$listbox cellcget $index,0 -text]

    AutomaticDownload $url

    return
}

###############################################################################
# SetResumeSitesMenu
#    Puts the entries in the "Resume Sites" menu.
###############################################################################
proc SetResumeSitesMenu {} {
    global getleftState

    .menus.fichero.resume delete 0 end

    foreach site [lsort $getleftState(urlQueue)] {
        .menus.fichero.resume add command -label $site \
                -command "AutomaticDownload $site"
    }
    return
}

###############################################################################
# MenusLabels
#    Puts the labels into the menus.
#
# Parameters:
#    labFile: file with the labels in the, hopefully, desired language
###############################################################################
proc MenusLabels {{labFile ""}} {
    global labelButtons labelTitles labelMessages labelDialogs labelMonths
    global labelMenus indexMenus dirGetleft
    variable menus

    if {$labFile==""} {
        set labFile en
    }

    ReadLangFile $labFile

    .menus      entryconfigure 1 -label $labelMenus(file)           \
            -underline $indexMenus(file)
    $menus(1)   entryconfigure 0 -label $labelMenus(enterUrl)...    \
            -underline $indexMenus(enterUrl)
    $menus(1)   entryconfigure 1 -label $labelMenus(siteMap)        \
            -underline $indexMenus(siteMap)
    $menus(1)   entryconfigure 2 -label $labelMenus(resumeSite)     \
            -underline $indexMenus(resumeSite)
    $menus(1)   entryconfigure 4 -label $labelMenus(stop)           \
            -underline $indexMenus(stop)
    $menus(1,4) entryconfigure 0 -label $labelMenus(stopPage)       \
            -underline $indexMenus(stopPage)
    $menus(1,4) entryconfigure 1 -label $labelMenus(stopFile)       \
            -underline $indexMenus(stopFile)
    $menus(1,4) entryconfigure 2 -label $labelMenus(stopNow)        \
            -underline $indexMenus(stopNow)
    $menus(1)   entryconfigure 5 -label $labelMenus(pause)          \
            -underline $indexMenus(pause)
    $menus(1,5) entryconfigure 0 -label $labelMenus(stopPage)       \
            -underline $indexMenus(stopPage)
    $menus(1,5) entryconfigure 1 -label $labelMenus(stopFile)       \
            -underline $indexMenus(stopFile)
    $menus(1)   entryconfigure 7 -label $labelMenus(exit)           \
            -underline $indexMenus(exit)

    .menus      entryconfigure 2 -label $labelMenus(options)        \
            -underline $indexMenus(options)
    $menus(2)   entryconfigure 0 -label $labelMenus(all)            \
            -underline $indexMenus(all)
    $menus(2)   entryconfigure 1 -label $labelMenus(uplinks)        \
            -underline $indexMenus(uplinks)
    $menus(2,0) entryconfigure 0 -label $labelMenus(linksFollow)    \
            -underline $indexMenus(linksFollow)
    $menus(2,0) entryconfigure 1 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 2 -label $labelMenus(levels)         \
            -underline $indexMenus(levels)
    $menus(2,0) entryconfigure 2 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2,1) entryconfigure 0 -label $labelMenus(noLimit)        \
            -underline $indexMenus(noLimit)
    $menus(2)   entryconfigure 3 -label $labelMenus(external)       \
            -underline $indexMenus(external)
    $menus(2,2) entryconfigure 0 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 4 -label $labelMenus(fileFilter)     \
            -underline $indexMenus(fileFilter)
    $menus(2,3) entryconfigure 0 -label $labelMenus(onlyHtml)       \
            -underline $indexMenus(onlyHtml)
    $menus(2,3) entryconfigure 1 -label $labelMenus(images)         \
            -underline $indexMenus(images)
    $menus(2,3,1) entryconfigure 0 -label $labelMenus(allImages)    \
            -underline $indexMenus(allImages)
    $menus(2,3,1) entryconfigure 1 -label $labelMenus(onlyThumb)    \
            -underline $indexMenus(onlyThumb)
    $menus(2,3,1) entryconfigure 2 -label $labelMenus(noThumb)      \
            -underline $indexMenus(noThumb)
    $menus(2,3) entryconfigure 2 -label $labelMenus(chooseFilter)   \
            -underline $indexMenus(chooseFilter)
    $menus(2)   entryconfigure 5 -label $labelMenus(resume)         \
            -underline $indexMenus(resume)
    $menus(2)   entryconfigure 6 -label $labelMenus(update)         \
            -underline $indexMenus(update)
    $menus(2)   entryconfigure 7 -label $labelMenus(cgi)            \
            -underline $indexMenus(cgi)
    $menus(2)   entryconfigure 8 -label $labelMenus(useProxy)       \
            -underline $indexMenus(useProxy)

    .menus      entryconfigure 3 -label $labelMenus(tools)          \
            -underline $indexMenus(tools)
    $menus(3)   entryconfigure 0 -label $labelMenus(purgeFiles)     \
            -underline $indexMenus(purgeFiles)
    $menus(3)   entryconfigure 1 -label $labelMenus(restoreOrig)    \
            -underline $indexMenus(restoreOrig)
    $menus(3)   entryconfigure 2 -label $labelMenus(configureProxy) \
            -underline $indexMenus(configureProxy)
    $menus(3)   entryconfigure 3 -label $labelMenus(getLog)         \
            -underline $indexMenus(getLog)
    $menus(3)   entryconfigure 4 -label $labelMenus(languages)      \
            -underline $indexMenus(languages)
    if {$::DEBUG==1} {
        $menus(3)   entryconfigure 5 -label $labelMenus(resource)   \
                -underline $indexMenus(resource)
    }

    .menus      entryconfigure 4 -label $labelMenus(help)           \
            -underline $indexMenus(help)
    $menus(4)   entryconfigure 0 -label $labelMenus(manual)         \
            -underline $indexMenus(manual)
    $menus(4)   entryconfigure 2 -label $labelMenus(changes)        \
            -underline $indexMenus(changes)
    $menus(4)   entryconfigure 3 -label $labelMenus(license)        \
            -underline $indexMenus(license)
    $menus(4)   entryconfigure 5 -label $labelMenus(about)          \
            -underline $indexMenus(about)

    return
}

###############################################################################
# ShowMenu1
#    Before posting the file menu, this procedure decides which entries
#    should be active and which will have to wait
###############################################################################
proc ShowMenu1 {} {
    global getleftState getleftOptions

    if {$getleftState(downloading)==0} {
        .menus.fichero entryconfigure 2 -state normal
        .menus.fichero entryconfigure 4 -state disabled
        .menus.fichero entryconfigure 5 -state disabled
        if {[llength $getleftState(urlQueue)]==0} {
            .menus.fichero entryconfigure 2 -state disabled
        }
    } else {
        .menus.fichero entryconfigure 2 -state disabled
        .menus.fichero entryconfigure 4 -state normal
        .menus.fichero entryconfigure 5 -state normal
    }

    return
}

###############################################################################
# CreateMenus
#   Creates the menus, without putting the labels
###############################################################################
proc CreateMenus {} {
    global getleftOptions dirGetleft supportedLang
    variable menus

    menu .menus -relief flat
    . config -menu .menus

    set menus(1) [menu .menus.fichero      -tearoff 0 \
            -postcommand mainWin::ShowMenu1]
    .menus add cascade -menu $menus(1)
    set menus(2) [menu .menus.editar       -tearoff 0]
    .menus add cascade -menu $menus(2)
    set menus(3) [menu .menus.herramientas -tearoff 0]
    .menus add cascade -menu $menus(3)
    set menus(4) [menu .menus.help         -tearoff 0]
    .menus add cascade -menu $menus(4)

    $menus(1) add command -command  EnterUrl::EnterUrl
    $menus(1) add command -command "EnterUrl::EnterUrl 1"
    $menus(1) add cascade -menu $menus(1).resume
    $menus(1) add separator
    $menus(1) add cascade -menu $menus(1).stop
    $menus(1) add cascade -menu $menus(1).pause
    $menus(1) add separator
    $menus(1) add command -command ExitGetleft

    set menus(1,2) [menu $menus(1).resume -tearoff 0]
    SetResumeSitesMenu

    set menus(1,4) [menu $menus(1).stop   -tearoff 0]
    $menus(1,4) add check -variable getleftOptions(stopPage)  \
            -onvalue 1 -offvalue 0 -command {
                if {$getleftState(autoDown)==1} mainWin::SetAutoDownload
            }
    $menus(1,4) add check -variable getleftOptions(stopFile)  \
            -onvalue 1 -offvalue 0 -command {
                if {$getleftState(autoDown)==1} mainWin::SetAutoDownload
            }
    $menus(1,4) add command -command {
        if {$getleftState(autoDown)==1} mainWin::SetAutoDownload
        ::Ventana::CancelDownloading .
    }

    set menus(1,5) [menu $menus(1).pause -tearoff 0]
    $menus(1,5) add check -variable getleftOptions(pausePage) \
            -onvalue 1 -offvalue 0
    $menus(1,5) add check -variable getleftOptions(pauseFile) \
            -onvalue 1 -offvalue 0

    $menus(2) add command -command mainWin::UrlListChangeOptions
    $menus(2) add cascade -menu $menus(2).directorios
    $menus(2) add cascade -menu $menus(2).levels
    $menus(2) add cascade -menu $menus(2).exLevels
    $menus(2) add cascade -menu $menus(2).filters
    $menus(2) add check   -variable downOptions(resume)   -onvalue 1 -offvalue 0 
    $menus(2) add check   -variable downOptions(update)   -onvalue 1 -offvalue 0 \
        -command {
            mainWin::UpdateUrlOptions
            catch {unset urlsDownloaded}
        }
    $menus(2) add check   -variable downOptions(cgi)      -onvalue 1 -offvalue 0 \
        -command {
            mainWin::UpdateUrlOptions
            catch {unset urlsDownloaded}
        }
    $menus(2) add check   -variable getleftOptions(proxy) -onvalue 1 -offvalue 0

    set menus(2,0) [menu $menus(2).directorios -tearoff 0]
    $menus(2,0) add radio -variable downOptions(dir) -value 1 \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    $menus(2,0) add radio -variable downOptions(dir) -value 0 \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }

    set menus(2,1) [menu $menus(2).levels -tearoff 0]
    $menus(2,1) add radio -variable downOptions(levels) -value -1
    for {set i 0} {$i<6} {incr i} {
        $menus(2,1) add radio -label $i -variable downOptions(levels) \
                -value $i -command {
                              mainWin::UpdateUrlOptions
                              catch {unset urlsDownloaded}
                          }
    }

    set menus(2,2) [menu $menus(2).exLevels -tearoff 0]
    $menus(2,2) add radio -variable downOptions(exLevels) -value 0    \
            -command {
                set downOptions(external) 0
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    for {set i 1} {$i<4} {incr i} {
        $menus(2,2) add radio -label $i -variable downOptions(exLevels) \
            -value $i -command {
                set downOptions(external) 1
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    }

    set menus(2,3) [menu $menus(2).filters -tearoff 0]
    $menus(2,3) add check -variable downOptions(onlyHtml) \
                -onvalue 1 -offvalue 0 -command {
                    mainWin::UpdateUrlOptions
                    catch {unset urlsDownloaded}
                }
    $menus(2,3) add cascade -menu $menus(2,3).images
    $menus(2,3) add command -command {
                Herramientas::FilterFiles $downOptions(filter)
                mainWin::UpdateUrlOptions
            }
    set menus(2,3,1) [menu $menus(2,3).images -tearoff 0]
    $menus(2,3,1) add radio -variable downOptions(images) -value 0 \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    $menus(2,3,1) add radio -variable downOptions(images) -value 1 \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    $menus(2,3,1) add radio -variable downOptions(images) -value 2 \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }

    $menus(3) add command -command Herramientas::PurgeFiles
    $menus(3) add command -command Herramientas::RestoreOriginals
    $menus(3) add command -command Herramientas::ConfProxy
    $menus(3) add command -command getLog::FileLog
    $menus(3) add cascade -menu $menus(3).idiomas

    if {$::DEBUG==1} {
        $menus(3) add command -command \
                "source [file join "$dirGetleft(main)" Getleft.tcl]"
    }

    set menus(3,0) [menu $menus(3).idiomas -tearoff 0]
    set i 0
    foreach lang $supportedLang(langList) {
        if {[file exists [file join $dirGetleft(languages) \
                menus.$supportedLang($lang)]]} {
            $menus(3,0) add command \
                    -command "ChangeLanguage $supportedLang($lang)"
            $menus(3,0) entryconfigure $i -label $lang
            incr i
        }
    }

    $menus(4) add command -command Ayuda::Manual
    $menus(4) add separator
    $menus(4) add command -command Ayuda::Changes
    $menus(4) add command -command Ayuda::Licence
    $menus(4) add separator
    $menus(4) add command -command Ayuda::About

    MenusLabels $getleftOptions(lang)

    return
}

###############################################################################
# MainWindow
#    Defines how the main window of Getleft looks.
###############################################################################
proc MainWindow {} {
    global   getleftState getleftOptions urlsToDownload labelMenus labelDialogs
    variable ctrFrameLabel
    variable listbox
    variable menus
    variable taskbar
    variable scrollBars

    wm title . "Getleft v1.0.5"
    wm minsize . 300 165

    if {[winfo exists .menus]} {
        destroy .menus
        destroy .menu
        destroy .extFrame
    }
    CreateMenus

    set extFrame  [frame .extFrame]
    set iconFrame [frame $extFrame.iconFrame]
    set listFrame [frame $extFrame.listFrame -bd 2 -relief sunken]
    set ctrFrame  [frame $extFrame.ctrFrame]

    set leftIconFrame  [frame $iconFrame.left ]
    set rightIconFrame [frame $iconFrame.right]
    DefineTaskbarButtons $leftIconFrame $rightIconFrame

    set scrollBars(y) [scrollbar $listFrame.scrolly -orient vertical       \
            -command  [list $listFrame.listbox yview]]
    set scrollBars(x) [scrollbar $listFrame.scrollx -orient horizontal     \
            -command  [list $listFrame.listbox xview]]

    set listbox [::tablelist::tablelist $listFrame.listbox                 \
            -bd 0 -height 10 -width 90 -stretch 3                          \
            -fg $getleftOptions(fg) -bg $getleftOptions(bg)                \
            -columns [list 44 $labelDialogs(url)      left                 \
                            0 $labelDialogs(resumed)  center               \
                           30 $labelDialogs(localDir) left]                \
            -xscrollcommand "$scrollBars(x) set"                           \
            -yscrollcommand "$scrollBars(y) set"]

    set intCtrFrame [frame $ctrFrame.int -bd 2 -relief sunken]
    set ctrLabel [label $intCtrFrame.label -textvariable mainWin::ctrFrameLabel]

    set backColor [$intCtrFrame cget -background]
    $ctrLabel configure -bg      $backColor
    $listbox  configure -labelbg $backColor

    grid $extFrame       -row 0 -sticky news
    grid $iconFrame             -sticky ew -padx 5
    grid $rightIconFrame -row 0 -sticky e
    grid $leftIconFrame  -row 0 -sticky w
    grid $taskbar(url)    $taskbar(start) $taskbar(auto)  $taskbar(pause) \
         $taskbar(stop)   $taskbar(conf)  $taskbar(delay) $taskbar(log)   \
         -sticky w
    grid $taskbar(fileLog) $taskbar(help)  $taskbar(get)  -sticky e
    grid columnconfigure $iconFrame 0 -weight 1

    grid $listFrame -sticky news -pady 2
    grid $scrollBars(y) -in $listFrame -row 0 -column 1 -sticky ns
    grid $scrollBars(x) -in $listFrame -row 1 -column 0 -sticky ew
    grid $listbox       -in $listFrame -row 0 -column 0 -sticky news     \
            -padx 0 -pady 0

    grid rowconfigure    .          0 -weight 1
    grid columnconfigure .          0 -weight 1
    grid columnconfigure $extFrame  0 -weight 1
    grid columnconfigure $listFrame 0 -weight 1
    grid columnconfigure $listFrame 1 -weight 0
    grid rowconfigure    $listFrame 0 -weight 1
    grid rowconfigure    $listFrame 1 -weight 0

    grid $ctrFrame       -sticky news
    grid $intCtrFrame    -sticky news
    grid $ctrLabel       -sticky w
    grid rowconfigure    $extFrame 0    -weight 0
    grid rowconfigure    $extFrame 1    -weight 1
    grid rowconfigure    $extFrame 2    -weight 0
    grid columnconfigure $ctrFrame 0    -weight 1
    grid columnconfigure $intCtrFrame 0 -weight 1

    if {![info exists getleftState(urlQueue)]} {
        return
    }

    focus [$listbox bodypath]

    bind [$listbox bodypath] <ButtonRelease-1> {
        mainWin::ActivateDownloadButtons
    }
    bind [$listbox bodypath] <Button-3> {
        mainWin::UrlListRightClick %W %x %y %X %Y
    }
    bind [$listbox bodypath] <Double-Button-1> {
        mainWin::UrlListDoubleClick %W %y
    }
    bind [$listbox bodypath] <Return> {
        mainWin::UrlListDoubleClick %W %y
    }
    bind [$listbox bodypath] <KP_Enter> {
        mainWin::UrlListDoubleClick %W %y
    }
    bind . <Key-Delete> {
        set mainWin::tmpUrl   [mainWin::UrlListSelectedUrl]
        set mainWin::tmpIndex [mainWin::UrlListUrlIndex $mainWin::tmpUrl]
        mainWin::UrlListDeleteEntry $mainWin::tmpUrl
        $mainWin::listbox selection set $mainWin::tmpIndex
    }

    wm protocol . WM_DELETE_WINDOW {
        ExitGetleft
    }

    bind Button <Return>   {%W invoke}
    bind Button <KP_Enter> {%W invoke}

    foreach site $getleftState(urlQueue) {
        UrlListEnterEntry $site $urlsToDownload($site,resume) \
                                $urlsToDownload($site,dir)
    }
    $listbox yview moveto 0

    ActivateTaskbarButtons

    set menus(rightClick) [menu .menu -tearoff 0]
    $menus(rightClick) add command -label $labelMenus(resume)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelMenus(options)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelDialogs(copyLink)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelMenus(top)
    $menus(rightClick) add command -label $labelMenus(up)
    $menus(rightClick) add command -label $labelMenus(down)
    $menus(rightClick) add command -label $labelMenus(bottom)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelMenus(delete)
 
    return
}
}
