###############################################################################
###############################################################################
##                                Opciones.tcl
###############################################################################
###############################################################################
## Creates a dialog in which to choose the options for the program
###############################################################################
###############################################################################
## (c) 2001 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the GPL v2
###############################################################################
###############################################################################

namespace eval Opciones {

###############################################################################
# UpdateSpinBox
#    Activates o deactivates the spinboxes.
#
# Parameter:
#    which: The spinbox to update
###############################################################################
proc UpdateSpinBox {which} {
    variable window
    variable optionsTemp

    if {$which=="internal"} {
        if {$optionsTemp(limitLevels)==1} {
            CuadroSpin::EnableSpin $window(intSpin) normal
            if {$optionsTemp(levels)==-1} {
                set CuadroSpin::CuadroSpins($window(intSpin)) 1
            } else {
                set CuadroSpin::CuadroSpins($window(intSpin)) $optionsTemp(levels)
            }
        } else {
            CuadroSpin::EnableSpin $window(intSpin) disable
            set CuadroSpin::CuadroSpins($window(intSpin)) 0
        }
    } else {
        if {$optionsTemp(external)==1} {
            CuadroSpin::EnableSpin $window(extSpin) normal
            if {$optionsTemp(exLevels)==0} {
                set CuadroSpin::CuadroSpins($window(extSpin)) 1
            } else {
                set CuadroSpin::CuadroSpins($window(extSpin)) $optionsTemp(exLevels)
            }
        } else {
            CuadroSpin::EnableSpin $window(extSpin) disable
            set CuadroSpin::CuadroSpins($window(extSpin)) 0
        }
    }
    return
}

###############################################################################
# SetDefaultOptions
#    This procedures set the downloading options to the default values.
#
# Parameter:
#    The name of the options to set, it can be 'downOptions' or
#    'downOptionsTemp'
###############################################################################
proc SetDefaultOptions {optionsToSet} {
    global defaultDownOptions

    upvar #0 $optionsToSet options
    array set options [array get defaultDownOptions]

    if {$defaultDownOptions(levels)==-1} {
        set options(limitLevels) 0
    } else {
        set options(limitLevels) 1
    }

    return
}

###############################################################################
# SaveDefaultOptions
#    Sets the current values of the options in the dialog as default.
#############################################################################
proc SaveDefaultOptions {} {
    global defaultDownOptions
    variable optionsTemp

    array set defaultDownOptions [array get optionsTemp]

    return
}

###############################################################################
# ActivateMap
#    It is invoked when the user clicks in the 'Get map' checkbutton in the
#    options dialog.
#
# Side effects:
#    It disables or activates the 'Only Html checkbutton' and the 'Choose
#    filter' button, sets optionsTemp(onlyHtml) to '1' to get the map and
#    to '0' not to get it.
###############################################################################
proc ActivateMap {} {
    variable optionsTemp
    variable window

    if {$optionsTemp(map)==1} {
        $window(onlyHtml)  configure -state disable
        $window(chooseFil) configure -state disable
        set optionsTemp(onlyHtml) 1
    } else {
        $window(onlyHtml)  configure -state normal
        $window(chooseFil) configure -state normal
        set optionsTemp(onlyHtml) 0
    }

    return
}

###############################################################################
# ChangeFilter
#    Invokes the dialog to choose the filter for the download.
#
# Side effect:
#    Puts the returned filter into 'optionsTemp(filter)'
###############################################################################
proc ChangeFilter {} {
    variable window
    variable optionsTemp

    set optionsTemp(filter) [Herramientas::FilterFiles \
            $optionsTemp(filter) $window(win)]

    return
}

###############################################################################
# ChooseOptionsWindow
#    Creates the dialog in which to select the different options.
#
# Parameters:
#    parent: Path of the dialog over which this one will appear.
###############################################################################
proc ChooseOptionsWindow {parent} {
    global labelTitles labelButtons labelDialogs labelFrames getleftOptions
    global dirGetleft
    variable optionsTemp
    variable done
    variable window

    if {[winfo exists .optionsWin]} {
        focus $window(win)
        return
    } 

    set win [toplevel .optionsWin]
    wm resizable $win 0 0
    wm title $win $labelTitles(optionsWin)

    set coord(x) [winfo rootx $parent]
    set coord(y) [winfo rooty $parent]
    wm geometry $win +[expr {$coord(x)+100}]+[expr {$coord(y)+15}]

    set extFrame  [frame $win.extFrame]
    set intFrame  [frame $extFrame.intFrame]
    set butFrame  [frame $extFrame.butFrame]

    set linkFrame [fl::FrameLabel $intFrame.links -bd 2 -relief groove  \
            -textvariable labelFrames(links)]
    set intLinks    [frame $linkFrame.int]
    set updirLinks  [checkbutton $intLinks.updirLinks      \
        -variable Opciones::optionsTemp(dir) -textvariable labelDialogs(uplinks)]
    set updateLinks [checkbutton $intLinks.updateLinks     \
        -variable Opciones::optionsTemp(update) -textvariable labelDialogs(update)]
    set cgiLinks    [checkbutton $intLinks.cgiLinks        \
        -variable Opciones::optionsTemp(cgi) -textvariable labelDialogs(cgi)]

    set levelFrame [fl::FrameLabel $intFrame.levels -bd 2 -relief groove \
            -textvariable labelFrames(levels)]
    set intLevels  [frame $levelFrame.int]
    set internal   [checkbutton $intLevels.internal    \
        -variable Opciones::optionsTemp(limitLevels)   \
        -textvariable labelDialogs(levels)             \
        -command "Opciones::UpdateSpinBox internal"]
    set external   [checkbutton $intLevels.external    \
        -variable Opciones::optionsTemp(external)      \
        -textvariable labelDialogs(external)           \
        -command "Opciones::UpdateSpinBox external"]
    set intSpin [CuadroSpin::CuadroSpin $intLevels.intspin     \
            -default 0 -min 0 -max 5 -width 3                  \
            -bg $getleftOptions(bg) -fg $getleftOptions(fg)    \
            -disbg $getleftOptions(disBg)                      \
            -disfg $getleftOptions(disFg)]
    set extSpin [CuadroSpin::CuadroSpin $intLevels.extspin     \
            -default 0 -min 0 -max 3 -width 3                  \
            -bg $getleftOptions(bg) -fg $getleftOptions(fg)    \
            -disbg $getleftOptions(disBg)                      \
            -disfg $getleftOptions(disFg)]

    set imageFrame   [fl::FrameLabel $intFrame.image   -bd 2 -relief groove  \
            -textvariable labelFrames(images)]
    set intImages    [frame $imageFrame.int]
    set all          [radiobutton $intImages.all        -value 0    \
            -textvariable labelDialogs(allImages)                   \
            -variable Opciones::optionsTemp(images)]
    set onlyThumbs   [radiobutton $intImages.onlyThumbs -value 1    \
            -variable Opciones::optionsTemp(images)                 \
            -textvariable labelDialogs(onlyThumb)]
    set noThumbs     [radiobutton $intImages.noThumbs   -value 2    \
            -variable Opciones::optionsTemp(images)                 \
            -textvariable labelDialogs(noThumb)]
    
    set filterFrame  [fl::FrameLabel $intFrame.filters -bd 2 -relief groove        \
            -textvariable labelFrames(filters)]
    set intFilters   [frame $filterFrame.int]
    set mapLinks     [checkbutton $intFilters.mapLinks                             \
	      -textvariable labelDialogs(getMap)  -variable Opciones::optionsTemp(map) \
            -command Opciones::ActivateMap]
    set onlyHtml     [checkbutton $intFilters.onlyHtml \
            -variable Opciones::optionsTemp(onlyHtml)  \
            -textvariable labelDialogs(onlyHtml)]
    set chooseFilter [button $intFilters.choose        \
            -textvariable labelButtons(chooseFilter)   \
            -command Opciones::ChangeFilter]

    set accept  [button $butFrame.accept  -textvariable labelButtons(ok)       \
		-width 8 -command "Opciones::ChooseOptionsControl"]
    set cancel  [button $butFrame.cancel -textvariable labelButtons(cancel)    \
		-width 8 -command "set Opciones::done 0"]
    set help    [button $butFrame.help    -textvariable  labelButtons(help)    \
        -width 8 -command "[list help::init [file join $dirGetleft(doc) help.help] howtoDown#Options]"]
    set saveOps [button $butFrame.save -textvariable labelButtons(save)        \
        -width 8 -command "Opciones::SaveDefaultOptions"]
    set loadOps [button $butFrame.load -textvariable labelButtons(load)        \
        -width 8 -command "
                Opciones::SetDefaultOptions Opciones::optionsTemp
                Opciones::ActivateMap
                Opciones::UpdateSpinBox internal
                Opciones::UpdateSpinBox external
        "]

    set window(win)       $win
    set window(intSpin)   $intSpin
    set window(extSpin)   $extSpin
    set window(onlyHtml)  $onlyHtml
    set window(chooseFil) $chooseFilter

    UpdateSpinBox internal
    UpdateSpinBox external
    if {$optionsTemp(map)==1} {
        ActivateMap
    }

    grid $extFrame -padx 5 -ipady 5
    grid $intFrame -sticky s

    grid $linkFrame   $levelFrame $butFrame -padx 5 -pady 5 -sticky news
    grid $filterFrame $imageFrame           -padx 5 -pady 5 -sticky news

    grid configure $butFrame -rowspan 2

    grid $intLinks          -sticky w -padx 5 -pady 5 
    grid $updirLinks        -sticky w
    grid $updateLinks       -sticky w
    grid $cgiLinks          -sticky w

    grid $intLevels         -sticky w -padx 5 -pady 5 
    grid $internal $intSpin -sticky w
    grid $external $extSpin -sticky w

    grid $intFilters        -sticky w -padx 5 -pady 5  
    grid $mapLinks          -sticky w
    grid $onlyHtml          -sticky w
    grid $chooseFilter  

    grid $intImages         -sticky w -padx 5 -pady 5
    grid $all               -sticky w
    grid $onlyThumbs        -sticky w
    grid $noThumbs          -sticky w

    grid $accept  -pady 3
    grid $cancel  -pady 3
    grid $loadOps -pady 3
    grid $saveOps -pady 3
    grid $help    -pady 3

    focus $win
    bind  $win <Escape> "set Opciones::done 0"

    return
}

###############################################################################
# ChooseOptionsControl
#    Makes sure that the options in the dialog make sense before closing.
###############################################################################
proc ChooseOptionsControl {} {
    global labelTitles labelMessages
    variable window
    variable done
    variable optionsTemp

    if {$optionsTemp(limitLevels)==1} {
        set intLimit $CuadroSpin::CuadroSpins($window(intSpin))
        if {![regexp {^[0-9]+$} $intLimit]} {
            tk_messageBox -title $labelTitles(error) -parent $window(win) \
                    -message $labelMessages(intNaN)  -type ok 
            return
        }
        if {$intLimit<0} {
            CuadroSpin::CuadroSpins($window(intSpin)) 0
        }
        if {$intLimit>5} {
            CuadroSpin::CuadroSpins($window(intSpin)) 5
        }
    }

    if {$optionsTemp(external)!=0} {
        set extLimit $CuadroSpin::CuadroSpins($window(extSpin))
        if {![regexp {^[0-9]+$} $extLimit]} {
            tk_messageBox -title $labelTitles(error) -parent $window(win) \
                    -message $labelMessages(extNaN)  -type ok 
            return
        }
        if {$extLimit<0} {
            CuadroSpin::CuadroSpins($window(intSpin)) 0
        }
        if {$extLimit>3} {
            CuadroSpin::CuadroSpins($window(intSpin)) 3
        }
    }

    set done 1

    return
}

###############################################################################
# ChooseOptions
#    Selects the different options.
#
# Parameter:
#    options: The array in which the options will be stored.
#    parent: Path of the window over which the dialog will appear
#
# Return
#    '1' if the user accepted the changes, '0' if he cancelled.
###############################################################################
proc ChooseOptions {optionsArray {parent .}} {
    variable optionsTemp
    variable window
    variable done

    upvar $optionsArray downOptions

    catch {unset optionsTemp}

    array set optionsTemp [array get downOptions]

    if {$optionsTemp(levels)==-1} {
        set optionsTemp(limitLevels) 0
    } else {
        set optionsTemp(limitLevels) 1
    }
    
    ChooseOptionsWindow $parent
    tkwait variable Opciones::done

    if {$done==1} {
        array set downOptions [array get optionsTemp]
        if {$optionsTemp(limitLevels)==0} {
            set downOptions(levels) -1
        } else {
            set downOptions(levels)   $CuadroSpin::CuadroSpins($window(intSpin))
        }
        if {$optionsTemp(external)==0} {
            set downOptions(exLevels) 0
        } else {
            set downOptions(exLevels) $CuadroSpin::CuadroSpins($window(extSpin))
        }
    }
    catch {destroy $window(win)}

    return $done
}

}

