##############################################################################
##############################################################################
#                               Cookies.tcl
##############################################################################
##############################################################################
# In this file are implemented the procedures to deal with the cookies.
##############################################################################
##############################################################################
# Copyright 2001 Andrs Garca Garca  -- fandom@retemail.es
# Distributed under the terms of the GPL v2
##############################################################################
##############################################################################

namespace eval Cookies  {

##############################################################################
# SaveCookie
#    Gets the cookie header and saves it in the 'cookies.txt' file.
#
# Parameter
#    header: the header we got from the server
##############################################################################
proc SaveCookie {header} {
    global dirGetleft

    set parsedUrl [HtmlParser::ParseUrl $Ventana::link]
    set domain [lindex $parsedUrl 1]
    set dir    [lindex $parsedUrl 2]
    if {$dir==""} {
        set dir /
    }

    if {![regexp {(?::\s)(.+?)(?:;)}   $header nada cookies]} return
    if {![regexp -nocase {(?:path=)(.+?)(?:\.*$|;| )} $header nada path]} {
        set path $dir
    }
    if {![regexp -nocase {(?:domain=)(.+?)(?:\.*$|;| )} $header nada domain]} {
        set domain $domain
    }
    if {[regexp {^\.} $domain]} {
        set share TRUE
    } else {
        set share FALSE
    }
    if {[regexp -nocase {(?:expires=)(.+?)(?:\.*$|;)} $header nada date]} {
        # clock scan doesn't like dates after 2037
        if {[catch {set expires [clock scan $date]}]} {
           set expires [clock scan "Thur, 31 Dec 2037 07:43:04 GMT"]
        }
    } else {
        set expires [expr {[clock seconds]} + 3600]
    }
    if {[regexp -nocase {secure} $header]} {
        set secure TRUE
    } else {
        set secure FALSE
    }
    if {[regexp {([^=]+)(?:=)(.*)} $cookies nada cookieName cookieValue]} {
        StoreCookie $domain $share $path $secure $expires                    \
                $cookieName $cookieValue
    }
    WriteCookies

    return
}

##############################################################################
# CheckCookies
#    Reads the file 'cookies.txt' and checks whether the cookies are current,
#    deletes those which are not and saves the resulting file.
##############################################################################
proc CheckCookies {} {
    variable cookies
    global   dirGetleft

    set now [clock seconds]

    if {[ReadCookies]==1} return

    set cookieJar  [file join $dirGetleft(conf) cookies.txt]
    set cookieTemp [file join $dirGetleft(conf) cookies.tmp]

    if {[catch {open "$cookieTemp" w+} tempHandle]} return

    for {set i 0} {[info exists cookies($i,domain)]} {incr i} {
        if {"$cookies($i,expires).0">$now} {
            puts -nonewline $tempHandle "$cookies($i,domain)\t"
            puts -nonewline $tempHandle "$cookies($i,share)\t"
            puts -nonewline $tempHandle "$cookies($i,path)\t"
            puts -nonewline $tempHandle "$cookies($i,secure)\t"
            puts -nonewline $tempHandle "$cookies($i,expires)\t"
            puts -nonewline $tempHandle "$cookies($i,cookie)\t"
            puts            $tempHandle "$cookies($i,value)"
        }
    }
    close $tempHandle

    if {[catch {file rename -force -- $cookieTemp $cookieJar}]} {
        catch {file delete $cookieTemp}
    }

    return
}

##############################################################################
# ReadCookies
#    Read the cookie.txt file into memory
#
# Returns:
#    '0' if all went well, '1' otherwise.
##############################################################################
proc ReadCookies {} {
    global dirGetleft
    variable cookies

    set cookieJar  [file join $dirGetleft(conf) cookies.txt]
    if {[catch {open "$cookieJar" r} cookieHandle]} {
        return 1
    }
    catch "unset Cookies::cookies"
    for {set i 0} {![eof $cookieHandle]} {} {
        set line [gets $cookieHandle]

        if {[regexp {^#|^\s|^$} $line]} continue

        set cookies($i,domain)  [lindex $line 0]
        set cookies($i,share)   [lindex $line 1]
        set cookies($i,path)    [lindex $line 2]
        set cookies($i,secure)  [lindex $line 3]
        set cookies($i,expires) [lindex $line 4]
        set cookies($i,cookie)  [lindex $line 5]
        set cookies($i,value)   [lindex $line 6]
        incr i
    }
    close $cookieHandle
    return 0
}

##############################################################################
# WriteCookies
#    Writes the cookies into the cookie file.
##############################################################################
proc WriteCookies {} {
    global dirGetleft
    variable cookies

    if {![info exists cookies(0,domain)]} return

    set cookieJar [file join $dirGetleft(conf) cookies.txt]
    set cookieHandle [open "$cookieJar" w]

    for {set i 0} {[info exists cookies($i,domain)]} {incr i} {
        puts -nonewline $cookieHandle "$cookies($i,domain)\t"
        puts -nonewline $cookieHandle "$cookies($i,share)\t"
        puts -nonewline $cookieHandle "$cookies($i,path)\t"
        puts -nonewline $cookieHandle "$cookies($i,secure)\t"
        puts -nonewline $cookieHandle "$cookies($i,expires)\t"
        puts -nonewline $cookieHandle "$cookies($i,cookie)\t"
        puts            $cookieHandle "$cookies($i,value)"
    }
    close $cookieHandle

    return
}

##############################################################################
# SeeCookies
#    Writes in the console the cookies we have in memory.
##############################################################################
proc SeeCookies {} {
    global dirGetleft
    variable cookies

    for {set i 0} {[info exists cookies($i,domain)]} {incr i} {
        puts -nonewline "$cookies($i,domain)\t$cookies($i,share)\t$cookies($i,path)\t"
        puts -nonewline "$cookies($i,secure)\t$cookies($i,expires)\t"
        puts            "$cookies($i,cookie)\t$cookies($i,value)"
    }
    return
}

##############################################################################
# StoreCookie
#    Either stores the new cookie or changes it if we already have it.
#
# Parameters:
#    I hope they are selfexplanatory.
##############################################################################
proc StoreCookie {domain share path secure expires cookie value} {
    variable cookies

    for {set i 0} {[info exists cookies($i,domain)]} {incr i} {
        if {($cookies($i,domain)==$domain)&&($cookies($i,path)==$path)&&
            ($cookies($i,secure)==$secure)&&($cookies($i,cookie)==$cookie)} {
            break
        }
    }
    set cookies($i,domain)  $domain
    set cookies($i,share)   $share
    set cookies($i,path)    $path
    set cookies($i,secure)  $secure
    set cookies($i,expires) $expires
    set cookies($i,cookie)  $cookie
    set cookies($i,value)   $value

    return
}

# when we read the namespace for the first time we check the cookies

CheckCookies

}
