##############################################################################
##############################################################################
#                               ChooseLinks.tcl
##############################################################################
##############################################################################
# In this namespace are the procedures that implement the 'Choose Links'
# dialog.
##############################################################################
##############################################################################
# Copyright 2001-2002 Andrs Garca Garca  -- fandom@retemail.es
# Distributed under the terms of the GPL v2
##############################################################################
##############################################################################

namespace eval ChooseLinks {

###############################################################################
# Traces
#    When one of the options change this procedure is invoked to change the
#    links.
###############################################################################
proc Traces {linkArray name1 name2 op} {
    variable traces
    global downOptions

    if {$downOptions($name2)!=$traces($name2)} {
        ChooseLinks::ChangeLinksToChoose $linkArray
        set traces($name2) $downOptions($name2)
    }

    return
}

###############################################################################
# SetChosenLinks
#    Puts into getleftState(chosenLinks) the links the user wants to download.
###############################################################################
proc SetChosenLinks {} {
    global downOptions getleftState

    if {($downOptions(map)==1)&&($getleftState(mapDone)==1)} {
        set getleftState(mapLinks) $getleftState(chosenLinks)
    }
    set getleftState(chosenLinks) ""
    for {set i 1} {$i<$HtmlParser::nLinks} {incr i} {
        lappend getleftState(chosenLinks) $HtmlParser::links($i,ok)
    }
    return
}


###############################################################################
# CopyLink
#   Copies the link to the clipboard
#
# Parameter
#    textPath: Path of the text widget where the links are.
###############################################################################
proc CopyLink {textPath} {

    regexp {(.*)(\.)} [$textPath index insert] nada line

    clipboard clear
    for {set i 1;set j 0} {$i<$HtmlParser::nLinks} {incr i} {
        incr j $HtmlParser::links($i,ok)
        if {$j==$line} {
            clipboard append $HtmlParser::links($i,url)
            break
        }
    }
    return
}

###############################################################################
# PutInQueue
#   Puts one of the links into the queue.
#
# Parameter
#    textPath: Path of the text widget where the links are.
###############################################################################
proc PutInQueue {textPath} {
    global dirGetleft
    variable win

    regexp {(.*)(\.)} [$textPath index insert] nada line
    for {set i 1;set j 0} {$i<$HtmlParser::nLinks} {incr i} {
        incr j $HtmlParser::links($i,shown)
        if {$j==$line} {
            set urlToDownload $HtmlParser::links($i,url)
            break
        }
    }

    EnterUrl::EnterUrl 0 $win(top) $urlToDownload

    return
}

###############################################################################
# ChooseAllOptions
#    Invokes the dialog to choose from all the downloading options, if
#    necessary it also makes sure that the links are changed.
#
# Parameter
#    The name of the array with the links
###############################################################################
proc ChooseAllOptions {linkArray} {
    global downOptions
    
    if {[Opciones::ChooseOptions downOptions]==1} {
        ChooseLinks::ChangeLinksToChoose $linkArray
    }

    return
}

###############################################################################
# ContextMenu
#    When the user right-clicks in the dialog, this will pop the context menu
#    up.
#
# Parameters:
#    link: The nearest link to the click.
#    x,y: coords where to show the menu
###############################################################################
proc ContextMenu {link x y} {
    variable win
    variable activeLink
    global   getleftOptions tcl_platform

    if {($tcl_platform(platform)!="windows")&&([winfo exists $link])} {
        $link configure -fg $getleftOptions(selFg)
        $link configure -bg $getleftOptions(selBg)
    }
    if {[winfo exists $link]} {
        $win(menu) entryconfigure 8 -state normal
        $win(menu) entryconfigure 9 -state normal
    } else {
        $win(menu) entryconfigure 8 -state disabled
        $win(menu) entryconfigure 9 -state disabled
    }
    set activeLink $link
    regexp {(?:enlaces)(.*)} $link nada line
    $ChooseLinks::win(text) mark set insert $line.0
    tk_popup $win(menu) $x $y

    return
}

###############################################################################
# Chosen
#    Selects and deselects the chosen files.
#
# Parameters:
#    pattern: regular expresion that identifies the files to (de)select.
#    select: '1' to select, '0' to deselect.
###############################################################################
proc Chosen {linkArray pattern select} {
    global siteMap downOptions

    upvar #0 $linkArray links

    for {set i 1} {[info exists links($i,url)]} {incr i} {
        if {$links($i,shown)==0} continue
        if {[regexp -nocase $pattern $links($i,file)]} {
            set links($i,ok) $select
        }
    }
    return
}

###############################################################################
# ChooseGroup
#    Activates or deactivates the links that are either before or after the
#    link where the user right-clicked.
#
# Parameters:
#    which: 'pre' the links before the click.
#           'fol' the links after.
#    action: '0' deactivates the links
#            '1' activates them.
###############################################################################
proc ChooseGroup {which action} {
    variable win

    regexp {(.*)(\.)} [$win(text) index insert] nada line
    if {$which=="pre"} {
        set next -1
    } else {
        set next  1
    }
    for {set i $line} {1} {incr i $next} {
        if {![winfo exists $win(text).enlaces$i]} {
	        if {($which=="pre")} {
		        if {$i==0} return else continue
			} else return
	    }
        if {$action==0} {
            $win(text).enlaces$i deselect
        } else {
            $win(text).enlaces$i select
        }
    }
    return
}

###############################################################################
# ChangeLinksToChoose
#    If the user changes the downloading options in the 'Choose Files' dialog
#    this procedure takes care of displaying the new links to choose from.
#
# Parameter:
#    linkArray: The name of the array where the links are stored.
###############################################################################
proc ChangeLinksToChoose {linkArray} {
    global filesDone
    variable win
    variable refererUrl

    $win(text) configure -cursor watch
    $win(text) configure -state  normal
    $win(text) delete 1.0 end
    update

    set file [lindex [array get filesDone] 0]
    catch {unset filesDone}
    set refererFile [::Commands::UrlToFile $refererUrl]

    HtmlParser::FilterLinks $refererUrl $linkArray
#    Commands::ChangePage $refererUrl

    LinksToChoose $linkArray

    $win(text) configure -state  disabled
    $win(text) configure -cursor left_ptr
    update

    return
}

###############################################################################
# LinksToChoose
#    Inserts the links in the 'Choose files' directory.
#
# Parameter
#    linkArray: Array where the links to be shown are stored.
###############################################################################
proc LinksToChoose {linkArray} {
    global siteIndex siteMap charSets downOptions getleftState labelMessages
    global getleftOptions
    variable win

    upvar #0 $linkArray links

    if {($downOptions(map)==1)&&($getleftState(mapDone)==1)} {
        set map 1
    } else {
        set map 0
    }
    for {set i 1 ; set j 1 ; set pre ""} {[info exists links($i,url)]} {incr i} {
        set links($i,shown) 0
        if {$links($i,ok)==0} continue
        set links($i,shown) 1

        set boton [checkbutton $win(text).enlaces$j -variable ${linkArray}($i,ok) \
                -bg $getleftOptions(bg) -fg $getleftOptions(fg)                   \
                -activebackground $getleftOptions(bg)                             \
                -highlightthickness 0]

        $win(text) window create $i.0 -window $win(text).enlaces$j
        set description [HtmlParser::TidyDescription \
                $links($i,descrip) $links($i,url)]
        if {$map==1} {
            for {set k 0 ; set spc "   " ; set pre ""} {$k<$links($i,level)} \
                    {incr k} {
                append pre $spc
            }
        }
        $win(text).enlaces$j configure -text $pre$description
        $win(text) insert insert \n

        bind $boton <Button-3> {
            ChooseLinks::ContextMenu %W %X %Y
        }
        incr j
    }
    if {$j==1} {
        $win(text) insert insert $labelMessages(noLinksOpt)
    }

    return
}

###############################################################################
# SetEncodingMenus
#    Creates the menus to choose from the different encodings
#
# Parameter
#    menu: The menu to create.
#    encodings: A list with the encodings to add.
#    referer: The page  from which we got the links.
#    linkArray: Name of the array with the links.
###############################################################################
proc SetEncodingMenus {menu encodings referer linkArray} {
    global charSets

    foreach codeName $encodings {
        if {![regexp {(?:\()(.*)(?:\))} $codeName nada code]} {
            set code $codeName
        }
        if {![info exists charSets($code)]} {
            continue
        }
        set code [string tolower $code]
        $menu add radio -variable ChooseLinks::tmp -value $code    \
                -label $codeName -command "
                    Descargar::Preprocessing $referer 0
                    HtmlParser::ChangeEncoding \$charSets(\$ChooseLinks::tmp)
                    ChooseLinks::ChangeLinksToChoose $linkArray
                    set ChooseLinks::tmp {}
                "
    }
    return
}

###############################################################################
# ChooseDownloads
#    Shows a dialog with all the links in the entered Url so you can choose
#    which ones will be followed and which will be ignored.
#
# Parameter
#    referer: url of the page from which the links where taken.
#    linkArray: Array where the links to be shown are stored.
###############################################################################
proc ChooseDownloads {referer linkArray} {
    global labelButtons labelTitles labelDialogs labelMenus
    global dirGetleft downOptions charSets getleftState getleftOptions
    variable refererUrl $referer
    variable traces
    variable win
    variable done
    variable menu

    catch {wm withdraw $Ventana::window(top)}

    if {$HtmlParser::nLinks==1} "return 0"

    set win(top) [toplevel .elegir]
    set marco    [frame $win(top).marco]
    set checks   [frame $win(top).checks]
    set botones  [frame $win(top).botones]

    wm title $win(top) $labelTitles(choose)

    set texto [text $marco.texto -yscrollcommand [list $marco.yscroll set]    \
            -xscrollcommand [list $marco.xscroll set] -wrap none              \
            -cursor left_ptr -bg $getleftOptions(bg)]
    set win(text) $texto

    set aceptar [button $botones.aceptar   -textvariable labelButtons(ok)     \
            -width 9 -command "
                mainWin::UpdateUrlOptions
                ChooseLinks::SetChosenLinks
                set ChooseLinks::done 1"]
    set cancelar [button $botones.cancelar -textvariable labelButtons(cancel) \
            -width 9 -command "set ChooseLinks::done 0" ]

    set mas   [image create bitmap  -file [file join $dirGetleft(images) mas]]
    set menos [image create bitmap  -file [file join $dirGetleft(images) menos]]
    set lhtml [label  $checks.lhtml -text "  $labelDialogs(html)"]
    set htmla [button $checks.htmla -image $mas   -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {(html$)|(htm$)|(/$)} 1"]
    set htmle [button $checks.htmle -image $menos -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {(html$)|(htm$)|(/$)} 0"]
    set lima  [label $checks.lima   -text "  $labelDialogs(images)"]
    set imaa  [button $checks.imaa  -image $mas   -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {(gif$)|(jpg$)|(jpeg$)|(bmp$)|(xbm$)|(tiff?$)|(png$)} 1"]
    set imae  [button $checks.imae  -image $menos -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {(gif$)|(jpg$)|(jpeg$)|(bmp$)|(xbm$)|(tiff?$)|(png$)} 0"]
    set lcom  [label $checks.lcom   -text "  $labelDialogs(archives)"]
    set coma  [button $checks.cona  -image $mas   -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {(tar$)|(gz$)|(z$)|(zip$)|(tgz$)|(rpm$)|(deb$)} 1"]
    set come  [button $checks.cone  -image $menos -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {(tar$)|(gz$)|(z$)|(zip$)|(tgz$)|(rpm$)|(deb$)} 0"]
    set ltodo [label $checks.ltodo  -text "  $labelDialogs(all)"]
    set todoa [button $checks.todoa -image $mas   -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {.} 1"]
    set todoe [button $checks.todoe -image $menos -width 8 -height 8 \
            -command "ChooseLinks::Chosen $linkArray {.} 0"]

    LinksToChoose $linkArray

    set barray [scrollbar $marco.yscroll -orient vertical \
        -command [list $texto yview]]
    set barrax [scrollbar $marco.xscroll -orient horizontal  \
        -command [list $texto xview]]

    set menu [menu $win(top).menu -tearoff 0]
    set win(menu) $menu
    $menu add cascade -label "$labelDialogs(html)"      -menu $menu.html
    $menu add cascade -label "$labelDialogs(images)"    -menu $menu.images
    $menu add cascade -label "$labelDialogs(archives)"  -menu $menu.archives
    $menu add cascade -label "$labelDialogs(all)" -menu $menu.all
    $menu add separator
    $menu add cascade -label "$labelDialogs(previous)"  -menu $menu.previous
    $menu add cascade -label "$labelDialogs(following)" -menu $menu.following
    $menu add separator
    $menu add command -label "$labelDialogs(copyLink)"   \
            -command "ChooseLinks::CopyLink $texto"
    $menu add command -label "$labelMenus(queue)"        \
            -command "ChooseLinks::PutInQueue $texto"
    $menu add separator
    $menu add cascade -label "$labelMenus(options)" -menu  $menu.options

    menu $menu.html      -tearoff 0
    menu $menu.images    -tearoff 0
    menu $menu.archives  -tearoff 0
    menu $menu.all       -tearoff 0
    menu $menu.previous  -tearoff 0
    menu $menu.following -tearoff 0
    menu $menu.options   -tearoff 0

    $menu.html      add command -label "$labelDialogs(activate)"   \
            -command "$htmla invoke"
    $menu.html      add command -label "$labelDialogs(deactivate)" \
            -command "$htmle invoke"
    $menu.images    add command -label "$labelDialogs(activate)"   \
            -command "$imaa  invoke"
    $menu.images    add command -label "$labelDialogs(deactivate)" \
            -command "$imae  invoke"
    $menu.archives  add command -label "$labelDialogs(activate)"   \
            -command "$coma  invoke"
    $menu.archives  add command -label "$labelDialogs(deactivate)" \
            -command "$come  invoke"
    $menu.all       add command -label "$labelDialogs(activate)"   \
            -command "$todoa invoke"
    $menu.all       add command -label "$labelDialogs(deactivate)" \
            -command "$todoe invoke"
    $menu.previous  add command -label "$labelDialogs(activate)"   \
            -command "ChooseLinks::ChooseGroup pre 1"
    $menu.previous  add command -label "$labelDialogs(deactivate)" \
            -command "ChooseLinks::ChooseGroup pre 0"
    $menu.following add command -label "$labelDialogs(activate)"   \
            -command "ChooseLinks::ChooseGroup fol 1"
    $menu.following add command -label "$labelDialogs(deactivate)" \
            -command "ChooseLinks::ChooseGroup fol 0"

    $menu.options add command                                   \
            -label $labelMenus(all)                             \
            -command "ChooseLinks::ChooseAllOptions $linkArray"
    $menu.options add cascade -menu $menu.options.directorios   \
            -label $labelMenus(uplinks)
    $menu.options add cascade -menu $menu.options.levels        \
            -label $labelMenus(levels)
    $menu.options add cascade -menu $menu.options.exLevels      \
            -label $labelMenus(external)
    $menu.options add cascade -menu $menu.options.filters       \
            -label $labelMenus(fileFilter)
    $menu.options add cascade -menu $menu.options.encodings     \
            -label $labelMenus(encodings)
    $menu.options add check   -variable downOptions(update)     \
            -label $labelMenus(update)   -onvalue 1 -offvalue 0
    $menu.options add check   -variable downOptions(cgi)        \
            -label $labelMenus(cgi)      -onvalue 1 -offvalue 0

    set menuDir [menu $menu.options.directorios -tearoff 0]
    $menuDir add radio -variable downOptions(dir)  -value 1 \
           -label $labelMenus(linksFollow)
    $menuDir add radio -variable downOptions(dir)  -value 0 \
            -label $labelMenus(linksIgnore)

    set menuLev [menu $menu.options.levels -tearoff 0]
    $menuLev add radio -variable downOptions(levels) -value -1     \
            -label $labelMenus(noLimit)
    for {set i 0} {$i<6} {incr i} {
        $menuLev add radio -label $i -variable downOptions(levels) \
                -value $i
    }

    set menuEx [menu $menu.options.exLevels -tearoff 0]
    $menuEx add radio -variable downOptions(exLevels)   \
            -label $labelMenus(linksIgnore) -value 0    \
            -command "
                set downOptions(external) 0
            "
    for {set i 1} {$i<4} {incr i} {
        $menuEx add radio -label $i -variable downOptions(exLevels) \
            -value $i -command "
                set downOptions(external) 1
            "
    }

    set menuFil [menu $menu.options.filters -tearoff 0]
    $menuFil add check -variable downOptions(onlyHtml) -onvalue 1 -offvalue 0 \
            -label $labelMenus(onlyHtml)
    $menuFil add cascade -menu $menuFil.images \
            -label $labelMenus(images)
    $menuFil add command -label $labelMenus(chooseFilter) -command "
        Herramientas::FilterFiles \"$downOptions(filter)\" $win(top)
        ChooseLinks::ChangeLinksToChoose $linkArray
    "

    set menuFilImage [menu $menuFil.images -tearoff 0]
    $menuFilImage add radio -variable downOptions(images) -value 0 \
	      -label $labelMenus(allImages)
    $menuFilImage add radio -variable downOptions(images) -value 1 \
            -label $labelMenus(onlyThumb)
    $menuFilImage add radio -variable downOptions(images) -value 2 \
            -label $labelMenus(noThumb)

    set menuEncodings [menu $menu.options.encodings -tearoff 0]
    $menuEncodings add cascade -label $labelMenus(europa) -menu $menuEncodings.europa
    $menuEncodings add cascade -label $labelMenus(asia)   -menu $menuEncodings.asia
    $menuEncodings add cascade -label $labelMenus(africa) -menu $menuEncodings.africa
    set encodings [lsort "
        \"$labelMenus(ascii)\"                     
        \"${labelMenus(westEuropa)} (iso8859-1)\"   \"${labelMenus(westEuropa)} (cp850)\"
        \"${labelMenus(westEuropa)} (macRoman)\"    \"${labelMenus(celtic)} (iso8859-7)\"
        \"${labelMenus(greek)} (iso8859-7)\"        \"${labelMenus(greek)} (macGreek)\"
        \"${labelMenus(iceland)} (macIceland)\"     \"${labelMenus(baltic)} (iso8859-4)\"
        \"${labelMenus(baltic)}\"                   \"${labelMenus(baltic)} (cp1257)\"
        \"${labelMenus(centerEuropa)} (cp852)\"     \"${labelMenus(centerEuropa)} (cp1250)\"
        \"${labelMenus(centerEuropa)} (iso8859-3)\" \"${labelMenus(centerEuropa)} (iso8859-2)\"
        \"${labelMenus(croatian)} (macCroatian)\"   \"${labelMenus(cyrillic)} (cp855)\"
        \"${labelMenus(cyrillic)} (macCyrillic)\"   \"${labelMenus(cyrillic)} (macUkraine)\"
        \"${labelMenus(cyrillic)} (iso8859-5)\"     \"${labelMenus(cyrillic)} (cp866)\"
        \"${labelMenus(cyrillic)} (koi8-r)\"
    "]
    set $menuEncodings.europa [menu $menuEncodings.europa -tearoff 0]
    SetEncodingMenus $menuEncodings.europa $encodings $referer $linkArray
    set encodings [lsort "
        \"${labelMenus(ascii)}\"               \"${labelMenus(simpleChin)} (gb2312)\"
        \"${labelMenus(tradChin)} (big5)\"     \"${labelMenus(japan)} (iso2022-jp)\"
        \"${labelMenus(japan)} (euc-jp)\"      \"${labelMenus(japan)} (shiftjis)\"
        \"${labelMenus(korean)} (euc-kr)\"     \"${labelMenus(turkish)} (cp857)\"
        \"${labelMenus(turkish)} (iso8859-9)\" \"${labelMenus(turkish)} (macTurkish)\"
        \"${labelMenus(hebrew)} (cp862)\"      \"${labelMenus(hebrew)} (iso8859-8)\"
        \"${labelMenus(arabic)} (cp864)\"      \"${labelMenus(arabic)} (cp1256)\"
        \"${labelMenus(unicode)} (utf-8)\"
    "]
    set $menuEncodings.asia [menu $menuEncodings.asia -tearoff 0]
    SetEncodingMenus $menuEncodings.asia $encodings $referer $linkArray
    set encodings [lsort "
        \"${labelMenus(ascii)}\"            \"${labelMenus(arabic)} (cp864)\"
        \"${labelMenus(arabic)} (cp1256)\"  \"${labelMenus(unicode)} (utf-8)\"
    "]
    set $menuEncodings.africa [menu $menuEncodings.africa -tearoff 0]
    SetEncodingMenus $menuEncodings.africa $encodings $referer $linkArray

    bind $win(top) <KeyPress-Prior> "$texto yview scroll -1 pages"
    bind $win(top) <KeyPress-Next>  "$texto yview scroll  1 pages"
    bind $win(top) <Key-Up>         "$texto yview scroll -1 units"
    bind $win(top) <Key-Down>       "$texto yview scroll  1 units"
    bind $win(top) <MouseWheel> "
        if {%D<0} {
            $texto yview scroll  1 pages
        } else {
            $texto yview scroll -1 pages
        }
    "
    bind $win(top) <Escape>   "$cancelar invoke"
    bind $win(top) <Button-4> "$texto yview scroll -1 pages"
    bind $win(top) <Button-5> "$texto yview scroll  1 pages"
    bind $texto  <Button-3>       {
        event generate %W <Button-1> -x %x -y %y
        event generate %W <ButtonRelease-1> -x %x -y %y
        regexp {(.*)(\.)} [$ChooseLinks::win(text) index insert] nada line
        ChooseLinks::ContextMenu $ChooseLinks::win(text).enlaces$line %X %Y
    }
    bind $texto  <Button> {
        %W tag remove sel 0.0 end
    }
    bind $menu   <Unmap> {
        if {($tcl_platform(platform)!="windows") \
                &&([winfo exists $ChooseLinks::activeLink])} {
            $ChooseLinks::activeLink configure -fg $getleftOptions(fg)
            $ChooseLinks::activeLink configure -bg $getleftOptions(bg)
        }
    }

    grid $marco -sticky news
    grid $texto $barray -sticky news
    grid $barrax -sticky ew
    grid $checks -sticky ew
    grid $lhtml $htmla $htmle $lima $imaa $imae $lcom $coma $come \
            $ltodo $todoa $todoe
    grid $botones -sticky ew -ipady 4
    grid $aceptar $cancelar  -padx 3

    grid rowconfigure    $win(top) 0 -weight 1
    grid columnconfigure $win(top) 0 -weight 1
    grid rowconfigure    $marco  0 -weight 1
    grid columnconfigure $marco  0 -weight 1

    $texto configure -state disabled

    wm protocol $win(top) WM_DELETE_WINDOW "
            set ChooseLinks::done 0
            destroy $win(top)
    "

    trace variable downOptions(cgi)      w "::ChooseLinks::Traces $linkArray"
    trace variable downOptions(dir)      w "::ChooseLinks::Traces $linkArray"
    trace variable downOptions(external) w "::ChooseLinks::Traces $linkArray"
    trace variable downOptions(images)   w "::ChooseLinks::Traces $linkArray"
    array set traces [array get downOptions]

    tkwait visibility $win(top)
    tkwait variable ChooseLinks::done

    trace vdelete downOptions(dir)      w "::ChooseLinks::Traces $linkArray"
    trace vdelete downOptions(external) w "::ChooseLinks::Traces $linkArray"
    trace vdelete downOptions(cgi)      w "::ChooseLinks::Traces $linkArray"
    trace vdelete downOptions(images)   w "::ChooseLinks::Traces $linkArray"

    destroy $win(top)

    return $done
}

}

