#!/bin/sh
###############################################################################
###############################################################################
#                               Getleft
###############################################################################
###############################################################################
# copyright (C) 1999-2002 Andres Garcia Garcia - fandom@retemail.es
# Distributed under the terms of the GPL v2
###############################################################################
###############################################################################
# Version 1.0.5
###############################################################################
###############################################################################
# This program recursively downloads Web sites, as it goes, it changes absolute
# paths in the links to relative ones and links to active content pages get
# changed to the resulting pages.
###############################################################################
# The following looks for the latest version of wish in you path, it is 
# specially useful in FreeBSD where 'wish' only gives a warning that it
# doesn't work. \
wish=`( for name in $(ls $(echo $PATH | sed 's/:/ /g') 2> /dev/null | grep wish8.4); do echo $name ; done ) | tail -1`
# \
[ -z "$wish" ] && wish=`(for name in $(ls $(echo $PATH | sed 's/:/ /g') 2> /dev/null | grep wish8.3); do echo $name ; done ) | tail -1`
# \
[ -z "$wish" ] && wish=`(for name in $(ls $(echo $PATH | sed 's/:/ /g') 2> /dev/null | grep wish8.2); do echo $name ; done ) | tail -1`
# \
[ -z "$wish" ] && wish=`(for name in $(ls $(echo $PATH | sed 's/:/ /g') 2> /dev/null | grep wish8.1); do echo $name ; done ) | tail -1`
# \
[ -z "$wish" ] && wish=wish
# \
exec $wish "$0" ${1+"$@"}

###############################################################################
# CheckPrograms
#    Checks whether the programs Getleft relies on are within reach.
#
# Side effects:
#    If one of the programs is not in the path, the procedure will exit the
#    program after issuing a error message.
###############################################################################
proc CheckPrograms {} {
    global tcl_version errorCode tcl_platform

    if {$tcl_version<8.1} {
        tk_messageBox -icon error -type ok -message "This program only works with\n\
                Tcl/Tk v 8.1 or newer, yours is:\n\t $tcl_version" -title Error
        exit
    }
    catch {exec curl}
    if {[lindex $errorCode 1]=="ENOENT"} {
        tk_messageBox -type ok -icon error -title "No curl" \
            -message "Getleft depends on program\n\tcURL\nPlease check the docs"
        exit
    }
    if {$tcl_platform(platform)=="windows"} {
        if {[catch {load winkill.dll} result]} {
            tk_messageBox -type ok -icon error -title Error \
                    -message "Couldn't load 'winkill.dll'."
            exit
        }
    } 
    return
}

###############################################################################
# SetGetleftColours
#     This is one of my world famous kludges, I use it in order to know
#     which colours the system is using. If you know how to do this using
#     the option database, please do so.
###############################################################################
proc SetGetleftColours {} {
    global getleftOptions

    set tmp  [entry .tmp]
    set getleftOptions(bg)    [$tmp cget -background]
    set getleftOptions(fg)    [$tmp cget -foreground]
    set getleftOptions(selBg) [$tmp cget -selectbackground]
    set getleftOptions(selFg) [$tmp cget -selectforeground] 
    destroy .tmp

    set tmp [frame .tmp]
    set getleftOptions(disBg) [$tmp cget -background]
    destroy .tmp

    button  .tmp
    if {($getleftOptions(bg)=="#dcdcdc")||($getleftOptions(bg)=="#d9d9d9")} {
        set getleftOptions(bg) white
    }
    .tmp configure -state disable
    set getleftOptions(disFg) [.tmp cget -disabledforeground]
    destroy .tmp

    return
}

###############################################################################
# InitializeDownload
#    Takes care of downloading the Url entered by the user starting the
#    download.
#
# Parameter:
#    url: The requested Url.
#
# Returns:
#    The requested url, completed, if necessary with protocol or changed
#    if there was a redirection.
#    An empty string in case of error.
###############################################################################
proc InitializeDownload {url} {
    global directories siteUrl getleftState getleftOptions
    global labelTitles labelMessages

    set getleftOptions(stopPage)          0
    set getleftOptions(stopFile)          0
    set getleftOptions(pauseFile)         0
    set getleftOptions(cancelDown)        0
    set getleftState(errorFound)          0
    set getleftOptions(pausePage)         0
    set Ventana::Rizo::curlReport(stop)   0

    catch "unset ::filesDone"
    catch "unset ::Descargar::filesNotFound"
    catch "unset ::Descargar::relocated"
    catch "unset ::Descargar::updated"
    catch "unset ::Commands::sedChanges"

    if {![regexp -nocase {([^:/])(/)} $url]} {
        set url $url/
    }

    set siteUrl(url) $url
    set parsedUrl [HtmlParser::ParseUrl $url]
    if {$parsedUrl!=1} {
        set siteUrl(prot) [lindex $parsedUrl 0]
        set siteUrl(www)  [lindex $parsedUrl 1]
        set siteUrl(dir)  [lindex $parsedUrl 2]
        set fileName      [lindex $parsedUrl 3]
        if {$fileName==""} {
            set isHtml 1
        } else {
            set isHtml [regexp -nocase {\.html?$} $fileName]
        }
        if {$siteUrl(prot)==""} {
            set siteUrl(prot) http
            set url http://$url
        } else {
            set siteUrl(prot) [string tolower $siteUrl(prot)]
        }
        set getleftState(noConnect,$siteUrl(www)) 0
    } else {
        tk_messageBox -type ok -title $labelTitles(initError) -icon error \
            -message $labelMessages(formatError)
        return
    }

    set directories(local)  $directories(base)
    set siteUrl(base)       $siteUrl(dir)

    set result [Descargar::Download $url "-" $isHtml]

    if {$result==2} {
        if {($Ventana::Rizo::errorMessage!="")&&($getleftState(autoDown)==0)} {
            tk_messageBox -icon error -type ok -title $labelTitles(error) \
                     -message "$Ventana::Rizo::errorMessage"
        }
        return
    }

    if {($result==1)&&($isHtml==0)} {
        set localFileName [::Commands::UrlToFile $url]
        if {(![regexp {\.html?$} $localFileName])&&(![file exists $localFileName.html])} {
            file rename -force $localFileName $localFileName.html
            Commands::Touch $localFileName
        }
        return $url
    }

    if {($result==3)||($result==4)} {
        set parsedUrl [HtmlParser::ParseUrl $Descargar::relocated(url)]
        if {$parsedUrl==1} {
            return
        }
        set fileName      [lindex $parsedUrl 3]

        set siteUrl(base) $siteUrl(dir)
        set url $Descargar::relocated(url)
        if {$result==4} {
            set localFileName [::Commands::UrlToFile $url]
            if {(![regexp {\.html?$} $localFileName])&&(![file exists $localFileName.html])} {
                file rename -force $localFileName $localFileName.html
                Commands::Touch $localFileName
            }
        }
    }
    return $url
}

###############################################################################
# TheEnd
#    After downloading is completed or stopped by the user, this procedure
#    takes care of telling the user all about it.
#
# Parameter
#    noEnd: '1' if we don't want the 'It's over' message box.
###############################################################################
proc TheEnd {{noEnd 0}} {
    global labelTitles labelMessages getleftState dirGetleft directories
    global siteUrl getleftOptions downOptions

    set index [lsearch -exact $getleftState(urlQueue) $getleftState(url)]
    set domain [lindex [HtmlParser::ParseUrl $getleftState(url)] 1]

    if {($getleftOptions(stopFile)==1)||($getleftOptions(stopPage)==1)} {
        if {($getleftState(autoDown)==1)||($getleftState(delayedDown)==1)} {
            set what yes
            if {$getleftState(autoDown)==1} {
                mainWin::UrlListMoveTopBottom $getleftState(url)
            }
            if {$getleftState(delayedDown)==1} {
                catch {wm withdraw $::Ventana::window(top)}
            }
        } else {
            catch {wm withdraw $::Ventana::window(top)}
            set what [tk_messageBox -icon question -title $labelTitles(resumeSite)\
                    -type yesno -message $labelMessages(resumeSite)]
        }
        if {[string match $what "yes"]} {
            SaveDownloadData   $getleftState(url)
        } else {
            mainWin::UrlListDeleteEntry $getleftState(url)
            if {[catch {file delete [file join $dirGetleft(conf) $domain.gdt]} errorMsg]} {
                if {$::DEBUG==1} {
                    tk_messageBox -type ok -icon error \
                        -message "Error deleting file:\n$errorMsg"
                }
            }
        }
    } else {
        if {$getleftState(autoDown)==0} {
            catch {wm withdraw $::Ventana::window(top)}
        }
        if {$noEnd==0} {
            tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                    -message $labelMessages(theEnd)
        }
        if {$index!=-1} {
            mainWin::UrlListDeleteEntry $getleftState(url)
        }
        if {[catch {file delete [file join $dirGetleft(conf) $domain.gdt]} errorMsg]} {
            if {$::DEBUG==1} {
                tk_messageBox -type ok -icon error \
                    -message "Error deleting file:\n$errorMsg"
            }
        }
    }
    SaveConfig
    catch {unset ::urlsDownloaded}
    catch {unset ::siteMap}

    if {$::getleftState(exiting)==1} {
        exit
    }
    if {$getleftState(autoDown)==1} {
        if {[llength $getleftState(urlQueue)]>0} {
            mainWin::NextToAutoDownload
        } else {
            catch {wm withdraw $::Ventana::window(top)}
            mainWin::SetAutoDownload
        }
    }
    if {$getleftState(delayedDown)==1} {
        set getleftState(delayedDown) 0
        if {$Delay::delayedDownload(days)==0} {
            set getleftState(delaySet) 0
        }
    }

    set getleftState(downloading) 0
    set getleftState(filesChosen) 0
    set getleftState(mapDone)     0

    catch {unset getleftState(mapLinks)}
    catch {unset ::siteMap}
    catch {unset ::HtmlParser::links}

    mainWin::ActivateTaskbarButtons

    return
}

###############################################################################
# AutomaticDownload
#    Resumes a download where the user told Getleft to leave it.
#
# Parameter:
#    url: The url to resume downloading.
###############################################################################
proc AutomaticDownload {requestedUrl} {
    global directories dirGetleft getleftState urlsDownloaded
    global labelMessages labelTitles

    if {$requestedUrl==""} return
    set domain [lindex [HtmlParser::ParseUrl $requestedUrl] 1]
    set domain [string tolower $domain]

    set getleftState(filesChosen) 0

    set optionsFile [file join $dirGetleft(conf) $domain.gdt]
    if {[file exists $optionsFile]} {
        if {[catch {source $optionsFile} errorMsg]} {
            if {$::DEBUG==1} {
                tk_messageBox -icon error -type ok -title Error \
                        -message "Error reading saved data:\n$errorMsg"
            }
        } else {
            mainWin::UrlListIncrResume $requestedUrl
        }
    } 
    StartDownload $requestedUrl

    return
}

###############################################################################
# PutUrlInQueue
#    Puts the url and all related data into urlsToDownload
#
# Parameters:
#    url: Url to download.
#    dir: Directory where it will be saved.
#    options: List with the options to use with the download.
###############################################################################
proc PutUrlInQueue {url dir options} {
    global getleftState urlsToDownload

    set urlsToDownload($url)     1
    set urlsToDownload($url,dir) $dir
    if {[lsearch $getleftState(urlQueue)  $url]==-1} {
        mainWin::UrlListEnterEntry        $url 0 $dir
        lappend getleftState(urlQueue)    $url
        mainWin::SetResumeSitesMenu
        set urlsToDownload($url,resume) 0
    }

    set urlsToDownload($url,options) $options

    return
}

################################################################################
# StartDownload
#    Gets all the data for a download from 'UrlsToDownload' and gets on with it.
#
# Parameter
#    requestedUrl: The url to download.
################################################################################
proc StartDownload {requestedUrl} {
    global getleftOptions downOptions siteIndex
    global directories dirGetleft getleftState urlsToDownload
    global labelMessages labelTitles

    if {![info exists urlsToDownload($requestedUrl)]} {
        tk_messageBox -icon error -title $labelTitles(error) -type ok \
                -message "No hay urlToDownload($requestedUrl)"
    }

    if {([info exists getleftOptions(useAuthProxy)])\
            &&($getleftOptions(useAuthProxy)==1)} {
        if {(![info exists getleftOptions(proxyPass)]) \
                ||($getleftOptions(proxyPass)=="")} {
            if {[::Herramientas::GetPass]==0} {
                return
            }
        }
    }

    set directories(base) $urlsToDownload($requestedUrl,dir)
    array set downOptions $urlsToDownload($requestedUrl,options)
    set getleftState(url) $requestedUrl
    incr urlsToDownload($requestedUrl,resume)

    set getleftState(downloading) 1

    mainWin::ActivateTaskbarButtons
    set requestedUrl  [InitializeDownload $requestedUrl]
    if {$requestedUrl==""} {
        TheEnd 1
        return
    }
    set localFileName [::Commands::UrlToFile $requestedUrl]
    if {($getleftOptions(stopFile)==1)||($downOptions(levels)==0)\
            ||(![regexp -nocase {\.html?$} $localFileName])} {
        TheEnd
        return
    }

    Descargar::Preprocessing $requestedUrl 0
    if {($getleftState(autoDown)==0)&&($getleftState(filesChosen)==0)} {
        if {$HtmlParser::nLinks==1} {
             tk_messageBox -icon info -title $labelTitles(noLinks) -type ok \
                     -message $labelMessages(noLinks)

            set getleftState(downloading) 0
            catch {wm withdraw $::Ventana::window(top)}
            TheEnd 1
            return
        }
        set getleftState(filesChosen) \
                [ChooseLinks::ChooseDownloads $requestedUrl HtmlParser::links]
    } else {
        if {$getleftState(autoDown)==0} {
            for {set i 1;set j 0} {$i<$HtmlParser::nLinks} {incr i;incr j} {
                if {($downOptions(map)==0)||($getleftState(mapDone)==0)} {
                    set HtmlParser::links($i,ok) [lindex $getleftState(chosenLinks) $j]
                } else {
                    set HtmlParser::links($i,ok) [lindex $getleftState(mapLinks) $j]
                }
            }
        } else {
            set getleftState(chosenLinks) ""
            for {set i 1;set j 0} {$i<$HtmlParser::nLinks} {incr i} {
                lappend getleftState(chosenLinks) $HtmlParser::links($i,ok)
            }
        }
        set getleftState(filesChosen) 1
    }
    if {$getleftState(filesChosen)==1} {
        if {$downOptions(map)==1} {
            set onlyHtmlTmp           $downOptions(onlyHtml)
            set downOptions(onlyHtml) 1
            set siteIndex             1
            if {![info exists getleftState(mapDone)]} {
                set getleftState(mapDone) 0
            }
        }
        #set ::start [clock seconds]
        Descargar::ControlDownloading $requestedUrl 1
        #set ::fin   [clock seconds]
        #puts "Time: [expr $::fin - $::start]"
        if {$downOptions(map)==0} {
            TheEnd $getleftState(autoDown)
        } else {
            if {$getleftState(autoDown)==0} {
                DownloadFromMap $requestedUrl
            } else {
                set getleftOptions(stopFile) 1
            }
            TheEnd
            catch "unset siteMap"
            catch "unset HtmlParser::links"
            set downOptions(onlyHtml) $onlyHtmlTmp
        }
        set getleftState(filesChosen) 0
    } else {
        set getleftOptions(stopFile) 1
        TheEnd
    }
    return
}

###############################################################################
# DownloadFromMap
#    Download the links chosen in the map of the site
#
# Parameter:
#    requestedUrl: The url the user entered to begin mapping
###############################################################################
proc DownloadFromMap {requestedUrl} {
    global siteIndex siteMap getleftOptions downOptions getleftState

    set downOptions(onlyHtml)  0
    set tmpLevels              $downOptions(levels)
    set downOptions(levels)    0

    if {$siteIndex==1} {
        return
    }

    if {$getleftState(mapDone)==0} {
        set getleftState(mapDone) 1
        HtmlParser::FilterLinks $requestedUrl ::siteMap
        set done [ChooseLinks::ChooseDownloads "" ::siteMap]
        if {$done==0} {
            set downOptions(levels)       $tmpLevels
            set downOptions(map)          0
            set getleftState(downloading) 0
            return
        }
    }

    for {set i 1} {$i<$siteIndex} {incr i} {
        if {$siteMap($i,ok)==1} {
            set enlace $siteMap($i,url)
            set filename [::Commands::UrlToFile $enlace]
            if {[file exists $filename]} {
                continue
            } else {
                set isHtml [regexp -nocase {\.html?$} $filename]
                Descargar::Download $enlace "-" $isHtml
            }
            if {$getleftOptions(stopFile)==1} {
                 set downOptions(levels) $tmpLevels
                 return
            }
            if {$getleftOptions(pauseFile)==1} {
                Ventana::Pause file
            }
        }
    }
    set downOptions(levels)   $tmpLevels
    set downOptions(map)      0
    set getleftState(mapDone) 0
    catch {unset getleftState(mapLinks)}

    return
}

###############################################################################
# SaveDownloadData
#     Saves the data needed to automatically resume a download.
#
# Parameter
#     url: The url we where downloading from.
###############################################################################
proc SaveDownloadData {url} {
    global dirGetleft getleftOptions downOptions siteUrl directories 
    global getleftState urlsDownloaded

    set domain [lindex [HtmlParser::ParseUrl $url] 1]
    set domain [string tolower $domain]
    set fileName [file join $dirGetleft(conf) $domain.gdt]
    catch {file delete -force "$fileName"}
    set file [open "$fileName" w]

    puts $file "# Data to resume downloading $domain"
    puts $file "# This file is generated automatically, please do not edit"
    puts $file "\n\n"
    puts $file "set getleftState(filesChosen)  $getleftState(filesChosen)"
    if {$getleftState(filesChosen)==1} {
        puts $file "set getleftState(mapDone)      $getleftState(mapDone)"
        puts $file "set getleftState(chosenLinks)  \[list $getleftState(chosenLinks)\]"

        if {[info exists getleftState(mapLinks)]} {
            puts $file "set getleftState(mapLinks) \[list $getleftState(mapLinks)\]"
        }
        if {$downOptions(map)==0} {
            puts $file "\n\n# Urls that have been already completed\n\n"
            foreach {url level} [array get urlsDownloaded] {
                regsub {\$} $url {\\$} url
                puts $file "set urlsDownloaded($url) $level"
            }
        }
        if {$downOptions(update)==1} {
            puts $file "\n\n# Urls that have been already updated\n\n"
            foreach {url nada} [array get Descargar::updated] {
                regsub {\$} $url {\\$} url
                puts $file "set \"Descargar::updated($url)\" 1"
            }
        }
    }
    puts $file "\n\n"
    close $file

    return
}

###############################################################################
# ReadConfig
#    Reads the configuration files
#
# Parameter:
#    Configuration file to read, if not present, the user configuration  file
#    will be read.
#
# Returns:
#    '0' if all went well, '1' if the file couldn't be found
###############################################################################
proc ReadConfig { {confFile {} } } {
    global urlHistory urlIndex dirGetleft getleftOptions urlsToDownload
    global getleftState defaultDownOptions env tcl_platform

    if {$confFile!=""} {
        set ficherorc $confFile
    } else {
        if {$tcl_platform(platform)=="windows"} {
            set ficherorc [file join "$dirGetleft(conf)" getleft.ini]
            set dirGetleft(lastDir) $dirGetleft(conf)
            set dirGetleft(toolDir) $dirGetleft(conf)
        } else {
            set ficherorc [file join "$dirGetleft(conf)" getleft.conf]
            set dirGetleft(lastDir) $env(HOME)
            set dirGetleft(toolDir) $env(HOME)
            set temp [file join $env(HOME) .Getleft]
            if {[file exists $temp]} {
                set ficherorc $temp
                set update 1
            }
        }
    }

    array set defaultDownOptions [list exLevels 0 resume 1 map 0 filter {} images 0 \
            onlyHtml 0 levels -1 external 0 update 0 cgi 0 dir 1]

    set getleftState(urlQueue) ""
    set getleftState(delaySet) 0
    set urlHistory ""
    if {[catch {open $ficherorc r} handle]} {
        if {![info exists getleftOptions(lang)]} {
            set getleftOptions(lang) en
        }
        set urlIndex 0
        return 1
    }

    for {} {![eof $handle]} {} {
        set line [gets $handle]
        if {[regexp {^#|^ |^$} $line]} {
            continue
        }
        if {[regexp {(?:\[)(.*)(?:\])} $line nada active]} {
            continue
        }
        switch -regexp -- $active {
            historial|urlHistory {
                 lappend urlHistory $line
            }
            urlIndex|hisIndex {
                set urlIndex $line
            }
            ^proxy$ {
                set getleftOptions(httpProxy) $line
                set getleftOptions(ftpProxy)  $line
                set getleftOptions(sameProxy) 1
                set getleftOptions(proxy)     1
            }
            httpProxy {
                set getleftOptions(httpProxy) $line
                set getleftOptions(proxy)     1
            }
            ftpProxy {
                set getleftOptions(ftpProxy)  $line
                set getleftOptions(proxy)     1
            }
            sameProxy {
                set getleftOptions(sameProxy) $line
            }
            proxyUser {
                set getleftOptions(proxyUser) $line
            }
            proxyPass {
                set getleftOptions(proxyPass) $line
            }
            useAuthProxy {
                set getleftOptions(useAuthProxy) $line
            }
            saveAuthPass {
                set getleftOptions(saveAuthPass) $line
            }
            ^language$ {
                set getleftOptions(lang) $line
            }
            scripts {
                set dirGetleft(scripts) $line
            }
            images {
                set dirGetleft(images) $line
            }
            doc {
                set dirGetleft(doc) $line
            }
            languages {
                set dirGetleft(languages) $line
            }
            lastDir {
                set dirGetleft(lastDir) $line
            }
            toolDir {
                set dirGetleft(toolDir) $line
            }
            urlQueue {
                lappend getleftState(urlQueue) $line
            }
            urlsToDownload {
                eval "$line"
            }
            delaySet {
                set getleftState(delaySet) $line
            }
            delayData {
                eval "$line"
            }
            defaultOptions {
                array set defaultDownOptions $line
            }
        }
    }

    close $handle
    if {[info exists update]} {
        file delete $ficherorc
        catch {file delete [file join $env(HOME) fileDown.log]}
        SaveConfig
    }
    if {![info exists getleftOptions(lang)]} {
        set getleftOptions(lang) en
    }
    if {![info exists getleftOptions(useAuthProxy)]} {
        set getleftOptions(useAuthProxy) 0
    }
    if {![info exists urlIndex]} {
        set urlIndex 0
    }
    if {$getleftState(delaySet)==1} {
        Delay::DelayedSetStartTime
    }
    # Since dirGetleft(toolDir) is a new thing it won't be in
    # many config files, so we give it a value just in case
    if {(![info exists dirGetleft(toolDir)])&&([info exists dirGetleft(lastDir)])} {
        set dirGetleft(toolDir) $dirGetleft(lastDir)
    }
    return 0
}

###############################################################################
# SaveConfig
#   Saves the configuration
###############################################################################
proc SaveConfig {} {
    global urlHistory urlIndex dirGetleft getleftOptions urlsToDownload
    global getleftState defaultDownOptions env tcl_platform
    global labelMessages labelTitles

    if {$tcl_platform(platform)=="windows"} {
        set ficherorc [file join "$dirGetleft(conf)" getleft.ini]

        # I got a bug report that said that Win2K doesn't like to
        # open hidden files, so I unhide it before opening
        if {[file exists $ficherorc]} {
            file attributes $ficherorc -hidden 0
        }
    } else {
        set ficherorc [file join "$dirGetleft(conf)" getleft.conf]
    }
    if {[catch {open  $ficherorc w+} fichero]} {
        tk_messageBox -title $labelTitles(error) -icon error \
            -message "$labelMessages(saveConfig)\n$fichero"
        return
    }

    puts $fichero "# Please do not edit unless you know what you are doing.\n\n"

    puts $fichero "\[urlHistory\]\n"
    if {![catch {llength $urlHistory} len]} {
        for {set i 0} {$i<$len} {incr i} {
            puts $fichero [lindex $urlHistory $i]
        }
    }

    puts $fichero "\n\[urlIndex\]\n"
    puts $fichero $urlIndex

    if {[info exists getleftOptions(httpProxy)]} {
        puts $fichero "\n\[httpProxy\]\n"
        puts $fichero "$getleftOptions(httpProxy)"
        puts $fichero "\n\[ftpProxy\]\n"
        puts $fichero "$getleftOptions(ftpProxy)"
        puts $fichero "\n\[sameProxy\]\n"
        puts $fichero "$getleftOptions(sameProxy)"
        if {$getleftOptions(useAuthProxy)==1} {
            if {$getleftOptions(saveAuthPass)==1} {
                puts $fichero "\n\[proxyPass\]\n"
                puts $fichero "$getleftOptions(proxyPass)"
            }
            puts $fichero "\n\[proxyUser\]\n"
            puts $fichero "$getleftOptions(proxyUser)"
            puts $fichero "\n\[useAuthProxy\]\n"
            puts $fichero "$getleftOptions(useAuthProxy)"
            puts $fichero "\n\[saveAuthPass\]\n"
            puts $fichero "$getleftOptions(saveAuthPass)"
        }
    }

    puts $fichero "\n\[language\]\n"
    puts $fichero "$getleftOptions(lang)"

    puts $fichero "\n\[lastDir\]\n"
    puts $fichero "$dirGetleft(lastDir)"

    puts $fichero "\n\[toolDir\]\n"
    puts $fichero "$dirGetleft(toolDir)"

    puts $fichero "\n\[delaySet\]\n"
    puts $fichero "$getleftState(delaySet)"

    puts $fichero "\n\[defaultOptions\]\n"
    puts $fichero [array get defaultDownOptions]

    if {$getleftState(delaySet)==1} {
        puts $fichero "\n\[delayData\]\n"
        puts $fichero "array set Delay::delayedDownload \
                      \{[array get Delay::delayedDownload]\}"
    }

    if {[llength $getleftState(urlQueue)]} {
        puts $fichero "\n\[urlQueue\]\n"
        foreach site $getleftState(urlQueue) {
            puts $fichero $site
        }
        puts $fichero "\n\[urlsToDownload\]\n"
        foreach site $getleftState(urlQueue) {
            puts $fichero "set urlsToDownload($site)         1"
            puts $fichero "set urlsToDownload($site,dir)     \"$urlsToDownload($site,dir)\""
            puts $fichero "set urlsToDownload($site,resume)  $urlsToDownload($site,resume)"
            puts $fichero "set urlsToDownload($site,options) [list $urlsToDownload($site,options)]"
        }
    }
    close $fichero

    if {$tcl_platform(platform)=="windows"} {
        file attributes $ficherorc -hidden 1
    } else {
        file attributes $ficherorc -permissions 0600
    }
      
    return
}

###############################################################################
# PutUrlInHistory
#    Put the given Url in the history of url
#
# Parameter
#    Puts, if need be, the given url in the history of urls.
###############################################################################
proc PutUrlInHistory {url} {
    global urlHistory urlIndex

    set listItems [llength $urlHistory]
    set newListItems [llength $Ccombinado::cbArgs(items)]
    if {$listItems!=$newListItems} {
        set urlHistory $Ccombinado::cbArgs(items)
        set urlIndex   $newListItems
        SaveConfig
    }

    if {![EnHistorial $url]} {
        if {[llength $urlHistory]<10} {
            lappend urlHistory $url
        } else {
            set urlHistory [lreplace $urlHistory $urlIndex $urlIndex $url]
        }
        if {$urlIndex==9} {
            set urlIndex 0
        } else {
            incr urlIndex
        }
        SaveConfig
    }
    return
}

###############################################################################
# MostrarHistorial
#    Shows the url history
###############################################################################
proc MostrarHistorial {} {
    global urlHistory urlIndex

    set len [llength $urlHistory]
    for {set i 0} {$i<$len} {incr i} {
        puts "$i: [lindex $urlHistory $i]"
    }
    return
}

###############################################################################
# EnHistorial
#    Checks whether the given url is already in the url history
#
# Parameter:
#    pagina: given url
#
# Returns:
#    1 if it is.
#    0 if it is not
###############################################################################
proc EnHistorial {pagina} {
    global urlHistory

    set len [llength $urlHistory]
    for {set i 0} {$i<$len} {incr i} {
        if {[catch {regsub {(http://)||(ftp://)} \
                [lindex $urlHistory $i] {} una}]} {
            return 0
        }
        regsub {(http://)||(ftp://)} $pagina {} dos

        if {[string match $una $dos]} {
            return 1
        }
    }
    return 0
}

###############################################################################
# ChangeLanguage
#    Changes the default language for the program
#
# Parameters:
#    lang: language to be used
###############################################################################
proc ChangeLanguage {lang} {
    global getleftOptions

    set getleftOptions(lang) $lang

    wm withdraw .

    mainWin::MainWindow

    wm deiconify .

    SaveConfig

    return
}

###############################################################################
# ReadSupportedLang
#    Reads the 'supported' file to know which languages this version supports
#    and puts then in supportedLang global variable
###############################################################################
proc ReadSupportedLang {} {
    global dirGetleft supportedLang getleftState
    global labelTitles

    if {[catch {open [file join $dirGetleft(languages) supported] r} handle]} {
        tk_messageBox -type ok -icon error -title Error -message $handle
        return
    }
    set encoding cp1252
    fconfigure $handle -encoding $encoding
    for {set supportedLang(langList) ""} {![eof $handle]} {} {
        set line [gets $handle]
        if {[regexp {^#|^ |^$} $line]} {
            continue
        }
        set key      [lindex $line 0]
        set encoding [lindex $line 2]
        if {[catch {encoding convertfrom $encoding [lindex $line 1]} language]} {
            continue
        }
        set supportedLang($key)         $encoding
        set supportedLang($language)    $key
        lappend supportedLang(langList) $language
    }
    set supportedLang(langList) [lsort $supportedLang(langList)]

    close $handle
    return
}

###############################################################################
# ReadLangFile
#    Reads one of the files with translations
#
# Parameter
#    The two letters code that identifies one language, i.e., 'es', 'en'
#
# Returns
#    - '0' all went well
#    - '1' there was an error
###############################################################################
proc ReadLangFile {sufix} {
    global dirGetleft supportedLang
    global labelMenus labelButtons  labelTitles labelMessages
    global labelDialogs labelMonths labelFrames indexMenus
    global labelCurlCodes confWizardLabels

    msgcat::mclocale $sufix
    msgcat::mcload   [file join $::tk_library msgs]

    set languageFile [file join $dirGetleft(languages) menus.$sufix]

    if {[catch {open $languageFile r} handle]} {
        tk_messageBox -icon error -type ok -title Error \
                -message "File not found"
        return 1
    }
    fconfigure $handle -encoding $supportedLang($sufix)

    for {set active ""} {![eof $handle]} {} {
        set line [gets $handle]
        if {[regexp {^#|^ |^$} $line]} {
            continue
        }
        if {![string compare $line "End"]} {
            set active ""
            continue
        }
        set key   [lindex $line 0]
        set value [lindex $line 1]

        switch -exact -- $active {
            Menus {
                set indexMenus($key) [string first & $value]
                if {$indexMenus($key)!=-1} {
                    regsub {&} $value {} value
                }
                set labelMenus($key) $value
            }
            Buttons {
                set labelButtons($key) $value
            }
            Titles {
                set labelTitles($key) $value
            }
            Messages {
                set labelMessages($key) $value
            }
            Dialogs {
                set labelDialogs($key) $value
            }
            Months {
                set labelMonths($key)   $value
                set labelMonths($value) $key
            }
            Frames {
                set labelFrames($key) $value
            }
            curlCodes {
                set labelCurlCodes($key) $value
            }
            confWizard {
                set confWizardLabels($key) $value
            }
            "" {
                set active $line
            }
        }
    }
    close $handle
    return 0
}

###############################################################################
# SetEncodings
#    Sets the array that maps the encoding names used in http to the
#    equivalent Tcl encodings.
###############################################################################
proc SetEncodings {} {
    global charSets

    set charSets(ansi_x3.4-1968)        ascii
    set charSets(iso-ir-6)              ascii
    set charSets(iso_646.irv:1991)      ascii
    set charSets(iso646-us)             ascii
    set charSets(us-ascii)              ascii
    set charSets(us)                    ascii
    set charSets(ibm367)                ascii
    set charSets(cp367)                 ascii
    set charSets(ascii)                 ascii

    set charSets(big5)                  big5

    set charSet(dingbats)               dingbats

    set charSets(ks_c_5601-1987)        ksc5601
    set charSets(iso-ir-149)            ksc5601
    set charSets(ks_c_5601-1989)        ksc5601
    set charSets(ksc_5601)              ksc5601
    set charSets(korean)                ksc5601

    set charSets(euc-cn)                euc-cn
    set charSets(euc-jp)                euc-jp
    set charSets(euc-kr)                euc-kr

    set charSets(gb12345)               gb12345
    set charSets(gb1988)                gb1988
    set charSets(gb2312)                gb2312

    set charSets(iso-2022)              iso2022
    set charSets(iso-2022-jp)           iso2022-jp
    set charSets(iso2022-jp)            iso2022-jp
    set charSets(iso-2022-kr)           iso2022-kr

    set charSets(iso_8859-1:1987)       iso8859-1
    set charSets(iso-ir-100)            iso8859-1
    set charSets(iso_8859-1)            iso8859-1
    set charSets(iso-8859-1)            iso8859-1
    set charSets(latin1)                iso8859-1
    set charSets(l1)                    iso8859-1
    set charSets(ibm819)                iso8859-1
    set charSets(cp819)                 iso8859-1
    set charSets(iso8859-1)             iso8859-1

    set charSets(iso_8859-2:1987)       iso8859-2
    set charSets(iso-ir-101)            iso8859-2
    set charSets(iso_8859-2)            iso8859-2
    set charSets(iso-8859-2)            iso8859-2
    set charSets(latin2)                iso8859-2
    set charSets(l2)                    iso8859-2
    set charSets(iso8859-2)             iso8859-2

    set charSets(iso_8859-3:1988)       iso8859-3
    set charSets(iso-ir-109)            iso8859-3
    set charSets(iso_8859-3)            iso8859-3
    set charSets(iso-8859-3)            iso8859-3
    set charSets(latin3)                iso8859-3
    set charSets(iso8859-3)             iso8859-3

    set charSets(iso_8859-4:1988)       iso8859-4
    set charSets(iso-ir-110)            iso8859-4
    set charSets(iso_8859-4)            iso8859-4
    set charSets(iso-8859-4)            iso8859-4
    set charSets(latin4)                iso8859-4
    set charSets(l4)                    iso8859-4
    set charSets(iso8859-4)             iso8859-4

    set charSets(iso_8859-5:1988)       iso8859-5
    set charSets(iso-ir-144)            iso8859-5
    set charSets(iso_8859-5)            iso8859-5
    set charSets(iso-8859-5)            iso8859-5
    set charSets(cyrillic)              iso8859-5
    set charSets(iso8859-5)             iso8859-5

    set charSets(iso_8859-6:1987)       iso8859-6
    set charSets(iso-ir-127)            iso8859-6
    set charSets(iso_8859-6)            iso8859-6
    set charSets(iso-8859-6)            iso8859-6
    set charSets(ecma-114)              iso8859-6
    set charSets(asmo-708)              iso8859-6
    set charSets(arabic)                iso8859-6
    set charSets(iso8859-6)             iso8859-6

    set charSets(iso_8859-7:1987)       iso8859-7
    set charSets(iso-ir-126)            iso8859-7
    set charSets(iso_8859-7)            iso8859-7
    set charSets(iso-8859-7)            iso8859-7
    set charSets(elot_928)              iso8859-7
    set charSets(ecma-118)              iso8859-7
    set charSets(greek)                 iso8859-7
    set charSets(greek8)                iso8859-7
    set charSets(iso8859-7)             iso8859-7

    set charSets(iso_8859-8:1988)       iso8859-8
    set charSets(iso-ir-138)            iso8859-8
    set charSets(iso_8859-8)            iso8859-8
    set charSets(iso-8859-8)            iso8859-8
    set charSets(hebrew)                iso8859-8
    set charSets(iso8859-8)             iso8859-8

    set charSets(iso_8859-9:1989)       iso8859-9
    set charSets(iso-ir-148)            iso8859-9
    set charSets(iso_8859-9)            iso8859-9
    set charSets(iso-8859-9)            iso8859-9
    set charSets(latin5)                iso8859-9
    set charSets(l5)                    iso8859-9
    set charSets(iso8859-9)             iso8859-9

    set charSets(jis_x0212-1990)        jis0212
    set charSets(x0212)                 jis0212
    set charSets(iso-ir-159)            jis0212
    set charSets(jis0212)               jis0212
    set charSets(shiftjis)              shiftjis

    set charSets(jis_x0201)             jis0201
    set charSets(x0201)                 jis0201
    set charSets(jis0201)               jis0201

    set charSets(jis_c6226-1983)        jis0208
    set charSets(iso-ir-87)             jis0208
    set charSets(x0208)                 jis0208
    set charSets(jis_x0208-1983)        jis0208
    set charSets(jis0208)               jis0208

    set charSets(ibm437)                cp437
    set charSets(cp437)                 cp437
    set charSets(437)                   cp437
    set charSets(windows-437)           cp437

    set charSets(ibm737)                cp737
    set charSets(cp737)                 cp737
    set charSets(737)                   cp737
    set charSets(windows-737)           cp737

    set charSets(ibm775)                cp775
    set charSets(cp775)                 cp775
    set charSets(775)                   cp775
    set charSets(windows-775)           cp775

    set charSets(ibm850)                cp850
    set charSets(cp850)                 cp850
    set charSets(850)                   cp850
    set charSets(windows-850)           cp850

    set charSets(ibm852)                cp852
    set charSets(cp852)                 cp852
    set charSets(852)                   cp852
    set charSets(windows-852)           cp852

    set charSets(ibm855)                cp855
    set charSets(cp855)                 cp855
    set charSets(855)                   cp855
    set charSets(windows-855)           cp855

    set charSets(ibm857)                cp857
    set charSets(cp857)                 cp857
    set charSets(857)                   cp857
    set charSets(windows-857)           cp857

    set charSets(ibm860)                cp860
    set charSets(cp860)                 cp860
    set charSets(860)                   cp860
    set charSets(windows-860)           cp860

    set charSets(ibm861)                cp861
    set charSets(cp861)                 cp861
    set charSets(861)                   cp861
    set charSets(cp-is)                 cp861
    set charSets(windows-861)           cp861

    set charSets(ibm862)                cp862
    set charSets(cp862)                 cp862
    set charSets(862)                   cp862
    set charSets(windows-862)           cp862

    set charSets(ibm863)                cp863
    set charSets(cp863)                 cp863
    set charSets(863)                   cp863
    set charSets(windows-863)           cp863

    set charSets(ibm864)                cp864
    set charSets(cp864)                 cp864
    set charSets(864)                   cp864
    set charSets(windows-864)           cp864

    set charSets(ibm865)                cp865
    set charSets(cp865)                 cp865
    set charSets(865)                   cp865
    set charSets(windows-865)           cp865

    set charSets(ibm866)                cp866
    set charSets(cp866)                 cp866
    set charSets(866)                   cp866
    set charSets(windows-866)           cp866

    set charSets(ibm869)                cp869
    set charSets(cp869)                 cp869
    set charSets(869)                   cp869
    set charSets(cp-gr)                 cp869
    set charSets(windows-869)           cp869

    set charSets(ibm874)                cp874
    set charSets(cp874)                 cp874
    set charSets(874)                   cp874
    set charSets(windows-874)           cp874

    set charSets(ibm932)                cp932
    set charSets(cp932)                 cp932
    set charSets(932)                   cp932
    set charSets(windows-932)           cp932

    set charSets(ibm936)                cp936
    set charSets(cp936)                 cp936
    set charSets(936)                   cp936
    set charSets(windows-936)           cp936

    set charSets(ibm949)                cp949
    set charSets(cp949)                 cp949
    set charSets(949)                   cp949
    set charSets(windows-949)           cp949

    set charSets(ibm950)                cp950
    set charSets(cp950)                 cp950
    set charSets(950)                   cp950
    set charSets(windows-950)           cp950

    set charSets(ibm1250)               cp1250
    set charSets(cp1250)                cp1250
    set charSets(1250)                  cp1250
    set charSets(windows-1250)          cp1250

    set charSets(ibm1251)               cp1251
    set charSets(cp1251)                cp1251
    set charSets(1251)                  cp1251
    set charSets(windows-1251)          cp1251

    set charSets(ibm1252)               cp1252
    set charSets(cp1252)                cp1252
    set charSets(1252)                  cp1252
    set charSets(windows-1252)          cp1252

    set charSets(ibm1253)               cp1253
    set charSets(cp1253)                cp1253
    set charSets(1253)                  cp1253
    set charSets(windows-1253)          cp1253

    set charSets(ibm1254)               cp1254
    set charSets(cp1254)                cp1254
    set charSets(1254)                  cp1254
    set charSets(windows-1254)          cp1254

    set charSets(ibm1255)               cp1255
    set charSets(cp1255)                cp1255
    set charSets(1255)                  cp1255
    set charSets(windows-1255)          cp1255

    set charSets(ibm1256)               cp1256
    set charSets(cp1256)                cp1256
    set charSets(1256)                  cp1256
    set charSets(windows-1256)          cp1256

    set charSets(ibm1257)               cp1257
    set charSets(cp1257)                cp1257
    set charSets(1257)                  cp1257
    set charSets(windows-1257)          cp1257

    set charSets(ibm1258)               cp1258
    set charSets(cp1258)                cp1258
    set charSets(1258)                  cp1258
    set charSets(windows-1258)          cp1258

    set charSets(koi8-r)                koi8-r
    set charSets(koi8-u)                koi8-u
    set charSets(ksc5601)               ksc5601

    set charSets(maccenteuro)           macCentEuro
    set charSets(maccroatian)           macCroatian
    set charSets(maccyrillic)           macCyrillic
    set charSets(macdingbats)           macDingbats
    set charSets(macgreek)              macGreek
    set charSets(maciceland)            macIceland
    set charSets(macjapan)              macJapan
    set charSets(macroman)              macRoman
    set charSets(macromania)            macRomania
    set charSets(macthai)               macThai
    set charSets(macturkish)            macTurkish
    set charSets(macukraine)            macUkraine

    set charSets(tis620)                tis-620
    set charSets(tis-620)               tis-620

    set charSets(unicode)               unicode
    set charSets(utf-8)                 utf-8

    return
}

###############################################################################
# ExitGetleft
#    This procedure is invoked when the user clicks to close the main window,
#    or in the 'Exit' menu entry, it checks whether the user is fooling around.
###############################################################################
proc ExitGetleft {} {
    global getleftState

    if {$getleftState(downloading)==1} {
        set getleftState(exiting) 1
        if {[Ventana::CancelDownloading $Ventana::window(top)]==0} {
            return
        }
        TheEnd
    }
    SaveConfig
    getLog::UrlLogWrite

    exit

    return
}

###############################################################################
#                                PROGRAMA PRINCIPAL
###############################################################################

set DEBUG 0

package require Tk

wm withdraw .

CheckPrograms

package require msgcat

set dirGetleft(main) [file dirname [info script]]

if {$tcl_platform(platform)=="windows"} {
    if {[info exists env(USERPROFILE)]} {
        set dirGetleft(conf) [file join $env(USERPROFILE) getleft]
        if {![file exists $dirGetleft(conf)]} {
            if {[catch {file mkdir $dirGetleft(conf)} result]} {
                tk_messageBox -icon error -type ok -title Error \
                        -message "Couldn't create directory\$result"
                exit
            }
        }
        # Since using the USERPROFILE folder is a new thing
        # we will have to help the users who have their data
        # in other places
        if {[file exists [file join $dirGetleft(main) getleft.ini]]} {
            set fileList [list getlefti.ini cookies.txt Geterror.log getLog.log]
            set fileList [concat $fileList \
                    [glob -nocomplain [file join $dirGetleft(main) *.gdt]]]
            foreach file $fileList {
                catch {file rename [file join $dirGetleft(main) $file] \
                        $dirGetleft(conf)}
            }
        }
    } else {
       if {![catch {librarypath}]} {
            set dirGetleft(conf) [pwd]
       } else {
            set dirGetleft(conf) $dirGetleft(main)
       }
    }
} else {
    set dirGetleft(conf) [file join $env(HOME) .getleft]
}

if {![file exists $dirGetleft(conf)]} {
    if {[catch {file mkdir $dirGetleft(conf)} error]} {
        tk_messageBox -title Error -icon error -type ok \
                -message "Couldn't create dir:\n$error"
        exit
    }
}

if {[file exists /etc/getlet.conf]} {
    ReadConfig /etc/getleft.conf
} elseif {[file exists /usr/local/etc/getleft.conf]} {
    ReadConfig /usr/local/etc/getleft.conf
} else {
    set dirGetleft(scripts)   [file join "$dirGetleft(main)" scripts]
    set dirGetleft(images)    [file join "$dirGetleft(main)" images]
    set dirGetleft(doc)       [file join "$dirGetleft(main)" doc]
    set dirGetleft(languages) [file join "$dirGetleft(main)" Languages]
}

source [file join "$dirGetleft(scripts)" mwutil.tcl]
source [file join "$dirGetleft(scripts)" tablelistWidget.tcl]
source [file join "$dirGetleft(scripts)" tablelistSortByColumn.tcl]
source [file join "$dirGetleft(scripts)" balloon.tcl]
source [file join "$dirGetleft(scripts)" mainWin.tcl]
source [file join "$dirGetleft(scripts)" delay.tcl]
source [file join "$dirGetleft(scripts)" Ventana.tcl]

if {![info exists getleftState(downloading)]} {
    set getleftState(downloading)             0
    set getleftState(mapDone)                 0
    set getleftState(exiting)                 0
    set getleftState(autoDown)                0
    set getleftState(startNow)                1
    set getleftState(delayedDown)             0
    set getleftState(waiting)                 0
    set getleftState(filesChosen)             0
    set ::Ventana::Rizo::curlReport(nextFile) 0
    set setIcon 1
}

SetGetleftColours

ReadSupportedLang
ReadLangFile en
SetEncodings

source [file join "$dirGetleft(scripts)" FrameLabel.tcl]
source [file join "$dirGetleft(scripts)" Herramientas.tcl]

if {[ReadConfig]} {
    source [file join "$dirGetleft(scripts)" getWizard.tcl]
}

mainWin::MainWindow
update

source [file join "$dirGetleft(scripts)" menuEntry.tcl]
source [file join "$dirGetleft(scripts)" Ccombinado.tcl]
source [file join "$dirGetleft(scripts)" Dialogos.tcl]
source [file join "$dirGetleft(scripts)" Ayuda.tcl]
source [file join "$dirGetleft(scripts)" HtmlParser.tcl]
source [file join "$dirGetleft(scripts)" Commands.tcl]
source [file join "$dirGetleft(scripts)" Opciones.tcl]
source [file join "$dirGetleft(scripts)" CuadroSpin.tcl]
source [file join "$dirGetleft(scripts)" Descargar.tcl]
source [file join "$dirGetleft(scripts)" Progreso.tcl]
source [file join "$dirGetleft(scripts)" Cookies.tcl]
source [file join "$dirGetleft(scripts)" ChooseLinks.tcl]
source [file join "$dirGetleft(scripts)" getLog.tcl]
source [file join "$dirGetleft(scripts)" help.tcl]
source [file join "$dirGetleft(scripts)" EnterUrl.tcl]

Ventana::Rizo::SetCurlVersion

Opciones::SetDefaultOptions downOptions

# We are going to delete the *.gdt files that shouldn't be there any more
set gdtFiles [glob -nocomplain [file join $dirGetleft(conf) *.gdt]]
foreach file $gdtFiles {
    set root [file rootname [file tail $file]]
    if {![regexp -nocase "$root" $getleftState(urlQueue)]} {
        file delete -force $file
    }
}

wm deiconify .

tkwait visibility .

if {$setIcon==1} {
    if {$tcl_platform(platform)=="windows"} {
        if {[regexp {((8\.3\.)(3|4))||(8.4)} $tcl_patchLevel]} {
            catch {wm iconbitmap . -default [file join $dirGetleft(images) icon.ico]}
        }
    }
    set setIcon 0
}


