###############################################################################
###############################################################################
##                          menuEntry
###############################################################################
###############################################################################
## Creates an entry widget with right-click context menu.
###############################################################################
###############################################################################
## (c) 2002 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the LGPL v2
###############################################################################
###############################################################################

namespace eval menuEntry {

###############################################################################
# ReadSel
#    Reads the selection from the clipboard
#
# Returns
#    Whatever was in the clipboard
###############################################################################
proc ReadSel {} {
    global tcl_platform

    if {$tcl_platform(platform)=="unix"} {
        if {![catch {selection get} sel]} {
            return $sel
        }
    }
    if {[catch {selection get -selection CLIPBOARD} sel]} return

    return $sel
}

###############################################################################
# PopUpMenu
#    Manages the menu that pops up when you right-click in the entry.
#
# Parameters:
#    menu: Path of the menu.
#    entry: Path of the entry
#    X-Y: Coordinates where the menu should appear.
###############################################################################
proc PopUpMenu {menu entry X Y} {

    set sel [ReadSel]
    if {$sel==""} {
        $menu entryconfigure 2 -state disabled
    } else {
        $menu entryconfigure 2 -state normal
    }
    if {[$entry select present]} {
        $menu entryconfigure 0 -state normal
        $menu entryconfigure 1 -state normal
        $menu entryconfigure 3 -state normal
    } else {
        $menu entryconfigure 0 -state disabled
        $menu entryconfigure 1 -state disabled
        $menu entryconfigure 3 -state disabled
    }
    if {[string length [$entry get]]==0} {
        $menu entryconfigure 5 -state disabled
    } else {
        $menu entryconfigure 5 -state normal
    }

    tk_popup $entry.menu $X $Y

    return
}
###############################################################################
# Cut - Copy - Paste - Clear - Selec
#    They do what they are supposed to.
#
# Parameters:
#    entryPath: Path of the entry
###############################################################################
proc Cut {entryPath} {
    if {[$entryPath selection present]} {
        event generate $entryPath <<Cut>>
    }
    return
}

proc Copy {entryPath} {
    if {[$entryPath selection present]} {
        event generate $entryPath <<Copy>>
    }
    return
}

proc Paste {entryPath} {
    event generate $entryPath <<Paste>>
    return
}

proc Clear {entryPath} {
    if {[$entryPath selection present]} {
        event generate $entryPath <<Clear>>
    }
    return
}

proc Selec {entryPath} {
    $entryPath select range 0 end
    clipboard clear
    return
}

###############################################################################
# FocusIn
#    When the entry gets the focus, this procedure will select the whole
#    content.
###############################################################################
proc FocusIn {entryPath} {
    variable entries

    if {$entries($entryPath)==0} {
        set entries($entryPath) 1
        $entryPath selection range 0 end
    }

    return
}

###############################################################################
# FocusOut
#    When the entry loses the focus, this procedure will prepare for the next
#    time it gets it.
###############################################################################
proc FocusOut {entryPath} {
    variable entries

    set entries($entryPath) 0

    return
}

###############################################################################
# RightClick
#    The following mess makes sure the insert cursor is placed in the
#    character where you right-click, all the while keeping the selection
#    if you had one.
###############################################################################
proc RightClick {entryPath x y X Y} {
    variable sel1
    variable sel2

    if {[$entryPath selection present]} {
        set selp 1
        set sel1 [$entryPath index sel.first]
        set sel2 [$entryPath index sel.last]
    } else {
        set selp 0
    }
    event generate $entryPath <Button-1>        -x $x -y $y
    event generate $entryPath <ButtonRelease-1> -x $x -y $y
    if {$selp==1} {
        $entryPath selection range $sel1 $sel2
    }
    menuEntry::PopUpMenu $entryPath.menu $entryPath $X $Y

    return
}

###############################################################################
# menuEntry
#    Creates the entry widget.
#
# Parameter:
#    entryPath: The path to the entry.
#    args: The arguments that will be passed whole to the standard entry.
#
# Returns:
#    The path of the entry
###############################################################################
proc menuEntry {entryPath args} {
    global labelMenus
    variable entries

    entry $entryPath
    eval $entryPath configure $args

    set menu [menu $entryPath.menu -tearoff 0]
    $menu add command -label $labelMenus(cut)       \
            -command "menuEntry::Cut   $entryPath"
    $menu add command -label $labelMenus(copy)      \
            -command "menuEntry::Copy  $entryPath"
    $menu add command -label $labelMenus(paste)     \
            -command "menuEntry::Paste $entryPath"
    $menu add command -label $labelMenus(clear)     \
            -command "menuEntry::Clear $entryPath"
    $menu add separator
    $menu add command -label $labelMenus(selectAll) \
            -command "menuEntry::Selec $entryPath"

    bind $entryPath <Button-3> "menuEntry::RightClick %W %x %y %X %Y"
    if {$::tcl_platform(platform)!="windows"} {
        bind $entryPath <Control-v> "
            event generate %W <<Paste>>
            break
        "
        bind $entryPath <ButtonRelease-2> break
    }
    bind $entryPath <Button-2> "
        event generate %W <<Paste>>
        break
    "
    bind $entryPath <Control-V> "
        event generate %W <<Paste>>
        break
    "
    bind $entryPath <FocusIn>  "menuEntry::FocusIn  $entryPath"
    bind $entryPath <FocusOut> "menuEntry::FocusOut $entryPath"

    set entries($entryPath) 0

    return $entryPath
}


}
