###############################################################################
###############################################################################
#####                            Progreso.tcl
###############################################################################
##### Namespace for creating and managing progress bars.
###############################################################################
##### Copyright 2001 Andrs Garca Garca - fandom@retemail.es
##### Distributed under the terms of the LGPL v2
###############################################################################

namespace eval Progreso {

###############################################################################
# ProgressBar
#    Creates the progress bar.
#
# Parameter
#    path: path of the progress bar.
#    args: list with the parameters for the bar:
#            - width: Default 275 pixels.
#            - height: Default 13 pixels.
#            - numbers: whether to show the percentage or not, default: '1'.
#            - bg: background color, defaults to white.
#            - fg: color that fills the bar, defaults to blue.
#
# Returns
#    path of the progress bar.
###############################################################################
proc ProgressBar {path args} {
    variable pbArgs

    ParseArguments $path $args

    set barFrame [frame $path]
    set whiteCanvas [canvas $barFrame.white -bg $pbArgs($path,-bg)      \
            -highlightthickness 0 -bd 2 -relief sunken                  \
            -width $pbArgs($path,-width) -height $pbArgs($path,-height)]
    set blueCanvas  [canvas $barFrame.blue  -bg $pbArgs($path,-fg)      \
            -highlightthickness 0]

    pack $whiteCanvas
    update

    return $barFrame
}

###############################################################################
# ParseArguments
#    Gets the optional parameters passed to the progress bar into the
#    namespace variable 'pbArgs', the rest get the default values.
###############################################################################
proc ParseArguments {path parameters} {
    variable pbArgs

    set pbArgs($path,-width)   275
    set pbArgs($path,-height)  12
    set pbArgs($path,-numbers) 1
    set pbArgs($path,-bg)      white
    set pbArgs($path,-fg)      blue
    if {$::tcl_platform(platform)=="windows"} {
        set pbArgs($path,-font) {"MS Sans Serif" 8}
    } else {
        set pbArgs($path,-font) {"Helvetica" 12}
    }

    foreach {parameter value} $parameters {
        set pbArgs($path,$parameter) $value
    }

    return
}

###############################################################################
# ProgressBarUpdate
#    Updates the progress bar.
#
# Parameter
#    path of the progress bar.
#    percen: porcentage of the task that has already been completed.
###############################################################################
proc ProgressBarUpdate {path percen} {
    variable pbArgs

    set pbArgs($path,-width) [winfo width $path.white]
    set done [expr {($pbArgs($path,-width)-4) * $percen/100}]
    if {$pbArgs($path,-numbers)==1} {
        set pbArgs($path,blackX) [expr {int($pbArgs($path,-width)/2)+2}]
        set pbArgs($path,blackY) [expr {int($pbArgs($path,-height)/2)+2}]
        set pbArgs($path,whiteX) [expr {int($pbArgs($path,-width)/2)}]
        set pbArgs($path,whiteY) [expr {int($pbArgs($path,-height)/2)}]
    }
    place $path.blue -in $path.white -x 2 -y 2 -bordermode inside \
            -width $done -height $pbArgs($path,-height)

    if {$pbArgs($path,-numbers)==0} return

    if {[info exists pbArgs($path,blackNumber)]} {
        $path.white delete $pbArgs($path,blackNumber)
        $path.blue  delete $pbArgs($path,whiteNumber)
    }
    set pbArgs($path,blackNumber) [$path.white create text \
            $pbArgs($path,blackX) $pbArgs($path,blackY)    \
            -text $percen% -fill black -anchor c -font $pbArgs($path,-font)]
    set pbArgs($path,whiteNumber) [$path.blue  create text \
            $pbArgs($path,whiteX) $pbArgs($path,whiteY)    \
            -text $percen% -fill white -anchor c -font $pbArgs($path,-font)]

    return
}

###############################################################################
# ProgressBarReset
#    Puts the progress bar back to 0%.
#
# Parameter
#    path of the progress bar.
###############################################################################
proc ProgressBarReset {path} {
    variable pbArgs

    place forget $path.blue
    catch {$path.white delete $pbArgs($path,blackNumber)}

    return
}

###############################################################################
# ProgressBarFull
#    Puts the progress bar at 100%.
#
# Parameter
#    path of the progress bar.
###############################################################################
proc ProgressBarFull {path} {

    ProgressBarUpdate $path 100

    return
}
}

###############################################################################
###############################################################################
#######                            Example
###############################################################################
###############################################################################
#proc StartCount {path percen} {
#    global progressId
#    Progreso::ProgressBarUpdate $path $percen
#    if {$percen<100} {
#        set progressId [after 250 "StartCount $path [incr percen 1]"]
#    }
#    return
#}
#set barFrame [frame .barFrame -relief groove -bd 2]
#set progreso [Progreso::ProgressBar $barFrame.progreso -width 300 -height 35 \
		-font {"Times new Roman" 20 bold}]
#set butFrame [frame .butFrame]
#set start  [button $butFrame.start  -text Start  -width 6\
        -command "StartCount $progreso 1"]
#set reset  [button $butFrame.reset  -text Reset  -width 6\
        -command "Progreso::ProgressBarReset $progreso"]
#set full   [button $butFrame.full   -text Full   -width 6\
        -command "Progreso::ProgressBarFull $progreso"]
#set cancel [button $butFrame.cancel -text Cancel -width 6\
        -command {after cancel $progressId}]

#pack $barFrame -padx 10 -pady 5
#pack $progreso -padx 10 -pady 5
#pack $butFrame -padx 7 -fill x
#pack $start $reset $full $cancel -side right -padx 3
###############################################################################
###############################################################################

