###############################################################################
###############################################################################
##                                CuadroSpin.tcl
###############################################################################
###############################################################################
## Creates a SpinBox. Next version of Tcl/Tk will have its own SpinBox widget,
## but up until then we need this.
###############################################################################
###############################################################################
## (c) 1997-2001 Eliseo Vergara, Andrs Garca Garca. fandom@retemail.es
## As with the combobox I don't know the whole story of this widget, Eliseo
## pass it to me at college, but he may not have done it completely himself.
##
## You may distribute the contents of this file under the terms of the LGPL v2
###############################################################################
###############################################################################

namespace eval CuadroSpin {

###############################################################################
# Before defining the CuadroSpin we are going to define the shape of the arrow
# points 'IncrValue' and 'DecValue'.
# If you would like to change the shape yourself you only have to save
# the numbers inside the '{}' in a file with the 'XBM' extension and
# edit it with a drawing program.
###############################################################################

set DecValue {
    #define abajo_width  9
    #define abajo_height 6
    static unsigned char abajo_bits[] = {
        0x00, 0x00, 0xfe, 0x00,
        0x7c, 0x00, 0x38, 0x00,
        0x10, 0x00, 0x00, 0x00
    };
}

image create bitmap DecValue -data [set DecValue]
unset DecValue

set IncrValue {
    #define arriba_width  9
    #define arriba_height 6
    static unsigned char arriba_bits[] = {
        0x00, 0x00, 0x10, 0x00,
        0x38, 0x00, 0x7c, 0x00,
        0xfe, 0x00, 0x00, 0x00
    };
}
image create bitmap IncrValue -data [set IncrValue]
unset IncrValue

##############################################################################
# EnableSpin
#    Enables or disables a spinbox.
#
# Parameter
#    spinPath: The path to the CuadroSpin.
#    which:    "normal" or "disable"
##############################################################################
proc EnableSpin {spinPath which} {
    variable csArgs

    $spinPath.e       configure -state $which
    if {$which=="disable"} {
        $spinPath.e   configure -bg $csArgs(-disbg)
        $spinPath.e   configure -fg $csArgs(-disfg)
    } else {
        $spinPath.e   configure -bg $csArgs(-bg)
        $spinPath.e   configure -fg $csArgs(-fg)
    }
    $spinPath.bg.incr configure -state $which
    $spinPath.bg.dec  configure -state $which

    return
}

##############################################################################
# IncrSpin
#    Increments or decrements the content of the spinBox by one.
#
# Parameter
#    spinPath: The path to the CuadroSpin.
#    max:      Maximun value allowed for the CuadroSpin
#    min:      Minumun value allowed for the CuadroSpin.
##############################################################################
proc IncrSpin {spinPath min max value} {
    variable CuadroSpins

    if {![regexp {^[-0-9]+$} $CuadroSpins($spinPath)]} {
        return
    }

    if {$value>0} {
        set tmp [expr $CuadroSpins($spinPath)+$CuadroSpins($spinPath,incr)]
        if {$tmp<$max} {
            set CuadroSpins($spinPath)  $tmp
        } else {
            set CuadroSpins($spinPath)  $max
        }
    } else {
        set tmp [expr $CuadroSpins($spinPath)-$CuadroSpins($spinPath,incr)]
        if {$tmp>$min} {
            set CuadroSpins($spinPath)  $tmp
        } else {
            set CuadroSpins($spinPath)  $min
        }
    }
    return
}    

##############################################################################
# CheckBounds
#    Checks that the content of the CuadroSpin is within the established bounds
#    if it isn't it sets it to the nearest limit.
#
# Parameter
#    spinPath: The path to the CuadroSpin.
#    max:      Maximun value allowed for the CuadroSpin
#    min:      Minumun value allowed for the CuadroSpin.
##############################################################################
proc CheckBounds {spinPath min max} {
    variable CuadroSpins

    if {![regexp {^[-0-9]+$} $CuadroSpins($spinPath)]} {
        return
    }

    if {$CuadroSpins($spinPath)>$max} {
        set CuadroSpins($spinPath) $max
    } elseif {$CuadroSpins($spinPath)<$min} {
        set CuadroSpins($spinPath) $min
    }
    return
}

###############################################################################
# ParseArguments
#    Gets the optional parameters passed to the SpinBox into the
#    namespace variable 'csArgs', the rest get the default values.
###############################################################################
proc ParseArguments {parameters} {
    variable csArgs

    upvar $parameters args

    set csArgs(-width)       5
    set csArgs(-incr)        1
    set csArgs(-min)         0
    set csArgs(-max)       100
    set csArgs(-default)     0
    set csArgs(-bg)        white
    set csArgs(-fg)        black
    set csArgs(-disbg)     grey
    set csArgs(-disfg)     black

    array set csArgs $args

    return
}

##############################################################################
# CuadroSpin
#    This is the procedure to create the CuadroSpin.
#
# Parameter
#    spinPath: The path to the CuadroSpin.
#    default:  Default value for the CuadroSpin.
#    max:      Maximun value allowed for the CuadroSpin
#    min:      Minumun value allowed for the CuadroSpin.
#    width:    The width of the entry in the spinbox, defaults to 5
#    incr:     The increment or decrement to apply to the value, defaults
#              to '1'
#
# Returns
#    The path to the CuadroSpin.
##############################################################################
proc CuadroSpin {spinPath args} {
    variable CuadroSpins
    variable csArgs

    ParseArguments args
    set CuadroSpins($spinPath)      $csArgs(-default)
    set CuadroSpins($spinPath,incr) $csArgs(-incr)

    frame $spinPath -relief sunken -bd 1

    entry $spinPath.e -bd 1 -justify right -relief sunken          \
            -textvariable ::CuadroSpin::CuadroSpins($spinPath)     \
            -width $csArgs(-width) -bg $csArgs(-bg) -fg $csArgs(-fg)

    frame  $spinPath.bg -bg $csArgs(-bg)
    button $spinPath.bg.incr -relief raised -image IncrValue       \
            -height 2 -width 4                                     \
            -command "CuadroSpin::IncrSpin $spinPath $csArgs(-min) \
            $csArgs(-max)  1"
    button $spinPath.bg.dec -relief raised  -image DecValue        \
            -height 2 -width 4                                     \
            -command "CuadroSpin::IncrSpin $spinPath $csArgs(-min) \
            $csArgs(-max) -1"
    pack $spinPath.e -side left   -fill both
    pack $spinPath.bg
    pack $spinPath.bg.incr $spinPath.bg.dec

    bind $spinPath.e <FocusIn>  "$spinPath.e selection range 0 end"
    bind $spinPath.e <FocusOut> "CuadroSpin::CheckBounds $spinPath \
            $csArgs(-min) $csArgs(-max)"

    return $spinPath
}

##############################################################################
# GetValue
#    Returns the value of the entry in a CuadroSpin.
#
# Parameter
#    spinPath: The path to the CuadroSpin.
#
# Returns:
#    Whatever is in the CuadroSpin's entry, or "" if there is not such
#    CuadroSpin.
##############################################################################
proc GetValue {spinPath} {
    variable CuadroSpins

    if {![info exists CuadroSpins($spinPath)]} {
        return ""
    }
    return $CuadroSpins($spinPath)
}

##############################################################################
# SetValue
#    Sets the value of the entry in a CuadroSpin.
#
# Parameter
#    spintPath: The path to the spinBox
#    value: The value to put in the entry
##############################################################################
proc SetValue {spinPath value} {
    variable CuadroSpins

    if {![info exists CuadroSpins($spinPath)]} {
        return ""
    }
    set CuadroSpins($spinPath) $value

    return
}

}

#################################EXAMPLE########################################
#
#wm title . "CuadroSpin"
#focus .
#
#CuadroSpin::CuadroSpin .cuadroSpin1 10 5 30
#CuadroSpin::CuadroSpin .cuadroSpin2 20 0 40
#
#pack .cuadroSpin1 .cuadroSpin2 -side top
################################################################################
