
/*

    File: ftp.c

    Copyright (C) 1999,2003,2004  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>

#include <time.h>
#include <signal.h>
#include <wait.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>

#include "ftp.h"
#include "ip-lib.h"
#include "lib.h"


unsigned get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "-ERR: can't get interface info: %m");
		exit (-1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}


int waitfd(ftp_t *x, int fd, int write, int timeout)
{
	int	rc;
	fd_set	*rs, *ws, fdset, available;
	struct timeval tv;

	while (1) {
		FD_ZERO(&fdset);
		FD_SET(fd, &fdset);
		memmove(&available, &fdset, sizeof(fd_set));
		tv.tv_sec  = timeout;
		tv.tv_usec = 0;

		if (write == 0) {
			rs = NULL;
			ws = &available;
			}
		else {
			rs  = &available;
			ws = NULL;
			}

		if ((rc = select(fd + 1, rs, ws, NULL, &tv)) < 0) {
			if (errno == EINTR)
				continue;

			fprintf (stderr, "%s: select error: %s\n", program, sys_errlist[errno]);
			exit (1);
			}
		else if (rc == 0) {
			fprintf (stderr, "%s: select timeout\n", program);
			exit (1);
			}

		if (FD_ISSET(fd, &available) == 0) {
			fprintf (stderr, "%s: select fd error\n", program);
			exit (1);
			}

		break;
		}

	return (0);
}

int getc_fd(ftp_t *x, int fd)
{
	int	c;
	static int here = 0, len = 0;
	static char buffer[1024];

	if (here >= len) {
		int	bytes;

		len = here = 0;
		waitfd(x, fd, 0, 60);
		if ((bytes = read(fd, buffer, sizeof(buffer) - 2)) <= 0)
			return (-1);

		len  = bytes;
		here = 0;
		}

	if (here >= len)
		return (-1);

	c = (unsigned char) buffer[here++];
	return (c);
}

char *readline_fd(ftp_t *x, int fd, char *line, int size)
{
	int	c, k;

	*line = 0;
	size = size - 2;

	c = getc_fd(x, fd);
	if (c < 0)
		return (NULL);

	k = 0;
	while (c > 0  &&  c != '\n'  &&  c != 0) {
		if (k < size)
			line[k++] = c;

		c = getc_fd(x, fd);
		}

	line[k] = 0;
	noctrl(line);

	return (line);
}


char *sfgets(ftp_t *x, char *line, int size)
{
	char *p;

	*line = 0;
	if ((p = readline_fd(x, x->control, line, size)) == NULL)
		return (NULL);
	else if (debug != 0)
		fprintf (stderr, "<<< SVR: %s\n", p);

	return (line);
}

int sfputs(ftp_t *x, char *format, ...)
{
	char	buffer[310];
	va_list	ap;

	va_start(ap, format);
	vsnprintf (buffer, sizeof(buffer) - 10, format, ap);
	va_end(ap);

	if (debug) {
		if (strncmp(buffer, "PASS ", 5) == 0)
			fprintf (stderr, ">>> SVR: PASS xxx\n");
		else
			fprintf (stderr, ">>> SVR: %s\n", buffer);
		}

	strcat(buffer, "\r\n");
	write(x->control, buffer, strlen(buffer));

	return (0);
}

int sfputc(ftp_t *x, char *command, char *parameter, char *line, int size, char **here)
{
	int	rc;
	char	*p, buffer[300];

	if (command != NULL  &&  *command != 0) {
		if (parameter != NULL  &&  *parameter != 0)
			snprintf (buffer, sizeof(buffer) - 2, "%s %s", command, skip_ws(parameter));
		else
			copy_string(buffer, command, sizeof(buffer));

		sfputs(x, buffer);
		}
	
	if (sfgets(x, line, size) == NULL)
		return (-1);

	rc = strtol(line, &p, 10);
	if (*p == '-') {
		while (1) {
			if (sfgets(x, line, size) == NULL)
				return (-1);

			p = noctrl(line);
			if (*p == ' ')
				/* weiter */ ;
			else if (atoi(line) == rc) {
				if (strlen(line) == 3)
					break;
				else if (strlen(line) > 3  &&  line[3] == ' ')
					break;
				}
			}
		}

	p = skip_ws(p);
	if (here != NULL)
		*here = p;

	return (rc);
}




int dotype(ftp_t *x, char *type)
{
	char	*p, line[200];

	x->ascii = (*type == 'A')? 1: 0;
	if (sfputc(x, "TYPE", type, line, sizeof(line), &p) != 200) {
		fprintf (stderr, "%s: can't set transfer type: %s\n", program, line);
		exit (1);
		}

	return (0);
}

int doport(ftp_t *x)
{
	unsigned int port;
	int	c, i;
	char	*p, ipnum[200], line[200];

	if (x->passive != 0) {
		int	k;
		char	ipbyte[5][20], portbyte[3][20];

		if (sfputc(x, "PASV", "", line, sizeof(line), &p) != 227) {
			fprintf (stderr, "%s: passive mode rejected: %s\n", program, line);
			exit (1);
			}

		/*
		 * PASV Response lesen, ftp.proxy entnommen.
		 */

		k = strlen(line);
		while (k > 0  &&  isdigit(line[k-1]) == 0)
			k--;

		if (isdigit(line[k-1])) {
			line[k--] = 0;
			while (k > 0  &&  (isdigit(line[k-1])  ||  line[k-1] == ','))
				k--;
			}

		/*
		 * line[k] sollte jetzt auf die erste Ziffer des PASV Response
		 * zeigen.
		 */

		if (isdigit(line[k]) == 0) {
			fprintf (stderr, "%s: can't read passive response: %s\n", program, line);
			exit (1);
			}

		/*
		 * Auslesen der PASV IP-Nummer und des Ports.
		 */

		p = &line[k];
		for (i=0; i < 4; i++)
			get_quoted(&p, ',', ipbyte[i], sizeof(ipbyte[i]));

		get_quoted(&p, ',', portbyte[0], sizeof(portbyte[0]));
		get_quoted(&p, ',', portbyte[1], sizeof(portbyte[1]));

		snprintf (x->data.interface, sizeof(x->data.interface) - 2, "%s.%s.%s.%s",
				ipbyte[0], ipbyte[1], ipbyte[2], ipbyte[3]);
		x->data.port = (strtoul(portbyte[0], NULL, 10) << 8) | strtoul(portbyte[1], NULL, 10);
		x->data.passive = 1;

		/*
		 * Wir muessen den Connect schon hier aufbauen, da z.B.
		 * proftpd ansonsten keine weiteren Kommandos annimmt.
		 */

		if (debug != 0) {
			fprintf (stderr, "*** connecting to %s:%u\n",
					x->data.interface, x->data.port);
			}

		if ((x->data.fd = openip(x->data.interface, x->data.port)) < 0) {
			fprintf (stderr, "%s: can't open data channel: %s:%u: %s\n",
					program, x->data.interface, x->data.port,
					sys_errlist[errno]);
			exit (1);
			}
		}
	else {
		get_interface_info(x->control, ipnum, sizeof(ipnum));
		x->data.fd = bind_to_port(ipnum, 0);
		port = get_interface_info(x->data.fd, ipnum, sizeof(ipnum));
		snprintf (line, sizeof(line) - 2, "%s,%d,%d", ipnum, (port >> 8) & 0XFF, port & 0XFF);

		for (i=0; (c = line[i]) != 0; i++) {
			if (c == '.')
				line[i] = ',';
			}

		if (sfputc(x, "PORT", line, line, sizeof(line), &p) != 200) {
			fprintf (stderr, "%s: port not accepted: %s\n", program, line);
			exit (1);
			}

		x->data.passive = 0;
		}

	return (0);
}

int connect_server(ftp_t *x)
{
	int	sock, adrlen;
	struct sockaddr_in adr;

	if (x->data.passive != 0)
		/* schon passiert, siehe doport() */ ;
	else {
		if (debug != 0)
			fprintf (stderr, "*** waiting for connection\n");

		adrlen = sizeof(struct sockaddr);
		sock = accept(x->data.fd, (struct sockaddr *) &adr, &adrlen);
		if (sock < 0) {
			fprintf (stderr, "%s: accept error: %s\n", program, sys_errlist[errno]);
			exit (1);
			}

		close(x->data.fd);
		x->data.fd = sock;
		}

	return (0);
}

int receive_data(ftp_t *x, char *filename)
{
	int	bytes;
	char	line[200], buffer[MAXBSIZE+10];
	FILE	*fp;

	if (*filename == 0  ||  strcmp(filename, "-") == 0)
		fp = stdout;
	else if ((fp = fopen(filename, "w")) == NULL) {
		fprintf (stderr, "%s: can't open file: %s\n", program, filename);
		exit (1);
		}

	connect_server(x);
	while (1) {
		waitfd(x, x->data.fd, 0, 300);
		if ((bytes = read(x->data.fd, buffer, x->bsize)) > 0) {
			if (x->ascii == 0)
				fwrite(buffer, sizeof(char), bytes, fp);
			else {
				int	c, i, k;

				k = 0;
				for (i=0; i<bytes; i++) {
					if ((c = buffer[i]) != '\r')
						buffer[k++] = c;
					}

				fwrite(buffer, sizeof(char), k, fp);
				}
			}
		else if (bytes == 0)
			break;
		else if (bytes < 0) {
			fprintf (stderr, "%s: broken connection: %s\n", program, sys_errlist[errno]);
			exit (1);
			}
		}

	close(x->data.fd);
	if (fp != stdout)
		fclose(fp);

	sfgets(x, line, sizeof(line));
	if (atoi(line) != 226) {
		fprintf (stderr, "%s: data transfer error: %s\n", program, line);
		exit (1);
		}

	return (0);
}

int dolist(ftp_t *x, char *command, char *parameter, char *output)
{
	char	*p, line[200];

	if (verbose != 0)
		fprintf (stderr, "getting file list%s%s\n", *parameter != 0? ": ": "", parameter);

	if (*x->lsopt != 0) {
		static char lsarg[200];

		snprintf (lsarg, sizeof(lsarg) - 2, "%s %s", x->lsopt, parameter);
		parameter = lsarg;
		}

	dotype(x, "A");
	doport(x);
	if (sfputc(x, command, parameter, line, sizeof(line), &p) != 150) {
		fprintf (stderr, "%s: can't get listing: %s\n", program, line);
		exit (1);
		}

	receive_data(x, output);
	return (0);
}

int doretr(ftp_t *x, char *remote, char *local)
{
	char	*p, line[200];

	if (verbose != 0)
		fprintf (stderr, "getting remote file: %s\n", remote);

	dotype(x, x->ascii != 0? "A": "I");
	doport(x);
	if (sfputc(x, "RETR", remote, line, sizeof(line), &p) != 150) {
		fprintf (stderr, "%s: can't retrieve data: %s\n", program, line);
		exit (1);
		}

	receive_data(x, local);
	return (0);
}

int dostor(ftp_t *x, char *remote, char *local)
{
	int	rc, bytes;
	char	*p, line[200], buffer[MAXBSIZE+10];
	fd_set	fdset, available;
	struct timeval tv;
	FILE	*fp;

	if (verbose != 0)
		fprintf (stderr, "storing local file: %s\n", remote);

	if (*local == 0  ||  strcmp(local, "-") == 0)
		fp = stdin;
	else if ((fp = fopen(local, "r")) == NULL) {
		fprintf (stderr, "%s: can't open file: %s\n", program , local);
		exit (1);
		}

	dotype(x, x->ascii != 0? "A": "I");
	doport(x);
	if (sfputc(x, "STOR", remote, line, sizeof(line), &p) != 150) {
		fprintf (stderr, "%s: can't store data: %s\n", program, line);
		exit (1);
		}

	connect_server(x);
	FD_ZERO(&fdset);
	FD_SET(x->data.fd, &fdset);
	while (1) {
		memmove(&available, &fdset, sizeof(fd_set));
		tv.tv_sec  = 600;
		tv.tv_usec = 0;

		if ((rc = select(x->data.fd + 1, NULL, &available, NULL, &tv)) < 0) {
			if (errno == EINTR)
				continue;

			fprintf (stderr, "%s: select error: %s\n", program, sys_errlist[errno]);
			exit (1);
			}
		else if (rc == 0) {
			fprintf (stderr, "%s: select timeout\n", program);
			exit (1);
			}

		if (FD_ISSET(x->data.fd, &available)) {
			if ((bytes = fread(buffer, sizeof(char), x->bsize, fp)) <= 0)
				break;
			else if (write(x->data.fd, buffer, bytes) != bytes) {
				fprintf (stderr, "%s: data write error\n", program);
				exit (1);
				}
			}
		}

	close (x->data.fd);
	if (fp != stdin)
		fclose (fp);

	sfgets(x, line, sizeof(line));
	if (atoi(line) != 226) {
		fprintf (stderr, "%s: data transfer error: %s\n", program, line);
		exit (1);
		}

	return (0);
}

int dodele(ftp_t *x, char *filename)
{
	char	*p, line[200];

	if (sfputc(x, "DELE", filename, line, sizeof(line), &p) != 250) {
		fprintf (stderr, "%s: can't remove file: %s, dir= %s\n", program, line, filename);
		exit (1);
		}

	return (0);
}

int domkdir(ftp_t *x, char *dirname)
{
	char	*p, line[200];

	if (sfputc(x, "MKD", dirname, line, sizeof(line), &p) != 257) {
		fprintf (stderr, "%s: can't create directory: %s, dir= %s\n", program, line, dirname);
		exit (1);
		}

	return (0);
}

int dormdir(ftp_t *x, char *dirname)
{
	char	*p, line[200];

	if (sfputc(x, "RMD", dirname, line, sizeof(line), &p) != 250) {
		fprintf (stderr, "%s: can't remove directory: %s, dir= %s\n", program, line, dirname);
		exit (1);
		}

	return (0);
}

int dorename(ftp_t *x, char *rnfr, char *rnto)
{
	char	*p, line[200];

	if (sfputc(x, "RNFR", rnfr, line, sizeof(line), &p) != 350) {
		fprintf (stderr, "%s: can't rename from: %s, filename= %s\n", program, line, rnfr);
		exit (1);
		}
	else if (sfputc(x, "RNTO", rnto, line, sizeof(line), &p) != 250) {
		fprintf (stderr, "%s: can't rename to: %s, filename= %s\n", program, line, rnto);
		exit (1);
		}

	return (0);
}

int domdtm(ftp_t *x, char *filename, char *buffer, int size)
{
	char	*p, line[300];

	copy_string(buffer, "-", size);
	if (sfputc(x, "MDTM", filename, line, sizeof(line), &p) != 213)
		return (0);

	copy_string(buffer, p, size);
	return (0);
}

int dosize(ftp_t *x, char *filename, char *buffer, int size)
{
	char	*p, line[300];

	copy_string(buffer, "-", size);
	if (sfputc(x, "SIZE", filename, line, sizeof(line), &p) != 213)
		return (0);

	copy_string(buffer, p, size);
	return (0);
}

int dosite(ftp_t *x, char *command)
{
	int	rc;
	char	*p, line[600];

	snprintf (line, sizeof(line) - 2, "SITE %s", command);
	sfputs(x, line);

	if (sfgets(x, line, sizeof(line)) == NULL) {
		fprintf (stderr, "%s: broken server connection\n", program);
		exit (1);
		}

	printf ("%s\n", line);
	rc = strtol(line, &p, 10);
	if (*p == '-') {
		while (1) {
			if (sfgets(x, line, sizeof(line)) == NULL) {
				fprintf (stderr, "%s: broken server connection\n", program);
				exit (1);
				}

			p = noctrl(line);
			printf ("%s\n", line);

			if (*p == ' ')
				/* weiter */ ;
			else if (atoi(line) == rc) {
				if (strlen(line) == 3)
					break;
				else if (strlen(line) > 3  &&  line[3] == ' ')
					break;
				}
			}
		}

	return (0);
}



int receive_nlist(ftp_t *x, char *pattern)
{
	char	*p, line[200];

	if (verbose)
		fprintf (stderr, "getting file list: %s\n", pattern);

/*	dotype(x, x->ascii != 0? "A": "I"); */
	doport(x);
	if (sfputc(x, "NLST", pattern, line, sizeof(line), &p) != 150) {
		fprintf (stderr, "%s: can't retrieve data: %s\n", program, line);
		exit (1);
		}

	connect_server(x);
	while (readline_fd(x, x->data.fd, line, sizeof(line)) != NULL) {
		p = skip_ws(noctrl(line));
		addfile(x, line);
		}

	close(x->data.fd);
	sfgets(x, line, sizeof(line));
	if (atoi(line) != 226) {
		fprintf (stderr, "%s: data transfer error: %s\n", program, line);
		exit (1);
		}

	return (0);
}

char *getfilename(char *path, char *file, int size)
{
	char	*p;

	if ((p = strrchr(path, '/')) == NULL)
		copy_string(file, path, size);
	else {
		p++;
		copy_string(file, p, size);
		}

	return (file);
}


int ftp_request(ftp_t *x, char *command, int argc, char **argv)
{
	int	i, k, rc;
	char	*p, filename[200], line[300];

	if (x->bsize <= 0)
		x->bsize = 2048;
	else if (x->bsize >= MAXBSIZE)
		x->bsize = MAXBSIZE;

	if ((x->control = openip(x->server, x->port)) < 0) {
		fprintf (stderr, "%s: can't connect to server: %s:%u: %s\n", program, x->server, x->port, sys_errlist[errno]);
		exit (1);
		}

	while (sfgets(x, line, sizeof(line)) != 0) {
		if (line[3] == ' ')
			break;
		}

	rc = strtoul(line, &p, 10);
	if (rc != 220) {
		fprintf (stderr, "%s: can't login: %s\n", program, line);
		exit (1);
		}

	rc = sfputc(x, "USER", x->username, line, sizeof(line), &p);
	if (rc == 230)
		/* ok, no password required */ ;
	else if (rc != 331  ||  sfputc(x, "PASS", x->password, line, sizeof(line), &p) != 230) {
		fprintf (stderr, "%s: can't login: %s\n", program, line);
		exit (1);
		}

	if (x->ndir > 0) {
		for (i=0; i<x->ndir; i++) {
			if (sfputc(x, "CWD", x->directory[i], line, sizeof(line), &p) != 250) {
				fprintf (stderr, "%s: can't change directory: %s, dir= %s\n", program, line, x->directory[i]);
				exit (1);
				}
			}
		}

	if (x->glob != 0) {
		if (strcmp(command, "get") == 0) {
			for (i=0; i<argc; i++)
				receive_nlist(x, argv[i]);

			argc = x->arg.argc;
			argv = x->arg.argv;
			}
		}

	if (argc == 0) {
		if (strcmp(command, "nlist") == 0  ||  strcmp(command, "mtime") == 0)
			/* continue */ ;
		else if (1  ||  *command == 0  ||  strcmp(command, "ls") == 0  ||  strcmp(command, "list") == 0) {
			dolist(x, "LIST", "", "-");
			exit (0);
			}
		else {
			fprintf (stderr, "%s: command requires arguments: %s\n", program, command);
			exit (1);
			}
		}


	k = 0;
	if (strcmp(command, "ls") == 0  ||  strcmp(command, "list") == 0) {
		if (argc == 0)
			dolist(x, "LIST", "", "-");
		else {
			for (i=k; i<argc; i++)
				dolist(x, "LIST", argv[i], "-");
			}
		}
	else if (strcmp(command, "nlist") == 0  ||  strcmp(command, "mtime") == 0) {
		if (argc == 0)
			receive_nlist(x, "");
		else {
			for (i=0; i<argc; i++)
				receive_nlist(x, argv[i]);
			}

		argc = x->arg.argc;
		argv = x->arg.argv;
		if (strcmp(command, "nlist") == 0) {
			for (i=0; i<argc; i++)
				printf ("%s\n", argv[i]);
			}
		else {
			char	mdtm[80], size[80];

			for (i=0; i < argc; i++) {
				domdtm(x, argv[i], mdtm, sizeof(mdtm));
				dosize(x, argv[i], size, sizeof(size));
				printf ("%14s %12s %s\n", mdtm, size, argv[i]);
				}
			}
		}
	else if (strcmp(command, "get") == 0) {
		if (x->pipe != 0)
			doretr(x, argv[k], "-");
		else {
			for (i=k; i<argc; i++)
				doretr(x, argv[i], getfilename(argv[i], filename, sizeof(filename) - 2));
			}
		}
	else if (strcmp(command, "put") == 0) {
		if (x->pipe != 0)
			dostor(x, argv[k], "-");
		else {
			for (i=k; i<argc; i++)
				dostor(x, argv[i], getfilename(argv[i], filename, sizeof(filename) - 2));
			}
		}
	else if (strcmp(command, "retr") == 0) {
		if (k+2 != argc) {
			fprintf (stderr, "%s: invalid number of arguments for rename\n", program);
			exit (1);
			}

		doretr(x, argv[0], argv[1]);
		}
	else if (strcmp(command, "stor") == 0) {
		if (k+2 != argc) {
			fprintf (stderr, "%s: invalid number of arguments for rename\n", program);
			exit (1);
			}

		dostor(x, argv[1], argv[0]);
		}
	else if (strcmp(command, "del") == 0) {
		for (i=k; i<argc; i++)
			dodele(x, argv[i]);
		}
	else if (strcmp(command, "mkdir") == 0) {
		for (i=k; i<argc; i++)
			domkdir(x, argv[i]);
		}
	else if (strcmp(command, "rmdir") == 0) {
		for (i=k; i<argc; i++)
			dormdir(x, argv[i]);
		}
	else if (strcmp(command, "rename") == 0) {
		if (k+2 != argc) {
			fprintf (stderr, "%s: invalid number of arguments for rename\n", program);
			exit (1);
			}

		dorename(x, argv[k], argv[k+1]);
		}
	else if (strcmp(command, "site") == 0) {
		for (i=k; i < argc; i++)
			dosite(x, argv[i]);
		}
	else {
		fprintf (stderr, "%s: unsupported command: %s\n", program, command);
		exit (1);
		}

	if (x->autolist != 0)
		dolist(x, "LIST", "", *x->outputfile == 0? "-": x->outputfile);

	return (0);
}


