/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */

/* Copyright (c) 1994 by Gunnar Rnning.  All Rights Reserved */

/*
 * Provide the basic functions for access to the World Wide Web 
 * from within the application. 
 * This code uses libwww from CERN.
 */

#if (defined WIN31 || defined WIN32)
#include <windows.h>            // required for all Windows applications
#include "windows/main.h"
#endif

#define WWWCOM

#ifdef WWWENABLE
#include "WWWLib.h"
#include "HTBind.h"
#include "HTHome.h"
#include "HTTP.h"
#ifdef TCPURL
#include "TCP.h"
#endif 
#include "HTGuess.h"
#include "WWWMIME.h"
#include "HTNetMan.h"
#include "HTStream.h"
#include "HTReqMan.h"
#include "HTAccess.h"
#include "HTEvntrg.h"
#include "HTInit.h"
#include "HTEvent.h"
struct _HTStream {
    HTStreamClass * isa;
};
#include "HTAtom.h"
#endif

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <stdlib.h>  
#if (defined WIN31 || defined WIN32)
#include <io.h>
#else
#include <unistd.h>
#endif
#ifdef X11
#include <X11/cursorfont.h>
#endif

#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>

#include "candle.h"
#include "fast_lis.h"
#include "wwwcom.h"
#include "parser.h"
#include "error.h"
#include "sysproto.h"
#include "protos/canutil.h"
#include "protos/fast_lis.h"
#include "protos/parser.h"
#include "protos/memory.h"

#ifdef WWWENABLE
static struct urlmap FAR *map=NULL;

PRIVATE HTList FAR * converters = NULL;
PRIVATE HTList FAR * presenters = NULL;
PRIVATE HTList FAR * encoders = NULL;

/* status field in reqmap */
#define WWW_ERROR   1
#define WWW_LOADED  2
#define WWW_LOADING 3

struct AweReqContext *blocking_request = NULL;
struct AweReqContext *active_requests = NULL;

/* 
 * Keep track of open URLs and the number of bytes the application
 * have read from them.
 */
struct _fdmap {
  struct urlmap *um;
  long rbytes;      /* Number of bytes read by the application from this fd */
  long wbytes;      /* Number of bytes written by the application to this fd */
  long bytes;       /* Number of bytes read from network. Used to decide if 
		     * we have to block and wait for more data to be received
		     * from the network.
		     */
  char blocking;    /* Defaults to blocking IO */
} *fdmap=NULL;


static unsigned short maxfd=0;

struct urlmap *new_urlmap(struct cw_status *gp, char *url)
{
  struct urlmap *cur;
  
  cur = (struct urlmap FAR *) CalCalloc(1, sizeof(struct urlmap));
  if(cur == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  cur->next = map;
  cur->url = url;
  map = cur;
  return cur;
}

void free_urlmap(struct cw_status *gp, struct urlmap *cur)
{
  struct urlmap *c;
  if(map == cur){
    map = map->next;
  } else {
    for(c = map; c != NULL && c->next != cur; c = c->next);
    if(c == NULL)
      return;
    c->next = cur->next;
  }
  if(cur->is_tmp)
    remove(cur->filename);
  CalFree(cur->filename);
  CalFree(cur->url);
  if(cur->content_type)
    CalFree(cur->content_type);
  CalFree(cur);
}


/*
 * Register a anchor as the new basis for computation of url's. 
 * It must be called before we start parsing a new candle-file.
 * The value of type must be either URL_ANCHOR or FILE_ANCHOR, depending
 * on the format of the supplied anchor. URL_ANCHOR's _must_ be absolute.
 */ 
void NewAnchor(struct cw_status *gp, char FAR *anchor, int type)
{
   if(gp->current_anchor != NULL){
      CalFree(gp->current_anchor);
      gp->current_anchor = NULL;
   } 
      
   if(type == URL_ANCHOR)
      gp->current_anchor = simplifyURL(gp, anchor);
   else if (type == FILE_ANCHOR)
      gp->current_anchor = lookupURL(gp, anchor);
}


static char FAR *startResolveURL(struct cw_status *gp, char FAR *url, int block)
{
  struct urlmap FAR *cur;
  char FAR *surl;
  char FAR *tmpfile;
  struct AweReqContext *req;
  
  surl = simplifyURL(gp, url);
  
  if(gp->current_anchor == NULL){
    NewAnchor(gp, surl, URL_ANCHOR);
  }
  cur = map;
  while(cur != NULL && strcmp(surl, cur->url) != 0)
    cur = cur->next;
  
  if(cur == NULL){
    cur = new_urlmap(gp, surl);

/*     printf ("Setting out to resolve URL\n"); */
    if(strncmp("file:", cur->url, 5) == 0){ 
      /*
       * Give local files special treatment because libwww doesn't 
       * handle file:... correctly.
       */
      char FAR *path;
      HTFormat format;
      double quality = 1.0;
      HTBind_getFormat    (cur->url, &format, NULL, NULL, NULL, &quality);
      cur->content_type = strdup(format->name);
      
      path = HTParse(cur->url, gp->current_anchor, PARSE_PATH);
      if((cur->filename = (char FAR *) CalMalloc(strlen(path)+2)) == NULL){
	userMsg(ErrNOMOREMEM);
	l_exit(NOT_OK);
      }
#ifdef UNIX
      strcpy(cur->filename, "/");
#else 
      cur->filename[0] = 0;
#endif
      strcat(cur->filename, path);
      HT_FREE(path);
    } else {
      req = GetURL(gp, cur, block);
      if(req->status == WWW_ERROR){
	CalFree(cur->filename);
	CalFree(cur->url);
	map = cur->next;
	CalFree(cur);
	CalFree(req);
	return NULL;
      } else if(req->status == WWW_LOADED)
	CalFree(req);
      else if(req->status == WWW_LOADING){
	printf("Still loading %s\n", cur->url);
      }
      
      cur->is_tmp = TRUE;
    }
  } else {
    CalFree(surl);
    if(cur->reqcon != NULL && block){
      req = GetURL(gp, cur, block);
      if(req->status == WWW_ERROR){
	CalFree(cur->filename);
	CalFree(cur->url);
	map = cur->next;
	CalFree(cur);
	CalFree(req);
	return NULL;
      } else if(req->status == WWW_LOADED)
	CalFree(req);
      else if(req->status == WWW_LOADING){
	printf("Still loading %s\n", cur->url);

      }
    }
  }
#ifdef GUNNAR   
  printf("Resolved URL to: %s\n",cur->filename);
#endif
  if((tmpfile = (char FAR *) CalMalloc(strlen(cur->filename) + 1)) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  strcpy(tmpfile, cur->filename);
/*   printf ("Resolved URL\n"); */
  return tmpfile;
}

/*
 * Fetch the object pointed by an URL, and return a filename to the local 
 * copy of the object. The string returned must be freed by the caller.
 * The URL can be either relative or absolute. Relative URLs are computed 
 * relative to the anchor registered by NewAnchor(). If gp->current_anchor 
 * == NULL set this URL as the gp->current anchor.
 */
char FAR *ResolveURL(struct cw_status *gp, char FAR *url)
{
  return startResolveURL(gp, url, TRUE);
}

void WWWInit(void)
{
  HTLibInit(versionname, systemname);
  HTProxy_getEnvVar();
  HTMIMEInit();
  HTProtocol_add("http", "tcp", NO, HTLoadHTTP, NULL); 
#ifdef TCPURL
  HTProtocol_add("tcp", "tcp", NO, HTLoadTCP, NULL); 
#endif
  converters = HTList_new();
  HTConversion_add(converters,"message/rfc822","*/*",HTMIMEConvert, 1.0, 0.0, 0.0);
  HTConversion_add(converters,"text/x-http","*/*",  HTTPStatus_new, 1.0, 0.0, 0.0);
#ifdef TCPURL
  HTConversion_add(converters,"text/x-tcp","*/*",  TCPStatus_new, 1.0, 0.0, 0.0);
#endif

  HTConversion_add(converters,"www/unknown",		"*/*",		
		   HTGuess_new,	1.0, 0.0, 0.0);
  
  HTBind_add("gz", NULL, "gzip", NULL, NULL, 1.0);	/* Gnu Compressed data	*/

  HTBind_addType("awe", "application/x-candleweb", 1.0);
  HTBind_addType("gif", "image/gif", 1.0);
  HTBind_addType("jpg", "image/jpeg", 1.0);
  HTBind_addType("html", "text/html", 1.0);
  HTBind_addType("htm", "text/html", 1.0);
  /** This dumps all other formats to local disk without any further
  ** action taken
  */
  
  presenters = HTList_new();
  encoders = HTList_new();
  HTEncoderInit(encoders);
  HTCoding_add(encoders, "gzip", NULL, HTChunkedDecoder, 1.0);
  HTFormat_setTransferCoding(encoders);


  HTEventrgInit();
  HTEventInit();

  HTFormat_setConversion(converters);
  HTNetCall_addBefore(HTLoadStart, NULL, 0);
  HTNetCall_addAfter(HTLoadTerminate, NULL, HT_ALL);
  HTNetCall_addAfter(terminate_handler, NULL, HT_ALL);   

  HTAlert_add(progress_handler, HT_A_PROGRESS);
  HTAlert_add(HTError_print, HT_A_MESSAGE);


  HTTransportInit();
}

/*
 * Free all memory allocated by this module.
 */
void WWWFree(void)
{
   struct urlmap FAR *next;  
   char mess[160];
   HTLibTerminate();
   for(;map != NULL;map = next){
      next = map->next;
      if(strncmp("file:", map->url, 5) && map->is_tmp && 
	 remove(map->filename)) {
	sprintf(mess, ErrWWWFREREM, map->filename);
	userMsg(mess);
      }
      CalFree(map->filename);
      CalFree(map->url);
      if(map->content_type)
	CalFree(map->content_type);
      CalFree(map);
   }
#if 0
   if(gp->current_anchor != NULL){
      CalFree(gp->current_anchor);
      gp->current_anchor = NULL;
   }
#endif
}

/*
 * Return a malloced URL associated with the filename. The filename must be
 * one that's already generated by ResolveURL().
 */
static char FAR *lookupURL(struct cw_status *gp, char FAR *filename)
{         
   char mess[160];
   struct urlmap FAR *cur=map;
   while(cur != NULL && strcmp(cur->filename, filename) != 0)
      cur = cur->next;

   if(cur != NULL){
      char FAR *t;
      if((t = (char FAR *) CalMalloc(strlen(cur->url)+1)) == NULL){
	userMsg(ErrNOMOREMEM);
	l_exit(NOT_OK);
      }
      strcpy(t, cur->url);
      return t;
   }

   sprintf(mess, ErrFILNOTEMP, filename);
   userMsg(mess);
#ifdef GUNNAR
   printURLs(gp);
#endif
   return NULL;
}

char *AbsoluteURL(char *anchor, char *relative)
{
   char FAR *u, FAR *tmp=NULL;
   char FAR *host, FAR *path, FAR *access;

   if((u = HTMemory_malloc(strlen(relative)+1)) == NULL){
     userMsg(ErrNOMOREMEM);
     l_exit(NOT_OK);
   }
   strcpy(u, relative);
   u = HTStrip(u);
   u = HTSimplify(&u);
   
   if(HTCleanTelnetString(u)){
     userMsg(ErrREMCHRURL);
   }

   access = HTParse(u, anchor, PARSE_ACCESS);
   host = HTParse(u, anchor, PARSE_HOST);
   path = HTParse(u, anchor, PARSE_PATH);
   tmp = CalMalloc(strlen(access) + strlen(host) + strlen(path) + 5);
   if(tmp == NULL){
     userMsg(ErrNOMOREMEM);
     l_exit(NOT_OK);
   }
   strcpy(tmp, access); strcat(tmp, "://"); strcat(tmp, host);
   if(path[0] != '/') /* Kludging because path[0] != '/' if the url is 
			 absolute */ 
     strcat(tmp, "/"); 
   strcat(tmp, path);
   HT_FREE(u);
   if((u = HTMemory_malloc(strlen(tmp)+1)) == NULL){
     userMsg(ErrNOMOREMEM);
     l_exit(NOT_OK);
   }
   strcpy(u, tmp);
   free(tmp);
   u = HTSimplify(&u);
   tmp = strdup(u);
   HT_FREE(access); HT_FREE(host); HT_FREE(path);
   HT_FREE(u);
   return tmp;
}

/*
 * This routine will do the following to a URL :
 * 1. strip whitespaces of the URL.
 * 2. Make it absolute.
 * 3. Canonicalizes and simplifies the URL.
 * 
 * Return a malloced URL.
 */
static char FAR *simplifyURL(struct cw_status *gp, char FAR *url)
{
   char FAR *u, FAR *tmp=NULL;
   char FAR *host, FAR *path, FAR *access;
   if((u = HTMemory_malloc(strlen(url)+1)) == NULL){
     userMsg(ErrNOMOREMEM);
     l_exit(NOT_OK);
   }
   strcpy(u, url);
   u = HTStrip(u);
   u = HTSimplify(&u);
   
   if(HTCleanTelnetString(u)){
     userMsg(ErrREMCHRURL);
   }
   if (gp->current_anchor != NULL){
     access = HTParse(u,gp->current_anchor, PARSE_ACCESS);
     host = HTParse(u,gp->current_anchor, PARSE_HOST);
     path = HTParse(u,gp->current_anchor, PARSE_PATH);
     tmp = CalMalloc(strlen(access) + strlen(host) + strlen(path) + 5);
     if(tmp == NULL){
       userMsg(ErrNOMOREMEM);
       l_exit(NOT_OK);
     }
     strcpy(tmp, access); strcat(tmp, "://"); strcat(tmp, host);
     if(path[0] != '/') /* Kludging because path[0] != '/' if the url is 
			   absolute */ 
       strcat(tmp, "/"); 
     strcat(tmp, path);
     HT_FREE(access); HT_FREE(host); HT_FREE(path);
     HT_FREE(u);
   }else{
     if(strstr(u,":") == 0){ 
       /*
	* 'u' is not an absolute url and we have no anchor, 
	* so we assume that 'u' is a local file.
	*/
       char FAR *cwd;
       if(u[0] == '/'){ /* u is an absolute local file */
	 if((tmp = (char FAR *) CalMalloc(strlen(u) + 17)) == NULL){
	   userMsg(ErrNOMOREMEM);
	   l_exit(NOT_OK);
	 }
	 strcpy(tmp, "file://localhost");
	 strcat(tmp, u);
       }else{ /* u is a relative local file */ 
	 if((cwd = getcwd((char FAR *)NULL, 256)) == NULL){
	   userMsg(ErrGETCWDURL);
	   return NULL;
	 }
	 tmp = (char FAR *) CalMalloc(strlen(u) + strlen(cwd) + 18);
	 if(tmp == NULL){
	   userMsg(ErrNOMOREMEM);
	   l_exit(NOT_OK);
	 }
	 strcpy(tmp, "file://localhost");
	 strcat(tmp, cwd);
	 strcat(tmp, "/");
	 strcat(tmp, u);
       }
       HT_FREE(u);
       u = HTSimplify(&tmp);
       if(tmp != u){
	 CalFree(tmp);
	 tmp = strdup(u);
	 HT_FREE(u);
       } 
     }else{ /* u is an absolute url */
       tmp = strdup(u);
       HT_FREE(u);
     }
   }
#ifdef GUNNAR
   printf("Simplified URL to %s\n", tmp);
#endif
   return tmp;
}

/*
 * Return pointer to a malloced canonized URL.
 */
char * canonizedURL(struct cw_status *gp, char *url)
{
  return simplifyURL(gp, url);
}

void printURLs(struct cw_status *gp)
{
  struct urlmap FAR *cur;

  cur = map;
  while(cur != NULL){
    /* printf("urlmap: %s - %s\n", cur->url, cur->filename); */
    cur = cur->next;
  }
}

/*
 * This function should reload the URL assosiciated with file 
 * and put the new contents into file.
 */
int ReloadFile(struct cw_status *gp, char FAR *filename)
{
  struct urlmap FAR *cur;
  char msg[256];
  struct AweReqContext *req;
  int status;

  cur = map;
  while(cur != NULL){
    if(!strcmp(filename, cur->filename)){
      if(strncmp("file:", cur->url, 5) == 0){
	setScanFileName(gp, strdup(cur->filename));
	return 1;
      }
      if(!cur->is_tmp){
	cur->filename = strdup(tmpnam(NULL));
	setScanFileName(gp, strdup(cur->filename));
	cur->is_tmp = TRUE;
      }
      req = GetURL(gp, cur, TRUE);
      status = req->status;
      CalFree(req);

      if(status == WWW_ERROR){
	sprintf(msg, ErrCONOLOURL, cur->url, 
		cur->filename);
	setStatus(gp, msg);
	return 0;
      } else {
	sprintf(msg, MessLOADURLLF, cur->url, 
		cur->filename);
	setStatus(gp, msg);
	return 1;
      }
    }
    cur = cur->next;
  }
  sprintf(msg, ErrCONOFIURL, 
	  filename);
  setStatus(gp, msg);
  return 0;
}

/*
 * This function removes and entry with filename from the map list.
 * Useful to invalidate cache members on a reload.
 */
int RemoveMapEntry(struct cw_status *gp, char *url)
{
  struct urlmap *cur;
  char msg[256];

  cur = map;
  while(cur != NULL){
    if(!strcmp(url, cur->url)){
      free_urlmap(gp, cur);
      return 1;
    }
    cur = cur->next;
  }
  sprintf(msg, ErrCONOFIURL, 
	  url);
  setStatus(gp, msg);
  return 0;
}

/*
 * Bind URL to filename if filename exists.
 */
void BindURL(struct cw_status *gp, char FAR *url, char FAR *filename)
{
  struct urlmap FAR *cur;

  if(filename == NULL || url == NULL)
    return;
  
  cur = map;
  while(cur != NULL){
    if(!strcmp(filename, cur->filename)){
      CalFree(cur->url);
      cur->url = strdup(url);
      return;
    }
    cur = cur->next;
  }
}

/*
 * Return pointer to URL associated with filename.
 */
char FAR *findURL(struct cw_status *gp, char FAR *filename)
{
  struct urlmap FAR *cur;

  cur = map;
  while(cur != NULL){
    if(!strcmp(filename, cur->filename)){
      return cur->url;
    }
    cur = cur->next;
  }
  return NULL;
}

static struct urlmap *URLEntry(struct cw_status *gp, char *url)
{
  struct urlmap FAR *cur;
  char *surl;

  surl = simplifyURL(gp, url);

  cur = map;
  while(cur != NULL){
    if(!strcmp(surl, cur->url)){
      break;
    }
    cur = cur->next;
  }
  CalFree(surl);
  return cur;
}

/*
 * Return a pointer to MIME content type of an already loaded URL.
 */
char *ContentType(struct cw_status *gp, char *url)
{
  struct urlmap *cur;
  char *fname;

  cur = URLEntry(gp, url);
  if(cur == NULL || cur->reqcon != NULL){
    fname = ResolveURL(gp, url);
    if(fname)
      CalFree(fname);
    else 
      return "";
    cur = URLEntry(gp, url);
  }
  return cur->content_type;
}

PRIVATE HTParentAnchor FAR * anchor;

/* 
 * Callback that handles events for the window system.
 */
int gui_handler(SOCKET fd, HTRequest *req, SockOps ops)
{
  struct AweReqContext *reqcon;

  reqcon = HTRequest_context(req);
  
  CalServeWinEvents(reqcon->gp);
  return HT_OK;
}

static int blocked_time;

static BOOL progress_handler(HTRequest FAR *request,
			     HTAlertOpcode op, int msgnum,
			     const char FAR * dfault, void FAR * input,
			     HTAlertPar FAR * reply)
{
  char msg[256];
  int len, n;
  struct AweReqContext *reqcon;
  
  reqcon = HTRequest_context(request);
  if(reqcon == NULL)
    return HT_OK;
  blocked_time = 0;
  switch(op){
  case HT_PROG_DNS:
    sprintf(msg, "Doing DNS resolution on %s", reqcon->map->url);
    break;
  case HT_PROG_CONNECT:
    sprintf(msg, "Connecting to %s", reqcon->map->url);
    break;
  case HT_PROG_ACCEPT:
    sprintf(msg, "Connecting passive");
    break;
  case HT_PROG_READ:
    n = HTRequest_bytesRead(request);
    len =  HTAnchor_length(anchor);
    sprintf(msg, "Loading %s %ld bytes.", reqcon->map->url, 
	    HTRequest_bytesRead(request));
    if(len != -1)
      sprintf(msg, "Read %d%%(%dK/%dK) of %s", 100 * n / len, n/1024, 
	      len/1024, reqcon->map->url);
    /*    if(n == len && len != -1){
      HTEvent_stopLoop ();
    }*/
    break;
  case HT_PROG_WRITE:
    sprintf(msg, "Writing data");
    break;
  case HT_PROG_DONE:
    sprintf(msg, "finished connection");
    break;
  case HT_PROG_WAIT:
    sprintf(msg, "waiting for socket");
    break;
  default:
    printf(msg, "if you see this message, please send a bug report.");
  }
  setStatus(reqcon->gp, msg);
  if(blocking_request == NULL){
    HTEventrg_stopLoop ();
  }
  CalServeWinEvents(reqcon->gp);
  return HT_OK;
}

int req_timer(HTRequest FAR *request)
{
  struct AweReqContext *reqcon;	
  int n, len;
  char msg[256];

  reqcon = HTRequest_context(request);
  
  /*  printf ("In reqtimer\n"); */

  if(blocked_time > 5){
	n = HTRequest_bytesRead(request);
    len =  HTAnchor_length(anchor);
    sprintf(msg, "Stalled on loading %s %ld bytes.", reqcon->map->url, 
	    HTRequest_bytesRead(request));
    if(len != -1)
      sprintf(msg, "Stalled %d%%(%dK/%dK) of %s", 100 * n / len, n/1024, 
	      len/1024, reqcon->map->url);
    setStatus(reqcon->gp, msg);
  }
  blocked_time++;
#ifdef WIN32
  CalServeWinEvents(reqcon->gp);
#endif
  /*  printf ("After req_timer\n"); */

  return HT_OK;
}

/*
 * Callback that handles the termination of a request. 
 */
static int terminate_handler(HTRequest * request, 
			     void *param, int status) 
{
  char msg[256];
  struct AweReqContext *reqcon;
  HTFormat fmt;
  
  reqcon = (struct AweReqContext *) HTRequest_context(request);
  if(reqcon == blocking_request){
    blocking_request = NULL;
    HTEventrg_stopLoop ();
  } else {
    if(reqcon == active_requests)
      active_requests = next(reqcon);
    remove_at(reqcon);
  }
  switch(status){
  case HT_LOADED:
    reqcon->status = WWW_LOADED;
    sprintf(msg, "Loaded %s succesfully", reqcon->map->url);
    break;
  case HT_ERROR:
    reqcon->status = WWW_ERROR;
    sprintf(msg, "Error occured on loading %s", reqcon->map->url);
    break;
  }
  reqcon->map->reqcon = NULL;
  setStatus(reqcon->gp, msg);
  fmt = HTAnchor_format(HTRequest_anchor(request));
  reqcon->map->content_type = strdup(fmt->name);
  HTRequest_delete(request);
  reqcon->req = NULL;
  CalServeWinEvents(reqcon->gp);
  return HT_OK;
}

int stop_loop_timer(HTRequest *req)
{
  return !HT_OK;
}

void ProcessRequests(struct cw_status *gp)
{
  struct timeval tp;
  tp.tv_sec = 0;
  tp.tv_usec = 1;

  if(active_requests != NULL){
    /*
     * This is not what we want, because it will block the request.
     */
    HTEventrg_registerTimeout (&tp, active_requests->req, stop_loop_timer, 1);
    HTEventrg_loop(active_requests->req);
    HTEventrg_unregisterTimeout();
  }
}


/*
 * Set up a request.
 */
struct AweReqContext *SetupRequest(struct cw_status *gp, struct urlmap *cur, int block)
{
  HTRequest FAR * request;
  struct AweReqContext *reqcon; 
  char msg[256];
  int loadstat = HT_INTERNAL;
  char *url = cur->url;
  char *tmpfile;

  if(cur->reqcon != NULL){
    reqcon = cur->reqcon;
    if(block){
      if(active_requests == reqcon){
	active_requests = next(reqcon);
      }
      if(is_list(reqcon))
	remove_at(reqcon);
      blocking_request = reqcon;
    }
  } else {
    if((reqcon = CalCalloc(1,sizeof(struct AweReqContext))) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    cur->reqcon = reqcon;
    reqcon->gp = gp;
    if(block)
      blocking_request = reqcon;
    else{
      if(active_requests != NULL)
	place_prev(active_requests, reqcon);
      else 
	init_list(reqcon);
      active_requests = reqcon;
    }
    
#ifdef WIN32
    tmpfile = _tempnam("c:\\WINDOWS\\TEMP", "AWE");
#else
    tmpfile = tmpnam(NULL);
#endif
    if(cur->filename != NULL)
      CalFree(cur->filename);
    cur->filename = strdup(tmpfile);
    if( (reqcon->fp = fopen(cur->filename, "wb")) == NULL){
/*       userMsg(strerror(errno)); */
      sprintf(msg, ErrFILE, cur->filename);
      userMsg(msg);
      
      reqcon->status = WWW_ERROR;
      return reqcon;
    }
    request =  HTRequest_new(); 
    
    /* Bind context information to request */
    reqcon->req = request;
    HTRequest_setContext(request, (void *)reqcon);
    reqcon->map = cur;
    cur->reqcon = reqcon;
    HTRequest_addRqHd(request, HT_C_HOST);
    HTRequest_addRqHd(request, HT_C_ACCEPT_ENC);

    HTRequest_setOutputStream(request,HTFWriter_new(request, reqcon->fp, NO));
    HTRequest_setOutputFormat(request, WWW_SOURCE);
    HTRequest_setConversion(request, presenters, NO);
    /* Set Referer field */
    HTRequest_setParent(request, (HTParentAnchor FAR *) 
			HTAnchor_findAddress(gp->current_anchor));
    
    anchor = (HTParentAnchor FAR *) HTAnchor_findAddress(url);
    
  }
  sprintf(msg, MessLOADURL, url);
  setStatus(gp, msg);
    
  return reqcon;
}

void DoRequest(struct cw_status *gp, HTRequest * request, int block)
{
  struct timeval tp;
  int fd;
  struct AweReqContext *reqcon;

  reqcon = (struct AweReqContext *) HTRequest_context(request);

#ifdef X11
    if(block)
      XDefineCursor(gp->system.dpy, XtWindow(gp->system.topLevel),
		    XCreateFontCursor(gp->system.dpy, XC_watch));
    
  
    fd = ConnectionNumber(XtDisplay(gp->system.topLevel));
    HTEventrg_register(fd, request,(SockOps)(FD_READ), 
		       gui_handler, 1);
  
#endif

  tp.tv_sec = 0;
  tp.tv_usec = 500000;
  blocked_time = 0;

#ifdef WIN32
  HTEventrg_registerTimeout (&tp, request, req_timer, 0);
#endif

  if(block)
    HTEventrg_loop(request);

#ifdef X11
  HTEventrg_unregister(fd, (SockOps)(FD_WRITE|FD_READ));

  if(block)
    XUndefineCursor(gp->system.dpy, XtWindow(gp->system.topLevel));
#endif
  if(block)
    fclose(reqcon->fp);
}

/*
 * Load url into filename using the GET method.
 */
struct AweReqContext *GetURL(struct cw_status *gp, struct urlmap *cur,
			     int block)
{
  struct AweReqContext *reqcon; 
  int loadstat;
  reqcon = SetupRequest(gp, cur, block);

  HTRequest_setMethod(reqcon->req, METHOD_GET);
  loadstat = HTLoadAnchor((HTAnchor FAR *) anchor, reqcon->req); 
  DoRequest(gp, reqcon->req, block);
    
 return reqcon;
}

/*
 * Write data to a stream.
 */
int WriteData(struct cw_status *gp, HTStream *target, char *buf, int len)
{
  int status;
  char msg[256];

  status = (*target->isa->put_block)(target, buf, len);

  if (status == HT_OK)
     return (*target->isa->flush)(target);
  if (status == HT_WOULD_BLOCK) {
    setStatus(gp, "POST Anchor. Target WOULD BLOCK\n");
    return HT_WOULD_BLOCK;
  } else if (status == HT_PAUSE) {
    setStatus(gp, "POST Anchor. Target PAUSED\n");
    return HT_PAUSE;
  } else if (status > 0) {              /* Stream specific return code */
    sprintf(msg, "POST Anchor. Target returns %d\n", status);
    setStatus(gp, msg);
    return status;
  } else {                                     /* We have a real error */
    setStatus(gp, "POST Anchor. Target ERROR\n");
    return status;
  }
}

static void extend_fdmap(struct cw_status *gp, int fd)
{
  if(fd > maxfd){
    if(fdmap != NULL){
      fdmap = CalRealloc(fdmap, fd * sizeof(struct _fdmap));
      if(fdmap == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      memset(&fdmap[maxfd], 0, 
	     (fd - maxfd) * sizeof(struct _fdmap));
    } else {
      fdmap = CalCalloc(1, fd * sizeof(struct _fdmap));
    }
    maxfd = fd;
  } else {
    memset(&fdmap[fd-1], 0, sizeof(struct _fdmap));
  }
}

/* Open URL for reading. Return fd on success, -1 on error.
 */
int OpenURL(struct cw_status *gp, char FAR *url, char FAR *mode)
{
  char FAR *filename, FAR *surl;
  char tmpname[128];
  int fd;
  int write_only=0;
  int flags=0, blocking = TRUE;
  
  surl = simplifyURL(gp, url);
  
  /* Assume read only if "w" not present */
  if(strchr(mode,'w')){
    flags = O_WRONLY|O_CREAT|O_TRUNC;
  } 
  if(strchr(mode,'r')){
    if(flags != 0)
      flags = O_RDWR|O_CREAT|O_TRUNC;
    else 
      flags = O_RDONLY;
  }
  if (strchr (mode, 'N')) blocking = FALSE;

  if(strncmp("file:", surl, 5) == 0){
    /* Check if we are allowed to read/write to local files */    
    if((flags & O_WRONLY) && !gp->allow_local_write){
      errorMsg(gp, 0, WarnFILEWRI);
      return -1;
    }
    if((flags & O_RDONLY) && !gp->allow_local_access){
      errorMsg(gp, 0, WarnFILEREA);
      return -1;
    }
    if((flags & O_RDWR) && ( !gp->allow_local_access || 
			     !gp->allow_local_write )){
      errorMsg(gp, 0, WarnFILEWRI);
      return -1;
    }
    filename = startResolveURL(gp, surl, FALSE);
    strncpy(tmpname, filename, 128);
    CalFree(surl);
    CalFree(filename);
    if((fd = open(tmpname, flags, 0777)) > 0){
      extend_fdmap(gp, fd);
      fdmap[fd-1].blocking = blocking;
      return fd;
    } else
      return -1;
  } else {
    if(write_only){
      errorMsg(gp, 0, WarnSOCKWRI);
      return -1;
    }else {
      filename = startResolveURL(gp, surl, FALSE);
      strncpy(tmpname, filename, 128);

      if((fd = open(tmpname, O_RDONLY)) > 0){
	extend_fdmap(gp, fd);
	fdmap[fd-1].um = URLEntry(gp, surl);
	fdmap[fd-1].blocking = blocking;
      } else
	fd = -1;
      CalFree(surl);
      CalFree(filename);
    }
  }
  return fd;
}

int Write(int fd, char FAR *buf, int nbytes)
{
  int n;
  int sum=0;
  struct urlmap *um;
  char *pbuf=buf;
  HTRequest *req;
  
  if(fd <= maxfd){
    if((um = fdmap[fd-1].um) != NULL && um->reqcon != NULL){
      /* This request has not yet finished, so I have to check if
       * there is enough data available to satisfy the write request. 
       * Block until there is enough data available.
       */
      req = um->reqcon->req;
      do {
	fdmap[fd-1].bytes = HTRequest_bytesRead(req);
	ProcessRequests(um->reqcon->gp);
      } while(fdmap[fd-1].bytes < fdmap[fd-1].wbytes + nbytes  &&
	      um->reqcon != NULL);
    }
    while(sum < nbytes){ 
      n = write(fd, pbuf, nbytes);
      if(n == 0){
	return sum;
      }
      sum += n;
      fdmap[fd-1].wbytes += n;
      pbuf = buf + sum;
    }
    return n;
  } else 
    return -1;
}

int Read(int fd, char FAR *buf, int nbytes)
{
  int n;
  int sum=0;
  struct urlmap *um;
  char *pbuf=buf;
  HTRequest *req;
  
  if(fd <= maxfd){
    if((um = fdmap[fd-1].um) != NULL && um->reqcon != NULL){
      /* This request has not yet finished, so I have to check if
       * there is enough data available to satisfy the read request. 
       * Block until there is enough data available.
       */
      req = um->reqcon->req;
      do {
	fdmap[fd-1].bytes = HTRequest_bytesRead(req);
	ProcessRequests(um->reqcon->gp);
      } while(fdmap[fd-1].bytes < fdmap[fd-1].rbytes + nbytes  &&
	      um->reqcon != NULL && fdmap[fd-1].blocking);
    }
    while(sum < nbytes){ 
      n = read(fd, pbuf, nbytes);
      if(n <= 0){
	return sum;
      }
      sum += n;
      fdmap[fd-1].rbytes += n;
      pbuf = buf + sum;
    }
    return n;
  } else 
    return -1;
}

HTAnchor *libExpandHref(struct cw_status *gp, char *url, size_t hreflen)
{
  char *ref;
  HTChildAnchor *child_anchor;
  HTParentAnchor *parent_anchor;
  HTAnchor *anchor;
    
  ref = simplifyURL(gp, url);


  anchor = (HTAnchor *) HTAnchor_findAddress(ref); /* we have to free anchor->address */
  child_anchor = (HTChildAnchor *) anchor;
  parent_anchor = HTAnchor_parent((HTAnchor *)child_anchor);

    
  if (HTAnchor_document(parent_anchor)) {
    fprintf(stderr,"libExpandHref: document structure for %s already in memory\n", ref);
  }

  CalFree(ref);
  return anchor;
}

int post_callback(struct cw_status *gp, HTRequest * request, HTStream * target)
{
  return 1;
}

int PostData(struct cw_status *gp, char *dest, char *data, long datalen, char **reply)
{
  HTAnchor *src;
  char *name;
  HTAnchor *anchor;
  struct AweReqContext *reqcon; 
  struct urlmap *cur;
  int loadstat;
  FILE *fp;
  char *buf;
  long buflen;

  anchor = libExpandHref(gp, dest, strlen(dest));

  if (!anchor)
    return 0;
  
  name = (char *)CalMalloc(strlen(dest)+16);
  sprintf(name, "internal:post/%s",dest);
  
  cur = new_urlmap(gp, name);
  cur->is_tmp = TRUE;
  reqcon = SetupRequest(gp, cur, TRUE);
  
  HTRequest_setMethod(reqcon->req, METHOD_POST);

  src = HTAnchor_findAddress(name);
  HTAnchor_setDocument((HTParentAnchor *)src, data);
  HTAnchor_setFormat((HTParentAnchor *)src, HTAtom_for("application/x-www-form-urlencoded"));
  HTAnchor_setLength((HTParentAnchor *)src, datalen);
  HTAnchor_link(src, anchor, NULL, METHOD_POST);	

  loadstat = HTUploadAnchor((HTAnchor *) src, reqcon->req,
		 HTUpload_callback);

  
  DoRequest(gp, reqcon->req, TRUE);

  if((fp = fopen(map->filename, "rb")) == NULL){

  }
  fseek(fp, 0, SEEK_END);

  buflen = ftell(fp) + 1;
  fseek(fp, 0, SEEK_SET);
  if((buf = CalMalloc(buflen)) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  *reply = buf;
  fread(buf, 1, buflen - 1, fp);
  buf[buflen - 1] = '\0';
  fclose(fp);
  free_urlmap(gp, map);
  return buflen - 1;
}

#else /* Provide dummy functions if compiled without WWWENABLE */
void NewAnchor(struct cw_status *gp, char FAR *anchor, int type)
{
  return;
}

char FAR *ResolveURL(struct cw_status *gp, char FAR *url)
{
  return url;
}

void WWWInit(struct cw_status *gp)
{
return;
}

void WWWFree(struct cw_status *gp)
{
return ;
}
int ReloadFile(struct cw_status *gp, char FAR *filename)
{
  return 1;
}
void BindURL(struct cw_status *gp, char FAR *url, char FAR *filename)
{
  return;
}

char FAR *findURL(struct cw_status *gp, char FAR *filename)
{
  return filename;
}
/* Open URL for reading. Return fd on success, -1 on error.
 */
int OpenURL(struct cw_status *gp, char FAR *url, char FAR *mode)
{
  char FAR *filename, FAR *surl;
  char tmpname[128];
  int fd;
  int write_only;

  surl = url;

  /* Assume read only if "w" not present */
  if(strncmp("w", mode, 1) == 0){
      write_only = 1;
  } else
    write_only = 0;

  if(write_only){
      /* Check if we are allowed to write to local files */
      if(gp->allow_local_write){
	  filename = ResolveURL(gp, surl);
	  strncpy(tmpname, filename, 128);
	  CalFree(filename);
	  if((fd = open(tmpname, O_WRONLY|O_CREAT)) > 0)
	      return fd;
	  else
	      return -1;
      } else {
	  errorMsg(gp, 0, WarnFILEWRI);
	  return -1;
      }
  } else
      /* Check if we are allowed to read from local files */
      if(gp->allow_local_access){
	  filename = ResolveURL(gp, surl);
	  strncpy(tmpname, filename, 128);
	  CalFree(filename);
	  if((fd = open(tmpname, O_RDONLY)) > 0)
	      return fd;
	  else
	      return -1;
      } else {
	errorMsg(gp, 0, WarnFILEREA);
	return -1;
    }
}

int Write(int fd, char FAR *buf, int nbytes)
{                               
  char tmp[1024];
  char  *p;
  int i;
  p = tmp;
  for (i=0; i<nbytes; i++) *p++ = *buf++;
  return write(fd, tmp, nbytes);
}

int Read(int fd, char FAR *buf, int nbytes)
{
  char tmp[1024];
  char  *p;
  int i;
  p = tmp;
  read(fd, tmp, nbytes);
  for (i=0; i<nbytes; i++) *buf++ = *p++;
  return 1;
}

char *ContentType(struct cw_status *gp, char *url)
{
  return NULL;
}

/*
 * Return pointer to a malloced canonized URL.
 */
char * canonizedURL(struct cw_status *gp, char *url)
{
  return strdup(url);
}

#endif

#ifdef PTHREAD
AweStream *fstream=NULL;

AweStream *AweOpen (struct cw_status *gp, char *URL)
{
  AweStream *as;
  message("AweOpen\n");
  as = CalCalloc (1, sizeof (AweStream));
  NPN_GetURL(gp->nppinstance, URL, NULL);
  if(fstream != NULL)
    fstream->prev = as;
  as->next = fstream;
  fstream = as;
  as->size = -1;
  as->url = strdup(URL);
  as->gp = gp;
  return as;
}

AweStream *NewAweStream(struct cw_status *gp, NPStream *nps)
{
  AweStream *as;
  char mess[256];
  as = CalCalloc (1, sizeof (AweStream));

  if(fstream != NULL)
    fstream->prev = as;
  as->next = fstream;
  fstream = as;
  as->size = -1;
  as->url = strdup(nps->url);
  as->nps = nps;
  as->gp = gp;
  sprintf(mess, "NAS fstream:%d\n", fstream);
  message(mess);
  return as;
}

void AweClose (AweStream *as)
{
  AweStream a;
  message("AweClose\n");
  if (as){
    if(as == fstream){
      fstream = as->next;
      if(fstream != NULL)
	fstream->prev = NULL;
    } else {
      if(as->next != NULL){
	as->next->prev = as->prev;
      } 
      as->prev->next = as->next;
    }
/*     NPN_DestroyStream(instance, as->nps, NPRES_DONE); */
    CalFree (as->buffer);
    CalFree (as);
  }
}

size_t AweRead (AweStream *as, char *buf, size_t count)
{

}

size_t AweWrite (AweStream *as, char *buf, size_t count)
{

}

int AweGetc (AweStream *as)
{
  char mess[256];
  sprintf(mess, "AweGetc as:%d\n", as);
  message(mess);
  if(as != NULL){
    if(as->position < as->csize){
      return (int)as->buffer[as->position++];
    } else if(as->size == -1){
      /* block and wait for character */
      sprintf(mess, "AweGetc block as:%d\n", as->csize);
      message(mess);
      while(as->position == as->csize && as->size == -1)
	CalServeEvents(as->gp);
      sprintf(mess, "AweGetc block as:%d\n", as->csize);
      message(mess);

      if(as->position < as->csize)
	return (int)as->buffer[as->position++];
    } 
  }
  return EOF;
}

int AweUngetc (int c, AweStream *as)
{
  if(as != NULL){
    if(as->position > 0){
      as->position--;
      as->buffer[as->position] = c;
    }
  }
  return TRUE;
}

AweStream *FindAweStream(NPStream *nps)
{
  AweStream *as;
  char mess[256];
  sprintf(mess, "fstream:%d\n", fstream);
  message(mess);
  for(as = fstream; as != NULL; as = as->next){
    sprintf(mess, "as:%s , nps:%s\n",as->url, nps->url);
    message(mess);
    if(!strcmp(as->url, nps->url))
      return as;
  }
  return NULL;
}
#else /* NOT PLUGIN */

AweStream *AweOpen (struct cw_status *gp, char *URL)
{
  AweStream *as = NULL;
  char *fname;

  fname = ResolveURL(gp, URL);
  if(fname != NULL){
    as = fopen(fname, "rb");
    CalFree(fname);
  }
  return as;
}

void AweClose (AweStream *as)
{
  fclose(as);
}

size_t AweRead (AweStream *as, char *buf, size_t count)
{
  return fread(buf, sizeof(char), count, as);
}

size_t AweWrite (AweStream *as, char *buf, size_t count)
{
  return fwrite(buf, sizeof(char), count, as);
}

int AweGetc (AweStream *as)
{
  return getc(as);
}

int AweUngetc (int c, AweStream *as)
{
  return ungetc(c, as);
}
#endif /* PLUGIN */
