/* Copyright (c) 1995 by Computers and Learning A/S (candleweb@candleweb.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (svein@candleweb.no), 
 *	    Gunnar Rnning (gunnar@candleweb.no)
 */

/* fast_lis.c */
#define FAST_LIST
#include <stdio.h>
#include "const.h"
#include "fast_lis.h"
#include "error.h"

#include "protos/canutil.h"
#include "protos/fast_lis.h"

#ifdef LISTDEBUG /* Change this if your compiler complains */
#define SEGFAULT {int foo; foo = 1/0;}
#else
#define SEGFAULT
#endif

static int valid_test2(void FAR *list_object);
static int valid_test(void FAR *list_object);

#define errorMsg(a,b,c) userMsg(c)

/* #define DEBUG 1 */

/*
 * fast_lists er rutiner og konvensjoner for standardisert listebehandling i
 * C. Hensikten med listesystemet er
 *
 * 1)  lette programmering med lister.          - 75%
 * 2)  lette og sikre resultatene av debugging. - 25%
 *
 * Prosentene til hyre viser vektleggingen av mlsettingene.
 *
 * 1) tillater ikke ekstra operasjoner ved pning og/eller lukking av lister.
 * 2) tillater ikke kompilator-warnings p.g.a. feil typer ved data-aksess.
 * 3) krever et felles sett med rutiner som kan benyttes hver gang.
 * 4) krever diverse valideringstester av listene ved debugging.
 *
 * fast_lis.c er gjennomsiktige for brukerne, kun lp m benyttes ved def av
 * structs som skal kunne listebehandles.
 */

/*
 * 11/9-1990 Knut Tvedten.
 *
 * Har lagt inn ekstra liste rutiner for } behandler de tilfeller hvor man
 * har to lp definisjoner etter hverandre. Dette er n|dvendig i tilfeller
 * hvor objekter skal v{re med i 2 lister. De nye rutinene er kopier av de
 * eksisterende. De nye rutinene har samme navnet som de gamle, men er
 * etterfulgt av 2.
 */

struct list
{
	lp(list)
	lp2(list)
};

/* dummy declaration for standardized list objects */

/****************************************************************************
						valid_test2/valid_test
  Called for a non-null object to test its validity and that of its nearest
  neighboring elements
*/

#if defined DEBUG && !defined OPTIMIZE
static int valid_test2(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */ 
  lo = (struct list FAR *) list_object;

  if (lo->id2 != LIST_ID)
    {
      errorMsg(gp, 1, ErrLISLISCAL); 
      SEGFAULT;
      return (NOT_OK);
    }
  if (lo->prev2 != NULL)
    {
      if (lo->prev2->id2 != LIST_ID)
	{
	  errorMsg(gp, 1, ErrLISPREPRE); 
	  SEGFAULT;
	  return (NOT_OK);
	}
      if (lo->prev2->next2 != lo)
	{
	  errorMsg(gp, 1, ErrLISPRENXT); 
	  SEGFAULT;
	  return (NOT_OK);
	}
    }
  if (lo->next2 != NULL)
    {
      if (lo->next2->id2 != LIST_ID)
	{
	  errorMsg(gp, 1, ErrLISNXTLIS); 
	  SEGFAULT;
	  return (NOT_OK);
	}
      if (lo->next2->prev2 != lo)
	{
	  errorMsg(gp, 1, ErrLISNXTPRE); 
	  SEGFAULT;
	  return (NOT_OK);
	}
    }
  return (OK);
}

static int valid_test(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */
  lo = (struct list FAR *) list_object;

  if (lo->id != LIST_ID)
    {
      errorMsg(gp, 1, ErrLITLISCAL); 
      SEGFAULT;
      return (NOT_OK);
    }
  if (lo->prev != NULL)
    {
      if (lo->prev->id != LIST_ID)
	{
	  errorMsg(gp, 1, ErrLITPREPRE); 
	  SEGFAULT;
	  return (NOT_OK);
	}
      if (lo->prev->next != lo)
	{
	  errorMsg(gp, 1, ErrLITPRENXT); 
	  SEGFAULT;
	  return (NOT_OK);
	}
    }
  if (lo->next != NULL)
    {
      if (lo->next->id != LIST_ID)
	{
	  errorMsg(gp, 1, ErrLITNXTLIS); 
	  SEGFAULT;
	  return (NOT_OK);
	}
      if (lo->next->prev != lo)
	{
	  errorMsg(gp, 1, ErrLITNXTPRE); 
	  SEGFAULT;
	  return (NOT_OK);
	}
    }
  return (OK);
}
#endif
#ifdef OPTIMIZE
#define valid_test(o) OK
#define valid_test2(o) OK
#endif

/****************************************************************************
  next2/next
  Returns the next object in a list, or NULL if the current object is NULL.
  */
#ifndef OPTIMIZE
void FAR *next2(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */
  
  lo = (struct list FAR *) list_object;
  
#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: next2 called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
  if (valid_test2(lo) != OK)
    {
      errorMsg(gp, 1, "next2.\n");
      SEGFAULT;
      return (NULL);
    }
#endif
  
  if (lo == NULL)
    return (NULL);
  return (lo->next2);
}

void FAR *next(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: next called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
  if (valid_test(lo) != OK)
    {
      errorMsg(gp, 1, "next.\n");
      lo = NULL;
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  return (lo->next);
}

/****************************************************************************
  prev2/prev
  Returns the prev object in a list, or NULL if the current object is NULL.
  */

void FAR *prev2(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: prev2 called for NULL object");
      SEGFAULT;
      return (NULL);
    }
  if (valid_test2(lo) != OK)
    {
      errorMsg(gp, 1, "prev2.\n");
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  return (lo->prev2);
}

void FAR *prev(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: prev called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
  if (valid_test(lo) != OK)
    {
      errorMsg(gp, 1, "prev.\n");
      lo = NULL;
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  return (lo->prev);
}

#endif

/****************************************************************************
  init_list2/init_list
  Place the first object in a list, thereby initializing a new list.
  Return value is OK or NOT_OK. Note that no input parameters
  are changed.
  Parameters: object1.
  */


int init_list2(void FAR *object1) {
  struct list FAR *lo;		/* object1 converted to list pointer */

  lo = (struct list FAR *) object1;

  if (lo == NULL)
    {
      errorMsg(gp, 1, ErrLITINTLS2); 
      return (NOT_OK);
    }
  lo->prev2 = NULL;
  lo->next2 = NULL;

  lo->id2 = LIST_ID;		/* lo validation stamp */

  return (OK);
}

int init_list(void FAR *object1) {
  struct list FAR *lo;		/* object1 converted to list pointer */

  lo = (struct list FAR *) object1;

  if (lo == NULL)
    {
      errorMsg(gp, 1, ErrLITINTILS); 
      return (NOT_OK);
    }
  lo->prev = NULL;
  lo->next = NULL;

  lo->id = LIST_ID;		/* lo validation stamp */

  return (OK);
}

#ifndef OPTIMIZE
/****************************************************************************
  is_last2/is_last
  Returns TRUE if this is the last object in a list. Else FALSE.
  */

int is_last2(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: is_last2 called for NULL object.");
      SEGFAULT;
      return (FALSE);
    }
#endif

  if (next2(lo) == NULL)
    return(TRUE);
  else
    return(FALSE);
}

int is_last(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: is_last called for NULL object.");
      SEGFAULT;
      return (FALSE);
    }
#endif

  if (next(lo) == NULL)
    return(TRUE);
  else
    return(FALSE);
}
#endif

/****************************************************************************
  last2/last
  Returns the last object in a list, or NULL if the current object is NULL.
  */


void FAR *last2(void FAR *list_object) {
  struct list FAR *lp;
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: last2 called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  lp = lo;
  while (next2(lp))
    lp = next2(lp);
  return (lp);
}

void FAR *last(void FAR *list_object) {
  struct list FAR *lp;
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: last called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  lp = lo;
  while (next(lp))
    lp = next(lp);
  return (lp);
}

#ifndef OPTIMIZE
/****************************************************************************
  is_first2/is_first
  Returns TRUE if this is the first object in a list. Else FALSE.
  */

int is_first2(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: is_first2 called for NULL object.");
      SEGFAULT;
      return (FALSE);
    }
#endif

  if (prev2(lo) == NULL)
    return(TRUE);
  else
    return(FALSE);
}

int is_first(void FAR *list_object) {
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: is_first called for NULL object.");
      SEGFAULT;
      return (FALSE);
    }
#endif

  if (prev(lo) == NULL)
    return(TRUE);
  else
    return(FALSE);
}
#endif

/****************************************************************************
  first2/first
  Returns the first object in a list, or NULL if the current object is NULL.
  */


void FAR *first2(void FAR *list_object) {
  struct list FAR *lp;
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: first2 called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  lp = lo;
  while (prev2(lp))
    lp = prev2(lp);
  return (lp);
}

void FAR *first(void FAR *list_object) {
  struct list FAR *lp;
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;

#ifdef DEBUG
  /* validation tests */
  if (lo == NULL)
    {
      errorMsg(gp, 1, "Internal error: first called for NULL object.");
      SEGFAULT;
      return (NULL);
    }
#endif

  if (lo == NULL)
    return (NULL);
  lp = lo;
  while (prev(lp))
    lp = prev(lp);
  return (lp);
}

/****************************************************************************
  count_object2/count_object.
  Return number of objects in the list.
  */

int count_object2(void FAR * list_object)
{
  struct list FAR *lo;		/* list_object converted to list pointer */
  int i = 0;

  for (lo = (struct list FAR *) first2(list_object);
       lo != NULL;
       lo = (struct list FAR *) next2(lo), i++);

  return(i);
}

int count_object(void FAR * list_object)
{
  struct list FAR *lo;		/* list_object converted to list pointer */
  int i = 0;

  for (lo = (struct list FAR *) first(list_object);
       lo != NULL;
       lo = (struct list FAR *) next(lo), i++);

  return(i);
}

/****************************************************************************
  place_prev2/place_prev
  Place a list object before (previous object of) another list object.
  Return value is OK or NOT_OK. Note that no input parameters
  are changed.
  Parameters: object1 is an object in the list, object2 is the object to
  be placed before object1.
  */


int place_prev2(void FAR *object1, void FAR *object2) {
  struct list FAR *lo1, FAR *lo2;	/* list objects converted to list pointers */

  lo1 = (struct list FAR *) object1;
  lo2 = (struct list FAR *) object2;

  if (lo1 == NULL)
    {
      errorMsg(gp, 1, ErrLITINTPP2); 
      return (NOT_OK);
    }
#ifdef DEBUG
  /* validation tests */
  if (valid_test2(lo1) != OK)
    {
      errorMsg(gp, 1, "place_prev2.\n"); 
      SEGFAULT;
      return (NOT_OK);
    }
  if (!lo2)
    {
      errorMsg(gp, 1, ErrLITINTPP2); 
      SEGFAULT;
      return (NOT_OK);
    }
#endif
  lo2->prev2 = lo1->prev2;
  lo2->next2 = lo1;
  lo1->prev2 = lo2;
  if (lo2->prev2)
    lo2->prev2->next2 = lo2;

  lo2->id2 = LIST_ID;		/* lo2 validation stamp */

  return (OK);
}

int place_prev(void FAR *object1, void FAR *object2) {
  struct list FAR *lo1, FAR *lo2;	/* list objects converted to list pointers */

  lo1 = (struct list FAR *) object1;
  lo2 = (struct list FAR *) object2;

  if (lo1 == NULL)
    {
      errorMsg(gp, 1, ErrLITINTPP1); 
      return (NOT_OK);
    }
#ifdef DEBUG
  /* validation tests */
  if (valid_test(lo1) != OK)
    {
      errorMsg(gp, 1, "place_prev.\n"); 
      SEGFAULT;
      return (NOT_OK);
    }
  if (!lo2)
    {
      errorMsg(gp, 1, ErrLITINTPP1); 
      SEGFAULT;
      return (NOT_OK);
    }
#endif
  lo2->prev = lo1->prev;
  lo2->next = lo1;
  lo1->prev = lo2;
  if (lo2->prev)
    lo2->prev->next = lo2;

  lo2->id = LIST_ID;		/* lo2 validation stamp */

  return (OK);
}


/****************************************************************************
  place_next2/place_next
  Place a list object after (next object of) another list object.
  Return value is OK or NOT_OK. Note that no input parameters
  are changed.
  */

int place_next2(void FAR *object1, void FAR *object2) {
  struct list FAR *lo1, FAR *lo2;	/* list objects converted to list pointers */

  lo1 = (struct list FAR *) object1;
  lo2 = (struct list FAR *) object2;

  if (lo1 == NULL)
    {
      errorMsg(gp, 1, ErrLITINTPN2); 
      return (NOT_OK);
    }
#ifdef DEBUG
  /* validation tests */
  if (valid_test2(lo1) != OK)
    {
      errorMsg(gp, 1, "place_next2.\n"); 
      SEGFAULT;
      return (NOT_OK);
    }
  if (!lo2)
    {
      errorMsg(gp, 1, ErrLITINTPN2); 
      SEGFAULT;
      return (NOT_OK);
    }
#endif
  lo2->next2 = lo1->next2;
  lo2->prev2 = lo1;
  lo1->next2 = lo2;
  if (lo2->next2)
    lo2->next2->prev2 = lo2;

  lo2->id2 = LIST_ID;		/* lo2 validation stamp */

  return (OK);
}

int place_next(void FAR *object1, void FAR *object2) {
  struct list FAR *lo1, FAR *lo2;	/* list objects converted to list pointers */

  lo1 = (struct list FAR *) object1;
  lo2 = (struct list FAR *) object2;

  if (lo1 == NULL)
    {
      errorMsg(gp, 1, ErrLITINTPN1); 
      return (NOT_OK);
    }
#ifdef DEBUG
  /* validation tests */
  if (valid_test(lo1) != OK)
    {
      errorMsg(gp, 1, "place_next.\n"); 
      SEGFAULT;
      return (NOT_OK);
    }
  if (!lo2)
    {
      errorMsg(gp, 1, ErrLITINTPN1); 
      SEGFAULT;
      return (NOT_OK);
    }
#endif
  lo2->next = lo1->next;
  lo2->prev = lo1;
  lo1->next = lo2;
  if (lo2->next)
    lo2->next->prev = lo2;

  lo2->id = LIST_ID;		/* lo2 validation stamp */

  return (OK);
}

/****************************************************************************
  remove_at2/remove_at
  Removes a list object from the list.
  Return value is OK or NOT_OK. Note that no input parameters
  are changed.
  */

int remove_at2(void FAR *object1) {
  struct list FAR *lo1;		/* list objects converted to list pointers */

  lo1 = (struct list FAR *) object1;

  if (lo1 == NULL)
    {
      errorMsg(gp, 1, ErrLITINTRA2); 
      return (NOT_OK);
    }
#ifdef DEBUG
  /* validation tests */
  if (valid_test2(lo1) != OK)
    {
      errorMsg(gp, 1, "remove_at2.\n"); 
      SEGFAULT;
      return (NOT_OK);
    }
#endif
  if (lo1->prev2)
    lo1->prev2->next2 = lo1->next2;
  if (lo1->next2)
    lo1->next2->prev2 = lo1->prev2;

  lo1->id2 = NOT_LIST;

  lo1->prev2 = lo1->next2 = NULL;
  return(OK);
}

int remove_at(void FAR *object1) {
  struct list FAR *lo1;		/* list objects converted to list pointers */

  lo1 = (struct list FAR *) object1;

  if (lo1 == NULL)
    {
      errorMsg(gp, 1, ErrLITINTRA1); 
      return (NOT_OK);
    }
#ifdef DEBUG
  /* validation tests */
  if (valid_test(lo1) != OK)
    {
      errorMsg(gp, 1, "remove_at.\n");
      SEGFAULT;
      return (NOT_OK);
    }
#endif
  if (lo1->prev)
    lo1->prev->next = lo1->next;
  if (lo1->next)
    lo1->next->prev = lo1->prev;

  lo1->id = NOT_LIST;

  lo1->prev = lo1->next = NULL;
  return (OK);
}


int copy_file(char dest_name[], char source_name[]) {
  char buf[1024];
  int i;
  FILE *dest, *source;

  if ((source = fopen(source_name, "rb")) == NULL)
    return (NOT_OK);
  dest = fopen(dest_name, "wb");
  if ((dest = fopen(dest_name, "wb")) == NULL) {
    fclose(source);
    return (NOT_OK);
  }
  while (!feof(source)) {
    i = fread(buf, 1, 1024, source);
    if (fwrite(buf, 1, i, dest) != (size_t) i)
      errorMsg(gp, 1, ErrFILDSKFUL);
  }
  fclose(source);
  fclose(dest);
  return (OK);
  /*
   * strcpy(command,"COPY "); strcat(command,source_name);
   * strcat(command," "); strcat(command,dest_name); strcat(command," >
   * NUL"); system(command); return(OK);
   */
}


#ifndef OPTIMIZE
/*
 * Return true if list_object is a valid list.
 */
int is_list(void FAR *list_object)
{
  struct list FAR *lo;		/* list_object converted to list pointer */

  lo = (struct list FAR *) list_object;
  
  if (lo->id != LIST_ID)
    return (FALSE);

  return (TRUE);
}
#endif

