/*
 * July 5, 1991
 * Copyright 1991 Lance Norskog And Sundry Contributors
 * This source code is freely redistributable and may be used for
 * any purpose.  This copyright notice must be maintained. 
 * Lance Norskog And Sundry Contributors are not responsible for 
 * the consequences of using this software.
 */

/*
 * Sound Tools raw format file.
 *
 * Includes .ub, .uw, .sb, .sw, and .ul formats at end
 */

/*
 * Notes: most of the headerless formats set their handlers to raw
 * in their startread/write routines.  
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include "st.h"
#include "libst.h"
#include "protos.h"

extern int summary, verbose;

void rawstartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
}


int
rawread(gp, ft, buf, nsamp) 
struct cw_status *gp;
ft_t ft;
long *buf, nsamp;
{
	long datum;
	int done = 0;

	switch(ft->info.size) {
		case ST_BYTE: switch(ft->info.style) {
			case SIGN2:
				while(done < nsamp) {
					if (read (ft->fp, &datum, 1) == 0)
						return done;
					/* scale signed up to long's range */
					*buf++ = LEFT(datum, 24);
					done++;
				}
				return done;
			case UNSIGNED:
				while(done < nsamp) {
				  if (read (ft->fp, &datum, 1) == 0)
				    return done;
					/* Convert to signed */
					datum ^= 128;
					/* scale signed up to long's range */
					*buf++ = LEFT(datum, 24);
					done++;
				}
				return done;
			case ULAW:
				/* grab table from Posk stuff */
				while(done < nsamp) {
					if (read (ft->fp, &datum, 1) == 0)
					  return done;
					datum = st_ulaw_to_linear(datum);
					/* scale signed up to long's range */
					*buf++ = LEFT(datum, 16);
					done++;
				}
				return done;
			case ALAW:
				fail(gp, "No A-Law support");
				return done;
			}
		case ST_WORD: switch(ft->info.style) {
			case SIGN2:
				while(done < nsamp) {
				  if (read (ft->fp, &datum, 2) == 0)
				    return done;
				  if (ft->swap) datum = swapw(gp, datum);
				  /* scale signed up to long's range */
					*buf++ = LEFT(datum, 16);
					done++;
				}
				return done;
			case UNSIGNED:
				while(done < nsamp) {
				  if (read (ft->fp, &datum, 2) == 0)
				    return done;
				  if (ft->swap) datum = swapw(gp, datum);
					/* Convert to signed */
					datum ^= 0x8000;
					/* scale signed up to long's range */
					*buf++ = LEFT(datum, 16);
					done++;
				}
				return done;
			case ULAW:
				fail(gp, "No U-Law support for shorts");
				return done;
			case ALAW:
				fail(gp, "No A-Law support");
				return done;
			}
		}
	fail(gp, "Drop through in rawread!");
	return done;
}

/*
 * Set parameters to the fixed parameters known for this format,
 * and change format to raw format.
 */

static void rawdefaults(struct cw_status *, ft_t);

/* Signed byte */
void sbstartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
	ft->info.size = ST_BYTE;
	ft->info.style = SIGN2;
	rawdefaults(gp, ft);
}

void ubstartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
	ft->info.size = ST_BYTE;
	ft->info.style = UNSIGNED;
	rawdefaults(gp, ft);
}

void uwstartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
	ft->info.size = ST_WORD;
	ft->info.style = UNSIGNED;
	rawdefaults(gp, ft);
}

void swstartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
	ft->info.size = ST_WORD;
	ft->info.style = SIGN2;
	rawdefaults(gp, ft);
}

void ulstartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
	ft->info.size = ST_BYTE;
	ft->info.style = ULAW;
	rawdefaults(gp, ft);
}

static
void rawdefaults(gp, ft)
struct cw_status *gp;
ft_t ft;
{
	if (ft->info.rate == 0)
		ft->info.rate = 8000;
	if (ft->info.channels == -1);
		ft->info.channels = 1;
}
