/*
 * September 25, 1991
 * Copyright 1991 Guido van Rossum And Sundry Contributors
 * This source code is freely redistributable and may be used for
 * any purpose.  This copyright notice must be maintained. 
 * Guido van Rossum And Sundry Contributors are not responsible for 
 * the consequences of using this software.
 */

/*
 * Sound Tools Sun format with header (SunOS 4.1; see /usr/demo/SOUND).
 * NeXT uses this format also, but has more format codes defined.
 * DEC uses a slight variation and swaps bytes.
 * We only support the common formats.
 * Output is always in big-endian (Sun/NeXT) order.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include "st.h"
#include "protos.h"
#include "protos/memory.h"

/* Magic numbers used in Sun and NeXT audio files */
#define SUN_MAGIC 	0x2e736e64		/* Really '.snd' */
#define SUN_INV_MAGIC	0x646e732e		/* '.snd' upside-down */
#define DEC_MAGIC	0x2e736400		/* Really '\0ds.' (for DEC) */
#define DEC_INV_MAGIC	0x0064732e		/* '\0ds.' upside-down */
#define SUN_HDRSIZE	24			/* Size of minimal header */
#define SUN_UNSPEC	((unsigned)(~0))	/* Unspecified data size */
#define SUN_ULAW	1			/* u-law encoding */
#define SUN_LIN_8	2			/* Linear 8 bits */
#define SUN_LIN_16	3			/* Linear 16 bits */
/* The other formats are not supported by sox at the moment */

/* Private data used by writer */
struct aupriv {
	unsigned long data_size;
};

void austartread(gp, ft) 
struct cw_status *gp;
ft_t ft;
{
	/* The following 6 variables represent a Sun sound header on disk.
	   The numbers are written as big-endians.
	   Any extra bytes (totalling hdr_size - 24) are an
	   "info" field of unspecified nature, usually a string.
	   By convention the header size is a multiple of 4. */
	unsigned long magic;
	unsigned long hdr_size;
	unsigned long data_size;
	unsigned long encoding;
	unsigned long sample_rate;
	unsigned long channels;

	register int i;
	unsigned char *buf;


	/* Check the magic word */
	magic = rlong(gp, ft);
	if (magic == DEC_INV_MAGIC) {
		ft->swap = 1;
		report(gp, "Found inverted DEC magic word");
	}
	else if (magic == SUN_INV_MAGIC) {
		ft->swap = 1;
		report(gp, "Found inverted Sun/NeXT magic word");
	}
	else if (magic == SUN_MAGIC) {
		ft->swap = 0;
		report(gp, "Found Sun/NeXT magic word");
	}
	else if (magic == DEC_MAGIC) {
		ft->swap = 0;
		report(gp, "Found DEC magic word");
	}
	else
		fail(gp, "Sun/NeXT/DEC header doesn't start with magic word\nTry the '.ul' file type with '-t ul -r 8000 filename'");

	/* Read the header size */
	hdr_size = rlong(gp, ft);
	if (hdr_size < SUN_HDRSIZE)
		fail(gp, "Sun/NeXT header size too small.");

	/* Read the data size; may be ~0 meaning unspecified */
	data_size = rlong(gp, ft);

	/* Read the encoding; there are some more possibilities */
	encoding = rlong(gp, ft);

	/* Translate the encoding into style and size parameters */
	switch (encoding) {
	case SUN_ULAW:
		ft->info.style = ULAW;
		ft->info.size = ST_BYTE;
		break;
	case SUN_LIN_8:
		ft->info.style = SIGN2;
		ft->info.size = ST_BYTE;
		break;
	case SUN_LIN_16:
		ft->info.style = SIGN2;
		ft->info.size = ST_WORD;
		break;
	default:
		report(gp, "encoding: 0x%lx", encoding);
		fail(gp, "Unsupported encoding in Sun/NeXT header.\nOnly U-law, signed bytes, and signed words are supported.");
		/*NOTREACHED*/
	}

	/* Read the sampling rate */
	sample_rate = rlong(gp, ft);
	ft->info.rate = sample_rate;

	/* Read the number of channels */
	channels = rlong(gp, ft);
	ft->info.channels = channels;

	/* Skip the info string in header; print it if verbose */
	hdr_size -= SUN_HDRSIZE; /* #bytes already read */
	if (hdr_size > 0) {
		buf = (unsigned char *) CalMalloc(hdr_size + 1);
		for(i = 0; i < hdr_size; i++) {
		  if (read (ft->fp, &buf[i], 1) == 0)
		    fail(gp, "Unexpected EOF in Sun/NeXT header info.");
		}
		buf[i] = 0;
		report(gp, "Input file %s: Sun header info: %s", ft->filename, buf);
	}
}













