#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>

/* Uses soundtools and playmidi */
#include "soundtoo/st.h"
#include "soundtoo/protos.h"
#include "playmidi/playmidi.h"
#ifdef linux
#include <sys/soundcard.h>
#endif

#include "candle.h"
#include "error.h"
#include "wwwcom.h"

#include "protos/canutil.h"
#include "protos/memory.h"

#if (defined UNIX && defined SOUNDSUPPORT)
/* These are globally shared resources */
extern int audio, seqfd, mtimes, wtimes;
void synth_setup (void);
void playmidi (char *, int);
#endif

static void reduceChannels (struct cw_status *gp, long *size, long *buffer)
{
  int i;

  for (i = 0; i < *size; i++)
    buffer[i] = buffer[2*i]/2+buffer[2*i+1]/2;
  *size /= 2;
}

/* Running as a separate process */

void waveReset (int x)
{
  wtimes = 0;
  if (audio != -1) {
    ioctl (audio, SNDCTL_DSP_RESET, 0);
    close (audio);
    audio = -1;
  }
  signal (SIGUSR1, waveReset);
}

void waveserver (struct cw_status *gp)
{
  int stereo, rate, size = 16;
  long bufsize, actualsize = 0, samplesread, i;
  char buf[1024], type, ext[1024];
  fd_set rfds;
  struct soundstream informat;
  long *ibuf, *ptrs32bit;
  short *ptrs16bit;
  unsigned char *ptru8bit;
  int pitch, n;

  FD_ZERO (&rfds);
  FD_SET (gp->wavefd[0], &rfds);

  signal (SIGUSR1, waveReset);

  while (1) {
    if ((n = select (gp->wavefd[0]+1, &rfds, NULL, NULL, NULL)) > 0) {
      wtimes = 1;
      n = read (gp->wavefd[0], &buf[0], 1024);
      sscanf (buf, "%ld %s %c", &bufsize, &ext[0], &type);
      
      init (gp, &informat);
      informat.filename = NULL;
      
      informat.info.rate = pitch;
      informat.fp = gp->wavefd[0];

      /* Assume ResolveURL gives reasonable names */
      informat.filetype = ext; /* File type extension */
      gettype (gp, &informat);
      (* informat.h->startread)(gp, &informat);
      checkformat(gp, &informat);

      actualsize = 0; samplesread = 0;
      if ((ibuf = (long *)CalMalloc (bufsize*sizeof (long))) == NULL) {
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit (gp, NOT_OK);
      }

      while (wtimes > 0 &&
	     (samplesread =
	      (*informat.h->read)(gp, &informat, &ibuf[actualsize], bufsize))
	     > 0) actualsize += samplesread;
      (* informat.h->stopread)(gp, &informat);

      /* Sample is read, now play it */
#ifdef linux
      if (wtimes > 0 &&(audio = open ("/dev/dsp", O_WRONLY, 0))) {
#endif
#ifdef sun
      if (wtimes > 0 && (audio = open ("/dev/audio", O_WRONLY, 0))) {
#endif
#if (defined linux || defined sun)
	ioctl (audio, SOUND_PCM_WRITE_BITS, &size);
	stereo = informat.info.channels;
	ioctl (audio, SOUND_PCM_WRITE_CHANNELS, &stereo);
	rate = informat.info.rate;
	ioctl (audio, SOUND_PCM_WRITE_RATE, &rate);
	
	/* Postprocess sample to approriate format */
	while (informat.info.channels > stereo)
	  reduceChannels (gp, &actualsize, ibuf);
	
	ptrs32bit = ibuf;
	if (size == 16) {
	  ptrs16bit = (short *)ptrs32bit;
	  for (i = 0; i < actualsize; i++)
	    /* 32-bit signed to 16 bit signed */
	    *(ptrs16bit++) = *(ptrs32bit++)/0x10000;
	}
	else if (size == 8) {
	  ptru8bit = (unsigned char *)ptrs32bit;
	  for (i = 0; i < actualsize; i++)
	    /* 32-bit signed to 8-bit unsigned */
	    *(ptru8bit++) = *(ptrs32bit++)/0x1000000^80;
	}
	while (audio != -1 && wtimes) {
	  write (audio, (char *)ibuf, actualsize*(size/8));
	  if (type == 'S') --wtimes;
	}
	close (audio);
	audio = -1;
	buf[0] = '\0';
      }
#endif
      free (ibuf);
    }
  }
}

/* MIDI, running as a separate process */

void midiReset (int x)
{
  assert (seqfd != -1); /* Should never be closed */
  mtimes = -1;
  ioctl (seqfd, SNDCTL_SEQ_RESET, 0);
  signal (SIGUSR1, midiReset);
}

void midiserver (struct cw_status *gp)
{
  char *buf, type, in[1024];
  fd_set rfds;
  long dataread, bufsize, actualsize;
  int n;

  FD_ZERO (&rfds);
  FD_SET (gp->midifd[0], &rfds);

  signal (SIGUSR1, midiReset);

  if ((seqfd = open(SEQUENCER_DEV, O_WRONLY, 0)) < 0) {
    /*    perror("open " SEQUENCER_DEV); */
    exit(OK);
  }
  synth_setup ();

  while (1) {
    if (select (gp->midifd[0]+1, &rfds, NULL, NULL, NULL) > 0) {
      mtimes = 1; /* Slightly awkward maybe, but it works... */
      n = read (gp->midifd[0], &in[0], 1024);
      sscanf (in, "%ld %c", &bufsize, &type);

      if ((buf = (char *)CalMalloc (bufsize)) == NULL) {
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit (gp, NOT_OK);
      }

      for (actualsize = 0; actualsize < bufsize; actualsize += dataread) {
	dataread = read (gp->midifd[0], &buf[actualsize], bufsize-actualsize);
      }
      while (seqfd != -1 && mtimes > 0) {
	playmidi (buf, actualsize);
	if (type == 'S') --mtimes;
      }
      free (buf);
    }
  }
}





