/* Copyright (c) 1995 by Computers and Learning A/S (candleweb@candleweb.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */

#include <stdio.h>
#include <string.h> 
#include <stdlib.h> 
#include <malloc.h>
#include <assert.h>


#include "candle.h"
#include "const.h"
#include "fast_lis.h"
#include "graphic.h"
#include "simulate.h"
#include "learnuni.h"
#include "funcname.h"
#include "function.h"
#include "operator.h"
#include "error.h"
#include "nodes.h"
#include "lex.h"
#include "parser.h"

#include "protos/memory.h"
#include "sysproto.h"
#include "protos/canutil.h"
#include "protos/simulate.h"
#include "protos/readsim.h"
#include "protos/fast_lis.h"
#include "protos/instsim.h"
#include "protos/instance.h"
#include "protos/creatsim.h"
#include "protos/parser.h"

int IF(struct cw_status *gp, struct sim_actions FAR *, FILE *);
int FOR(struct cw_status *gp, struct sim_actions FAR *, FILE *);
int SWITCH(struct cw_status *gp, struct sim_actions FAR *, FILE *);
int WHILE(struct cw_status *gp, struct sim_actions FAR *, FILE *);


static long el_in_array(struct cw_status *gp, struct intarray FAR *array) 
{
  long i=0, FAR *ind, dim=0, index=0;
  struct vinst FAR *vi;
  struct param FAR *par, *fpar;
  char errormsg[256];

#ifdef DEBUG
  if(array->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal array->type in el_in_array.\n");
  }
#endif
  vi = array->which.vi;
  

  fpar = par = array->indeks;
  while(dim < vi->parent->ndim){
    if(par != NULL)
      i = par_value(gp, par);
    else {
      sprintf(errormsg, "Error : Too few indexes in access to array %s.", 
	      vi->parent->name);
      paramError(gp, fpar, errormsg);
      return (NOT_OK);
    }
    if(i < 0 || i >= (long) vi->size[dim]){
      sprintf(errormsg, ErrARROUTBOU,
	      vi->parent->name, i);
      paramError(gp, par, errormsg);
      return (NOT_OK);
    }
    index += i * vi->size[dim+vi->parent->ndim]; 
    par = (struct param FAR *)next(par);
    dim++;
  }
  ind = vi->value.elem;
  if (ind) {
    ind += index;
    return(*ind);
  } else
    return(0);
}

static double el_fl_array(struct cw_status *gp, struct intarray FAR *array) 
{
  long i=0, dim=0, index=0;
  double FAR *ind;
  struct vinst FAR *vi;
  struct param FAR *par;
  char errormsg[256];

#ifdef DEBUG
  if(array->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal array->type in elp_in_array.\n");
  }
#endif  
  vi = array->which.vi;
  

  par = array->indeks;
  while(dim < vi->parent->ndim){
    if(par != NULL)
      i = par_value(gp, par);
    else {
      sprintf(errormsg, ErrTOOFEWIND, 
	      vi->parent->name);
      errorMsg(gp, 1, errormsg);
      return ((float) NOT_OK);
    }
    if(i < 0 || i >= (long)vi->size[dim]){
      sprintf(errormsg, ErrARROUTBOU,
	      vi->parent->name, i);
      paramError(gp, par, errormsg);
      return ((double) NOT_OK);
    }
    index += i * vi->size[dim+vi->parent->ndim]; 
    par = (struct param FAR *)next(par);
    dim++;
  }
  ind = vi->value.felem;
  if (ind) {
    ind += index;
    return(*ind);
  } else
    return(0.0F);
}

static char FAR *el_txt_array(struct cw_status *gp, struct intarray FAR *array) 
{
  long i=0, dim=0, index=0;
  char FAR *FAR *ind;
  struct vinst FAR *vi;
  struct param FAR *par;
  char errormsg[256];

#ifdef DEBUG
  if(array->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal array->type in el_txt_array.\n");
  }
#endif
  vi = array->which.vi;
  

  par = array->indeks;
  while(dim < vi->parent->ndim){
    if(par != NULL)
      i = par_value(gp, par);
    else {
      sprintf(errormsg, ErrTOOFEWIND, 
	      vi->parent->name);
      errorMsg(gp, 1, errormsg);
      return (NULL);
    }
    if(i < 0 || i >= (long) vi->size[dim]){
      sprintf(errormsg, ErrARROUTBOU,
	      vi->parent->name, i);
      paramError(gp, par, errormsg);
      return NULL;
    }
    index += i * vi->size[dim+vi->parent->ndim]; 
    par = (struct param FAR *)next(par);
    dim++;
  }
  ind = vi->value.telem;
  if (ind) {
    ind += index;
    return(*ind);
  } else
    return NULL;
}

int compute_valtype(struct cw_status *gp, struct oper FAR *op) 
{
  char errormsg[256];
  if (!op) {
    sprintf (errormsg, ErrINTOPNUVA);
    errorMsg(gp, 1, errormsg);
    return(0);
  }
  switch(op->flag) {
  case INTARRAY :
    if(op->left.array->indeks != NULL)
      return op->valtype = T_INT;
    else
      return op->valtype = T_INT | T_ARRAY;  
  case INTVAL  :
  case INTOPER :
  case INTVAR :
  case INTCONST :
  case BINTFUNC :
  case UINTFUNC :
    return op->valtype = T_INT;
  case FLOATARRAY :
    if(op->left.array->indeks != NULL)
      return op->valtype = T_FLOAT;
    else
      return op->valtype = T_FLOAT | T_ARRAY;
  case FLOATVAL :
  case FLOATOPER :
  case FLOATVAR :
  case FLOATCONST :
  case BFLOATFUNC : 
  case UFLOATFUNC :
    return op->valtype = T_FLOAT;
  case TEXTARRAY :
    if(op->left.array->indeks != NULL)
      return op->valtype = T_TEXT;
    else
      return op->valtype = T_TEXT | T_ARRAY;
  case TEXTVAL :
  case TEXTOPER:
  case TEXTVAR :
  case TEXTCONST : 
  case BTEXTFUNC :
  case UTEXTFUNC :
    return op->valtype = T_TEXT;
  case ATTROPER :
  case ATTROPER | T_ARRAY:
  case T_OPER :
  case T_VOID:
    return op->valtype = T_VOID;
  default : 
    sprintf(errormsg, 
	    ErrINTILLFVA, 
	    op->flag);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

double f_calculate(struct cw_status *gp, struct oper FAR *op);
char FAR *t_calc_dup(struct cw_status *gp, struct oper FAR *op);

/********************************************************
**                                          calculate  **
**  Return value of op (expression).                   **
********************************************************/
long calculate(struct cw_status *gp, struct oper FAR *op) 
{
  long lval, rval;
  char errormsg[256];

  assert(op);

#ifdef DEBUG
  if(op->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal op->type in calculate.\n");
  }
#endif


  switch(op->flag){
  case INTOPER :  /* indicates an operator, unary or binary */
    /* op-flag also indicates the desired type of the current intermediate
       value (either as specified by a cast, or as inherited from
       the sub-expressions). This routine is not called if the
       op-flag is 'r' (float) */
    if(op->operation.lng == l_ass || op->operation.lng == l_assplus ||
       op->operation.lng == l_assminus || op->operation.lng == l_assmod ||
       op->operation.lng == l_assdiv || op->operation.lng == l_assmult ||
       op->operation.lng == l_asslshift || op->operation.lng == l_assrshift ||
       op->operation.lng == l_assbor || op->operation.lng == l_assband){
      rval = calculate(gp, op->right.op);
      return((*op->operation.lng)(gp, op,0,rval));
    } else if(op->operation.ltxt == t_eq || op->operation.ltxt == t_ne ||
	      op->operation.ltxt == t_lt || op->operation.ltxt == t_le ||
	      op->operation.ltxt == t_gt || op->operation.ltxt == t_ge){
      char *tlval=NULL, *trval=NULL;
      int retval;
      switch(valtype(op->left.op)){
      case T_TEXT:
	tlval = t_calc_dup(gp, op->left.op);
	break;
      default:
	sprintf(errormsg, ErrINTILVTCA);
	errorMsg(gp, 1, errormsg);
	return(0);
      }
      if (RETURN)
	return(1);
      if (op->right.op) { /* NULL if unary operator */
	switch(valtype(op->right.op)) {
	case  T_TEXT: 
	  trval = t_calc_dup(gp, op->right.op);
	  break;
	default: 
	  sprintf(errormsg, ErrILLINTEXP);
	  errorMsg(gp, 1, errormsg);
	  return(NOT_OK);
	}
      }
      if (RETURN)
	return(1);
      retval = (*op->operation.ltxt)(gp, op, tlval, trval);
      if(tlval != NULL)
	free(tlval);
      if(trval != NULL)
	free(trval);
      return retval;
    }
    /* calculate left and possibly right oper */
    switch(valtype(op->left.op)) {
    case  T_INT : 
      lval = calculate(gp, op->left.op);
      break;
    case  T_FLOAT : 
      lval = (long) f_calculate(gp, op->left.op);
      break;
    default : 
      sprintf(errormsg, ErrINTILVTCA);
      errorMsg(gp, 1, errormsg);
      return(0);
    }
    if(op->operation.lng == l_or && lval) return 1;
    else if(lval == 0 && op->operation.lng == l_and) return 0;
		    
    if (RETURN)
      return(1);
    if (op->right.op) { /* NULL if unary operator */
      switch(valtype(op->right.op)) {
      case  T_INT : 
	rval = calculate(gp, op->right.op);
	break;
      case  T_FLOAT   : 
	rval = (long) f_calculate(gp, op->right.op);
	break;
      default         : 
	sprintf(errormsg, ErrILLINTEXP);
	errorMsg(gp, 1, errormsg);
	return(NOT_OK);
      }
    }
    if (RETURN)
      return(1);
    return((*op->operation.lng)(gp, op,lval,rval));
  case  INTVAL : return(op->left.value);
  case  FLOATVAL : return((long)op->left.fvalue);
  case  INTVAR : 
#ifdef DEBUG
    switch(op->type){
    case DO_VDECL :
      return(op->left.vd->vi->value.value);
    case DO_VINST :
#endif
      return(op->left.vi->value.value);
#ifdef DEBUG
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return(NOT_OK);
    }
#endif
  case  FLOATVAR : 
#ifdef DEBUG
    switch(op->type){
    case DO_VDECL :
      return((long) op->left.vd->vi->value.fvalue);
    case DO_VINST :
#endif
	return((long) op->left.vi->value.fvalue);
#ifdef DEBUG
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return(NOT_OK);
    }
#endif
  case INTCONST : return(op->left.cconst->thisc.value);
  case FLOATCONST : return((long)op->left.cconst->thisc.fvalue);
  case INTARRAY : return(el_in_array(gp, op->left.array));
  case FLOATARRAY : return((long)el_fl_array(gp, op->left.array));
  case BFLOATFUNC : return((long)(*op->left.do_act->function.fdo_act)(gp, op->left.do_act->actual.par));
  case BINTFUNC : return((*op->left.do_act->function.do_act)
		       (gp, op->left.do_act->actual.par));
  case UINTFUNC : 
    return(IntFunc(gp, op->left.do_act->function.func_act, op->left.do_act->actual.par));	
  case UFLOATFUNC :
    return((long) FloatFunc(gp, op->left.do_act->function.func_act, op->left.do_act->actual.par));
  case FLOATOPER :
    return ((long) f_calculate(gp, op));
  case TEXTVAL :
  case TEXTARRAY :
  case TEXTVAR :
  case TEXTCONST :
  case BTEXTFUNC :
  case UTEXTFUNC :
    sprintf(errormsg, ErrTXTINTEXP, op->line);
    errorMsg(gp, 1, errormsg);
    return(NOT_OK);
  default   : 
    sprintf(errormsg, ErrINTUNFLCA, 
	    op->flag);
    errorMsg(gp, 1, errormsg);
    return(NOT_OK);
  }
}

double f_calculate(struct cw_status *gp, struct oper FAR *op) 
{
  double lval, rval;
  char errormsg[256];

  if (!op) return(0.0F);

#ifdef DEBUG
  if(op->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal op->type in f_calculate.\n");
  }
#endif

  switch(op->flag){
  case FLOATOPER : /* indicates an operator, unary or binary */
    /* op-flag also indicates the desired type of the current intermediate
       value (either as specified by a cast, or as inherited from
       the sub-expressions). This routine is not called if the
       op-flag is 'i' (int) */

    if(op->operation.flt == r_ass || op->operation.flt == r_assplus ||
       op->operation.flt == r_assminus || op->operation.flt == r_assdiv ||
       op->operation.flt == r_assmod || op->operation.flt == r_assmult){
      rval = f_calculate(gp, op->right.op);
      return((*op->operation.flt)(gp, op,0,rval));
    }

    /* calculate  left and possibly right oper */
    switch(valtype(op->left.op)) {
    case  T_INT : 
      lval = (double) calculate(gp, op->left.op);
      break;
    case  T_FLOAT :	
      lval = f_calculate(gp, op->left.op);
      break;
    default :
      sprintf(errormsg, ErrINTILVTCA);
      errorMsg(gp, 1, errormsg);
      return((float) NOT_OK);
    }
    if(lval && op->operation.flt == r_or) return 1.0F;
    if(lval == 0 && op->operation.flt == r_and) return 0.0F;
    if (RETURN)
      return(1.0F);
    if (op->right.op) { /* NULL if unary operator */
      switch(valtype(op->right.op)) {
      case  T_INT : 
	rval = (float) calculate(gp, op->right.op);
	break;
      case  T_FLOAT : 
	rval = f_calculate(gp, op->right.op);
	break;
      default : 
	sprintf(errormsg, 
		ErrINTIRVTCA);
	errorMsg(gp, 1, errormsg);
	return(0.0F);
      }
    }
    if (RETURN)
      return(1.0F);
    return((*op->operation.flt)(gp, op,lval,rval));
  case  INTVAL : return((double)op->left.value);
    case  FLOATVAL : return(op->left.fvalue);
  case  INTVAR : 
#ifdef DEBUG
    switch(op->type){
    case DO_VDECL :
      return((double) op->left.vd->vi->value.value);
    case DO_VINST :
#endif
      return((double) op->left.vi->value.value);
#ifdef DEBUG
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return((float)NOT_OK);
    }
#endif
  case  FLOATVAR : 
#ifdef DEBUG
    switch(op->type){
    case DO_VDECL :
      return(op->left.vd->vi->value.fvalue);
    case DO_VINST :
#endif
      return(op->left.vi->value.fvalue);
#ifdef DEBUG
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return((float)NOT_OK);
    }
#endif
  case INTCONST : return((double)op->left.cconst->thisc.value);
  case FLOATCONST : return(op->left.cconst->thisc.fvalue);
  case INTARRAY : return((double)el_in_array(gp, op->left.array));
  case FLOATARRAY : return(el_fl_array(gp, op->left.array));
  case BINTFUNC : return((double)(*op->left.do_act->function.do_act)(gp, op->left.do_act->actual.par));
  case BFLOATFUNC : return((*op->left.do_act->function.fdo_act)(gp, op->left.do_act->actual.par));
  case UINTFUNC : 
    return((double) IntFunc(gp, op->left.do_act->function.func_act, op->left.do_act->actual.par));	
  case UFLOATFUNC :
    return(FloatFunc(gp, op->left.do_act->function.func_act, op->left.do_act->actual.par));
  case INTOPER :
    return ((double) calculate(gp, op));
  case TEXTVAL :
  case TEXTARRAY :
  case TEXTVAR :
  case TEXTCONST :
  case BTEXTFUNC :
  case UTEXTFUNC :
    errorMsg(gp, 1, ErrTXTFLOEXP);
    return((float)NOT_OK);
  default : 
    sprintf(errormsg, ErrINTUFFCAL, 
	    op->flag);
    errorMsg(gp, 1, errormsg);
    return((float)NOT_OK);
    break;
  }
}

char FAR *t_calc_dup(struct cw_status *gp, struct oper FAR *op)
{
  char *lval=NULL, *rval=NULL;
  char errormsg[256];
  char *retval;

  if (!op) return(NULL);

#ifdef DEBUG
  if(op->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal op->type in t_calc_dup.\n");
  }
#endif


  switch(op->flag) {
  case TEXTOPER :
    /* indicates an operator, unary or binary */
    /* op-flag also indicates the desired type of the current intermediate
       value (either as specified by a cast, or as inherited from
       the sub-expressions). This routine is not called if the
       op-flag is 'i' (int) */

    if(op->operation.txt == t_ass){
      rval = t_calc_dup(gp, op->right.op);
      return((*op->operation.txt)(gp, op,lval,rval));
    }

    /* calculate  left and possibly right oper */
    switch(valtype(op->left.op)) {
    case  T_TEXT : 
      lval = t_calc_dup(gp, op->left.op);
      break;
    default :
      sprintf(errormsg, ErrINTILVTTDCA);
      errorMsg(gp, 1, errormsg);
      return NULL;
    }

    if (op->right.op) { /* NULL if unary operator */
      rval = t_calc_dup(gp, op->right.op);
    }

    retval = (*op->operation.txt)(gp, op,lval,rval);
    if(lval)
      CalFree(lval);
    if(rval)
      CalFree(rval);
    return retval;
  case TEXTVAL : return(strdup(op->left.text));
  case TEXTVAR : 
#ifdef DEBUG
    switch(op->type){
    case DO_VDECL :
      return(strdup(op->left.vd->vi->value.text));
    case DO_VINST :
#endif
      return(strdup(op->left.vi->value.text));
#ifdef DEBUG
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return NULL;
    }
#endif
  case TEXTCONST : return(strdup(op->left.cconst->thisc.text));
  case TEXTARRAY : return(strdup(el_txt_array(gp, op->left.array)));
  case UTEXTFUNC : 
    /* We must use the safestrdup function here, so we don't evaluate
     * the argument twice with the strdup define.
     */
    return TextFunc(gp, op->left.do_act->function.func_act, op->left.do_act->actual.par);
  case BTEXTFUNC : return((*op->left.do_act->function.tdo_act)(gp, op->left.do_act->actual.par));
  case INTVAL :
  case INTARRAY :
  case INTVAR :
  case INTCONST :
  case BINTFUNC :
  case UINTFUNC :
  case INTOPER :
    errorMsg(gp, 1, ErrIINTEXEXP);
    return(NULL);
  case FLOATVAL :
  case FLOATARRAY :
  case FLOATVAR :
  case FLOATCONST :
  case BFLOATFUNC :
  case UFLOATFUNC :
  case FLOATOPER :
    errorMsg(gp, 1, ErrFINTEXEXP);
    return NULL;
  default   : 
    sprintf(errormsg, ErrINTUFTCAD, op->flag);
    errorMsg(gp, 1, errormsg);
    return NULL;
  }
}

/************************************************************
**                                          eval_condition **
**  Evaluate test:                                         **
**          left_term test_operation right_term            **
**	returns 1 (TRUE) or 0 (FALSE)			   **
************************************************************/
int eval_condition(struct cw_status *gp,  struct oper FAR *op)
{
  char errormsg[256];
  if(op == NULL)
    return TRUE;
  switch(valtype(op)) {
  case T_INT : 
    if (calculate(gp, op)) 
      return(TRUE); 
    else 
      return(FALSE);
    break;
  case T_FLOAT : 
    if (f_calculate(gp, op)) 
      return(TRUE); 
    else return(FALSE);
    break;
  default : 
    sprintf(errormsg, ErrINTEVACON, op->flag);
    errorMsg(gp, 1, errormsg);
    return -1;
  }
}

#ifdef DEBUG
void var_log(struct cw_status *gp, struct vdecl FAR *var, FILE *logfile)
{
  switch(var->type){
  case INTVAL : 
    fprintf(logfile,"%s %ld\n",var->name,var->vi->value.value);
    break;
  case FLOATVAL : 
    fprintf(logfile,"%s %f\n",var->name,var->vi->value.fvalue);
    break;
  case TEXTVAL:
    fprintf(logfile, "%s %s\n", var->name, var->vi->value.text);
    break;
  }
}
#endif

/*
 * This function will put the attributes connected to this vinst 
 * into the global list of changed attributes.
 */
void update_changed_attr(struct cw_status *gp, struct vinst FAR *vi)
{
  struct outlist FAR *ol;

  for(ol = vi->out.out; ol != NULL; ol = (struct outlist FAR *)next(ol)){
    if(!is_list(ol->attr)){
      if(gp->curlunit->nattr == NULL){
	init_list(ol->attr);
	gp->curlunit->nattr = ol->attr;
      } else 
	place_next(gp->curlunit->lnattr, ol->attr);
      gp->curlunit->lnattr = ol->attr;
      if(ol->attr->attrtype == S_IMAGE){
	/* Register attribute, so URL can be preloaded... */
      }
    }
  }
}


/**********************************************************
**                                           simulate    **
**  Simulate actions in the block pointed to by "sim".   **
**  This sequence of actions teminates with a }.         **
**********************************************************/
int simulate(struct cw_status *gp, struct sim_actions FAR *sim, FILE *logfile)
{
  struct sim_actions FAR *s;
  char FAR *freestring;
  char errormsg[256];
    
  while(gp->freeze_sim)
    CalServeEvents (gp);

  if (RETURN)
    return(1);

  for (s = sim; (s != NULL);){
    if (s->cond_act != NULL) {
      (*s->cond_act->actions)(gp, s->cond_act,logfile);
    } else if (s->do_act != NULL) {
      switch(s->do_act->type) {
      case BINTFUNC: 
	(*s->do_act->function.do_act)(gp, s->do_act->actual.par);
	break;
      case BFLOATFUNC:
	(*s->do_act->function.fdo_act)(gp, s->do_act->actual.par);
	break;
      case BTEXTFUNC:
        CalFree((*s->do_act->function.tdo_act)(gp, s->do_act->actual.par));
        break;
      case UINTFUNC:
        IntFunc(gp, s->do_act->function.func_act, s->do_act->actual.par);
        break;
      case UFLOATFUNC:
        FloatFunc(gp, s->do_act->function.func_act, s->do_act->actual.par);
        break;
      case UTEXTFUNC:
        CalFree(TextFunc(gp, s->do_act->function.func_act, s->do_act->actual.par));
        break;
      case T_OBJECT:
	GenObject(gp, s->do_act->function.obj_act, s->do_act->actual.attr);
	break;
      default : 
        sprintf(errormsg, 
		ErrINTUNFUTY, 
		s->do_act->type);
	errorMsg(gp, 1, errormsg);
	break;
      }
    } else if (s->eq != NULL) {
      switch(valtype(s->eq)){
      case T_INT:
	calculate(gp, s->eq);
	break;
      case T_FLOAT:
	f_calculate(gp, s->eq);
	break;
      case T_TEXT:
	freestring = t_calc_dup(gp, s->eq);
	CalFree(freestring);
	break;
      };
    } else if(s->ret_act != NULL){ 
      gp->curlunit->ret_blockinst = s->parent;
      switch(s->parent->parent->rettype){
      case T_INT:
	s->parent->parent->retval.value = 
	   calculate(gp, s->ret_act);
	break;
      case T_FLOAT:
	s->parent->parent->retval.fvalue = 
           f_calculate(gp, s->ret_act);
	break;
      case T_TEXT:
	s->parent->parent->retval.text = 
	   t_calc_dup(gp, s->ret_act);
	break;
      default:
	sprintf(errormsg, ErrINTUNREIS, 
		s->parent->parent->rettype);
	errorMsg(gp, 1, errormsg);
	return(NOT_OK);
      }
      gp->command |= RETURN_ACTION;
      return(1);
    } else if(s->win != NULL){
      create_winobj(gp, (struct blockinst *)gp->curlunit->cur_funcinst,
		    s->win);
    } else if(s->graph != NULL){
      create_graphobj(gp, (struct blockinst *)gp->curlunit->cur_funcinst,
		      s->graph);
    } else if(s->inp != NULL){
      create_inpobj(gp, (struct blockinst *)gp->curlunit->cur_funcinst,
		    s->inp);
    }
      
    if (RETURN)
      return(1);

    CalServeEvents (gp);

    s = (struct sim_actions FAR *) next(s);
  }
  return(1);
}

void sim(struct cw_status *gp, struct learnunit FAR *learn)
{
  struct function FAR *mainfunc;

  mainfunc = get_function((struct commonblock *)learn, "main");
  if(mainfunc == NULL){
    errorMsg(gp, 1, ErrNOMAINAWE);
    return;
  }
  IntFunc(gp, mainfunc, NULL);
}

/*
 * Used for user defined integer functions.
 */
long IntFunc(struct cw_status *gp, struct function FAR *fp,
	     struct param FAR *act_par)
{
  long val;
  FUNC(gp, fp, act_par);
  val = fp->retval.value;
  fp->retval.value = 0;
  return val;
}

/*
 * Used for user defined floating point functions.
 */
double FloatFunc(struct cw_status *gp, struct function FAR *fp,
		 struct param FAR *act_par)
		
{
  double fval;
  FUNC(gp, fp, act_par);
  fval = fp->retval.fvalue;
  fp->retval.fvalue = 0;
  return fval;  
}

/*
 * Used for user defined text functions.
 */
char FAR *TextFunc(struct cw_status *gp, struct function FAR *fp,
		   struct param FAR *act_par)
{
  char FAR *str;
  FUNC(gp, fp, act_par);
  if(fp->retval.text){
    str = fp->retval.text;
    fp->retval.text = NULL;
  } else {
    str = (char *) CalMalloc(sizeof(char));
    *str = '\0';
  }
  return str;
}
 
void FUNC(struct cw_status *gp, struct function FAR *fp,
	  struct param FAR *act_par)
{
  struct funcinst FAR *fi, FAR *oldfi;
  struct blockinst *oldretinst;
  FILE *file=NULL;
  
  if((fi = NEWFUNCINST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }

  fi->caller = gp->curlunit->cur_blockinst;
  fi->parent = fp;
  oldretinst = gp->curlunit->ret_blockinst;

  /*
   * 1. Instantiate function hierarchy.
   *
   * 2. Execute function body.
   *
   * 3. Return from function.
   */

  /* Initializes function fp, and records information about the old 
   * function in 'fi'
   */
  i_function(gp, fp, act_par, fi);


  gp->curlunit->cur_blockinst = (struct blockinst FAR *)fi;
  oldfi = gp->curlunit->cur_funcinst;
  gp->curlunit->cur_funcinst = fi;
  simulate(gp, fi->sim, file);
  if(gp->curlunit->ret_blockinst == (struct blockinst FAR *)fi){
    gp->command &= ~RETURN_ACTION;
    gp->curlunit->ret_blockinst = oldretinst;
  }
  gp->command |= CONTINUE;
  
#ifdef GUNNAR
  if((file=fopen("logfile", "a")) == NULL){
    fprintf(stderr, ErrINTOPEFIL);
  }
  dbug_function(fp, file);
  fclose(file);
#endif

  /* Put the contents of 'fi' back into the function.  
   * Free all objects in the function, except those who are bound to 
   * reference parameters.
   */

  r_function(gp, fp, fi);

  gp->curlunit->cur_blockinst = fi->caller;
  gp->curlunit->cur_funcinst = oldfi;
  CalFree(fi);
}

void GenObject(struct cw_status *gp, struct objectproto FAR *object, struct attrparam FAR *act_par)
{
  struct objectinst FAR *oi;
  struct blockinst *oldretinst;
  FILE *file=NULL;

  if((oi = NEWOBJECTINST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }

  oi->caller = gp->curlunit->cur_blockinst;
  oi->parent = object;
  oldretinst = gp->curlunit->ret_blockinst;


  /* Initializes function fp, and records information about the old 
   * function in 'fi'
   */
  i_object(gp, object, act_par, oi);


  gp->curlunit->cur_blockinst = (struct blockinst FAR *)oi;
  simulate(gp, oi->sim, file);

  if(gp->curlunit->ret_blockinst == (struct blockinst FAR *)oi){
    gp->command &= ~RETURN_ACTION;
    gp->curlunit->ret_blockinst = oldretinst;
  }
  gp->command |= CONTINUE;
  

  /* Put the contents of 'fi' back into the function.  
   * Free all objects in the function, except those who are bound to 
   * reference parameters.
   */

  r_object(gp, object, oi);

  gp->curlunit->cur_blockinst = oi->caller;
  CalFree(oi);
}

int IF(struct cw_status *gp, struct sim_actions FAR *act, FILE *logfile)
{
  if (eval_condition(gp, act->eq_op)) {
    if (RETURN)
      return(1);
    simulate(gp, act->cond_act,logfile);
  } else {
    if (RETURN)
      return(1);
    simulate(gp, (struct sim_actions FAR *) next(act),logfile);
  }	
  return(1);
}

int WHILE(struct cw_status *gp, struct sim_actions FAR *act, FILE *logfile)
{
  while (!RETURN &&
	 eval_condition(gp, act->eq_op))
    simulate(gp, act->cond_act,logfile);
  
  return(1);
}

int FOR(struct cw_status *gp, struct sim_actions FAR *act, FILE *logfile)
{
  struct sim_actions FAR *for_act;

  for_act = act;

  for ( ;eval_condition(gp, for_act->eq_op);){
    simulate(gp, for_act->cond_act,logfile);
    if(RETURN)
      return(1);
  }
  return(1);
}

int SWITCH(struct cw_status *gp, struct sim_actions FAR *act, FILE *logfile)
{
  struct sim_actions FAR *switch_act;
  long lval;
  double rval;
  char type;

#ifdef OLDP
  switch(type = valtype(act->eq_op)) {
    case S_INTEGER : lval = calculate(gp, act->eq->eq_op);
		     break;
    case S_FLOAT   : rval = f_calculate(gp, act->eq->eq_op);
		     break;
  }
#endif

  if (RETURN)
    return(1);

  for ( switch_act = (struct sim_actions FAR *) next(act);
	((switch_act != NULL) && (type == T_INT ?

	   (valtype(switch_act->eq_op) == T_INT ?
	      calculate(gp, switch_act->eq_op) != lval :
	      f_calculate(gp, switch_act->eq_op) != (double) lval) :

	   (valtype(switch_act->eq_op) == T_INT ?
	      calculate(gp, switch_act->eq_op) != (long) rval :
	      f_calculate(gp, switch_act->eq_op) != rval) ));

	switch_act = (struct sim_actions FAR *) next(switch_act) )
    if (RETURN)
      return(1);

  if (switch_act != NULL)
    simulate(gp, switch_act->cond_act,logfile);
  return(1);
}

void set_time(struct cw_status *gp)
{
  gp->curlunit->starttime = millitime(gp);
}




/**********************************************************
**                                            start_sim  **
**  Initialize simulation, set START_TIME to starting    **
**  time for simulation. Timing in simulation is related **
**  to START_TIME.                                       **
**                                                       **
**  This function should be called when:                 **
**                                                       **
**          1. simulation is to be started and has not   **
**             yet been started and halted.              **
**          2. simulation is to be restarted after halt  **
**             and continue without resetting variables  **
**             to initial values.                        **
**********************************************************/

int start_sim(struct cw_status *gp, struct learnunit FAR *learn) 
{
  struct vdecl FAR *v;

  if(!RETURN){
    for(v = learn->firstv; v != NULL; v = (struct vdecl FAR *)next(v)){
      v->vi = c_vinst(gp, v);
    }
    set_time(gp);
    sim(gp, learn);
  }
  return(gp->command);
}

