/* Copyright (c) 1995 by Computers and Learning A/S (candleweb@candleweb.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */


/*
 * Functions related to instantation and termination of functions.
 * Some _very_ simple code improvements are done one expressions.
 */

#include <stdio.h>
#include <string.h> 
#include <stdlib.h> 
#include <malloc.h>
#include <assert.h>

#include "const.h"
#include "candle.h"
#include "simulate.h"
#include "error.h"
#include "fast_lis.h"
#include "function.h"
#include "graphic.h"
#include "nodes.h"

#include "protos/memory.h"
#include "protos/optimize.h"
#include "protos/simulate.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/freesim.h"
#include "protos/instsim.h"

/*
 * Simplify and remove variable declarations from a expression.
 * 
 * op is the expression to be simplified.
 *
 * The function will return a pointer to a new oper structure.
 * If simple is true on return, then the value of the returned oper is 
 * constant over time and may be simplified with one of the calculate 
 * functions.
 * If sideeffect is true on return the expression may have sideeffects. 
 */
struct oper FAR *SimplifyOper(struct cw_status *gp, struct oper FAR *op,
			      struct attribute FAR *attr, 
			      char FAR *simple, char FAR *sideeffect)
{                                          
  char errormsg[256];
  struct oper FAR *new, FAR *tmp_op;
  char easy, e1, e2;
  int skiprest = 0;
  struct ptoperlist FAR *opto;
  int i, n;
  struct oper FAR * FAR *oparr;

  e1 = e2 = easy = TRUE;
  
  assert(op);
  
  if(op->type == DO_VINST){
    errorMsg(gp, 2, ErrINTVINSTO);
    c_exit(gp, NOT_OK);
  }

  if((new = NEWOPER) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  new->line = op->line;
  /* Simplify left part */
  switch(op->flag){
  case INTVAL:
    new->flag = INTVAL;
    new->left.value = op->left.value;
    break;
  case FLOATVAL:
    new->flag = FLOATVAL;
    new->left.fvalue = op->left.fvalue;
    break;
  case TEXTVAL:
    new->flag = TEXTVAL;
    new->left.text = strdup(op->left.text);
    break;
  case INTARRAY:
  case FLOATARRAY:
  case TEXTARRAY:
    new->flag = op->flag;
    new->type = DO_VINST;
    if((new->left.array = NEWINTARRAY) == NULL){
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    new->left.array->type = DO_VINST;
    new->left.array->which.vi = op->left.array->which.vd->vi;
    addattr(gp, new->left.array->which.vi, attr);
    new->left.array->indeks = SimplifyParam(gp, op->left.array->indeks, attr, 
					    sideeffect);
    easy = FALSE;
    break;
  case INTVAR:
  case FLOATVAR:
  case TEXTVAR:
    if(op->left.vd->array != NULL){
      if(op->flag == INTVAR)
	new->flag = INTARRAY;
      else if(op->flag == FLOATVAR)
	new->flag = FLOATARRAY;
      else if(op->flag == TEXTVAR)
	new->flag = TEXTARRAY;
      else {
	sprintf(errormsg, 
		ErrINTUNSIOP, 
		op->flag);
	errorMsg(gp, 1, errormsg);
	return NULL;
      }
      if((new->left.array = NEWINTARRAY) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      new->left.array->type = DO_VINST;
      new->left.array->which.vi = op->left.vd->array->which.vi;
      if((new->left.array->indeks = NEWPARAM) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      init_list(new->left.array->indeks);
      if((new->left.array->indeks->exp = NEWOPER) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      new->left.array->indeks->exp->line = op->line;
      new->left.array->indeks->exp->flag = INTVAL;
      compute_valtype(gp, new->left.array->indeks->exp);

      tmp_op = i_oper(gp, op->left.vd->array->indeks->exp);
      new->left.array->indeks->exp->left.value =  calculate(gp, tmp_op);
      f_oper(gp, tmp_op);

      new->left.array->indeks->line = op->left.vd->array->indeks->line;
      addattr(gp, new->left.array->which.vi, attr);
     }else {
       new->flag = op->flag;       
       new->type = DO_VINST;
       new->left.vi = op->left.vd->vi;
       addattr(gp, new->left.vi, attr);
     }
    easy = FALSE;
    break;
  case INTCONST:	/* Constants will be simplified to concrete values, 
			 * as they never are subject to changes.
			 */
    new->flag = INTVAL;
    new->left.value = op->left.cconst->thisc.value;
    break;
  case FLOATCONST:
    new->flag = FLOATVAL;
    new->left.fvalue = op->left.cconst->thisc.fvalue;
    break;
  case TEXTCONST:
    new->flag = TEXTVAL;
    new->left.text = strdup(op->left.cconst->thisc.text);
    break;
  case UINTFUNC:
  case UFLOATFUNC:
  case UTEXTFUNC:
  case BINTFUNC:
  case BFLOATFUNC:
  case BTEXTFUNC:
    *sideeffect = TRUE;
    new->flag = op->flag;
    new->left.do_act = SimplifyAction(gp, op->left.do_act, attr, sideeffect);
    easy = FALSE;
    break;
  case ATTROPER:
  case ATTROPER | T_ARRAY:
    switch(op->left.ap->type){
    case T_EXPR:
      if(op->flag & T_ARRAY){
	if(op->left.ap->actval.oper && 
	   valtype(op->left.ap->actval.oper) & T_ARRAY){
	  op->right.array->which.vd = 
	    op->left.ap->actval.oper->left.array->which.vd;
	  op->right.array->type = 
	    op->left.ap->actval.oper->left.array->type;
	  new->type = DO_VINST;
	  if((new->left.array = NEWINTARRAY) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  new->left.array->type = DO_VINST;
	  new->left.array->which.vi = op->right.array->which.vd->vi;
	  addattr(gp, new->left.array->which.vi, attr);
	  new->left.array->indeks = SimplifyParam(gp, op->right.array->indeks, 
						  attr, sideeffect);
	  new->flag = op->left.ap->actval.oper->flag;
	  easy = FALSE;
	}
      }	else if(op->left.ap->actval.oper){
	CalFree(new);
	new = SimplifyOper(gp, op->left.ap->actval.oper, 
			   attr, &e1, sideeffect);
	easy = easy && e1;
      } else {
	new->flag = INTVAL;
      }
      break;
    case T_POINTS:
      new->flag = ATTROPER;
      new->type = DO_VINST;
      if((new->left.pt = NEWPOINTARRAY) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      n = new->left.pt->n = op->left.ap->npoints;
      oparr = new->left.pt->op = CalCalloc(1, 2*n*sizeof(struct oper));
      opto = op->left.ap->actval.pl;
      for(i = 0;
	  opto != NULL;
	  opto = (struct ptoperlist FAR *) next(opto), i++){
	oparr[2*i] = SimplifyOper(gp, opto->x, attr, &e1, sideeffect);
	oparr[2*i+1] = SimplifyOper(gp, opto->y, attr, &e2, sideeffect);
	easy = easy && e1 && e2;
      }
      break;
    }
    break;
  case INTOPER: 
    if(op->operation.lng == l_exprval) {
      e1 = e2 = TRUE;
    } else {
      new->left.op = SimplifyOper(gp, op->left.op, attr, &e1, sideeffect);
      if(e1){
	if(op->operation.lng == l_or){
	  skiprest = e2 = TRUE;
	} else if(op->operation.lng == l_and){
	  skiprest = e2 = FALSE;
	}
      }
      if(!skiprest && op->right.op != NULL){
	new->right.op =  SimplifyOper(gp, op->right.op, attr, &e2, sideeffect);
      }
    }

    skiprest = FALSE;
    if(e1 && e2) {
      f_oper(gp, new->left.op);
      if(new->right.op != NULL){
	f_oper(gp, new->right.op);
	new->right.op = NULL;
      }
      tmp_op = i_oper(gp, op);
      new->left.value = calculate(gp, tmp_op);
      f_oper(gp, tmp_op);

      new->flag = INTVAL;
    } else {
      new->flag = INTOPER;
      new->operation.lng = op->operation.lng;
      easy = FALSE;
    }
    break;
  case FLOATOPER:
    if(op->operation.flt == r_exprval) {
      e1 = e2 = TRUE;
    } else {
      new->left.op = SimplifyOper(gp, op->left.op, attr, &e1, sideeffect);
      if(e1){
	if(op->operation.flt == r_or){
	  skiprest = e2 = TRUE;
	} else if(op->operation.flt == r_and){
	  skiprest = e2 = FALSE;
	}
      }
      if(!skiprest && op->right.op != NULL){
	new->right.op =  SimplifyOper(gp, op->right.op, attr, &e2, sideeffect);
      }
    }
    skiprest = FALSE;
    if(e1 && e2) {
      f_oper(gp, new->left.op);
      if(new->right.op != NULL){
	f_oper(gp, new->right.op);
	new->right.op = NULL;
      }
      tmp_op = i_oper(gp, op);
      new->left.fvalue = f_calculate(gp, tmp_op);
      f_oper(gp, tmp_op);

      new->flag = FLOATVAL;
    } else {
      new->flag = FLOATOPER;
      new->operation.flt = op->operation.flt;
      easy = FALSE;
    }
    break;
  case TEXTOPER:
    if(op->operation.txt == t_exprval) {
      e1 = e2 = TRUE;
    } else {
      new->left.op = SimplifyOper(gp, op->left.op, attr, &e1, sideeffect);
      if(op->right.op != NULL){
	new->right.op =  SimplifyOper(gp, op->right.op, attr, &e2, sideeffect);
      }
    }
    if(e1 && e2) {
      f_oper(gp, new->left.op);
      if(new->right.op != NULL){
	f_oper(gp, new->right.op);
	new->right.op = NULL;
      }

      tmp_op = i_oper(gp, op);
      new->left.text = t_calc_dup(gp, tmp_op);
      f_oper(gp, tmp_op);

      new->flag = TEXTVAL;
    } else {
      new->flag = TEXTOPER;
      new->operation.txt = op->operation.txt;
      easy = FALSE;
    }
    break;
  case T_OPER:
    if(op->operation.lng == l_exprval || op->operation.flt == r_exprval ||
       op->operation.txt == t_exprval) {
      tmp_op = i_oper(gp, op);
      switch(valtype(tmp_op)){
      case INTVAL:
	new->left.value = calculate(gp, tmp_op);
	new->flag = T_INT;
	break;
      case FLOATVAL:
	new->left.fvalue = f_calculate(gp, tmp_op);
	new->flag = T_FLOAT;
	break;
      case TEXTVAL:
	new->left.text = t_calc_dup(gp, tmp_op);
	new->flag = T_TEXT;
	break;
      default:
	sprintf(errormsg, "Internal error: Illegal valtype '%d' in optimize", 
		valtype(tmp_op));
	errorMsg(gp, 2, errormsg);
	c_exit(gp, NOT_OK);
      }
      f_oper(gp, tmp_op);

    } else {
      sprintf(errormsg, "Internal error: Illegal T_OPER in optimize");
      errorMsg(gp, 2, errormsg);
      c_exit(gp, NOT_OK);
    }
    break;
  default:
    sprintf(errormsg, ErrINTILVAFO, op->flag);
    errorMsg(gp, 2, errormsg);
    c_exit(gp, NOT_OK);
  }
  
  compute_valtype(gp, new);
  *simple = easy;
  return new;
}

struct action FAR *SimplifyAction(struct cw_status *gp,
				  struct action FAR *act,
				  struct attribute FAR *attr, 
				  char FAR *sideeffect)
{
  struct action FAR *new;
  char errmsg[256];
   
  if((new = NEWACTION) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  new->type = act->type;
  switch(act->type){
  case BINTFUNC:
    new->function.do_act = act->function.do_act;
    break;
  case BFLOATFUNC:
    new->function.fdo_act = act->function.fdo_act;
    break;
  case BTEXTFUNC:
    new->function.tdo_act = act->function.tdo_act;
    break;
  case UINTFUNC:
  case UFLOATFUNC:
  case UTEXTFUNC:
    new->function.func_act = act->function.func_act;
    break;
  default:
    sprintf(errmsg, ErrINTUNSIAC, act->type);
    errorMsg(gp, 2, errmsg);
    c_exit(gp, NOT_OK);
  }
  
  if(act->type == T_OBJECT)
    new->actual.attr = SimplifyAttrparam(gp, act->actual.attr, attr,
					 sideeffect);
  else
    new->actual.par = SimplifyParam(gp, act->actual.par, attr, sideeffect);

  return new;
}

struct attrparam FAR 
*SimplifyAttrparam(struct cw_status *gp, struct attrparam FAR *par,
		   struct attribute *attr, char FAR *sideeffect)
{
  struct attrparam FAR *new;
  char simple=TRUE;

  if(par == NULL)
    return NULL;

  if((new = NEWATTRPARAM) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  new->type = par->type;
  new->name = strdup(par->name);
  switch(new->type){
  case T_EXPR:
    new->actval.oper = SimplifyOper(gp, par->actval.oper, attr, &simple, 
				    sideeffect);
    break;
  case T_POINTS:
    break;
  case T_TRANSLATION:
    break;
  };
  
  if((par = next(par)) != NULL)
    place_prev(SimplifyAttrparam(gp, par, attr, sideeffect), new);
  else
    init_list(new);
  return new;
} 

struct param FAR *SimplifyParam(struct cw_status *gp, struct param FAR *par,
				struct attribute FAR *attr,
				char FAR *sideeffect)
{
  struct param FAR *new;
  char simple=TRUE;

  if(par == NULL)
    return NULL;

  if((new = NEWPARAM) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  new->line = par->line;
  new->exp = SimplifyOper(gp, par->exp, attr, &simple, sideeffect);
  
  if((par = next(par)) != NULL)
    place_prev(SimplifyParam(gp, par, attr, sideeffect), new);
  else
    init_list(new);
  return new;
}

/*
 * Add attr to vi's list of attributes.
 */
void addattr(struct cw_status *gp, struct vinst FAR *vi,
	     struct attribute FAR *attr)
{
  struct outlist FAR *ol;
  struct volist FAR *vol;

  if((ol = NEWOUTLIST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  if((vol = NEWVOLIST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }

  ol->attr = attr;
  vol->ol = ol;
  vol->vi = vi;

  if(vi->out.out == NULL)
    init_list(ol);
  else
    place_prev(vi->out.out, ol);

  vi->out.out = ol;

  if(attr->vinstlist == NULL)
    init_list(vol);
  else {
    place_prev(attr->vinstlist, vol);
  }
  attr->vinstlist = vol;
}

/*
 * Return FALSE if there is a pointer to vi in list, 
 * else return TRUE.
 */
int is_remote(struct cw_status *gp, struct vinst FAR *list,
	      struct vinst FAR *vi)
{
  for(;list != NULL; list = (struct vinst FAR *) next(list))
    if(list == vi)
      return FALSE;
  
  return TRUE;
}

