/* Copyright (c) 1995 by Computers and Learning A/S (candleweb@candleweb.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */

/* 
 * Provide functions to instantiate and terminate functions.
 */

#include <stdio.h>
#include <string.h> 
#include <stdlib.h> 
#include <malloc.h>
#include <assert.h>

#include "const.h"
#include "candle.h"
#include "error.h"
#include "simulate.h"
#include "fast_lis.h"
#include "candle.h"
#include "graphic.h"
#include "lex.h"
#include "nodes.h"
#include "input.h"
#include "pbmplus.h"        /* For min/max-macros */
#include "function.h"
#include "parser.h"
#include "sysproto.h"

#include "protos/memory.h"
#include "protos/instsim.h"
#include "protos/fast_lis.h"
#include "protos/simulate.h"
#include "protos/creatsim.h"
#include "protos/optimize.h"
#include "protos/instance.h"
#include "protos/canutil.h"
#include "protos/freesim.h"
#include "protos/update.h"
#include "protos/parser.h"


struct action FAR *i_action(struct cw_status *gp, struct action FAR *old_act)
{
  struct action FAR *act;
  char errormsg[256];

  if(old_act == NULL)
    return NULL;
  if((act = NEWACTION) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  if(old_act->type == T_OBJECT)
    act->actual.attr = i_attrparam(gp, old_act->actual.attr);
  else 
    act->actual.par = i_param(gp, old_act->actual.par);
  
  act->type = old_act->type;
  act->line = old_act->line;
  switch(old_act->type){
  case BINTFUNC:
    act->function.do_act = old_act->function.do_act;
    break;
  case BFLOATFUNC:
    act->function.fdo_act = old_act->function.fdo_act;
    break;
  case BTEXTFUNC:
    act->function.tdo_act = old_act->function.tdo_act;
    break;
  case UINTFUNC:
  case UFLOATFUNC:
  case UTEXTFUNC:
    act->function.func_act = old_act->function.func_act;
    break;
  case T_OBJECT:
    act->function.func_act = old_act->function.func_act;
    break;
  default:
    sprintf(errormsg, "Internal error: Unknown old_act->type in i_action");
    errorMsg(gp, 1, errormsg);
    break;
  }
  return act;
}

struct param FAR *i_param(struct cw_status *gp, struct param FAR *old_par)
{
  struct param FAR *par;
  
  if(old_par == NULL)
    return NULL;
  if((par = NEWPARAM) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  par->line = old_par->line;
  par->exp = i_oper(gp, old_par->exp);
  if((old_par = (struct param FAR *) next(old_par)) != NULL)
    place_prev(i_param(gp, old_par), par);
  else
    init_list(par);
  return par;
}

struct attrparam FAR *i_attrparam(struct cw_status *gp,
				  struct attrparam FAR *old_par)
{
  struct attrparam FAR *par;

  if(old_par == NULL)
    return NULL;
  if((par = NEWATTRPARAM) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  par->type = old_par->type;
  par->name = strdup(old_par->name);
  switch(old_par->type){
  case T_EXPR:
    if(old_par->indirect)
      par->actval.oper = old_par->defval.ap->actval.oper;
    else 
      par->actval.oper = old_par->defval.oper;
    break;
  case T_TRANSLATION:
    if(old_par->indirect)
      par->actval.tl = old_par->defval.ap->actval.tl;
    else
      par->actval.tl = old_par->defval.tl;
    break;
  case T_POINTS:
    if(old_par->indirect){
      par->actval.pl = old_par->defval.ap->actval.pl;
      par->npoints = old_par->defval.ap->npoints;
    } else {
      par->actval.pl = old_par->defval.pl;
      par->npoints = old_par->npoints;
    }
    break;
  };

  if((old_par = (struct attrparam FAR *) next(old_par)) != NULL)
    place_prev(i_attrparam(gp, old_par), par);
  else
    init_list(par);
  return par;
}

struct intarray FAR *i_intarray(struct cw_status *gp, struct intarray *old_ia)
{
  struct intarray FAR *ia;

  if(old_ia == NULL)
    return NULL;
  if((ia = NEWINTARRAY) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  ia->type = DO_VINST;
  if(old_ia->type == DO_VDECL)
    ia->which.vi = old_ia->which.vd->vi;
  else
    ia->which.vi = old_ia->which.vi;
  ia->indeks = i_param(gp, old_ia->indeks);
  return ia;
}

struct oper FAR *i_oper(struct cw_status *gp, struct oper FAR *old_op)
{
  struct oper FAR *op, FAR *tmp_op;
  char errormsg[256];
  struct ptoperlist FAR *opto;
  int i, n;
  struct oper FAR * FAR *oparr;
  struct param *par, *lpar, *opar;

  assert(old_op);

  if((op = NEWOPER) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  op->flag = old_op->flag;
  op->line = old_op->line;
  switch(old_op->flag){
  case INTVAL:
    op->left.value = old_op->left.value;
    break;
  case FLOATVAL:
    op->left.fvalue = old_op->left.fvalue;
    break;
  case TEXTVAL:
    op->left.text = safestrdup(old_op->left.text);
    break;
  case INTARRAY:
  case FLOATARRAY:
  case TEXTARRAY:
    op->left.array = i_intarray(gp, old_op->left.array);
    break;
  case INTVAR:
  case FLOATVAR:
  case TEXTVAR:
    if(old_op->type == DO_VDECL){
      if(old_op->left.vd->array != NULL){
	if(op->flag == INTVAR)
	  op->flag = INTARRAY;
	else if(op->flag == FLOATVAR)
	  op->flag = FLOATARRAY;
	else if(op->flag == TEXTVAR)
	  op->flag = TEXTARRAY;
	else {
	  sprintf(errormsg, ErrINTUNOPFL, op->flag);
	  errorMsg(gp, 1, errormsg);
	  return NULL;
	}
	if((op->left.array = NEWINTARRAY) == NULL){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	op->left.array->type = DO_VINST;
	op->left.array->which.vi = old_op->left.vd->array->which.vi;
	
	for(opar = old_op->left.vd->array->indeks; opar != NULL; 
	    opar = next(opar))
	  {
	    if((par = NEWPARAM) == NULL){
	      errorMsg(gp, 2, ErrNOMOREMEM);
	      c_exit(gp, NOT_OK);
	    }
	    if(op->left.array->indeks == NULL){
	      init_list(par);
	      op->left.array->indeks = par;
	    } else {
	      place_next(par, par);
	    }
	    lpar = par;
	    if((par->exp = NEWOPER) == NULL){
	      errorMsg(gp, 2, ErrNOMOREMEM);
	      c_exit(gp, NOT_OK);
	    }
	    par->exp->line = old_op->line;
	    par->line = opar->line;
	    par->exp->flag = INTVAL;
	    compute_valtype(gp, par->exp);
	    par->exp->left.value =  
	      calculate(gp, opar->exp);
	  }
      } else 
	op->left.vi = old_op->left.vd->vi;
    } else if(old_op->type == DO_VINST)
	     op->left.vi =  old_op->left.vi;
    break;
  case INTCONST:
    op->flag = INTVAL;
    op->left.value = old_op->left.cconst->thisc.value;
    break;
  case FLOATCONST:
    op->flag = FLOATVAL;
    op->left.fvalue = old_op->left.cconst->thisc.fvalue;
    break;
  case TEXTCONST:
    op->flag = TEXTVAL;
    op->left.text = strdup(op->left.cconst->thisc.text);
    break;
  case BINTFUNC:
  case BTEXTFUNC:
  case BFLOATFUNC:
  case UINTFUNC:
  case UFLOATFUNC:
  case UTEXTFUNC:
    op->left.do_act = i_action(gp, old_op->left.do_act);
    break;
  case INTOPER:
  case T_OPER: /* Handle operators of unknown types */
    if(old_op->operation.lng == l_exprval){
      tmp_op = i_oper(gp, old_op->left.op);
      op->left.value = calculate(gp, tmp_op);
      f_oper(gp, tmp_op);
      op->flag = INTVAL;
    }else if((old_op->operation.lng == l_ass || 
	      old_op->operation.lng == l_assplus ||
	      old_op->operation.lng == l_assminus || 
	      old_op->operation.lng == l_assdiv || 
	      old_op->operation.lng == l_assmod || 
	      old_op->operation.lng == l_assmult || 
	      old_op->operation.lng == l_assbor || 
	      old_op->operation.lng == l_assband || 
	      old_op->operation.lng == l_asslshift || 
	      old_op->operation.lng == l_assrshift) && 
	     (old_op->left.op->flag == ATTROPER || 
	      valtype(old_op) == T_VOID)){
      op->left.op = i_oper(gp, old_op->left.op);
      switch(valtype(op->left.op)){
      case T_INT:
	op->operation.lng = old_op->operation.lng;
	op->flag = INTOPER;
	break;
      case T_FLOAT:
	if(old_op->operation.lng == l_ass)
	  op->operation.flt = r_ass;
	else if(old_op->operation.lng == l_assplus)
	   op->operation.flt = r_assplus;
	else if(old_op->operation.lng == l_assminus)
	   op->operation.flt = r_assminus;
	else if(old_op->operation.lng == l_assdiv)
	   op->operation.flt = r_assdiv;
	else if(old_op->operation.lng == l_assmod)
	   op->operation.flt = r_assmod;
	else if(old_op->operation.lng == l_assmult)
	   op->operation.flt = r_assmult;
	else {
	  sprintf(errormsg, "Line %d: Illegal assignment operator for floating point assignment.", old_op->line);
	  errorMsg(gp, 1, errormsg);
	}
	op->flag = FLOATOPER;
	break;
      case T_TEXT:
	op->operation.txt = t_ass;
	op->flag = TEXTOPER;
	break;
      }
      op->right.op = i_oper(gp, old_op->right.op);
    } else {
      op->left.op = i_oper(gp, old_op->left.op);
      if(old_op->right.op)
	op->right.op = i_oper(gp, old_op->right.op);
      op->operation.lng = old_op->operation.lng;
    }
    break;
  case FLOATOPER:
    if(old_op->operation.flt == r_exprval){
      tmp_op = i_oper(gp, old_op->left.op);
      op->left.fvalue = f_calculate(gp, tmp_op);
      f_oper(gp, tmp_op);
      op->flag = FLOATVAL;
    } else {
      op->left.op = i_oper(gp, old_op->left.op);
      if(old_op->right.op)
	op->right.op = i_oper(gp, old_op->right.op);
      op->operation.flt = old_op->operation.flt;
    }
    break;
  case TEXTOPER:
    if(old_op->operation.txt == t_exprval){
      tmp_op = i_oper(gp, old_op->left.op);
      op->left.text = t_calc_dup(gp, tmp_op);
      f_oper(gp, tmp_op);
      op->flag = TEXTVAL;
    } else {
      op->left.op = i_oper(gp, old_op->left.op);
      if(old_op->right.op)
	op->right.op = i_oper(gp, old_op->right.op);
      op->operation.txt = old_op->operation.txt;
    }
    break;
  case ATTROPER:
  case ATTROPER | T_ARRAY:
    switch(old_op->left.ap->type){
    case T_EXPR:
      if(old_op->flag & T_ARRAY){
	if(old_op->left.ap->actval.oper && 
	   valtype(old_op->left.ap->actval.oper) & T_ARRAY){
	  old_op->right.array->which.vd = 
	    old_op->left.ap->actval.oper->left.array->which.vd;
	  old_op->right.array->type = 
	    old_op->left.ap->actval.oper->left.array->type;
	  op->left.array = i_intarray(gp, old_op->right.array);
	  op->flag = old_op->left.ap->actval.oper->flag;
	}
      } else if(old_op->left.ap->actval.oper){
	CalFree(op);
	op = i_oper(gp, old_op->left.ap->actval.oper);
      } else {
	op->flag = INTVAL;
      }
      break;
    case T_POINTS:
      op->flag = ATTROPER;
      if((op->left.pt = NEWPOINTARRAY) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      n = op->left.pt->n = old_op->left.ap->npoints;
      oparr = op->left.pt->op = CalCalloc(1, 2*n*sizeof(struct oper));
      opto = old_op->left.ap->actval.pl;
      for(i = 0;
	  opto != NULL;
	  opto = (struct ptoperlist FAR *) next(opto), i++){
	oparr[2*i] = i_oper(gp, opto->x);
	oparr[2*i+1] = i_oper(gp, opto->y);
      }
      break;
    }
    break;
  default:
    sprintf(errormsg, 
	    ErrINTILVAFL, 
	    old_op->flag);
    errorMsg(gp, 1, errormsg);
    return NULL;
  }
  op->type = DO_VINST;
  compute_valtype(gp, op);
  return op;
}

/*
 * Copy and modify a sequence of sim_actions so they can be used in 
 * object attributes. 
 */
struct sim_actions FAR *i_simactions(struct cw_status *gp,
				     struct sim_actions FAR *old_act, 
				     struct blockinst FAR *bi)
{
  struct sim_actions FAR *act;

  if(old_act == NULL)
    return NULL;
  if((act = NEWSIMACTION) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  act->parent = bi;
  if(old_act->do_act != NULL)
    act->do_act = i_action(gp, old_act->do_act);
  else if(old_act->eq != NULL)
    act->eq = i_oper(gp, old_act->eq);
  else if(old_act->ret_act != NULL)
    act->ret_act = i_oper(gp, old_act->ret_act);
  else if(old_act->win != NULL)
    act->win = old_act->win;
  else if(old_act->graph != NULL)
    act->graph = old_act->graph;
  else if(old_act->inp != NULL)
    act->inp = old_act->inp;
  
  
  if(old_act->cond_act != NULL)
    act->cond_act = i_simactions(gp, old_act->cond_act, bi);
  act->actions = old_act->actions;
  if(old_act->eq_op)
    act->eq_op = i_oper(gp, old_act->eq_op);
  if((old_act = (struct sim_actions FAR *) next(old_act)) != NULL)
    place_prev(i_simactions(gp, old_act, bi), act);
  else
    init_list(act);
  return act;
}


int getXDefPoint(struct cw_status *gp, struct commonobj FAR *obj, int i)
{
  if(i == 1){
    switch(obj->type){
    case S_IMAGE:
      break;
    }
  } else if(i == 2) {
    
  }
  return 0;
}

int getYDefPoint(struct cw_status *gp, struct commonobj FAR *obj, int i)
{
  return 0;
}

/*
 * Return pointer to an instantiated attribute object. 
 * If *constant == TRUE on return, then the attribute is static.
 */
static struct attribute FAR *
i_attribute(struct cw_status *gp, struct attribute FAR *attr,
	    struct commonobj FAR *parent, 
	    char FAR *constant)
{
  struct attribute FAR *nattr, *lattr;
  struct ptoperlist FAR *opto;
  struct translation FAR *tl, FAR *ntl, FAR *ptl;
  char simple=TRUE;
  char sideeffect=FALSE;
  char errormsg[256];
  int i, npoints;

  if(attr == NULL)
    return NULL;

  if((nattr = NEWATTRIBUTE) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  nattr->valid = FALSE;
  nattr->parenttype = attr->parenttype;
  switch(nattr->parenttype){
  case WIN :
    nattr->parent.typwin = (struct winobj FAR *) parent;
    break;
  case GRAPH :
    nattr->parent.typgraph = (struct graphobj FAR *) parent;
    break;
  case INP :
    nattr->parent.typinp = (struct inpobj FAR *) parent;
    break;
  default :
    sprintf(errormsg, ErrINTUNKATT, 
	    nattr->parenttype);
    errorMsg(gp, 1,errormsg);
    return NULL;
  }

  assert(nattr->parent.typwin);

  nattr->attrtype = attr->attrtype;
  switch(nattr->attrtype){
  case S_POINTS:
    if(attr->indirect) {
      npoints = attr->exprval.ap->npoints;
      opto = attr->exprval.ap->actval.pl; 
    } else {
      opto = attr->exprval.pl;
      npoints = attr->parent.obj->ptcount;
    }
    if(npoints < minPoints(gp, parent))
      npoints = minPoints(gp, parent);
    parent->ptcount = npoints;

    nattr->exprval.op = CalCalloc(1, npoints*2*sizeof(struct oper *));
    for(i = 0; opto != NULL; 
	opto = (struct ptoperlist FAR *) next(opto), i++){
      nattr->exprval.op[2*i] = SimplifyOper(gp, opto->x, nattr,
					    &simple, &sideeffect);
      *constant = *constant && simple;
      nattr->exprval.op[2*i+1] = SimplifyOper(gp, opto->y, nattr,
					      &simple, &sideeffect);
      *constant = *constant && simple;
    }
    /* Initialize curval and directval to nil.
     * TODO: Add routines that calculates proper default values.
     */
    nattr->curval.ptarr = CalCalloc(1, npoints*2*sizeof(int));
    nattr->directval.ptarr = CalCalloc(1, npoints*2*sizeof(int));
    for(;i < npoints; i++){
      nattr->curval.ptarr[2*i] = getXDefPoint(gp, parent, i);
      nattr->curval.ptarr[2*i+1] = getYDefPoint(gp, parent, i);
    }
    break;
  case S_COLOR:
  case S_ACTIVE:
  case S_FILL:
  case S_LINEWIDTH:
  case S_STARTANGLE:
  case S_ENDANGLE:
  case S_SAVEBG:
  case S_DECIMALS:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typint = attr->curval.typint;
    nattr->directval.typint = attr->directval.typint;
    *constant = *constant && simple;
    break;
  case S_LEVEL:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typint = attr->curval.typint;
    nattr->directval.typlev = attr->directval.typlev;
    *constant = *constant && simple;
    break;
  case S_TEXTURE:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typint = attr->curval.typint;
    nattr->directval.typimage = attr->directval.typimage;
    *constant = *constant && simple;
    break;
  case S_DASHES:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typint = attr->curval.typint;
    nattr->directval.typdash = attr->directval.typdash;
    *constant = *constant && simple;
    break;
  case S_OUTINT:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typint = attr->curval.typint;
    nattr->directval.typtxtnode = attr->directval.typtxtnode;
    *constant = *constant && simple;
    break;
  case S_OUTFLOAT:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typflt = attr->curval.typflt;
    nattr->directval.typtxtnode = attr->directval.typtxtnode;
    *constant = *constant && simple;
    break;
  case S_IMAGE:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typtxt = safestrdup(attr->curval.typtxt);
    nattr->directval.typimage = attr->directval.typimage;
    *constant = *constant && simple;
    break;
  case S_FONT:
  case S_OUTTEXT:
    if(attr->exprval.oper)
      nattr->exprval.oper = SimplifyOper(gp, attr->exprval.oper, nattr,
					 &simple, &sideeffect);
    nattr->curval.typtxt = safestrdup(attr->curval.typtxt);
    *constant = *constant && simple;
    nattr->directval.typtxtnode = attr->directval.typtxtnode;
    break;
  case S_TRANSLATION:
    if(attr->indirect) 
      tl = attr->exprval.ap->actval.tl; 
    else 
      tl = attr->exprval.tl;

    for(; tl != NULL; tl = next(tl)){
      if((ntl = NEWTRANSLATION) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      ntl->event = tl->event;
      ntl->act = i_simactions(gp, tl->act, parent->parentbi);
      if(nattr->exprval.tl == NULL){
	init_list(ntl);
	nattr->exprval.tl = ntl;
      } else 
	place_next(ptl, ntl);
      ptl = ntl;
    }
    return nattr;
  default:
    sprintf(errormsg, ErrINTUNKATT, 
	    nattr->attrtype);
    errorMsg(gp, 1, errormsg);
    break;
  }
  
  if(sideeffect == TRUE){
    if(gp->curlunit->fattr == NULL){
      init_list(nattr);
      gp->curlunit->fattr = nattr;
    } else {
      lattr = last(gp->curlunit->fattr);
      place_next(lattr, nattr);
    }
  }else {
    if(gp->curlunit->nattr == NULL){
      init_list(nattr);
      gp->curlunit->nattr = nattr;
    } else 
      place_next(gp->curlunit->lnattr, nattr);
    gp->curlunit->lnattr = nattr;
  }

  return nattr;
}

/*
 * Return pointer to an instantiated graphic object.
 */
static struct graphobj FAR *i_graphobj(struct cw_status *gp,
				       struct graphobj FAR *go, 
				       struct blockinst FAR *bi,
				       struct winobj FAR *parentwin)
{
  struct graphobj FAR *gi;
  char constant = OPTIMIZABLE;
  struct commonobj FAR *co;

  if((gi = NEWGRAPHOBJ) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  co = (struct commonobj FAR *) gi;
  gi->parentbi = bi;
  gi->parentwin = parentwin;
  gi->type 	= go->type;
  gi->visible	= FALSE;
  gi->status	= VALID;
  gi->ptcount 	= go->ptcount;

   
  gi->active   	= i_attribute(gp, go->active, co, &constant);
  gi->saveback 	= i_attribute(gp, go->saveback, co, &constant);
  gi->level 	= i_attribute(gp, go->level, co, &constant);
  gi->color 	= i_attribute(gp, go->color, co, &constant);
  gi->dashes 	= i_attribute(gp, go->dashes, co, &constant);
  gi->linewidth = i_attribute(gp, go->linewidth, co, &constant);
  gi->fill 	= i_attribute(gp, go->fill, co, &constant);
  gi->image 	= i_attribute(gp, go->image, co, &constant);
  gi->texture 	= i_attribute(gp, go->texture, co, &constant);
  gi->startangle =i_attribute(gp, go->startangle, co, &constant);
  gi->endangle 	= i_attribute(gp, go->endangle, co, &constant);
  gi->pointlist	= i_attribute(gp, go->pointlist, co, &constant);
  gi->font 	= i_attribute(gp, go->font, co, &constant);
  gi->decimals 	= i_attribute(gp, go->decimals, co, &constant);
  gi->outint 	= i_attribute(gp, go->outint, co, &constant);
  gi->outfloat 	= i_attribute(gp, go->outfloat, co, &constant);
  gi->outtext 	= i_attribute(gp, go->outtext, co, &constant);

/* If the instance is static, its attributes are computed only here */
  gi->objstatic = constant;

  gi->live = TRUE;
  return gi;
}

/*
 * Return pointer to an instantiated graphic object.
 */
static struct inpobj FAR *i_inpobj(struct cw_status *gp,
				   struct inpobj FAR *io, 
				   struct blockinst FAR *bi,
				   struct winobj FAR *parentwin)
{
  struct inpobj FAR *ii;
  char constant = TRUE;
  struct commonobj FAR *co;

  if((ii = NEWINPOBJ) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  co = (struct commonobj FAR *) ii;
  ii->parentbi = bi;
  ii->parentwin = parentwin;
  ii->ptcount 	= io->ptcount;
  ii->pointlist = i_attribute(gp, io->pointlist, co, &constant);
  ii->level 	= i_attribute(gp, io->level, co, &constant);
  ii->active 	= i_attribute(gp, io->active, co, &constant);
  ii->translation = i_attribute(gp, io->translation, co, &constant);

  ii->live	= TRUE;

  return ii;
}


/*
 * Return pointer to an instantiated window object.
 */
static struct winobj FAR *i_winobj(struct cw_status *gp,
				   struct winobj FAR *wo, struct blockinst *bi)
{
  struct winobj FAR *wi;
  struct graphobj FAR *go, FAR *gn;
  struct inpobj FAR *io, FAR *in;
  char constant=OPTIMIZABLE;
  struct commonobj FAR *co;

  if((wi = NEWWINOBJ) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  co = (struct commonobj FAR *) wi;
  wi->parentbi = bi;
  wi->status 	= VALID;
  wi->visible	= FALSE; 
  wi->ptcount 	= wo->ptcount;
  wi->pointlist = i_attribute(gp, wo->pointlist, co, &constant);
  wi->active 	= i_attribute(gp, wo->active, co, &constant);
  wi->saveback 	= i_attribute(gp, wo->saveback, co, &constant);
  wi->level 	= i_attribute(gp, wo->level, co, &constant);
  init_list(wi->level0 = newLevNode (gp, 0));


  for(go = wo->graphs; go != NULL; go = (struct graphobj FAR *) next2(go)){
    gn = i_graphobj(gp, go, bi, wi);
    constant = constant && gn->objstatic;
    if(wi->graphs != NULL)
      place_prev2(wi->graphs, gn);
    else
      init_list2(gn);
    wi->graphs = gn;
  }
  for(io = wo->inps; io != NULL; io = (struct inpobj FAR *) next2(io)){
    in = i_inpobj(gp, io, bi, wi);
    if(wi->inps != NULL)
      place_prev2(wi->inps, in);
    else
      init_list2(in);
    wi->inps = in;
  }
  wi->objstatic = constant;
  wi->live	= TRUE;

  return wi;
}

void optOCleanup (struct cw_status *gp)
{
  struct levlist *gll, *wll;
  struct winobj *wo;
  struct graphobj *go;

  for (gll = first(gp->curlunit->level0); gll; gll = next(gll)) {
    if (gll->ownsmask) {
      killImage (gp, gll->maskstatic);
      gll->ownsmask = FALSE;
    }
    gll->maskstatic = NULL;
    for (go = gll->levgraphs; go; go = next(go))
      if (go->objstatic == OPTIMIZED) go->objstatic = OPTIMIZABLE;
    for (wo = gll->levwins; wo; wo = next(wo)) {
      for (wll = first(wo->level0); wll; wll = next(wll)) {
	if (wll->ownsmask) {
	  killImage (gp, wll->maskstatic);
	  wll->ownsmask = FALSE;
	}
	wll->maskstatic = NULL;
      }
      if (wo->objstatic == OPTIMIZED) wo->objstatic = OPTIMIZABLE;
    }
  }
/*  killImage (gp->curlunit->picstatic);
  gp->curlunit->picstatic =
    blankImage (gp->curlunit->win->width,
		gp->curlunit->win->height,
		gp->curlunit->win->color); */
  clearImage (gp, gp->curlunit->picstatic->picture,
	      gp->curlunit->picstatic->width,
	      gp->curlunit->picstatic->height);
}

/*
 * Return from function.
 */

void r_function(struct cw_status *glp, struct function FAR *fp,
		struct funcinst FAR *fi)
{
  struct winobj FAR *wp;
  struct graphobj FAR *gp;
  struct inpobj FAR *ip;
  struct vdecl FAR *vd;
  struct vinst FAR *vi, FAR *vip;
  struct vinstlist FAR *vil, FAR *vilp;
  struct vdecllist *vl, *nvl;
  struct blockinst *bi;

  int sb;

  /* Put objects referred to by ref-parameters into the objectlist 
   * of the caller.
   */
  if(fi->caller != NULL){
    for(vl = fi->caller->vl; vl != NULL; vl = next(vl)){
      vl->vd->array = vl->array;
    }
    for(wp = fi->fwin; wp != NULL;){
	  sb = wp->saveback ?
	    wp->saveback->directval.typint : DEF_SAVEBACKGROUND;
	  if (sb && wp->active->directval.typint == 1) {
	    wp->active->directval.typint = wp->active->curval.typint = 0;
	    wp->status = max(wp->status, STALESBGEOM);
	  }
	  wp = (struct winobj FAR *)next2(wp);
    }
    for(gp = fi->fgraph; gp != NULL;){
	  sb = gp->saveback ?
	    gp->saveback->directval.typint : DEF_SAVEBACKGROUND;
	  if (sb && gp->active->directval.typint == 1) {
	    gp->active->directval.typint = gp->active->curval.typint = 0;
	    gp->status = max(gp->status, STALESBGEOM);
	  }
	  gp = (struct graphobj FAR *)next2(gp);
    }
    for(ip = fi->finp; ip != NULL;){
	  ip->active->curval.typint = ip->active->directval.typint = 0;
	  ip = (struct inpobj FAR *)next2(ip);
    }
  }

  if (fi->optimized == 0) refresh (glp, FALSE);
  /* Remove all the objects to be killed from screen */
  /*  refresh (); Should be done by statobjSetup */

  /* INVAR: There is no ref-parameters left in either of the lists : 
   * fi->fwin, fi->fgraph and fi->finp.
   */
  f_winobj(glp, fi->fwin);
  f_graphobj(glp, fi->fgraph);
  f_inpobj(glp, fi->finp);


  vi = (struct vinst FAR *)next(fi->firstvi);
  remove_at(fi->firstvi);
  CalFree(fi->firstvi);
  if(fi->caller != NULL){
    for(;vi != NULL;){
      vip = vi;
      vi = (struct vinst FAR *)next(vi);
      remove_at(vip);
      if(vip->live){ 
	/* Move variable instance to caller */
	place_next(fi->caller->firstvi, vip);
	vip->live = FALSE;
      } else
	f_vinst(glp, vip);
    }
  } else {
    for(;vi != NULL;){
      vip = vi;
      vi = (struct vinst FAR *)next(vi);
      remove_at(vip);
      f_vinst(glp, vip);
    }
  }

  for (bi = fi->caller;
       bi && bi->parent->type != FUNCTIONBLOCK;
       bi = bi->caller);
  if (fi->optimized > 0 && bi) {
    /* Set up all remaining objects again */
    optOCleanup (glp);
    glp->curlunit->output -= fi->optimized;
    statobjSetup (glp, bi);
  }

  /* Possibly also have to redraw all dynamic objects here, too */

  f_simulate(glp, fi->sim, FALSE);

  /*
   * Restore old values of fp->firstv
   */
  for(vd=fp->firstv, vil=fi->foldvi; vd != NULL;){
    vd->vi = vil->vi;
    vilp = vil;
    vil = (struct vinstlist FAR *)next(vil);
    vd = (struct vdecl FAR *)next(vd);
    remove_at(vilp);
    CalFree(vilp);
  }
  for(vl=fi->vl; vl != NULL;){
    nvl = vl;
    vl = next(vl);
    f_intarray(glp, nvl->vd->array);
    nvl->vd->array = NULL;
    remove_at(nvl);
    CalFree(nvl);
  }
}

/*
 * Instantiate the function 'fp', save the old values of the function 
 * in 'fi' before overwriting them.
 */
void i_function(struct cw_status *gp, struct function FAR *fp,
		struct param FAR *act_par, struct funcinst FAR *fi)
{
  struct vdecl FAR *v;
  struct fparam FAR *fpar;
  struct vinstlist FAR *vil, FAR *vilp;
  struct vdecllist FAR *nvl, *fvl=NULL;
  char errormsg[256];
  
  
  if(fi->caller != NULL){
    for(nvl = fi->caller->vl; nvl != NULL; nvl = next(nvl)){
      nvl->vd->array = NULL;
    }
  }
  if((fi->firstvi = NEWVINST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(fi->firstvi);
  fi->foldvi = NULL;
  for(v = fp->firstv; v != NULL; v = (struct vdecl FAR *)next(v)){
    if((vil = NEWVINSTLIST) == NULL){
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    vil->vi = v->vi;
    if(fi->foldvi == NULL)
      init_list(fi->foldvi = vil);
    else
      place_next(vilp, vil);
    vilp = vil;
  }

  /* Check and compute the actual parameters */
  for(fpar = fp->fpar;
      fpar != NULL;
      fpar = (struct fparam FAR *)next(fpar)){
    if(act_par == NULL){
      sprintf(errormsg, 
	      ErrTOOFEWPAR, 
	      fp->name);
      errorMsg(gp, 1, errormsg);
      return;
    }
    if(fpar->is_ref == TRUE){
      switch(act_par->exp->flag){
      case INTVAR:
	if(fpar->var->type == INTVAL){
	  fpar->var->vi = act_par->exp->left.vi;
	} else {
	  sprintf(errormsg, 
		  ErrFORINTVAR, fpar->var->name, fp->name);
	  errorMsg(gp, 1, errormsg);
	  return;
	} 
	break;
      case FLOATVAR:
	if(fpar->var->type == FLOATVAL){
	  fpar->var->vi = act_par->exp->left.vi;
	} else {
	  sprintf(errormsg, 
		  ErrFORFLOVAR, fpar->var->name, fp->name);
	  errorMsg(gp, 1, errormsg);
	  return;
	}
	break;
      case TEXTVAR:
	if(fpar->var->type == TEXTVAL){
	  fpar->var->vi = act_par->exp->left.vi;
	} else {
	  sprintf(errormsg, 
		  ErrFORTXTVAR, fpar->var->name, fp->name);
	  errorMsg(gp, 1, errormsg);
	  return;
	}
	break;
      case INTARRAY:
	if(fpar->var->type == INTARRAY){
	  if(fpar->var->ndim == 
	     act_par->exp->left.array->which.vi->parent->ndim)
	    {
	      fpar->var->vi = act_par->exp->left.array->which.vi;
	    } else {
	      sprintf(errormsg, 
		      "Actual array parameter '%s' does not match dimension of formal parameter '%s' in function '%s'.", 
		      act_par->exp->left.array->which.vi->parent->name, 
		      fpar->var->name, fp->name);
	      paramError(gp, act_par, errormsg);
	    }
	} else if(fpar->var->type == INTVAL){
	  if((nvl = NEWVDECLLIST) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  nvl->vd = fpar->var;
	  nvl->array = i_intarray(gp, act_par->exp->left.array);
	  if(fvl == NULL)
	    init_list(fvl = nvl);
	  else
	    place_next(fvl, nvl);
	} else {
	  sprintf(errormsg, 
		  ErrFORINTARR, fpar->var->name, fp->name);
	  errorMsg(gp, 1, errormsg);
	  return;
	} 
	break;
      case FLOATARRAY:
      	if(fpar->var->type == FLOATARRAY){
	  if(fpar->var->ndim == 
	     act_par->exp->left.array->which.vi->parent->ndim)
	    {
	      fpar->var->vi = act_par->exp->left.array->which.vi;
	    } else {
	      sprintf(errormsg, 
		      "Actual array parameter '%s' does not match dimension of formal parameter '%s' in function '%s'.", 
		      act_par->exp->left.array->which.vi->parent->name, 
		      fpar->var->name, fp->name);
	      paramError(gp, act_par, errormsg);
	    }
	} else if(fpar->var->type == FLOATVAL) {
	  if((nvl = NEWVDECLLIST) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  nvl->vd = fpar->var;
	  nvl->array = i_intarray(gp, act_par->exp->left.array);
	  if(fvl == NULL)
	    init_list(fvl = nvl);
	  else
	    place_next(fvl, nvl);
	} else {
	  sprintf(errormsg, 
		  ErrFORFLOARR, fpar->var->name, fp->name);
	  errorMsg(gp, 1, errormsg);
	  return;
	} 
	break;
      case TEXTARRAY:
	if(fpar->var->type == TEXTARRAY){
	  if(fpar->var->ndim == 
	     act_par->exp->left.array->which.vi->parent->ndim)
	    {
	      fpar->var->vi = act_par->exp->left.array->which.vi;
	    } else {
	      sprintf(errormsg, 
		      "Actual array parameter '%s' does not match dimension of formal parameter '%s' in function '%s'.", 
		      act_par->exp->left.array->which.vi->parent->name, 
		      fpar->var->name, fp->name);
	      paramError(gp, act_par, errormsg);
	    }
	} else if(fpar->var->type == TEXTVAL) {
	  if((nvl = NEWVDECLLIST) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  nvl->vd = fpar->var;
	  nvl->array = i_intarray(gp, act_par->exp->left.array);
	  if(fvl == NULL)
	    init_list(fvl = nvl);
	  else
	    place_next(fvl, nvl);
	} else {
	  sprintf(errormsg, 
		  ErrFORTXTARR, fpar->var->name, fp->name);
	  errorMsg(gp, 1, errormsg);
	  return;
	} 
	break;
      default:
	sprintf(errormsg, ErrILLPARFUN, fp->name);
	errorMsg(gp, 1, errormsg);
        return;
      }
    } else {
      fpar->var->vi = c_vinst(gp, fpar->var);
      place_next(fi->firstvi, fpar->var->vi);
      switch(fpar->var->type){
      case INTVAL:
	fpar->var->vi->value.value = calculate(gp, act_par->exp);
	break;
      case FLOATVAL:
	fpar->var->vi->value.fvalue = f_calculate(gp, act_par->exp);
	break;
      case TEXTVAL:
	CalFree(fpar->var->vi->value.text);
	fpar->var->vi->value.text = t_calc_dup(gp, act_par->exp);
	break;
      default:
	sprintf(errormsg, 
		ErrINTVDECLT, 
		fpar->var->type);
	errorMsg(gp, 1, errormsg);
	return;
      }
    }
    act_par = (struct param FAR *)next(act_par);
  } /* end for */  

  fi->vl = fvl;
  /* Make sure references to arrays are instantiated correctly */
  for(nvl = fvl; nvl != NULL; nvl = next(nvl))
    nvl->vd->array = nvl->array;

  for(v = fp->firstv; v != fp->fparvar; v = (struct vdecl FAR *)next(v)){
    v->vi = c_vinst(gp, v);
    place_next(fi->firstvi, v->vi);
  }
  

  /* Instantiate code */
  fi->sim = i_simactions(gp, fp->sim, (struct blockinst FAR *)fi);
}

void r_object(struct cw_status *gp, struct objectproto FAR *object,
	      struct objectinst FAR *oi)
{
  struct vinst FAR *vi, FAR *vip;

  vi = (struct vinst FAR *)next(oi->firstvi);
  remove_at(oi->firstvi);
  CalFree(oi->firstvi);
  
  for(;vi != NULL;){
    vip = vi;
    vi = (struct vinst FAR *)next(vi);
    remove_at(vip);
    /* Move variable instance to caller */
    place_next(oi->caller->firstvi, vip);
    vip->live = FALSE;
  }
  
  f_simulate(gp, oi->sim, FALSE);
}

void i_object(struct cw_status *gp, struct objectproto FAR *object,
	      struct attrparam FAR *par, struct objectinst FAR *oi)
{
  struct vdecl FAR *v;
  char errormsg[256];
  struct attrparam FAR *fpar;

  if((oi->firstvi = NEWVINST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(oi->firstvi);

  /*
   * Save pointers to "old" variable instances, so they can be put 
   * back when returning from the function. 
   * NOTE: We should probably let the simulation code address the 
   * variable instances directly(through offset), so we could get rid 
   * of all this shit.
   */

  /* We don't need this if we don't have recursion. */
  /*
  oi->foldvi = NULL;
  for(v = object->firstv; v != NULL; v = (struct vdecl FAR *)next(v)){
    if((vil = NEWVINSTLIST) == NULL){
      errorMsg(2, ErrNOMOREMEM);
      c_exit(NOT_OK);
    }
    vil->vi = v->vi;
    if(oi->foldvi == NULL)
      init_list(oi->foldvi = vil);
    else
      place_next(vilp, vil);
    vilp = vil;
  }
  */

  /* Do params */
  for(;par != NULL;par = next(par)){
    fpar = get_attr(object, par->name);
    if(fpar == NULL){
      sprintf(errormsg, "Error: Illegal attribute %s", par->name);
      errorMsg(gp, 1, errormsg);
    }
    switch(fpar->type){
    case T_EXPR:
      fpar->actval.oper = par->actval.oper;
      break;
    case T_TRANSLATION:
      if(par->indirect)
	fpar->actval.tl = par->defval.tl;
      else
	fpar->actval.tl = par->actval.tl;
      break;
    case T_POINTS:
      if(par->indirect){
	fpar->actval.pl = par->defval.ap->actval.pl;
	fpar->npoints = par->defval.ap->npoints;
      } else {
	fpar->actval.pl = par->actval.pl;
	fpar->npoints = par->npoints;
      }
      break;
    }
  }


  /*
   * Create new variable instances.
   */
  for(v = object->firstv; v != NULL; 
      v = (struct vdecl FAR *)next(v)){
    v->vi = c_vinst(gp, v);
    place_next(oi->firstvi, v->vi);
  }
  
  /* Instantiate code */
  oi->sim = i_simactions(gp, object->sim, (struct blockinst FAR *)
			 gp->curlunit->cur_funcinst);

}

void create_winobj(struct cw_status *gp, struct blockinst FAR *bi,
		   struct winobj FAR *wo)
{
  struct winobj FAR *wn;
  wn = i_winobj(gp, wo, bi);
  /*  if(bi->caller != NULL || gp->curlunit->output)
      wn->objstatic = UNOPTIMIZABLE; */
  if(bi->fwin != NULL)
    place_prev2(bi->fwin, wn); 
  else
    init_list2(wn);
  bi->fwin = wn; 
}

void create_graphobj(struct cw_status *gp, struct blockinst FAR *bi,
		     struct graphobj FAR *go)
{
  struct graphobj FAR *gn;
  gn = i_graphobj(gp, go, bi, NULL);
  /*  if(bi->caller != NULL || gp->curlunit->output) 
    gn->objstatic = UNOPTIMIZABLE; */
  if(bi->fgraph != NULL)
    place_prev2(bi->fgraph, gn); 
  else
    init_list2(gn);
  bi->fgraph = gn;
}

void create_inpobj(struct cw_status *gp, struct blockinst FAR *bi,
		   struct inpobj FAR *io)
{
  struct inpobj FAR *in;
  in = i_inpobj(gp, io, bi, NULL);
  if(bi->finp != NULL)
    place_prev2(bi->finp, in); 
  else
    init_list2(in);
  bi->finp = in;
}




