/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */

#include <stdio.h>
#include <string.h> 
#include <stdlib.h> 
#include <malloc.h>

#include "candle.h"
#include "const.h"
#include "graphic.h"
#include "fast_lis.h"
#include "learnuni.h"
#include "simulate.h"
#include "function.h"
#include "attr.h"

#include "protos/memory.h"
#include "protos/freesim.h"
#include "protos/fast_lis.h"
#include "protos/objects.h"

void dbug_param(struct cw_status *gp, struct param FAR *par) {
struct param FAR *p;

  p = par;
  while (p) {
/*    op_debug(par->exp); */
    p = (struct param FAR *) next(p);
  }
}

/*
 * Free a generic list.
 * Assumes that the object does not contain pointers 
 * to anything that needs to be freed.
 */  
void f_fastlist(struct cw_status *gp, void FAR *elem)
{
  struct fparam FAR *f, FAR *fpar;
  fpar = (struct fparam FAR *) elem;
  while(fpar){
    f = (struct fparam FAR *) next(fpar);
    remove_at(fpar);
    CalFree(fpar);
    fpar = f;
  }
}


void f_winobj(struct cw_status *gp, struct winobj FAR *wp)
{
  struct winobj FAR *w;
  while(wp != NULL){
    w = (struct winobj FAR *) next2(wp);
    remove_at2(wp);
    killWinObj(gp, wp);
    wp = w;
  }
}

void f_graphobj(struct cw_status *glp, struct graphobj FAR *gp)
{
  struct graphobj FAR *g;
  while(gp != NULL){
    g = (struct graphobj FAR *) next2(gp);
    remove_at2(gp);
    killGraphObj(glp, gp);
    gp = g;
  }
}

void f_inpobj(struct cw_status *gp, struct inpobj FAR *ip)
{
  struct inpobj FAR *i;
  while(ip != NULL){
    i = (struct inpobj FAR *) next2(ip);
    remove_at2(ip);
    killInpObj(gp, ip);
    ip = i;
  }
}

void f_vinst(struct cw_status *gp, struct vinst FAR *v)
{
  long i, total_size=1;

  if(v){
    switch(v->parent->type){
    case INTARRAY:
      CalFree(v->value.elem);
      break;
    case FLOATARRAY:
      CalFree(v->value.felem);
      break;
    case TEXTARRAY:
      for(i = 0; i < v->parent->ndim;i++)
	total_size *= v->size[i];
      for(i = 0; i < total_size; i++)
	if(*(v->value.telem+i) != NULL)
	  CalFree(*(v->value.telem+i));
      CalFree(v->value.telem);
      break;
    case TEXTVAL:
      if(v->value.text != NULL)
	CalFree(v->value.text);
      break;
    }  
    if(v->size) 
      CalFree(v->size);
    CalFree(v);
  }
}

void f_vinsts(struct cw_status *gp, struct vinst FAR *vi)
{
  struct vinst FAR *v;
  long i, total_size;
  
  while(vi){
    v = vi;
    total_size = 1;
    if(v->parent){
      switch(v->parent->type){
      case INTARRAY:
	CalFree(v->value.elem);
	break;
      case FLOATARRAY:
	CalFree(v->value.felem);
	break;
      case TEXTARRAY:
	for(i = 0; i < v->parent->ndim;i++)
	  total_size *= v->size[i];
	for(i = 0; i < total_size;i++)
	  if(*(v->value.telem+i) != NULL)
	    CalFree(*(v->value.telem+i));
	CalFree(v->value.telem);
	break;
      case TEXTVAL:
	if(v->value.text != NULL)
	  CalFree(v->value.text);
	break;
      }  
      if(v->size) 
	CalFree(v->size);
    }
    vi = next(vi);
    remove_at(v);
    CalFree(v);
  }
}

void f_param(struct cw_status *gp, struct param FAR *par) 
{
  struct param FAR *p;

  while (par) {
    p = (struct param FAR *) next(par);
    remove_at(par);
    f_oper(gp, par->exp);
    CalFree(par);
    par = p;
  }
}

void f_fparam(struct cw_status *gp, struct fparam FAR *fpar)
{
  struct fparam FAR *f;
  while(fpar){
    f = (struct fparam FAR *) next(fpar);
    remove_at(fpar);
    CalFree(fpar);
    fpar = f;
  }
}

void f_notdecl(struct cw_status *gp, struct notdecl FAR *nd)
{
  struct notdecl FAR *n;
  while(nd){
    n = (struct notdecl FAR *) next(nd);
    remove_at(nd);
    if(nd->name)
      CalFree(nd->name);
    f_param(gp, nd->actual.par);
    CalFree(nd);
    nd = n;
  }
}

void f_functions(struct cw_status *gp, struct function FAR *fp)
{
  struct function FAR *f;
  while(fp){
    f = (struct function FAR *) next(fp);
    remove_at(fp);
    if(fp->name)
      CalFree(fp->name);
    f_constants(gp, fp->firstc);
    f_variables(gp, fp->firstv);
    f_fparam(gp, fp->fpar);
    f_simulate(gp, fp->sim, TRUE);
    f_notdecl(gp, fp->nd);
    CalFree(fp);
    fp = f;
  }
}

void f_attrparam(struct cw_status *gp, struct attrparam FAR *ap)
{
  struct attrparam FAR *a;
  while(ap != NULL){
    a = ap;
    ap = next(ap);
    remove_at(a);
    CalFree(a->name);
    switch(a->type){
    case T_EXPR:
      f_oper(gp, a->defval.oper);
      break;
    case T_POINTS:
      if(!a->indirect)
	killPoints(gp, a->defval.pl); 
      break;
    case T_TRANSLATION:
      if(!a->indirect)
	f_translations(gp, a->defval.tl, TRUE);
      break;
    }
    CalFree(a);
  }
}


void f_objectprotos(struct cw_status *gp, struct objectproto FAR *op)
{
  struct objectproto FAR *o;
  while(op){
    o = (struct objectproto FAR *) next(op);
    remove_at(op);
    if(op->name)
      CalFree(op->name);
    f_constants(gp, op->firstc);
    f_variables(gp, op->firstv);
    f_attrparam(gp, op->fparam);
    f_simulate(gp, op->sim, TRUE);

    CalFree(op);
    op = o;
  }
}



void f_funcinst(struct cw_status *gp, struct funcinst FAR *fi)
{
  if(fi){
    f_fastlist(gp, fi->foldvi);
    f_vinsts(gp, fi->firstvi);
    f_simulate(gp, fi->sim, FALSE);
    f_winobj(gp, fi->fwin);
    f_graphobj(gp, fi->fgraph);
    f_inpobj(gp, fi->finp);

    CalFree(fi);
  }
}

void f_objectinst(struct cw_status *gp, struct objectinst FAR *oi)
{
  if(oi){
    f_vinsts(gp, oi->firstvi);
    f_simulate(gp, oi->sim, FALSE);
    f_winobj(gp, oi->fwin);
    f_graphobj(gp, oi->fgraph);
    f_inpobj(gp, oi->finp);

    CalFree(oi);
  }
}



void f_action(struct cw_status *gp, struct action FAR *do_act) 
{
  if(do_act->type == T_OBJECT)
    f_attrparam(gp, do_act->actual.attr); 
  else if (do_act->actual.par) 
    f_param(gp, do_act->actual.par);
  
  CalFree(do_act);
}

void f_intarray(struct cw_status *gp, struct intarray FAR *ia)
{
  if(ia){
    f_param(gp, ia->indeks);
    CalFree(ia);
  }
}

void f_oper(struct cw_status *gp, struct oper FAR *o) 
{
  int i;
  if (o) {
    switch (o->flag) {
    case INTARRAY :
    case FLOATARRAY : 
    case TEXTARRAY :
      f_intarray(gp, o->left.array);
      break;
    case BINTFUNC :
    case BFLOATFUNC : 
    case BTEXTFUNC :
    case UINTFUNC :
    case UFLOATFUNC :
    case UTEXTFUNC :
      f_action(gp, o->left.do_act);
      break;
    case TEXTVAL : 
      CalFree(o->left.text);
      break;
    case INTOPER :
    case FLOATOPER : 
    case TEXTOPER :
      if (o->left.op)
	f_oper(gp, o->left.op);
      break;
    case ATTROPER :
      if(o->type == DO_VDECL){
	
      } else  if (o->type == DO_VINST){
	for(i=0;i < o->left.pt->n * 2;i++){
	  f_oper(gp, o->left.pt->op[i]);
	}
	CalFree(o->left.pt->op);
	CalFree(o->left.pt);
      }
    default  : 
      break;
    }
    if (o->right.op) f_oper(gp, o->right.op);
    CalFree(o);
  }
}


void f_simulate(struct cw_status *gp, struct sim_actions FAR *sim,
		int is_proto) 
{
  struct sim_actions FAR *s;

  while (sim) {
    s = (struct sim_actions FAR *) next(sim);
    remove_at(sim);
    if(sim->cond_act) f_simulate(gp, sim->cond_act, is_proto);
    if(sim->do_act) f_action(gp, sim->do_act);
    if(sim->eq) 
      f_oper(gp, sim->eq);
    if(sim->eq_op) f_oper(gp, sim->eq_op);
    if(sim->ret_act) f_oper(gp, sim->ret_act);
    if(sim->win && is_proto) killWinObj(gp, sim->win);
    if(sim->graph && is_proto) 
      killGraphObj(gp, sim->graph);
    if(sim->inp && is_proto) killInpObj(gp, sim->inp);
    CalFree(sim);
    sim = s;
  }
}

void f_constants(struct cw_status *gp, struct c_decl FAR *c) 
{
  struct c_decl FAR *hc;

  while (c) {
    hc = (struct c_decl FAR *) next(c);
    remove_at(c);
    if (c->name)
      CalFree(c->name);
    else {
    }
    if (c->type == TEXTVAL && c->thisc.text) CalFree(c->thisc.text);
    CalFree(c);
    c = hc;
  }
}

void f_outlist(struct cw_status *gp, struct outlist FAR *out) 
{
  struct outlist FAR *o;

  while (out) {
    o = (struct outlist FAR *) next(out);
    remove_at(out);
    CalFree(out);
    out = o;
  }
}

void f_vinit(struct cw_status *gp, struct vinit FAR *v)
{
  struct vinit FAR *vp;
  while(v){
    vp = v;
    f_vinit(gp, v->next_dim);
    f_oper(gp, v->exp);
    v = next(v);
    remove_at(vp);
    CalFree(vp);
  }
}



void f_variables(struct cw_status *gp, struct vdecl FAR *v) 
{
  struct vdecl FAR *hv;

  while (v) {
    hv = (struct vdecl FAR *) next(v);
    remove_at(v);
    
    f_vinit(gp, v->init);
    f_param(gp, v->dim_size);
    if (v->name)
      CalFree(v->name);

    CalFree(v);
    v = hv;
  }
}

void f_vdecl(struct cw_status *gp, struct vdecl FAR *v) 
{
  if(v) {
    remove_at(v);
    
    f_vinit(gp, v->init);
    f_param(gp, v->dim_size);
    if (v->name)
      CalFree(v->name);

    CalFree(v);
  }
}

void f_blockinsts(struct cw_status *gp, struct blockinst FAR *bi)
{
  struct blockinst FAR *b;
  while(bi){
    b = (struct blockinst FAR *) bi->caller;
    if(bi->parent->type == FUNCTIONBLOCK)
      f_funcinst(gp, (struct funcinst FAR *)bi);
    else if(bi->parent->type == OBJECTBLOCK)
      f_objectinst(gp, (struct objectinst FAR *)bi);
    bi = b;
  }
  gp->curlunit->cur_blockinst = NULL;
}

int freesim(struct cw_status *gp, struct learnunit FAR *learn) 
{
  struct vdecl FAR *v, FAR *cv;
  
  for(v = learn->firstv;v != NULL;){
    cv = next(v);
    f_vinst(gp, v->vi);
    f_vdecl(gp, v);
    v = cv;
  }
  
  f_blockinsts(gp, learn->cur_blockinst);
  f_functions(gp, learn->fp);
  f_objectprotos(gp, learn->fobject);
  f_notdecl(gp, learn->firstnd);
  flush_input(gp, NULL);
  return(1);
}
