/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnarr@ifi.uio.no)
 */

/*
 * Provide creation functions which initializes the structures in 
 * simulation.h properly.
 */

#include <stdio.h>
#include <string.h> 
#include <stdlib.h> 
#include <malloc.h>

#include "const.h"
#include "candle.h"
#include "simulate.h"
#include "error.h"
#include "fast_lis.h"

#include "protos/memory.h"

#include "protos/canutil.h"
#include "protos/fast_lis.h"
#include "protos/simulate.h"
#include "protos/instsim.h"
#include "protos/freesim.h"

struct fparam FAR *c_fparam(struct cw_status *gp, struct vdecl FAR *vd)
{
  struct fparam FAR *fpar;

  if((fpar = NEWFPARAM) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(fpar);
  fpar->var = vd;
  return fpar;
}

struct function FAR *c_function(struct cw_status *gp)
{
  struct function FAR *fp;
  
  if((fp = NEWFUNCTION) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(fp);
  return fp;
}

struct notdecl FAR *c_notdecl(struct cw_status *gp)
{
  struct notdecl FAR *nd;

  if((nd = NEWNOTDECL) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(nd);
  return nd;
}

void int_init(struct cw_status *gp, struct vdecl FAR *v,
	      struct vinst FAR *vi, struct vinit FAR *init, 
	      long FAR *elem, int dim)
{
  int dim_ind=0;
  struct oper FAR *iexp;
  while(init){
    if(dim_ind++ >= (int) vi->size[dim]){
      errorMsg(gp, 1, ErrTMINIEINT);
      return ;
    }

    if(init->next_dim != NULL){
      int_init(gp, v, vi, init->next_dim, elem, dim + 1);
    } else {
      iexp = i_oper(gp, init->exp);
      *elem = calculate(gp, iexp);
      f_oper(gp, iexp);
    }
    elem += vi->size[v->ndim+dim];
    init = next(init);
  }
}

void float_init(struct cw_status *gp, struct vdecl *v, struct vinst FAR *vi, 
		struct vinit FAR *init, double FAR *felem, int dim)
{
  int dim_ind=0;
  struct oper FAR *iexp;
  while(init){
    if(dim_ind++ >= (int) vi->size[dim]){
      errorMsg(gp, 1, ErrTMINIEFLO);
      return ;
    }
    if(init->next_dim != NULL){
      float_init(gp, v, vi, init->next_dim, felem, dim + 1);
    } else {
      iexp = i_oper(gp, init->exp);
      *felem = f_calculate(gp, iexp);
      f_oper(gp, iexp);
    }
    felem += vi->size[v->ndim+dim];
    init = next(init);
  }
}

void text_init(struct cw_status *gp, struct vdecl FAR *v,
	       struct vinst FAR *vi, struct vinit FAR *init, 
	       char FAR *FAR *telem, int dim)
{
  int dim_ind=0;
  int fill=0;
  struct oper FAR *iexp;

  while(init){
    if(dim_ind++ >= (int) vi->size[dim]){
      errorMsg(gp, 1, ErrTMINIETXT);
      return;
    }
    if(init->next_dim != NULL){
      text_init(gp, v, vi, init->next_dim, telem, dim + 1);
    } else {
      iexp = i_oper(gp, init->exp);
      *telem = t_calc_dup(gp, iexp);
      f_oper(gp, iexp);
      fill = 1;
    }
    telem += vi->size[v->ndim+dim];
    init = next(init);
  }
  while(fill && dim_ind++ < (int) vi->size[dim]){
    *telem = CalMalloc(1);
    **telem = '\0';
    telem += vi->size[v->ndim+dim];
  }
}


struct vinst FAR *c_vinst(struct cw_status *gp, struct vdecl FAR *v)
{
  struct vinst FAR *vi;
  unsigned long i=0, total_size;
  struct param FAR *par;
  char errormsg[256];
  struct oper FAR *iexp;

  if((vi = NEWVINST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(vi);
  vi->parent = v;
  switch(v->type){
  case INTVAL:
    if(v->init){
      iexp = i_oper(gp, v->init->exp);
      vi->value.value = calculate(gp, iexp);
      f_oper(gp, iexp);
    }
    break;
  case FLOATVAL:
    if(v->init){
      iexp = i_oper(gp, v->init->exp);
      vi->value.fvalue = f_calculate(gp, iexp);
      f_oper(gp, iexp);
    }
    break;
  case TEXTVAL:
    if(v->init){
      iexp = i_oper(gp, v->init->exp);
      vi->value.text = t_calc_dup(gp, iexp);
      f_oper(gp, iexp);
    }
    else{
      vi->value.text = CalMalloc(1);
      vi->value.text[0] = '\0';
    }
    break;
  case INTARRAY:
    vi->size =
      (unsigned long FAR *) CalMalloc(sizeof(unsigned long) * v->ndim * 2);
    vi->size[2 * v->ndim-1] = 1;
    total_size = 1;
    for(par = v->dim_size; par != NULL; par = next(par)){
      iexp = i_oper(gp, par->exp);
      vi->size[i] = calculate(gp, iexp);
      total_size *= vi->size[i++];
      f_oper(gp, iexp);
    }
    for(i = (unsigned long)(2*v->ndim) - 2; i > (unsigned long) v->ndim - 1; i--){
      vi->size[i] = vi->size[i+1] * vi->size[i + 1 - v->ndim];
    }
    vi->value.elem = (long FAR *) CalCalloc(1, (size_t) total_size*sizeof(long));
    int_init(gp, v, vi, v->init, vi->value.elem, 0);
    break;
  case FLOATARRAY:
    vi->size = (unsigned long FAR *) CalMalloc(sizeof(unsigned long) * v->ndim * 2);
    vi->size[2 * v->ndim-1] = 1;
    total_size = 1;
    for(par = v->dim_size; par != NULL; par = next(par)){
      iexp = i_oper(gp, par->exp);
      vi->size[i] = calculate(gp, iexp);
      total_size *= vi->size[i++];
      f_oper(gp, iexp);
    }
    for(i = (int)(2*v->ndim) - 2; i > (unsigned long) v->ndim - 1; i--){
      vi->size[i] = vi->size[i+1] * vi->size[i + 1 - v->ndim];
    }
    vi->value.felem =
      (double FAR *) CalCalloc(1, (size_t) total_size*sizeof(double));
    float_init(gp, v, vi, v->init, vi->value.felem, 0);
    break;
  case TEXTARRAY:
    vi->size =
      (unsigned long FAR *) CalMalloc(sizeof(unsigned long) * v->ndim * 2);
    vi->size[2 * v->ndim-1] = 1;
    total_size = 1;
    for(par = v->dim_size; par != NULL; par = next(par)){
      iexp = i_oper(gp, par->exp);
      vi->size[i] = calculate(gp, iexp);
      total_size *= vi->size[i++];
      f_oper(gp, iexp);
    }
    for(i = (int)(2*v->ndim) - 2; i > (unsigned long) v->ndim - 1; i--){
      vi->size[i] = vi->size[i+1] * vi->size[i + 1 - v->ndim];
    }
    vi->value.telem =
      (char FAR *FAR *)CalCalloc(1, (size_t) total_size*sizeof(char FAR *));
    text_init(gp, v, vi, v->init, vi->value.telem, 0);
    for(i = 0;i < total_size;i++){
      if(*(vi->value.telem+i) == NULL){
	 *(vi->value.telem+i) = CalMalloc(1);
	 **(vi->value.telem+i) = '\0';
      }
    }
    break;
  default:
    sprintf(errormsg, ErrINTUNKVTY, 
	    v->type, v->name);
    errorMsg(gp, 1, errormsg);
    return NULL;
  }
  return vi;
}

struct vinstlist FAR *c_vinstlist(struct cw_status *gp, struct vinst FAR *vi)
{
  struct vinstlist FAR *vil;

  if((vil = NEWVINSTLIST) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(vil);
  vil->vi = vi;
  return vil;
}

