/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no)
 */

/*
 * This file holds routines used for updating objects on the screen.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#ifndef MAXINT
#define MAXINT 32767
#else
#include <values.h>
#endif
#include "const.h"
#include "candle.h"
#include "lex.h"
#include "fast_lis.h"
#include "graphic.h"
#include "attr.h"
#include "learnuni.h"
#include "nodes.h"
#include "pbmplus.h"        /* For min/max-macros */
#include "error.h"

#include "protos/memory.h"

#include "sysproto.h"
#include "protos/fast_lis.h"
#include "protos/region.h"
#include "protos/simulate.h"
#include "protos/canutil.h"
#include "protos/instance.h"
#include "protos/update.h"

#include "awethor.h"

#ifdef OLD
/* Sort two coordinates according to position */
static void sort2Points (struct cw_status *gp, struct ptvallist FAR *list)
{
  struct ptvallist FAR *pl1, FAR *pl2;
  int x, y;

  pl1 = list;
  pl2 = (struct ptvallist FAR *)next (pl1);
  if (pl1->x > pl2->x) {
    x = pl1->x;
    pl1->x = pl2->x;
    pl2->x = x;
  }
  if (pl1->y > pl2->y) {
    y = pl1->y;
    pl1->y = pl2->y;
    pl2->y = y;
  }
}

static int equalPts(struct cw_status *gp, struct ptvallist FAR *plist1, struct ptvallist FAR *plist2)
{
  while (plist1 && plist2)
    if (plist1->x != plist2->x || plist1->y != plist2->y)
      return FALSE;
  if (plist1 == NULL && plist2 == NULL) return TRUE;
  return FALSE;
}

#endif

#define bbOverlap(BOX1, BOX2) ((BOX1)->width != 0 && (BOX1)->height != 0 \
			       && (BOX2)->width != 0 && (BOX2)->height != 0 \
			       && !((BOX1)->x > (BOX2)->x+(BOX2)->width \
				    || (BOX1)->y > (BOX2)->y+(BOX2)->height \
				    || (BOX1)->x + (BOX1)->width < (BOX2)->x \
				    || (BOX1)->y + (BOX1)->height < (BOX2)->y))

void getWinBB (struct cw_status *gp, struct winobj FAR *object)
{
  long *dval = object->pointlist->directval.ptarr;

  if (object->active && !object->active->directval.typint)
    object->boundingbox.width = 0;
  else {
    object->boundingbox.x = dval[0];
    object->boundingbox.y = dval[1];
    object->boundingbox.width = dval[2] - object->boundingbox.x;
    object->boundingbox.height = dval[3] - object->boundingbox.y;
  }
}

void getGraphBB(struct cw_status *gp, struct graphobj FAR *object)
{
  CalText FAR *textnode;
  int left, top, right, bottom;
  int lw, i;
  long *dval = object->pointlist->directval.ptarr;

  if (object->active && !object->active->directval.typint)
    object->boundingbox.width = 0;
  else {
/* The linewidth may increase the size of the object */
    lw = (object->linewidth &&
	  (!object->fill
	   || (object->fill && !object->fill->directval.typint))) ?
	     (object->linewidth->directval.typint+1)/2 : 0;
    if (lw == 0) lw++;

    switch (object->type) {
    case S_ARC:
    case S_BOX:
      left = dval[0];
      top = dval[1];
      right = dval[2];
      bottom = dval[3];
      if (right > left) {
	left -= lw; right += lw;
      }
      else {
	left += lw; right -= lw;
      }
      if (bottom > top) {
	top -= lw; bottom += lw;
      }
      else {
	top += lw; bottom -= lw;
      }
      break;
    case S_IMAGE:
      left = dval[0];
      top = dval[1];
      if(object->ptcount > 1){
	right = dval[2];
	bottom = dval[3];
	/* If width or height == 0 then the respective original size is to be used */
	if (left == right)
	  dval[2] = right = left + object->image->directval.typimage->width;
	if (top == bottom)
	  dval[3] = bottom = top + object->image->directval.typimage->height;
      } else {
	right = left + object->image->directval.typimage->width;
	bottom = top + object->image->directval.typimage->height;
      }
      break;
    case S_TEXTOBJECT:
/* Does merely read the CalText font instance, never changes it */
      textnode = object->outtext ? object->outtext->directval.typtxtnode
	: object->outint ? object->outint->directval.typtxtnode
	  : object->outfloat->directval.typtxtnode;
/* Object always has one of these */
#if (defined WIN31 || defined WIN32)
      left = dval[0];
      top = dval[1];
      right = left+textnode->maxwidth;
      bottom = top+textnode->lincount*(textnode->fontinfo->ascent
				       +textnode->fontinfo->descent);
#else
      left = dval[0]+textnode->fontinfo->xoffs;
      top = dval[1]-3; /* Compensate for  (dumb X-font) */
      right = left+textnode->maxwidth
	+textnode->fontinfo->descent+textnode->fontinfo->ascent;
      /* Compensate for italics */
      bottom = dval[1]+textnode->lincount*(textnode->fontinfo->ascent
				       +textnode->fontinfo->descent);
#endif

      break;
    case S_POINT:
    case S_LINE:
    case S_POLYGON:
      left = right = dval[0]; 
      top = bottom = dval[1];
      for(i = 1; i < object->ptcount; i++) {
	if (dval[2*i] < left) 
	  left = dval[2*i];
	else if (dval[2*i] > right) 
	  right = dval[2*i];
	if (dval[2*i+1] < top) 
	  top = dval[2*i+1];
	else if (dval[2*i+1] > bottom) 
	  bottom = dval[2*i+1];
      }
      left -= lw; top -= lw; right += lw; bottom += lw;
      break;
    default:
      errorMsg(gp, 1, ErrUNKNOBJ);
    }
    if (object->parentwin) {
      left += object->parentwin->boundingbox.x;
      top += object->parentwin->boundingbox.y;
      right = min(right+object->parentwin->boundingbox.x,
		  object->parentwin->boundingbox.x
		  +object->parentwin->boundingbox.width);
      bottom = min(bottom+object->parentwin->boundingbox.y,
		   object->parentwin->boundingbox.y
		   +object->parentwin->boundingbox.height);
    }
    object->boundingbox.x=left;
    object->boundingbox.y=top;
    object->boundingbox.width=right-left;
    object->boundingbox.height=bottom-top;
  }
}


CalRect screenRect (struct cw_status *gp)
{
  CalRect rectangle;

  rectangle.x = rectangle.y = 0;
  rectangle.width = gp->curlunit->win->width;
  rectangle.height = gp->curlunit->win->height;
  return rectangle;
}

CalRect isectRects (struct cw_status *gp, CalRect *rect1, CalRect *rect2)
{
  CalRect rectangle;

  if (rect1->width < 0) {
    rect1->x += rect1->width;
    rect1->width = -rect1->width;
  }
  if (rect1->height < 0) {
    rect1->y += rect1->height;
    rect1->height = -rect1->height;
  }
  if (rect2->width < 0) {
    rect2->x += rect2->width;
    rect2->width = -rect2->width;
  }
  if (rect2->height < 0) {
    rect2->y += rect2->height;
    rect2->height = -rect2->height;
  }

  rectangle.x = max (rect1->x, rect2->x);
  rectangle.y = max (rect1->y, rect2->y);
  rectangle.width =
    min (rect1->x + rect1->width, rect2->x + rect2->width) - rectangle.x;
  rectangle.height =
    min (rect1->y + rect1->height, rect2->y + rect2->height) - rectangle.y;

  if (rectangle.width < 0) rectangle.width = 0;

  return rectangle;
}

void insRect (struct cw_status *gp, struct rectlist FAR *FAR *rlistp, int key, CalRect FAR *rectangle)
{
  struct rectlist FAR *newrectnode;

  if (gp->rectcount >= gp->rectmaxcount) {
    gp->rectmaxcount *= 2;
    CalFree (gp->rectspace);
    if ((gp->rectspace =
	 CalMalloc (gp->rectmaxcount*sizeof (struct rectlist))) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      exit (NOT_OK);
    }
  }

  newrectnode = gp->rectspace+(gp->rectcount++);

  newrectnode->rectangle = *rectangle;
  newrectnode->level = key;

  if (*rlistp) {
    if (key < 0) {
      place_prev(*rlistp, newrectnode);
      (*rlistp) = newrectnode;
    }
    else place_next(last(*rlistp), newrectnode);
  }
  else init_list ((*rlistp) = newrectnode);
}

CalText FAR *newTxtNode (struct cw_status *gp) {
  CalText FAR *tn;

  if (!(tn = (CalText FAR *)CalCalloc (1, sizeof (CalText)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    exit (NOT_OK);
  }
  getDefFInfo (gp, tn);
  tn->defaultfont = TRUE;

  return tn;
}

void getFont (struct cw_status *gp, struct attribute FAR *attr)
{            
  char mess[160];
  if (attr->parent.typgraph->outtext)
    attr->directval.typtxtnode =
      attr->parent.typgraph->outtext->directval.typtxtnode;
  else if (attr->parent.typgraph->outint)
    attr->directval.typtxtnode =
      attr->parent.typgraph->outint->directval.typtxtnode;
  else if (attr->parent.typgraph->outfloat)
    attr->directval.typtxtnode =
      attr->parent.typgraph->outfloat->directval.typtxtnode;
  
  if (!attr->directval.typtxtnode)
    attr->directval.typtxtnode = newTxtNode (gp);
  else freeFont (gp, attr->directval.typtxtnode);

  if (!getFInfo (gp, attr->directval.typtxtnode, attr->curval.typtxt)) {
    sprintf(mess, WarnFONTERR, attr->curval.typtxt);
    errorMsg(gp, 0, mess);
    getDefFInfo (gp, attr->directval.typtxtnode);
    attr->directval.typtxtnode->defaultfont = TRUE;
  }
  else
    attr->directval.typtxtnode->defaultfont = FALSE;
}

void newTextLine (struct cw_status *gp, CalText FAR *textnode,
		  char FAR *start, int length)
{
  char FAR *line;
  struct txtlist FAR *tl;

  if (length > 0 && !(line = (char FAR *)CalMalloc (length+1))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  if (!(tl = (struct txtlist FAR *)CalCalloc (1, sizeof (struct txtlist)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  
  tl->length = length;
  tl->line = strncpy (line, start, length);
  if(length > 0){
	  line[length] = '\0';
   	  if(line[length-1] == '\n' || line[length-1] == '\r'){
		line[length-1] = '\0';
	    tl->length--;
	   }
  }
  if (textnode->linlist) place_next(last(textnode->linlist), tl);
  else init_list(textnode->linlist = tl);
}

void tScanText (struct cw_status *gp, struct attribute FAR *attr,
		int width, int height)
{
  char FAR *c, FAR *linstart = attr->curval.typtxt, FAR *lastspc = NULL;
  int linwidth = 0, curheight = 0, maxwidth = 0, lincount = 0;
  int fontheight, cw;
  struct txtlist FAR *tl;

  /*  if (width <= 0) width = MAXINT;
  if (height <= 0) height = MAXINT; */

  if (!attr->directval.typtxtnode && attr->parent.typgraph->font)
    attr->directval.typtxtnode =
      attr->parent.typgraph->font->directval.typtxtnode;

  if (!attr->directval.typtxtnode)
    attr->directval.typtxtnode = newTxtNode (gp);

  fontheight = attr->directval.typtxtnode->fontinfo->ascent
    +attr->directval.typtxtnode->fontinfo->descent;
/* First free old nodes if any */
  while ((tl = attr->directval.typtxtnode->linlist)) {
    attr->directval.typtxtnode->linlist =
      (struct txtlist FAR *)next(attr->directval.typtxtnode->linlist);
    remove_at(tl);
    if (tl->line && tl->length > 0) {
      CalFree (tl->line);
      tl->line = NULL;
    }
    CalFree (tl);
  }

  for (c = linstart;; c++) {
    if (*c == '\0' || *c == '\n') {
      maxwidth = max(maxwidth, linwidth);
      lincount++;
      if ((curheight += fontheight) > height && height > 0) goto textOK;
      newTextLine (gp, attr->directval.typtxtnode, linstart, c-linstart);
      linstart = c+1;
      lastspc = NULL;
      linwidth = 0;
      if (*c == '\n') continue;
      else goto textOK;
    }
    cw = calCharWidth (gp, attr->directval.typtxtnode->fontinfo, c);
    if (width > 0 && linwidth+cw > width) {
/*First character outside legal region */
      maxwidth = max(maxwidth, linwidth);
      lincount++;
      if ((curheight += fontheight) > height && height > 0) goto textOK;
      if (lastspc) {
	newTextLine (gp, attr->directval.typtxtnode, linstart,
		     (*lastspc == ' ') ?
/* If space, discard */
		     lastspc-linstart : lastspc+1-linstart);
	linstart = (c = lastspc)+1;
	lastspc = NULL;
      }
      else {
	newTextLine (gp, attr->directval.typtxtnode, linstart, c-linstart);
/* In case first character on next line is space */
	if (*c == ' ') c++;
	linstart = c;
      }
      linwidth = 0;
      continue;
    }
    linwidth += cw;
/* Wrap characters */
    if (*c == ' ' || ispunct (*c)) lastspc = c;
  }
 textOK:
  attr->directval.typtxtnode->maxwidth = maxwidth;
  attr->directval.typtxtnode->lincount = lincount;
}

void tScanInt (struct cw_status *gp, struct attribute FAR *attr, int width)
{
  char txt[80];
  struct txtlist FAR *tl;

  /* Check whether the textnode is already created by a font-attribute */
  if (attr->parent.typgraph->font)
    attr->directval.typtxtnode =
      attr->parent.typgraph->font->directval.typtxtnode;

  /* Allocate new textnode if it doesn't already exist.
     It could be created over or first time if value-change */
  if (!attr->directval.typtxtnode)
    attr->directval.typtxtnode = newTxtNode (gp);

  /* Textnode contains at most one txtlist, pointed to by linlist */
  if (!(tl = attr->directval.typtxtnode->linlist)) {
    if (!(tl = (struct txtlist FAR *)CalCalloc (1, sizeof (struct txtlist)))) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      exit (NOT_OK);
    }
    init_list(attr->directval.typtxtnode->linlist = tl);
  }

  /* New value, discard previous */
  if (attr->directval.typtxtnode->linlist->line)
    CalFree (attr->directval.typtxtnode->linlist->line);
  if (!(attr->directval.typtxtnode->linlist->line =
	(char FAR *)CalMalloc (width+1))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    exit (NOT_OK);
  }

  sprintf (txt, "%ld", attr->curval.typint);
  strncpy (attr->directval.typtxtnode->linlist->line, txt, width);
  attr->directval.typtxtnode->linlist->length =
    strlen (attr->directval.typtxtnode->linlist->line);
  attr->directval.typtxtnode->lincount = 1;
  computeTxtWidth (gp, attr->directval.typtxtnode);
}

void tScanFlt (struct cw_status *gp, struct attribute FAR *attr, int width)
{
  char txt[1024], FAR *dotpos;
  struct txtlist FAR *tl;

  if (attr->parent.typgraph->font)
    attr->directval.typtxtnode =
      attr->parent.typgraph->font->directval.typtxtnode;

  if (!attr->directval.typtxtnode)
    attr->directval.typtxtnode = newTxtNode (gp);

  if (!attr->directval.typtxtnode->linlist) {
    if (!(tl = (struct txtlist FAR *)CalCalloc (1, sizeof (struct txtlist)))) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      exit (NOT_OK);
    }
    init_list(attr->directval.typtxtnode->linlist = tl);
  }
  if (attr->directval.typtxtnode->linlist->line)
    CalFree (attr->directval.typtxtnode->linlist->line);
  if (!(attr->directval.typtxtnode->linlist->line =
	(char FAR *)CalMalloc (width+1+1))) { /* width+'.'+terminating \0 */
    errorMsg(gp, 2, ErrNOMOREMEM);
    exit (NOT_OK);
  }

  sprintf (txt, "%f", attr->curval.typflt);
  if ((dotpos = strchr (txt, '.')) && dotpos- (char FAR *) txt < width)
    width++;
  strncpy (attr->directval.typtxtnode->linlist->line, txt, width);
  /* Terminate string */
  *(attr->directval.typtxtnode->linlist->line+width) = '\0';
  attr->directval.typtxtnode->linlist->length =
    strlen (attr->directval.typtxtnode->linlist->line);
  attr->directval.typtxtnode->lincount = 1;
  computeTxtWidth (gp, attr->directval.typtxtnode);
}

void recalcText (struct cw_status *gp, struct graphobj FAR *go)
{
  int tw, th, tl;
  long *dval = go->pointlist->directval.ptarr;

/* Checked in newGraphAttr that only one text-attrib is set */
  if (go->outtext) {
    if (go->ptcount > 1){
      tw = dval[2] - dval[0];
      th = dval[3] - dval[1];
    }
    else
      tw = th = 0;
    tScanText (gp, go->outtext, tw, th);
    return;
  }
  if (!go->decimals || (tl = go->decimals->directval.typint) == 0)
    tl = 16;
  if (go->outint) {
    tScanInt (gp, go->outint, tl);
    return;
  }
  if (go->outfloat) {
    tScanFlt (gp, go->outfloat, tl);
    return;
  }
}

void clearUpdateArea (struct cw_status *gp, CalRect FAR *urect)
{
/* Uses win to find values to force urect to be updated on first union-call */
  urect->x = gp->curlunit->win->width;
  urect->y = gp->curlunit->win->height;
  urect->width = -gp->curlunit->win->width;
  urect->height = -gp->curlunit->win->height;
}
  
void unionUpdateArea (struct cw_status *gp, CalRect FAR *urect,
		      CalRect FAR *newrect)
{
  register int uminx = urect->x, uminy = urect->y;
  register int umaxx = uminx+urect->width, umaxy = uminy+urect->height;
  if (newrect->width < 0) {
    newrect->x += newrect->width;
    newrect->width = -newrect->width;
  }
  if (newrect->height < 0) {
    newrect->y += newrect->height;
    newrect->height = -newrect->height;
  }
  
  uminx = min(uminx, newrect->x);
  uminy = min(uminy, newrect->y);
  umaxx = max(umaxx, newrect->x+newrect->width);
  umaxy = max(umaxy, newrect->y+newrect->height);
  
  urect->x = uminx;
  urect->y = uminy;
  urect->width = umaxx-uminx;
  urect->height = umaxy-uminy;
}

/* #define unionUpdateArea(URECT, NEWRECT) \ */
/*   if ((NEWRECT)->width < 0) { \ */
/*     (NEWRECT)->x += (NEWRECT)->width; \ */
/*     (NEWRECT)->width = -(NEWRECT)->width; \ */
/*   } \ */
/*   if ((NEWRECT)->height < 0) { \ */
/*     (NEWRECT)->y += (NEWRECT)->height; \ */
/*     (NEWRECT)->height = -(NEWRECT)->height; \ */
/*   } \ */
/*   (URECT)->x = min((URECT)->x, (NEWRECT)->x); \ */
/*   (URECT)->y = min((URECT)->y, (NEWRECT)->y); \ */
/*   (URECT)->width = \ */
/*     max((URECT)->x+(URECT)->width, (NEWRECT)->x+(NEWRECT)->width) \ */
/*      -(URECT)->x; \ */
/*   (URECT)->height = \ */
/*     max((URECT)->y+(URECT)->height, (NEWRECT)->y+(NEWRECT)->height) \ */
/*      -(URECT)->y; */
  
CalRect queryUpdateArea (struct cw_status *gp, CalRect FAR *urect)
{
  CalRect norect = {0, 0, 0, 0};

  if (urect->width > 0)
    return *urect;
  else
    return norect;
}

/*
static char graphOptimized (struct graphobj *go)
{
  struct blockinst *bi;

  if (go->objstatic != OPTIMIZED) return FALSE;
  for (bi = (struct blockinst *)gp->curlunit->cur_funcinst;
       bi != go->parentbi;
       bi = bi->caller) {
    assert (bi->parent);
    if (bi->parent->type == FUNCTIONBLOCK
	&& ((struct funcinst *)bi)->optimized > 0) return TRUE;
  }
  return (((struct funcinst *)bi)->optimized > 0);
}

static char winOptimized (struct winobj *wo)
{
  struct blockinst *bi;

  if (wo->objstatic != OPTIMIZED) return FALSE;
  for (bi = (struct blockinst *)gp->curlunit->cur_funcinst;
       bi != wo->parentbi;
       bi = bi->caller) {
    assert (bi->parent);
    if (bi->parent->type == FUNCTIONBLOCK
	&& ((struct funcinst *)bi)->optimized > 0) return TRUE;
  }
  return (((struct funcinst *)bi)->optimized > 0);
}
*/

void updateWinAttr (struct cw_status *gp, struct attribute FAR *attr)
{
  struct oper FAR * FAR *op = attr->exprval.op;
  long *cval, *dval;
  int i;
  struct levlist FAR *lnod1;
  struct winobj FAR *wlnod;
  int sb;
  char changed = FALSE;
  int valint;    
  char mess[160];

  assert (attr);

  sb = attr->parent.typwin->saveback ?
    attr->parent.typwin->saveback->directval.typint : DEF_SAVEBACKGROUND;

  switch (attr->attrtype) {
  case S_POINTS:
    cval = attr->curval.ptarr;
    dval = attr->directval.ptarr;
    for (i = 0; i < attr->parent.typwin->ptcount; i++) {
      if (op[2*i] != NULL) {
	cval[2*i] = calculate (gp, op[2*i]);
	cval[2*i+1] = calculate (gp, op[2*i+1]);
      }
      if (cval[2*i] != dval[2*i]) {
	changed = TRUE;
	dval[2*i] = cval[2*i];
      }
      if (cval[2*i+1] != dval[2*i+1]) {
	changed = TRUE;
	dval[2*i+1] = cval[2*i+1];
      }
    }
    if (changed || !attr->valid) {
      attr->parent.typwin->status =
	max(attr->parent.typwin->status, sb ? STALESBGEOM : STALEGEOM);
      attr->valid = TRUE;
    }
    break;
  case S_ACTIVE:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      attr->parent.typwin->status =
	max(attr->parent.typwin->status, sb ? STALESBGEOM : STALEGEOM);
      attr->directval.typint = attr->curval.typint;
      attr->valid = TRUE;
    }
    break;
  case S_SAVEBG:
    if (attr->exprval.oper) attr->curval.typint =
      calculate(gp, attr->exprval.oper);
    attr->directval.typint = attr->curval.typint;
    attr->valid = TRUE;
/* No need to set status here, because VALID has minimum value */
    break;
  case S_LEVEL:
/* Should not be called unless the level-value really has changed */
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      attr->parent.typwin->status =
	max(attr->parent.typwin->status, sb ? STALESBAPP : STALEAPP);
      /* Object must be taken out of this lev node */
      for (lnod1 = (struct levlist FAR *)first(gp->curlunit->level0);
	   lnod1;
	   lnod1 = (struct levlist FAR *)next(lnod1))
	for (wlnod = lnod1->levwins;
	     wlnod;
	     wlnod = (struct winobj FAR *)next(wlnod)) 
	  if (wlnod == attr->parent.typwin) {
	    /* Found object in levlist, remove it */
	    if (is_first(wlnod))
	      lnod1->levwins = (struct winobj FAR *)next(wlnod);
	    remove_at(wlnod);
	    if (!lnod1->ownsmask
		&& !lnod1->levwins && !lnod1->levgraphs && !lnod1->levinps
		&& lnod1->level != 0) {
	      remove_at(lnod1);
	      killLevNode (gp, lnod1);
	    }
/* Break all for-loops */
	    goto outOfList;
	  }
    outOfList:
      attr->directval.typlev =
	getLevNode (gp, &gp->curlunit->level0, attr->curval.typint);
      if (!attr->directval.typlev->maskstatic
	  && !is_last(attr->directval.typlev)
	  && ((struct levlist FAR *)next(attr->directval.typlev))->maskstatic)
	attr->directval.typlev->maskstatic =
	  ((struct levlist FAR *)next(attr->directval.typlev))->maskstatic;
      
      if (attr->directval.typlev->levwins)
	place_prev(attr->directval.typlev->levwins, attr->parent.typwin);
      else
	init_list(attr->parent.typwin);
      attr->directval.typlev->levwins = attr->parent.typwin;
    }
    attr->valid = TRUE;
    break;
  default:       
    sprintf(mess, ErrUNKNATTR, attr->attrtype);
    errorMsg(gp, 1, mess);
  }
}

void updateWins (struct cw_status *gp)
{
  struct attribute FAR *attr, FAR *attr_next;

  for (attr = gp->curlunit->fattr; 
       attr; attr = (struct attribute FAR *)next(attr)){
    if (attr->parenttype == WIN && attr->parent.obj->live)
      updateWinAttr (gp, attr);
  }

  for(attr = gp->curlunit->nattr; attr;) {
    if (attr->parenttype == WIN) {
      updateWinAttr(gp, attr);
    
      attr_next = (struct attribute FAR *)next(attr);
      if (attr == gp->curlunit->nattr) 
	gp->curlunit->nattr = attr_next;
      if(attr == gp->curlunit->lnattr)
	gp->curlunit->lnattr = prev(attr);
      remove_at(attr);
      attr = attr_next;
    }
    else 
      attr = next(attr);
  }
}

#ifdef AWETHOR
extern struct awe_object *ao_buf[];
#endif

void updateGraphAttr (struct cw_status *gp, struct attribute FAR *attr)
{
  struct oper FAR * FAR *op = attr->exprval.op;
  int i, tw, th, tl;
  struct levlist FAR *lnod1;
  struct graphobj FAR *glnod;
  int sb, pwsb;
  char changed = FALSE;
  int valint; char FAR *valtxt; double valflt;
  long *cval, *dval;

  assert (attr);

  sb = attr->parent.typgraph->saveback ?
    attr->parent.typgraph->saveback->directval.typint : DEF_SAVEBACKGROUND;
  if (attr->parent.typgraph->parentwin)
    pwsb = attr->parent.typgraph->parentwin->saveback ?
      attr->parent.typgraph->parentwin->saveback->directval.typint :
	DEF_SAVEBACKGROUND;

  switch (attr->attrtype) {
  case S_POINTS:
    cval = attr->curval.ptarr;
    dval = attr->directval.ptarr;
    for (i = 0; i < attr->parent.typgraph->ptcount; i++) {
      if (op[2*i] != NULL) {
	cval[2*i] = calculate (gp, op[2*i]);
	cval[2*i+1] = calculate (gp, op[2*i+1]);
      }
      if (cval[2*i] != dval[2*i]) {
	changed = TRUE;
	dval[2*i] = cval[2*i];
      }
      if (cval[2*i+1] != dval[2*i+1]) {
	changed = TRUE;
	dval[2*i+1] = cval[2*i+1];
      }
    }
/* If IMAGE is given 0-size, expand to original size */
    if (attr->parent.typgraph->type == S_IMAGE && 
	attr->parent.obj->ptcount > 1 && 
	(cval[0] == cval[2] || cval[1] == cval[3])) {
      dval[2] = cval[0] + 
	attr->parent.typgraph->image->directval.typimage->width;
      dval[3] = cval[1] + 
	+attr->parent.typgraph->image->directval.typimage->height;
    }

    if (changed || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBGEOM : STALEGEOM);
      if (attr->parent.typgraph->type == S_TEXTOBJECT)
	recalcText (gp, attr->parent.typgraph);
      attr->valid = TRUE;
    }
    break;
  case S_COLOR:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate (gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, STALEAPP);
      attr->directval.typint = colorOfRGB (attr->curval.typint);
      attr->valid = TRUE;
    }
    break;
  case S_IMAGE:
    if ((changed = attr->exprval.oper != NULL)) {
      changed &= (strcmp ((valtxt = t_calc_dup(gp, attr->exprval.oper)),
			  attr->curval.typtxt) != 0);
      if(changed){
	if(attr->curval.typtxt != NULL)
	  CalFree(attr->curval.typtxt);
	attr->curval.typtxt = valtxt;
      } else CalFree(valtxt);
    }
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBAPP : STALEAPP);
      if (attr->valid && attr->directval.typimage
	  && !attr->directval.typimage->in_cache)
	killImage (gp, attr->directval.typimage);

      attr->directval.typimage = getImage (gp, attr->curval.typtxt);
/* INVAR : getImage never returns NULL. Either correct image or default */
/* Update the points now that its dimension is known */
#if 0
      attr->parent.typgraph->pointlist->valid = FALSE;
      updateGraphAttr (gp, attr->parent.typgraph->pointlist);
#endif
      attr->valid = TRUE;
    }
    break;
  case S_ACTIVE:
  case S_FILL:
  case S_LINEWIDTH:
  case S_STARTANGLE:
  case S_ENDANGLE:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBGEOM : STALEGEOM);
      attr->directval.typint = attr->curval.typint;
      attr->valid = TRUE;
    }
    break;
  case S_SAVEBG:
    if (attr->exprval.oper) attr->curval.typint =
      calculate(gp, attr->exprval.oper);
    attr->directval.typint = attr->curval.typint;
    attr->valid = TRUE;
/* No need to set status here, because VALID has minimum value */
    break;
  case S_LEVEL:
/* Should not be called unless the level-value really has changed */
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBAPP : STALEAPP);

      for (lnod1 =
	   (struct levlist FAR *)
	     first(attr->parent.typgraph->parentwin ?
		    attr->parent.typgraph->parentwin->level0 :
		    gp->curlunit->level0);
	   lnod1;
	   lnod1 = (struct levlist FAR *)next(lnod1))
	for (glnod = lnod1->levgraphs;
	     glnod;
	     glnod = (struct graphobj FAR *)next(glnod)) 
	  if (glnod == attr->parent.typgraph) {
	    if (is_first(glnod))
	      lnod1->levgraphs = (struct graphobj FAR *)next(glnod);
	    remove_at(glnod);
	    if (!lnod1->ownsmask
		&& !lnod1->levwins && !lnod1->levgraphs && !lnod1->levinps
		&& lnod1->level != 0) {
	      remove_at(lnod1);
	      killLevNode (gp, lnod1);
	    }
/* Break all for-loops */
	    goto outOfList;
	  }
    outOfList:
      attr->directval.typlev =
	getLevNode (gp, attr->parent.typgraph->parentwin ?
		    &attr->parent.typgraph->parentwin->level0 :
		    &gp->curlunit->level0, attr->curval.typint);
      if (!attr->directval.typlev->maskstatic
	  && !is_last(attr->directval.typlev)
	  && ((struct levlist *)next(attr->directval.typlev))->maskstatic)
	attr->directval.typlev->maskstatic =
	  ((struct levlist *)next(attr->directval.typlev))->maskstatic;
      
      if (attr->directval.typlev->levgraphs)
	place_prev(attr->directval.typlev->levgraphs,
		    attr->parent.typgraph);
      else
	init_list(attr->parent.typgraph);
      attr->directval.typlev->levgraphs = attr->parent.typgraph;
    }
    attr->valid = TRUE;
    break;
  case S_DECIMALS:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBAPP : STALEAPP);
      attr->directval.typint = attr->curval.typint;
      attr->valid = TRUE;
    }
    break;
  case S_TEXTURE:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBAPP : STALEAPP);
      attr->directval.typimage = &gp->stipple[attr->curval.typint];
      attr->valid = TRUE;
    }
    break;
  case S_DASHES:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBAPP : STALEAPP);
      attr->directval.typdash = &gp->dashes[attr->curval.typint-1];
      attr->valid = TRUE;
    }
    break;
  case S_FONT:
    if ((changed = attr->exprval.oper != NULL)) {
      valtxt = t_calc_dup(gp, attr->exprval.oper);
      changed &= (strcmp (valtxt, attr->curval.typtxt) != 0);
      if (changed) {
	if (attr->curval.typtxt != NULL) 
	  CalFree (attr->curval.typtxt);
	attr->curval.typtxt = valtxt;
      }
      else CalFree (valtxt);
    }
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBTEXT : STALETEXT);
      getFont (gp, attr);
      attr->valid = TRUE;
    }
    break;
  case S_OUTTEXT:
/* The atribute's object should have at least one point here (checkObjAttrs) */
    if ((changed = attr->exprval.oper != NULL)) {
      changed &= (strcmp ((valtxt = t_calc_dup(gp, attr->exprval.oper)),
			 attr->curval.typtxt) != 0);
      if (changed) {
	if (attr->curval.typtxt != NULL)
	  CalFree (attr->curval.typtxt);
	attr->curval.typtxt = valtxt;
      }
      else CalFree(valtxt);
    }
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBTEXT : STALETEXT);
      
      dval = attr->parent.obj->pointlist->directval.ptarr;
      if(attr->parent.obj->ptcount > 1){
	tw = dval[2] - dval[0];
	th = dval[3] - dval[1];
      } else
	tw = th = 0;
      tScanText (gp, attr, tw, th);
      attr->valid = TRUE;
    }
    break;
  case S_OUTINT:
    if ((changed =
	 (attr->exprval.oper
	  && (valint =
	      calculate(gp, attr->exprval.oper)) != attr->curval.typint)))
      attr->curval.typint = valint;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBTEXT : STALETEXT);
      if (attr->parent.typgraph->decimals)
	tl = attr->parent.typgraph->decimals->directval.typint;
      else
	tl = 16;
      tScanInt (gp, attr, tl);
      attr->valid = TRUE;
    }
    break;
  case S_OUTFLOAT:
    if ((changed =
	 (attr->exprval.oper
	  && (valflt =
	      f_calculate(gp, attr->exprval.oper)) != attr->curval.typflt)))
      attr->curval.typflt = valflt;
    if (changed || !attr->exprval.oper || !attr->valid) {
      if (attr->parent.typgraph->parentwin)
	attr->parent.typgraph->parentwin->status =
	  max(attr->parent.typgraph->parentwin->status,
	      pwsb ? STALESBAPP : STALEAPP);
      else
	attr->parent.typgraph->status =
	  max(attr->parent.typgraph->status, sb ? STALESBTEXT : STALETEXT);
      if (!attr->parent.typgraph->decimals
	  || (tl = attr->parent.typgraph->decimals->directval.typint) == 0)
	tl = 16;
      tScanFlt(gp, attr, tl);
      attr->valid = TRUE;
    }
    break;
  default:
#ifdef UNIX
    /* Unknown attribute */
    assert (0);
#endif
	;
  }
}

void updateGraphs (struct cw_status *gp)
{
  struct attribute FAR *attr, FAR *attr_next;

/* Do not serve static objects requests */
  for (attr = gp->curlunit->fattr; 
       attr; attr = (struct attribute FAR *)next(attr)){
    if (attr->parenttype == GRAPH)
      updateGraphAttr (gp, attr);
  }

  for(attr = gp->curlunit->nattr; attr;) {
    if (attr->parenttype == GRAPH && attr->parent.obj->live) {
      updateGraphAttr(gp, attr);
    
      attr_next = (struct attribute FAR *)next(attr);
      if (attr == gp->curlunit->nattr) 
	gp->curlunit->nattr = attr_next;
      if(attr == gp->curlunit->lnattr)
	gp->curlunit->lnattr = prev(attr);
      remove_at(attr);
      attr = attr_next;
    }
    else 
      attr = next(attr);
  }

/* Now, all the graph-attributes in the list are removed, their respective
   objects are notified about the changes, and the objects' attributes are
   updated */
}

void refresh (struct cw_status *gp, char clearall)
{  
  struct levlist FAR *lnod1;
  struct winobj FAR *wlnod;
  struct rectlist FAR *rnod, FAR *levrects = NULL;
  int upkey = 0, lowkey = 0;   /* chronological keys for the level-rects */
  CalRect staleArea, defrect = screenRect (gp);
  struct graphobj FAR *glnod;
  char mess[160];

  gp->rectcount = 0;

/* Traverse all objects. Build a list of rectangles, describing the areas
   that must be restored. The levellist must be up to date */

  clearUpdateArea (gp, &gp->dynupdate);
  if (clearall) unionUpdateArea (gp, &gp->dynupdate, &defrect);

  for (lnod1 = (struct levlist FAR *)first(gp->curlunit->level0);
       lnod1;
       lnod1 = (struct levlist FAR *)next(lnod1)) {

    for (wlnod = lnod1->levwins;
	 wlnod;
	 wlnod = (struct winobj FAR *)next(wlnod)) {
/* Never update static objects */
      if (wlnod->objstatic == OPTIMIZED) continue;
      switch (wlnod->status) {
/* The use of upkey indecates that redrawing is necessary only from
   the level and up. Lowkey means that all levels must be redrawn */

      case STALESBAPP:
	if (wlnod->visible) {
	  copyStatPic (gp, gp->curlunit->picstatic, wlnod->boundingbox);
	  insRect (gp, &levrects, --lowkey, &wlnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &wlnod->boundingbox);
	}
      case VALID:
	break;
	
      case STALESBGEOM:
	if (wlnod->visible) {
	  copyStatPic (gp, gp->curlunit->picstatic, wlnod->boundingbox);
	  insRect (gp, &levrects, --lowkey, &wlnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &wlnod->boundingbox);
	}
      case STALEGEOM:
	getWinBB (gp, wlnod);
	wlnod->visible = bbOverlap (&defrect, &wlnod->boundingbox);
      case STALEAPP:
	if (wlnod->visible) {
	  insRect (gp, &levrects, upkey++, &wlnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &wlnod->boundingbox);
	}
	break;

/* Code removed because window objects cannot have text attribute */
      default:
	sprintf(mess, ErrUNKNSTAT, wlnod->status);
	errorMsg(gp, 2, mess);
	c_exit (gp, NOT_OK);
      }
    }

/* The same code as above, actually, only the window types are replaced with
   graphic object types */

    for (glnod = lnod1->levgraphs;
	 glnod;
	 glnod = (struct graphobj FAR *)next(glnod)) {
/* Never update static objects */
      if (glnod->objstatic == OPTIMIZED) continue;
      switch (glnod->status) {
/* The use of upkey indecates that redrawing is necessary only from
   the level and up. Lowkey means that all levels must be redrawn */

      case STALESBAPP:
	if (glnod->visible) {
	  copyStatPic (gp, gp->curlunit->picstatic, glnod->boundingbox);
	  insRect (gp, &levrects, --lowkey, &glnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &glnod->boundingbox);
	}
      case VALID:
	break;
	
      case STALESBGEOM:
	if (glnod->visible) {
	  copyStatPic (gp, gp->curlunit->picstatic, glnod->boundingbox);
	  insRect (gp, &levrects, --lowkey, &glnod->boundingbox);
	  unionUpdateArea (gp,&gp->dynupdate, &glnod->boundingbox);
	}
      case STALEGEOM:
	getGraphBB (gp, glnod);
	glnod->visible = bbOverlap (&defrect, &glnod->boundingbox);
      case STALEAPP:
	if (glnod->visible) {
	  insRect (gp, &levrects, upkey++, &glnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &glnod->boundingbox);
	}
	break;
	
      case STALESBTEXT:
	if (glnod->visible) {
	  copyStatPic (gp, gp->curlunit->picstatic, glnod->boundingbox);
	  insRect (gp, &levrects, --lowkey, &glnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &glnod->boundingbox);
	}
      case STALETEXT:
	recalcText (gp, glnod);
	getGraphBB (gp, glnod);
	if ((glnod->visible = bbOverlap (&defrect, &glnod->boundingbox))) {
	  insRect (gp, &levrects, upkey++, &glnod->boundingbox);
	  unionUpdateArea (gp, &gp->dynupdate, &glnod->boundingbox);
	}
	break;
      default:
	sprintf(mess, ErrUNKNSTAT, glnod->status);
	errorMsg(gp, 2, mess);
	c_exit (gp, NOT_OK);
      }
    }
  }

/* Now, we have a list of rectangles describing the boundingboxes for
   the objects which were changed by the attrlist input-parameter,
   and their relative stacking (key) */

/* Traverse the list again (the list was not altered in any way above)
   to find out which objects are to be redrawn */

  upkey = 0;
  for (lnod1 = (struct levlist FAR *)first(gp->curlunit->level0);
       lnod1;
       lnod1 = (struct levlist FAR *)next(lnod1)) {

    for (wlnod = lnod1->levwins;
	 wlnod;
	 wlnod = (struct winobj FAR *)next(wlnod)) {

      if (wlnod->objstatic == OPTIMIZED) continue;
      switch (wlnod->status) {
      case STALEAPP:
      case STALEGEOM:
      case STALESBGEOM:
      case STALETEXT:
      case STALESBTEXT:
	upkey++;
      case STALESBAPP:
	if (bbOverlap (&defrect, &wlnod->boundingbox))
	  drawWin (gp, wlnod, NULL,
		   isectRects (gp, &defrect, &wlnod->boundingbox));
	wlnod->status = VALID;
	break;
      case VALID:
/* Determine whether the object must be redrawn, i.e. whether the object is
   inside one of the rectangles that must be redrawn */
	for (rnod = levrects;
	     rnod && upkey >= rnod->level;
	     rnod = (struct rectlist FAR *)next(rnod))
	  if (bbOverlap (&rnod->rectangle, &wlnod->boundingbox)) {
	    drawWin (gp, wlnod, NULL,
		     isectRects (gp, &rnod->rectangle, &wlnod->boundingbox));
	  }
	break;
      default:
	sprintf(mess, ErrUNKNSTAT, wlnod->status);
	errorMsg(gp, 2, mess);
	c_exit (gp, NOT_OK);
      }
    }

/* Same code as above, for graphic objects */
    for (glnod = lnod1->levgraphs;
	 glnod;
	 glnod = (struct graphobj FAR *)next(glnod)) {
      if (glnod->objstatic == OPTIMIZED) continue;
      switch (glnod->status) {
      case STALEAPP:
      case STALEGEOM:
      case STALESBGEOM:
      case STALETEXT:
      case STALESBTEXT:
	upkey++;
      case STALESBAPP:
	if (bbOverlap (&defrect, &glnod->boundingbox)) {
	  drawGraph (gp, glnod, NULL,
		     isectRects (gp, &defrect, &glnod->boundingbox),
		     glnod->parentwin ? glnod->parentwin->boundingbox.x : 0,
		     glnod->parentwin ? glnod->parentwin->boundingbox.y : 0);
	}
	glnod->status = VALID;
	break;
      case VALID:
/* Determine whether the object must be redrawn, i.e. whether the object is
   inside one of the rectangles that must be redrawn */
	for (rnod = levrects;
	     rnod && upkey >= rnod->level;
	     rnod = (struct rectlist FAR *)next(rnod))
	  if (bbOverlap (&rnod->rectangle, &glnod->boundingbox)) {
	    drawGraph (gp, glnod, NULL,
		       isectRects (gp, &rnod->rectangle, &glnod->boundingbox),
		       glnod->parentwin ? glnod->parentwin->boundingbox.x : 0,
		       glnod->parentwin ? glnod->parentwin->boundingbox.y : 0);
	  }
	break;
      default:
	sprintf(mess, ErrUNKNSTAT, glnod->status);
	errorMsg(gp, 2, mess);
       }
    }
  }
  staleArea = queryUpdateArea (gp, &gp->dynupdate);
  if (staleArea.width > 0 && staleArea.height > 0)
    fakeExpose (gp, staleArea);
}

void UpdateInpAttr(struct cw_status *gp, struct attribute FAR *attr)
{
  struct oper FAR * FAR *op = attr->exprval.op;
  struct levlist FAR *lnod1;
  struct inpobj FAR *ilnod;
  int ptcount, i;
  long *cval;
  char errormsg[256];

  assert (attr);

  ptcount = attr->parent.typinp->ptcount;

  switch (attr->attrtype) {
  case S_POINTS:
    cval = attr->curval.ptarr;
    for(i = 0; i < ptcount; i++){
      cval[2*i] = calculate (gp, op[2*i]);
      cval[2*i+1] = calculate (gp, op[2*i+1]);
    }
#if (defined WIN31 || defined WIN32)
    if(attr->parent.typinp->boundingregion != (HRGN) 0)
#else
    if(attr->parent.typinp->boundingregion != NULL)
#endif
      DestroyRegion(attr->parent.typinp->boundingregion);
    if(ptcount == 2) 
      attr->parent.typinp->boundingregion =  
 	BoxRegion(attr->curval.ptarr); 
    else 
      attr->parent.typinp->boundingregion =  
 	PolygonRegion(attr->curval.ptarr, attr->parent.typinp->ptcount);
     break; 
  case S_ACTIVE:
    if(attr->exprval.oper != NULL)
      attr->curval.typint = calculate(gp, attr->exprval.oper);
    attr->directval.typint = attr->curval.typint;
    break;
  case S_LEVEL:
/* Should not be called unless the level-value really has changed */
    if(attr->exprval.oper != NULL)
      attr->curval.typint = calculate(gp, attr->exprval.oper);
    for (lnod1 = (struct levlist FAR *)first(gp->curlunit->level0);
	 lnod1;
	 lnod1 = (struct levlist FAR *)next(lnod1))
      for (ilnod = lnod1->levinps;
	   ilnod;
	   ilnod = (struct inpobj FAR *)next(ilnod)) 
	if (ilnod == attr->parent.typinp) {
	  if (is_first(ilnod))
	    lnod1->levinps = (struct inpobj FAR *)next(ilnod);
	  remove_at(ilnod);
	  if (!lnod1->ownsmask
	      && !lnod1->levwins && !lnod1->levgraphs && !lnod1->levinps
	      && lnod1->level != 0) {
	    remove_at(lnod1);
	    killLevNode (gp, lnod1);
	  }
/* Break all for-loops */
	  goto L_INP_outOfList;
	}
L_INP_outOfList:
    attr->directval.typlev =
      getLevNode (gp, &gp->curlunit->level0, attr->curval.typint);
    if (!attr->directval.typlev->maskstatic
	&& !is_last(attr->directval.typlev)
	&& ((struct levlist FAR *)
	    next(attr->directval.typlev))->maskstatic)
      attr->directval.typlev->maskstatic =
	((struct levlist FAR *)next(attr->directval.typlev))->maskstatic;
	
    if (attr->directval.typlev->levinps)
      place_prev(attr->directval.typlev->levinps, attr->parent.typinp);
    else
      init_list(attr->parent.typinp);
    attr->directval.typlev->levinps = attr->parent.typinp;
    break;
  case S_TRANSLATION:
    /* do nothing */
    break;
  default:
    sprintf(errormsg, ErrUNKNATTR, attr->attrtype);
    errorMsg(gp, 1, errormsg);
  }
}

void UpdateInps(struct cw_status *gp)
{
  struct attribute FAR *attr, FAR *attr_next;
  for(attr = gp->curlunit->fattr; attr != NULL; attr = next(attr)){
    if(attr->parenttype == INP && attr->parent.obj->live){
      UpdateInpAttr(gp, attr);
    }
  }
  for(attr = gp->curlunit->nattr; attr != NULL;){
    if(attr->parenttype == INP){
      UpdateInpAttr(gp, attr);
      attr_next = next(attr);
      if(attr == gp->curlunit->nattr)
	gp->curlunit->nattr = attr_next;
      if(attr == gp->curlunit->lnattr)
	gp->curlunit->lnattr = prev(attr);
      remove_at(attr);
      attr = attr_next;
    } else 
      attr = next(attr);
  }
}





