/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>

#include "const.h"
#include "parser.h"
#include "lex.h"
#include "graphic.h"
#include "input.h"
#include "fast_lis.h"
#include "attr.h"
#include "nodes.h"
#include "candle.h"
#include "error.h"

#include "protos/memory.h"
#include "sysproto.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/instance.h"
#include "protos/objects.h"
#include "protos/freesim.h"
#include "protos/parser.h"

/*
* These are the functions for building the window-, graphic- and input-
* objects:
*
* object=newXXXObj() returns a pointer to a new object-struct,
* or NULL if memory is full.
*
* status=newXXXAttr(object, attribute_type, expression) allocates and
* ties a new attribute of type attribute_type and value according to
* expression to the given object. The object-parameter should be one
* earlier returned by newXXXObj(). If a fatal error occurs, the function
* gives an errormessage and returns FALSE, otherwise it returns TRUE.
* This routines handles also multiple calls with the same attribute-type.
*
* killXXXObj(object) terminates(i.e frees) the object and all its attributes.
* The object-parameter should be one earlier returned by newXXXObj().
*/

void killPoints(struct cw_status *gp, struct ptoperlist FAR *pto1)
{
  struct ptoperlist FAR *pto2;
  
  while (pto1) {
    pto2 = (struct ptoperlist FAR *)next(pto1);
    remove_at(pto1);
    f_oper (gp, pto1->x);
    f_oper (gp, pto1->y);
    CalFree (pto1);
    pto1 = pto2;
  }
}

void f_events(struct cw_status *gp, struct event *ep)
{
  struct event *e;
  while(ep){
    e = next(ep);
    remove_at(ep);
    CalFree(ep);
    ep = e;
  }
}

void f_translations(struct cw_status *gp, struct translation *tr, int is_proto)
{
  struct translation *t;
  while(tr != NULL){
    if(is_proto)
      f_events(gp, tr->event);
    f_simulate(gp, tr->act, is_proto);
    t = next(tr);
    remove_at(tr);
    CalFree(tr);
    tr = t;
  }
}

void killAttrib (struct cw_status *gp, struct attribute FAR *attr)
{
  struct volist FAR *vol, FAR *volp;
  struct outlist *ol;
  struct txtlist FAR *tl;
  int i;

  if (attr == NULL) return;

  switch(attr->attrtype){
  case S_POINTS:
    if(attr->parent.obj->is_proto && !attr->indirect){
	killPoints(gp, attr->exprval.pl);
    } else if(!attr->indirect){
      if(attr->directval.ptarr)
	CalFree(attr->directval.ptarr);
      if(attr->curval.ptarr)
	CalFree(attr->curval.ptarr);
      for(i = 0; i < attr->parent.obj->ptcount;i++){
	f_oper(gp, attr->exprval.op[2*i]);
	f_oper(gp, attr->exprval.op[2*i+1]);
      }
      if(attr->exprval.op)
	CalFree(attr->exprval.op);
    }
    break;
  case S_OUTTEXT:
    if (attr->curval.typtxt) 
      CalFree (attr->curval.typtxt);
  case S_OUTINT:
  case S_OUTFLOAT:
    f_oper(gp, attr->exprval.oper);
    /* First free old nodes if any */
    if(attr->directval.typtxtnode != NULL){
      while ((tl = attr->directval.typtxtnode->linlist)) {
	attr->directval.typtxtnode->linlist =
	  (struct txtlist FAR *)next(attr->directval.typtxtnode->linlist);
	remove_at(tl);
	if (tl->line && tl->length > 0) {
	  CalFree (tl->line);
	  tl->line = NULL;
	}
	CalFree (tl);
      }
      CalFree(attr->directval.typtxtnode);
      attr->directval.typtxtnode = NULL;
    }
    break;
  case S_IMAGE:
    if (attr->curval.typtxt) 
      CalFree (attr->curval.typtxt);
    f_oper(gp, attr->exprval.oper);
    break;
  case S_FONT:
    if (attr->directval.typtxtnode) freeFont (gp, attr->directval.typtxtnode);
    if (attr->curval.typtxt) CalFree (attr->curval.typtxt);
    f_oper(gp, attr->exprval.oper);
    break;
  case S_TRANSLATION:
    if(!attr->indirect)
      f_translations(gp, attr->exprval.tl, attr->parent.obj->is_proto);
    break;
  default:
    f_oper(gp, attr->exprval.oper);
    break;
  }
  
  if(is_list(attr)){
    if(attr == gp->curlunit->fattr)
      gp->curlunit->fattr = next(attr);
    else if(attr == gp->curlunit->nattr)
      gp->curlunit->nattr = next(attr);
    else if(attr == gp->curlunit->lnattr)
      gp->curlunit->lnattr = prev(attr);
    remove_at(attr);
  }
  for(vol = attr->vinstlist;vol != NULL;){
    ol = vol->ol;
    if(is_first(ol))
      vol->vi->out.out = next(ol);
    remove_at(ol);
    CalFree(ol);
    volp = vol;
    vol = next(vol);
    remove_at(volp);
    CalFree(volp);
  }
  CalFree(attr);
}

void assignAttrib (struct cw_status *gp, struct attribute FAR *FAR *attr,
		   struct A_val FAR *expr, struct commonobj FAR *parent)
{
  if (*attr == NULL) {
    *attr=(struct attribute FAR *)CalCalloc (1, sizeof (struct attribute));
    if (*attr == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      exit(1);
    }
  }
  else errorMsg(gp, 1, ErrATTRTWICE);
  (*attr)->attrtype = expr->type;
  switch(expr->type){
  case S_POINTS:
    if(expr->indirect){
      (*attr)->indirect = TRUE;
      (*attr)->exprval.ap = expr->val.ap;
    } else {
      (*attr)->exprval.pl = expr->val.pl;
      parent->ptcount = expr->npoints;
    }
    break;
  case S_TRANSLATION:
    if(expr->indirect){
      (*attr)->indirect = TRUE;
      (*attr)->exprval.ap = expr->val.ap;
    } else
      (*attr)->exprval.tl = expr->val.tl;
    break;
  default:
    (*attr)->exprval.oper = expr->val.oper;
  }
}

struct winobj *newWinObj (struct cw_status *gp)
{
  struct winobj FAR *newobject;

  if (!(newobject = (struct winobj FAR *)CalCalloc (1, sizeof (struct winobj)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  newobject->is_proto = TRUE;
  newobject->visible = FALSE;
  newobject->status = VALID;
  newobject->type = S_WINDOW;

/*   init_list (newobject->level0 = newLevNode (0)); */

  return newobject;
}

void newWinAttr (struct cw_status *gp, struct commonobj FAR *obj,
		 struct A_val FAR *woaexpr)
{
  struct winobj *object = (struct winobj *) obj;
  switch (woaexpr->type) {
  case S_POINTS:
    assignAttrib (gp, &object->pointlist, woaexpr,
		  (struct commonobj FAR *) object);
    object->pointlist->parenttype = WIN;
    object->pointlist->parent.typwin = object;
    object->pointlist->curval.ptarr =
      object->pointlist->directval.ptarr = NULL;
    break;
  case S_ACTIVE:
    assignAttrib (gp, &object->active, woaexpr, (struct commonobj *)object);
    object->active->parenttype = WIN;
    object->active->parent.typwin = object;
    object->active->curval.typint =
      object->active->directval.typint = DEF_ACTIVE;
    break;
  case S_SAVEBG:
    assignAttrib (gp, &object->saveback, woaexpr, (struct commonobj *)object);
    object->saveback->parenttype = WIN;
    object->saveback->parent.typwin = object;
    object->saveback->curval.typint =
      object->saveback->directval.typint = DEF_SAVEBACKGROUND;
    break;
  case S_LEVEL:
    assignAttrib (gp, &object->level, woaexpr, (struct commonobj *)object);
    object->level->parenttype = WIN;
    object->level->parent.typwin = object;
    object->level->curval.typint = DEF_LEVEL;
    object->level->directval.typlev = NULL;
    break;
  default:
    ;/* errorMsg(1, UNKNATTR, woaexpr->type); */
  }
}


void killWinObj (struct cw_status *gp, struct winobj FAR *object)
{
  struct levlist FAR *ll, *ll1, *ll2;

/* Remove from levlist */
  if(is_list(object)){
    if (!object->is_proto && is_first(object)) {
/* Parent must be found first */
      for (ll = (struct levlist FAR *)(gp->curlunit->level0 ?
				       first(gp->curlunit->level0)
				       : NULL);
	   ll && ll->levwins != object;
	   ll = (struct levlist FAR *)next(ll));
      if (ll) {
	ll->levwins = (struct winobj FAR *)next(ll->levwins);
	if (!ll->ownsmask
	    && !ll->levwins && !ll->levgraphs && !ll->levinps
	    && ll->level != 0) {
	  remove_at(ll);
	  killLevNode (gp, ll);
	}
      }
    }
    remove_at(object);
  }  

  /* Free window's local levellist */
  if (object->level0) {
    ll1 = (struct levlist *)first(object->level0);
    while ((ll2 = ll1)) {
      ll1 = (struct levlist FAR *)next(ll2);
      remove_at(ll2);
      killLevNode (gp, ll2);
      /* Kill levnode including all its window's levnodes */
    }
    object->level0 = NULL;
  }

  killAttrib (gp, object->pointlist);
  killAttrib (gp, object->active); killAttrib (gp, object->saveback);
  killAttrib (gp, object->level);

  f_graphobj(gp, object->graphs);
  f_inpobj(gp, object->inps);

  CalFree (object);
}

struct graphobj FAR *newGraphObj(struct cw_status *gp, int gotype,
				 struct winobj FAR *parentwin)
{
  struct graphobj FAR *newobject;

  if (!(newobject =
	(struct graphobj FAR *)CalCalloc (1, sizeof (struct graphobj)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  newobject->objstatic = OPTIMIZABLE; /* Object is static until one or more */
				      /* non-static attributes are found */

  newobject->type = gotype;
  newobject->visible = FALSE;
  newobject->status = VALID;
  newobject->parentwin = parentwin;
  newobject->is_proto = TRUE;

  return newobject;
}

void newGraphAttr (struct cw_status *gp, struct commonobj FAR *obj,
		   struct A_val FAR *goaexpr)
{
  struct graphobj *object = (struct graphobj *) obj;
/* Allocate attribute, initialize it and set value to default */
  switch (goaexpr->type) {
  case S_POINTS:
    assignAttrib (gp, &object->pointlist, goaexpr, (struct commonobj *)object);
    object->pointlist->parenttype = GRAPH;
    object->pointlist->parent.typgraph = object;
    object->pointlist->curval.ptarr =
      object->pointlist->directval.ptarr = NULL;
    break;
  case S_COLOR:
    if (object->type != S_IMAGE) {
      assignAttrib (gp, &object->color, goaexpr, (struct commonobj *)object);
      object->color->parenttype = GRAPH;
      object->color->parent.typgraph = object;
      object->color->curval.typint = 0;
    }
    break;
  case S_IMAGE:
    if (object->type != S_LINE) {
      assignAttrib (gp, &object->image, goaexpr, (struct commonobj *)object);
      object->image->parenttype = GRAPH;
      object->image->parent.typgraph = object;
      object->image->curval.typtxt = strdup("");
      object->image->directval.typimage = &gp->imgcache->img;
    }
    break;
  case S_ACTIVE:
    assignAttrib (gp, &object->active, goaexpr, (struct commonobj *)object);
    object->active->parenttype = GRAPH;
    object->active->parent.typgraph = object;
    object->active->curval.typint =
      object->active->directval.typint = DEF_ACTIVE;
    break;
  case S_SAVEBG:
    assignAttrib (gp, &object->saveback, goaexpr, (struct commonobj *)object);
    object->saveback->parenttype = GRAPH;
    object->saveback->parent.typgraph = object;
    object->saveback->curval.typint =
      object->saveback->directval.typint = DEF_SAVEBACKGROUND;
    break;
  case S_LEVEL:
    assignAttrib (gp, &object->level, goaexpr, (struct commonobj FAR *)object);
    object->level->parenttype = GRAPH;
    object->level->parent.typgraph = object;
    object->level->curval.typint = DEF_LEVEL;
    object->level->directval.typlev = NULL;
    break;
  case S_FILL:
    if (object->type == S_BOX
	|| object->type == S_POLYGON
	|| object->type == S_ARC) {
      assignAttrib(gp, &object->fill, goaexpr, (struct commonobj *)object);
      object->fill->parenttype = GRAPH;
      object->fill->parent.typgraph = object;
      object->fill->curval.typint =
	object->fill->directval.typint = DEF_FILL;
    }
/*    else errorMsg(0, WarnINVALFILL); */
    break;
  case S_TEXTURE:
    assignAttrib (gp, &object->texture, goaexpr, (struct commonobj *)object);
    object->texture->parenttype = GRAPH;
    object->texture->parent.typgraph = object;
    object->texture->curval.typint = DEF_TEXTURE;
    object->texture->directval.typimage = NULL;
    break;
  case S_DASHES:
    if (object->type != S_IMAGE
	&& object->type != S_TEXTOBJECT) {
      assignAttrib (gp, &object->dashes, goaexpr, (struct commonobj *) object);
      object->dashes->parenttype = GRAPH;
      object->dashes->parent.typgraph = object;
      object->dashes->curval.typint = DEF_DASHES;
      object->dashes->directval.typdash = NULL;
    }
/*    else errorMsg(0, WarnINVALDSH); */
    break;
  case S_LINEWIDTH:
    if (object->type != S_IMAGE
	&& object->type != S_TEXTOBJECT) {
      assignAttrib (gp, &object->linewidth, goaexpr,
		    (struct commonobj *)object);
      object->linewidth->parenttype = GRAPH;
      object->linewidth->parent.typgraph = object;
      object->linewidth->curval.typint =
	object->linewidth->directval.typint = DEF_LINEWIDTH;
    }
/*    else errorMsg(0, WarnINVALLW); */
    break;
  case S_STARTANGLE:
    if (object->type == S_ARC) {
      assignAttrib (gp, &object->startangle, goaexpr,
		    (struct commonobj *)object);
      object->startangle->parenttype = GRAPH;
      object->startangle->parent.typgraph = object;
      object->startangle->curval.typint =
	object->startangle->directval.typint = DEF_STARTANGLE;
    }
/*    else errorMsg(0, WarnINVALSA); */
    break;
  case S_ENDANGLE:
    if (object->type == S_ARC) {
      assignAttrib (gp, &object->endangle, goaexpr,
		    (struct commonobj *) object);
      object->endangle->parenttype = GRAPH;
      object->endangle->parent.typgraph = object;
      object->endangle->curval.typint =
	object->endangle->directval.typint = DEF_ENDANGLE;
    }
/*    else errorMsg(0, WarnINVALEA); */
    break;
  case S_FONT:
    if (object->type == S_TEXTOBJECT) {
      assignAttrib (gp, &object->font, goaexpr, (struct commonobj *)object);
      object->font->parenttype = GRAPH;
      object->font->parent.typgraph = object;
      object->font->curval.typtxt = strdup("");
      object->font->directval.typtxtnode = NULL;
    }
/*    else errorMsg(0, WarnINVALFNT); */
    break;
  case S_OUTTEXT:
    if (object->type == S_TEXTOBJECT) {
      if (!object->outint && !object->outfloat) {
	assignAttrib (gp, &object->outtext, goaexpr,
		      (struct commonobj *)object);
	object->outtext->parenttype = GRAPH;
	object->outtext->parent.typgraph = object;
	object->outtext->curval.typtxt = strdup(DEF_OUTTEXT);
	object->outtext->directval.typtxtnode = NULL;
      } else errorMsg(gp, 0, WarnOTXTTWICE);
    }
/*    else errorMsg(0, WarnINVALOTX); */
    break;
  case S_OUTINT:
    if (object->type == S_TEXTOBJECT) {
      if (!object->outtext && !object->outfloat) {
	assignAttrib (gp, &object->outint, goaexpr,
		      (struct commonobj *)object);
	object->outint->parenttype = GRAPH;
	object->outint->parent.typgraph = object;
	object->outint->curval.typint = DEF_OUTINT;
	object->outint->directval.typtxtnode = NULL;
      } else errorMsg(gp, 0, WarnOINTTWICE);
    } else errorMsg(gp, 0, WarnINVALONT);
    break;
  case S_OUTFLOAT:
    if (object->type == S_TEXTOBJECT) {
      if (!object->outtext && !object->outfloat) { 
	assignAttrib (gp, &object->outfloat, goaexpr,
		      (struct commonobj *)object);
	object->outfloat->parenttype = GRAPH;
	object->outfloat->parent.typgraph = object;
	object->outfloat->curval.typflt = (float) DEF_OUTFLOAT;
	object->outfloat->directval.typtxtnode = NULL;
      } else errorMsg(gp, 0, WarnOFLTTWICE);
    } else errorMsg(gp, 0, WarnINVALOLT);
    break;
  case S_DECIMALS:
    if (object->type == S_TEXTOBJECT) {
      assignAttrib (gp, &object->decimals, goaexpr,
		    (struct commonobj *)object);
      object->decimals->parenttype = GRAPH;
      object->decimals->parent.typgraph = object;
      object->decimals->curval.typint =
	object->decimals->directval.typint = DEF_DECIMALS;
    } else errorMsg(gp, 0, WarnINVALDEC);
    break;
  default:
    ;/* errorMsg(1, UNKNATTR, goaexpr->type); */
  }
}

void killGraphObj (struct cw_status *gp, struct graphobj FAR *object)
{
  struct levlist FAR *ll;
  struct txtlist FAR *tl;

/* Remove from levlist */
  if(is_list(object)){
    if (!object->is_proto && is_first(object)) {
/* Parent must be found first */
      for (ll = (struct levlist FAR *)
	     (object->parentwin ?
	      object->parentwin->level0 ?
	      first(object->parentwin->level0) : NULL
	      : gp->curlunit->level0 ?
	      first(gp->curlunit->level0) : NULL);
	   ll && ll->levgraphs != object;
	   ll = (struct levlist FAR *)next(ll));
      if (ll) {
	ll->levgraphs = (struct graphobj FAR *)next(ll->levgraphs);
	if (!ll->ownsmask
	    && !ll->levwins && !ll->levgraphs && !ll->levinps
	    && ll->level != 0) {
	  remove_at(ll);
	  killLevNode (gp, ll);
	}
      }
    }
    remove_at(object);
  }

  if (!object->is_proto
      && object->image
      && !object->image->directval.typimage->in_cache)
    killImage (gp, object->image->directval.typimage);

  killAttrib (gp, object->pointlist); killAttrib (gp, object->color);
  killAttrib (gp, object->image); killAttrib (gp, object->active);
  killAttrib (gp, object->saveback); killAttrib (gp, object->level);
  killAttrib (gp, object->fill); killAttrib (gp, object->texture);
  killAttrib (gp, object->dashes); killAttrib (gp, object->linewidth);
  killAttrib (gp, object->startangle); killAttrib (gp, object->endangle);

  if (object->font && object->font->directval.typtxtnode) {
    freeFont (gp, object->font->directval.typtxtnode);
/* First free old nodes if any */
    while ((tl = object->font->directval.typtxtnode->linlist)) {
      object->font->directval.typtxtnode->linlist =
	(struct txtlist FAR *)next(object->font->directval.typtxtnode->linlist);
      remove_at(tl);
      if (tl->line && tl->length > 0) {
	CalFree (tl->line);
	tl->line = NULL;
      }
      CalFree (tl);
    }
  }

  killAttrib (gp, object->font); killAttrib (gp, object->outtext);
  killAttrib (gp, object->outint); killAttrib (gp, object->outfloat);
  killAttrib (gp, object->decimals);

  CalFree (object);
}

struct inpobj FAR *newInpObj (struct cw_status *gp, int iotype,
			      struct winobj FAR *parentwin)
{
  struct inpobj FAR *newobject;

  if (!(newobject =
	(struct inpobj FAR *)CalCalloc (1, sizeof (struct inpobj)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  newobject->is_proto = TRUE;
  newobject->parentwin = parentwin;
  newobject->type = iotype;
  return newobject;
}

void newInpAttr (struct cw_status *gp, struct commonobj FAR *obj,
		 struct A_val FAR *ioaexpr)
{
  struct inpobj *object = (struct inpobj *) obj;
  switch (ioaexpr->type) {
  case S_POINTS:
    assignAttrib (gp, &object->pointlist, ioaexpr, 
		  (struct commonobj FAR *) object);
    object->pointlist->parenttype = INP;
    object->pointlist->parent.typinp = object;
    object->pointlist->curval.ptarr =
      object->pointlist->directval.ptarr = NULL;
    break;
  case S_LEVEL:
    assignAttrib (gp, &object->level, ioaexpr, (struct commonobj *)object);
    object->level->parenttype = INP;
    object->level->parent.typinp = object;
    object->level->curval.typint = DEF_LEVEL;
    object->level->directval.typlev = NULL;
    break;
  case S_ACTIVE:
    assignAttrib (gp, &object->active, ioaexpr, (struct commonobj *)object);
    object->active->parenttype = INP;
    object->active->parent.typinp = object;
    object->active->curval.typint =
      object->active->directval.typint = DEF_ACTIVE;
    break;
  case S_TRANSLATION:
    assignAttrib(gp, &object->translation, ioaexpr,
		 (struct commonobj *)object);
    object->translation->parenttype = INP;
    object->translation->parent.typinp = object;
    break;
  default:
    parse_error(gp, "Warning: Illegal attribute for inputobject");
  }
}

void killInpObj (struct cw_status *gp, struct inpobj FAR *object)
{
  struct levlist FAR *ll;

/* Remove from levlist */
  if (is_list(object)){
    if(!object->is_proto && is_first(object)) {
/* Parent must be found first */
      for (ll = (struct levlist FAR *)
	     (object->parentwin ?
	      object->parentwin->level0 ? first(object->parentwin->level0)
	      : NULL
	      : gp->curlunit->level0 ? first(gp->curlunit->level0)
	      : NULL);
	   ll && ll->levinps != object;
	   ll = (struct levlist FAR *)next(ll));
      if (ll) {
	ll->levinps = (struct inpobj FAR *)next(ll->levinps);
	if (!ll->ownsmask
	    && !ll->levwins && !ll->levgraphs && !ll->levinps
	    && ll->level != 0) {
	  remove_at(ll);
	  killLevNode (gp, ll);
	}
      }
    }
    remove_at(object);
  }


  killAttrib (gp, object->pointlist);
  killAttrib (gp, object->level); 
  killAttrib (gp, object->active);
  killAttrib (gp, object->translation);

  CalFree (object);
}








