/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#ifndef MAXINT
#define MAXINT 32767
#else
#include <values.h>
#endif

#include "candle.h"
#include "nodes.h"
#include "learnuni.h"
#include "graphic.h"
#include "attr.h"
#include "lex.h"
#include "error.h"
#include "input.h"
#include "simulate.h"
#include "pbmplus.h"        /* For min/max-macros */

#include "protos/memory.h"

#include "parser.h"
#include "sysproto.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/freesim.h"
#include "protos/objects.h"
#include "protos/update.h"
#include "protos/instance.h"
#include "protos/creatsim.h"
#include "protos/parser.h"

/*
 * This file holds some routines used when new learnunits and functions are
 * started and when old ones are terminated
 */

void newLearnUnit (struct cw_status *gp, int width, int height, int bgcolor)
{
  struct learnunit FAR *learnnode;

  if (gp->rectmaxcount != RECTPOWER) {
    CalFree (gp->rectspace);
    if ((gp->rectspace = 
	 CalMalloc ((gp->rectmaxcount = RECTPOWER)
		    *sizeof (struct rectlist))) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }
  } 

  if (!(learnnode = 
	(struct learnunit FAR *)CalCalloc (1, sizeof (struct learnunit)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  
  if (!gp->parseOnly) {
    learnnode->win = blankImage (gp, width, height, bgcolor);
    learnnode->picstatic = blankImage (gp, width, height, bgcolor);
  }
  learnnode->defX = learnnode->defY = 8;
  learnnode->defCol = 0x888888/*defColor (gp, bgcolor)*/;
  init_list(learnnode->level0 = newLevNode (gp, 0));
  if (gp->curlunit) {
    place_prev(gp->curlunit, learnnode);
    gp->curlunit = learnnode;
  }
  else init_list(gp->curlunit = learnnode);
  gp->exposed = 0;
}

void killLearnUnit (struct cw_status *gp)
{
  struct learnunit FAR *learnnode = gp->curlunit;
  struct levlist FAR *ll1 = (struct levlist *)first(learnnode->level0),
    FAR *ll2;
  struct imgnode FAR *in1 = (struct imgnode *)next(gp->imgcache),
    FAR *in2;

  while ((ll2 = ll1)) {
    ll1 = (struct levlist FAR *)next(ll2);
    remove_at(ll2);
    killLevNode (gp, ll2);
    /* Kill levnode including all its window's levnodes */
  }
  learnnode->level0 = NULL;

  while ((in2 = in1)) {
    in1 = (struct imgnode FAR *)next(in2);
    if (in2->lunit == learnnode) freeImage (gp, in2);
  }
  if(learnnode->filename != NULL){
    CalFree(learnnode->filename);
    learnnode->filename = NULL;
  }
  freesim(gp, learnnode);
  if (learnnode->picstatic) killImage (gp, learnnode->picstatic);
  killImage (gp, learnnode->win);
  gp->curlunit = (struct learnunit FAR *)next(gp->curlunit);
  remove_at(learnnode);
  CalFree(learnnode);
  if (gp->curlunit && !(gp->command & QUIT_CALUNIT))
    resizeMainWin (gp, gp->curlunit->win->width,
		   gp->curlunit->win->height);
}

struct levlist FAR *newLevNode (struct cw_status *gp, int key)
{
  struct levlist FAR *lnod;

  if (!(lnod = (struct levlist FAR *)CalCalloc (1, sizeof (struct levlist)))) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  lnod->level = key;
  lnod->ownsmask = FALSE;

  return lnod;
}


/*
 * Return a pointer to the levelnode at the given level
 */

struct levlist FAR *getLevNode (struct cw_status *gp,
				struct levlist **levlist, int key)
{
  struct levlist FAR *lnod1, FAR *lnod2;

  if (!(*levlist)) {
    init_list((*levlist) = newLevNode (gp, key));
    return (*levlist);
  }

  for (lnod1 = (struct levlist FAR *)first(*levlist);
       lnod1;
       lnod1 = (struct levlist FAR *)next(lnod1)) {

    if (lnod1->level == key) return lnod1;
    if (lnod1->level > key) {
      lnod2 = newLevNode (gp, key);
/* This is the only case where new levnodes are added in front of old,
 make sure maskstatic points right */
      lnod2->maskstatic = lnod1->maskstatic;
      place_prev(lnod1, lnod2);
      return lnod2;
    }
  }
/* This key must be higher than any other key in the level-list */
  lnod2 = newLevNode (gp, key);
  place_next(last(*levlist), lnod2);

  return lnod2;
}

/*
 * Free levelnode and all memory it occupies
 */

void killLevNode (struct cw_status *gp, struct levlist FAR *lnod)
{
  struct winobj FAR *wnod;
  struct levlist FAR *wll1, FAR *wll2;

  if (lnod->ownsmask) killImage (gp, lnod->maskstatic);
  for (wnod = lnod->levwins;
       wnod;
       wnod = (struct winobj FAR *)next(wnod)) {
    wll1 = (struct levlist FAR *)first(wnod->level0);
    while ((wll2 = wll1)) {
      wll1 = (struct levlist FAR *)next(wll2);
      remove_at(wll2);
      killLevNode (gp, wll2);
    }
    wnod->level0 = NULL;
  }
  CalFree (lnod);
}

static char winOptimize (struct cw_status *gp, struct winobj *wo,
			 struct blockinst *bi)
{
  if (wo->objstatic == UNOPTIMIZABLE) return FALSE;
  while (bi != wo->parentbi) {
    assert (bi->parent);
    if (bi->parent->type == FUNCTIONBLOCK
	&& ((struct funcinst *)bi)->optimized > 0) {
      wo->objstatic = OPTIMIZED;
      return TRUE;
    }
    bi = bi->caller;
  }
  if (((struct funcinst *)bi)->optimized > 0) {
    wo->objstatic = OPTIMIZED;
    return TRUE;
  }
  return FALSE;
}

static char graphOptimize (struct cw_status *gp, struct graphobj *go,
			   struct blockinst *bi)
{
  if (go->objstatic == UNOPTIMIZABLE) return FALSE;
  while (bi != go->parentbi) {
    assert (bi->parent);
    if (bi->parent->type == FUNCTIONBLOCK
	&& ((struct funcinst *)bi)->optimized > 0) {
      go->objstatic = OPTIMIZED;
      return TRUE;
    }
    bi = bi->caller;
  }
  if (((struct funcinst *)bi)->optimized > 0) {
    go->objstatic = OPTIMIZED;
    return TRUE;
  }
  return FALSE;
}


void statobjSetup (struct cw_status *gp, struct blockinst *bi)
{
  struct levlist FAR *ll;
  struct winobj FAR *wl;
  struct graphobj FAR *gl;
  CalRect defrect = screenRect (gp);

/* Now fix all the static graphic objects (including windows) */
  for (ll = (struct levlist FAR *)last(gp->curlunit->level0);
       ll;
       ll = (struct levlist FAR *)prev(ll)) {

    for (wl = ll->levwins; wl; wl = (struct winobj FAR *)next(wl)) {
      if (winOptimize (gp, wl, bi)) {
	getWinBB (gp, wl);
	drawWin (gp, wl, ll, isectRects (gp, &defrect, &wl->boundingbox));
      }
      else wl->status = max(wl->status, STALESBAPP);
    }

    for (gl = ll->levgraphs; gl; gl = (struct graphobj FAR *)next(gl)) {
      if (graphOptimize (gp, gl, bi)) {
	getGraphBB (gp, gl);
	drawGraph (gp, gl, ll,
		   isectRects (gp, &defrect, &gl->boundingbox), 0, 0);
      }
      else gl->status = max(gl->status, STALESBAPP);
    }

/* If there is no static mask on the level, make sure the above is used
   if it exists (i.e. level 0 must use level 2's even if there is none
   at level 1) */
    if (!ll->maskstatic
	&& !is_last(ll)
	&& ((struct levlist FAR *)next(ll))->maskstatic)
      ll->maskstatic = ((struct levlist FAR *)next(ll))->maskstatic;
  }
  copyStatPic (gp, gp->curlunit->picstatic, screenRect (gp));
  fakeExpose (gp, screenRect (gp));
}

/*
 * Return the maximum number of points allowed for 
 * objects of this type.
 */
int maxPoints(struct cw_status *gp, struct commonobj FAR *obj)
{
  switch(obj->type){
  case S_POLYGON:
    return MAXINT;
  default:
    return 2;
  }
}

/*
 * Return the maximum number of points allowed for 
 * objects of this type.
 */
int minPoints(struct cw_status *gp, struct commonobj FAR *obj)
{
  switch(obj->type){
  case S_IMAGE:
  case S_POINT:
  case S_TEXTOBJECT:
    return 1;
  default:
    return 2;
  }
}

static void checkPoints(struct cw_status *gp, struct commonobj FAR *obj)
{
  int ptcount, max, min;
  struct ptoperlist FAR *pol;

  max = maxPoints(gp, obj);
  min = minPoints(gp, obj);
  /* 
   * If the object's essential attribute pointlist is not already created,
   * create it now 
   */
  if (!obj->pointlist) {
    struct A_val expr;
    expr.type = S_POINTS;
    expr.val.pl = NULL;
    expr.indirect = FALSE;
    assignAttrib (gp, &obj->pointlist, &expr, obj);
    obj->pointlist->parenttype = GRAPH;
    obj->pointlist->parent.obj = obj;
    obj->pointlist->curval.ptarr =
      obj->pointlist->directval.ptarr = NULL;
  }

  if(obj->pointlist->indirect) /* We can check the points yet */
    return;

/* Count the points */
  pol = obj->pointlist->exprval.pl;
  ptcount = 0;
  while (pol) {
    pol = (struct ptoperlist FAR *)next(pol);
    ptcount++;
  }

/* Check the points : */
/* Too many points ? */
  if (ptcount > max) {
    parse_error (gp, "Warning: Object contains redundant points. Ignoring");
    obj->ptcount = max;
  } else if (ptcount < min) {
    parse_error (gp, "Warning: Essential attribute(s) missing - too few points.\nUsing default values");
  }
  obj->ptcount = ptcount;
}

void checkWinAttrs (struct cw_status *gp, struct winobj FAR *wnod)
{
  struct ptoperlist FAR *pol;

  checkPoints(gp, (struct commonobj FAR *)wnod);
#if 0  
/* If window's essential attribute pointlist is not already created,
   create it now */
  if (!wnod->pointlist) {
    struct A_val expr;
    expr.type = S_POINTS;
    expr.val.pl = NULL;
    assignAttrib (&wnod->pointlist, &expr, wnod);
    wnod->pointlist->parenttype = WIN;
    wnod->pointlist->parent.typwin = wnod;
    wnod->pointlist->curval.typptlist =
      wnod->pointlist->directval.typptlist = NULL;
  }
/* Count the points and make this many pt curval-nodes */
  pol = wnod->pointlist->exprval.pl;
  ptcount = 0;
  while (pol) {
    if (!(plnod1 =
	  (struct ptvallist FAR *)CalCalloc (1, sizeof (struct ptvallist)))) {
      errorMsg(2, ErrNOMOREMEM);
      c_exit (NOT_OK);
    }
    if (wnod->pointlist->curval.typptlist)
      place_next(wnod->pointlist->curval.typptlist, plnod1);
    else
      init_list(wnod->pointlist->curval.typptlist = plnod1);
    pol = (struct ptoperlist FAR *)next(pol);
    ptcount++;
  }
/* Check the points : */
/* Too many points ? */
  if (ptcount > 2) {
    parse_error("Warning: Object contains redundant points. Ignoring");
  }
/* Too few points ? */
  else if (ptcount < 2) {
    parse_error("Warning: Essential attribute(s) missing - too few points.\nUsing default values");
    if (ptcount == 1)
      place_next(wnod->pointlist->curval.typptlist,
		 graphDefPoint2 (wnod->pointlist->curval.typptlist->x,
				 wnod->pointlist->curval.typptlist->y));
    else {
      init_list(wnod->pointlist->curval.typptlist = graphDefPoint1 ());
      place_next(wnod->pointlist->curval.typptlist,
		  graphDefPoint2 (wnod->pointlist->curval.typptlist->x,
				  wnod->pointlist->curval.typptlist->y));
    }
  }
  wnod->ptcount = 2;
  ptcount = 0;
  for (plnod1 = wnod->pointlist->curval.typptlist;
       ptcount++ < wnod->ptcount;
       plnod1 = (struct ptvallist FAR *)next (plnod1)) {
    if (!(plnod2 =
	  (struct ptvallist FAR *)CalCalloc (1, sizeof (struct ptvallist)))) {
      errorMsg(2, ErrNOMOREMEM);
      c_exit (NOT_OK);
    }
    plnod2->x = plnod1->x; plnod2->y = plnod1->y;
    if (wnod->pointlist->directval.typptlist)
      place_next (last (wnod->pointlist->directval.typptlist), plnod2);
    else
      init_list (wnod->pointlist->directval.typptlist = plnod2);
  }
#endif

/* All objects must have the level-attribute */
  if (!wnod->level) {
    struct A_val expr;
    expr.type = S_LEVEL;
    expr.val.oper = NULL;
    assignAttrib (gp, &wnod->level, &expr, (struct commonobj FAR *)wnod);
    wnod->level->parenttype = WIN;
    wnod->level->parent.typwin = wnod;
    wnod->level->curval.typint = DEF_LEVEL;
    wnod->level->directval.typlev = gp->curlunit->level0;
  }

/* All objects must have the active-attribute (for function-termination) */
  if (!wnod->active) {
    struct A_val expr;
    expr.type = S_ACTIVE;
    expr.val.oper = NULL;
    assignAttrib (gp, &wnod->active, &expr, (struct commonobj FAR *)wnod);
    wnod->active->parenttype = WIN;
    wnod->active->parent.typwin = wnod;
    wnod->active->curval.typint = DEF_ACTIVE;
  }
}


void checkGraphAttrs (struct cw_status *gp, struct graphobj FAR *gnod)
{
  struct ptoperlist FAR *pol;
  int ptcount, ptneed, ptmax;

  checkPoints(gp, (struct commonobj FAR *)gnod);

/* Check the COLOR-attribute */
  if (gnod->type != S_IMAGE) {

/* If essential color-attribute is not already created,
   create it now */
    if (!gnod->color) {
      struct A_val expr;
      expr.type = S_COLOR;
      expr.val.oper = NULL;

      if (!gnod->fill || !gnod->image) 
	  parse_error (gp, "Warning: Essential attribute COLOR missing for the object.\nUsing default color");
      assignAttrib (gp, &gnod->color, &expr, (struct commonobj FAR *)gnod);
      gnod->color->parenttype = GRAPH;
      gnod->color->parent.typgraph = gnod;
      gnod->color->curval.typint = gp->curlunit->defCol;
/* Directval will always be set when instanciating */
    }
  }
/* Check the IMAGE-attribute */
  else {
    if (!gnod->image) {
      struct A_val expr;
      expr.type = S_IMAGE;
      expr.val.oper = NULL;
      
      parse_error (gp, "Warning: Essential attribute IMAGE missing for the object.\nUsing default image");
      assignAttrib (gp, &gnod->image, &expr, (struct commonobj FAR *)gnod);
      gnod->image->parenttype = GRAPH;
      gnod->image->parent.typgraph = gnod;
      gnod->image->curval.typtxt = NULL;
    }
  }
/* Check TEXT-object */
  if (gnod->type == S_TEXTOBJECT
      && !gnod->outtext && !gnod->outint && !gnod->outfloat) {
    struct A_val expr;
    expr.type = S_OUTTEXT;
    expr.val.oper = NULL;
    parse_error (gp, "Warning: Essential attribute OUTTEXT or OUTINT or OUTFLOAT missing.\n Using default text");
    assignAttrib (gp, &gnod->outtext, &expr, (struct commonobj FAR *)gnod);
    gnod->outtext->parenttype = GRAPH;
    gnod->outtext->parent.typgraph = gnod;
    gnod->outtext->curval.typtxt = strdup("0");
  }
/* All objects must have the level-attribute */
  if (!gnod->level) {
    struct A_val expr;
    expr.type = S_LEVEL;
    expr.val.oper = NULL;
    assignAttrib (gp, &gnod->level, &expr, (struct commonobj FAR *)gnod);
    gnod->level->parenttype = GRAPH;
    gnod->level->parent.typgraph = gnod;
    gnod->level->curval.typint = 0;
/*    gnod->level->directval.typlev = gnod->parentwin ?
      gnod->parentwin->level0 : gp->curlunit->level0; */
  }
/* All objects must have the active-attribute (for function termination) */
  if (!gnod->active) {
    struct A_val expr;
    expr.type = S_ACTIVE;
    expr.val.oper = NULL;
    assignAttrib (gp, &gnod->active, &expr, (struct commonobj FAR *)gnod);
    gnod->active->parenttype = GRAPH;
    gnod->active->parent.typgraph = gnod;
    gnod->active->curval.typint = DEF_ACTIVE;
  }
}  

void checkInpAttrs (struct cw_status *gp, struct inpobj FAR *inod)
{
  struct ptoperlist FAR *pol;

  checkPoints(gp, (struct commonobj FAR *)inod);

#if 0
/* If inpobj's essential attribute pointlist is not already created,
   create it now */
  if (!inod->pointlist) {
    struct A_val expr;
    expr.type = S_POINTS;
    expr.val.pl = NULL;
    assignAttrib (gp, &inod->pointlist, &expr, (struct commonobj FAR *)inod);
    inod->pointlist->parenttype = INP;
    inod->pointlist->parent.typinp = inod;
    inod->pointlist->curval.typptlist =
      inod->pointlist->directval.typptlist = NULL;
  }

  pol = inod->pointlist->exprval.pl;
  ptcount = 0;
  while (pol) {
    if (!(plnod1 =
	  (struct ptvallist FAR *)CalCalloc (1, sizeof (struct ptvallist)))) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }
    if (inod->pointlist->curval.typptlist)
      place_next (gp, inod->pointlist->curval.typptlist, plnod1);
    else
      init_list (gp, inod->pointlist->curval.typptlist = plnod1);
    pol = (struct ptoperlist FAR *)next (gp, pol);
    ptcount++;
  }

  if (!inod->pointlist->curval.typptlist) {
    inod->pointlist->curval.typptlist = inpDefPtlist (gp);
    inod->ptcount = 4;           /* Always returns list of 4 points */
  }
  else
    inod->ptcount = ptcount;
  ptcount = 0;
  for (plnod1 = inod->pointlist->curval.typptlist;
       ptcount++ < inod->ptcount;
       plnod1 = (struct ptvallist FAR *)next (gp, plnod1)) {
    if (!(plnod2 =
	  (struct ptvallist FAR *)CalAlloc (1, sizeof (struct ptvallist)))) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }
    plnod2->x = plnod1->x; plnod2->y = plnod1->y;
    if (inod->pointlist->directval.typptlist)
      place_next (last (inod->pointlist->directval.typptlist), plnod2);
    else
      init_list (inod->pointlist->directval.typptlist = plnod2);
  }
#endif

  /* Inputobjects must have the translation attribute */
  if(inod->translation == NULL){
    parse_error(gp, "Error: Missing translation attribute for inputarea object");
  }

/* All objects must have the level-attribute */
  if (!inod->level) {
    struct A_val expr;
    expr.type = S_LEVEL;
    expr.val.oper = NULL;
    assignAttrib (gp, &inod->level, &expr, (struct commonobj FAR *)inod);
    inod->level->parenttype = INP;
    inod->level->parent.typinp = inod;
    inod->level->curval.typint = DEF_LEVEL;
    inod->level->directval.typlev = inod->parentwin ?
      inod->parentwin->level0 : gp->curlunit->level0;
  }

/* All objects must have the active-attribute (for function termination) */
  if (!inod->active) {
    struct A_val expr;
    expr.type = S_ACTIVE;
    expr.val.oper = NULL;
    assignAttrib (gp, &inod->active, &expr, (struct commonobj FAR *)inod);
    inod->active->parenttype = INP;
    inod->active->parent.typinp = inod;
    inod->active->curval.typint = DEF_ACTIVE;
  }
}
