/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no) 
 */

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, David Koblas.                                     | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */

/*
 * Architecture-independent routines for reading GIFs
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>


#include "candle.h"
#include "ppm.h"
#include "gif.h"
#include "error.h"
#include "wwwcom.h" 

#include "sysproto.h"
#include "protos/canutil.h"
#include "protos/gif.h"

int GetDataBlock(struct cw_status *gp, AweStream *fd, unsigned char  FAR *buf);

char FAR * FAR *
pm_allocarray(gp,  cols, rows, size )
     struct cw_status *gp;
     int cols, rows, size;
{
  char FAR * FAR * its;
  int i;
  
  its = (char FAR * FAR *) malloc( rows * sizeof(char FAR *) );
  if ( its == (char FAR * FAR *) 0 )
    errorMsg(gp, 2, ErrNOMOREMEM);
  its[0] = (char FAR *) malloc( rows * cols * size );
  if ( its[0] == (char FAR *) 0 )
    errorMsg(gp, 2, ErrNOMOREMEM);
  for ( i = 1; i < rows; ++i )
    its[i] = &(its[0][i * cols * size]);
  return its;
}

static int ReadColorMap (struct cw_status *gp, AweStream *fd,int number,long *map)
{
  int		i;
  unsigned char	rgb[3];
#ifndef UNIX
  HPALETTE pal;
  pal = gp->system.colormap;
#endif
  for (i = 0; i < number; ++i) {
    if (! ReadOK(fd, rgb, sizeof(rgb)))
      errorMsg(gp, 1, ErrBADCMAP);

#ifdef UNIX
    map[i] = gp->system.depth == 24 ? rgb[0]*256*256+rgb[1]*256+rgb[2]
      : gp->system.depth == 16 ?
      ((rgb[0]&0xF8)<<8)|((rgb[1]&0xFC)<<3)|((rgb[2]&0xF8)>>3)
      : colorOfRGB (rgb[0]*256*256+rgb[1]*256+rgb[2]);
#else
	map[i] = GetNearestPaletteIndex(pal, RGB(rgb[0], rgb[1], rgb[2]));
#endif

#ifdef logcmaps
    printf("Color %d: %lX - %d\n",
      i, map[i], (rgb[0]/32)*(4*8)+(rgb[1]/32)*4+rgb[2]/64);
#endif
  }
  return FALSE;
}

int DoExtension (struct cw_status *gp, AweStream *fd, int label)
{
  char		 FAR *str;
  
  switch (label) {
  case 0x01:		/* Plain Text Extension */
    str = "Plain Text Extension";
#ifdef notdef
    if (GetDataBlock(gp, fd, buf) == 0)
      ;
    
    lpos   = LM_to_uint(buf[0], buf[1]);
    tpos   = LM_to_uint(buf[2], buf[3]);
    width  = LM_to_uint(buf[4], buf[5]);
    height = LM_to_uint(buf[6], buf[7]);
    cellw  = buf[8];
    cellh  = buf[9];
    foreground = buf[10];
    background = buf[11];
    
    while (GetDataBlock(gp, fd, buf) != 0) {
      PPM_ASSIGN(gp, image[ypos][xpos],
		 cmap[CM_RED][v],
		 cmap[CM_GREEN][v],
		 cmap[CM_BLUE][v]);
      ++index;
    }
    
    return FALSE;
#else
    break;
#endif
  case 0xff:		/* Application Extension */
    str = "Application Extension";
    break;
  case 0xfe:		/* Comment Extension */
    str = "Comment Extension";
    while (GetDataBlock(gp, fd, gp->GIFbuf) != 0);
      return FALSE;
  case 0xf9:		/* Graphic Control Extension */
    str = "Graphic Control Extension";
    (void) GetDataBlock(gp, fd, gp->GIFbuf);
    gp->Gif89.disposal    = ((gp->GIFbuf)[0] >> 2) & 0x7;
    gp->Gif89.inputFlag   = ((gp->GIFbuf)[0] >> 1) & 0x1;
    gp->Gif89.delayTime   = LM_to_uint((gp->GIFbuf)[1],(gp->GIFbuf)[2]);
    if (((gp->GIFbuf)[0] & 0x1) != 0) {
      gp->Gif89.transparent = (gp->GIFbuf)[3];
      assert (gp->Gif89.transparent < 256);
    }
    while (GetDataBlock(gp, fd, gp->GIFbuf) != 0)
      ;
    return FALSE;
  default:
    str = gp->GIFbuf;
    sprintf(gp->GIFbuf, "UNKNOWN (0x%02x)", label);
    break;
  }
  
#ifdef UNIX
  printf("got a '%s' extension - please report this to koblas@mips.com\n",
	 str);
#endif
  
  while (GetDataBlock(gp, fd, gp->GIFbuf) != 0)
    ;
  
  return FALSE;
}

#ifdef WIN31
#define memcpy _fmemcpy
#endif

int GetDataBlock(struct cw_status *gp, AweStream *fd, unsigned char  FAR *buf)
{
  unsigned char	count;
  unsigned char lbuf[256]; /* fread can use local memory only */
  if (! ReadOK(fd,&count,1)) {
    errorMsg(gp, 1, ErrDBSIZ);
    return -1;
  }
  
  gp->GIFZeroDataBlock = count == 0;
  
  if (count != 0) {
    if (! ReadOK(fd, lbuf, count)) {
      errorMsg(gp, 1, ErrDBREAD);
      return -1; 
    } else memcpy(buf, lbuf, count);
  }
  
  return count;
}

#ifdef WIN31
#undef memcpy
#endif


static int
GetCode(struct cw_status *gp, AweStream *fd, int code_size, int flag)
{
  int			i, j, ret;
  unsigned char		count;
  
  if (flag) {
		gp->GIFcurbit = 0;
		gp->GIFlastbit = 0;
		gp->GIFdone = FALSE;
		return 0;
  }
  
  if ( (gp->GIFcurbit+gp->GIFcode_size) >= gp->GIFlastbit) {
    if (gp->GIFdone) {
      if (gp->GIFcurbit >= gp->GIFlastbit)
	errorMsg(gp, 1, ErrBITS);
      return -1;
    }
    /*    assert (last_byte >= 2); */
    /* There's an error here, indexes -2 and -1 may be indexed,
       using a kludge for now */
    (gp->GIFbuf)[0] = gp->GIFlast_byte >= 2 ?
      (gp->GIFbuf)[gp->GIFlast_byte-2] : 0;
    (gp->GIFbuf)[1] = gp->GIFlast_byte >= 2 ?
      (gp->GIFbuf)[gp->GIFlast_byte-1] : 0;
    
    if ((count = GetDataBlock(gp, fd, &(gp->GIFbuf)[2])) == 0)
      gp->GIFdone = TRUE;
    
    gp->GIFlast_byte = 2 + count;
    gp->GIFcurbit = (gp->GIFcurbit - gp->GIFlastbit) + 16;
    gp->GIFlastbit = (2+count)*8 ;
  }
  
  ret = 0;
  for (i = gp->GIFcurbit, j = 0; j < gp->GIFcode_size; ++i, ++j)
    ret |= (((gp->GIFbuf)[ i / 8 ] & (1 << (i % 8))) != 0) << j;
  
  gp->GIFcurbit += gp->GIFcode_size;
  
  return ret;
}

int LWZReadByte (struct cw_status *gp, AweStream *fd, int flag, int input_code_size)
{
  int		code, incode;
  register int	i;

  if (flag) {
    gp->GIFset_code_size = input_code_size;
    gp->GIFcode_size = gp->GIFset_code_size+1;
    gp->GIFclear_code = 1 << gp->GIFset_code_size ;
    gp->GIFend_code = gp->GIFclear_code + 1;
    gp->GIFmax_code_size = 2*gp->GIFclear_code;
    gp->GIFmax_code = gp->GIFclear_code+2;
    
    GetCode(gp, fd, 0, TRUE);
    
    gp->GIFfresh = TRUE;
    
    for (i = 0; i < gp->GIFclear_code; ++i) {
      (gp->GIFtable)[0][i] = 0;
      (gp->GIFtable)[1][i] = i;
    }
    for (; i < (1<<MAX_LWZ_BITS); ++i)
      (gp->GIFtable)[0][i] = (gp->GIFtable)[1][0] = 0;
    
    gp->GIFsp = gp->GIFstack;
    
    return 0;
  } else if (gp->GIFfresh) {
    gp->GIFfresh = FALSE;
    do {
      gp->GIFfirstcode = gp->GIFoldcode =
	GetCode(gp, fd, gp->GIFcode_size, FALSE);
    } while (gp->GIFfirstcode == gp->GIFclear_code);
    return gp->GIFfirstcode;
  }
  
  if (gp->GIFsp > gp->GIFstack)
    return *--(gp->GIFsp);
  
  while ((code = GetCode(gp, fd, gp->GIFcode_size, FALSE)) >= 0) {
    if (code == gp->GIFclear_code) {
      for (i = 0; i < gp->GIFclear_code; ++i) {
	(gp->GIFtable)[0][i] = 0;
	(gp->GIFtable)[1][i] = i;
      }
      for (; i < (1<<MAX_LWZ_BITS); ++i)
	(gp->GIFtable)[0][i] = (gp->GIFtable)[1][i] = 0;
      gp->GIFcode_size = gp->GIFset_code_size+1;
      gp->GIFmax_code_size = 2*gp->GIFclear_code;
      gp->GIFmax_code = gp->GIFclear_code+2;
      gp->GIFsp = gp->GIFstack;
      gp->GIFfirstcode = gp->GIFoldcode =
	GetCode(gp, fd, gp->GIFcode_size, FALSE);
      return gp->GIFfirstcode;
    } else if (code == gp->GIFend_code) {
      int		count;
      unsigned char	buf[260];
      
      if (gp->GIFZeroDataBlock)
	return -2;
      
      while ((count = GetDataBlock(gp, fd, buf)) > 0)
	;
      
      if (count != 0)
	errorMsg(gp, 1, ErrNEOD);
      return -2;
    }
    
    incode = code;
    
    if (code >= gp->GIFmax_code) {
      *(gp->GIFsp)++ = gp->GIFfirstcode;
      code = gp->GIFoldcode;
    }
    
    while (code >= gp->GIFclear_code) {
      *(gp->GIFsp)++ = (gp->GIFtable)[1][code];
      if (code == (gp->GIFtable)[0][code])
	errorMsg(gp, 1, ErrCIRCTAB);
      code = (gp->GIFtable)[0][code];
    }
    
    *(gp->GIFsp)++ = gp->GIFfirstcode = (gp->GIFtable)[1][code];
    
    if ((code = gp->GIFmax_code) <(1<<MAX_LWZ_BITS)) {
      (gp->GIFtable)[0][code] = gp->GIFoldcode;
      (gp->GIFtable)[1][code] = gp->GIFfirstcode;
      ++(gp->GIFmax_code);
      if ((gp->GIFmax_code >= gp->GIFmax_code_size) &&
	  (gp->GIFmax_code_size < (1<<MAX_LWZ_BITS))) {
	gp->GIFmax_code_size *= 2;
	++(gp->GIFcode_size);
      }
    }
    
    gp->GIFoldcode = incode;
    
    if (gp->GIFsp > gp->GIFstack)
      return *--(gp->GIFsp);
  }
  return code;
}

void ReadImage(struct cw_status *gp, AweStream *fd, int len, int height,
	       int interlace, int ignore, long *gifColormap, CalImage *cip)
{
  unsigned char	c;	

  /*
  **  Initialize the Compression routines
  */
  if (! ReadOK(fd,&c,1)){
    fprintf (stderr, "Error: EOF / read error on image data\n");
    return;
  }
  if (LWZReadByte(gp, fd, TRUE, c) < 0)
    fprintf (stderr, "Error: error reading image\n");

  /*
  **  If this is an "uninteresting picture" ignore it.
  */
  if (ignore) {
    while (LWZReadByte(gp, fd, FALSE, c) >= 0)
      ;
    return;
  }
  gifSendData (gp, fd, c, gifColormap, cip, interlace);
}

void ReadGIF(struct cw_status *gp, AweStream *fd, long *gifColormap,
	     CalImage  FAR *imgptr)
{
  unsigned char	buf[16];
  unsigned char	c;
  int		useGlobalColormap;
  int		bitPixel;
  int		imageCount = 0;
  char		version[4];
  
  gp->Gif89.transparent = -1;
  if (! ReadOK(fd,buf,6)) {
    fprintf(stderr, "Error: error reading magic number\n");
    imgptr->width = 0;
    return;
  }

  if (strncmp(buf,"GIF",3) != 0) {
    fprintf(stderr, "Error: not a GIF file\n");
    imgptr->width = 0;
    return;
  }
  
  strncpy(version, buf + 3, 3);
  version[3] = '\0';
  
  if ((strcmp(version, "87a") != 0) && (strcmp(version, "89a") != 0)) {
    fprintf(stderr, "Error: bad version number, not '87a' or '89a'\n");
    imgptr->width = 0;
    return;
  }
  
  if (! ReadOK(fd,buf,7)) {
    fprintf(stderr, "Error: failed to read screen descriptor\n");
    imgptr->width = 0;
    return;
  }

  gp->GifScreen.Width=LM_to_uint(buf[0],buf[1]);   /* update g->width */
  gp->GifScreen.Height=LM_to_uint(buf[2],buf[3]);  /* and height */
  gp->GifScreen.BitPixel        = 2<<(buf[4]&0x07);
  gp->GifScreen.ColorResolution = (((buf[4]&0x70)>>3)+1);
  gp->GifScreen.Background      = buf[5];
  gp->GifScreen.AspectRatio     = buf[6];
  
  if (BitSet(buf[4], LOCALCOLORMAP)) {	/* Global Colormap */
    if (ReadColorMap(gp, fd,gp->GifScreen.BitPixel, gifColormap)) {
      fprintf(stderr, "Error: error reading global colormap\n");
      imgptr->width = 0;
      return;
    }
  }
  
  if (gp->GifScreen.AspectRatio != 0) {
    float	r;
    r =  (float) ((gp->GifScreen.AspectRatio + 15.0 ) / 64.0);
    fprintf (stderr, "Warning : non-square pixels in GIF-file (%c %f)\n",
	     r < 1.0 ? 'x' : 'y',
	     r < 1.0 ? 1.0 / r : r);
  }

  for (;;) {
    if (! ReadOK(fd,&c,1)){
      fprintf (stderr, "Error: EOF / read error on image data\n");
      imgptr->width = 0;
      return;
    }
    if (c == ';') {		/* GIF terminator */
      if (imageCount < 1)
	fprintf (stderr, "Error: only %d image%s found in file\n",
		 imageCount, imageCount>1?"s":"");
      imgptr->width = 0;
      return;
    }
    
    if (c == '!') { 	/* Extension */
      if (! ReadOK(fd,&c,1)) {
	fprintf(stderr, "Error: OF / read error on extention function code\n");
	imgptr->width = 0;
	return;
      }
      DoExtension(gp, fd, c);
      continue;
    }
    
    if (c != ',') {		/* Not a valid start character */
      if(gp->err_file == NULL)
	fprintf (stderr, "Warning: bogus character 0x%02x, ignoring\n",
		 (int) c);
      continue;
    }
    
    ++imageCount;
    
    if (! ReadOK(fd,buf,9)) {
      fprintf (stderr, "Error: couldn't read left/top/width/height\n");
      imgptr->width = 0;
      return;
    }
    
    useGlobalColormap = ! BitSet (buf[8], LOCALCOLORMAP);
    
    bitPixel = 1<<((buf[8]&0x07)+1);
    
    imgptr->width = gp->GifScreen.Width;
    imgptr->height = gp->GifScreen.Height;
    imgptr->color = gp->Gif89.transparent != -1 ?
      gifColormap[gp->Gif89.transparent] : -1;

    if (! useGlobalColormap) {
      if (ReadColorMap (gp, fd, bitPixel, gifColormap)) {
	fprintf (stderr, "Error: error reading local colormap\n");
	imgptr->width = 0;
	return;
      }
      ReadImage (gp, fd, LM_to_uint(buf[4],buf[5]),
		 LM_to_uint(buf[6],buf[7]),
		 BitSet(buf[8], INTERLACE), imageCount != 1,
		 gifColormap, imgptr);
      return;
    } else {
      ReadImage (gp, fd, LM_to_uint(buf[4],buf[5]),
		 LM_to_uint (buf[6],buf[7]),
		 BitSet (buf[8], INTERLACE), imageCount != 1,
		 gifColormap, imgptr);
      return;
    }
  }
/* Quit when first image is read, CandleWeb does not know what to do
   with GIFs containing several */
}








