/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no), 
 *	    Gunnar Rnning (gunnarr@ifi.uio.no)
 */

#ifndef SIMULATE_H
#define SIMULATE_H

#include "candle.h"
#include "fast_lis.h"
#include "types.h"

enum {
  DO_VDECL = 1, 
  DO_VINST
};

/* 
 * Constants for the flags in struct oper.
 */
#define INTVAL		(T_INT)
#define FLOATVAL	(T_FLOAT)
#define TEXTVAL		(T_TEXT)
#define INTARRAY	(T_INT   | T_ARRAY)
#define FLOATARRAY	(T_FLOAT | T_ARRAY)
#define TEXTARRAY	(T_TEXT  | T_ARRAY)
#define INTVAR		(T_INT   | T_VAR)
#define FLOATVAR	(T_FLOAT | T_VAR)
#define TEXTVAR		(T_TEXT  | T_VAR)
#define INTCONST	(T_INT   | T_CONST)
#define FLOATCONST	(T_FLOAT | T_CONST)
#define TEXTCONST	(T_TEXT  | T_CONST)
#define BINTFUNC	(T_INT   | T_FUNC | T_BIN)
#define BFLOATFUNC	(T_FLOAT | T_FUNC | T_BIN)
#define BTEXTFUNC	(T_TEXT  | T_FUNC | T_BIN)
#define UINTFUNC	(T_INT   | T_FUNC)
#define UFLOATFUNC      (T_FLOAT | T_FUNC)
#define UTEXTFUNC	(T_TEXT  | T_FUNC)
#define INTOPER		(T_INT   | T_OPER)
#define FLOATOPER	(T_FLOAT | T_OPER)
#define TEXTOPER	(T_TEXT  | T_OPER)
#define ATTROPER	(T_OBJECT)
/********************************************************/
/*                               struct oper            */
/* Oper consists of an arithmetic operation (+,-,...),  */
/* left term, which may be another oper or variable,    */
/* and right term, which is another oper or NULL.       */
/********************************************************/

struct oper {
  unsigned char flag;
  /* If flag=='i' - (int) use operation.lng,
   * if flag=='r' (float/real) use operation.flt,
   * else : we don't have a value, but should just fetch the 
   * value from left.value, left.fvalue etc.
   */

  unsigned char type; /* Used to differ between vdecl and vinst. 
	      *	type = DO_VDECL then VDECL_ACCESS
	      *	type = DO_VINST then VINST_ACCESS
	      */
  unsigned char valtype;
  /* flag = 'D' left.value.
     flag = 'F' left.fvalue.
     flag = 'T' left.text.
     flag = 'd' left.array. (integer).
     flag = 'f' left.array. (float).
     flag = 't' left.array. (text).
     flag = 'V' left.variable. (integer).
     flag = 'v' left.variable. (float).
     flag = 'S' left.variable. (text).
     flag = 'C' left.cconst. (integer).
     flag = 'c' left.cconst. (float).
     flag = 's' left.cconst. (text).
     flag = 'A' left.do_act. (integer).
     flag = 'a' left.do_act. (float).
     flag = 'U' left.do_act. (integer). User defined function.
     flag = 'u' left.do_act. (float). User defined function.
     flag = 'M' left.do_act. (text). User defined function.
     flag = 'i' left.op && (left->flag is int && (unary || right->flag is int)).
     flag = 'r' left.op && (left->flag is float || right->flag is float)
     flag does not affect right, which is always op (oper pointer or NULL)
   */

  union {
    long (*lng)(struct cw_status *, struct oper *, long, long);
    double (*flt)(struct cw_status *, struct oper *, double, double);
    char *(*txt)(struct cw_status *, struct oper *, char *, char *);
    long (*ltxt)(struct cw_status *, struct oper *, char *, char *);
  } operation;
#ifdef FASTEXPR
  union {
    long (*l_calc)(struct cw_status *, struct oper FAR *);
    double (*r_calc)(struct cw_status *, struct oper FAR *);
    char *(*t_calc)(struct cw_status *, struct oper FAR *);
  } calc;
  union {
    long val;
    double rval;
    char FAR *tval;
  }left, right;
#endif
  union {
    long value;
    double fvalue;
    char FAR *text;
    struct vinst FAR *vi;
    struct vdecl FAR *vd;
    struct c_decl FAR *cconst;
    struct intarray FAR *array;
    struct action FAR *do_act;
    struct oper FAR *op;
    struct attrparam *ap;
    struct translation FAR *tl;
    struct pointarray FAR *pt;
  } left, right;
  short line;
};

/*
 * List of identifiers which is used before they are declared.
 */
struct notdecl {
  lp (notdecl)
  char FAR *name;			/* Name of the identifier */ 

  unsigned char type;			/* Expected type of the declaration.
				 * Legal values are :
				 * 'V'	- Variable
				 * 'A'	- Array
				 * 'F'	- Function 
				 */
  union {				   
  struct param FAR *par;	/* Pointer to the parameters of the identfier.
				 * if type == 'V' then act_par == NULL 
				 */  
  } actual;
  struct oper FAR *op;		/* Pointer to the oper structure of which this 
				 * identifier is the left part.
				 */
  struct action FAR *do_act;	/* Pointer to the action structure 
				 * associated with this identifier.
				 */
};

/********************************************************/
/*                                           c_decl     */
/*  Listed declarations of constants.                   */
/********************************************************/

struct c_decl {
  lp (c_decl)
  char FAR *name;
  union {
    long value;
    double fvalue;
    char FAR *text;
  } thisc;
  unsigned char type;
};

struct outlist {
  lp (outlist)
  int FAR *out;
  struct attribute FAR *attr;
};

/* This structure contains the initializing expression(s) 
 * for variables and arrays.
 */  
struct vinit {
  lp(vinit)
  struct vinit FAR *next_dim;	/* Points to vinit strcuture for the next 
				 * dimension
				 */
				  
  struct oper FAR *exp;		/* Points to initializing expression if 
				 * next_dim == NULL
				 */
};

/* 
 * Variable instance.
 */
struct vinst {
  lp(vinst)	/* Used to link variable instances in a function instance */
  struct vdecl FAR *parent;
  union {
    struct outlist FAR *out;      /* used for connection to output unit */
    struct outlist FAR * FAR *outarray; 
  } out;
  union {
    /* The next fields are used for normal variables */
    long value;
    double fvalue;
    char FAR *text;
    /* And these fields are used for array variables */
    long FAR *elem;
    double FAR *felem;
    char FAR * FAR *telem;
    void *void_el;
  } value;
  unsigned long FAR *size;	/* size[0..ndim-1] contains the size of each 
			 * dimension.
			 * size[ndim..2*ndim-1] contains the multiplicative 
			 * operand to be associated with the dimension. 
			 * That is 
			 * size[ndim] == size[1] * ... * size[ndim-1], 
			 * size[ndim+1] == size[2] * ... * size[ndim-1], ...
			 * size[ndim*2-1] == 1.
			 */
  unsigned live;	/* Used when terminating a function to determine if this 
			 * vinst should propagate to the caller(because it is 
			 * bound to a propagating object).
			 */
};

/********************************************************/
/*                                           vdecl      */
/*  Listed declarations of variables. Contains type,    */
/*  name, start value, value and a pointer to output    */
/*  unit (out).                                         */
/********************************************************/
struct vdecl {
  lp (vdecl)
  char FAR *name;

  struct vinit FAR *init;   /* Contains the initializing expressions */

  struct param FAR *dim_size; /* List of expressions describing the dimension 
			   * size(s) in an array.
			   */
			 
  struct vinst FAR *vi;	/* Pointer to an instance of the variable. This 
			 * instance should _always_ be that of the last 
			 * executing version of the associated function.
			 */
  struct c_decl FAR *cconst;
  struct intarray FAR *array; /* Pointer to an intarray which should replace this
			   * vdecl in expressions. Only used when we reference
			   * variables points to an array element.
			   */ 
  unsigned char type;
  unsigned char ndim;		/* Number of dimensions in array */
  unsigned char log;
};

/********************************************************/
/*                                             action   */
/*  Used to link predefined functions to simulations.   */
/*  These functions allows interaction through input.   */
/********************************************************/

struct action {
  union {
    long (*do_act)();  /* pointer to actual function */
    double (*fdo_act)();
    char FAR *(*tdo_act)();
    struct function FAR *func_act;
    struct objectproto FAR *obj_act;
  } function;
  
  union {
    struct param FAR *par; /* list of parameters */
    struct attrparam FAR *attr;
  }actual;
  short line;
  unsigned char type;
};



/********************************************************/
/*                                             param    */
/*  Listed parameters used in function calls.           */
/********************************************************/

struct param {
  lp (param)
  struct oper FAR *exp;
  short line;
};


struct intarray {
  union {
    struct vinst FAR *vi;
    struct vdecl FAR *vd;
  } which;
  struct param FAR *indeks;
  unsigned char type;		/* Used to differ between vdecl and vinst. 
			 * type = DO_VDECL then VDECL_ACCESS
			 * type = DO_VINST then VINST_ACCESS
			 */
};

#ifdef OLDP
/**********************************************************/
/*                                           equation     */
/*  Struct equation consists of a pointer to the variable */
/*  that are to be assigned a value. The left term of the */
/*  equation is the expression (equation) eq_op.          */
/**********************************************************/

struct equation {
  union {
    struct vinst FAR *vi;
    struct vdecl FAR *vd;
  } which;
  struct intarray FAR *array;
  struct oper FAR *eq_op;
  unsigned char type;		/* Used to differ between vdecl and vinst. 
			 * type = DO_VDECL then VDECL_ACCESS
			 * type = DO_VINST then VINST_ACCESS
			 */
};
#endif

/* fparam - formal parameter.
 *
 * Holds information about the formal parameters to functions.
 */
struct fparam {
  lp(fparam)
  struct vdecl FAR *var;
  char is_ref;		/* is_ref == TRUE if this is a reference parameter */
};

/* 
 * Valid types for blocks
 */ 
#define FUNCTIONBLOCK	1L
#define OBJECTBLOCK	2L
#define GLOBALBLOCK	3L
/*
 * This can for the moment be either a function or a 
 * global(learnunit) block. 
 */
struct commonblock {
  lp(commonblock)
  unsigned char type;
  char *name;
  struct vdecl *firstv, *lastv;
  struct c_decl FAR *firstc,FAR *lastc; /* Pointers to constants in the block */
  struct vdecl FAR *fparvar;	/* dummy */
  struct commonblock FAR *parent_block;
  struct sim_actions FAR *sim;	/* Pointer to the list of statements in the 
				 * function.
				 */
  unsigned char rettype;       /* S_INTEGER || S_FLOAT || S_TEXT || S_VOID */
  union {
    long value;
    double fvalue;
    char FAR *text;
  } retval;
  struct function *fp;
};

struct blockinst {
  struct commonblock *parent;
  struct blockinst FAR *caller;
  struct vinst FAR *firstvi;
  struct winobj FAR *fwin;
  struct graphobj FAR *fgraph;
  struct inpobj FAR *finp;
  struct sim_actions FAR *sim;
  /*  List of ref parameters pointing to array elements*/
  struct vdecllist FAR *vl;
};

struct function {
  lp(function)
  unsigned char type;
  char FAR *name;			/* Name of the function */  
  struct vdecl FAR *firstv, 	/* Points to first local variable. */
    FAR *lastv; 			/* Points to last local variable */
  struct c_decl FAR *firstc,FAR *lastc; /* Pointers to constants in the function */
  struct vdecl FAR *fparvar;	/* Points to first parameter in firstv list */

  struct commonblock FAR *parent_block;
  struct sim_actions FAR *sim;	/* Pointer to the list of statements in the 
				 * function.
				 */
  unsigned char rettype;       /* S_INTEGER || S_FLOAT || S_TEXT || S_VOID */
  union {
    long value;
    double fvalue;
    char FAR *text;
  } retval;
  struct function *fp;
  /* End of commonblock part */

  struct winobj FAR *fwin;
  struct graphobj FAR *fgraph;
  struct inpobj FAR *finp;
  struct vdecl FAR *fref;		/* Points to first reference parameter */


/*
 * The following relationships should always hold between the pointers above:
 *
 * firstv |- lastv |- fref 
 * firstv==first(firstv) && (fref == NULL || last(fref) == last(firstv)
 */

  struct fparam FAR *fpar;		/* Points to first formal parameter */
  struct notdecl FAR *nd;		/* Pointer to list of identifiers which have 
				 * not yet been declared. 
				 * Only used while parsing.
				 */
};

struct attrparam {
  lp(attrparam)
  char type;      /* Attribute type : T_EXPR, T_TRANSLATION, T_POINTS */
  char indirect;
  char FAR *name;    /* Attribute name */

  /* Default value pointer may be NULL */
  union {
    struct oper FAR *oper;
    struct ptoperlist FAR *pl;
    struct translation FAR *tl;
    struct attrparam FAR *ap;
  } defval;
  union {
    struct oper FAR *oper;
    struct ptoperlist FAR *pl;
    struct translation FAR *tl;
  } actval;
  int npoints;	/* Number of points in ptoperlist */
};

struct pointarray {
  struct oper FAR * FAR *op;
  int n;
};

struct objectproto {
  lp(objectproto)
  unsigned char type;
  char FAR *name;

  struct vdecl FAR *firstv, 	/* Points to first local variable. */
    FAR *lastv;			/* Points to last local variable */

  struct c_decl FAR *firstc,FAR *lastc; /* Pointers to constants in the function */
  struct attrparam FAR *fparam; /* Pointer to first object attribute */

  struct commonblock FAR *parent_block;
  struct sim_actions FAR *sim;
  unsigned char rettype;       /* S_INTEGER || S_FLOAT || S_TEXT || S_VOID */
  union {
    long value;
    double fvalue;
    char FAR *text;
  } retval;
  struct function *fp;
};

struct objectinst {
  struct objectproto FAR *parent;
  struct blockinst FAR *caller;
  struct vinst FAR *firstvi;
  struct winobj FAR *fwin;
  struct graphobj FAR *fgraph;
  struct inpobj FAR *finp;

  struct sim_actions FAR *sim;
  /*  List of ref parameters pointing to array elements*/
  struct vdecllist FAR *vl;
};

struct vinstlist {
  lp(vinstlist)
  struct vinst FAR *vi;
};

struct vdecllist {
  lp(vdecllist)
  struct vdecl FAR *vd;
  struct intarray FAR *array;
};

struct arraylist {
  lp(arraylist)
  struct intarray FAR *array; 
};

struct funcinst {
  struct function FAR *parent;
  struct blockinst FAR *caller;	/* Points to the funcinst that called 
				 * the current function.
				 */
  
  struct vinst FAR *firstvi;

  struct winobj FAR *fwin;
  struct graphobj FAR *fgraph;
  struct inpobj FAR *finp;

  struct sim_actions FAR *sim;

/* Pointer to the instantiated objects */

  /*  List of ref parameters pointing to array elements*/
  struct vdecllist FAR *vl;

  struct vinstlist FAR *foldvi;		

  /* Corresponding list of array elements 
     found in the calling function. */
/*   struct arraylist FAR *al; */

  /* # of times optimizeObject was called within funcinst */
  int optimized;
};

/**********************************************************/
/*                                        sim_actions     */
/*   Main structure for simulation. There are three diff- */
/*   erent types of simulation actions:                   */
/*             1. function call.    *do_act.              */
/*             2. equation.         *eq.                  */
/*             3. if_else, for, while or switch statement.*/
/*   (*actions)(), *eq_op are used in the last case to    */
/*   hold test information.                               */
/**********************************************************/

struct sim_actions {
  lp (sim_actions)
  struct blockinst FAR *parent;     
  int (*actions)();
  struct action FAR *do_act;
#ifdef OLDP
  struct equation FAR *eq;
#else
  struct oper FAR *eq;
#endif
  struct oper FAR *eq_op;
  struct sim_actions FAR *cond_act;
  struct oper FAR *ret_act;
  struct winobj FAR *win;
  struct graphobj FAR *graph;
  struct inpobj FAR *inp;
};

#define valtype(op) ((op)->valtype)

#define NEWSIMACTION (struct sim_actions FAR *) CalCalloc(1, sizeof(struct sim_actions))
#define NEWPARAM  (struct param FAR *) CalCalloc(1, sizeof(struct param))
#define NEWOPER   (struct oper FAR *) CalCalloc(1, sizeof(struct oper))
#define NEWEQUATION  (struct equation FAR *) CalCalloc(1, sizeof(struct equation))
#define NEWCDECL  (struct c_decl FAR *) CalCalloc(1, sizeof(struct c_decl))
#define NEWVDECL  (struct vdecl FAR *) CalCalloc(1, sizeof(struct vdecl))
#define NEWVINST  (struct vinst FAR *) CalCalloc(1, sizeof(struct vinst))
#define NEWVINIT  (struct vinit FAR *) CalCalloc(1, sizeof(struct vinit))
#define NEWACTION (struct action FAR *) CalCalloc(1, sizeof(struct action))
#define NEWINTARRAY (struct intarray FAR *) CalCalloc(1, sizeof(struct intarray))
#define NEWOUTLIST (struct outlist FAR *) CalCalloc(1, sizeof(struct outlist))
#define NEWNOTDECL (struct notdecl FAR *) CalCalloc(1, sizeof(struct notdecl))
#define NEWFUNCTION  (struct function FAR *) CalCalloc(1, sizeof(struct function))
#define NEWFUNCINST  (struct funcinst FAR *) CalCalloc(1, sizeof(struct funcinst))
#define NEWOBJECTPROTO  (struct objectproto FAR *) CalCalloc(1, sizeof(struct objectproto))
#define NEWOBJECTINST  (struct objectinst FAR *) CalCalloc(1, sizeof(struct objectinst))
#define NEWVINSTLIST  (struct vinstlist FAR *) CalCalloc(1, sizeof(struct vinstlist))
#define NEWVDECLLIST (struct vdecllist FAR *) CalCalloc(1, sizeof(struct vdecllist))
#define NEWVARRAYLIST (struct arraylist FAR *) CalCalloc(1, sizeof(struct arraylist))
#define NEWFPARAM  (struct fparam FAR *) CalCalloc(1, sizeof(struct fparam))
#define NEWATTRPARAM  (struct attrparam FAR *) CalCalloc(1, sizeof(struct attrparam))
#define NEWPOINTARRAY  (struct pointarray FAR *) CalCalloc(1, sizeof(struct pointarray))

#endif /* SIMULATE_H */
