/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no), 
 *	    Gunnar Rnning (gunnarr@ifi.uio.no)
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h> 
#include <string.h> 
#include <malloc.h> 


#include "candle.h"
#include "simulate.h"
#include "pbmplus.h"
#include "nodes.h"
#include "error.h"

#include "sysproto.h"
#include "protos/canutil.h"
#include "protos/fast_lis.h"
#include "protos/funcutil.h"
#include "protos/memory.h"

#define lmin(a, b)  (((a) < (b)) ? (a) : (b))


long project (struct cw_status *gp, struct param FAR *par) {
  long i, arraysize, dn;
  long scalar, FAR *oldx, FAR *oldy, FAR *oldz, FAR *newx, FAR *newy;
  long oldxsize, oldysize, oldzsize, newxsize, newysize;
  int arg = 0;
  char FAR *funcname = "project";
  char mess[80];
  
  scalar = getIntPar (gp, &par, ++arg, funcname);

  getIntArrPar (gp, &par, ++arg, funcname,(void **)&oldx, &oldxsize, 1);
  arraysize = oldxsize;

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldy, &oldysize, 1);
  arraysize = lmin(arraysize, oldysize);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldz, &oldzsize, 1);
  arraysize = lmin(arraysize, oldzsize);

  changed_par(gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newx, &newxsize, 1);
  arraysize = lmin(arraysize, newxsize);

  changed_par(gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newy, &newysize, 1);
  arraysize = lmin(arraysize, newysize);
  checkPars (gp, par, arg, funcname);

  for (i = 0; i < arraysize; i++) {
    if ((dn = oldz[i]+scalar) <= 0) {
      sprintf(mess, WarnBEHINDPOV, i);
      errorMsg(gp, 0, mess);
      dn = 1;
    }
    else {
      newx[i] = (scalar == 0) ? oldx[i] : (oldx[i]*scalar)/dn;
      newy[i] = (scalar == 0) ? oldy[i] : (oldy[i]*scalar)/dn;
    }
  }
  return 0;
}

long rotate (struct cw_status *gp, struct param FAR *par)
{
  double rotmat[3][3];
  long i, arraysize;
  long x, y, z;

  double xr, yr, zr, d;

  long FAR *oldx, FAR *oldy, FAR *oldz, FAR *newx, FAR *newy, *newz;
  long oldxsize, oldysize, oldzsize, newxsize, newysize, newzsize;
  int arg = 0;
  char FAR *funcname = "rotate";

  xr = (double)getFltPar (gp, &par, ++arg, funcname);
  yr = (double)getFltPar (gp, &par, ++arg, funcname);
  zr = (double)getFltPar (gp, &par, ++arg, funcname);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldx, &oldxsize, 1);
  arraysize = oldxsize;

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldy, &oldysize, 1);
  arraysize = lmin(arraysize, oldysize);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldz, &oldzsize, 1);
  arraysize = lmin(arraysize, oldzsize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newx, &newxsize, 1);
  arraysize = lmin(arraysize, newxsize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newy, &newysize, 1);
  arraysize = lmin(arraysize, newysize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newz, &newzsize, 1);
  arraysize = lmin(arraysize, newzsize);

  checkPars (gp, par, arg, funcname);

/* Calculate rotation matrix */
  rotmat[0][0] = cos ((double)yr)*cos ((double)zr);
  rotmat[0][1] = -cos ((double)xr)*cos ((double)yr)*sin ((double)zr)
    +sin ((double)xr)*sin((double)yr);
  rotmat[0][2] = sin ((double)xr)*cos ((double)yr)*sin ((double)zr)
    +cos ((double)xr)*sin ((double)yr);
  rotmat[1][0] = sin ((double)zr);
  rotmat[1][1] = cos ((double)xr)*cos ((double)zr);
  rotmat[1][2] = -sin ((double)xr)*cos ((double)zr);
  rotmat[2][0] = -sin ((double)yr)*cos ((double)zr);
  rotmat[2][1] = cos ((double)xr)*sin ((double)yr)*sin ((double)zr)
    +sin ((double)xr)*cos ((double)yr);
  rotmat[2][2] = -sin ((double)xr)*sin ((double)yr)*sin ((double)zr)
    +cos ((double)xr)*cos ((double)yr);

  for (i = 0; i < arraysize; i++) {
    x = oldx[i]; 
    y = oldy[i]; 
    z = oldz[i];
    d       = rotmat[0][0]*x+rotmat[0][1]*y+rotmat[0][2]*z;
    newx[i] = (long) d;
    d       = rotmat[1][0]*x+rotmat[1][1]*y+rotmat[1][2]*z;
    newy[i] = (long) d;
    d       = rotmat[2][0]*x+rotmat[2][1]*y+rotmat[2][2]*z;
    newz[i] = (long) d;
  }
  return 0;
}

long scale (struct cw_status *gp, struct param FAR *par)
{
  long i, arraysize;

  double xscale, yscale, zscale;

  long FAR *oldx, FAR *oldy, FAR *oldz, FAR *newx, FAR *newy, *newz;
  long oldxsize, oldysize, oldzsize, newxsize, newysize, newzsize;
  int arg = 0;
  char FAR *funcname = "scale";

  xscale = (double)getFltPar (gp, &par, ++arg, funcname);
  yscale = (double)getFltPar (gp, &par, ++arg, funcname);
  zscale = (double)getFltPar (gp, &par, ++arg, funcname);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldx, &oldxsize, 1);
  arraysize = oldxsize;

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldy, &oldysize, 1);
  arraysize = lmin(arraysize, oldysize);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldz, &oldzsize, 1);
  arraysize = lmin(arraysize, oldzsize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newx, &newxsize, 1);
  arraysize = lmin(arraysize, newxsize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newy, &newysize, 1);
  arraysize = lmin(arraysize, newysize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newz, &newzsize, 1);
  arraysize = lmin(arraysize, newzsize);

  checkPars (gp, par, arg, funcname);

/* Calculate rotation matrix */
  for (i = 0; i < arraysize; i++) {
    newx[i] = (long) xscale*oldx[i];
    newy[i] = (long) yscale*oldy[i];
    newz[i] = (long) zscale*oldz[i];
  }
  return 0;
}

long visiblesurface (struct cw_status *gp, struct param FAR *par)
{
  long x1, y1, x2, y2, x3, y3;

  long vec1x, vec1y;
  long vec2x, vec2y;
  int arg = 0;
  char FAR *funcname = "visibleSurface";

  x1 = getIntPar (gp, &par, ++arg, funcname);
  y1 = getIntPar (gp, &par, ++arg, funcname);
  x2 = getIntPar (gp, &par, ++arg, funcname);
  y2 = getIntPar (gp, &par, ++arg, funcname);
  x3 = getIntPar (gp, &par, ++arg, funcname);
  y3 = getIntPar (gp, &par, ++arg, funcname);

  checkPars (gp, par, arg, funcname);

  vec1x = x1-x2; vec1y = y1-y2;
  vec2x = x2-x3; vec2y = y2-y3;

/* Visible if z component of crossproduct is positive */
  return (vec1x*vec2y-vec1y*vec2x) > 0;
}

int distcmp (const void *s1, const void *s2)
{
  struct sortnode FAR *sn1, FAR *sn2;

  sn1 = (struct sortnode FAR *)s1;
  sn2 = (struct sortnode FAR *)s2;
  return (sn1->dist < sn2->dist) ? 1 :
    (sn1->dist > sn2->dist) ? -1 : 0;
}

long distsort (struct cw_status *gp, struct param FAR *par)
{
  struct sortnode *sortarray;

  long scalar, FAR *oldx, FAR *oldy, FAR *oldz, FAR *ordered;
  long oldxsize, oldysize, oldzsize, orderedsize, arraysize, i, elmcount = 0;
  int arg = 0;
  char FAR *funcname = "distSort";

  scalar = getIntPar (gp, &par, ++arg, funcname);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldx, &oldxsize, 1); 
  arraysize = oldxsize;

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldy, &oldysize, 1);
  arraysize = lmin(arraysize, oldysize);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldz, &oldzsize, 1);
  arraysize = lmin(arraysize, oldzsize);

  changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&ordered, &orderedsize, 1);
  arraysize = lmin(arraysize, orderedsize);

  checkPars (gp, par, arg, funcname);

#if (defined UNIX || defined WIN32)
  sortarray =
    (struct sortnode *)CalCalloc (1, (size_t)(arraysize*sizeof (struct sortnode)));
#elif WIN31
  sortarray = (struct sortnode *)_ncalloc (1, (size_t)(arraysize*sizeof (struct sortnode)));
#endif

  if (!sortarray)
    errorMsg(gp, 2, ErrNOMOREMEM);
    
/* Use qsort to get o(n*log n), even if a bit of overhead on small sets */

  for (i = 0; i < arraysize; i++)
    if (oldz[i]+scalar > 0) {
      sortarray[elmcount].dist =
	(int)sqrt ((double)(oldx[i]*oldx[i]
			    +oldy[i]*oldy[i]
			    +(oldz[i]+scalar)*(oldz[i]+scalar)));
      sortarray[elmcount++].pos = i;
    }
  qsort ((void *) sortarray, (size_t) elmcount, sizeof (struct sortnode),
	 distcmp);
  for (i = 0; i < arraysize; i++) ordered[i] = -1;
  for (i = 0; i < elmcount; i++) ordered[sortarray[i].pos] = i;

#if (defined UNIX || defined WIN32)
  CalFree (sortarray);
#elif WIN31
  _nfree (sortarray);
#endif

  return 0;
}

long addpoint (struct cw_status *gp, struct param FAR *par)
{
  long n, offs, xad, yad, zad, xmod, ymod, zmod;
  long FAR *oldx, FAR *oldy, FAR *oldz, FAR *newx, FAR *newy, FAR *newz;
  long arraysize, oldxsize, oldysize, oldzsize, newxsize, newysize, newzsize;
  long i;
  int arg = 0;
  char FAR *funcname = "addPoint";

  n = getIntPar (gp, &par, ++arg, funcname);
  offs = getIntPar (gp, &par, ++arg, funcname);
  xad = getIntPar (gp, &par, ++arg, funcname); 
  yad = getIntPar (gp, &par, ++arg, funcname);
  zad = getIntPar (gp, &par, ++arg, funcname);
  xmod = getIntPar (gp, &par, ++arg, funcname);
  ymod = getIntPar (gp, &par, ++arg, funcname);
  zmod = getIntPar (gp, &par, ++arg, funcname);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldx, &oldxsize, 1);
  arraysize = oldxsize;

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldy, &oldysize, 1);
  arraysize = lmin(arraysize, oldysize);

  getIntArrPar (gp, &par, ++arg, funcname, (void **)&oldz, &oldzsize, 1);
  arraysize = lmin(arraysize, oldzsize);

  if (xad != 0) changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newx, &newxsize, 1);
  arraysize = lmin (arraysize, newxsize);

  if (yad != 0) changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newy, &newysize, 1);
  arraysize = lmin(arraysize, newysize);

  if (zad != 0) changed_par (gp, par);
  getIntArrPar (gp, &par, ++arg, funcname, (void **)&newz, &newzsize, 1);
  arraysize = lmin(arraysize, newzsize);

  checkPars (gp, par, arg, funcname);

  if (offs < 0 || n < 0 || offs+n > arraysize) {
    errorMsg(gp, 0, WarnVECILLOFF);
    offs = abs (offs);
    n = abs (n);
    offs = lmin (offs, arraysize);
    n = lmin (n, arraysize);
    if (offs+n > arraysize) n = arraysize-offs;
  }

  for (i = offs; i < offs+n; i++) {
    newx[i] = xmod ? (oldx[i]+xad)%xmod : oldx[i]+xad;
    newy[i] = ymod ? (oldy[i]+yad)%ymod : oldy[i]+yad;
    newz[i] = zmod ? (oldz[i]+zad)%zmod : oldz[i]+zad;
  }
  return 0;
}


long pointdist (struct cw_status *gp, struct param FAR *par)
{
  long x0, y0, z0;
  long x1, y1, z1;
  int arg = 0;
  char FAR *funcname = "pointDist";
  double d;

  x0 = getIntPar (gp, &par, ++arg, funcname);
  y0 = getIntPar (gp, &par, ++arg, funcname);
  z0 = getIntPar (gp, &par, ++arg, funcname);
  x1 = getIntPar (gp, &par, ++arg, funcname);
  y1 = getIntPar (gp, &par, ++arg, funcname);
  z1 = getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  d = sqrt ((x1-x0)*(x1-x0)+(y1-y0)*(y1-y0)+(z1-z0)*(z1-z0));
  return( (long) d);
}

long linedist (struct cw_status *gp, struct param FAR *par)
{
  long x0, y0, z0;
  long x1, y1, z1, x2, y2, z2;
  double t;
  int arg = 0;
  char FAR *funcname = "lineDist";
  double d;
  
  x0 = getIntPar (gp, &par, ++arg, funcname);
  y0 = getIntPar (gp, &par, ++arg, funcname);
  z0 = getIntPar (gp, &par, ++arg, funcname);
  x1 = getIntPar (gp, &par, ++arg, funcname);
  y1 = getIntPar (gp, &par, ++arg, funcname);
  z1 = getIntPar (gp, &par, ++arg, funcname);
  x2 = getIntPar (gp, &par, ++arg, funcname);
  y2 = getIntPar (gp, &par, ++arg, funcname);
  z2 = getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

/* Determine where on the line the normal intersect */
  t = (double)((x2-x1)*(x0-x1)+(y2-y1)*(y0-y1)+(z2-z1)*(z0-z1))
    /(double)((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1)+(z2-z1)*(z2-z1));

/* t == 0 indicates p1, t==1 indicates p2 */
  if (t<=0)
    d = sqrt ((x1-x0)*(x1-x0)+(y1-y0)*(y1-y0)+(z1-z0)*(z1-z0));
  else if (t >= 1) 
    d = sqrt ((x2-x0)*(x2-x0)+(y2-y0)*(y2-y0)+(z2-z0)*(z2-z0));
  else
    d = sqrt ((x1+t*(x2-x1)-x0)*(x1+t*(x2-x1)-x0)
		+(y1+t*(y2-y1)-y0)*(y1+t*(y2-y1)-y0)
		+(z1+t*(z2-z1)-z0)*(z1+t*(z2-z1)-z0));
  return((long) d);
}

long norm_vector(struct cw_status *gp, struct param FAR *par) 
{
#ifdef CANDLE_V1
/*  parameters: array1 only, or:   array1, array2, mass1, mass2 
 *  array1 only: normalizes array1 by scaling to have a total sum of 10000 
 *  4 pars: array1 is set to the weighted sum of array1 and array2 using 
 *  mass1 and mass2 as weights, and then array1 is normalized 
 */
  struct param FAR *p;
  long FAR *tilarray;
  long FAR *fraarray;
  long FAR *helpv;
  long tilmengde, framengde;
  long length1,length2,minlength;
  long i;
  long sumvec=0;

  tilarray = par->exp->left.array->which->elem.elem;
  length1 = par->exp->left.array->which->start.start;
  p = (struct param FAR *) next(par);
  if (p!=(struct param FAR *) NULL) {
    fraarray = p->exp->left.array->which->elem.elem;
    length2 = par->exp->left.array->which->start.start;
    p = (struct param FAR *) next(p);
    tilmengde = par_value(p);
    p = (struct param FAR *) next(p);
    framengde = par_value(p);
    minlength = length1>length2?length2:length1;

    helpv = (long FAR *) CalCalloc(minlength*sizeof(long));
    for (i = 0; i < minlength ; i++, helpv++, tilarray++, fraarray++) {
      *helpv = ((*tilarray) * tilmengde) + ((*fraarray) * framengde);
      sumvec += *helpv;
    }
    helpv--;tilarray--;
    for (i = minlength - 1; i >= 0 ;i-- , helpv--, tilarray--)
      if (sumvec > 1000000)
	*tilarray = (*helpv) / (sumvec / 10000);
      else if (sumvec>10000)
	*tilarray = ((*helpv) * 100) / (sumvec / 100);
      else if (sumvec != 0)
	*tilarray = ((*helpv) * 10000) / sumvec;
      else
	*tilarray = 0;
    helpv++;
    CalFree(helpv);
  } else {
    /* only normalization of vector tilarray */
    for (i = 0; i < length1 ; i++)
      sumvec += (*tilarray++);
    tilarray--;
    for (i = 0; i < length1 ; i++,tilarray--)
      if (sumvec != 0)
	*tilarray = ((*tilarray) * 10000) / sumvec;
      else
	*tilarray = 0;
  }
#endif /* CANDLE_V1 */
  return(0L);

}

long VectorSub(struct cw_status *gp, struct param FAR *par) {
#ifdef CANDLE_V1
/* parameters: array1, array2, array3, mass1
 * array3 is set to the dot product of array1 and array2 divided by 10000
 * this is subtracted from array1, which is normalized,
 * and the mass of the subtracted cut is returned
 */
  struct param FAR *p;
  long FAR *array1, FAR *array2, FAR *array3;
  long FAR *helpv;
  long mass1, mass3=0;
  long length1,length2,length3;
  long i, retval;
  long sum1=0, sum3=0;

  /*  get all parameters */
  array1 = par->exp->left.array->which->elem.elem;
  length1 = par->exp->left.array->which->start.start;
  p = (struct param FAR *) next(par);
  if (p==(struct param FAR *) NULL) return(-1);
  array2 = p->exp->left.array->which->elem.elem;
  length2 = p->exp->left.array->which->start.start;
  p = (struct param FAR *) next(p);
  if (p==(struct param FAR *) NULL) return(-1);
  array3 = p->exp->left.array->which->elem.elem;
  length3 = p->exp->left.array->which->start.start;
  p = (struct param FAR *) next(p);
  if (p==(struct param FAR *) NULL) return(-1);
  mass1 = par_value(p);
  
  if (length1!=length2 || length2!=length3) return(-1);

  for (i = 0; i < length1 ; i++, array1++, array2++, array3++) {
    *array3 = ((*array1) * (*array2))/10000;	/*  calculate */
    *array1 -= *array3;				/*  subtract */
    sum3 += *array3;			       /* compute sums */
    sum1 += *array1;
  }
  retval = (mass1*sum3)/10000;			/* compute subtracted mass */
  if (sum1 != 0) {			/* normalization of vector array1 */
    array1--;
    for (i = 0; i < length1 ; i++) {
      *array1 = ((*array1) * 10000) / sum1;
      array1--;
    }
  }
  if (sum3 != 0) {			/*  normalization of vector array3 */
    array3--;
    for (i = 0; i < length1 ; i++) {
      *array3 = ((*array3) * 10000) / sum3;
      array3--;
    }
  }
  return(retval);

#endif /* CANDLE_V1 */
  return(0L);
}

long VectorProd(struct cw_status *gp, struct param FAR *par) {
/* multiplies each element of array1 with array2 like this:
 * for (i=0;i<length;i++) array1[i] = array1[i]*array2[i];
 */
#ifdef CANDLE_V1
  struct param FAR *p;
  long FAR *array1, FAR *array2;
  long length1, length2;
  long i;

  array1 = par->exp->left.array->which->elem.elem;
  length1 = par->exp->left.array->which->start.start;
  p = (struct param FAR *) next(par);
  if (p!=(struct param FAR *) NULL) {
    array2 = p->exp->left.array->which->elem.elem;
    length2 = par->exp->left.array->which->start.start;
    if (length1!=length2) return((long) -1);  /* ----- lengths equal only */

    for (i = 0; i < length1 ; i++, array1++, array2++) {
      *array1 = (*array1) * (*array2);
    }
      return(0);
  }
#endif /* CANDLE_V1 */
  return((long) -1);
}

long VectorSum(struct cw_status *gp, struct param FAR *par) {
/*  returns the sum of all elements of array parameter */
#ifdef CANDLE_V1

  struct param FAR *p;
  long FAR *array;
  long tilmengde, framengde;
  long length, i, sumvec=0;

  array = par->exp->left.array->which->elem.elem;
  length = par->exp->left.array->which->start.start;

  for (i = 0; i < length ; i++, array++) {
    sumvec += *array;
  }
  return(sumvec);

#endif /* CANDLE_V1 */
  return(0L);
}








