/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (svein@candleweb.no), 
 *	    Gunnar Rnning (gunnar@candleweb.no)
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include <malloc.h>

#include "regex.h"

#include "candle.h"
#include "simulate.h"
#include "error.h"

#include "protos/memory.h"
#include "sysproto.h"
#include "protos/canutil.h"
#include "protos/fast_lis.h"
#include "protos/funcutil.h"

long Char2Text(struct cw_status *gp, struct param FAR *par)
{
  return 1;
}

long Strcat(struct cw_status *gp, struct param FAR *par) {
  char FAR *par1, FAR *par2;
  struct param FAR *p;
  struct oper FAR *op;
  int length1;
  char c[2];
  char FAR *newpar1;

  if(par == NULL){
    paramError(gp, NULL, ErrSTRCATNOP);
    return 0;
  }
  /*  get text pointers */
  op = par->exp;
  if(op->flag != TEXTVAR && op->flag != TEXTARRAY){
    paramError(gp, par, ErrSTRCATFPN);
    return 0;
  }
  par1 = (char FAR *) t_par_value(gp, par);
  p = (struct param FAR *) next(par);
  if(p == NULL){
    paramError(gp, NULL, ErrSTRCATSPM);
    return 1;
  }
  if(par_type(gp, p) == T_INT){
      c[1] = '\0';
      c[0] = par_value(gp, p) % 256;
      par2 = c;
  } else if(par_type(gp, p) == T_TEXT){
      par2 = (char FAR *) t_par_value(gp, p);
  } else {
    paramError(gp, p, ErrSTRCATSPN);
    return 1;
  }

  if (!par1 || !par2) {
    paramError(gp, par, ErrSTRCATNPN);
    return(0);
  }

  /*  compute length of both strings */
  length1 = strlen(par1) + strlen(par2);
  /*  allocate new string for par1 */
  newpar1 = (char FAR *) CalMalloc(length1+1);
  /*  copy old par1 string and par2 string to new char * */
  strcpy(newpar1,par1);
  strcat(newpar1,par2);
  /*  deallocate par1   */
  CalFree(par1);
  
  t_set_par_value(gp, par, newpar1);
  return 1;
}

long Strcmp(struct cw_status *gp, struct param FAR *par) 
{
  char FAR *s1, FAR *s2;
  long len;

  s1 = t_par_value(gp, par);
  s2 = t_par_value(gp, (struct param FAR *) next(par));
  len = strcmp(s1, s2);
  CalFree(s1);
  CalFree(s2);
  return len;
}

long Stricmp(struct cw_status *gp, struct param FAR *par) 
{
  char FAR *s1, FAR *s2;
  long len;
  s1 = t_par_value(gp, par);
  s2 = t_par_value(gp, (struct param FAR *) next(par));
  len = (long) stricmp(s1, s2);
  CalFree(s1);
  CalFree(s2);
  return len;
}

long Strncmp(struct cw_status *gp, struct param FAR *par) 
{
  struct param FAR *par2;
  char FAR *s1, FAR *s2;
  long len;

  s1 = t_par_value(gp, par);
  par2 = (struct param FAR *) next(par);

  s2 = t_par_value(gp, par2);

  len = strncmp(s1, s2, par_value(gp, (struct param FAR *) next(par2)));
  CalFree(s1);
  CalFree(s2);
  return len;
}

long Strlen(struct cw_status *gp, struct param FAR *par) 
{
  long len;
  char FAR *s1;
  
  s1 = t_par_value(gp, par);
  len = strlen(s1);
  CalFree(s1);
  return len; 
}

long Strgetchar(struct cw_status *gp, struct param FAR *par) 
{
  struct param FAR *par2;
  char FAR *string;
  int nindex;
  long nvalue;

/* 1st parameter: a string, 2nd: an index
 *  return string[index], or -1 if index out of range
 */
  par2 = (struct param FAR *) next(par);
  nindex = (int) par_value(gp, par2);  
  string = t_par_value(gp, par);
  if (nindex >= 0 && ((int) strlen(string)>nindex)) 
    nvalue = string[nindex];
  else 
    nvalue = -1;
  CalFree(string);
  return nvalue;
}

char FAR *IntAsText(struct cw_status *gp, struct param FAR *par)
{
  int i;
  int arg = 0;
  char FAR *funcname = "intAsText", FAR *res;

  i = (int) getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if ((res = CalMalloc (200)) == NULL) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    exit (NOT_OK);
  }
  sprintf (res, "%d", i);
  return res;
}

long Int2Text(struct cw_status *gp, struct param FAR *par) 
{
/* set parameter1 to the text containing a single character as specified
 * by the long value parameter2
 */
  char FAR *par1;
  struct param FAR *p;
  struct oper FAR *op;
  char FAR *newpar1;

   /*  get text pointer */
  par1 = (char FAR *) t_par_value(gp, par);
  p = (struct param FAR *) next(par);
  if(par_type(gp, p) != T_INT){
    paramError(gp, p, ErrINTTOTNAI);
    return 0;
  }

  if (!par1) {
    paramError(gp, par, ErrINTTOTNTP);
    return(0);
  }
  /*  check type of par1, don't allow changes to constants */
  op = par->exp;
  if (op->flag == TEXTCONST) {
     paramError(gp, par, ErrINTTOTCTC);
     return(0);
  }

  /*  allocate new string for par1 */
  newpar1 = (char FAR *) CalMalloc(22); /* no integer will have more 
				    than 20 decimals */

  sprintf(newpar1, "%ld", par_value(gp, p)); 

  /*  deallocate par1   */
  CalFree(par1);

  t_set_par_value(gp, par, newpar1);
  
  return 1;
}

long TextAsInt (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0, i;
  char FAR *funcname = "textAsInt", FAR *t;

  t = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  
  sscanf (t, "%i", &i);
  
  CalFree(t);
  return (long)i;
}

long text2Int (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "text2int", FAR *t;
  char tmp[80];

  t = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  strncpy(tmp, t, 80);
  CalFree(t);
  return (long)atoi (tmp);
}

double text2Float(struct cw_status *gp, struct param *par)
{
  int arg = 0;
  char FAR *funcname = "getFloat", FAR *t;
  double rval=0.0;

  t = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

/*   sscanf(t, "%*[^0-9.] %lf", &rval); */
  sscanf(t, "%lf", &rval);
  CalFree(t);
  return rval;
}

long textextents (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0, a, d, w, fontexists = 0;
  char FAR *funcname = "textExtents", FAR *t, FAR *fn;
  struct param FAR *apar, FAR *dpar, FAR *wpar;

  t = getTextPar (gp, &par, ++arg, funcname); 
  fn = getTextPar (gp, &par, ++arg, funcname);
  apar = par;
  getIntPar (gp, &par, ++arg, funcname);
  dpar = par;
  getIntPar (gp, &par, ++arg, funcname);
  wpar = par;
  getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  fextents (gp, t, fn, &a, &d, &w);
  if (fn) {
    /* This is barely correct, is it ? */
    fontexists = TRUE;
    CalFree (fn);
  }
  set_par_value (gp, apar, a);
  set_par_value (gp, dpar, d);
  set_par_value (gp, wpar, w);

  CalFree(t);

  return fontexists;
}

#define MAXEXP 308
#define MAXFRACT 39 /* 2^128 ensure 128 bit prec. */

struct sp_str {
  int left; 
  int sign;
  int space;
  int zeropad;
  int alternate;
  int precision;
  long width;
  char *res;
  int rlen;
  int credit; /* Number of unused characters in res */
};

void use_credit(struct cw_status *gp, int len, char **str, struct sp_str *sps)
{
    char FAR *t;
    if(sps->credit < len){
	int offset = *str - sps->res;
	t = CalRealloc(sps->res, sps->rlen + len);
	sps->res = t;
	sps->rlen += len;
	*str = sps->res + offset;
	sps->credit = 0;
    } else
	sps->credit -= len;
} 

#define LOWER 1
void do_number(struct cw_status *gp, char **str, int val, int base,
	       int type, struct sp_str *sps)
{
    char FAR *digits= (type == LOWER) ? "0123456789abcdef" : "0123456789ABCDEF";
    char pad = sps->zeropad ? '0' : ' ';
    char sign = 0;
    char tmp[36];
    int i = 0;
    int len=0;
    char FAR *buf;
    if(val < 0){
	sign = '-';
	val = -val;
    } else if(sps->sign){
	sign = '+';
    }
    if(sign) 
	len++;

    if(sps->alternate)
	if(base == 8) 
	    len++;
	else if(base == 16)
	    len += 2;

    /* Generate string in reverse order */
    if(val == 0)
	tmp[i++] = '0';
    else while(val != 0){
	tmp[i++] = digits[val % base];
	val = val / base;
    }
    if(i > sps->precision) 
	sps->precision = i;
    len += sps->precision;

    if(len < sps->width){
	sps->width = sps->width - len;
	len += sps->width;
    }
    use_credit(gp, len, str, sps);
    buf = *str;
    if(!sps->zeropad && !sps->left)
	while(sps->width-- > 0)
	    *buf++ = '0';
	
    if(sign)
	*buf++ = sign;

    if(sps->alternate)
	if(base == 8)
	    *buf++ = '0';
	else if(base == 16){
	    *buf++ = '0';
	    *buf++ = (type == LOWER) ? 'x' : 'X';
	}
    if(!sps->left)
	while(sps->width-- > 0)
	    *buf++ = pad;
    while(i < sps->precision--)
	*buf++ = '0';
    while(i-- > 0)
	*buf++ = tmp[i];
    while(sps->width-- > 0)
	*buf++ = ' ';
    *str = buf;
}

#define	todigit(c)	((c) - '0')
#define	tochar(n)	((n) + '0')
static char FAR *
round(struct cw_status *gp, double fract, int *exp, register char *start,
      register char *end, char ch, char *signp, struct sp_str *sps)
{
	double tmp;

	if (fract)
		(void)modf(fract * 10, &tmp);
	else
		tmp = todigit(ch);
	if (tmp > 4)
		for (;; --end) {
			if (*end == '.')
				--end;
			if (++*end <= '9')
				break;
			*end = '0';
			if (end == start) {
				if (exp) {	/* e/E; increment exponent */
					*end = '1';
					++*exp;
				}
				else {		/* f; add extra digit */
					*--end = '1';
					--start;
				}
				break;
			}
		}
	/* ``"%.3f", (double)-0.0004'' gives you a negative 0. */
	else if (*signp == '-')
		for (;; --end) {
			if (*end == '.')
				--end;
			if (*end != '0')
				break;
			if (end == start)
				*signp = 0;
		}
	return(start);
}

static char FAR *
exponent(struct cw_status *gp, register char *p, register int exp,
	 u_char fmtch, struct sp_str *sps)
{
	register char FAR *t;
	char expbuf[MAXEXP];
	
	*p++ = fmtch;
	if (exp < 0) {
		exp = -exp;
		*p++ = '-';
	}
	else
		*p++ = '+';
	t = expbuf + MAXEXP;
	if (exp > 9) {
		do {
			*--t = tochar(exp % 10);
		} while ((exp /= 10) > 9);
		*--t = tochar(exp);
		for (; t < expbuf + MAXEXP; *p++ = *t++);
	}
	else {
		*p++ = '0';
		*p++ = tochar(exp);
	}
	return(p);
}

void do_float(struct cw_status *gp, char FAR **inbuf, double fval,
	      unsigned char mode, struct sp_str *sps)
{
    int fprec=0, fsize;
    char softsign;
    char buf[MAXFRACT+MAXEXP+1];  /* Maximum size */
    double integer, fract, tmp;
    int dotrim, expcnt, gformat;
    char FAR *p, FAR *t, FAR *startp, FAR *endp;
    char FAR *str = *inbuf;
    int i;

    startp = buf;
    endp = buf + sizeof(buf);
    buf[0] = 0;
    dotrim = expcnt = gformat = 0;

    /* Keep reasonable precision */
    if(sps->precision > MAXFRACT) {
	if((mode != 'g' && mode != 'G') || sps->alternate)
	    fprec = sps->precision - MAXFRACT;
	sps->precision = MAXFRACT;
    } if(sps->precision == -1) /* Default precision */
	sps->precision = 6;

    /* avoid negative 0 */
    if(fval < 0){
	softsign = '-';
	fval = -fval;
    } else
	softsign = 0;

    fract = modf(fval, &integer);
    t = ++startp;

    for (p = endp - 1; integer; ++expcnt) {
	tmp = modf(integer / 10, &integer);
	*p-- = tochar((int)((tmp + .01) * 10));
    }
    switch(mode) {
    case 'f':
		/* reverse integer into beginning of buffer */
		if (expcnt)
			for (; ++p < endp; *t++ = *p);
		else
			*t++ = '0';
		/*
		 * if precision required or sps->alternate flag set, add in a
		 * decimal point.
		 */
		if (sps->precision || sps->alternate)
			*t++ = '.';
		/* if requires more precision and some fraction left */
		if (fract) {
			if (sps->precision)
				do {
					fract = modf(fract * 10, &tmp);
					*t++ = tochar((int)tmp);
				} while (--sps->precision && fract);
			if (fract)
				startp = round(gp, fract, (int FAR *)NULL,
					       startp, t - 1, (char)0,
					       &softsign, sps);
		}
		for (; sps->precision--; *t++ = '0');
		break;
	case 'e':
	case 'E':
eformat:	if (expcnt) {
			*t++ = *++p;
			if (sps->precision || sps->alternate)
				*t++ = '.';
			/* if requires more precision and some integer left */
			for (; sps->precision && ++p < endp; --sps->precision)
				*t++ = *p;
			/*
			 * if done precision and more of the integer component,
			 * round using it; adjust fract so we don't re-round
			 * later.
			 */
			if (!sps->precision && ++p < endp) {
				fract = 0;
				startp = round(gp, (double)0, &expcnt, startp,
				    t - 1, *p, &softsign, sps);
			}
			/* adjust expcnt for digit in front of decimal */
			--expcnt;
		}
		/* until first fractional digit, decrement exponent */
		else if (fract) {
			/* adjust expcnt for digit in front of decimal */
			for (expcnt = -1;; --expcnt) {
				fract = modf(fract * 10, &tmp);
				if (tmp)
					break;
			}
			*t++ = tochar((int)tmp);
			if (sps->precision || sps->alternate)
				*t++ = '.';
		}
		else {
			*t++ = '0';
			if (sps->precision || sps->alternate)
				*t++ = '.';
		}
		/* if requires more precision and some fraction left */
		if (fract) {
			if (sps->precision)
				do {
					fract = modf(fract * 10, &tmp);
					*t++ = tochar((int)tmp);
				} while (--sps->precision && fract);
			if (fract)
				startp = round(gp, fract, &expcnt, startp,
				    t - 1, (char)0, &softsign, sps);
		}
		/* if requires more precision */
		for (; sps->precision--; *t++ = '0');

		/* unless sps->alternate flag, trim any g/G format trailing 0's */
		if (gformat && !(sps->alternate)) {
			while (t > startp && *--t == '0');
			if (*t == '.')
				--t;
			++t;
		}
		t = exponent(gp, t, expcnt, mode, sps);
		break;
	case 'g':
	case 'G':
		/* a precision of 0 is treated as a precision of 1. */
		if (!sps->precision)
			++sps->precision;
		/*
		 * ``The style used depends on the value converted; style e
		 * will be used only if the exponent resulting from the
		 * conversion is less than -4 or greater than the precision.''
		 *	-- ANSI X3J11
		 */
		if ((expcnt > sps->precision) || 
		    (!expcnt && fract && fract < .0001)) {
			/*
			 * g/G format counts "significant digits, not digits of
			 * precision; for the e/E format, this just causes an
			 * off-by-one problem, i.e. g/G considers the digit
			 * before the decimal point significant and e/E doesn't
			 * count it as precision.
			 */
			--sps->precision;
			mode -= 2;		/* G->E, g->e */
			gformat = 1;
			goto eformat;
		}
		/*
		 * reverse integer into beginning of buffer,
		 * note, decrement precision
		 */
		if (expcnt)
			for (; ++p < endp; *t++ = *p, --sps->precision);
		else
			*t++ = '0';
		/*
		 * if precision required or sps->alternate flag set, add in a
		 * decimal point.  If no digits yet, add in leading 0.
		 */
		if (sps->precision || sps->alternate) {
			dotrim = 1;
			*t++ = '.';
		}
		else
			dotrim = 0;
		/* if requires more precision and some fraction left */
		if (fract) {
			if (sps->precision) {
				do {
					fract = modf(fract * 10, &tmp);
					*t++ = tochar((int)tmp);
				} while(!tmp);
				while (--sps->precision && fract) {
					fract = modf(fract * 10, &tmp);
					*t++ = tochar((int)tmp);
				}
			}
			if (fract)
				startp = round(gp, fract, (int FAR *)NULL,
					       startp, t - 1, (char)0,
					       &softsign, sps);
		}
		/* sps->alternate format, adds 0's for precision, else trim 0's */
		if (sps->alternate)
			for (; sps->precision--; *t++ = '0');
		else if (dotrim) {
			while (t > startp && *--t == '0');
			if (*t != '.')
				++t;
		}
	}
    fsize = fprec + t - startp;
    if(softsign)
	fsize++;
    
    use_credit(gp, MAX(fsize, sps->width), inbuf, sps);
    str = *inbuf;
    if (!(sps->zeropad || sps->left) && sps->width)
	for (i = fsize; i < sps->width; i++)
	    *str++ = ' ';

    /* prefix */
    if (softsign)
	*str++ = '-';

    /* right-adjusting zero padding */
    if (sps->zeropad && !sps->left)
	for (i = fsize; i < sps->width; i++)
	    *str++ = '0';
#ifdef UNIX
    bcopy(buf+1, str, t - startp);
#else
	memmove(str, buf+1, t - startp);
#endif 
    str += t - startp;

    /* trailing f.p. zeroes */
    
    while (--fprec >= 0)
	*str++ = '0';

    /* left-adjusting padding (always blank) */
    if (sps->left)
	for(i = fsize; i < sps->width; i++)
	    *str++ = ' ';
    *inbuf = str;
}

char FAR *Sprintf(struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "sprintf";
  struct param FAR *p = par;
  char FAR *orig_fmt, FAR *fmt, FAR *t;
  char FAR *str;
  int val;
  double fval;
  char FAR *sp, *oldsp;
  int len;
  int i=0;
  int flen;
  char mess[128];
  struct sp_str *sps, sp_struct;

  sps = &sp_struct;
  memset(sps, 0, sizeof(struct sp_str));
  
  orig_fmt = fmt = getTextPar(gp, &p, ++arg, funcname);
  flen = strlen(fmt);
  str = sps->res = (char FAR *) CalMalloc(flen+1);
  sps->rlen = flen + 1;
  sps->credit = 0;

  while((*str = *fmt) != '\0'){
    for(;(*str = *fmt) && *fmt != '%';str++,fmt++);
    if(!*fmt)
      break;
    
    sps->left=sps->sign=sps->space=sps->zeropad=sps->alternate=0;
    /* Read flags */
    while(1){
      sps->credit++;
      switch(*(++fmt)){
      case '-':
	sps->left = 1;
	break;
      case '+':
	sps->sign = 1;
	break;
      case ' ':
	sps->space = 1;
	break;
      case '0':
	sps->zeropad = 1;
	break;
      case '#':
	sps->alternate = 1;
	break;
      default:
	goto L_WIDTH;
      }
    }
    L_WIDTH:
    sps->width=0;
    if(isdigit(*fmt)){
      sps->width = strtol(fmt, &t, 10);
      sps->credit += t - fmt;
      fmt = t;
    }
    sps->precision = -1;
    if(*fmt == '.' && isdigit(*(++fmt))){
      sps->credit++;
      sps->precision = strtol(fmt, &t, 10);
      sps->credit += t - fmt;
      fmt = t;
    }
    sps->credit++;
    switch(*fmt++){
    case 'd':
    case 'i':
      val = getIntPar(gp, &p, ++arg, funcname);
      do_number(gp, &str, val, 10, 0, sps); 
      break;
    case 'o':
    case 'x':
      val = getIntPar(gp, &p, ++arg, funcname);
      do_number(gp, &str, val, 16, LOWER, sps); 
      break;
    case 'X':
      val = getIntPar(gp, &p, ++arg, funcname);
      do_number(gp, &str, val, 16, 0, sps); 
      break;
    case 'u':
      /* no meaning */
      break;
    case 'c':
      val = getIntPar(gp, &p, ++arg, funcname);
      use_credit(gp, MAX(1, sps->width), &str, sps);
      if(!sps->left)
	  while(sps->width--)
	      *str++ = ' ';
      *str++ = val % 256;
      if(sps->left)
	while(sps->width--)
	  *str++ = ' ';
      break;
    case 's':
      sp = getTextPar(gp, &p, ++arg, funcname);
      if(!sp)
	  sp = strdup("<NULL>");
      len = strlen(sp);
      oldsp = sp;
      if(sps->precision < 0)
	  sps->precision = len;
      else if(len > sps->precision)
	  len = sps->precision;
      
      use_credit(gp, MAX(len, sps->width), &str, sps);

      if(!sps->left)
	  while(len < sps->width--)
	      *str++ = ' ';
      for(i = 0;i < len; i++)
	  *str++ = *sp++;
      while(len < sps->width--)
	  *str++ = ' ';
      CalFree(oldsp);
      break;
    case 'f':
    case 'e':
    case 'E':
    case 'g':
    case 'G':
      fval = getFltPar(gp, &p, ++arg, funcname);
      do_float(gp, &str, fval, *(fmt-1), sps);
      break;
    case 'p':
      break;
    case 'n':
      if(p == NULL){
	  sprintf (mess, ErrFUNCMISPAR, funcname, arg);
	  errorMsg(gp, 1, mess);
	  CalFree(orig_fmt);
	  return sps->res;
      }
      set_par_value(gp, p, str - sps->res);
      p = next(p);
      ++arg;
      break;
    case '%':

    default:
      break;
    }
  }
  CalFree(orig_fmt);
  return sps->res;
}

long RegMatch(struct cw_status *gp, struct param *par)
{
  int arg = 0;
  char *funcname = "regmatch", *str, *pat;
  long so, eo, retval=FALSE;
  int nm = 1;
  regmatch_t pm[1];
  regex_t preg;

  str = getTextPar (gp, &par, ++arg, funcname);
  pat = getTextPar (gp, &par, ++arg, funcname);
  regcomp(&preg, pat, REG_EXTENDED);

  if(!regexec(&preg, str, nm, pm, REG_NOTBOL)){
    so = pm[0].rm_so;
    eo = pm[0].rm_eo;
    set_par_value(gp, par, so);
    par = next(par);
    set_par_value(gp, par, eo);
    retval = TRUE;
  }
  CalFree(str);
  CalFree(pat);
  return retval;
}

char *SubString(struct cw_status *gp, struct param *par)
{
  int arg = 0;
  char *funcname = "substr", *str, *buf;
  long so, eo;
  char errormsg[256];
  struct param *p1;

  p1 = par;
  str = getTextPar (gp, &par, ++arg, funcname);
  so = getIntPar (gp, &par, ++arg, funcname);
  eo = getIntPar(gp, &par, ++arg, funcname);
/*   printf("%d - %d : len %d\n", so, eo, strlen(str)); */
  if(so > eo || eo > (long) strlen(str) || so < 0){
    sprintf(errormsg, "Illegal substring specification on line %d(%ld-%ld:%d)", 
	    p1->line, so, eo, strlen(str));
    errorMsg(gp, 0, errormsg);
  } else {
    buf = CalMalloc(eo - so + 1);
    if(buf == NULL){
      errorMsg(gp, 1, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    memcpy(buf, str+so, eo - so);
    buf[eo - so] = '\0';
    CalFree(str);
    str = buf;
  } 
  
  return str;
}




