/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no), 
 *	    Gunnar Rnning (gunnarr@ifi.uio.no)
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "candle.h"
#include "const.h"
#include "fast_lis.h"
#include "simulate.h"
#include "error.h"
#include "function.h"

#include "protos/memory.h"
#include "protos/simulate.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"

extern int left_type(struct cw_status *gp, struct oper FAR *op);
extern int right_type(struct cw_status *gp, struct oper FAR *op);
extern void op_debug(struct cw_status *gp, struct oper FAR *op);

long Pow(struct cw_status *gp, long b, long e) {
long ret,i;

  ret = 1;
  if (e > 0){
    for (i = 1; i <= e;i++)
      ret = ret * b;
  }
  return(ret);
}

double f_Pow(struct cw_status *gp, double b, double e) 
{
  double ret;
  ret = pow (b, e);
  return ret;
}

static long *elp_in_array(struct cw_status *gp, struct intarray FAR *array) 
{
  long i=0, FAR *ind, dim=0, index=0;
  struct vinst FAR *vi;
  struct param FAR *par;
  char errormsg[256];
#ifdef DEBUG
  if(array->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal array->type in elp_in_array.\n");
  }
#endif

  vi = array->which.vi;
  

  par = array->indeks;
  while(dim < vi->parent->ndim){
    if(par != NULL)
      i = par_value(gp, par);
    else {
      sprintf(errormsg, ErrTOOFEWIND, 
	      vi->parent->name);
      errorMsg(gp, 1, errormsg);
      return NULL;
    }
    if(i < 0 || i >= (long) vi->size[dim]){
      sprintf(errormsg, ErrARROUTBOU,
	      vi->parent->name, i);
      paramError(gp, par, errormsg);
      return NULL;
    }
    index += i * vi->size[dim+vi->parent->ndim]; 
    par = (struct param FAR *)next(par);
    dim++;
  }
  ind = vi->value.elem;
  if (ind) {
    ind += index;
    return(ind);
  } else
    return NULL;
}

static double *elp_fl_array(struct cw_status *gp, struct intarray FAR *array) 
{
  long i=0, dim=0, index=0;
  double *ind;
  struct vinst FAR *vi;
  struct param FAR *par;
  char errormsg[256];
#ifdef DEBUG
  if(array->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal array->type in elp_in_array.\n");
  }
#endif

  vi = array->which.vi;
  

  par = array->indeks;
  while(dim < vi->parent->ndim){
    if(par != NULL)
      i = par_value(gp, par);
    else {
      sprintf(errormsg, ErrTOOFEWIND, 
	      vi->parent->name);
      errorMsg(gp, 1, errormsg);
      return NULL;
    }
    if(i < 0 || i >= (long) vi->size[dim]){
      sprintf(errormsg, ErrARROUTBOU,
	      vi->parent->name, i);
      paramError(gp, par, errormsg);
      return NULL;
    }
    index += i * vi->size[dim+vi->parent->ndim]; 
    par = (struct param FAR *)next(par);
    dim++;
  }
  ind = vi->value.void_el;
  if (ind) {
    ind = ind + index;
    return(ind);
  } else
    return NULL;
}

static double mod_op(struct cw_status *gp, struct oper FAR *op, double val)
{
  long *lval;
  double *fval;
  double retval;
  char errormsg[256];
#ifdef DEBUG
  if(op->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal op->type in inc_op.\n");
  }
#endif
  switch(op->flag){
  case  INTVAR : 
    switch(op->type){
#ifdef DEBUG
    case DO_VDECL :
      retval = op->left.vd->vi->value.value += val;
      update_changed_attr(gp, op->left.vd->vi);
      break;
#endif
    case DO_VINST :
      retval = op->left.vi->value.value += (long)val;
      update_changed_attr(gp, op->left.vi);
      break;
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return 0;
    }
    break;
  case INTARRAY :
    lval = elp_in_array(gp, op->left.array);
    if(lval)
      retval = (*lval) += (long)val;
    switch(op->left.array->type){
#ifdef DEBUG
    case DO_VDECL :
      update_changed_attr(gp, op->left.array->which.vd->vi);
      break;
#endif
    case DO_VINST :
      update_changed_attr(gp, op->left.array->which.vi);
      break;
    }
    break;
  case  FLOATVAR : 
    switch(op->type){
#ifdef DEBUG
    case DO_VDECL :
      retval = op->left.vd->vi->value.fvalue += val;
      update_changed_attr(gp, op->left.vd->vi);
      break;
#endif
    case DO_VINST :
      retval = op->left.vi->value.fvalue += val;
      update_changed_attr(gp, op->left.vi);
      break;
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return 0;
    }
    break;
  case FLOATARRAY :
    fval = elp_fl_array(gp, op->left.array);
    if(fval)
      retval = (*fval) += val;
    switch(op->left.array->type){
#ifdef DEBUG
    case DO_VDECL :
      update_changed_attr(gp, op->left.array->which.vd->vi);
      break;
#endif
    case DO_VINST :
      update_changed_attr(gp, op->left.array->which.vi);
      break;
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return 0;
    }
    break;
  }
  return retval;
}

static double set_op(struct cw_status *gp, struct oper FAR *op, double val)
{
  long *lval;
  double *fval;
  double retval;
  char errormsg[256];
#ifdef DEBUG
  if(op->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal op->type in inc_op.\n");
  }
#endif
  switch(op->flag){
  case  INTVAR : 
    switch(op->type){
#ifdef DEBUG
    case DO_VDECL :
      retval = op->left.vd->vi->value.value = val;
      update_changed_attr(gp, op->left.vd->vi);
      break;
#endif
    case DO_VINST :
      retval = op->left.vi->value.value = (long)val;
      update_changed_attr(gp, op->left.vi);
      break;
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return 0;
    }
    break;
  case INTARRAY :
    lval = elp_in_array(gp, op->left.array);
    if(lval)
      retval = (*lval) = (long)val;
    switch(op->left.array->type){
#ifdef DEBUG
    case DO_VDECL :
      update_changed_attr(gp, op->left.array->which.vd->vi);
      break;
#endif
    case DO_VINST :
      update_changed_attr(gp, op->left.array->which.vi);
      break;
    }
    break;
  case  FLOATVAR : 
    switch(op->type){
#ifdef DEBUG
    case DO_VDECL :
      retval = op->left.vd->vi->value.fvalue = val;
      update_changed_attr(gp, op->left.vd->vi);
      break;
#endif
    case DO_VINST :
      retval = op->left.vi->value.fvalue = val;
      update_changed_attr(gp, op->left.vi);
      break;
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return 0;
    }
    break;
  case FLOATARRAY :
    fval = elp_fl_array(gp, op->left.array);
    if(fval)
      retval = (*fval) = val;
    switch(op->left.array->type){
#ifdef DEBUG
    case DO_VDECL :
      update_changed_attr(gp, op->left.array->which.vd->vi);
      break;
#endif
    case DO_VINST :
      update_changed_attr(gp, op->left.array->which.vi);
      break;
    default :
      sprintf(errormsg, ErrINTILLMOP); 
      errorMsg(gp, 1, errormsg);
      return 0;
    }
    break;
  }
  return retval;
}

long l_preplus(struct cw_status *gp, struct oper *op, long lval, long rval) 
{
  mod_op(gp, op->left.op, 1);
  return lval+1;
}

long l_postplus(struct cw_status *gp, struct oper *op, long lval, long rval) 
{
  mod_op(gp, op->left.op, 1);
  return lval;
}

long l_preminus(struct cw_status *gp, struct oper *op, long lval, long rval) 
{
  mod_op(gp, op->left.op, -1);
  return lval-1;
}

long l_postminus(struct cw_status *gp, struct oper *op, long lval, long rval) 
{
  mod_op(gp, op->left.op, -1);
  return lval;
}


long l_exprval(struct cw_status *gp, struct oper *op, long lval, long rval) 
{
  return lval;
}
double r_exprval(struct cw_status *gp, struct oper *op, double lval,
		 double rval) 
{
  return lval;
}
char *t_exprval(struct cw_status *gp, struct oper *op, char *lval, char *rval) 
{
  return lval;
}

long l_not(struct cw_status *gp, struct oper FAR *op, long lval, long rval) {
  return(! lval);
}

double r_not(struct cw_status *gp, struct oper *op, double lval, double rval) 
{ 
  if(lval != 0.0)
    return ((double) 1.0);
  else 
    return((double) 0.0);
}

long l_unadd(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval);
}

double r_unadd(struct cw_status *gp, struct oper *op, double lval,
	       double rval) 
{ 
  return(lval); 
}

long l_unsub(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(- lval);
}

double r_unsub(struct cw_status *gp, struct oper *op, double lval,
	       double rval) 
{ 
  return(- lval); 
}

long l_pow(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(Pow(gp, lval,rval));
}

double r_pow(struct cw_status *gp, struct oper *op, double lval, double rval) 
{ 
  return(f_Pow(gp, lval, rval)); 
}

long l_mul(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  return(lval * rval);
}

double r_mul(struct cw_status *gp, struct oper *op, double lval, double rval)
{ 
  return(lval * rval); 
}


double l_div(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[256];
  if (rval == 0) {
    sprintf(errormsg, "Error: Division by zero in line %d\n", op->line);
    errorMsg(gp, 1, errormsg);
    return((long) 2);
  }
  return(lval / rval);
}

double r_div(struct cw_status *gp, struct oper *op, double lval, double rval)
{
  char errormsg[256];
  if (rval == 0) {
    sprintf(errormsg, "Error: Division by zero in line %d\n", op->line);
    errorMsg(gp, 1, errormsg);
    return(2.0F);
  }
  return(lval/rval);
}

long l_rest(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[256];
  if (rval == (long) 0) {
    sprintf(errormsg, "Error: Division by zero in line %d\n", op->line);
    errorMsg(gp, 1, errormsg);
    return(1);
  }
  return(lval % rval);
}

double r_rest(struct cw_status *gp, struct oper *op, double lval, double rval) 
{ 
  return((double) 0.0); 
}

long l_add(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval + rval);
}

double r_add(struct cw_status *gp, struct oper *op, double lval, double rval) 
{ 
  return(lval + rval); 
}

char *t_add(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{ 
  char *buf;
  buf = CalMalloc(strlen(lval) + strlen(rval) + 1);
  if(buf == NULL){
    errorMsg(gp, 1, ErrNOMOREMEM);
    c_exit(gp, 1);
  }
  strcpy(buf, lval);
  
  return strcat(buf, rval); 
}


long l_sub(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval - rval);
}

double r_sub(struct cw_status *gp, struct oper *op, double lval, double rval) 
{
  return(lval - rval);
}

long l_rshift(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval>>rval);
}

double r_rshift(struct cw_status *gp, struct oper *op, double lval,
		double rval) 
{ 
  return((double) 0.0); 
}

long l_lshift(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval<<rval);
}

double r_lshift(struct cw_status *gp, struct oper *op, double lval,
		double rval) 
{ 
  return((double) 0.0); 
}

long l_lt(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  if (lval < rval) 
    return((long)1); 
  else 
    return((long)0);
}

double r_lt(struct cw_status *gp, struct oper *op, double lval, double rval) 
{ 
  if(lval < rval)
    return ((double) 1.0);
  else
    return((double) 0.0); 
}

long t_lt(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{
  if (strcmp(lval,rval) < 0) 
    return 1; 
  else 
    return 0;
}


long l_le(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  if (lval <= rval) 
    return((long)1); 
  else 
    return((long)0);
}

double r_le(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  if(lval <= rval)
    return((double) 1.0);
  else
    return((double) 0.0); 
}

long t_le(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{
  if (strcmp(lval,rval) <= 0) 
    return 1; 
  else 
    return 0;
}

long l_gt(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  if (lval > rval) 
    return((long)1); 
  else 
    return((long)0);
}

double r_gt(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  if(lval > rval)
    return((double) 1.0);
  else
    return((double) 0.0); 
}

long t_gt(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{
  if (strcmp(lval,rval) > 0) 
    return 1; 
  else 
    return 0;
}


long l_ge(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  if (lval >= rval) 
    return((long)1); 
  else 
    return((long)0);
}

double r_ge(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  if (lval >= rval)
    return((double) 1.0);
  else
    return((double) 0.0); 
}

long t_ge(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{
  if (strcmp(lval,rval) >= 0) 
    return 1; 
  else 
    return 0;
}

long l_eq(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  if (lval == rval) 
    return((long)1); 
  else 
    return((long)0);
}

long t_eq(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{
  if (!strcmp(lval,rval)) 
    return 1; 
  else 
    return 0;
}


double r_eq(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  if (lval == rval) 
    return((double) 1.0); 
  else 
    return((double) 0.0);
}

long l_ne(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  if (lval != rval) 
    return((long)1); 
  else 
    return((long)0);
}

double r_ne(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  if (lval != rval) 
    return((double) 1.0); 
  else 
    return((double) 0.0);
}

long t_ne(struct cw_status *gp, struct oper FAR *op, char *lval, char *rval) 
{
  if (!strcmp(lval,rval)) 
    return 0; 
  else 
    return 1;
}

long l_bitand(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval&rval);
}

double r_bitand(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{
 return((long)lval & (long)rval); 
}

long l_bitor(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return(lval|rval);
}

double r_bitor(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  return((long)lval|(long)rval); 
}

long l_and(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return((long) lval && rval);
}

double r_and(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  return((double) lval && rval); 
}

long l_or(struct cw_status *gp, struct oper FAR *op, long lval, long rval) 
{
  return((long) lval || rval);
}

double r_or(struct cw_status *gp, struct oper FAR *op, double lval, double rval) 
{ 
  return((double) lval || rval); 
}

static long compute_index(struct cw_status *gp, struct vinst FAR *vi, struct param FAR *par)
{
  long dim, index, i;
  char errormsg[128];

  dim = 0;
  index = 0;
  while(dim < vi->parent->ndim){
    if(par != NULL)
      i = par_value(gp, par);
    else {
      sprintf(errormsg, ErrTOOFEWIND, 
	      vi->parent->name);
      errorMsg(gp, 1, errormsg);
      return (NOT_OK);
    }
    if(i < 0 || i >= (long) vi->size[dim]){
      sprintf(errormsg, 
	      ErrARROUTBOU,
	      vi->parent->name, i);
      paramError(gp, par, errormsg);
      return (NOT_OK);
    }
    index += i * (long) vi->size[dim+vi->parent->ndim]; 
    par = (struct param FAR *)next(par);
    dim++;
  }
  return index;
}

long l_ass(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  struct vinst *vi;
  char errormsg[128];
  long *in;
  struct param *par;

  switch(op->left.op->flag){
  case INTVAR:
    vi = op->left.op->left.vi;
    vi->value.value = rval;
    break;
  case INTARRAY:
    vi = op->left.op->left.array->which.vi;
    par = op->left.op->left.array->indeks;
    in = vi->value.elem + compute_index(gp, vi, par);
    *in = rval;
    break;
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return rval;
  }
  update_changed_attr(gp, vi);
  return rval;
}

long l_assplus(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)mod_op(gp, op->left.op, rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assminus(struct cw_status *gp, struct oper FAR *op, long lval, 
		long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)mod_op(gp, op->left.op, - rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assdiv(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op) / rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assmod(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op)%rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assmult(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op)*rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assband(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op) & rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_assband", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assbor(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op) | rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_assbor", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}


long l_asslshift(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op) << rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

long l_assrshift(struct cw_status *gp, struct oper FAR *op, long lval, long rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case INTVAR:
  case INTARRAY:
    return (long)set_op(gp, op->left.op, calculate(gp, op->left.op) >> rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

double r_ass(struct cw_status *gp, struct oper FAR *op, double lval, double rval)
{
  struct vinst *vi;
  char errormsg[128];
  double *in;
  struct param *par;

  switch(op->left.op->flag){
  case FLOATVAR:
    vi = op->left.op->left.vi;
    vi->value.fvalue = rval;
    break;
  case FLOATARRAY:
    vi = op->left.op->left.array->which.vi;
    par = op->left.op->left.array->indeks;
    in = vi->value.felem + compute_index(gp, vi, par);
    *in = rval;
    break;
  default:
    sprintf(errormsg, "Error: Illegal lvalue in r_ass");
    errorMsg(gp, 1, errormsg);
    break;
  }
  update_changed_attr(gp, vi);
  return rval;
}

double r_assplus(struct cw_status *gp, struct oper FAR *op, double lval, 
		 double rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case FLOATVAR:
  case FLOATARRAY:
    return mod_op(gp, op->left.op, rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

double r_assminus(struct cw_status *gp, struct oper FAR *op, double lval, 
		double rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case FLOATVAR:
  case FLOATARRAY:
    return mod_op(gp, op->left.op, - rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

double r_assdiv(struct cw_status *gp, struct oper FAR *op, double lval, 
	      double rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case FLOATVAR:
  case FLOATARRAY:
    return set_op(gp, op->left.op, calculate(gp, op->left.op) / rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

double r_assmod(struct cw_status *gp, struct oper FAR *op, double lval, 
		double rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case FLOATVAR:
  case FLOATARRAY:
    return set_op(gp, op->left.op, fmod(calculate(gp, op->left.op),rval));
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}

double r_assmult(struct cw_status *gp, struct oper FAR *op, double lval, 
	       double rval)
{
  char errormsg[128];
  switch(op->left.op->flag){
  case FLOATVAR:
  case FLOATARRAY:
    return set_op(gp, op->left.op, calculate(gp, op->left.op)*rval);
  default:
    sprintf(errormsg, "Error line %d: Illegal lvalue in l_ass", op->line);
    errorMsg(gp, 1, errormsg);
    return 0;
  }
}


char FAR *t_ass(struct cw_status *gp, struct oper FAR *op, char FAR *lval, char FAR *rval)
{
  struct vinst *vi;
  char errormsg[128];
  char **in;
  struct param *par;

  switch(op->left.op->flag){
  case TEXTVAR:
    vi = op->left.op->left.vi;
    CalFree(vi->value.text);
    vi->value.text = rval;
    break;
  case TEXTARRAY:
    vi = op->left.op->left.array->which.vi;
    par = op->left.op->left.array->indeks;
    in = vi->value.telem + compute_index(gp, vi, par);
    CalFree(*in);
    *in = rval;
    break;
  default:
    sprintf(errormsg, "Error at line %d: Illegal lvalue in text assignment", op->line);
    errorMsg(gp, 1, errormsg);
    break;
  }
  update_changed_attr(gp, vi);
  return strdup(rval);
}

#ifdef FASTEXPR
long l_intval(struct cw_status *gp, struct oper *op)
{
  return op->left.val;
}

double r_intval(struct cw_status *gp, struct oper *op)
{
  return op->left.val;
}

long l_floatval(struct cw_status *gp, struct oper *op)
{
  return op->left.rval;
}

double r_floatval(struct cw_status *gp, struct oper *op)
{
  return op->left.rval;
}

char *t_textval(struct cw_status *gp, struct oper *op)
{
  return strdup(op->left.tval);
}

long l_add(struct cw_status *gp, struct oper *op)
{
  return (op->left.op->calc.l_calc)(gp, op->left.op) + 
    (op->right.op->calc.l_calc)(gp, op->right.op);
}

#endif
