/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no), 
 *	    Gunnar Rnning (gunnarr@ifi.uio.no)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>

#ifdef UNIX
#include <sys/types.h>  
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#endif
#ifdef WIN32
#include <Windows.h>
#include <Mmsystem.h>
#endif

#include "candle.h"
#include "parser.h"
#include "lex.h"
#include "learnuni.h"
#include "simulate.h"
#include "graphic.h"
#include "nodes.h"
#include "input.h"
#include "error.h"
#include "wwwcom.h"

#include "sysproto.h"
#include "protos/memory.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/simulate.h"
#include "protos/region.h"
#include "protos/update.h"
#include "protos/readsim.h"
#include "protos/instance.h"
#include "protos/input.h"
#include "protos/funcutil.h"
#include "protos/freesim.h"
#include "protos/instsim.h"
#include "protos/parser.h"

#define MAX_BEHIND 1000  /* This value means that simulation is
			  allowed to be 10 seconds behind specified time */


extern long calculate(struct cw_status *gp, struct oper FAR *op);
extern double f_calculate(struct cw_status *gp, struct oper FAR *op);
extern char FAR *t_calc_dup(struct cw_status *gp, struct oper FAR *op);


#ifdef X11
long millitime(struct cw_status *gp) { /* Not in funclib */
  struct timeval tp;
  struct timezone tzp;
 
  if(gettimeofday(&tp, &tzp) == -1){
    errorMsg(gp, 1, ErrSYSTIMDAY);
    return(-1);
  }
  return (tp.tv_sec * 1000 + tp.tv_usec/1000);
}
#elif WIN32
long millitime (struct cw_status *gp)
{
  SYSTEMTIME st;

  GetSystemTime((LPSYSTEMTIME)&st);
  return (st.wSecond*1000+st.wMilliseconds);
}
#endif

long init_random(struct cw_status *gp, struct param FAR *par)
{
  int i;
  int arg = 0;
  char FAR *funcname = "initRandom";

  i = (int) getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  srandom(i);
  return(TRUE);
}

/***********************************************************/
/*                                        int get_random   */
/* This function return a random number (int) between      */
/* lower- (l) and upper (u) limit.                         */
/* Tested 881026 Yngvar Berg.                              */
/***********************************************************/

long get_random(struct cw_status *gp, struct param FAR *par) 
{
  long l,u;
  int arg = 0;
  char FAR *funcname = "random";

  l = getIntPar (gp, &par, ++arg, funcname);
  u = getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  
  if (u - l > 0){
    if (l < 0){
      l = (-1)*l;
      u = u + 2*l;
      return(l + ((long)random() % (u - l + 1)) - 2*l);
    }else
      return(l + ((long)random() % (u - l + 1)));
  }else
    return(l);
}

/*
 * Check if incoming event i matches event specified in a translation. 
 * Return true if it matches.
 */
static int equal_event(struct cw_status *gp, struct event FAR *i,
		       struct event FAR *t) 
{ /* Not in funclib */
  if(((t->keysym != 0 && i->keysym == t->keysym && 
      i->type == t->type) 
     || (t->keysym == 0 && 
     (i->type == t->type && (i->detail.key == t->detail.key || 
			     t->detail.key == 0))))
      && (t->modifiermask == 0 || 
	  (i->modifiermask | t->modifiermask) == i->modifiermask))
    {
      if(t->transparent)
	return 2;
      else 
	return 1;
    } 
  return 0;
}

static int is_point_insensitive(struct cw_status *gp, struct event *ep)
{
  switch(ep->type){
  case ResizeWindow:
    return TRUE;
  default:
    return FALSE;
  }
}

/*
 * Return true if event matched some translation in the inputobject.
 * The matching translation will be executed.
 */
static int match_event(struct cw_status *gp, struct event *e,
		       struct inpobj *io)
{ /* Not in funclib */
  struct translation FAR *tr;
  struct event FAR *ep;
  int etype;

  if((!io->active || io->active->directval.typint) && 
     (PointInRegion(io->boundingregion, e->x, e->y) || 
      is_point_insensitive(gp, e))){
    for(tr = io->translation->exprval.tl; tr != NULL; tr = next(tr)){
      for(ep = tr->event; ep != NULL; ep = next(ep)){
	etype = equal_event(gp, e, ep);
	if(etype == 1){
	  if(gp->curlunit->fevent == gp->curlunit->levent)
	    gp->curlunit->fevent = gp->curlunit->levent = NULL;
	  else 
	    gp->curlunit->fevent = next(e);
	  remove_at(e);
	  gp->curlunit->last_pr_event = e;
	  simulate(gp, tr->act, NULL);
	  CalFree(e);
	  gp->curlunit->last_pr_event = NULL;
	  return 1;
	} else if(etype == 2){
	  gp->curlunit->last_pr_event = e;
	  simulate(gp, tr->act, NULL);
	  gp->curlunit->last_pr_event = NULL;
	  return 0;
	}
      }
    }
  }
  return 0;
}

#define PointInBB(bb, x1, y1) (x1 >= bb.x && x1 < bb.x + bb.width && \
			     y1 >= bb.y && y1 < bb.y + bb.height)

long input(struct cw_status *gp, struct param FAR *par) 
{
  struct event FAR *e;
  struct levlist FAR *ll, FAR *wll;
  struct inpobj FAR *il;
  struct winobj FAR *wl;
  
  checkPars (gp, par, 0, "input");
  UpdateInps(gp);
L_MATCHED: 
  for(e = gp->curlunit->fevent; e != NULL;e = gp->curlunit->fevent){
    for(ll = last(gp->curlunit->level0);
	ll != NULL;
	ll = prev(ll)){
      for(il = ll->levinps;
	  il != NULL;
	  il = (struct inpobj FAR *) next(il)){
	if(match_event(gp, e, il))
	  goto L_MATCHED;
      }
      for(wl = ll->levwins; wl != NULL; wl = next(wl)){
	if(PointInBB(wl->boundingbox, e->x, e->y)){
	  e->x -= wl->boundingbox.x;
	  e->y -= wl->boundingbox.y;
	  for(wll = last(wl->level0); wll != NULL; wll = prev(wll)){
	    for(il = wll->levinps; il != NULL; 
		il = (struct inpobj FAR *)next(il)){
	      if(match_event(gp, e, il))
		goto L_MATCHED;
	    }
	  }
	  e->x += wl->boundingbox.x;
	  e->y += wl->boundingbox.y;
	}
      }
    }
    gp->curlunit->fevent = next(e);
    remove_at(e);
    CalFree(e);
  }
  return TRUE;
}

long Getc(struct cw_status *gp, struct param FAR *par)
{
  char FAR *t, FAR *tn;
  struct param FAR *p2;
  int arg = 0;
  char FAR *funcname = "getc";
  struct event FAR *event = gp->curlunit->last_pr_event;
  int fd, c;
  char buf;      
  char mess[160];

  fd = (int) getIntPar(gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if(fd < 0){
    sprintf(mess, ErrFILNEGFD, fd);
    errorMsg(gp, 1, mess);
    return -1;
  } else if(fd == 0){
    if(event == NULL || (event->type != KeyDown && event->type != KeyUp))
      return -1; /* Return -1 if not key event */
    c = event->detail.key;
  } else {
    if(Read(fd, &buf, 1) <= 0)
      return -1;
    c = buf;
  } 

  if(par != NULL){
    p2 = par;
    t = getTextPar (gp, &par, ++arg, funcname);
    checkPars (gp, par, arg, funcname);

    if((tn = CalMalloc(strlen(t) + 2)) == NULL)
      errorMsg(gp, 2, ErrNOMOREMEM);
    strcpy(tn, t);
    tn[strlen(t)] = c;
    tn[strlen(t)+1] = 0;
/*     printf("tn %s\n", tn); */
    CalFree(t);
    t_set_par_value(gp, p2, tn);
  }

  return c;
}

long ReadFD(struct cw_status *gp, struct param FAR *par)
{
  struct param *bufpar;
  int arg = 0;
  char FAR *funcname = "read";
  int fd;
  long n, c;
  char *buf;      
  char mess[160];

  fd = (int) getIntPar(gp, &par, ++arg, funcname);
  bufpar = par;
  par = next(par);
  n = getIntPar(gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  
  if(fd <= 0){
    sprintf(mess, ErrFILNEGFD, fd);
    errorMsg(gp, 1, mess);
    return -1;
  } else {
    buf = CalMalloc(n+1);
    if((c = Read(fd, buf, n)) <= 0){
      CalFree(buf);
      return -1;
    }
    buf[c] = '\0';
    t_set_par_value(gp, bufpar, buf);
  } 

  return c;
}

long WriteFD(struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "read";
  int fd;
  long n, c;
  char *buf;      
  char mess[160];

  fd = (int) getIntPar(gp, &par, ++arg, funcname);
  buf = getTextPar(gp, &par, ++arg, funcname);
  n = getIntPar(gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  
  if(fd <= 0){
    sprintf(mess, ErrFILNEGFD, fd);
    errorMsg(gp, 1, mess);
    c = -1;
  } else 
    if((c = Write(fd, buf, n)) <= 0)
      c = -1;

  CalFree(buf);
  return c;
}



char *Post(struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "post";
  char *url, *data;
  char *reply;

  url = getTextPar(gp, &par, ++arg, funcname);
  data = getTextPar(gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
#ifdef WWWENABLE
  PostData(gp, url, data,(long) strlen(data), &reply);
#endif WWWENABLE
  CalFree(url);
  CalFree(data);
  return reply;
}

long Putc(struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "putc";
  int fd;
  char buf;

  fd = (int) getIntPar(gp, &par, ++arg, funcname);
  buf = (int) getIntPar(gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if(fd < 0){
    errorMsg(gp, 1, ErrFILNEGFD);
    return -1;
  } else if(fd == 0){
    return -1;
  } else {
    if(Write(fd, &buf, 1) <= 0)
      return -1;
    return 1;
  } 
}

/* Open a URL for reading and return file descriptor.
 */
long url_open(struct cw_status *gp, struct param FAR *par)
{
  char FAR *url;
  char FAR *mode;
  char FAR *funcname = "open";
  long fd;
  int arg=0;
  
  url = getTextPar(gp, &par, ++arg, funcname);
  if(url == NULL)
    return -1;

  mode = getTextPar(gp, &par, ++arg, funcname);

  if(mode == NULL){
    CalFree(url);
    return -1;
  }
  fd = OpenURL(gp, url, mode);
  CalFree(url);
  CalFree(mode);
  return fd;
}

long url_close(struct cw_status *gp, struct param FAR *par)
{
  return 1;
}

long link_url(struct cw_status *gp, struct param *par)
{
  char FAR *url;
  char FAR *funcname = "link";
  int arg=0;
  
  url = getTextPar(gp, &par, ++arg, funcname);
  checkPars(gp, par, arg, funcname);

  if(url != NULL){
#ifdef WWWENABLE
    LinkUrl(gp, url);
#endif
    CalFree(url);
    return TRUE;
  } else {
    return FALSE;
  }
}

char *absolute_url(struct cw_status *gp, struct param *par)
{
  char *anchorurl, *relativeurl, *absoluteurl;
  char FAR *funcname = "absoluteURL";
  int arg=0;
  
  anchorurl = getTextPar(gp, &par, ++arg, funcname);
  if(anchorurl == NULL)
    return strdup("");
  relativeurl = getTextPar(gp, &par, ++arg, funcname);
  if(relativeurl == NULL){
    CalFree(anchorurl);
    return strdup("");
  }
  checkPars(gp, par, arg, funcname);

#ifdef WWWENABLE
  absoluteurl = AbsoluteURL(anchorurl, relativeurl);
#endif
  CalFree(anchorurl);
  CalFree(relativeurl);
  return absoluteurl;
}

char *content_type(struct cw_status *gp, struct param *par)
{
  char FAR *url;
  char *ctype;
  char FAR *funcname = "contentType";
  int arg=0;
  
  url = getTextPar(gp, &par, ++arg, funcname);
  checkPars(gp, par, arg, funcname);

  if(url != NULL){
#ifdef WWWENABLE
    ctype = ContentType(gp, url);
#endif
    CalFree(url);
    /*    assert (ctype); */
    return strdup (ctype);
  } else {
    return strdup("");
  }
}

long get_char(struct cw_status *gp, struct param FAR *par)
{
  char FAR *t, FAR *tn;
  struct param FAR *p2 = par;
  int arg = 0;
  char FAR *funcname = "getchar";
  struct event FAR *event = gp->curlunit->last_pr_event;
 
  if(event == NULL || (event->type != KeyDown && event->type != KeyUp))
    return 0; /* Return 0 if not key event */
  
  if(par != NULL){
    if(event == NULL) return 0;
    t = getTextPar (gp, &par, ++arg, funcname);
    checkPars (gp, par, arg, funcname);
    if(event->detail.key == '\b'){
      if(strlen(t) > 0){
	t[strlen(t)-1] = 0;
	tn = strdup(t);
      } else {
	CalFree(t);
	goto L_GETC_END;
      }
    } else {
      if((tn = CalMalloc(strlen(t) + 2)) == NULL)
	errorMsg(gp, 2, ErrNOMOREMEM);
      strcpy(tn, t);
      tn[strlen(t)] = event->detail.key;
      tn[strlen(t)+1] = 0;
    }
    CalFree(t);
    t_set_par_value(gp, p2, tn);
  }
 L_GETC_END:
  return event->detail.key;
}

/****************************************************************
**                                                  sim_time   **
**  Return simulated time since start of simulation.  Start of **
**  simulation is stored in gp->curlunit->starttime        **
**  This function is called from sync_input() and may be       **
**  called before entering main loop in simulation.            **
****************************************************************/
long sim_time(struct cw_status *gp, struct param FAR *par) 
{
  checkPars (gp, par, 0, "elapsedTime");
  return(millitime(gp) - gp->curlunit->starttime);
}

long wait_for_input(struct cw_status *gp, struct param FAR *par)
{
  checkPars (gp, par, 0, "waitOnInput");
  CalWaitForEvent(gp);
  return 0;
}

/*
 * Wait(delay). Waits for delay/100 seconds.
 */
long halt(struct cw_status *gp, struct param FAR *par)
{
  long halttime;
  int arg = 0;
  char FAR *funcname = "wait";

  halttime = getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  CalTimeOut (gp, (unsigned long)halttime);

  return(sim_time(gp, (struct param *)NULL));
}

long set_start_time(struct cw_status *gp, struct param FAR *par) 
{
  checkPars (gp, par, 0, "startTime");
  set_time(gp);
  return 0;
}


long sync_input(struct cw_status *gp, struct param FAR *par) 
{
  long halttime, tot, now;
  int arg = 0;
  char FAR *funcname = "sync";
  
  halttime = getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  if(gp->curlunit->lastsync != 0){
    tot = gp->curlunit->lastsync+halttime;
    if (tot > (now = millitime(gp))) {
      CalTimeOut (gp, (unsigned long)(tot-now));
    }
  }
  gp->curlunit->lastsync = millitime(gp);
  return gp->curlunit->lastsync;
}

long log_variable(struct cw_status *gp, struct param FAR *par) { /* Unused */
  struct param FAR *p;
  
  for ( p = par ; p ; p = (struct param FAR *) next(p))
    if (p->exp->left.vd != NULL)
      p->exp->left.vd->log = 1;
  return(TRUE);
}

long unlog_variable(struct cw_status *gp, struct param FAR *par) { /* Unused */
  struct param FAR *p;

  for ( p = par ; p ; p = (struct param FAR *) next(p))
    if (p->exp->left.vd != NULL)
      p->exp->left.vd->log = 0;
  return(TRUE);
}

/* The next two routines are not using the standard parameter-fetching
   routines. I dared not rewrite them since I'm not sure how they work
   22/6-95 SAJ */

long lower_limit(struct cw_status *gp, struct param FAR *par) 
{
  struct param FAR *p;
  long limit;
  double f_limit, f_value;

  switch(par_type(gp, par)) {
  case  T_INT :
    limit = par_value(gp, par);
    f_limit = (double) limit;
    for (p=(struct param FAR *) next(par);
	 p ;
	 p=(struct param FAR *) next(p)) {
      switch(par_type(gp, p)) {
      case T_INT :
	if (par_value(gp, p) < limit)
	  set_par_value(gp, p,limit);
	break;
      case T_FLOAT   :
	if (f_par_value(gp, p) < f_limit)
	  f_set_par_value(gp, p,f_limit);
	break;
      default        :
      paramError(gp, p, ErrLIMVARLOW);
	return 0;
      }
    }
    break;
  case  T_FLOAT   :
    f_limit = f_par_value(gp, par);
    for (p=(struct param FAR *) next(par);
	 p ;
	 p=(struct param FAR *) next(p)) {
      switch(par_type(gp, p)) {
      case T_INT :
	f_value = (double) par_value(gp, p);
	if (f_value < f_limit)
	  set_par_value(gp, p,(long)f_limit);
	break;
      case T_FLOAT   :
	if (f_par_value(gp, p) < f_limit)
	  f_set_par_value(gp, p,f_limit);
	break;
      default        :
	paramError(gp, p, ErrLIMVARLOW);
	return 0;
      }
    }
    break;
  default         :
    paramError(gp, par, ErrLIMVARLOW);
    return 0;
  }

  return(TRUE);
}

long upper_limit(struct cw_status *gp, struct param FAR *par) 
{
  struct param FAR *p;
  long limit;
  double f_limit, f_value;

  switch(par_type(gp, par)) {
  case  T_INT :
    limit = par_value(gp, par);
    f_limit = (double) limit;
    for (p=(struct param FAR *) next(par);
	 p ;
	 p=(struct param FAR *) next(p)) {
      switch(par_type(gp, p)) {
      case T_INT :
	if (par_value(gp, p) > limit)
	  set_par_value(gp, p,limit);
	break;
      case T_FLOAT   :
	if (f_par_value(gp, p) > f_limit)
	  f_set_par_value(gp, p,f_limit);
	break;
      default        :
	paramError(gp, p, ErrLIMVARHIG);
	return 0;
      }
    }
    break;
  case  T_FLOAT   :
    f_limit = f_par_value(gp, par);
    for (p=(struct param FAR *) next(par);
	 p ;
	 p=(struct param FAR *) next(p)) {
      switch(par_type(gp, p)) {
      case T_INT :
	f_value = (double) par_value(gp, p);
	if (f_value > f_limit)
	  set_par_value(gp, p,(long)f_limit);
	break;
      case T_FLOAT   :
	if (f_par_value(gp, p) > f_limit)
	  f_set_par_value(gp, p,f_limit);
	break;
      default        :
	paramError(gp, p, ErrLIMVARHIG);
	return 0;
      }
    }
    break;
  default         :
    paramError(gp, par, ErrLIMVARHIG);
    return 0;
  }
  return(TRUE);
}

void resizeAweWin(struct cw_status *gp, int width, int height)
{
  int bgcolor = gp->curlunit->win->color;
  CalImage FAR *oldwin = gp->curlunit->win;
  CalImage FAR *oldstat = gp->curlunit->picstatic;
  CalRect oldrect = screenRect (gp);

  /* Do not load connection if not necessary */
  if (gp->curlunit->win->width == width
      && gp->curlunit->win->height == height)
    return ;

/* As copyStatPic copies to gp->curlunit->win, it must temporarly be set
   to point to gp->curlunit->picstatic */
  if (oldstat) {
    gp->curlunit->win = blankImage (gp, width, height, bgcolor);
    copyStatPic (gp, oldstat, oldrect);
    gp->curlunit->picstatic = gp->curlunit->win;
    killImage (gp, oldstat);
  }

  gp->curlunit->win = blankImage (gp, width, height, bgcolor);
  copyStatPic (gp, oldwin, oldrect);
  killImage (gp, oldwin);
}

long resize_window(struct cw_status *gp, struct param FAR *par)
{
  int height, width;
  int arg = 0;
  char FAR *funcname = "resizeWindow";

  width = (int) getIntPar (gp, &par, ++arg, funcname);
  height = (int) getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  resizeAweWin(gp, width, height);

  resizeMainWin (gp, width, height);
  return TRUE;
}

/*
 * Put width and height of the current candleweb window into par_w and par_h.
 */
long getWindowSize(struct cw_status *gp, struct param *par)
{
  struct param FAR *par_h, FAR *par_w;
  int w, h;
  int arg = 0;
  char FAR *funcname = "getWindowSize";

  par_w = par;
  getIntPar (gp, &par, ++arg, funcname);
  par_h = par;
  getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  w = gp->curlunit->win->width;
  h = gp->curlunit->win->height;
  if(par_w)
    set_par_value(gp, par_w, w);
  if(par_h)
    set_par_value(gp, par_h, h);
  return 1;
}

/*
 * Put x position in first parameter and y in second.
 */
long mouse_pos(struct cw_status *gp, struct param FAR *par)
{
  struct param FAR *par_x, FAR *par_y;
  int arg = 0;
  char FAR *funcname = "mousePos";

  par_x = par;
  getIntPar (gp, &par, ++arg, funcname);
  par_y = par;
  getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  /*  GetMousePos(&x, &y); */
  set_par_value(gp, par_x, gp->mousex);
  set_par_value(gp, par_y, gp->mousey);

  return 1;
}

long set_mouse(struct cw_status *gp, struct param FAR *par) 
{
  int x,y;
  int arg = 0;
  char FAR *funcname = "setCursorPos";

  x = (int) getIntPar (gp, &par, ++arg, funcname);
  y = (int) getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  SetMousePos(gp, x,y);

  return 1;
}

long flush_input(struct cw_status *gp, struct param FAR *par)
{
  struct event FAR *e;

  checkPars (gp, par, 0, "flushInput");
  while( (e = gp->curlunit->fevent) ){
    gp->curlunit->fevent = next(e);
    remove_at(e);
    CalFree(e);
  }
  return 1;
}

/* No standard parameter-fetching, because it is unneccessary */
long Exit(struct cw_status *gp, struct param FAR *par) 
{
  int parval=0;
  if(par)
    parval = (int) par_value(gp, par);
  gp->command |= RETURN_CALUNIT;

  return(parval);
};

long UnixTime(struct cw_status *gp, struct param FAR *par) 
{
  time_t timer;

  checkPars (gp, par, 0, "unixTime");
  time(&timer);
  return((long)timer);
}

long Output(struct cw_status *gp, struct param FAR *par)
{
  checkPars (gp, par, 0, "output");

  CalSync(gp);

  updateWins (gp);
  updateGraphs (gp);

  if (gp->exposed) refresh (gp, FALSE);
  else {
    gp->exposed = TRUE; /* The initial picture should no more be drawn */
    refresh (gp, TRUE);
    CalSync (gp);
  }
  return 1;
}

long OptimizeObjects (struct cw_status *gp, struct param FAR *par)
{
  checkPars (gp, par, 0, "optimizeObjects");
  if (gp->curlunit->output > 0) optOCleanup (gp);
  updateWins (gp);
  updateGraphs (gp);
  gp->curlunit->output++;
  gp->curlunit->cur_funcinst->optimized++;
  statobjSetup (gp, (struct blockinst *)gp->curlunit->cur_funcinst);

  return 1;
}

long getXPoint (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "getXPoint";
  struct param *fpar;
  struct pointarray *pa;
  int i;

  fpar = par;
  pa = par->exp->left.pt;

  par = next(par);
  ++arg;

  i = getIntPar (gp, &par, ++arg, funcname);

  if(i > pa->n){
    paramError(gp, fpar, "Error in getXPoint() parameter: index larger than number of points in pointlist.");
    return 0;
  } else if(i <= 0) {
    paramError(gp, fpar, "Error in getXPoint() parameter: index into pointlist is less or equal 0.");
    return 0;
  }
  
  return calculate(gp, pa->op[2*(i-1)]);
}

long getYPoint (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "getYPoint";
  struct param *fpar;
  struct pointarray *pa;
  int i;

  fpar = par;
  pa = par->exp->left.pt;

  par = next(par);
  ++arg;

  i = getIntPar (gp, &par, ++arg, funcname);

  if(i > pa->n){
    paramError(gp, fpar, "Error in getYPoint() parameter: index larger than number of points in pointlist.");
    return 0;
  } else if(i <= 0) {
    paramError(gp, fpar, "Error in getYPoint() parameter: index into pointlist is less or equal 0.");
    return 0;
  }
  
  return calculate(gp, pa->op[2*(i-1)+1]);
}

long Bell (struct cw_status *gp, struct param FAR *par)
{
  long vol, pitch, dur;
  int arg = 0;
  char FAR *funcname = "bell";

  vol = getIntPar (gp, &par, ++arg, funcname);
  if (vol != -1 && (vol <0 || vol > 100)) {
    errorMsg(gp, 0, WarnBELRANGE1);
    vol = -1;
  }
  pitch = getIntPar (gp, &par, ++arg, funcname);
  if (pitch != -1 && pitch < 0) {
    errorMsg(gp, 0, WarnBELRANGE2);
    pitch = -1;
  }
  dur = getIntPar (gp, &par, ++arg, funcname);
  if (dur != -1 && dur < 0) {
    errorMsg(gp, 0, WarnBELRANGE3);
    dur = -1;
  }
  checkPars (gp, par, arg, funcname);
  calBell (gp, vol, pitch, dur);
  return 1;
}

/* Type: 0 - stop, 1 - play once, -1 - play forever */

#ifdef WIN32
UINT wDeviceID;
char inuse = 0;
MCI_PLAY_PARMS mciPlayParms;

void restartDevice (void)
{
/*  MCIERROR err;
  char *errmsg[128];

  err =*/ 
  if (inuse == 2) {
    mciSendCommand(wDeviceID, MCI_SEEK, MCI_WAIT | MCI_SEEK_TO_START, (DWORD)NULL);
/*  if (err != 0) {
    mciGetErrorString(err, &errmsg[0], 128);
    MessageBox (NULL, errmsg, "Notify", MB_OK);
    exit (-1);
  } */
    mciSendCommand(wDeviceID, MCI_PLAY, MCI_NOTIFY, (DWORD)(LPVOID) &mciPlayParms);
    inuse = 2;
  }
  else {
    mciSendCommand(wDeviceID, MCI_STOP, MCI_WAIT, (DWORD)NULL);
    mciSendCommand(wDeviceID, MCI_CLOSE, MCI_WAIT, (DWORD)NULL);
    inuse = 0;
  }
}

void stopDevice (void)
{
  if (inuse) {
    mciSendCommand(wDeviceID, MCI_STOP, MCI_WAIT, (DWORD)NULL);
    mciSendCommand(wDeviceID, MCI_CLOSE, MCI_WAIT, (DWORD)NULL);
/*  mciSendCommand(MCI_ALL_DEVICE_ID, MCI_CLOSE, 0, NULL); */
    }
  inuse = 0;
}
#endif /* WIN32 */

static long Sound (struct cw_status *gp, struct param FAR *par, int type)
{
#ifdef SOUNDSUPPORT
  long arg = 0, n, bufsize, dataread;
  char FAR *funcname = "sound", *ext, *URL, mess[160];
  AweStream *sfd;
  char ctrlbuf[1024], *buffer;
#ifdef WIN32
  DWORD dwReturn;
  MCI_OPEN_PARMS mciOpenParms;
  char backbuf[128], *fn2;
#endif
#ifdef UNIX
  fd_set wfds;
  
  FD_ZERO (&wfds);
  FD_SET (gp->wavefd[1], &wfds);
#endif
  if (type != 0) URL = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if (type != 0) {
#ifdef UNIX
    if((sfd = AweOpen (gp, URL)) == NULL){
      sprintf(mess, ErrRESOLVURL, URL);
      errorMsg(gp, 1, mess);
      return 0;
    }

    bufsize = 1024; dataread = 0;
    if ((buffer = (char *)CalMalloc (bufsize)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }

    while ((n = AweRead (sfd, &buffer[dataread], 1024)) > 0) {
      dataread += n;
      if (dataread+1024 > bufsize) {
	/* Aint gonna last another round */
	bufsize += 1024;
	if ((buffer = CalRealloc (buffer, bufsize)) == NULL) {
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit (gp, NOT_OK);
	}
      }
    }

    ext = strrchr (URL, '.');
    if (ext == NULL) {
      free (URL);
      return 0;
    }
    free (URL);
    AweClose (sfd);
    sprintf (ctrlbuf, "%ld %s %c", dataread, ext+1, type == 1 ? 'S' : 'C');

    kill (gp->wavePid, SIGUSR1);
    select (gp->wavefd[1]+1, NULL, &wfds, NULL, NULL);
    write (gp->wavefd[1], &ctrlbuf[0], strlen (ctrlbuf)+1);

    select (gp->wavefd[1]+1, NULL, &wfds, NULL, NULL);
    write (gp->wavefd[1], &buffer[0], dataread);

    CalFree (buffer);
#endif
#ifdef WIN32
    /*
     * Plays a given waveform-audio file using MCI_OPEN and MCI_PLAY. 
     * Returns when playback begins.
     * Open the device by specifying the device and filename.
     * MCI will choose a device capable of playing the given file.
     */

    stopDevice ();
    mciOpenParms.lpstrDeviceType = "waveaudio";
    mciOpenParms.lpstrElementName = ResolveURL (gp, URL);
    if (mciOpenParms.lpstrElementName == NULL) {
      free (URL);
      sprintf(mess, ErrRESOLVURL, URL);
      errorMsg(gp, 1, mess);
      return 0;
    }
    free (URL);
    if (dwReturn = mciSendCommand(0, MCI_OPEN,
				  MCI_OPEN_TYPE | MCI_OPEN_ELEMENT,
				  (DWORD)(LPVOID) &mciOpenParms)) {
      /* Failed to open device. Dont close it; just return error. */
/*      mciGetErrorString(dwReturn, &backbuf, 128); */
      return 0;
    }
    /* The device opened successfully; get the device ID. */
    wDeviceID = mciOpenParms.wDeviceID;

    /*
     * Begin playback. The window procedure function for the parent 
     * window will be notified with an MM_MCINOTIFY message when 
     * playback is complete. At this time, the window procedure closes 
     * the device.
     */
    mciPlayParms.dwCallback = (DWORD)gp->system.workArea;
    if (dwReturn = mciSendCommand(wDeviceID, MCI_PLAY, MCI_NOTIFY, 
				  (DWORD)(LPVOID) &mciPlayParms)) {
      mciSendCommand(wDeviceID, MCI_CLOSE, 0, (DWORD)NULL);
      return 1;
    }
    inuse = type == 1 ? 1 : 2;
#endif
    return 1;
  }
  else {
#ifdef UNIX
    kill (gp->wavePid, SIGUSR1);
#else
    stopDevice ();
#endif
  }
#endif
  return 0;
}

long SndSingle (struct cw_status *gp, struct param *par)
{
  return Sound (gp, par, 1);
}

long SndCont (struct cw_status *gp, struct param *par)
{
  return Sound (gp, par, -1);
}

long SndStop (struct cw_status *gp, struct param *par)
{
  return Sound (gp, par, 0);
}

static long Play (struct cw_status *gp, struct param *par, int type)
{
#ifdef SOUNDSUPPORT
#ifdef WIN32
    DWORD dwReturn;
    MCI_OPEN_PARMS mciOpenParms;
    MCI_STATUS_PARMS mciStatusParms;
    MCI_SEQ_SET_PARMS mciSeqSetParms;
#endif
  int arg = 0, n, written;
  char *funcname = "play", *URL, mess[160], ctrlbuf[1024], *buffer;
  AweStream *pas;
  long bufsize, dataread;
#ifdef UNIX
  fd_set mfds;

  FD_ZERO (&mfds);
  FD_SET (gp->midifd[1], &mfds);
#endif
  if (type != 0) URL = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if (type != 0) {
#ifdef UNIX
    if((pas = AweOpen (gp, URL)) == NULL){
      sprintf(mess, ErrRESOLVURL, URL);
      errorMsg(gp, 1, mess);
      return 0;
    }
    bufsize = 1024; dataread = 0;
    if ((buffer = (char *)CalMalloc (bufsize)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }

    while ((n = AweRead (pas, &buffer[dataread], 1024)) > 0) {
      dataread += n;
      if (dataread+1024 > bufsize) {
	/* Aint gonna last another round */
	bufsize += 1024;
	if ((buffer = CalRealloc (buffer, bufsize)) == NULL) {
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit (gp, NOT_OK);
	}
      }
    }
    free (URL);
    AweClose (pas);
    sprintf (ctrlbuf, "%ld %c", dataread, type == 1 ? 'S' : 'C');

    kill (gp->midiPid, SIGUSR1);
    select (gp->midifd[1]+1, NULL, &mfds, NULL, NULL);
    n = write (gp->midifd[1], &ctrlbuf[0], strlen (ctrlbuf)+1);

    for (written = 0; written < dataread; written += n) {
      select (gp->midifd[1]+1, NULL, &mfds, NULL, NULL);
      n = write (gp->midifd[1], &buffer[written], dataread);
    }
    CalFree (buffer);
#endif
#ifdef WIN32
	// Plays a specified MIDI file by using MCI_OPEN and MCI_PLAY. Returns 
// as soon as playback begins. The window procedure function for the 
// given window will be notified when playback is complete. Returns 0L 
// on success; otherwise, it returns an MCI error code.
    // Open the device by specifying the device and filename.
    // MCI will attempt to choose the MIDI mapper as the output port.
    stopDevice ();
    inuse = 0;
    mciOpenParms.lpstrDeviceType = "sequencer";
    mciOpenParms.lpstrElementName = ResolveURL (gp, URL);
    if (mciOpenParms.lpstrElementName == NULL) {
      free (URL);
      sprintf(mess, ErrRESOLVURL, URL);
      errorMsg(gp, 1, mess);
      return 0;
    }
    free (URL);  

    if (dwReturn = mciSendCommand((UINT)NULL, MCI_OPEN,
        MCI_OPEN_TYPE | MCI_OPEN_ELEMENT,
        (DWORD)(LPVOID) &mciOpenParms))
    {
        // Failed to open device. Dont close it; just return error.
        return (dwReturn);
    }

    // The device opened successfully; get the device ID.
    wDeviceID = mciOpenParms.wDeviceID;

    // Check if the output port is the MIDI mapper.
    mciStatusParms.dwItem = MCI_SEQ_STATUS_PORT;
    if (dwReturn = mciSendCommand(wDeviceID, MCI_STATUS, 
        MCI_STATUS_ITEM, (DWORD)(LPVOID) &mciStatusParms))
    {
        mciSendCommand(wDeviceID, MCI_CLOSE, 0, (DWORD)NULL);
        return (dwReturn);
    }

    // The output port is not the MIDI mapper. 
    // Ask if the user wants to continue.
    if (LOWORD(mciStatusParms.dwReturn) != MIDI_MAPPER)
    {
//        if (MessageBox(hMainWnd,
//            "The MIDI mapper is not available. Continue?",
//            "", MB_YESNO) == IDNO)
//        {
            // User does not want to continue. Not an error;
            // just close the device and return.
//            mciSendCommand(wDeviceID, MCI_CLOSE, 0, NULL);
//            return (0L);
//        }
    }

    // Begin playback. The window procedure function for the parent 
    // window will be notified with an MM_MCINOTIFY message when 
    // playback is complete. At this time, the window procedure closes 
    // the device.
    mciPlayParms.dwCallback = (DWORD)gp->system.workArea;
    if (dwReturn = mciSendCommand(wDeviceID, MCI_PLAY, MCI_NOTIFY, 
        (DWORD)(LPVOID) &mciPlayParms))
    {
        mciSendCommand(wDeviceID, MCI_CLOSE, 0, (DWORD)NULL);
        return (dwReturn);
    }
    inuse = type == 1 ? 1 : 2;
#endif
    return 1;
  }
  else {
#ifdef UNIX
    kill (gp->midiPid, SIGUSR1);
#else
    stopDevice ();
#endif
  }
#endif
    return 0;
}

long PlSingle (struct cw_status *gp, struct param *par)
{
  return Play (gp, par, 1);
}

long PlCont (struct cw_status *gp, struct param *par)
{
  return Play (gp, par, -1);
}

long PlStop (struct cw_status *gp, struct param *par)
{
  return Play (gp, par, 0);
}

long EvalStatement(struct cw_status *gp, struct param FAR *par)
{
  int arg=0;
  char FAR *funcname = "eval";
  struct sim_actions FAR *si, FAR *sa;
  char FAR *str;

  str = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);


  if(!str || RETURN)
    return 0;

  sa = parse_statement(gp, str);
  CalFree(str);

  si = i_simactions(gp, sa, gp->curlunit->cur_blockinst);

  simulate(gp, si, NULL);
  
  f_simulate(gp, si, FALSE);
  f_simulate(gp, sa, TRUE);
  return 1;
}



long imageextents (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  int ret;
  char FAR *funcname = "imageExtents", FAR *in;
  struct param FAR *wpar, FAR *hpar;
  CalImage *cip;

  in = getTextPar (gp, &par, ++arg, funcname);
  wpar = par;
  getIntPar (gp, &par, ++arg, funcname);
  hpar = par;
  getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if ((cip = getImage (gp, in)) == &gp->imgcache->img){
    ret = FALSE;
  } else {
    set_par_value (gp, wpar, cip->width);
    set_par_value (gp, hpar, cip->height);
    ret = TRUE;
  }
  CalFree(in);
  return ret;
}

long minLevel (struct cw_status *gp, struct param FAR *par)
{
  checkPars (gp, par, 0, "minLevel");

  return (((struct levlist *)first(gp->curlunit->level0))->level);
}

long maxLevel (struct cw_status *gp, struct param FAR *par)
{
  checkPars (gp, par, 0, "maxLevel");

  return (((struct levlist *)last(gp->curlunit->level0))->level);
}

long SetAnchor(struct cw_status *gp, struct param FAR *par)
{
  char FAR *url;
  int arg = 0;
  char FAR *funcname = "setAnchor";

  url = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  NewAnchor(gp, url, URL_ANCHOR);

  setURL (gp, url);
  CalFree(url);
  return 1;
}

long CalCall(struct cw_status *gp, struct param FAR *par) 
{
  CalRect defrect;
  char FAR *url, *curl;
  int arg = 0;
  char FAR *funcname = "call";
  char *ctype;
  AweStream *as;

  url = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  SndStop (gp, NULL);
  PlStop (gp, NULL);

  setScanFileName(gp, url);
  curl = canonizedURL(gp, url); 

  NewAnchor(gp, curl, URL_ANCHOR);
#ifdef WWWENABLE
  ctype = ContentType(gp, curl);
#endif

  if(!strcmp(ctype, "application/x-candleweb")){
    if((as = AweOpen(gp, curl)) == NULL){
      CalFree(curl);
      return 0;
    }

    start_candle_unit(gp, as, curl);
    
    killLearnUnit(gp);
    if(gp->curlunit){
      NewAnchor(gp, gp->curlunit->url, URL_ANCHOR);
      setScanFileName(gp, strdup(gp->curlunit->url));
      setURL(gp, gp->curlunit->url);
      setStatus(gp, "Continuing simulation");
      defrect = screenRect (gp);
      fakeExpose (gp, defrect);
    }
    if(gp->command & RETURN_CALUNIT)
      gp->command &= ~RETURN_CALUNIT;
  } else if(!strcmp(ctype, "text/html")){
    char cmd[256];
    int ret;
    sprintf(cmd, "netscape -remote openURL\\(\"%s\"\\)", curl);
    ret = system(cmd);
/*     printf("ret %d\n", ret); */
    if(ret != 0){
      sprintf(cmd, "netscape %s &", curl);
      ret = system(cmd);
    }
  } else {
    fprintf(stderr, "Unsupported content-type '%s' in call(\"%s\")\n", 
	    ctype, curl);
  }
  CalFree(curl);
  return(OK);
}
