#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <math.h>
#include <ctype.h>

#include "candle.h"
#include "const.h"
#include "simulate.h"
#include "learnuni.h"
#include "function.h"
#include "funcname.h"
#include "error.h"
#include "parser.h"
#include "input.h"
#include "lex.h"

#include "protos/canutil.h"
#include "protos/creatsim.h"
#include "protos/fast_lis.h"
#include "protos/freesim.h"
#include "protos/parser.h"
#include "protos/memory.h"

struct sym_type {
  char *name;
  long type;
};

/*
 * This procedure checks if a string 'str' is in the array set.
 */
long get_symbol(char  *str, struct sym_type set[],
		int max)
{
  int i;
  for(i = 0;i < max; i++)
    if(!strcmp(set[i].name, str))
      return set[i].type;

  return -1;
}

static struct sym_type mask_syms[] = {
  {"CtrlMask", CMaskCtrl}, 
  {"ShiftMask", CMaskShift}, 
  {"AltMask", CMaskAlt}, 
  {"Button1Mask", CMaskButton1}, 
  {"Button2Mask", CMaskButton2}, 
  {"Button3Mask", CMaskButton3}, 
  {"DoubleMask", CMaskDouble} 
};

long mask_sym_type(char *name)
{
  return get_symbol(name, mask_syms, 
		    sizeof(mask_syms)/sizeof(struct sym_type));
}


static struct sym_type key_syms[] = {
  {"Up", K_Up}, 
  {"Down", K_Down}, 
  {"Left", K_Left}, 
  {"Right", K_Right}, 
  {"Return", K_Return}, 
  {"BackSpace", K_BackSpace}, 
  {"Delete", K_Delete}, 
  {"Insert", K_Insert}, 
  {"Home", K_Home}, 
  {"End", K_End}, 
  {"PageUp", K_PageUp}, 
  {"PageDown", K_PageDown}, 
  {"Escape", K_Escape}, 
  {"Shift", K_Shift}, 
  {"Control", K_Control}, 
  {"Tab", K_Tab}, 
  {"F1", K_F1}, 
  {"F2", K_F2}, 
  {"F3", K_F3}, 
  {"F4", K_F4}, 
  {"F5", K_F5}, 
  {"F6", K_F6}, 
  {"F7", K_F7}, 
  {"F8", K_F8}, 
  {"F9", K_F9}, 
  {"F10", K_F10},
};

long key_sym_type(char *name)
{
  return get_symbol(name, key_syms, 
		    sizeof(key_syms)/sizeof(struct sym_type));
}


static struct sym_type specevent_syms[] = {
  {"Motion", MouseMotion}, 
  {"ResizeWindow", ResizeWindow}
};

long specevent_sym_type(char *name)
{
  return get_symbol(name, specevent_syms, 
		    sizeof(specevent_syms)/sizeof(struct sym_type));
}

static struct sym_type press_syms[] = {
  {"Press", KeyDown}, 
  {"Release", KeyUp}
};

long press_sym_type(char *name)
{
  return get_symbol(name, press_syms, 
		    sizeof(press_syms)/sizeof(struct sym_type));
}

static struct sym_type mouse_syms[] = {
  {"Button1", 1}, 
  {"Button2", 2}, 
  {"Button3", 3}
};

long mouse_sym_type(char *name)
{
  return get_symbol(name, mouse_syms, 
		    sizeof(mouse_syms)/sizeof(struct sym_type));
}

static struct sym_type attr_syms[] = {
  {"expr", T_EXPR}, 
  {"points", T_POINTS},
  {"translation", T_TRANSLATION}
};

long attr_sym_type(char *name)
{
  return get_symbol(name, attr_syms, 
		    sizeof(attr_syms)/sizeof(struct sym_type));
}

static struct sym_type battr_syms[] = {
  {"act", S_ACTIVE}, 
  {"active", S_ACTIVE}, 
  {"col", S_COLOR}, 
  {"color", S_COLOR}, 
  {"dashes", S_DASHES}, 
  {"dec", S_DECIMALS}, 
  {"decimals", S_DECIMALS}, 
  {"dsh", S_DASHES}, 
  {"ea", S_ENDANGLE}, 
  {"endangle", S_ENDANGLE}, 
  {"f", S_FILL}, 
  {"fill", S_FILL}, 
  {"font", S_FONT}, 
  {"fn", S_FONT}, 
  {"img", S_IMAGE}, 
  {"image", S_IMAGE}, 
  {"lev", S_LEVEL}, 
  {"level", S_LEVEL}, 
  {"linewidth", S_LINEWIDTH}, 
  {"lw", S_LINEWIDTH}, 
  {"of", S_OUTFLOAT}, 
  {"outfloat", S_OUTFLOAT}, 
  {"oi", S_OUTINT}, 
  {"outint", S_OUTINT}, 
  {"ot", S_OUTTEXT}, 
  {"outtext", S_OUTTEXT}, 
  {"points", S_POINTS},
  {"sa", S_STARTANGLE}, 
  {"savebackground", S_SAVEBG}, 
  {"sb", S_SAVEBG}, 
  {"startangle", S_STARTANGLE}, 
  {"tex", S_TEXTURE}, 
  {"texture", S_TEXTURE}, 
  {"translation", S_TRANSLATION}, 
  {"tr", S_TRANSLATION}
};

long battr_sym_type(char *name)
{
  return get_symbol(name, battr_syms, 
		    sizeof(battr_syms)/sizeof(struct sym_type));
}

static struct sym_type bobj_syms[] = {
  {"arc", S_ARC}, 
  {"box", S_BOX},
  {"endwindow", S_END_WINDOW},
  {"image", S_IMAGE}, 
  {"inputarea", S_INPUTAREA},
  {"line", S_LINE}, 
  {"point", S_POINT},
  {"polygon", S_POLYGON},
  {"textobj", S_TEXTOBJECT}, 
  {"window", S_WINDOW},
};

long bobj_sym_type(char *name)
{
  return get_symbol(name, bobj_syms, 
		    sizeof(bobj_syms)/sizeof(struct sym_type));
}

char var_type(struct vdecl FAR *var) 
{
  switch(var->type){
  case INTVAL : return(INTVAR);
  case FLOATVAL : return(FLOATVAR);
  case TEXTVAL : return(TEXTVAR);
  case INTARRAY : return(INTARRAY);
  case FLOATARRAY : return(FLOATARRAY);
  case TEXTARRAY : return(TEXTARRAY);
  default  : return('\0');
  }
}

char const_type(struct c_decl FAR *cc) 
{
  switch(cc->type){
  case INTVAL : return(INTCONST);
  case FLOATVAL : return(FLOATCONST);
  case TEXTVAL : return(TEXTCONST);
  default  : return('\0');
  }
}

static struct vdecl *lget_var(struct commonblock FAR *cb,
		       char FAR *name) 
{
  struct vdecl FAR *var;
  for (var = cb->firstv; var != NULL; var = next (var))
    if (strcmp (var->name, name) == 0) return var;
  return NULL;
}


/*************************************************************
**                                                 get_var  **
**  Return pointer to a variable named "name" else NULL.    **
*************************************************************/
struct vdecl FAR *get_var(struct commonblock FAR *cb,
			  char FAR *name) 
{
  struct vdecl FAR *var;
  for (; cb != NULL; cb = cb->parent_block)
    if((var = lget_var(cb, name)) != NULL)
      return var;
  return NULL;
}



/*
 * Return TRUE if "name" is an lvalue.
 */
int is_lvalue(struct commonblock FAR *cb, char FAR *name) 
{
   if(get_var(cb, name))
    return TRUE;
  if(cb->type == OBJECTBLOCK && get_attr((struct objectproto *)cb, name))
    return TRUE;

  return FALSE;
}


static struct c_decl *lget_const(struct commonblock FAR *cb, char FAR *name) 
{
  struct c_decl FAR *c;
  for (c = cb->firstc; c != NULL; c = next (c))
    if (strcmp (c->name, name) == 0) 
      return c;
  return NULL;
}


/*************************************************************
**                                               get_const  **
**  Return pointer to a constant named "name" else NULL.    **
*************************************************************/
struct c_decl FAR *get_const(struct commonblock FAR *cb, char FAR *name) 
{
  struct c_decl FAR *c;
  for (; cb != NULL; cb = cb->parent_block)
    if((c = lget_const(cb, name)) != NULL)
      return c;
  return NULL;
}

/*
 * Return pointer to builtin function named 'name' else NULL.
 */
struct bfunction FAR *get_func(char FAR *name) 
{
  int i;

  for(i = 0; funcarray[i].name != NULL;i++){
    if (!strcmp(funcarray[i].name,name)) {
      return &funcarray[i];
    }
  }
  return(NULL);
}


static struct function *lget_function(struct commonblock *cb, char FAR *name)
{
  struct function FAR *fp;
  for (fp = cb->fp; fp != NULL; fp = next (fp))
    if (strcmp (fp->name, name) == 0) return fp;
  return NULL;
} 

/*
 * Return pointer to function defined in 'learn' named 'name' else NULL.
 */ 
struct function FAR *get_function(struct commonblock *cb, char FAR *name)
{
  struct function FAR *fp;
  for (; cb != NULL; cb = cb->parent_block)
    if((fp = lget_function(cb, name)) != NULL)
      return fp;
  return NULL;
} 

/*
 * Return TRUE if identifier 'name' is already declared in block or 
 * enclosing blocks,
 * else return FALSE.
 */
int ident_declared(struct commonblock FAR *block,
		   char FAR *name)
{
  for (; block != NULL; block = block->parent_block)
    if (get_var (block, name) || get_const (block, name) || 
       get_object (block, name) || get_function(block, name))
      return TRUE;
  return FALSE;
}

/*
 * Return TRUE if identifier 'name' is already declared in _this_ block.
 */
int local_declared(struct commonblock FAR *block,
		   char FAR *name)
{
  if (lget_var(block, name) || lget_const(block, name) || 
      lget_function(block, name))
    return TRUE;
  return FALSE;
}

int attr_type(struct objectproto *op, char *attr)
{
  struct attrparam *ap;
  for(ap = op->fparam; ap != NULL; ap = next(ap)){
    if(!strcmp(ap->name, attr))
      return ap->type;
  }
  return 0;
}

struct attrparam FAR *get_attr(struct objectproto *op, char *attr)
{
  struct attrparam *ap;
  for(ap = op->fparam; ap != NULL; ap = next(ap)){
    if(!strcmp(ap->name, attr))
      return ap;
  }
  return NULL;
}

struct objectproto *get_object(struct commonblock FAR *cb, char *ident)
{
  struct objectproto *op;
  struct learnunit *learn;

  while (cb->parent_block != NULL) cb = cb->parent_block;
  learn = (struct learnunit *)cb;

  for(op = learn->fobject;op != NULL;
      op = (struct objectproto FAR *) next(op)){
    if (!strcmp(op->name,ident)) {
      return op;
    }
  }

  return NULL;
}

/*
 * Insert variable into function. 
 */
void insert_variable(struct commonblock FAR *bp,
		     struct vdecl FAR *vd, int is_ref)
{
  if(bp->firstv == NULL){
    init_list(vd);
    bp->firstv = bp->lastv = vd;
  } else {
    if(bp->firstv == bp->fparvar){
      place_prev(bp->fparvar, vd);
      bp->firstv = bp->lastv = vd;
    } else{
      place_next(bp->lastv, vd);
      bp->lastv = vd;
    }
  }	
}

void insert_parameter(struct function FAR *fp,
		      struct vdecl FAR *vd, int is_ref)
{
  if(is_ref == TRUE){
    if(fp->firstv == NULL){
      init_list(vd);
      fp->firstv = fp->lastv = fp->fparvar = fp->fref = vd;
    } else {
      if(fp->fref == NULL){
	place_next(last(fp->firstv), vd);
	fp->fref = vd;
      } else
	place_next(fp->fref, vd);
    } 
  } else {
    if(fp->firstv == NULL){
      init_list(vd);
    } else {
      place_prev(fp->fparvar, vd);
    } 
    fp->firstv = fp->lastv = fp->fparvar = vd;
  }
}

