/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnarr@ifi.uio.no)
 */
#define READSIM
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>

#include <math.h>
#include <ctype.h>

#include "candle.h"
#include "const.h"
#include "simulate.h"
#include "learnuni.h"
#include "lex.h"
#include "parser.h"
#include "function.h"
#include "funcname.h"
#include "error.h"

#include "protos/memory.h"
#include "protos/canutil.h"
#include "protos/creatsim.h"
#include "protos/fast_lis.h"
#include "protos/readsim.h"
#include "protos/readobjs.h"
#include "protos/freesim.h"
#include "protos/lex.h"
#include "protos/parser.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define OP_LEVELS 11

/* List of identfiers which have been used but not declared */
static void matchglobal(struct cw_status *gp, struct learnunit FAR *);


/*
 * Parse the statement in str and return a pointer to 
 * the statement structure.
 */
struct sim_actions *parse_statement(struct cw_status *gp, char *str)
{
  struct sim_actions *sa;
  int token;
  gp->LEXyy.str = str;
  gp->LEXyy.strind = 0;

  token = yylex(gp);
  if((sa = NEWSIMACTION) == NULL)
    errorMsg(gp,2, ErrNOMOREMEM);
  init_list(sa);
  token = statement(gp, gp->curlunit->cur_blockinst->parent, 
		    S_NOSYMBOL, token, sa);

  gp->LEXyy.strind = -1;
  return sa;
}


int error(struct cw_status *gp, int ss, char FAR *text)
{
  /* write text error message and scan input using yylex until a token
   * ss is found 
   */
  int token;

  yyerror(gp, text);
  token = yylex(gp);
  while(token != S_NOSYMBOL && token != ss ) token = yylex(gp);
  return(token);
}

int op_(struct cw_status *gp, struct commonblock *block, int s,
	struct oper **node, int level);

/*
 * Parse actual parameters to a function or parameters to an array.
 */
int parameters(struct cw_status *gp, struct commonblock *block, int s,
	       int intoken, struct param *par)
{
  struct param FAR *p, FAR *f=NULL;
  int firsttime=TRUE;
  int token;

  token = intoken;
 L_PARAMETER:	
  switch (token) {
  case S_BEGBRACKET :
  case S_BEGPAR     :
    if(firsttime){
      f = p = par;
      firsttime = FALSE;
      break;
    }
  case S_PAREXPSEP  :
    if ((p = NEWPARAM) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      return(NOT_OK);
    }
    p->line = gp->LEXyy.line;
    place_next(f,p);
    f = p;
    break;
  default:
    return token;
  }

  token = op_(gp, block, s, &p->exp, 1);
  
  if(token == S_ENDBRACKET){
    token = yylex(gp);
    if(token != S_BEGBRACKET)
      return token;
  } else if(token == S_ENDPAR){
    return token;
  } else if (token != S_PAREXPSEP){
    token = error(gp, s, ErrPARAMLIST);
    return token;
  } 
  goto L_PARAMETER;	
}

/*
 * Parse one factor of an expression.
 */
int factor(struct cw_status *gp, struct commonblock *block, int s,
	   struct oper **op, int intoken) 
{
  struct vdecl FAR *var;
  struct c_decl FAR *c;
  struct action FAR *act;
  struct intarray FAR *arr;
  struct bfunction FAR *fp;
  int token, flag;
  char errormsg[256];
  struct attrparam FAR *ap;

  token = intoken;
  L_FACTOR :
    switch (intoken) {
    case S_BEGPAR :
      token = op_(gp, block, S_ENDPAR, op, 1);
      if (token != S_ENDPAR)
	yyerror(gp, ErrXPRNCLPAR);
      else
	token = yylex(gp);
      break;
    case S_IDENT   :
      if ((*op = NEWOPER) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      (*op)->type = DO_VDECL;
      (*op)->line = gp->LEXyy.line;
      if ((var = get_var(block, gp->LEXyy.ident)) != NULL) {
	flag = var_type(var);
	if ((flag == INTARRAY) || (flag == FLOATARRAY) || 
	    (flag == TEXTARRAY)) {
	  if ((arr = (*op)->left.array = NEWINTARRAY) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  (*op)->left.array->which.vd = var;
	  (*op)->left.array->type = DO_VDECL;
	  (*op)->flag = flag;
	  token = yylex(gp);
	  if (token == S_BEGBRACKET) {
	    if ((arr->indeks = NEWPARAM) == NULL){
	      errorMsg(gp, 2, ErrNOMOREMEM);
	      c_exit(gp, NOT_OK);
	    }
	    arr->indeks->line = gp->LEXyy.line;
	    init_list(arr->indeks);
	    token = parameters(gp, block, S_ENDBRACKET,token,arr->indeks);
	  }
	} else {
	  (*op)->left.vd = var;
	  (*op)->flag = flag;
	  token = yylex(gp);
	}
      } else if ((c = get_const(block, gp->LEXyy.ident)) != NULL) {
	(*op)->left.cconst = c;
	(*op)->flag = const_type(c);
	token = yylex(gp);
      } else if ((fp = get_func(gp->LEXyy.ident)) != NULL) {
	if ((act = (*op)->left.do_act = NEWACTION) == NULL){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	act->line = gp->LEXyy.line;
	switch(fp->type){
	case BINTFUNC:
	  (*op)->left.do_act->function.do_act = fp->do_act;
	  break;
	case BFLOATFUNC:
	  (*op)->left.do_act->function.fdo_act = fp->fdo_act;
	  break;
	case BTEXTFUNC:
	  (*op)->left.do_act->function.tdo_act = fp->tdo_act;
	  break;
	default:
	  sprintf(errormsg, ErrSIMINTERR, fp->type);
	  errorMsg(gp, 1, errormsg);
	  return(0);
	}
	(*op)->flag = fp->type;
	(*op)->left.do_act->type = fp->type;

	token = yylex(gp);
	if (token == S_BEGPAR) {
	  if ((act->actual.par = NEWPARAM) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  act->actual.par->line = gp->LEXyy.line;
	  init_list(act->actual.par);
	  token = parameters(gp, block, S_ENDPAR,token,act->actual.par);
	  token = yylex(gp);
	}
      } else if (block->type == OBJECTBLOCK && 
		 (ap = get_attr((struct objectproto FAR *)block,
				gp->LEXyy.ident)) != NULL){
	(*op)->flag = ATTROPER;
	(*op)->left.ap = ap;
	token = yylex(gp);
	if(token == S_BEGBRACKET){
	  (*op)->flag |= T_ARRAY;
	  if ((arr = (*op)->right.array = NEWINTARRAY) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  (*op)->right.array->which.vd = var;
	  (*op)->right.array->type = DO_VDECL;
	  if ((arr->indeks = NEWPARAM) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  arr->indeks->line = gp->LEXyy.line;
	  init_list(arr->indeks);
	  token = parameters(gp, block, S_ENDBRACKET,token,arr->indeks);
	}
      } else {
	/* 
	 * The identifier has not yet been declared, it may however be 
	 * declared at a later stage. The identfier is therefore inserted 
	 * into a list in the current learnunit.
	 */
	struct notdecl FAR *nd;
	(*op)->left.value = 0;
	(*op)->flag = INTVAL;

	nd = c_notdecl(gp);

	nd->name = (char FAR *) CalMalloc(strlen(gp->LEXyy.ident) + 1);
	if(nd->name == NULL){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	strcpy(nd->name, gp->LEXyy.ident);
	nd->type = 'F';
	nd->op = *op;
	token = yylex(gp);	  
	if(token == S_BEGPAR){
	  nd->type = 'F';
	  if((nd->actual.par = NEWPARAM) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  nd->actual.par->line = gp->LEXyy.line;
	  init_list(nd->actual.par);
	  token = parameters(gp, block, S_ENDPAR, token, nd->actual.par);
	  token = yylex(gp);
	} 	
	/* Insert new element first in list */
	if(gp->curlunit->firstnd != NULL)
	  place_prev(gp->curlunit->firstnd, nd);
	gp->curlunit->firstnd = nd;
      }
      break;
    case S_INTEGERKONST :
      if ((*op = NEWOPER) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      (*op)->type = DO_VDECL;
      (*op)->left.value = gp->LEXyy.lval;
      (*op)->flag = INTVAL;
      token = yylex(gp);
      break;
    case S_REALKONST :
      if ((*op = NEWOPER) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      (*op)->type = DO_VDECL;
      (*op)->left.fvalue = gp->LEXyy.rval;
      (*op)->flag = FLOATVAL;
      (*op)->line = gp->LEXyy.line;
      token = yylex(gp);
      break;
    case S_TTEXTKONST :
      if ((*op = NEWOPER) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      (*op)->type = DO_VDECL;
      (*op)->flag = TEXTVAL;
      (*op)->left.text = gp->LEXyy.tval;
      (*op)->line = gp->LEXyy.line;
      token = yylex(gp);
      break;
    default :
      error(gp, s, ErrTOKILLCON);
      if ( (token == S_BEGPAR) || (token == S_IDENT) ||
	   (token == S_INTEGERKONST) ||
	   (token == S_REALKONST) ) goto L_FACTOR;
      break;
  }
  compute_valtype(gp, (*op));
  return(token);
}

int op_m(struct cw_status *gp, struct commonblock *block, int s,
	 struct oper **node, int intoken, int level, char lefttype)
{
  int token;
  struct oper FAR *f_part=NULL, FAR *tmark=NULL, FAR *optrav=NULL;
  char f_type, optype;

  switch (level) {
    case 1:	switch (intoken) {
			case S_OR : break;
			default :   return(intoken);
		}
		break;
    case 2:	switch (intoken) {
			case S_AND: break;
			default :   return(intoken);
		}
		break;
    case 3:	switch (intoken) {
			case S_BITOR : break;
			default :   return(intoken);
		}
		break;
    case 4:	switch (intoken) {
			case S_BITAND : break;
			default :   return(intoken);
		}
		break;
    case 5:	switch (intoken) {
			case S_EQ : break;
			case S_NE : break;
			default :   return(intoken);
		}
		break;
    case 6:	switch (intoken) {
			case S_LT : break;
			case S_LE : break;
			case S_GE : break;
			case S_GT : break;
			default :   return(intoken);
		}
		break;
    case 7:	switch (intoken) {
			case S_LSHIFT : break;
			case S_RSHIFT : break;
			default :   return(intoken);
		}
		break;
    case 8:	switch (intoken) {
			case S_ADD : break;
			case S_SUB : break;
			default :   return(intoken);
		}
		break;
    case 9:	switch (intoken) {
			case S_MUL : break;
			case S_DIV : break;
			case S_REST : break;
			default :   return(intoken);
		}
		break;
    case 10:	switch (intoken) {
			case S_PRIMARY : break;
			default :   return(intoken);
		}
		break;
    default:	yyerror(gp, ErrLEV10);
  }

  if ((*node = NEWOPER) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  (*node)->type = DO_VDECL;
  (*node)->line = gp->LEXyy.line;

  token = op_(gp, block, s, &f_part, level+1);

#if 0
 if (!f_part)
   printf("f_part NULL, op_m calls op_, level %d\n",level);
#endif

  f_type = compute_valtype(gp, f_part) == T_INT ? INTOPER : FLOATOPER;
  (*node)->flag = optype = (lefttype==INTOPER && f_type==INTOPER) ?
    INTOPER : FLOATOPER;
  compute_valtype(gp, *node);
  /* now determine and set the proper function for the operator */
  switch (intoken) {
	case S_OR:	if (optype==INTOPER) (*node)->operation.lng = l_or;
			else (*node)->operation.flt = r_or;
			break;
	case S_AND:	if (optype==INTOPER) (*node)->operation.lng = l_and;
			else (*node)->operation.flt = r_and;
			break;
	case S_BITOR:	if (optype==INTOPER) (*node)->operation.lng = l_bitor;
			else (*node)->operation.flt = r_bitor;
			break;
	case S_BITAND:	if (optype==INTOPER) (*node)->operation.lng = l_bitand;
			else (*node)->operation.flt = r_bitand;
			break;
	case S_EQ:	if (optype==INTOPER) (*node)->operation.lng = l_eq;
			else (*node)->operation.flt = r_eq;
			break;
	case S_NE:	if (optype==INTOPER) (*node)->operation.lng = l_ne;
			else (*node)->operation.flt = r_ne;
			break;
	case S_LT:	if (optype==INTOPER) (*node)->operation.lng = l_lt;
			else (*node)->operation.flt = r_lt;
			break;
	case S_LE:	if (optype==INTOPER) (*node)->operation.lng = l_le;
			else (*node)->operation.flt = r_le;
			break;
	case S_GE:	if (optype==INTOPER) (*node)->operation.lng = l_ge;
			else (*node)->operation.flt = r_ge;
			break;
	case S_GT:	if (optype==INTOPER) (*node)->operation.lng = l_gt;
			else (*node)->operation.flt = r_gt;
			break;
	case S_LSHIFT:	(*node)->operation.lng = l_lshift;
			break;
	case S_RSHIFT:	(*node)->operation.lng = l_rshift;
			break;
	case S_ADD:	
	  if(optype == TEXTOPER)
	    (*node)->operation.txt = t_add;
	  else if (optype==INTOPER) 
	    (*node)->operation.lng = l_add;
	  else 
	    (*node)->operation.flt = r_add;
			break;
	case S_SUB:	if (optype==INTOPER) (*node)->operation.lng = l_sub;
			else (*node)->operation.flt = r_sub;
			break;
	case S_MUL:	if (optype==INTOPER) (*node)->operation.lng = l_mul;
			else (*node)->operation.flt = r_mul;
			break;
	case S_DIV:
	  (*node)->flag = optype = FLOATOPER;
	  (*node)->operation.flt = r_div;
	  break;
	case S_REST:	if (optype==INTOPER) (*node)->operation.lng = l_rest;
			else (*node)->operation.flt = r_rest;
			break;
	case S_PRIMARY:	if (optype==INTOPER) (*node)->operation.lng = l_pow;
			else (*node)->operation.flt = r_pow;
			break;
    }

  f_type = (*node)->flag;
  (*node)->right.op = f_part;
  /* then read T'-part of T = FT' */
  token = op_m(gp, block, s, &tmark, token, level, f_type);

  if (tmark) {
    /* assuming now: F-part is a valid tree, except possibly missing its
       bottom left branch, while the T'-part is definitely missing
       its bottom left branch, which is where the F-part fits */
    for (optrav = tmark;
	(optrav->flag == INTOPER || optrav->flag == FLOATOPER)
	   && optrav->left.op ;
	optrav=optrav->left.op);
    optrav->left.op = *node;
    *node = tmark;
  }
  compute_valtype(gp, *node);
  return(token);
}

int op_(struct cw_status *gp, struct commonblock FAR *block, int s,
	struct oper **node, int level) 
{
  int token,oldtoken;
  struct oper FAR *f_part=NULL, FAR *tmark=NULL, FAR *optrav=NULL;
  char f_type; /* indicates the type of the f_part (int or float) */
  
  if (level==OP_LEVELS) {
    switch (token = yylex(gp)) {
    case S_NOT :
    case S_ADD :
    case S_SUB : 
    case S_ADDADD :
    case S_EXPRVAL:
      if ((*node = NEWOPER) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      (*node)->type = DO_VDECL;
      (*node)->line = gp->LEXyy.line;
      break;
    default    : 
      token = factor(gp, block, s, node, token);
      return(token);
    }
    oldtoken = token;
    token = yylex(gp);
    token = factor(gp, block, s, &f_part, token);
    if (f_part==NULL)
      yyerror(gp, ErrUNOPNOOP);
    else {
      if (compute_valtype(gp, f_part) == T_INT
	  || valtype(f_part) == T_VOID) {
	(*node)->flag = INTOPER;
	switch(oldtoken) {
	case S_NOT : 
	  (*node)->operation.lng = l_not;
	  break;
	case S_ADD : 
	  (*node)->operation.lng = l_unadd;
	  break;
	case S_SUB : 
	  (*node)->operation.lng = l_unsub;
	  break;
	case S_EXPRVAL:
	  (*node)->operation.lng = l_exprval;
	   break;
	case S_ADDADD:
	  (*node)->operation.lng = l_preplus;
	  break;
	}
      } else if(valtype(f_part) == T_FLOAT){
	(*node)->flag = FLOATOPER;
	switch(oldtoken) {
	case S_NOT : 
	  (*node)->operation.flt = r_not;
	  break;
	case S_ADD : 
	  (*node)->operation.flt = r_unadd;
	  break;
	case S_SUB : 
	  (*node)->operation.flt = r_unsub;
	  break;
	case S_EXPRVAL:
	  (*node)->operation.flt = r_exprval;
	  break;
	}
      } else {
	(*node)->flag = TEXTOPER;
	switch(oldtoken) {
	case S_EXPRVAL:
	  (*node)->operation.txt = t_exprval;
	  break;
	}
      }
    }
    (*node)->left.op = f_part;
    compute_valtype(gp, *node);
    return(token);
  }
  /* not highest level operator, first read F-part of T = FT' */
  token = op_(gp, block, s, &f_part, level+1);
#if 0
  if (!f_part) printf("f_part NULL, op_ calls op_, level %d\n",level);
#endif
  f_type = compute_valtype(gp, f_part)==T_INT ? INTOPER : FLOATOPER;
  /* then read T'-part of T = FT' */
  token = op_m(gp, block, s, &tmark, token, level, f_type);
  *node = f_part;
  if (tmark) {
    /* assuming now: F-part is a valid tree, possibly missing its bottom
       left node, while the T'-part is definitely missing its
       bottom left branch, which is where the F-part fits */

    for (optrav = tmark;
	 (optrav->flag == INTOPER || optrav->flag == FLOATOPER) 
	   && optrav->left.op ;
	 optrav=optrav->left.op);

    optrav->left.op = f_part;
    *node = tmark;
  }
  return(token);
}

int const_int_decl(struct cw_status *gp, struct commonblock FAR *block, int s)
{
  int token, ssub = FALSE;
  struct c_decl FAR *c;
  char errormsg[256];

L_CONST_INT_DECL:
  token = yylex(gp);
  switch (token) {
  case S_IDENT : 
    /* New constant name = gp->LEXyy.ident */
    if ((block->lastc->name =
	 (char FAR *) CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    strcpy(block->lastc->name,gp->LEXyy.ident);
    if ((token = yylex(gp)) == S_ASSIGN) {
      token = yylex(gp);
      if (token == S_SUB) {
	ssub = TRUE;
	token = yylex(gp);
      }
      if (token == S_INTEGERKONST) {
	if (ssub)
	  block->lastc->thisc.value = - gp->LEXyy.lval;
	else
	  block->lastc->thisc.value = gp->LEXyy.lval;
	block->lastc->type = INTVAL;
	token = yylex(gp);
      } else {
	yyerror(gp, ErrCONSTILLASG);
	token = yylex(gp);
	goto L_PAREXPSEP;
      }
    } else {
      sprintf(errormsg, ErrCONSTEXPEQ, (char FAR *) gp->LEXyy.match_buffer);
      yyerror(gp, errormsg);
      goto L_PAREXPSEP;
    }
    break;
  default: 
    sprintf(errormsg, ErrCONSTEXPID, (char FAR *) gp->LEXyy.match_buffer);
    yyerror(gp, errormsg);
    break;
  }

L_PAREXPSEP :
  if (token == S_PAREXPSEP) {
    if ((c = NEWCDECL) != NULL) {
      place_next(block->lastc,c);
      block->lastc = c;
    } else {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    ssub = FALSE;
    goto L_CONST_INT_DECL;
  } else {
    if (token == S_STATSEP) {
      return(token);
    }
    token = error(gp, S_PAREXPSEP, ErrINTDECL);
    if (token == S_PAREXPSEP) 
      goto L_PAREXPSEP;
  }
  return 0;
}


int const_float_decl(struct cw_status *gp, struct commonblock *block, int s)
{
  int token;
  struct c_decl FAR *c;
  char errormsg[256];

L_CONST_FLOAT_DECL:
  token = yylex(gp);
  switch( token ){
  case S_IDENT : 
    /* Ny konstant navn = gp->LEXyy.ident */
    if ((block->lastc->name = (char FAR *) 
	 CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    strcpy(block->lastc->name,gp->LEXyy.ident);
    if ( ( token = yylex(gp) ) == S_ASSIGN ){
      if ( ( token = yylex(gp) ) == S_REALKONST ){
	block->lastc->thisc.fvalue = gp->LEXyy.rval;
	block->lastc->type = FLOATVAL;
	token = yylex(gp);
      }else{
	yyerror(gp, ErrCONSTILLASG);
	token = yylex(gp);
	goto L_PAREXPSEP;
      }
    }else{
      sprintf(errormsg, ErrCONSTEXPEQ, (char FAR *) gp->LEXyy.match_buffer);
      yyerror(gp, errormsg);       
      goto L_PAREXPSEP;
    }
    break;
  default : 
    sprintf(errormsg, ErrCONSTEXPID, (char FAR *) gp->LEXyy.match_buffer);
    yyerror(gp, errormsg);
    break;
  }

L_PAREXPSEP:
  if ( token == S_PAREXPSEP ){
    if ( (c = NEWCDECL) != NULL){
      place_next(block->lastc,c);
      block->lastc = c;
    }else{
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    goto L_CONST_FLOAT_DECL;
  }else{
    if ( token == S_STATSEP ){
      return(token);
    }
    token = error(gp, S_PAREXPSEP, ErrFLTDECL);
    if ( token == S_PAREXPSEP ) 
      goto L_PAREXPSEP;
  }
  return 0;
}
               
int const_text_decl(struct cw_status *gp, struct commonblock *block, int s)
{
  int token;
  struct c_decl FAR *c;
  char errormsg[256];

L_CONST_TEXT_DECL :
  token = yylex(gp);
  switch (token) {
     case S_IDENT : 
       /* Ny konstant navn = gp->LEXyy.ident */
       if ((block->lastc->name =
	    (char *)CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
	 errorMsg(gp, 2, ErrNOMOREMEM);
	 c_exit(gp, NOT_OK);
       }
       strcpy(block->lastc->name,gp->LEXyy.ident);
       if ((token = yylex(gp)) == S_ASSIGN) {
	 token = yylex(gp);
	 if (token == S_TTEXTKONST) {
	   block->lastc->thisc.text = gp->LEXyy.tval;
	   block->lastc->type = TEXTVAL;
	   token = yylex(gp);
	 } else {
	   yyerror(gp, ErrCONSTILLASG); 
	   token = yylex(gp);
	   goto L_PAREXPSEP;
	 }
       } else {
	 sprintf(errormsg, ErrCONSTEXPEQ, (char FAR *) gp->LEXyy.match_buffer);
	 yyerror(gp, errormsg);
	 goto L_PAREXPSEP;
       }
       break;
  default: 
    sprintf(errormsg, ErrCONSTEXPID, (char FAR *) gp->LEXyy.match_buffer);
    yyerror(gp, errormsg);
    break;
  }

  L_PAREXPSEP :
    if (token == S_PAREXPSEP) {
      if ((c = NEWCDECL) != NULL) {
	place_next(block->lastc,c);
	block->lastc = c;
      } else {
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      goto L_CONST_TEXT_DECL;
    } else {
      if (token == S_STATSEP) {
	return(token);
      }
      token = error(gp, S_PAREXPSEP, ErrTXTCONDEC);
      if (token == S_PAREXPSEP) 
	goto L_PAREXPSEP;
    }
  return 0;
}


int const_decl(struct cw_status *gp, struct commonblock FAR *block, int s)
{
  int token;
  struct c_decl FAR *c;

  if (block->lastc != NULL) {
    if ((c = NEWCDECL) != NULL) {
      place_next(block->lastc, c);
      block->lastc = c;
    } else {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
  } else {
    if ((block->lastc = NEWCDECL) != NULL)
      init_list(block->firstc = block->lastc);
    else {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
  }
  token = yylex(gp);
  switch (token) {
    case S_INT    : token = const_int_decl(gp, block, s); break;
    case S_FFLOAT : token = const_float_decl(gp, block, s); break;
    case S_TTEXT  : token = const_text_decl(gp, block, s); break;
    default       : 
      yyerror(gp, ErrCONSTMISSTYP);
      break;
  }
L_STATSEP :
  if (token == S_STATSEP)
    return(yylex(gp));
  else if(token == 0)
    return 0;
  else {
    token = error(gp, s,ErrERICONDEC);
    if (token == S_STATSEP) 
      goto L_STATSEP;
  }
  return 0; /* Should never be executed */
}


/*
 * Parse array initialization for one dimension, and call function
 * recursively for each sub-dimension.
 */
int array_init(struct cw_status *gp, struct commonblock *block,
	       struct vinit *vi, int dim)
{
  int token;
  struct vinit FAR *nvi;
  int firsttime=TRUE;

  if ((token = yylex(gp)) == S_BEGIN) {
    if(dim == 0){
      while(token != S_END){
	if(firsttime == FALSE){
	  if((nvi = NEWVINIT) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  place_next(vi, nvi);
	  vi = nvi;
	} else
	  firsttime = FALSE;
	token = op_(gp, block, S_END, &vi->exp, 1);
      }
    } else {
      while(token != S_END && token != S_ERROR){
	if(firsttime == FALSE){
	  if((nvi = NEWVINIT) == NULL){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  place_next(vi, nvi);
	  vi = nvi;
	} else
	  firsttime = FALSE;
	if((nvi = NEWVINIT) == NULL){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	init_list(nvi);
	vi->next_dim = nvi;
	token = array_init(gp, block, nvi, dim - 1);
	if(token == S_END)
	  token = yylex(gp);
      }
    }
    return token;
  } else {
    error(gp, S_STATSEP, ErrTOOFEWBRA);
    return 0;
  } 
}


int array_decl(struct cw_status *gp, struct commonblock *block,
	       struct vdecl *vd)
{
  int token=S_BEGBRACKET;
  struct param FAR *ind, FAR *lind=NULL;
  
  do {
    if((ind = NEWPARAM) == NULL){
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    ind->line = gp->LEXyy.line;
    vd->ndim++;
    if(lind == NULL){
      init_list(ind);
      vd->dim_size = ind;
    } else
      place_next(lind, ind);
    lind = ind;
    token = op_(gp, block, S_ENDBRACKET, &ind->exp, 1);
    token = yylex(gp);
  } while(token == S_BEGBRACKET);

  if (token == S_ASSIGN) {
    if((vd->init = NEWVINIT) == NULL){
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    init_list(vd->init);
    token = array_init(gp, block, vd->init, vd->ndim - 1);
    if(token == S_END){
      token = yylex(gp);
      return token;
    } else if(token == S_ERROR){
      CalFree(vd->init);
      vd->init = NULL;
      return 0;
    } 
    
  }
  return token;
}

int int_decl(struct cw_status *gp, struct commonblock FAR *block, int s)
{
  int token;
  struct vdecl FAR *v;

  if((v = NEWVDECL) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  insert_variable(block, v, FALSE);

L_INT_DECL:
  switch (token = yylex(gp)) {
  case S_IDENT : /* New variable name = gp->LEXyy.ident */
    if ((v->name = (char FAR *)CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    strcpy(v->name, gp->LEXyy.ident);
    v->type = INTVAL;
    switch (token = yylex(gp)) {
    case S_ASSIGN     :
      if((v->init = NEWVINIT) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      init_list(v->init);
      token = op_(gp, block, S_STATSEP, &v->init->exp, 1);
      goto L_PAREXPSEP;
    case S_BEGBRACKET :
      v->type = INTARRAY;
      token = array_decl(gp, block, v);
    default :
      goto L_PAREXPSEP;
    }
    token = yylex(gp);
    break;
  default :     
    yyerror(gp, ErrINTIDNAM);
    break;
  }
L_PAREXPSEP:
  if (token == S_PAREXPSEP) {
    if ((v = NEWVDECL) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    insert_variable(block, v, FALSE);
    goto L_INT_DECL;
  } else {
    if (token == S_STATSEP) {
      return(token = yylex(gp));
    }
    token = error(gp, S_PAREXPSEP, ErrININTDECL);
    if (token == S_PAREXPSEP) 
      goto L_PAREXPSEP;
  }
  return 0; /* Should never be executed */
}


int float_decl(struct cw_status *gp, struct commonblock FAR *block, int s)
{
  int token;
  struct vdecl FAR *v;

  if((v = NEWVDECL) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  insert_variable(block, v, FALSE);

L_FLOAT_DECL:
  switch (token = yylex(gp)) {
  case S_IDENT : /* New variable name = gp->LEXyy.ident */
    if ((v->name = (char FAR *)CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    strcpy(v->name, gp->LEXyy.ident);
    v->type = FLOATVAL;
    switch (token = yylex(gp)) {
    case S_ASSIGN     :
      if((v->init = NEWVINIT) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      init_list(v->init);
      token = op_(gp, block, S_STATSEP, &v->init->exp, 1);
      goto L_PAREXPSEP;
    case S_BEGBRACKET :
      v->type = FLOATARRAY;
      token = array_decl(gp, block, v);
    default :
      goto L_PAREXPSEP;
    }
    token = yylex(gp);
    break;
  default : 
    yyerror(gp, ErrFLTIDNAM);
    break;
  }
L_PAREXPSEP:
  if (token == S_PAREXPSEP) {
    if ((v = NEWVDECL) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    insert_variable(block, v, FALSE);
    goto L_FLOAT_DECL;
  } else if (token == S_STATSEP) 
    return(token = yylex(gp));
  else {
    token = error(gp, S_STATSEP,ErrINFLODECL);
    if (token == S_PAREXPSEP) 
      goto L_FLOAT_DECL;
    else if (token == S_STATSEP)
      goto L_PAREXPSEP;
  }
  return 0; 
}

int text_decl(struct cw_status *gp, struct commonblock FAR *block, int s) 
{
  int token;
  struct vdecl FAR *v;

  if((v = NEWVDECL) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  insert_variable(block, v, FALSE);

L_TEXT_DECL:
  switch (token = yylex(gp)) {
  case S_IDENT : /* New variable name = gp->LEXyy.ident */
    if ((v->name = (char FAR *)CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    strcpy(v->name, gp->LEXyy.ident);
    v->type = TEXTVAL;
    switch (token = yylex(gp)) {
    case S_ASSIGN     :
      if((v->init = NEWVINIT) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      init_list(v->init);
      token = op_(gp, block, S_STATSEP, &v->init->exp, 1);
      goto L_PAREXPSEP;
    case S_BEGBRACKET :
      v->type = TEXTARRAY;
      token = array_decl(gp, block, v);
    default :
      goto L_PAREXPSEP;
    }
    token = yylex(gp);
    break;
  default : 
    yyerror(gp, ErrTXTIDNAM);
    break;
  }
L_PAREXPSEP:
  if (token == S_PAREXPSEP) {
    if ((v = NEWVDECL) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    insert_variable(block, v, FALSE);
    goto L_TEXT_DECL;
  } else {
    if (token == S_STATSEP) {
      return(token = yylex(gp));
    }
    token = error(gp, S_PAREXPSEP, ErrINTXTDECL);
    if (token == S_PAREXPSEP) 
      goto L_PAREXPSEP;
  }
  return 0;
}

int declarations(struct cw_status *gp, struct commonblock *block, int s,
		 int intoken) 
{
  int token;

  token = intoken;
  token = yylex(gp);
L_S_DECL:
  switch (token) {
    case S_INT      : 
      token = int_decl(gp, block, s); goto L_S_DECL;
    case S_FFLOAT   : 
      token = float_decl(gp, block, s); goto L_S_DECL;
    case S_TTEXT    :
      token = text_decl(gp, block, s);
      goto L_S_DECL;
    case S_CONST    : 
      token = const_decl(gp, block, s); goto L_S_DECL;


#ifdef OLD
    case S_NOSYMBOL : 
      yyerror(ErrNSIMCOD); break;
    default         : 
      if ( ( token == S_WHILE ) || (token == S_IF ) ||
	   ( token == S_FOR ) || (token == S_SWITCH ) ||
	   ( token == S_IDENT ) || token == S_RETURN ) 
      return(token);
      else {
	token = error(S_STATSEP, ErrERRINIFUN);
	goto L_S_DECL;
      }
#endif

  }
  return(token);
}

int EQUATION(struct cw_status *gp, struct commonblock FAR *block, int s,
	     struct sim_actions FAR *sa) 
{
  struct vdecl FAR *var;
  struct bfunction  FAR *fp;
  struct attrparam FAR *ap;
  struct intarray *arr;
  int token;
  char errormsg[256];
  
  if(is_lvalue(block, gp->LEXyy.ident)){
    struct oper FAR *lop;
    
    if ((sa->eq = NEWOPER) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      return(NOT_OK);
    }
    sa->eq->line = gp->LEXyy.line;
    
    if((lop = sa->eq->left.op = NEWOPER) == NULL){
      errorMsg(gp, 2, ErrNOMOREMEM);
      return(NOT_OK);
    }
    lop->line = gp->LEXyy.line;
    if ((var = get_var(block,gp->LEXyy.ident)) != NULL) {
      switch (var->type) {
      case  INTARRAY :
      case  FLOATARRAY :
      case  TEXTARRAY :
	token = yylex(gp);
	if (token != S_BEGBRACKET) goto L_ERROR;
	if ((lop->left.array = NEWINTARRAY) == NULL) {
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  return(NOT_OK);
	}
	lop->left.array->which.vd = var;
	lop->left.array->type = DO_VDECL;
	if ((lop->left.array->indeks = NEWPARAM) == NULL) {
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  return(NOT_OK);
	}
	lop->left.array->indeks->line = gp->LEXyy.line;
	init_list(lop->left.array->indeks);
	token  = parameters(gp, block, S_ENDBRACKET, token,
			    lop->left.array->indeks);
	lop->flag = var->type | T_ARRAY;
	break;
      case  INTVAL :
      case  FLOATVAL :
      case  TEXTVAL :
	lop->type = DO_VDECL;
	lop->left.vd = var;
	lop->flag = var->type | T_VAR;
	token = yylex(gp);
	break;
      }
    } else if((ap = get_attr((struct objectproto *)block, gp->LEXyy.ident))
	      != NULL){
      if(ap->type != T_EXPR){
	sprintf(errormsg, "Attribute parameter '%s' as lvalue", gp->LEXyy.ident);
	yyerror(gp, errormsg);
	goto L_ERROR;
      } 
      lop->left.ap = ap;
      lop->flag = ATTROPER;
      token = yylex(gp);
      if(token == S_BEGBRACKET){
	lop->flag |= T_ARRAY;
	if ((arr = lop->right.array = NEWINTARRAY) == NULL){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	lop->right.array->which.vd = var;
	lop->right.array->type = DO_VDECL;

	if ((arr->indeks = NEWPARAM) == NULL){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	arr->indeks->line = gp->LEXyy.line;
	init_list(arr->indeks);
	token = parameters(gp, block, S_ENDBRACKET,token,arr->indeks);
      }
    }
    compute_valtype(gp, lop);
    if(token != S_ASSIGN) 
	goto L_ERROR;
    token = op_(gp, block, s, &sa->eq->right.op, 1);
    sa->eq->type = DO_VDECL;
    switch(valtype(lop)){
    case T_TEXT:
      sa->eq->operation.txt = t_ass;
      sa->eq->flag = TEXTOPER;
      break;
    case T_FLOAT:
      sa->eq->operation.flt = r_ass;
      sa->eq->flag = FLOATOPER;
      break;
    case T_INT:
      sa->eq->operation.lng = l_ass;
      sa->eq->flag = INTOPER;
      break;
    default:
      sa->eq->operation.lng = l_ass;
      sa->eq->flag = INTOPER;
/* sa->eq->right.op->flag; */
    }
    compute_valtype(gp, sa->eq);
    goto L_STATSEP;
  } else {
    if ((fp = get_func(gp->LEXyy.ident)) != NULL) {
      if ((sa->do_act = NEWACTION) == NULL) {
	errorMsg(gp, 2, ErrNOMOREMEM);
	return(NOT_OK);
      }
     sa->do_act->line = gp->LEXyy.line;
      switch(fp->type){
      case BINTFUNC:
	sa->do_act->function.do_act = fp->do_act;
	break;
      case BFLOATFUNC:
	sa->do_act->function.fdo_act = fp->fdo_act;
	break;
      case BTEXTFUNC:
	sa->do_act->function.tdo_act = fp->tdo_act;
	break;
      default:
	sprintf(errormsg, ErrSIMINTERR, fp->type);
	errorMsg(gp, 1, errormsg);
	return(NOT_OK);
      }
      sa->do_act->type = fp->type;
      token = yylex(gp);
      if ( token != S_BEGPAR ) goto L_STATSEP;
      if ((sa->do_act->actual.par = NEWPARAM) == NULL) {
	errorMsg(gp, 2, ErrNOMOREMEM);
	return(NOT_OK);
      }
      sa->do_act->actual.par->line = gp->LEXyy.line;
      init_list(sa->do_act->actual.par);
      token = parameters(gp, block, S_ENDPAR, token, sa->do_act->actual.par);
      if (token != S_ENDPAR)
	goto L_ERROR;
      else {
	token = yylex(gp);
	goto L_STATSEP;
      }
    } else {
      /* Assume user defined function */
      struct notdecl FAR *nd;
      nd = c_notdecl(gp);

      nd->name = (char FAR *) CalMalloc(strlen(gp->LEXyy.ident) + 1);
      if(nd->name == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      strcpy(nd->name, gp->LEXyy.ident);
      nd->type = 'F';
      if ((sa->do_act = NEWACTION) == NULL) {
	errorMsg(gp, 2, ErrNOMOREMEM);
	return(NOT_OK);
      }
      sa->do_act->line = gp->LEXyy.line;
      nd->do_act = sa->do_act;
      /* Insert new element first in list */
      if(gp->curlunit->firstnd != NULL)
	place_prev(gp->curlunit->firstnd, nd);
      gp->curlunit->firstnd = nd;
      token = yylex(gp);
      if(token != S_BEGPAR) 
	goto L_STATSEP;
      if((nd->do_act->actual.par = NEWPARAM) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      nd->do_act->actual.par->line = gp->LEXyy.line;
      init_list(nd->do_act->actual.par);
      token = parameters(gp, block, S_ENDPAR, token, nd->do_act->actual.par);
      if (token != S_ENDPAR)
	goto L_ERROR;
      else {
	token = yylex(gp);
	goto L_STATSEP;
      }
    }
  }
L_ERROR:
  token = error(gp, s, ErrERRILLSTA);
L_STATSEP:
  if (token == s) return(token);
  token = error(gp, s,ErrERRILLSTA);
  return(token);
}

int sequence(struct cw_status *gp, struct commonblock *block, int s,
	     int intoken, struct sim_actions *sa);

int while_sentence(struct cw_status *gp, struct commonblock *block,
		   int s, struct sim_actions *sa) 
{
  /* first create while object one level inside statement list */
  int token;

  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);
  sa = sa->cond_act;
  sa->actions = WHILE;			/* function pointer */
  token = yylex(gp);

  /* read and test condition in parentheses */
  if ( token != S_BEGPAR ){
    error(gp, ';', ErrNOTLEFTWT);
    return(token);
  }
  token = op_(gp, block, s, &sa->eq_op, 1);

  /* then read block, single statement or ";" */
  token = yylex(gp);

  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);
  
  token = statement(gp, block, s, token, sa->cond_act);
  return(token);
}

int if_sentence(struct cw_status *gp, struct commonblock *block,
		int s, struct sim_actions *sa) 
{
  int token;

  /* first create if object one level inside statement list */
  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);
  sa = sa->cond_act;
  sa->actions = IF;	/* function pointer */

  /* read and test condition in parentheses */
  token = yylex(gp);
  if ( token != S_BEGPAR ){
    yyerror(gp, ErrNOTLEFTIT);
    if ( token != S_STATSEP )
      token = error(gp, s, ErrNOTSEMIIF);
    return(token=yylex(gp));
  }
  token = op_(gp, block, s, &sa->eq_op, 1);
  /* then read block, single statement or ";" */
  token = yylex(gp);

  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);

  token = statement(gp, block, s, token, sa->cond_act);
  token = else_sentence(gp, block, s, sa, token);
  return token;
}

int else_sentence(struct cw_status *gp, struct commonblock *block, int s,
		  struct sim_actions *sa, int token) 
{
  struct sim_actions FAR *sim;

  if (token != S_ELSE) return(token);

  if((sim = NEWSIMACTION) == NULL){ 
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  place_next(sa,sim);
  sa = sim;
  token = yylex(gp);


  token = statement(gp, block, s, token, sa);
  return(token);
}

int for_sentence(struct cw_status *gp, struct commonblock *block,
		   int s, struct sim_actions *sa) 
{
  struct sim_actions FAR *la, *sim;
  int token;

  if ((sim  = NEWSIMACTION) == NULL ) {
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  place_next(sa,sim);

  token = yylex(gp);
  if ( token != S_BEGPAR ){
    yyerror(gp, ErrNOTLEFTFS);
    return(token);
  }
  token = yylex(gp);

  /* Place initializer statement _before_ the for-statement */
/*   token = statement(block, S_STATSEP, token, sa); */

  token = EQUATION(gp, block, S_STATSEP, sa); 
  
  sa = sim;

  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);
  sa = sa->cond_act;
  sa->actions = FOR;


#ifdef OLD
  if ( (la = NEWSIMACTION) == NULL ){
    errorMsg(2, ErrNOMOREMEM);
    c_exit(NOT_OK);
  }
  place_next(sa, la);
  sa = la;
#endif

  token = op_(gp, block, S_STATSEP, &sa->eq_op, 1);

  if ( (la = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }  
  init_list(la);

/*   token = statement(func, S_ENDPAR, token, la); */
  token = yylex(gp);
  token = EQUATION(gp, block, S_ENDPAR, la); 

  if ( token != S_ENDPAR ) {
    token = error(gp, S_ENDPAR, ErrNOTRIGHFS);
    if (token == S_NOSYMBOL) return(token);
  }

  token = yylex(gp);
  
  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);
  
  token = statement(gp, block, s, token, sa->cond_act);
  place_next(last(sa->cond_act), la);
  return(token);
}

int switch_sentence(struct cw_status *gp, struct commonblock FAR *block,
		    int s, struct sim_actions FAR *sa) 
{
  int token;
  struct sim_actions FAR *fnostat = NULL, FAR *la1, FAR *la2, FAR *ha;

  if ( (sa->cond_act = NEWSIMACTION) == NULL ){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(sa->cond_act);
  sa = sa->cond_act;
  sa->actions = SWITCH;
  token = yylex(gp);

  if ( token != S_BEGPAR ) {
    yyerror(gp, ErrNOTLEFTSS);
    return(token);

  }
  token = op_(gp, block, S_ENDPAR, &sa->eq_op, 1);

  /* then read block, single statement or ";" */
  token = yylex(gp);
  switch ( token ) {
  case S_BEGIN : 
    if ( (sa->cond_act = NEWSIMACTION) == NULL ){
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    init_list(sa->cond_act);
    la1 = sa;
    token = sequence(gp, block, S_END, token, sa->cond_act);
    while (token != S_END) {
      switch (token) {
      case S_CASE:
	if ( (ha = NEWSIMACTION) == NULL ){
	  errorMsg(gp, 2, ErrNOMOREMEM);
	  c_exit(gp, NOT_OK);
	}
	place_next(sa,ha);
	sa = ha;
	if (!fnostat) fnostat = sa;
	token = op_(gp, block, S_LABELSEP, &la1->eq_op, 1);
	if (token != S_LABELSEP) {
	  error(gp, S_END,ErrCOLMISACE);
	  return(token);
	}
	token = sequence(gp, block, S_END, token, la1->cond_act);
	break;
      case S_BREAK:
	if (la1->cond_act) {
	  for (la2 = la1->cond_act; 
	       next(la2); la2 = (void FAR *)next(la2));
	  if ( (ha = NEWSIMACTION) == NULL ){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  place_next(la2,ha);
	} else {
	  if ( (la1->cond_act = NEWSIMACTION) == NULL ){
	    errorMsg(gp, 2, ErrNOMOREMEM);
	    c_exit(gp, NOT_OK);
	  }
	  init_list(la1->cond_act);
	}
	while (fnostat) {
	  if (fnostat != la1) {
	    fnostat->cond_act = la1->cond_act;
	    fnostat = next(fnostat);
	  } else
	    fnostat = NULL;
	}
	token = yylex(gp);
	if ((token != S_CASE) || (token != S_END))
	  token = error(gp, s, ErrUNRSIMCOD);
	break;
      default :
	token = error(gp, s, ErrSWIMISRIG);
	break;
      }
    }
    if ( token != S_END )
      token = error(gp, s, ErrSWIMISRIG);
    return(token);
  case S_STATSEP :
  case S_IDENT : 
    token = error(gp, S_STATSEP, ErrSWIMISLEF);
    return(token);
    
  }
  return(token);
}

int return_sentence(struct cw_status *gp, struct commonblock FAR *block,
		    int s, struct sim_actions FAR *sa)
{
  int token;

  token = op_(gp, block, s, &sa->ret_act, 1);
  return token;
}

/*
 * Read a sequence of statements.
 */
int sequence(struct cw_status *gp, struct commonblock FAR *block, int s,
	     int intoken, struct sim_actions FAR *sa) 
{
  int token, first = TRUE;
  struct sim_actions FAR *sim, FAR *last = NULL;
  struct objectproto *object;

  token = intoken;
  
L_S_SENTENCE:

  if ( token == S_NOSYMBOL || token == S_END ) {
    return(token);
  }
  if ( first != TRUE ) {
    if ((sim  = NEWSIMACTION) == NULL ) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    place_next(last,sim);
    last = sim;
  } else if ( last == NULL )
    last = sa;
  switch (token) {
  case S_BEGIN    :
    token = yylex(gp);
    goto L_S_SENTENCE;
  case S_WHILE    :
    token = while_sentence(gp, block, s, last);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_FOR      :
    token = for_sentence(gp, block, s, last);
    last = next(last);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_SWITCH   :
    token = switch_sentence(gp, block, s, last);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_IF       :
    token = if_sentence(gp, block, s, last);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_IDENT    :
    if((object = get_object(block, gp->LEXyy.ident)) != NULL)
      token = P_object(gp, object, block, &last->do_act, token);
    else
      token = EQUATION(gp, block, S_STATSEP, last);
    token = yylex(gp);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_RETURN:
    token = return_sentence(gp, block, s, last);
    token = yylex(gp);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_STATSEP :
    token = yylex(gp);
    first = FALSE;
    goto L_S_SENTENCE;
  case S_WINDOW:
    token = P_start_window(gp, block, &last->win, token);
    first = FALSE;
    if(token == S_STATSEP)
      token = yylex(gp);
    goto L_S_SENTENCE;
  case S_BOX :
  case S_LINE :
  case S_ARC :
  case S_IMAGE :
  case S_POINT :
  case S_POLYGON :
  case S_TEXTOBJECT :
    token = P_graphobj(gp, block, &last->graph, token);
    first = FALSE;
    if(token == S_STATSEP)
      token = yylex(gp);
    goto L_S_SENTENCE;
  case S_INPUTAREA:
    token = P_inputobj(gp, block, &last->inp, token);
    first = FALSE;
    if(token == S_STATSEP)
      token = yylex(gp);
    goto L_S_SENTENCE;
  case S_END      : break;
  case S_NOSYMBOL : break;
  case S_CASE     : break;
  case S_BREAK    : break;
  default      :
    token = error(gp, S_END, ErrPARUNSTAT);
    goto L_S_SENTENCE;
  }
  return(token);
}


/*
 * Parse a compound statement. It should _always_ return the token after 
 * statement. 
 */
int statement(struct cw_status *gp, struct commonblock FAR *block, int s,
	      int intoken, struct sim_actions FAR *sa)
{
  char errormsg[256];
  int token;
  struct objectproto FAR *object;
  token = intoken;
  
  switch(token){
  case S_BEGIN:
    token = sequence(gp, block, s, token, sa);
    if(token != S_END){
      error(gp, S_END, WarnOPENBRAC);
    }
    token = yylex(gp);
    break;
  case S_WHILE:
    token = while_sentence(gp, block, s, sa);
    break;
  case S_FOR:
    token = for_sentence(gp, block, s, sa);
    sa = next(sa);
    break;
  case S_SWITCH:
    token = switch_sentence(gp, block, s, sa);
    break;
  case S_IF:
    token = if_sentence(gp, block, s, sa);
    break;
  case S_IDENT:
    if((object = get_object(block, gp->LEXyy.ident)) != NULL)
      token = P_object(gp, object, block, &sa->do_act, token);
    else
      token = EQUATION(gp, block, S_STATSEP, sa);
    break;
  case S_RETURN:
    token = return_sentence(gp, block, s, sa);
    break;
  case S_STATSEP :
    token = yylex(gp);
    break;
  case S_WINDOW:
    token = P_start_window(gp, block, &sa->win, token);
    break;
  case S_BOX :
  case S_LINE :
  case S_ARC :
  case S_IMAGE :
  case S_POINT :
  case S_POLYGON :
  case S_TEXTOBJECT :
    token = P_graphobj(gp, block, &sa->graph, token);
    break;
  case S_INPUTAREA:
    token = P_inputobj(gp, block, &sa->inp, token);
    break;
  default :
    sprintf(errormsg, ErrUNKNSYM, (char FAR *) gp->LEXyy.match_buffer);
    token = error(gp, S_STATSEP, errormsg); 
    break;
  }
  if(token == S_STATSEP)
    token = yylex(gp);
  return token;
}

/*
 * Parse a function declaration. 
 * Return the first token after the function definition.
 */
int readfunc(struct cw_status *gp, struct learnunit FAR *learn,
	     int token, int type, char FAR *name)
{
  struct function FAR *func;
  struct fparam FAR *f, FAR *p=NULL;
  struct vdecl FAR *vd;
  char errormsg[256];

  func = c_function(gp);
  if(learn->fp == NULL)
    init_list(func);
  else 
    place_prev(learn->fp, func);
  learn->fp = func;
  func->parent_lunit = learn;



  func->type = FUNCTIONBLOCK;
  func->rettype = type;
  func->name = name;

  if(token != S_BEGPAR){
    yyerror(gp, ErrFUNCNOPPAR);
    return 0;
  }
  token = yylex(gp);

  while(token != S_ENDPAR){
    if((vd = NEWVDECL) == NULL){
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }


    /* Place formal parameter last in list */
    f = c_fparam(gp, vd);


    if(p == NULL){
      func->fpar = p = f;
    } else {
      place_next(p, f);
      p = f;
    }


    if(token == S_REF){
      f->is_ref = TRUE;
      token = yylex(gp);
    }/* else
      func->fparvar = vd; */


    switch(token){
    case S_FFLOAT:
      vd->type = FLOATVAL;
      break;
    case S_INT:
      vd->type = INTVAL;
      break;
    case S_TTEXT:
      vd->type = TEXTVAL;
      break;
    default:
      sprintf(errormsg, ErrILLTYPIFD, (char FAR *) gp->LEXyy.match_buffer);
      token = error(gp, S_BEGIN, errormsg);
      goto L_DECL_START;
    }
    token = yylex(gp);
    if(token != S_IDENT){
      sprintf(errormsg, ErrPLEXPID, (char FAR *) gp->LEXyy.match_buffer);
      yyerror(gp, errormsg);
      return 0;
    }
    if(ident_declared((struct commonblock FAR *)func, gp->LEXyy.ident)){
      sprintf(errormsg, ErrPARALRDEC, (char FAR *) gp->LEXyy.ident);
      yyerror(gp, errormsg);
      return 0;
    }
    vd->name = strdup(gp->LEXyy.ident);

    token = yylex(gp);
    
    while(token == S_BEGBRACKET){
      token = yylex(gp);
      if(token != S_ENDBRACKET){
	yyerror(gp, ErrNOTCLOBRT);
	return 0;
      }
      token = yylex(gp);
      vd->ndim++;
      SET_ARRAY(vd->type);
      f->is_ref = TRUE;		/* Make all array parameters 
				 *  reference parameters 
				 */
    }

    insert_parameter(func, vd, f->is_ref);

    if(token != S_ENDPAR && token != S_PAREXPSEP){
      sprintf(errormsg, ErrIDILLSYM, (char FAR *) gp->LEXyy.match_buffer);
      yyerror(gp, errormsg);
      return 0;
    }
    if(token == S_PAREXPSEP)
      token = yylex(gp);
  }
/*  if(func->fparvar == NULL)
    func->fparvar = func->fref;*/   /* Make sure fparvar points to first parameter
				   * in the functions variable list.
				   */
  token = yylex(gp);
  if(token != S_BEGIN){
    sprintf(errormsg, ErrEXPOPBRAC, (char FAR *) gp->LEXyy.match_buffer);
      yyerror(gp, errormsg);
      return 0;
  }
L_DECL_START:
  token = declarations(gp, (struct commonblock *)func, S_END, token);
/*   token = readobjdecls(learn, func, token); */
  if((func->sim = NEWSIMACTION) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(func->sim);
  if ((token = sequence(gp, (struct commonblock *)func, S_END , 
			token, func->sim)) != S_END)
    token = error(gp, S_END, ErrABNTERMFU);
  token = yylex(gp);

  return token;
}

int readobject(struct cw_status *gp, struct learnunit FAR *learn, int token)
{
  struct objectproto FAR *og;
  struct attrparam FAR *attr;
  char errormsg[256];

  if((og = NEWOBJECTPROTO) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  if(learn->fobject == NULL){
    init_list(og);
  } else {
    place_prev(learn->fobject, og);
  }
  learn->fobject = og;
  og->parent_lunit = learn;

  token = yylex(gp);
  if(token != S_IDENT){
    yyerror(gp, ErrEXPFUNID);
    return 0;
  }
  og->name = strdup(gp->LEXyy.ident);
  og->type = OBJECTBLOCK;

  token = yylex(gp);

  if(token == S_BEGPAR){
    while(token != S_ENDPAR){
      token = yylex(gp);
      if(token == S_IDENT){
	if((attr = NEWATTRPARAM) == NULL){
	  error (gp, S_END, ErrCOLMISACE);
	  return(token);
	}
	if(! strcmp(gp->LEXyy.ident, "expr")){
	  attr->type = T_EXPR;
	} else if(! strcmp(gp->LEXyy.ident, "translation")){
	  attr->type = T_TRANSLATION;
	} else if(! strcmp(gp->LEXyy.ident, "points")){
	  attr->type = T_POINTS;
	} else {
	  sprintf(errormsg, "Expected attribute type specifier, found %s", 
		  gp->LEXyy.ident);
	  yyerror(gp, errormsg);
	}
	token = yylex(gp);
	if(token != S_IDENT){
	  sprintf(errormsg, "Expected attribute identifier, found %s", 
		  gp->LEXyy.match_buffer);
	  yyerror(gp, errormsg);
	}
	attr->name = strdup(gp->LEXyy.ident);
	if(og->fparam == NULL){
	  init_list(attr);
	} else {
	  place_prev(og->fparam, attr);
	}
	og->fparam = attr;
	if(token == S_PAREXPSEP)
	  token = yylex(gp);
      }
    }
  }
  token = yylex(gp);
  if(token != S_BEGIN){
    sprintf(errormsg, ErrEXPOPBRAC, (char FAR *) gp->LEXyy.match_buffer);
    yyerror(gp, errormsg);
    return 0;
  }
  token = declarations(gp, (struct commonblock *)og, S_END, token);
  if((og->sim = NEWSIMACTION) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  init_list(og->sim);
  if ((token = sequence(gp, (struct commonblock *)og, S_END , 
			token, og->sim)) != S_END)
    token = error(gp, S_END, ErrABNTERMFU);
  token = yylex(gp);
  return token;
}

int readvar(struct cw_status *gp, struct learnunit FAR *learn, int token,
	    int type, char *name)
{
  struct vdecl FAR *v;
  struct commonblock *block;
  int firsttime=TRUE;
  block = (struct commonblock FAR *) learn;
  if((v = NEWVDECL) == NULL){
    errorMsg(gp, 2, ErrNOMOREMEM);
    c_exit(gp, NOT_OK);
  }
  insert_variable(block, v, FALSE);


L_DECL:
  if(firsttime)
    goto START_DECL;
  switch (token=yylex(gp)) {
  case S_IDENT : /* New variable name = gp->LEXyy.ident */
START_DECL:
    if(firsttime){
      v->name = name;
    } else { 
      if ((v->name =
	   (char FAR *) CalMalloc(strlen(gp->LEXyy.ident)+1)) == NULL) {
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      strcpy(v->name, gp->LEXyy.ident);
    }
    v->type = type;
    if(!firsttime)
      token = yylex(gp);
    else 
      firsttime = FALSE;
    switch (token) {
    case S_ASSIGN     :
      if((v->init = NEWVINIT) == NULL){
	errorMsg(gp, 2, ErrNOMOREMEM);
	c_exit(gp, NOT_OK);
      }
      init_list(v->init);
      token = op_(gp, block, S_STATSEP, &v->init->exp, 1);
      goto L_PAREXPSEP;
    case S_BEGBRACKET :
      v->type = type | T_ARRAY;
      token = array_decl(gp, block, v);
    default :
      goto L_PAREXPSEP;
    }
    token = yylex(gp);
    break;
  default :     
    yyerror(gp, ErrINTIDNAM);
    break;
  }
L_PAREXPSEP:
  if (token == S_PAREXPSEP) {
    if ((v = NEWVDECL) == NULL) {
      errorMsg(gp, 2, ErrNOMOREMEM);
      c_exit(gp, NOT_OK);
    }
    insert_variable(block, v, FALSE);
    goto L_DECL;
  } else {
    if (token == S_STATSEP) {
      return(token = yylex(gp));
    }
    token = error(gp, S_PAREXPSEP, ErrININTDECL);
    if (token == S_PAREXPSEP) 
      goto L_PAREXPSEP;
  }
  return S_NOSYMBOL; /* Should never be executed */
}

int readglobal(struct cw_status *gp, struct learnunit FAR *learn, int token)
{
  int type, ret;
  char FAR *ident;

  switch(token){
  case S_OBJECT:
    token = readobject(gp, learn, token);
    return token;
  case S_FFLOAT:
    type = T_FLOAT;
    token = yylex(gp);
    break;
  case S_TTEXT:
    type = T_TEXT;
    token = yylex(gp);
    break;
  case S_INT:
    type = T_INT;
    token = yylex(gp);
  default:
    type = T_INT;
    break;
  }

  if(token != S_IDENT){
    yyerror(gp, ErrEXPFUNID);
    return 0;
  }
  ident = strdup(gp->LEXyy.ident);
  token = yylex(gp);
  if(token == S_BEGPAR){
    ret = readfunc(gp, learn, token, type, ident);
    return ret;
  } else {
    return readvar(gp, learn, token, type, ident);
  }
}

/* 
 * This funtions will read and parse the entire simulation.
 */
void readsim(struct cw_status *gp, AweStream *calFile) 
{
  struct learnunit FAR *learn;
  int token;

  learn = gp->curlunit;
  gp->LEXyy.line = 1;
  gp->LEXyy.file = calFile;
  gp->LEXyy.strind = -1;
  token = yylex(gp);
  while(token != S_NOSYMBOL && token != 0){
    token = readglobal(gp, learn, token);
  }
  matchglobal(gp, learn);
}

/*
 * Match entries in the 'struct notdecl *firstnd' list against global 
 * declarations.
 */
static void matchglobal(struct cw_status *gp, struct learnunit FAR *learn)
{
  char errormsg[256];
  struct notdecl FAR *nd;
  struct oper FAR *op;
  struct function FAR *fp;

  if(learn != NULL){
    for(nd = learn->firstnd;
	nd != NULL;
	nd = (struct notdecl FAR *) next(nd)){
      if((fp = get_function(learn->fp, nd->name)) != NULL){
	if((op = nd->op) != NULL){
	  switch(nd->type){
	  case 'V':
	  case 'F':
	    switch(fp->rettype){
	    case T_INT :
	      op->flag = UINTFUNC;
	      break;
	    case T_FLOAT :
	      op->flag = UFLOATFUNC;
	      break;
	    case T_TEXT :
	      op->flag = UTEXTFUNC;
	      break;
	    default :
	      sprintf(errormsg, ErrINTUNFUNT, fp->rettype);
	      errorMsg(gp, 1, errormsg);
	      return;
	    }
	    compute_valtype(gp, op);
	    if((op->left.do_act = NEWACTION) == NULL){
	      errorMsg(gp, 2, ErrNOMOREMEM);
	      c_exit(gp, NOT_OK);
	    }
	    op->left.do_act->type = op->flag;
	    op->left.do_act->function.func_act = fp;
	    op->left.do_act->actual.par = nd->actual.par;
	    
	    nd->actual.par = NULL;
	    break;
	  default :
	    sprintf(errormsg, ErrINTUNNDTT, nd->type);
	    errorMsg(gp, 1, errormsg);
	    return;
	  }
	} else if(nd->do_act != NULL){
	  nd->do_act->function.func_act =  fp;
	  switch(fp->rettype){
	  case T_INT :
	    nd->do_act->type = UINTFUNC;
	    break;
	  case T_FLOAT :
	    nd->do_act->type = UFLOATFUNC;
	    break;
	  case T_TEXT :
	    nd->do_act->type = UTEXTFUNC;
	    break;
	  default:
	    sprintf(errormsg, ErrINTUFPRET, fp->rettype);
	    errorMsg(gp, 1, errormsg);
	    return;
	  }
	}
      } else {
	sprintf(errormsg, ErrNIDDECL, (char FAR *) nd->name);
	errorMsg(gp, 1, errormsg);
	return;
      }
    }
  } else {
    sprintf(errormsg, ErrMGNULARG);
    errorMsg(gp, 1, errormsg);
    return;
  }
  
  f_notdecl(gp, learn->firstnd);
  learn->firstnd = NULL;
}

/*
 * Backpatch 
 */
void bp_objectproto(struct cw_status *gp, struct objectproto *object,
		    struct learnunit *learn)
{

}





