/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>

#define LEX_MAIN
#include "candle.h"
#include "const.h"
#include "error.h"
#include "protos/canutil.h"
#include "lex.h"
#include "protos/parser.h"
#include "protos/memory.h"

static void FreeHashlist(struct cw_status *gp);

#define unputc(c) {(void)AweUngetc((char)c,gp->LEXyy.file);\
		     gp->LEXyy.ant_char_in_match_buffer--;\
		     gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';}

static void unput(struct cw_status *gp, int c)
{
  if(gp->LEXyy.strind >= 0){
    gp->LEXyy.ant_char_in_match_buffer--;
    gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
    gp->LEXyy.strind--;
  } else
    unputc(c);
}


/*
 * Free memory allocated by lexer.
 */
void yyfree(struct cw_status *gp)
{
  if(gp->LEXyy.text_buffer) CalFree(gp->LEXyy.text_buffer);
  gp->LEXyy.text_buffer = NULL;
  if(gp->LEXyy.match_buffer) CalFree(gp->LEXyy.match_buffer);
  gp->LEXyy.match_buffer = NULL;
  FreeHashlist(gp);
}

/*****************************************************************************
								   INPUT    */

long linput(struct cw_status *gp) {
  int yytchar;
  do
    {
      if(gp->LEXyy.strind < 0)
	yytchar = AweGetc(gp->LEXyy.file);
      else {
	yytchar = gp->LEXyy.strind < (long) strlen(gp->LEXyy.str) ? gp->LEXyy.str[gp->LEXyy.strind] : EOF;
	gp->LEXyy.strind++;
      }
    } while (yytchar==0);
  if(gp->LEXyy.ant_char_in_match_buffer+1>=gp->LEXmatch_buffer_length)
    {
      if(gp->LEXmatch_buffer_length==0)
	{
	  gp->LEXmatch_buffer_length=64;
	  gp->LEXyy.match_buffer=
	    (char FAR *)CalMalloc(gp->LEXmatch_buffer_length);
	} else
	{
	  gp->LEXmatch_buffer_length*=2;
	  gp->LEXyy.match_buffer=
	    (char FAR *)CalRealloc(gp->LEXyy.match_buffer,
				   gp->LEXmatch_buffer_length);
	}
    }
  gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer++]=yytchar;
  gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';

  return(yytchar);
}

#define  newlexchar (gp->LEXyy.lexchar=linput(gp))


/******************************************************************************
  							      PUTCHARTEXT    */

void putchartext(struct cw_status *gp, char character) {
  if(gp->LEXyy.ant_char_in_text_buffer+1>=gp->LEXtext_buffer_length)
    {
      if(gp->LEXtext_buffer_length==0)
	{
	  gp->LEXtext_buffer_length=64;
	  gp->LEXyy.text_buffer=(char FAR *)CalMalloc(gp->LEXtext_buffer_length);
	} else
	{
	  gp->LEXtext_buffer_length*=2;
	  gp->LEXyy.text_buffer=(char FAR *)CalRealloc(gp->LEXyy.text_buffer,gp->LEXtext_buffer_length);
	}
    }
  gp->LEXyy.text_buffer[gp->LEXyy.ant_char_in_text_buffer++]=character;
}

/******************************************************************************
   KONSTANT-TEKST-OBJEKTER          				GETTEXT      */

char FAR *yygettext(struct cw_status *gp) 
{
  char FAR *s;

  gp->LEXyy.text_buffer[gp->LEXyy.ant_char_in_text_buffer++]='\0';
  s= (char FAR *) CalMalloc(gp->LEXyy.ant_char_in_text_buffer);
  (void)strcpy(s,gp->LEXyy.text_buffer);
  gp->LEXyy.ant_char_in_text_buffer=0;
  return(s);
}

/******************************************************************************
                                                                  YYERROR    */

void yyerror(struct cw_status *gp, char *msg) 
{   
  char errormsg[256];
  int gc;
  if(!strncmp(msg, "Error:", 5))
    gp->command |= RETURN_ACTION;

  sprintf(errormsg, "\"%s\", at line %d:\n   %s.\n", scanFileName(gp), gp->LEXyy.line, 
	  msg);

  gc = gp->command;
  errorMsg(gp, 1,errormsg);
  gp->command = gc;
}

struct namebuff {
  struct namebuff FAR *next;
  char text[1];
};

static void FreeHashlist(struct cw_status *gp)
{
  int i;
  struct namebuff FAR *atHashlist, FAR *nextList;
  for (i=0;i<NAMEPRIMEVALUE;i++) {
    atHashlist = gp->LEXhashlist[i];
    gp->LEXhashlist[i] = NULL;
    while (atHashlist) {
      nextList = atHashlist->next;
      CalFree(atHashlist);
      atHashlist = nextList;
    }
  }
}

/******************************************************************************
								SYSTAG       */
struct namebuff FAR *systag(struct cw_status *gp, char FAR t[], long leng)
{
  long hash;
  struct namebuff FAR *list,FAR *prevlist=NULL;
  if(leng>3)
    hash=t[0]+8*t[1]+64*t[2]+(long)512*t[leng-1]+leng; else
      hash=t[0]+(long)512*t[leng-1]+leng;
  hash=hash%NAMEPRIMEVALUE;
  for(list=gp->LEXhashlist[hash];list!=NULL;list=list->next){
    if(!strcmp(list->text,t))return(list);
    prevlist=list;
  }
  list = (struct namebuff FAR *) CalMalloc((size_t)(sizeof(struct namebuff FAR *)+sizeof(long)+leng+1));
  (void)strcpy(list->text,t);
  list->next=NULL;
  if(prevlist==NULL)gp->LEXhashlist[hash]=list; else
    prevlist->next=list;
  return(list);
}



/******************************************************************************
								TAG          */
char FAR FAR *tag(struct cw_status *gp, char FAR t[], long leng)
{
  return(systag(gp, t,leng)->text);
}

/* 
 * Return a token to the parser. This function is meant to be used by the 
 * simulation code parser.
 */
int yylex(struct cw_status *gp)
{
  char mess[160];
  int escape=FALSE;

  if(gp->LEXfirsttime){ 
    gp->LEXfirsttime=FALSE;
    gp->LEXyy.line=1;
  }

  for(;;){
    gp->LEXyy.ant_char_in_match_buffer=0;
    if(isalpha(newlexchar)){
      while(isalnum(gp->LEXyy.lexchar) || gp->LEXyy.lexchar == '_')newlexchar;
        unput(gp, gp->LEXyy.lexchar);
	gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
        switch(gp->LEXyy.match_buffer[0])
	  {
	  case 'a':
	    if(! strcmp(gp->LEXyy.match_buffer,"arc"))  return(S_ARC);
	    break;
	  case 'b':
	    if(! strcmp(gp->LEXyy.match_buffer,"box"))  return(S_BOX);
	    if(! strcmp(gp->LEXyy.match_buffer,"break")) return(S_BREAK);
	    break;
	  case 'c':
	    if(! strcmp(gp->LEXyy.match_buffer,"case")) return(S_CASE);
	    if(! strcmp(gp->LEXyy.match_buffer,"const")) return(S_CONST);
	    break;
	  case 'd':
	    if(! strcmp(gp->LEXyy.match_buffer,"default")) return(S_DEFAULT);
	    break;
	  case 'e':
	    if(! strcmp(gp->LEXyy.match_buffer,"else")) return(S_ELSE);
	    if(! strcmp(gp->LEXyy.match_buffer,"endwindow"))  return(S_END_WINDOW);
	    break;
	  case 'f':
	    if(! strcmp(gp->LEXyy.match_buffer,"float")) return(S_FFLOAT);
	    if(! strcmp(gp->LEXyy.match_buffer,"for")) return(S_FOR);
	    break;
	  case 'i':
	    if(! strcmp(gp->LEXyy.match_buffer,"if")) return(S_IF);
	    if(! strcmp(gp->LEXyy.match_buffer,"image"))  return(S_IMAGE);
	    if(! strcmp(gp->LEXyy.match_buffer,"inputarea"))  return(S_INPUTAREA);
	    if(! strcmp(gp->LEXyy.match_buffer,"int")) return(S_INT);
	    break;
	  case 'l':
	    if(! strcmp(gp->LEXyy.match_buffer,"line"))  return(S_LINE);
	    break;
	  case 'o':
	    if(! strcmp(gp->LEXyy.match_buffer,"object")) return(S_OBJECT);
	    break;
	  case 'p':
	    if(! strcmp(gp->LEXyy.match_buffer,"point"))  return(S_POINT);
	    if(! strcmp(gp->LEXyy.match_buffer,"polygon"))  return(S_POLYGON);
	    break;
	  case 'r':
	    if(! strcmp(gp->LEXyy.match_buffer,"ref")) return(S_REF);
	    if(! strcmp(gp->LEXyy.match_buffer,"return")) return(S_RETURN);
	    break;	
	  case 's':
	    if(! strcmp(gp->LEXyy.match_buffer,"switch"))  return(S_SWITCH);
	    break;
	  case 't':
	    if(! strcmp(gp->LEXyy.match_buffer,"text"))  return(S_TTEXT);
	    if(! strcmp(gp->LEXyy.match_buffer,"textobj"))  return(S_TEXTOBJECT);
	    break;
	  case 'w':
	    if(! strcmp(gp->LEXyy.match_buffer,"while"))  return(S_WHILE);
	    if(! strcmp(gp->LEXyy.match_buffer,"window"))  return(S_WINDOW);
	    break;
	  }
	/* IDENTIFIKATORER    BLIR */
	/* LAGT INN I NAVNELAGERET */
	gp->LEXyy.ident=tag(gp, gp->LEXyy.match_buffer,gp->LEXyy.ant_char_in_match_buffer);
	return(S_IDENT);
      }
    switch(gp->LEXyy.lexchar)
      {
      case '=':
	if(newlexchar=='=') return(S_EQ);
	unput(gp, gp->LEXyy.lexchar);
	return(S_ASSIGN);
      case '!':
	if(newlexchar=='='){ return(S_NE);}
	unput(gp, gp->LEXyy.lexchar);
	return (S_NOT);
      case '>':
	if(newlexchar=='=') return(S_GE);
	if(gp->LEXyy.lexchar=='>'){ return(S_RSHIFT);}
	unput(gp, gp->LEXyy.lexchar);
	return(S_GT);
      case '<':
	if(newlexchar=='=') return(S_LE);
	if(gp->LEXyy.lexchar=='>'){ return(S_NE);}
	if(gp->LEXyy.lexchar=='<'){ return(S_LSHIFT);}
	unput(gp, gp->LEXyy.lexchar);
	return(S_LT);
      case '+': 
	if(newlexchar=='+') return(S_ADDADD);
	unput(gp, gp->LEXyy.lexchar);
	return(S_ADD);
      case '-': 
	if(newlexchar=='-') return(S_SUBSUB);
	unput(gp, gp->LEXyy.lexchar);
	return(S_SUB);
      case '*':
	if(newlexchar=='*') return(S_PRIMARY);
	unput(gp, gp->LEXyy.lexchar);
	return(S_MUL);
      case '/':
	if(newlexchar=='/')
	  {
	    while(newlexchar!='\n' && gp->LEXyy.lexchar!=EOF);
	    if(gp->LEXyy.lexchar!=EOF)unput(gp, gp->LEXyy.lexchar);
	    continue;
	  }
	if(gp->LEXyy.lexchar=='*')
	  {
	    while((newlexchar!=EOF))
	      if(gp->LEXyy.lexchar=='\n')gp->LEXyy.line++;else
	      if(gp->LEXyy.lexchar=='*')
		if(newlexchar=='/')break;else
		  unput(gp, gp->LEXyy.lexchar);
	    continue;
	  }
	unput(gp, gp->LEXyy.lexchar);
	return(S_DIV);
      case ',': return (S_PAREXPSEP);
      case ':':
	return (S_LABELSEP);
      case ';': return (S_STATSEP);
      case '(': return (S_BEGPAR);
      case ')': return (S_ENDPAR);
      case '%': return (S_REST);
      case '{': return (S_BEGIN);
      case '}': return (S_END);
      case '[': return (S_BEGBRACKET);
      case ']': return (S_ENDBRACKET);
      case '#': return (S_EXPRVAL);
      case '|':
	if(newlexchar=='|') return(S_OR);
	unput(gp, gp->LEXyy.lexchar);
	return(S_BITOR);
      case '&':
	if(newlexchar=='&') return(S_AND);
	unput(gp, gp->LEXyy.lexchar);
	return(S_BITAND);
      case '\"':
	newlexchar;

	for(;;){
	  if(gp->LEXyy.lexchar == '\n') gp->LEXyy.line++;
	  if(escape == TRUE){
	    switch(gp->LEXyy.lexchar){
	    case '\"':
	      gp->LEXyy.ant_char_in_text_buffer--;
	      putchartext(gp, '\"');
	      break;
	    case 'n':
	      gp->LEXyy.ant_char_in_text_buffer--;
	      putchartext(gp, '\n');
	      break;
	    case 'r':
	      gp->LEXyy.ant_char_in_text_buffer--;
	      putchartext(gp, '\r');
	      break;
	    case '\\':
	      break;
	    }
	    escape = FALSE;
	  } else { 
	    switch(gp->LEXyy.lexchar){
	    case '\\':
	      escape = TRUE;
	      break;
	    case '\"':
	      putchartext(gp, '\0'); /* Make sure empty strings have allocated 
				    space in gp->LEXyy.text_buffer */
	      gp->LEXyy.tval = yygettext(gp);
	      return S_TTEXTKONST;
	    }
	    putchartext(gp, (char)gp->LEXyy.lexchar);
	  }
	  newlexchar;
	}
      case '0':
        newlexchar;
	if(gp->LEXyy.lexchar == 'x' || gp->LEXyy.lexchar == 'X' || isdigit(gp->LEXyy.lexchar)) 
	  newlexchar;
	else
	  goto L_DEC;
	while(isdigit(gp->LEXyy.lexchar) || gp->LEXyy.lexchar == 'a' || gp->LEXyy.lexchar == 'b' || 
	      gp->LEXyy.lexchar == 'c' || gp->LEXyy.lexchar == 'd' || gp->LEXyy.lexchar == 'e' || 
	      gp->LEXyy.lexchar == 'f' || gp->LEXyy.lexchar == 'A' || gp->LEXyy.lexchar == 'B' || 
	      gp->LEXyy.lexchar == 'C' || gp->LEXyy.lexchar == 'D' || gp->LEXyy.lexchar == 'E' || 
	      gp->LEXyy.lexchar == 'F') newlexchar;
	unput(gp, gp->LEXyy.lexchar);
	gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
	strcpy(mess, gp->LEXyy.match_buffer);
	sscanf(mess,"%li",&gp->LEXyy.lval);
	return(S_INTEGERKONST);
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
L_DEC:	while(isdigit(gp->LEXyy.lexchar))newlexchar;
	if(gp->LEXyy.lexchar!='.' && gp->LEXyy.lexchar!='e' && gp->LEXyy.lexchar!='E')
	  {
	    unput(gp, gp->LEXyy.lexchar);
            gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
	    strcpy(mess, gp->LEXyy.match_buffer);
	    sscanf(mess,"%ld",&gp->LEXyy.lval);
	    return(S_INTEGERKONST);
	  }
	/* No break after previous statement */
      case '.':
	if(gp->LEXyy.lexchar=='.') newlexchar;
	while(isdigit(gp->LEXyy.lexchar))newlexchar;
	if((gp->LEXyy.lexchar=='e' || gp->LEXyy.lexchar=='E')
	   && (newlexchar=='+' || gp->LEXyy.lexchar=='+'))newlexchar;
	while(isdigit(gp->LEXyy.lexchar))newlexchar;
	unput(gp, gp->LEXyy.lexchar);
	gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
	strcpy(mess, gp->LEXyy.match_buffer);
	sscanf(mess,"%lf",&gp->LEXyy.rval);
	return (S_REALKONST);
      case EOF: return(S_NOSYMBOL);
      case '\n':/* NL (LF) */
	gp->LEXyy.line++;
	continue;
      case ' ':
      case '\b':/* BS */
      case '\t':/* HT */
      case '\v':/* VT */
      case '\f':/* FF */
      case '\r':/* CR */
	continue;
      default:
	yyerror(gp, "Illegal character in input");
	continue;
      }
    }
}

/*
 * Return next token to the parser. This function is meant to be used by the 
 * object-parser.
 */
int yylexobject(struct cw_status *gp)
{              
    char mess[160];
    for(;;){
    gp->LEXyy.ant_char_in_match_buffer=0;
    if(isalpha(newlexchar)){
      while(isalnum(gp->LEXyy.lexchar) || gp->LEXyy.lexchar == '_')newlexchar;
        unput(gp, gp->LEXyy.lexchar);
	gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
        switch(gp->LEXyy.match_buffer[0])
	  {
	  case 'a':
	    if(! strcmp(gp->LEXyy.match_buffer,"act"))  return(S_ACTIVE);
	    if(! strcmp(gp->LEXyy.match_buffer,"active"))  return(S_ACTIVE);
	    break;
	  case 'c':
	    if(! strcmp(gp->LEXyy.match_buffer,"col"))  return(S_COLOR);
	    if(! strcmp(gp->LEXyy.match_buffer,"color"))  return(S_COLOR);
	    break;
	  case 'd':
	    if(! strcmp(gp->LEXyy.match_buffer,"dashes"))  return(S_DASHES);
	    if(! strcmp(gp->LEXyy.match_buffer,"dec"))  return(S_DECIMALS);
	    if(! strcmp(gp->LEXyy.match_buffer,"decimals"))  return(S_DECIMALS);
	    if(! strcmp(gp->LEXyy.match_buffer,"dsh"))  return(S_DASHES);
	    break;
	  case 'e':
	    if(! strcmp(gp->LEXyy.match_buffer,"ea"))  return(S_ENDANGLE);    
	    if(! strcmp(gp->LEXyy.match_buffer,"endangle"))  return(S_ENDANGLE);
	    break;
	  case 'f':
	    if(! strcmp(gp->LEXyy.match_buffer,"f"))  return(S_FILL);
	    if(! strcmp(gp->LEXyy.match_buffer,"fill"))  return(S_FILL);
	    if(! strcmp(gp->LEXyy.match_buffer,"font"))  return(S_FONT);
	    if(! strcmp(gp->LEXyy.match_buffer,"fn"))  return(S_FONT);
	    break;
	  case 'i':
	    if(! strcmp(gp->LEXyy.match_buffer,"img"))  return(S_IMAGE);
	    if(! strcmp(gp->LEXyy.match_buffer,"image"))  return(S_IMAGE);
	    break;
	  case 'l':
	    if(! strcmp(gp->LEXyy.match_buffer,"lev"))  return(S_LEVEL);
	    if(! strcmp(gp->LEXyy.match_buffer,"level"))  return(S_LEVEL);
	    if(! strcmp(gp->LEXyy.match_buffer,"linewidth"))  return(S_LINEWIDTH);	
            if(! strcmp(gp->LEXyy.match_buffer,"lw"))  return(S_LINEWIDTH);
	    break;
	  case 'o':
	    if(! strcmp(gp->LEXyy.match_buffer,"of"))  return(S_OUTFLOAT);
	    if(! strcmp(gp->LEXyy.match_buffer,"outfloat"))  return(S_OUTFLOAT);
	    if(! strcmp(gp->LEXyy.match_buffer,"oi"))  return(S_OUTINT);
	    if(! strcmp(gp->LEXyy.match_buffer,"outint"))  return(S_OUTINT);
	    if(! strcmp(gp->LEXyy.match_buffer,"ot"))  return(S_OUTTEXT);
	    if(! strcmp(gp->LEXyy.match_buffer,"outtext"))  return(S_OUTTEXT);
	    break;
	  case 'p':
	    if(! strcmp(gp->LEXyy.match_buffer,"points"))  return(S_POINTS);
	    break;
	  case 's':
	    if(! strcmp(gp->LEXyy.match_buffer,"sa"))  return(S_STARTANGLE);
	    if(! strcmp(gp->LEXyy.match_buffer,"savebackground"))  return(S_SAVEBG);
	    if(! strcmp(gp->LEXyy.match_buffer,"sb"))  return(S_SAVEBG);
	    if(! strcmp(gp->LEXyy.match_buffer,"startangle"))  return(S_STARTANGLE);
	    break;
	  case 't':
	    if(! strcmp(gp->LEXyy.match_buffer,"tex"))  return(S_TEXTURE);
	    if(! strcmp(gp->LEXyy.match_buffer,"texture"))  return(S_TEXTURE);
	    if(! strcmp(gp->LEXyy.match_buffer,"translation"))  return(S_TRANSLATION);
	    if(! strcmp(gp->LEXyy.match_buffer,"tr"))  return(S_TRANSLATION);
	    break;
	  }
	/* IDENTIFIKATORER    BLIR */
	/* LAGT INN I NAVNELAGERET */
	gp->LEXyy.ident=tag(gp, gp->LEXyy.match_buffer,gp->LEXyy.ant_char_in_match_buffer);
	return(S_IDENT);
      }
    switch(gp->LEXyy.lexchar)
      {
      case '=':
	if(newlexchar=='=') return(S_EQ);
	unput(gp, gp->LEXyy.lexchar);
	return(S_ASSIGN);
      case '!':
	if(newlexchar=='='){ return(S_NE);}
	unput(gp, gp->LEXyy.lexchar);
	return (S_NOT);
      case '>':
	if(newlexchar=='=') return(S_GE);
	if(gp->LEXyy.lexchar=='>'){ return(S_RSHIFT);}
	unput(gp, gp->LEXyy.lexchar);
	return(S_GT);
      case '<':
	if(newlexchar=='=') return(S_LE);
	if(gp->LEXyy.lexchar=='>'){ return(S_NE);}
	if(gp->LEXyy.lexchar=='<'){ return(S_LSHIFT);}
	unput(gp, gp->LEXyy.lexchar);
	return(S_LT);
      case '+': return(S_ADD);
      case '-': return(S_SUB);
      case '*':
	if(newlexchar=='*') return(S_PRIMARY);
	unput(gp, gp->LEXyy.lexchar);
	return(S_MUL);
      case '/':
	if(newlexchar=='/')
	  {
	    while(newlexchar!='\n' && gp->LEXyy.lexchar!=EOF);
	    if(gp->LEXyy.lexchar!=EOF)unput(gp, gp->LEXyy.lexchar);
	    continue;
	  }
	if(gp->LEXyy.lexchar=='*')
	  {
	    while((newlexchar!=EOF))
	      if(gp->LEXyy.lexchar=='\n')gp->LEXyy.line++;else
	      if(gp->LEXyy.lexchar=='*')
		if(newlexchar=='/')break;else
		  unput(gp, gp->LEXyy.lexchar);
	    continue;
	  }
	unput(gp, gp->LEXyy.lexchar);
	return(S_DIV);
      case ',': return (S_PAREXPSEP);
      case ':':
	return (S_LABELSEP);
      case ';': return (S_STATSEP);
      case '(': return (S_BEGPAR);
      case ')': return (S_ENDPAR);
      case '%': return (S_REST);
      case '{': return (S_BEGIN);
      case '}': return (S_END);
      case '[': return (S_BEGBRACKET);
      case ']': return (S_ENDBRACKET);
      case '|':
	if(newlexchar=='|') return(S_OR);
	unput(gp, gp->LEXyy.lexchar);
	return(S_BITOR);
      case '&':
	if(newlexchar=='&') return(S_AND);
	unput(gp, gp->LEXyy.lexchar);
	return(S_BITAND);
      case '\"':
	newlexchar;
	for(;;)
	  {
	    if((isgraph(gp->LEXyy.lexchar) && gp->LEXyy.lexchar!='\"') || gp->LEXyy.lexchar==' ');else
	      if(gp->LEXyy.lexchar=='\"')
		{
		  if(newlexchar!='\"')
		    {
		      unput(gp, gp->LEXyy.lexchar);
		      gp->LEXyy.tval=yygettext(gp);
		      return(S_TTEXTKONST);
		    }
		} else yyerror(gp, "Illegal character in text constant");
	    putchartext(gp, (char)gp->LEXyy.lexchar);
	    newlexchar;
	  }
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	while(isdigit(gp->LEXyy.lexchar))newlexchar;
	if(gp->LEXyy.lexchar!='.' && gp->LEXyy.lexchar!='e' && gp->LEXyy.lexchar!='E')
	  {
	    unput(gp, gp->LEXyy.lexchar);
            gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
        strcpy(mess, gp->LEXyy.match_buffer);
	    sscanf(mess,"%ld",&gp->LEXyy.lval);
	    return(S_INTEGERKONST);
	  }
	/* No break after previous statement */
      case '.':
	if(gp->LEXyy.lexchar=='.')newlexchar;
	while(isdigit(gp->LEXyy.lexchar))newlexchar;
	if((gp->LEXyy.lexchar=='e' || gp->LEXyy.lexchar=='E')
	   && (newlexchar=='+' || gp->LEXyy.lexchar=='+'))newlexchar;
	while(isdigit(gp->LEXyy.lexchar))newlexchar;
	unput(gp, gp->LEXyy.lexchar);
	gp->LEXyy.match_buffer[gp->LEXyy.ant_char_in_match_buffer]='\0';
	strcpy(mess, gp->LEXyy.match_buffer);
	sscanf(mess,"%lf",&gp->LEXyy.rval);
	return (S_REALKONST);
      case EOF: return(S_NOSYMBOL);
      case '\n':/* NL (LF) */
	gp->LEXyy.line++;
	continue;
      case ' ':
      case '\b':/* BS */
      case '\t':/* HT */
      case '\v':/* VT */
      case '\f':/* FF */
      case '\r':/* CR */
	continue;
      default:
	yyerror(gp, "Illegal character in input");
	continue;
      }
    }
}


