/* Copyright (c) 1996 by CandleWeb AS (candleweb@candleweb.no). 
 * See Copyright.txt for details.
 *
 * Authors: Gunnar Rnning (gunnar@candleweb.no)
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "candle.h"
#include "fast_lis.h"
#include "awethor.h"
#include "awelex.h"
#include "parser.h"
#include "error.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/parser.h"
#include "protos/memory.h"


#define awe gp->awe
#define parsecomment awe.parsecomment
#define parsenewline awe.parsenewline
#define fblock awe.fblock
#define lblock awe.lblock
#define cblock awe.cblock
#define nblocks awe.nblocks
#define ao_buf awe.ao_buf
#define ao_size awe.ao_size

#if 0
/* 
 * List of block in the current -file.
 */

static struct awe_func *fblock=NULL, *lblock=NULL;
int nblocks;

/* 
 * Array of with pointers to all the objects in the current function. 
 */
struct awe_object **ao_buf = NULL;
long ao_size=0;

/*
 * Pointer to the current composite object/function.
 */
static struct awe_func *cblock;
#endif
#if 0
extern int parsecomment;
extern int parsenewline;
#endif

static int readstatement(struct cw_status *gp, struct awe_func *af, int token);


int set_current_block(struct cw_status *gp, char *name)
{
  struct awe_func *af;
  struct awe_object *ao;
  int i;

  for(af = fblock; af != NULL; af = next(af))
    if(!strcmp(af->name, name))
      break;
  if(af == NULL)
    return FALSE;

  if(ao_buf != NULL){
    CalFree(ao_buf);
  }

  ao_size = af->nobjects;
  ao_buf = CalMalloc(ao_size * sizeof(struct aweobject *));
  if(ao_buf == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  for(i = 0, ao = af->fo; i < ao_size; i++, ao = next(ao)){
    ao_buf[i] = ao;
  }
  cblock = af;
  return TRUE;
}

/*
 * Create an array with all the block names in the current unit.
 */
int get_awe_blocks(struct cw_status *gp, char ***names)
{
  int i;
  struct awe_func *af;
  char **na;

  if(nblocks == 0)
    return 0;

  (*names) = na = CalMalloc(nblocks * sizeof(char *));

  if(*names == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }

  for(i = 0, af = fblock; i < nblocks; i++, af = next(af))
    na[i] = strdup(af->name);

  return nblocks;
}

static void add_chunk(struct cw_status *gp, struct awe_func *af)
{
  char *buf;
  struct awe_chunk *ac;

  buf = getbuf(gp);
/*   printf("%s", buf); */
  if((ac = NEWAWECHUNK) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  ac->buf = buf;
  ac->size = strlen(buf);
  if(af->start == NULL){
    init_list(ac);
    af->start = af->end = ac;
  } else {
    place_next(af->end, ac);
    af->end = ac;
  }
}

/*
 * Flush the current chunk.
 * Useful to ignore input you don't want to store.
 */
static void flush_chunk(struct cw_status *gp)
{
  char *buf;
  buf = getbuf(gp);
  if(buf != NULL)
    CalFree(buf);
}

/*
 * Return pointer to user defined object.
 */
static struct awe_func *get_user_object(struct cw_status *gp, char *name)
{
  struct awe_func *af;

  for(af=fblock; af != NULL; af = next(af)){
    if(!strcmp(af->name, name) && af->type == T_USEROBJECT)
      return af;
  }
  return NULL;
}

static int is_object(struct cw_status *gp, int token)
{
  switch(token){
  case S_ARC:
  case S_BOX:
  case S_IMAGE:
  case S_INPUTAREA:
  case S_LINE:
  case S_POINT:
  case S_POLYGON:
  case S_TEXTOBJECT:
  case S_WINDOW:
  case S_END_WINDOW:
    return TRUE;
  case S_IDENT:
    return get_user_object(gp, awe.ident) != NULL;
  default:
    return FALSE;
  }
}

#define SET_TVAL(dst, src) if(dst != NULL) CalFree(dst); dst = src;

void set_attrraw(struct cw_status *gp, struct awe_object *ao, char *name, 
		 char *tval)
{
  int sym;
  sym = battr_sym_type(name);

  switch(sym){
  case S_ACTIVE:
    SET_TVAL(ao->tactive,tval);
    break;
  case S_COLOR:
    SET_TVAL(ao->tcolor, tval);
    break;
  case S_DASHES:
    SET_TVAL(ao->tdashes,tval);
    break;
  case S_ENDANGLE:
    SET_TVAL(ao->tendangle,tval);
    break;
  case S_FILL:
    SET_TVAL(ao->tfill,tval;);
    break;
  case S_LEVEL:
    SET_TVAL(ao->tlevel,tval);
    break;
  case S_LINEWIDTH:
    SET_TVAL(ao->tlinewidth,tval);
    break;
  case S_SAVEBG:
    SET_TVAL(ao->tsavebackground,tval);
    break;
  case S_STARTANGLE:
    SET_TVAL(ao->tstartangle,tval);
    break;
  case S_TEXTURE:
    SET_TVAL(ao->ttexture,tval);
    break;
  case S_FONT:
    SET_TVAL(ao->tfont,tval);
    break;
  case S_IMAGE:
    SET_TVAL(ao->timage,tval);
    break;
  case S_OUTTEXT:
    SET_TVAL(ao->touttext,tval);
    break;
  default:
    /* TODO: Set raw value for user defined object attribute */
    break;
  }
}

#define SET_ATTRLVAL(name, lval, valtype) \
	if(ao->d ## name != ADIRVAL){\
	ao->name = lval;\
	ao->d ## name = valtype;\
				  }				       

#define SET_ATTRTVAL(name, tval, valtype) \
	if(ao->d ## name != ADIRVAL){\
	ao->name = strdup(tval);\
	ao->d ## name = valtype;\
				  }				       

void set_attrval(struct cw_status *gp, struct awe_object *ao, char *name, 
		 char *tval, long lval, int valtype)
{
  int sym;
  sym = battr_sym_type(name);

  switch(sym){
  case S_ACTIVE:
    SET_ATTRLVAL(active, lval, valtype);
    break;
  case S_COLOR:
    SET_ATTRLVAL(color, lval, valtype);
    break;
  case S_DASHES:
    SET_ATTRLVAL(dashes, lval, valtype);
    break;
  case S_ENDANGLE:
    SET_ATTRLVAL(endangle, lval, valtype);
    break;
  case S_FILL:
    SET_ATTRLVAL(fill, lval, valtype);
    break;
  case S_LEVEL:
    SET_ATTRLVAL(level, lval, valtype);
    break;
  case S_LINEWIDTH:
    SET_ATTRLVAL(linewidth, lval, valtype);
    break;
  case S_SAVEBG:
    SET_ATTRLVAL(savebackground, lval, valtype);
    break;
  case S_STARTANGLE:
    SET_ATTRLVAL(startangle, lval, valtype);
    break;
  case S_TEXTURE:
    SET_ATTRLVAL(texture, lval, valtype);
    break;
  case S_FONT:
    SET_ATTRTVAL(font, tval, valtype);
    break;
  case S_IMAGE:
    SET_ATTRTVAL(image, tval, valtype);
    break;
  case S_OUTTEXT:
    SET_ATTRTVAL(outtext, tval, valtype);
    break;
  case S_TRANSLATION:
    SET_ATTRTVAL(translation, tval, valtype);
    break;
  default:
    /* TODO: set attribute value for attributes in user defined objects */
    break;
  }
}

#define OPERAND		1
#define OPERATOR	2

struct expr {
  int type;
  char *val;
  struct expr *left, *right;
};


static int is_operand(int token)
{
  switch(token){
  case S_IDENT:
  case S_INTEGERKONST:
  case S_REALKONST:
  case S_TTEXTKONST:
    return TRUE;
  }
  return FALSE;
}

static int is_operator(int token)
{
  switch(token){
  case S_EQ:
  case S_ASSIGN:
  case S_NE:
  case S_NOT:
  case S_GE:
  case S_RSHIFT:
  case S_GT:
  case S_LE:
  case S_LSHIFT:
  case S_LT:
  case S_ADDADD:
  case S_ADD:
  case S_SUBSUB:
  case S_PRIMARY:
  case S_SUB:
  case S_MUL:
  case S_DIV:
  case S_OR:
  case S_BITOR:
  case S_EXPRVAL:
  case S_AND:
  case S_BITAND:
  case S_BEGPAR:
  case S_ENDPAR:
  case S_REST:
    return TRUE;
  }
  return FALSE;
}
static int readexpr(struct cw_status *gp, int token)
{
  int stoken=S_NOSYMBOL;
  char errormsg[256];
  int nestlev=0;

  while(is_operand(token) || is_operator(token)){
    if(token == S_IDENT){
      token = awelex(gp);
      while(token == S_BEGPAR || token == S_BEGBRACKET){
	if(token == S_BEGPAR)
	  stoken = S_ENDPAR;
	else if(token == S_BEGBRACKET)
	  stoken = S_ENDBRACKET;
	  
	do {
	  token = awelex(gp);
	  token = readexpr(gp, token);
	} while(token == S_PAREXPSEP);
	if(token != stoken){
	  sprintf(errormsg, "')' or ']' expected");
	  aweerror(gp, errormsg);
	  return(0);
	}
	token = awelex(gp);
      }
    } else if(token == S_BEGPAR){
      nestlev++;
      token = awelex(gp);
    } else if(token == S_ENDPAR){
      if(nestlev > 0){
	nestlev--;
	token = awelex(gp);
      } else
	return token;
    } else {
      token = awelex(gp);
    }
  }
  return token;
}

/*
 * Check if the object name is a builtuin object.
 */
static int is_builtin(char *name)
{
  if(bobj_sym_type(name) != -1)
    return TRUE;
  return FALSE;
}

static struct awe_attr *get_user_attribute(struct awe_attr *fa, char *name)
{
  struct awe_attr *aa;
  for(aa = fa; aa != NULL; aa = next(aa))
    if(!strcmp(aa->name, name))
      return aa;
  return NULL;
}

/*
 * Check if "id" is a translation attribute for the object ao.
 */
static int is_translation(struct cw_status *gp, struct awe_object *ao, 
			  char *id)
{
  struct awe_func *af;
  struct awe_attr *aa;
  
  if(is_builtin(ao->name) && battr_sym_type(id) == S_TRANSLATION)
    return TRUE;
  else if((af = get_user_object(gp, ao->name)) != NULL){
    aa = get_user_attribute(af->fa, id);
    if(aa != NULL && aa->type == T_TRANSLATION)
      return TRUE;
  }
  return FALSE;
}

static int is_points(struct cw_status *gp, struct awe_object *ao, char *id)
{
  struct awe_func *af;
  struct awe_attr *aa;
  
  if(is_builtin(ao->name) && battr_sym_type(id) == S_POINTS)
    return T_TRANSLATION;
  else if((af = get_user_object(gp, ao->name)) != NULL){
    aa = get_user_attribute(af->fa, id);
    if(aa != NULL && aa->type == T_TRANSLATION)
      return TRUE;
  }
  return FALSE;
}

static struct awe_attr *c_aweattr(struct awe_object *ao, char *name, int type)
{
  struct awe_attr *na;

  if((na = NEWAWEATTR) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if(ao->fa == NULL)
    init_list(na);
  else 
    place_prev(ao->fa, na);
  ao->fa = na;
  na->type = type;
  na->name = strdup(name);
  return na;
}
#define GET_CONST_EXPR       token = awelex(gp);\
      if(token == S_TTEXTKONST){ \
	tval = strdup(awe.tval); \
	is_const = TRUE; \
      } else if(token == S_INTEGERKONST){ \
	lval = awe.lval; \
	is_const = TRUE; \
      } else if(token == S_REALKONST){ \
	lval = awe.rval; \
	is_const = TRUE; \
      } 


static int readattribute(struct cw_status *gp, struct awe_func *af, 
			 struct awe_object *ao, int token)
{
  char *name;
  char *tval=NULL;
  long lval=0;
  int is_const=FALSE;
  int nestlev=0;
  struct awe_attr *aa, *na, *oa;
  char **tbuf;
  char errormsg[256];
  if(token == S_PAREXPSEP)
    token = awelex(gp);

  name = strdup(awe.match_buffer);
  if(!strcmp("points", name)){
    token = awelex(gp);
    token = awelex(gp);
    if(token == S_BEGPAR){
      do {
	token = awelex(gp);
	if(token != S_BEGPAR){
	  aweerror(gp, "Missing opening parenthesis in points attribute");
	}
	ao->npoints++;
	ao->pa = (ao->pa) ?
	  CalRealloc(ao->pa, ao->npoints * sizeof(long) * 2)
	  : CalCalloc (1, ao->npoints * sizeof(long) * 2);
	ao->tpa = (ao->tpa) ?
	  CalRealloc(ao->tpa, ao->npoints * sizeof(char *) * 2)
	  : CalCalloc (1, ao->npoints * sizeof(char *) * 2);
	ao->dpa = (ao->dpa) ?
	  CalRealloc(ao->dpa, ao->npoints * sizeof(char) * 2)
	  : CalCalloc (1, ao->npoints * sizeof(char) * 2);
	ao->dpa[(ao->npoints-1)*2] = ADEFVAL;
	ao->dpa[(ao->npoints-1)*2+1] = ADEFVAL;
	  
	flush_chunk(gp);
	GET_CONST_EXPR;
	if(is_const){
	  token = awelex(gp);
	  if(token == S_PAREXPSEP || token == S_ENDPAR){
	    ao->pa[(ao->npoints-1)*2] = lval;
	    ao->dpa[(ao->npoints-1)*2] = ADIRVAL;
	  } 
	}  
	if(ao->dpa[(ao->npoints-1)*2] == ADEFVAL)
	  ao->pa[(ao->npoints-1)*2] = 0;

	token = readexpr(gp, token);
	ao->tpa[(ao->npoints-1)*2] = getbuf(gp);
	  
	if(token != S_PAREXPSEP)
	  aweerror(gp, "Missing "," between x and y point in pointlist");
	flush_chunk(gp);
	GET_CONST_EXPR;
	if(is_const){
	  token = awelex(gp);
	  if(token == S_PAREXPSEP || token == S_ENDPAR){
	    ao->pa[(ao->npoints-1)*2+1] = lval;
	    ao->dpa[(ao->npoints-1)*2+1] = ADIRVAL;
	  }
	} 
	if(ao->dpa[(ao->npoints-1)*2+1] == ADEFVAL)
	  ao->pa[(ao->npoints-1)*2+1] = 0;

	token = readexpr(gp, token);
	ao->tpa[(ao->npoints-1)*2+1] = getbuf(gp);

	if(token != S_ENDPAR)
	  aweerror(gp, "Missing closing parenthesis in points attribute");
	token = awelex(gp);
      } while(token != S_ENDPAR);
    }
  } else if(is_translation(gp, ao, name)){
    if((oa = get_user_attribute(ao->fa, name)) != NULL) {
      na = c_aweattr(ao, name, T_TRANSLATION);
      tbuf = &na->tval;
    } else {
      tbuf = &ao->translation;
    }
    token = awelex(gp);
    flush_chunk(gp);
    token = awelex(gp);
    if(token == S_BEGPAR){
      do {
	if(token == S_BEGPAR)
	  nestlev++;
	else if(token == S_ENDPAR)
	  nestlev--;
	token = awelex(gp);
      } while(nestlev > 0);
    } else if(token == S_IDENT && af->type == T_USEROBJECT){
      for(aa = af->fa; aa != NULL;aa = next(aa))
	if(!strcmp(awe.ident, aa->name) && aa->type == T_TRANSLATION)
	  break;
      if(aa == NULL){
	sprintf(errormsg, "%s is not a legal translation attribute", 
		awe.ident);
	aweerror(gp, errormsg);
      }	  
    }
    set_attrval(gp, ao, name, getbuf(gp), 0, ADIRVAL);
  } else {
    token = awelex(gp);
    flush_chunk(gp);
    GET_CONST_EXPR;
    if(is_const){
      token = awelex(gp);
      if(token == S_PAREXPSEP || token == S_STATSEP){
	set_attrval(gp, ao, name, tval, lval, ADIRVAL);
      }
      if(tval){
	CalFree(tval);
      }
    }
    token = readexpr(gp, token);
    delbufc(gp);
    set_attrraw(gp, ao, name, getbuf(gp));
  }
  CalFree(name);
  

  return token;
}

/*
 * Read and set values corresponding to the points attribute in a comment. 
 */ 
static int readdefpoints(struct cw_status *gp, struct awe_object *ao, 
			 char *name)
{
  int n = 0;
  int token;
  char *tval;
  int is_const;
  long lval=0;

  token = awelex(gp);
  if(token == S_BEGPAR){
    do {
      token = awelex(gp);
      if(token != S_BEGPAR){
	aweerror(gp, "Missing opening parenthesis in points attribute");
      }

      if(n >= ao->npoints){
	aweerror(gp, "More points in fallback-comment than in the object itself");
	return token;
      }
      GET_CONST_EXPR;

      if(ao->dpa[n*2] == ADEFVAL){
	ao->pa[n*2] = lval;
	ao->dpa[n*2] = ACOMVAL;
      }

      token = awelex(gp);
      if(token != S_PAREXPSEP)
	aweerror(gp, "Missing "," between x and y point in pointlist");

      GET_CONST_EXPR;

      if(ao->dpa[n*2+1] == ADEFVAL){
	ao->pa[n*2+1] = lval;
	ao->dpa[n*2+1] = ACOMVAL;
      }

      token = awelex(gp);
      if(token != S_ENDPAR)
	aweerror(gp, "Missing closing parenthesis in points attribute");
      token = awelex(gp);
      n++;
    } while(token != S_ENDPAR);
  } else {
    aweerror(gp, "points attribute does not start with '('");
  }
  return token;
}

static int readdefaults(struct cw_status *gp, struct awe_object *ao)
{
  int token;
  char *name;
  int lval=0, is_const;
  char *tval;

  parsecomment = TRUE;
  token = awelex(gp);
  parsecomment = FALSE;
  if(token == S_COMMENT){
    parsenewline = TRUE;
    token = awelex(gp);
    if(token == S_IDENT && !strcmp(awe.ident, "FALLBACK")){
      token = awelex(gp);
      while(token != S_NEWLINE){
	tval = NULL;
	name = strdup(awe.match_buffer);
	token = awelex(gp);
	if(token != S_ASSIGN){
	  CalFree(name);
	  continue;
	}
	if(is_points(gp, ao, name)){
	  token = readdefpoints(gp, ao, name);
	} else {
	  GET_CONST_EXPR;
	  set_attrval(gp, ao, name, tval, lval, ACOMVAL);
	}
	token = awelex(gp);
	CalFree(name);
      }
      flush_chunk(gp);
    } else {
      while(token != S_NOSYMBOL || token != S_NEWLINE)
	token = awelex(gp);
    }
  parsenewline = FALSE;
  token = awelex(gp);
  }
  return token;
}


static void initialize_object(struct awe_object *ao)
{
  if(ao->dactive == ADEFVAL){
    ao->active = DEF_ACTIVE;
  }
  if(ao->dcolor == ADEFVAL){
    ao->color = DEF_COLOR;
  }
  if(ao->ddashes == ADEFVAL){
    ao->dashes = DEF_DASHES;
  }
  if(ao->dendangle == ADEFVAL){
    ao->endangle = DEF_ENDANGLE;
  }
  if(ao->dfill == ADEFVAL){
    ao->fill = DEF_FILL;
  }
  if(ao->dlevel == ADEFVAL){
    ao->level = DEF_LEVEL;
  }
  if(ao->dlinewidth == ADEFVAL){
    ao->linewidth = DEF_LINEWIDTH;
  }
  if(ao->dsavebackground == ADEFVAL){
    ao->savebackground = DEF_SAVEBACKGROUND;
  }
  if(ao->dstartangle == ADEFVAL){
    ao->startangle = DEF_STARTANGLE;
  }
  if(ao->dtexture == ADEFVAL){
    ao->texture = DEF_TEXTURE;
  }
  if(ao->dfont == ADEFVAL){
    ao->font = strdup(DEF_FONT);
  }
  if(ao->douttext == ADEFVAL){
    ao->outtext = strdup(DEF_OUTTEXT);
  }
  if(ao->dimage == ADEFVAL){
    ao->image = strdup(DEF_IMAGE);
  }
  if(ao->dtranslation == ADEFVAL){
    ao->translation = strdup(DEF_TRANSLATION);
  }
}

/*
 * Read object-statement.
 */
static int readobjectstat(struct cw_status *gp, struct awe_func *af, int token)
{
  struct awe_object *ao;
  int i;
  
  af->nobjects++;

  if((ao = NEWAWEOBJECT) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if(af->fo == NULL){
    init_list(ao);
    af->fo = af->lo = ao;
  } else {
    place_next(af->lo, ao);
    af->lo = ao;
  }
  ao->tag = strdup ("Duppeditt");
  ao->name = strdup(awe.match_buffer);
  for(i=0;i < strlen(ao->name);i++)
    delbufc(gp);
  
  add_chunk(gp, af);
  do {
    token = awelex(gp);
    if(token != S_STATSEP)
      token = readattribute(gp, af, ao, token);
  } while(token != S_STATSEP);
  if(ao->translation == NULL){
    ao->translation = strdup(DEF_TRANSLATION);
  }
  add_chunk(gp, af);
  ao->raw = af->end;

  token = readdefaults(gp, ao);

  initialize_object(ao);
  return token;
}



static int readfor(struct cw_status *gp, struct awe_func *af, int token)
{
  char errormsg[256];

  token = awelex(gp);
  if(token != S_BEGPAR){
    sprintf(errormsg, "Opening parenthesis '(' expected");
    aweerror(gp, errormsg);
    return(0);
  }
  token = awelex(gp);
  if(token != S_STATSEP)
    token = readexpr(gp, token);
  token = awelex(gp);
  if(token != S_STATSEP)
    token = readexpr(gp, token);
  token = awelex(gp);
  if(token != S_ENDPAR)
    token = readexpr(gp, token);

  if(token != S_ENDPAR){
    sprintf(errormsg, "Closing parenthesis ')' expected");
    aweerror(gp, errormsg);
    return(0);
  }
  token = awelex(gp);
  token = readstatement(gp, af, token);
  return token;
}

static int readwhile(struct cw_status *gp, struct awe_func *af, int token)
{
  char errormsg[256];

  token = awelex(gp);
  if(token != S_BEGPAR){
    sprintf(errormsg, "Opening parenthesis '(' expected");
    aweerror(gp, errormsg);
    return(0);
  }
  token = awelex(gp);
  token = readexpr(gp, token);
  if(token != S_ENDPAR){
    sprintf(errormsg, "Closing parenthesis ')' expected");
    aweerror(gp, errormsg);
    return(0);
  }
  token = awelex(gp);
  token = readstatement(gp, af, token);
  return token;
}

static int readif(struct cw_status *gp, struct awe_func *af, int token)
{
  char errormsg[256];

  token = awelex(gp);
  if(token != S_BEGPAR){
    sprintf(errormsg, "Opening parenthesis '(' expected");
    aweerror(gp, errormsg);
    return(0);
  }
  token = awelex(gp);
  token = readexpr(gp, token);
  if(token != S_ENDPAR){
    sprintf(errormsg, "Closing parenthesis ')' expected");
    aweerror(gp, errormsg);
    return(0);
  }
  token = awelex(gp);
  token = readstatement(gp, af, token);
  if(token == S_ELSE){
    token = awelex(gp);
    token = readstatement(gp, af, token);
  }
  return token;
}

static int readblock(struct cw_status *gp, struct awe_func *af, int token)
{
  token = awelex(gp);
  while(token != S_END)
    token = readstatement(gp, af, token);
  return token;
}

static int readstatement(struct cw_status *gp, struct awe_func *af, int token)
{
  if(is_object(gp, token)){
    return readobjectstat(gp, af, token);
  } else if(token == S_BEGIN){
    token = readblock(gp, af, token);
    token = awelex(gp);
  } else if(token == S_IF){
    token = readif(gp, af, token);
  } else if(token == S_WHILE){
    token = readwhile(gp, af, token);
  } else if(token == S_FOR){
    token = readfor(gp, af, token);
  } else if(token == S_IDENT){
    token = readexpr(gp, token);
    token = awelex(gp);
  } else if(token == S_RETURN){
    token = awelex(gp);
    token = readexpr(gp, token);
    token = awelex(gp);
  }
  add_chunk(gp, af);
  return token;
}

static int readvar(struct cw_status *gp, int token)
{
  while(token != S_STATSEP)
    token = awelex(gp);
  token = awelex(gp);
  return token;
}

static int readdecls(struct cw_status *gp, int token)
{
  while(token == S_CONST || token == S_INT || token == S_TTEXT || 
	token == S_FFLOAT){
    token = readvar(gp, token);
  }
  return token;
}


static int readfunc(struct cw_status *gp, int token, char *name)
{
  struct awe_func *af;
  
  nblocks++;
  while(token != S_BEGIN)
    token = awelex(gp);
  token = awelex(gp);
  token = readdecls(gp, token);
  if((af = NEWAWEFUNC) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  af->name = name;
  if(fblock == NULL){
    init_list(af);
    fblock = lblock = af;
  } else {
    place_next(lblock, af);
    lblock = af;
  }
    
  while(token != S_END){
    token = readstatement(gp, af, token);
  }

  add_chunk(gp, af);

  token = awelex(gp);
  return token;
}

static int readobjdecl(struct cw_status *gp, int token)
{
  struct awe_func *af;
  char errormsg[256];
  struct awe_attr *aa;

  nblocks++;
  token = awelex(gp);
  if((af = NEWAWEFUNC) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if(token != S_IDENT){
    sprintf(errormsg, "Object identifier expected");
    aweerror(gp, errormsg);
    return(0);
  }
  af->name = strdup(awe.ident);
  af->type = T_USEROBJECT;
  token = awelex(gp);
  if(token == S_BEGPAR){
    do { 
      if(token == S_PAREXPSEP)
	token = awelex(gp);
      token = awelex(gp);
      if((aa = NEWAWEATTR) == NULL){
	userMsg(ErrNOMOREMEM);
	l_exit(NOT_OK);
      }
      aa->type = attr_sym_type(awe.ident);
      token = awelex(gp);
      aa->name = strdup(awe.ident);
      if(af->fa == NULL){
	init_list(aa);
	af->fa = aa;
      } else
	place_next(af->fa, aa);
    } while(token != S_END);
  }
  while(token != S_BEGIN)
    token = awelex(gp);
  token = awelex(gp);
  token = readdecls(gp, token);

  if(fblock == NULL){
    init_list(af);
    fblock = lblock = af;
  } else {
    place_next(lblock, af);
    lblock = af;
  }
    
  while(token != S_END){
    token = readstatement(gp, af, token);
  }
  add_chunk(gp, af);

  token = awelex(gp);

  return token;
}

static int global_parse(struct cw_status *gp, int token)
{
  char errormsg[256];
  char *name;
  fblock = lblock = NULL;

  while(token != S_NOSYMBOL){
    switch(token){
    case S_OBJECT:
      token = readobjdecl(gp, token);
      continue;
    case S_FFLOAT:
      token = awelex(gp);
      break;
    case S_TTEXT:
      token = awelex(gp);
      break;
    case S_INT:
      token = awelex(gp);
    default:
      break;
    }
    if(token != S_IDENT){
      sprintf(errormsg, "Function or variable identifier expected");
      aweerror(gp, errormsg);
      return(0);
    }
    name = strdup(awe.ident);
    token = awelex(gp);
    if(token == S_BEGPAR){
      token = readfunc(gp, token, name);
    } else{
      token = readvar(gp, token);
      CalFree(name);
    }
  }
  return token;
}

int awethor_parse(struct cw_status *gp, char *filename)
{
  FILE *fp;
  int token;
  awethor_free(gp);
  if((fp = fopen(filename, "rb")) == NULL){
    return FALSE;
  }
  awe.line = 1;
  awe.file = fp;
  awe.strind = -1;
  awe.filename = strdup(filename);
  nblocks = 0;
  token = awelex(gp);
  token = global_parse(gp, token);
  add_chunk(gp, lblock);
  return TRUE;
}

static int obj_has_attr(struct awe_object *ao, char *aname)
{
  /* Only textobjects have font and outtext attributes */
  if((!strcmp(aname, "outtext") || !strcmp(aname, "font")) &&
     strcmp(ao->name, "textobj"))
    return FALSE;

  if((!strcmp(aname, "endangle") || !strcmp(aname, "startangle")) &&
     strcmp(ao->name, "arc"))
    return FALSE;

  if((!strcmp(aname, "translation")) && strcmp(ao->name, "inputarea"))
    return FALSE;

  return TRUE;
}

static int bufsize;
static void bufcat(char **buf, char *str)
{
  if(strlen(*buf) + strlen(str) > bufsize){
    *buf = CalRealloc(*buf, strlen(*buf) + strlen(str) * 2);
  }
  strcat(*buf, str);
} 

void update_object(struct awe_object *ao)
{
  char pointbuf[2048];
  char intbuf[64];
  char *buf;
  int i;
  int seen_comval = FALSE;
  int do_point_comment = FALSE;

  bufsize = 1024;
  if(ao->raw == NULL){
    if((ao->raw = NEWAWECHUNK) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
  }
  if(ao->raw->buf)
    CalFree(ao->raw->buf);
  
  buf = ao->raw->buf = CalMalloc(bufsize * sizeof(char));

  if(buf == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  strcpy(buf, ao->name);
  strcat(buf, " points = (");
  
  pointbuf[0] = 0;
  for(i = 0; i < ao->npoints; i++){

    if(ao->dpa != NULL && ao->dpa[2*i] == AOMIVAL)
      continue; /* Skip this point */

    if(i > 0)
      strcat(pointbuf, ",");

    if(ao->dpa == NULL || ao->dpa[2*i] == ADIRVAL){
      sprintf(intbuf, "(%ld,", ao->pa[2*i]);
      strcat(pointbuf, intbuf);
    } else {
      strcat(pointbuf, "(");
      strcat(pointbuf, ao->tpa[2*i]);
    }
    if(ao->dpa == NULL || ao->dpa[2*i+1] == ADIRVAL){
      sprintf(intbuf, "%ld)", ao->pa[2*i+1]);
      strcat(pointbuf, intbuf);
    } else {
      strcat(pointbuf, ao->tpa[2*i+1]);
    }
  }
  strcat(pointbuf, ")");
  if(strlen(pointbuf) + strlen(buf) > bufsize){
    bufsize = 2 * (strlen(pointbuf) + strlen(buf));
    buf = CalRealloc(buf, bufsize);
  }
  strcat(buf, pointbuf);


#define UPDATEVAL(name, defname)   if(obj_has_attr(ao, #name))\
  if((ao->t ## name == NULL && ao->name != defname)){ \
    sprintf(intbuf, ", %s = %ld", #name, ao->name);\
    bufcat(&buf, intbuf); \
    ao->d ## name = ADIRVAL;\
  } else { \
    if(ao->t ## name != NULL){ \
      sprintf(intbuf, ", %s = ", #name);\
      bufcat(&buf, intbuf); \
      bufcat(&buf, ao->t ## name);\
    }\
  }

#define UPDATETVAL(name, defname)   if(obj_has_attr(ao, #name))\
  if(ao->t ## name == NULL && ao->name != NULL && ((strcmp(ao->name,defname) && strcmp(ao->name, "")) || (ao->d ## name == ADIRVAL))){ \
    sprintf(intbuf, ", %s = \"", #name);\
    bufcat(&buf, intbuf); \
    bufcat(&buf, ao->name);\
    bufcat(&buf, "\"");\
    ao->d ## name = ADIRVAL;\
  } else { \
    if(ao->t ## name != NULL){ \
      sprintf(intbuf, ", %s = ", #name);\
      bufcat(&buf, intbuf); \
      bufcat(&buf, ao->t ## name);\
    }\
  }

  UPDATEVAL(active, DEF_ACTIVE);
  UPDATEVAL(color, DEF_COLOR);
  UPDATEVAL(dashes, DEF_DASHES);
  UPDATEVAL(endangle, DEF_ENDANGLE);
  UPDATEVAL(fill, DEF_FILL);
  UPDATEVAL(level, DEF_LEVEL);
  UPDATEVAL(linewidth, DEF_LINEWIDTH);
  UPDATEVAL(savebackground, DEF_SAVEBACKGROUND);
  UPDATEVAL(startangle, DEF_STARTANGLE);
  UPDATEVAL(texture, DEF_TEXTURE);

  UPDATETVAL(font, DEF_FONT);
  UPDATETVAL(outtext, DEF_OUTTEXT);
  UPDATETVAL(image, DEF_IMAGE);

  if(obj_has_attr(ao, "translation")) {
    if (ao->translation != NULL){
      sprintf(intbuf, ", %s = ", "translation");
      bufcat(&buf, intbuf);
      bufcat(&buf, ao->translation);
      ao->dtranslation = ADIRVAL;
    } else {
      ao->translation = strdup(DEF_TRANSLATION);
    }
  } 
  bufcat(&buf, ";\n");

  /* Update eventual comment values */

  if(ao->dpa != NULL){
    for(i = 0; i < ao->npoints * 2; i++)
      if(ao->dpa[i] == ACOMVAL || (ao->dpa[i] == ADEFVAL && ao->pa[i] != 0)){
	seen_comval = do_point_comment = TRUE;
      break;
      }

    if(do_point_comment){
      bufcat(&buf, "  // FALLBACK ");
      bufcat(&buf, "points = (");
      for(i = 0; i < ao->npoints * 2; i++){
	if(i % 2 == 0){
	  if(i > 1)
	    bufcat(&buf, ",");
	  
	  bufcat(&buf, "(");
	}
	if(ao->dpa[i] == ACOMVAL || (ao->dpa[i] == ADEFVAL && ao->pa[i] != 0)){
	  sprintf(intbuf, "%ld", ao->pa[i]);
	  bufcat(&buf, intbuf);
	} else 
	  bufcat(&buf, "0");
	if(i % 2 == 1)
	  bufcat(&buf, ")"); 
	else
	  bufcat(&buf, ","); 
      }
      bufcat(&buf, ")"); 
    }
  }

#define UPDATECVAL(name,defname)   if(obj_has_attr(ao, #name) && (ao->d ## name == ACOMVAL || (ao->d ## name == ADEFVAL && ao->name != defname))){ \
    if(! seen_comval){\
      bufcat(&buf, "  // FALLBACK ");\
      sprintf(intbuf, "%s = %ld", #name, ao->name);\
      seen_comval = TRUE;\
    } else \
      sprintf(intbuf, ", %s = %ld", #name, ao->name);\
    bufcat(&buf, intbuf); \
  }

#define UPDATECTVAL(name,defname)   if(obj_has_attr(ao, #name) && (ao->d ## name == ACOMVAL || (ao->d ## name == ADEFVAL && ao->name != NULL && strcmp(ao->name,defname)))){ \
    if(! seen_comval){\
      bufcat(&buf, "  // FALLBACK ");\
      seen_comval = TRUE;\
    } else \
      bufcat(&buf, ",");\
    sprintf(intbuf, "%s = \"", #name);\
    bufcat(&buf, intbuf); \
    bufcat(&buf, ao->name);\
    bufcat(&buf, "\"");\
  }


  UPDATECVAL(active, DEF_ACTIVE);
  UPDATECVAL(color, DEF_COLOR);
  UPDATECVAL(dashes, DEF_DASHES);
  UPDATECVAL(endangle, DEF_ENDANGLE);
  UPDATECVAL(fill, DEF_FILL);
  UPDATECVAL(level, DEF_LEVEL);
  UPDATECVAL(linewidth, DEF_LINEWIDTH);
  UPDATECVAL(savebackground, DEF_SAVEBACKGROUND);
  UPDATECVAL(startangle, DEF_STARTANGLE);
  UPDATECVAL(texture, DEF_TEXTURE);

  UPDATECTVAL(font, DEF_FONT);
  UPDATECTVAL(outtext, DEF_OUTTEXT);
  UPDATECTVAL(image, DEF_IMAGE);
  UPDATECTVAL(translation, DEF_TRANSLATION);
  
  if(seen_comval)
    bufcat(&buf, "\n");

  ao->raw->size = strlen(buf);
  ao->raw->buf = buf;
}

static void writefunc(FILE *fp, struct awe_func *af)
{
  struct awe_object *ao;
  struct awe_chunk *ac;
  
  /* Update objects */
  for(ao = af->fo; ao != NULL; ao = next(ao)){
    update_object(ao);
  }

  for(ac = af->start; ac != NULL; ac = next(ac)){
    fwrite(ac->buf, 1, ac->size, fp);
  }
}

int awethor_write(struct cw_status *gp, char *filename)
{
  FILE *fp;
  struct awe_func *af;

  if((fp = fopen(filename, "wb")) == NULL){
    return FALSE;
  }
  for(af = fblock; af != NULL; af = next(af))
    writefunc(fp, af);

  fclose(fp);
  return TRUE;
}

#define f_mem(str)  if(str != NULL) CalFree(str)

static void f_awe_chunk(struct awe_chunk *ac)
{
  assert(ac);
  remove_at(ac);
  CalFree(ac->buf);
  CalFree(ac);
}

static void f_awe_attrs(struct awe_attr *aa)
{
  struct awe_attr *pa;
  while(aa != NULL){
    pa = aa;
    aa = next(aa);
    remove_at(aa);
    CalFree(pa);
  }
}

static void f_awe_object(struct awe_object *ao)
{
  int i;

  f_awe_chunk(ao->raw);
  f_awe_attrs(ao->fa);

  f_mem(ao->name);
  f_mem(ao->tag);

  if(ao->tpa != NULL){
    for(i = 0; i < ao->npoints; i++){
      if(ao->tpa[i] != NULL)
	CalFree(ao->tpa[i]);
    }
    CalFree(ao->tpa);
  }
  f_mem(ao->pa);
  f_mem(ao->dpa);
  f_mem(ao->font);
  f_mem(ao->outtext);
  f_mem(ao->image);
  f_mem(ao->translation);

  f_mem(ao->tactive);
  f_mem(ao->tcolor);
  f_mem(ao->tdashes);
  f_mem(ao->tendangle);
  f_mem(ao->tfill);
  f_mem(ao->tlevel);
  f_mem(ao->tlinewidth);
  f_mem(ao->tsavebackground);
  f_mem(ao->tstartangle);
  f_mem(ao->ttexture);
  f_mem(ao->tfont);
  f_mem(ao->touttext);
  f_mem(ao->timage);
  f_mem(ao->ttranslation);

  CalFree(ao);
}

void delete_awe_object(struct cw_status *gp, int i)
{
  struct awe_object *ao;
  char errormsg[256];

  if(i >= ao_size){
    sprintf(errormsg, "Internal error: Trying to kill non-existant object with index %d", i);
    userMsg(errormsg);
    l_exit(NOT_OK);
  }
  ao = ao_buf[i];
  ao_buf[i] = NULL;
  assert(ao);
  
  if(ao == cblock->fo){
    cblock->fo = next(ao);
  } 
  if(ao == cblock->lo){
    cblock->lo = prev(ao);
  }
  remove_at(ao);
  f_awe_object(ao);
}

static void f_awe_objects(struct awe_object *ao)
{
  struct awe_object *a;
  while(ao != NULL){
    a = ao;
    ao = next(ao);
    remove_at(a);
    f_awe_object(a);
  }
}

static void f_awe_chunks(struct awe_chunk *ac)
{
  struct awe_chunk *a;
  while(ac != NULL){
    a = ac;
    ac = next(ac);
    f_awe_chunk(a);
  }
}


/*
 * Free all memory allocated by this module. 
 * The function will also initialize static datastructures, so 
 * they can be reused by subsequent calls on the module. 
 */
void awethor_free(struct cw_status *gp)
{
  struct awe_func *c;

  ao_size = 0;
  if (!ao_buf) return;

  CalFree(ao_buf);
  ao_buf = NULL;

  while(fblock != NULL){
    c = fblock;
    fblock = next(fblock);
    f_awe_objects(c->fo);
    f_awe_attrs(c->fa);
    f_mem(c->name);
    f_awe_chunks(c->start);
    remove_at(c);
    CalFree(c);
  }
  awefree(gp);
  nblocks = 0;
  cblock = fblock = lblock = NULL;
}

#define FUNCSTARTTEXT "main ()\n{\n"
#define FUNCENDTEXT "\nwhile (1) {\ninput; \noutput;\n}\n}\n"


/*
 * Create the initial function generated by AweThor when the user 
 * starts drawing from scratch.
 */
static void create_init_func(struct cw_status *gp)
{
  if ((cblock = CalCalloc (1, sizeof (struct awe_func))) == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  init_list (cblock);
  fblock = lblock = cblock;
  nblocks = 1;
  if ((cblock->start = NEWAWECHUNK) == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  cblock->start->buf = strdup (FUNCSTARTTEXT);
  cblock->start->size = strlen (FUNCSTARTTEXT);
  init_list (cblock->start);
  
  if ((cblock->end = NEWAWECHUNK) == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  cblock->end->buf = strdup (FUNCENDTEXT);
  cblock->end->size = strlen (FUNCENDTEXT);
  place_next (cblock->start, cblock->end);
  
  cblock->name = strdup ("main");
  cblock->type = T_USERFUNC;
}

struct awe_object *add_awe_object (struct cw_status *gp, char *oname)
{
  struct awe_object *ao;

  if ((ao = NEWAWEOBJECT) == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  ao->name = strdup (oname);

  ao_buf = ao_buf ? CalRealloc(ao_buf, ++ao_size * sizeof(struct awe_object *))
    : CalMalloc (++ao_size * sizeof(struct awe_object *));

  if(ao_buf == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  ao_buf[ao_size-1] = ao;
  
  if (cblock == NULL) 
    create_init_func(gp);

  update_object (ao);

  if(cblock->lo != NULL){
    /* Insert new objects after the last object in the function */
    place_next(cblock->lo->raw, ao->raw);  
    place_next(cblock->lo, ao);
    cblock->lo = ao;
  } else {
    /* Insert the new object just before the of the function */
    init_list(ao);
    cblock->fo = cblock->lo = ao;
    place_prev(cblock->end, ao->raw);
  }

  return ao;
}

#ifdef MAIN
void main()
{
  int n, i;
  char **name;
  awethor_parse("foo.awe");

  printf("get_awe_blocks\n");
  n = get_awe_blocks(name);
  for(i = 0; i < n; i++){
    printf("Name : %s\n", name[i]);
  }
  set_current_block("main");
  awethor_write(gp, "bar.awe");
}

void c_exit(int exval)
{
  exit(exval);
}
#endif
