#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>

#include <sys/types.h>  
#include <sys/stat.h>
#include <fcntl.h>
#ifdef UNIX
#include <sys/time.h>
#include <unistd.h>
#include <dirent.h>
#endif

#include "candle.h"
#include "wwwcom.h"
#include "learnuni.h"
#include "parser.h"
#include "lex.h"
#include "simulate.h"
#include "graphic.h"
#include "nodes.h"
#include "input.h"
#include "error.h"
#include "attr.h"
#ifdef AWETHOR
#include "awethor.h"
#include "pbmplus.h"
#include "protos/objects.h"
#endif

#include "sysproto.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/simulate.h"
#include "protos/region.h"
#include "protos/update.h"
#include "protos/readsim.h"
#include "protos/instance.h"
#include "protos/input.h"
#include "protos/funcutil.h"
#include "protos/freesim.h"
#include "protos/instsim.h"
#include "protos/optimize.h"
#include "protos/memory.h"
#include "protos/parser.h"

#define lmin(a, b)  (((a) < (b)) ? (a) : (b))

#define awe gp->awe
#define ao_buf awe.ao_buf
#define ao_size awe.ao_size

void set_oper(struct cw_status *gp, struct attribute *attr, struct vinst *vi, 
	      long n)
{
  struct oper *op, *po;
  struct intarray *arr;

  if((op = NEWOPER) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if((arr = op->left.array = NEWINTARRAY) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  arr->type = DO_VINST;
  arr->which.vi = vi;
  if((arr->indeks = NEWPARAM) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  init_list(arr->indeks);
  if((po = arr->indeks->exp = NEWOPER) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  po->flag = INTVAL;
  po->type = DO_VINST;
  po->left.value = n;
  compute_valtype(gp, po);
  
  op->flag = var_type(vi->parent);
  op->type = DO_VINST;
  compute_valtype(gp, op);

  attr->exprval.oper = op;
  addattr(gp, vi, attr);
}

void set_pl_opers(struct cw_status *gp, struct attribute *attr,
		  struct vinst *xvi, struct vinst *yvi, 
		  struct vinst *nvi, long n)
{
  struct oper *op, *po;
  struct intarray *arr;
  long npoints, i;
  struct param *par;
  struct vinst *vi;

  npoints = nvi->value.elem[n];
  
  attr->exprval.op = malloc(2*npoints*sizeof(struct oper *));
  for(i = 0; i < npoints; i++){
    /* Bind X coordinate */
    vi = xvi;
    if((op = attr->exprval.op[2*i] = NEWOPER) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    if((arr = op->left.array = NEWINTARRAY) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    arr->type = DO_VINST;
    arr->which.vi = vi;
    /* Set up first dimension */
    if((arr->indeks = NEWPARAM) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    init_list(arr->indeks);
    if((po = arr->indeks->exp = NEWOPER) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    po->flag = INTVAL;
    po->type = DO_VINST;
    po->left.value = n;
    compute_valtype(gp, po);
    /* Setup second dimension */
    if((par = NEWPARAM) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    place_next(arr->indeks, par);
    if((po = par->exp = NEWOPER) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    po->flag = INTVAL;
    po->type = DO_VINST;
    po->left.value = i;
    compute_valtype(gp, po);
    
    op->flag = var_type(vi->parent);
    op->type = DO_VINST;
    compute_valtype(gp, op);

    /* Bind Y coordinate */
    vi = yvi;
    if((op = attr->exprval.op[2*i+1] = NEWOPER) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    if((arr = op->left.array = NEWINTARRAY) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    arr->type = DO_VINST;
    arr->which.vi = vi;
    /* Set up first dimension */
    if((arr->indeks = NEWPARAM) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    init_list(arr->indeks);
    if((po = arr->indeks->exp = NEWOPER) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    po->flag = INTVAL;
    po->type = DO_VINST;
    po->left.value = n;
    compute_valtype(gp, po);
    /* Setup second dimension */
    if((par = NEWPARAM) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    place_next(arr->indeks, par);
    if((po = par->exp = NEWOPER) == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit(NOT_OK);
    }
    po->flag = INTVAL;
    po->type = DO_VINST;
    po->left.value = i;
    compute_valtype(gp, po);
    
    op->flag = var_type(vi->parent);
    op->type = DO_VINST;
    compute_valtype(gp, op);
  }
  if((attr->curval.ptarr = calloc (2*npoints, sizeof (long))) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if((attr->directval.ptarr = calloc (2*npoints, sizeof (long))) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  
  addattr(gp, xvi, attr);
  addattr(gp, yvi, attr);
}

/* A set of functions to be called from Awethor */

long terminate (struct cw_status *gp, struct param FAR *par)
{
  checkPars (gp, par, 0, "terminate");
  c_exit(gp, OK);
  return 1; /* Only to please compiler */
}



  /* Windowsystem specific routine */
char FAR *textEditPopup (struct cw_status *, char FAR *);

long TextPopup (struct cw_status *gp, struct param FAR *par)
{
  int arg = 0;
  char FAR *funcname = "textPopup", FAR *buf;
  struct param FAR *p = par;

  buf = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  buf = textEditPopup (gp, buf); /* Parameter must be freed in callee if altered */
  t_set_par_value (gp, p, strdup (buf));
  free (buf);

  return 1;
}

long Load (struct cw_status *gp, struct param *par)
{
  int arg = 0, res;
  char FAR *funcname = "load", *filename;

  filename = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);
  res = awethor_parse (gp, filename);
  free (filename);
  return res;
}

static void *rallarr (struct cw_status *gp, struct param *par, int n, 
		      int elsiz, int arrdim, char dofree)
{
  unsigned long i;
  long oldsize = par->exp->left.array->which.vi->size[0];
  char **tel;

  changed_par (gp, par);
  if(dofree && par->exp->left.array->which.vi->parent->type == TEXTARRAY){
    for(i = 0, tel = par->exp->left.array->which.vi->value.telem; 
	i < par->exp->left.array->which.vi->size[0];i++) {
      if(tel[i] != NULL)
	CalFree(tel[i]);
    }
    oldsize = 0;
  }
  if ((par->exp->left.array->which.vi->value.elem =
       CalRealloc (par->exp->left.array->which.vi->value.elem,
		n*elsiz)) == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit (NOT_OK);
  }
  par->exp->left.array->which.vi->size[0] = n;
  if(par->exp->left.array->which.vi->parent->type == TEXTARRAY){
    for(i = oldsize, tel = par->exp->left.array->which.vi->value.telem; 
	i < par->exp->left.array->which.vi->size[0];i++)
      {
	tel[i] = strdup("");
      }
  }
  
  return par->exp->left.array->which.vi->value.elem;
}

#define AWETHORMAXPTS 16

static char *ao_symbols[] = {
  "point", "line", "arc", "box", "polygon",
  "textobj", "image", "inputarea", "window", NULL
};

static long ao_constants[] = {
  15, 16, 17, 18, 19, 20, 2, 21, 22, -1
};

#define AWETHORPOINT     15
#define AWETHORLINE      16
#define AWETHORARC       17
#define AWETHORBOX       18
#define AWETHORPOLYGON   19
#define AWETHORTEXTOBJ   20
#define AWETHORIMAGE      2
#define AWETHORINPUTAREA 21
#define AWETHORWINDOW    22

static long ao_symb2const (char *t)
{
  int i, nul = 0;

  assert (t);

  for (i = 0; ao_symbols[i]; i++)
    if (strcmp (t, ao_symbols[i]) == 0)
      return ao_constants[i];
  i = i/nul;
  return -1;
}

static char *ao_const2symb (long l)
{
  int i, nul = 0;

  for (i = 0; ao_constants[i] != -1; i++)
    if (l == ao_constants[i])
      return ao_symbols[i];
  i = i/nul;
  return NULL;
}

long LoadView (struct cw_status *gp, struct param *par)
{
  int arg = 0, i, j;
  char *funcname = "loadView", *tofunc;
  long *obj, *xattr, *yattr, *colattr, *actattr, *sbattr, *levattr, *fattr,
    *texattr, *dshattr, *lwattr, *saattr, *eaattr, *npoints;
  char **objname, **fnattr, **otattr, **imgattr, **trlattr;

  tofunc = getTextPar (gp, &par, ++arg, funcname);
  if (!set_current_block (gp, tofunc)) {
    free (tofunc);
    return 0;
  }
  free (tofunc);

  /* Reallocate all incoming arrays */
  obj = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE);
  par = next (par);
  xattr = rallarr (gp, par, ao_size, sizeof(long)*AWETHORMAXPTS, 2, TRUE);
  par->exp->left.array->which.vi->size[1] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[2] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[3] = 1;
  par = next (par);
  yattr = rallarr (gp, par, ao_size, sizeof(long)*AWETHORMAXPTS, 2, TRUE);
  par->exp->left.array->which.vi->size[1] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[2] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[3] = 1;
  par = next (par);
  npoints = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  colattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  actattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  sbattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  levattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  fattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  texattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  dshattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  lwattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  saattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  eaattr = rallarr (gp, par, ao_size, sizeof(long), 1, TRUE); par = next (par);
  objname = rallarr (gp, par, ao_size, sizeof(char *), 1, TRUE); par = next (par);
  fnattr = rallarr (gp, par, ao_size, sizeof(char *), 1, TRUE); par = next (par);
  otattr = rallarr (gp, par, ao_size, sizeof(char *), 1, TRUE); par = next (par);
  imgattr = rallarr (gp, par, ao_size, sizeof(char *), 1, TRUE); par = next (par);
  trlattr = rallarr (gp, par, ao_size, sizeof(char *), 1, TRUE); par = next (par);
  checkPars (gp, par, arg, funcname);

  /* Got and checked all parameters, all arrays now have the right size */
  for (i = 0; i < ao_size; i++) {
    obj[i] = ao_symb2const (ao_buf[i]->name);
    npoints[i] = ao_buf[i]->npoints;
    for (j = 0; j < AWETHORMAXPTS; j++) {
      xattr[i*AWETHORMAXPTS+j] = ao_buf[i]->pa[2*j];
      yattr[i*AWETHORMAXPTS+j] = ao_buf[i]->pa[2*j+1];
    }
    actattr[i] = ao_buf[i]->active;
    colattr[i] = ao_buf[i]->color;
    dshattr[i] = ao_buf[i]->dashes;
    saattr[i] = ao_buf[i]->startangle;
    eaattr[i] = ao_buf[i]->endangle;
    fattr[i] = ao_buf[i]->fill;
    levattr[i] = ao_buf[i]->level;
    lwattr[i] = ao_buf[i]->linewidth;
    sbattr[i] = ao_buf[i]->savebackground;
    texattr[i] = ao_buf[i]->texture;
    objname[i] = strdup (ao_buf[i]->tag);
    fnattr[i] = strdup (ao_buf[i]->font);
    otattr[i] = strdup (ao_buf[i]->outtext);
    imgattr[i] = strdup (ao_buf[i]->image);
    trlattr[i] = strdup (ao_buf[i]->translation);
  }
  return ao_size;
}

long Save (struct cw_status *gp, struct param *par)
{
  int arg = 0, res;
  char FAR *funcname = "save", *filename;

  filename = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  res = awethor_write (gp, filename);
  free (filename);
  return res;
}

long SaveView (struct cw_status *gp, struct param *par)
{
  int arg = 0, i, j;
  char FAR *funcname = "saveView";
  long size, arraysize;
  long *obj, *xattr, *yattr, *colattr, *actattr, *sbattr, *levattr, *fattr,
    *texattr, *dshattr, *lwattr, *saattr, *eaattr, *npoints;
  char **objname, **fnattr, **otattr, **imgattr, **trlattr;

  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&obj, &size, 1);
  arraysize = size;
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&xattr, &size, 2);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&yattr, &size, 2);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&npoints, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&colattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&actattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&sbattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&levattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&fattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&texattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&dshattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&lwattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&saattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getIntArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&eaattr, &size, 1);
  arraysize = lmin (arraysize, size);

  getTextArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&objname, &size, 1);
  arraysize = lmin (arraysize, size);
  getTextArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&fnattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getTextArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&otattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getTextArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&imgattr, &size, 1);
  arraysize = lmin (arraysize, size);
  getTextArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&trlattr, &size, 1);
  arraysize = lmin (arraysize, size);
  checkPars (gp, par, arg, funcname);

  for (i = 0; i < arraysize-1; i++) {
    ao_buf[i]->name = strdup (ao_const2symb (obj[i]));
    ao_buf[i]->npoints = (short) npoints[i];
    if(ao_buf[i]->pa)
      free(ao_buf[i]->pa);
    ao_buf[i]->pa = calloc(ao_buf[i]->npoints * 2, sizeof(long));
    if(ao_buf[i]->pa == NULL){
      userMsg(ErrNOMOREMEM);
      l_exit (NOT_OK);
    }
    for (j = 0; j < ao_buf[i]->npoints; j++) {
      ao_buf[i]->pa[2*j] = xattr[i*AWETHORMAXPTS+j];
      ao_buf[i]->pa[2*j+1] = yattr[i*AWETHORMAXPTS+j];
    }
    ao_buf[i]->active = actattr[i];
    ao_buf[i]->color = colattr[i];
    ao_buf[i]->dashes = dshattr[i];
    ao_buf[i]->startangle = saattr[i];
    ao_buf[i]->endangle = eaattr[i];
    ao_buf[i]->fill = fattr[i];
    ao_buf[i]->level = levattr[i];
    ao_buf[i]->linewidth = lwattr[i];
    ao_buf[i]->savebackground = sbattr[i];
    ao_buf[i]->texture = texattr[i];
    ao_buf[i]->tag = strdup (objname[i]);
    ao_buf[i]->font = strdup (fnattr[i]);
    ao_buf[i]->outtext = strdup (otattr[i]);
    ao_buf[i]->image = strdup (imgattr[i]);
    ao_buf[i]->translation = strdup (trlattr[i]);
  }
  return arraysize-1;
}

struct attribute *natt (struct cw_status *gp, unsigned char ptype, 
			unsigned char atype,
			struct commonobj *parent)
{
  struct attribute *aret;

  aret = NEWATTRIBUTE;
  aret->parenttype = ptype;
  aret->parent.obj = parent;
  aret->valid = 0;
  aret->attrtype = atype;

  if(gp->curlunit->nattr == NULL){
    init_list(aret);
    gp->curlunit->nattr = aret;
  } else 
    place_next(gp->curlunit->lnattr, aret);
  gp->curlunit->lnattr = aret;

  return aret;
}

struct vinst *next_par_vi (struct param **pp)
{
  struct vinst *vi;

  vi = (*pp)->exp->left.array->which.vi;
  *pp = next (*pp);
  return vi;
}

struct translation *ntrans (struct cw_status *gp, char type, char button, 
			    unsigned long modmask,
			    char *funcname, int actparval)
{
  struct translation *t;
  struct param *par;
  struct oper *po;

  if ((t = NEWTRANSLATION) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if ((t->event = NEWEVENT) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  init_list (t->event);
  if ((t->act = NEWSIMACTION) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  init_list(t->act);
  if ((t->act->do_act = NEWACTION) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  if((par = NEWPARAM) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  init_list(par);
  if((po = par->exp = NEWOPER) == NULL){
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  po->flag = INTVAL;
  po->type = DO_VINST;
  po->left.value = actparval;
  compute_valtype(gp, po);

  t->event->type = type;
  t->event->detail.button = button;
  t->event->modifiermask = modmask;
  t->act->do_act->function.func_act =
    get_function ((struct commonblock *)gp->curlunit, funcname);
  t->act->do_act->actual.par = par;
  t->act->do_act->type = UINTFUNC;
  
  return t;
}

struct graphobj *ngraph (int gotype, struct winobj *parentwin)
{
  struct graphobj *newobject;
  
  if ((newobject = (struct graphobj *)calloc (1, sizeof (struct graphobj)))
      == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  
  newobject->type = gotype;
  newobject->status = VALID;
  newobject->visible = TRUE;

  return newobject;
}

struct inpobj *ninp (int iotype, struct winobj *parentwin)
{
  struct inpobj FAR *newobject;

  if (!(newobject = (struct inpobj FAR *)calloc (1, sizeof (struct inpobj)))) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  /*  newobject->parentwin = parentwin; */
  newobject->type = iotype;

  return newobject;
}

void makeObject (struct cw_status *gp, 
		 int n, struct param *par, long *obj, int new_obj)
{
  int a, d, w;
  struct awe_object *ao = ao_buf[n];
  struct vinst *xvi, *yvi, *npvi, *fnvi, *imgvi, *otvi, *actvi, *levvi;
  long *npts;
  CalImage *cip;

  switch (obj[n]) {
  case AWETHORWINDOW:
    break;
  case AWETHORPOINT:
    ao->baseobject = ngraph (S_POINT, NULL);
    break;
  case AWETHORLINE:
    ao->baseobject = ngraph (S_LINE, NULL);
    break;
  case AWETHORBOX:
    ao->baseobject = ngraph (S_BOX, NULL);
    break;
  case AWETHORARC:
    ao->baseobject = ngraph (S_ARC, NULL);
    break;
  case AWETHORPOLYGON:
    ao->baseobject = ngraph (S_POLYGON, NULL);
    break;
  case AWETHORTEXTOBJ:
    ao->baseobject = ngraph (S_TEXTOBJECT, NULL);
    break;
  case AWETHORIMAGE:
    ao->baseobject = ngraph (S_IMAGE, NULL);
    break;
  default:
    break;
  }
  if (!ao->baseobject) ao->baseobject = ngraph (S_BOX, NULL);
  ao->xtraobject = ninp (S_INPUTAREA, NULL);

  if (gp->curlunit->cur_funcinst->fgraph)
    place_prev2 (gp->curlunit->cur_funcinst->fgraph, ao->baseobject);
  else
    init_list2 (ao->baseobject);
  gp->curlunit->cur_funcinst->fgraph = ao->baseobject;

  if (gp->curlunit->cur_funcinst->finp)
    place_prev2 (gp->curlunit->cur_funcinst->finp, ao->xtraobject);
  else
    init_list2 (ao->xtraobject);
  gp->curlunit->cur_funcinst->finp = ao->xtraobject;

  /* Fix attributes for baseobject */
  ao->baseobject->pointlist =
    natt (gp, GRAPH, S_POINTS, (struct commonobj *)ao->baseobject);
  xvi = next_par_vi (&par);
  yvi = next_par_vi (&par);
  npvi = next_par_vi (&par);

  /* Use same var.instances for inputobject */
  ao->xtraobject->pointlist =
    natt (gp, INP, S_POINTS, (struct commonobj *)ao->xtraobject);

  npts = npvi->value.elem; /* The npoints-array */
  ao->baseobject->ptcount = ao->xtraobject->ptcount = npts[n];

  if (obj[n] == AWETHORPOINT || obj[n] == AWETHORLINE
      || obj[n] == AWETHORBOX || obj[n] == AWETHORARC
      || obj[n] == AWETHORPOLYGON || obj[n] == AWETHORTEXTOBJ
      || obj[n] == AWETHORINPUTAREA || obj[n] == AWETHORWINDOW)
    set_oper (gp, (ao->baseobject->color =
	       natt (gp, GRAPH, S_COLOR, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
  else {
    ao->baseobject->color = NULL;
    next_par_vi (&par);
  }
  set_oper (gp, (ao->baseobject->active =
	     natt (gp, GRAPH, S_ACTIVE, (struct commonobj *)ao->baseobject)),
	    (actvi = next_par_vi (&par)), n);
  set_oper (gp, (ao->xtraobject->active =
	     natt (gp, INP, S_ACTIVE, (struct commonobj *)ao->xtraobject)),
	    actvi, n);
  if (obj[n] != AWETHORINPUTAREA)
    set_oper (gp, (ao->baseobject->saveback =
	       natt (gp, GRAPH, S_SAVEBG, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
  else {
    ao->baseobject->saveback = NULL;
    next_par_vi (&par);
  }
  set_oper (gp, (ao->baseobject->level =
	     natt (gp, GRAPH, S_LEVEL, (struct commonobj *)ao->baseobject)),
	    (levvi = next_par_vi (&par)), n);

  set_oper (gp, (ao->xtraobject->level =
	     natt (gp, INP, S_LEVEL, (struct commonobj *)ao->xtraobject)),
	    levvi, n);

  if (obj[n] == AWETHORBOX || obj[n] == AWETHORARC
      || obj[n] == AWETHORPOLYGON || obj[n] == AWETHORINPUTAREA)
    set_oper (gp, (ao->baseobject->fill =
	       natt (gp, GRAPH, S_FILL, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
  else {
    ao->baseobject->fill = NULL;
    next_par_vi (&par);
  }
  if (obj[n] != AWETHORWINDOW)
    set_oper (gp, (ao->baseobject->texture =
	       natt (gp, GRAPH, S_TEXTURE, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
  else {
    ao->baseobject->texture = NULL;
    next_par_vi (&par);
  }
  if (obj[n] == AWETHORLINE || obj[n] == AWETHORBOX
      || obj[n] == AWETHORARC || obj[n] == AWETHORPOLYGON) {
    set_oper (gp, (ao->baseobject->dashes =
	       natt (gp, GRAPH, S_DASHES, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
    set_oper (gp, (ao->baseobject->linewidth =
	       natt (gp, GRAPH, S_LINEWIDTH, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
  }
  else {
    ao->baseobject->dashes = ao->baseobject->linewidth = NULL;
    next_par_vi (&par); next_par_vi (&par);
  }
  if (obj[n] == AWETHORARC) {
    set_oper (gp, (ao->baseobject->startangle =
	       natt (gp, GRAPH, S_STARTANGLE, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
    set_oper (gp, (ao->baseobject->endangle =
	       natt (gp, GRAPH, S_ENDANGLE, (struct commonobj *)ao->baseobject)),
	      next_par_vi (&par), n);
  }
  else {
    ao->baseobject->startangle = ao->baseobject->endangle = NULL;
    next_par_vi (&par); next_par_vi (&par);
  }
  next_par_vi (&par); /* Skip objname-attribute */

  if (obj[n] == AWETHORTEXTOBJ) {
    set_oper (gp, (ao->baseobject->font =
	       natt (gp, GRAPH, S_FONT, (struct commonobj *)ao->baseobject)),
	      (fnvi = next_par_vi (&par)), n);
    ao->baseobject->font->curval.typtxt = strdup("");
    set_oper (gp, (ao->baseobject->outtext =
	       natt (gp, GRAPH, S_OUTTEXT, (struct commonobj *)ao->baseobject)),
	      (otvi = next_par_vi (&par)), n);
    ao->baseobject->outtext->curval.typtxt = strdup("");
  }
  else {
    ao->baseobject->font = ao->baseobject->outtext = NULL;
    fnvi = next_par_vi (&par);
    otvi = next_par_vi (&par);
  }
  if (obj[n] == AWETHORBOX || obj[n] == AWETHORARC
      || obj[n] == AWETHORPOLYGON || obj[n] == AWETHORTEXTOBJ
      || obj[n] == AWETHORIMAGE) {
    set_oper (gp, (ao->baseobject->image =
	       natt (gp, GRAPH, S_IMAGE, (struct commonobj *)ao->baseobject)),
	      (imgvi = next_par_vi (&par)), n);
    ao->baseobject->image->curval.typtxt = strdup("");
    ao->baseobject->image->directval.typimage = &gp->imgcache->img;
  }
  else {
    ao->baseobject->image = NULL; 
    imgvi = next_par_vi (&par);
  }

  /* 
   * Ensure that we always use two points on textobjects and 
   * image-objects.
   * Try to use text/image-extent if extents are unspecified.
   */
  if ((obj[n] == AWETHORTEXTOBJ || obj[n] == AWETHORIMAGE) &&
       (npts[n] == 1 || (xvi->value.elem[n*AWETHORMAXPTS]
			 == xvi->value.elem[n*AWETHORMAXPTS+1]
			 && yvi->value.elem[n*AWETHORMAXPTS]
			 == yvi->value.elem[n*AWETHORMAXPTS+1]))) {
    if (obj[n] == AWETHORTEXTOBJ) {
      fextents (gp, otvi->value.telem[n], fnvi->value.telem[n], &a, &d, &w);
      xvi->value.elem[n*AWETHORMAXPTS+1] =
	xvi->value.elem[n*AWETHORMAXPTS]+w;
      yvi->value.elem[n*AWETHORMAXPTS+1] =
	yvi->value.elem[n*AWETHORMAXPTS]+a+d;
    } else if (obj[n] == AWETHORIMAGE) {
      if ((cip = getImage (gp, imgvi->value.telem[n]))
	  == &gp->imgcache->img){
	xvi->value.elem[n*AWETHORMAXPTS+1] =
	  xvi->value.elem[n*AWETHORMAXPTS]+88;
	yvi->value.elem[n*AWETHORMAXPTS+1] =
	  yvi->value.elem[n*AWETHORMAXPTS]+123;
      }
      else {
	xvi->value.elem[n*AWETHORMAXPTS+1] =
	  xvi->value.elem[n*AWETHORMAXPTS]+cip->width;
	yvi->value.elem[n*AWETHORMAXPTS+1] =
	  yvi->value.elem[n*AWETHORMAXPTS]+cip->height;
      }
    }
    ao->npoints = (short) npts[n] = 2;
    ao->pa = ao->pa ? realloc(ao->pa, 2*ao->npoints*(sizeof(long)))
      : calloc (2*ao->npoints, sizeof(long));
    ao->dpa = ao->dpa ? realloc(ao->dpa, 2*ao->npoints*(sizeof(char)))
      : calloc (2*ao->npoints, sizeof(char));
    if(new_obj){
      ao->dpa[0] = ADIRVAL;
      ao->dpa[1] = ADIRVAL;
    }
    /* Mark the second point for omission in the save, because it
     * is either redundant or even worse : plain wrong.
     */ 
    ao->dpa[2] = ao->dpa[3] = AOMIVAL;
    ao->pa[2] = ao->pa[3] = 0;
  }
  ao->baseobject->ptcount = ao->xtraobject->ptcount = npts[n];
  set_pl_opers (gp, ao->baseobject->pointlist, xvi, yvi, npvi, n);
  set_pl_opers (gp, ao->xtraobject->pointlist, xvi, yvi, npvi, n);

  ao->xtraobject->translation =
    natt (gp, INP, S_TRANSLATION, (struct commonobj *)ao->xtraobject);


  ao->xtraobject->translation->exprval.tl = 
    ntrans (gp, MouseDown, 1, CMaskShift, "actSBut", n);

  init_list(ao->xtraobject->translation->exprval.tl);
  
  place_next(ao->xtraobject->translation->exprval.tl, 
	     ntrans (gp, MouseDown, 1, 0, "actBut", n));

  /* Give life to the objects */
  ao->baseobject->live = TRUE;
  ao->xtraobject->live = TRUE;

}

long GenerateObject (struct cw_status *gp, struct param *par)
{
  struct awe_object *o;
  int arg = 0, n, i;
  char FAR *funcname = "generateObject";
  long *obj, size;
  struct param *opar;

  n = (int) getIntPar (gp, &par, ++arg, funcname);
  changed_par (gp, opar = par);
  getIntArrPar (gp, &par, 2, "generateObject", (void **)&obj, &size, 1);

  if (n == -1) {
    for (i = 0; i < ao_size; i++)
      makeObject (gp, i, par, obj, FALSE);
  }
  else {
    if (n == ao_size) o = add_awe_object (gp, ao_const2symb (obj[n]));
    makeObject (gp, n, par, obj, TRUE);
  }

  n = ao_size+1;

  /*
   * Every time before generateObject returns, it realloc all arrays
   * to fit with one more object
   * (Possibly do some bigger increments later...)
   * n is number of elements in the new arrays
   */
  par = opar;

  /*obj =*/rallarr (gp, par, n, sizeof(long), 1, FALSE);
  par = next (par);
  /*xattr =*/rallarr (gp, par, n, sizeof(long)*AWETHORMAXPTS, 2, FALSE);
  par->exp->left.array->which.vi->size[1] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[2] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[3] = 1;
  par = next (par);
  /*yattr =*/ rallarr (gp, par, n, sizeof(long)*AWETHORMAXPTS, 2, FALSE);
  par->exp->left.array->which.vi->size[1] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[2] = AWETHORMAXPTS;
  par->exp->left.array->which.vi->size[3] = 1;
  par = next (par);
  /*npoints =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*colattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*actattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*sbattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*levattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*fattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*texattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*dshattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*lwattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*saattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*eaattr =*/ rallarr (gp, par, n, sizeof(long), 1, FALSE); par = next (par);
  /*objname =*/ rallarr (gp, par, n, sizeof(char *), 1, FALSE); par = next (par);
  /*fnattr =*/ rallarr (gp, par, n, sizeof(char *), 1, FALSE); par = next (par);
  /*otattr =*/ rallarr (gp, par, n, sizeof(char *), 1, FALSE); par = next (par);
  /*imgattr =*/ rallarr (gp, par, n, sizeof(char *), 1, FALSE); par = next (par);
  /*trlattr =*/ rallarr (gp, par, n, sizeof(char *), 1, FALSE); par = next (par);
  checkPars (gp, par, arg, funcname);

  return 1;
}

void termObject (struct cw_status *gp, int n)
{
  struct awe_object *ao = ao_buf[n];


  if (is_first2 (ao->baseobject))
    gp->curlunit->cur_funcinst->fgraph = next2 (ao->baseobject);
  remove_at2 (ao->baseobject);
  killGraphObj (gp, ao->baseobject);

  if (is_first2 (ao->xtraobject))
    gp->curlunit->cur_funcinst->finp = next2 (ao->xtraobject);
  remove_at2 (ao->xtraobject);
  killInpObj (gp, ao->xtraobject);
}

long KillObject (struct cw_status *gp, struct param *par)
{
  int arg = 0, n, i;
  char FAR *funcname = "killObject";

  n = (int) getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

  if (n == -1) { /* Remove objects from screen before killing */
    for (i = 0; i < ao_size; i++)
      if (ao_buf[i]) { /* Assert that object not killed */
	ao_buf[i]->baseobject->active->directval.typint =
	  ao_buf[i]->baseobject->active->curval.typint = 0;
	ao_buf[i]->baseobject->status =
	  max(ao_buf[i]->baseobject->status, STALESBGEOM);
	ao_buf[i]->xtraobject->active->curval.typint =
	  ao_buf[i]->xtraobject->active->directval.typint = 0;
      }
    refresh (gp, FALSE);

    for (i = 0; i < ao_size; i++)
      termObject (gp, i);
  }
  else if (ao_buf[n]) {
    ao_buf[n]->baseobject->active->directval.typint =
      ao_buf[n]->baseobject->active->curval.typint = 0;
    ao_buf[n]->baseobject->status =
      max(ao_buf[n]->baseobject->status, STALESBGEOM);
    ao_buf[n]->xtraobject->active->curval.typint =
      ao_buf[n]->xtraobject->active->directval.typint = 0;
    refresh (gp, FALSE);
    
    termObject (gp, n);
    delete_awe_object (gp, n);
  }
  return 1;
}

long IsLegalDir (struct cw_status *gp, struct param *par)
{
  int arg = 0, res;
  char FAR *funcname = "isLegalDir", *filename;
#ifdef UNIX
  struct stat stbuf;
#endif
#ifdef WIN32
  WIN32_FIND_DATA FindData;
  HANDLE hFindFile;
#endif

  filename = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

#ifdef UNIX
  res =
    stat (filename, &stbuf) != -1 && (stbuf.st_mode & S_IFMT) == S_IFDIR;
#endif
#ifdef WIN32
	hFindFile = FindFirstFile (filename, &FindData);
	res = (hFindFile != INVALID_HANDLE_VALUE);
	res = res && ((FindData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0);
#endif
  free (filename);
  return res;
}

long GetDirectory (struct cw_status *gp, struct param *par)
{
  int arg = 0;
  long size, index = 0;
  char *funcname = "getDirectory", **names, *filename;
#ifdef UNIX
  DIR *dfd;
  struct dirent *dp;
#endif
#ifdef WIN32
  WIN32_FIND_DATA FindData;
  HANDLE hFind;
#endif
  struct param *apar;
  
  filename = getTextPar (gp, &par, ++arg, funcname);
  apar = par;
  getTextArrPar (gp, &par, ++arg, funcname, (void **)&names, &size, 1);
  /* size if void, will be realloced */
  checkPars (gp, par, arg, funcname);

/* Open directory */
#ifdef UNIX
  if ((dfd = opendir (filename)) == NULL) {
#endif
#ifdef WIN32
  if ((hFind = FindFirstFile (__TEXT ("*.*"), &FindData)) == INVALID_HANDLE_VALUE) {
#endif
	free (filename);
	return -1;
  }

/* Allocate some space for directory names */
  for (size = 1024;; size += 1024) {
    if ((names = (char **)realloc ((void *)names,
				   size*sizeof(char *))) == NULL) {
      userMsg(ErrNOMOREMEM);
      l_exit (NOT_OK);
    }
/* Fill up space with dirnames as long as they exist */
#ifdef UNIX
    for (; index < size && (dp = readdir (dfd)) != NULL; index++)
      names[index] = strdup (dp->d_name ? dp->d_name : "");
    if (dp == NULL) break;
#endif
#ifdef WIN32
	while(index < size) {
		names[index++] = strdup (FindData.cFileName ? FindData.cFileName : "");
		if (!FindNextFile (hFind, &FindData))
			goto DIRENDWIN;
	}
#endif
  }
/* Close directory */
#ifdef UNIX
  closedir (dfd);
#endif
#ifdef WIN32
DIRENDWIN:
  if (hFind != INVALID_HANDLE_VALUE) FindClose (hFind);
#endif
/* Reallocate to exact size */
  if ((names = (char **)realloc ((void *)names,
				 index*sizeof(char *))) == NULL) {
    userMsg(ErrNOMOREMEM);
    l_exit(NOT_OK);
  }
  apar->exp->left.array->which.vi->size[0] = index;
  apar->exp->left.array->which.vi->value.telem = names;
  changed_par (gp, apar);

  return index;
}

long ArraySize (struct cw_status *gp, struct param *par)
{
  int arg = 0;
  long index;
  char *funcname = "arraySize";
  struct param *apar = par;
  struct vinst FAR *vi;
  
  par = (struct param *)next (par);      /* Any type OK */
  index = getIntPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

#ifdef DEBUG
  if(apar->exp->left.array->type == DO_VDECL){
    fprintf(stderr,"Internal error: Illegal par->exp->value in par_array_value.\n");
  }
#endif
  vi = apar->exp->left.array->which.vi;
  
  if(vi->parent->ndim < index){
    paramError(gp, apar, ErrNOTONEDIM);
    return 0;
  }
  return vi->size[index-1];
}

long ChangeDir (struct cw_status *gp, struct param *par)
{
  int arg = 0;
  long res;
  char *funcname = "changeDir", *dir;
  
  dir = getTextPar (gp, &par, ++arg, funcname);
  checkPars (gp, par, arg, funcname);

#ifdef UNIX
  res = chdir (dir);
#endif
#ifdef WIN32
  res = SetCurrentDirectory (dir);
#endif
  free (dir);

  return res;
}

long GetFunctions (struct cw_status *gp, struct param *par)
{
  int arg = 0;
  long size;
  char *funcname = "getFunctions", **funcs, **old;
  struct param *apar;

  changed_par (gp, apar = par);
  getTextArrPar (gp, &par, ++arg, funcname, (void FAR *FAR *)&funcs, &size, 1);
  checkPars (gp, par, arg, funcname);

  old = funcs;
  if ((size = get_awe_blocks (gp, &funcs)) > 0) {
    free (old);
    apar->exp->left.array->which.vi->size[0] = size;
    apar->exp->left.array->which.vi->value.telem = funcs;
  }
  else funcs = old;

  return size;
}

long FreeFile (struct cw_status *gp, struct param *par)
{
  checkPars (gp, par, 0, "freeFile");

  awethor_free(gp);
  return 1;
}

/*
#define ANCHORBEGIN "href"

long LinkBrowse (struct param *par)
{
  int arg = 0, fd, r, total = 0;
  long size, index = 0, tsiz;
  char *funcname = "linkBrowse", **names, *filename, *url, *mtype,
    *rms, *rme, *txt, *tbuf, *mess;
  struct param *apar;
  
  url = getTextPar (&par, ++arg, funcname);
  apar = par;
  getTextArrPar (&par, ++arg, funcname, (void **)&names, &size, 1);

  checkPars (par, arg, funcname);

  filename = ResolveURL(url);
  if(filename == NULL) {
    free (url);
    sprintf(mess, ErrRESOLVURL, url);
    errorMsg(0, mess);
    return 0;
  }

  fd = open (filename, O_RDONLY); free (filename);
  if ((txt = tbuf = (char *)malloc (1024)) == NULL) {
    errorMsg(2, ErrNOMOREMEM);
    exit (NOT_OK);
  }
  for (tsiz = 1024;; tsiz += 1024) {
    r = read (fd, tbuf+total, 1024);
    total += r;
    if (r < 1024) break;
    if ((tbuf = (char *)realloc ((void *)txt, tsiz)) == NULL) {
      errorMsg(2, ErrNOMOREMEM);
      exit (NOT_OK);
    }
  }
  close (fd);

  for (size = 1024;; size += 1024) {
    if ((names = (char **)realloc ((void *)names,
				   size*sizeof(char *))) == NULL) {
      errorMsg(2, ErrNOMOREMEM);
      exit (NOT_OK);
    }

    for (; index < size; index++)
      while (1)
	if (strncmp (txt++, ANCHORBEGIN, strlen (ANCHORBEGIN)) == 0) {

	  for (;*txt != '='; txt++) if (*txt == '0') goto gotEOT;
	  for (;*txt != '\"'; txt++) if (*txt == '\0') goto gotEOT;

	  for (rms = rme = txt; *rme != '\"'; rme++)
	    if (*rme == '\0') goto gotEOT;

	  *(rme-1) = '\0';
	  names[index] = strdup (rms);
	  if (*(txt = rme) == '\0') goto gotEOT;
	}
  }
gotEOT:
  free (tbuf);

  if ((names = (char **)realloc ((void *)names,
				 index*sizeof(char *))) == NULL) {
    errorMsg(2, ErrNOMOREMEM);
    exit (NOT_OK);
  }
  apar->exp->left.array->which.vi->size[0] = index;
  apar->exp->left.array->which.vi->value.telem = names;
  changed_par (apar);

  return index;
}
*/
