
/* Copyright (c) 1995 by Computers and Learning A/S (candle@sn.no). 
 * See Copyright.txt for details.
 *
 * Authors: Svein Arne Johansen (sveinj@ifi.uio.no), 
 *	    Gunnar Rnning (gunnarr@ifi.uio.no)
 */

#include <stdio.h>
#include <stdlib.h>
#include <jpeglib.h>
#include <values.h>
#include <assert.h>
#include <X11/Intrinsic.h>
#include <X11/Xcms.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>
#include <X11/Shell.h>
#include "guilib/AweP.h"

#include "candle.h"
#include "error.h"
#include "nodes.h"
#include "lex.h"
#include "pbmplus.h"

#include "sysproto.h"
#include "protos/fast_lis.h"
#include "protos/canutil.h"
#include "protos/update.h"
#include "protos/gif.h"
#include "protos/memory.h"

/*
 * In plugin: No global state variables allowed.
 * Read-only variables should be allright, though
 */


extern unsigned char defpicdata[];

/* The palette in CIEXYZ-format */
float CIEpal[MAXCOLS][3] = {
  {0.000000, 0.000000, 0.000000},
  {0.016367, 0.007391, 0.081834},
  {0.097953, 0.044237, 0.489764},
  {0.248346, 0.112156, 1.241729},
  {0.002083, 0.004427, 0.000930},
  {0.018450, 0.011818, 0.082764},
  {0.100036, 0.048664, 0.490694},
  {0.250429, 0.116583, 1.242659},
  {0.014110, 0.029984, 0.006299},
  {0.030477, 0.037375, 0.088133},
  {0.112063, 0.074221, 0.496063},
  {0.262456, 0.142140, 1.248028},
  {0.041460, 0.088102, 0.018509},
  {0.057826, 0.095493, 0.100342},
  {0.139412, 0.132338, 0.508273},
  {0.289805, 0.200258, 1.260238},
  {0.087044, 0.184968, 0.038859},
  {0.103411, 0.192360, 0.120693},
  {0.184997, 0.229205, 0.528623},
  {0.335390, 0.297124, 1.280588},
  {0.150962, 0.320795, 0.067394},
  {0.167329, 0.328186, 0.149227},
  {0.248915, 0.365031, 0.557158},
  {0.399308, 0.432951, 1.309123},
  {0.230500, 0.489812, 0.102902},
  {0.246866, 0.497203, 0.184735},
  {0.328452, 0.534048, 0.592666},
  {0.478845, 0.601968, 1.344631},
  {0.320257, 0.680546, 0.142972},
  {0.336624, 0.687938, 0.224806},
  {0.418210, 0.724783, 0.632736},
  {0.568603, 0.792703, 1.384701},
  {0.003007, 0.001636, 0.000168},
  {0.019374, 0.009027, 0.082002},
  {0.100960, 0.045873, 0.489932},
  {0.251353, 0.113792, 1.241897},
  {0.005090, 0.006063, 0.001098},
  {0.021457, 0.013454, 0.082932},
  {0.103043, 0.050299, 0.490862},
  {0.253436, 0.118219, 1.242827},
  {0.017117, 0.031620, 0.006468},
  {0.033484, 0.039011, 0.088301},
  {0.115070, 0.075857, 0.496232},
  {0.265463, 0.143776, 1.248196},
  {0.044467, 0.089737, 0.018677},
  {0.060833, 0.097129, 0.100511},
  {0.142420, 0.133974, 0.508441},
  {0.292813, 0.201894, 1.260406},
  {0.090051, 0.186604, 0.039027},
  {0.106418, 0.193996, 0.120861},
  {0.188004, 0.230841, 0.528791},
  {0.338397, 0.298760, 1.280756},
  {0.153969, 0.322430, 0.067562},
  {0.170336, 0.329822, 0.149396},
  {0.251922, 0.366667, 0.557326},
  {0.402315, 0.434587, 1.309291},
  {0.233507, 0.491447, 0.103070},
  {0.249873, 0.498839, 0.184904},
  {0.331460, 0.535684, 0.592834},
  {0.481853, 0.603604, 1.344799},
  {0.323264, 0.682182, 0.143140},
  {0.339631, 0.689574, 0.224974},
  {0.421217, 0.726419, 0.632904},
  {0.571610, 0.794338, 1.384869},
  {0.015803, 0.008597, 0.000885},
  {0.032170, 0.015988, 0.082719},
  {0.113756, 0.052834, 0.490649},
  {0.264149, 0.120753, 1.242614},
  {0.017886, 0.013024, 0.001815},
  {0.034253, 0.020415, 0.083649},
  {0.115839, 0.057260, 0.491579},
  {0.266232, 0.125180, 1.243544},
  {0.029913, 0.038581, 0.007184},
  {0.046280, 0.045972, 0.089018},
  {0.127866, 0.082818, 0.496948},
  {0.278259, 0.150737, 1.248913},
  {0.057263, 0.096698, 0.019394},
  {0.073629, 0.104090, 0.101227},
  {0.155215, 0.140935, 0.509158},
  {0.305608, 0.208855, 1.261123},
  {0.102847, 0.193565, 0.039744},
  {0.119214, 0.200957, 0.121578},
  {0.200800, 0.237802, 0.529508},
  {0.351193, 0.305721, 1.281473},
  {0.166765, 0.329391, 0.068279},
  {0.183132, 0.336783, 0.150112},
  {0.264718, 0.373628, 0.558043},
  {0.415111, 0.441548, 1.310008},
  {0.246303, 0.498408, 0.103787},
  {0.262669, 0.505800, 0.185620},
  {0.344255, 0.542645, 0.593551},
  {0.494648, 0.610565, 1.345515},
  {0.336060, 0.689143, 0.143857},
  {0.352427, 0.696535, 0.225691},
  {0.434013, 0.733380, 0.633621},
  {0.584406, 0.801299, 1.385586},
  {0.044250, 0.024072, 0.002478},
  {0.060617, 0.031464, 0.084312},
  {0.142203, 0.068309, 0.492242},
  {0.292596, 0.136228, 1.244207},
  {0.046334, 0.028499, 0.003408},
  {0.062700, 0.035891, 0.085242},
  {0.144286, 0.072736, 0.493172},
  {0.294679, 0.140655, 1.245137},
  {0.058361, 0.054056, 0.008777},
  {0.074727, 0.061448, 0.090611},
  {0.156313, 0.098293, 0.498541},
  {0.306706, 0.166212, 1.250506},
  {0.085710, 0.112174, 0.020987},
  {0.102077, 0.119565, 0.102820},
  {0.183663, 0.156411, 0.510751},
  {0.334056, 0.224330, 1.262716},
  {0.131294, 0.209041, 0.041337},
  {0.147661, 0.216432, 0.123171},
  {0.229247, 0.253277, 0.531101},
  {0.379640, 0.321197, 1.283066},
  {0.195213, 0.344867, 0.069872},
  {0.211579, 0.352258, 0.151706},
  {0.293165, 0.389104, 0.559636},
  {0.443558, 0.457023, 1.311601},
  {0.274750, 0.513884, 0.105380},
  {0.291117, 0.521275, 0.187213},
  {0.372703, 0.558121, 0.595144},
  {0.523096, 0.626040, 1.347109},
  {0.364508, 0.704619, 0.145450},
  {0.380874, 0.712010, 0.227284},
  {0.462460, 0.748855, 0.635214},
  {0.612853, 0.816775, 1.387179},
  {0.092718, 0.050439, 0.005192},
  {0.109085, 0.057830, 0.087026},
  {0.190671, 0.094676, 0.494956},
  {0.341064, 0.162595, 1.246921},
  {0.094802, 0.054866, 0.006122},
  {0.111168, 0.062257, 0.087956},
  {0.192754, 0.099102, 0.495886},
  {0.343147, 0.167022, 1.247851},
  {0.106828, 0.080423, 0.011491},
  {0.123195, 0.087814, 0.093325},
  {0.204781, 0.124660, 0.501255},
  {0.355174, 0.192579, 1.253220},
  {0.134178, 0.138540, 0.023701},
  {0.150545, 0.145932, 0.105535},
  {0.232131, 0.182777, 0.513465},
  {0.382524, 0.250697, 1.265430},
  {0.179762, 0.235407, 0.044051},
  {0.196129, 0.242799, 0.125885},
  {0.277715, 0.279644, 0.533815},
  {0.428108, 0.347563, 1.285780},
  {0.243681, 0.371233, 0.072586},
  {0.260047, 0.378625, 0.154420},
  {0.341633, 0.415470, 0.562350},
  {0.492026, 0.483390, 1.314315},
  {0.323218, 0.540250, 0.108094},
  {0.339585, 0.547642, 0.189927},
  {0.421171, 0.584487, 0.597858},
  {0.571564, 0.652407, 1.349823},
  {0.412976, 0.730985, 0.148164},
  {0.429342, 0.738377, 0.229998},
  {0.510928, 0.775222, 0.637928},
  {0.661321, 0.843141, 1.389893},
  {0.164187, 0.089318, 0.009194},
  {0.180554, 0.096709, 0.091028},
  {0.262140, 0.133555, 0.498958},
  {0.412533, 0.201474, 1.250923},
  {0.166270, 0.093745, 0.010124},
  {0.182637, 0.101136, 0.091958},
  {0.264223, 0.137981, 0.499888},
  {0.414616, 0.205901, 1.251853},
  {0.178297, 0.119302, 0.015494},
  {0.194664, 0.126693, 0.097327},
  {0.276250, 0.163539, 0.505258},
  {0.426643, 0.231458, 1.257223},
  {0.205647, 0.177419, 0.027703},
  {0.222014, 0.184811, 0.109537},
  {0.303600, 0.221656, 0.517467},
  {0.453993, 0.289576, 1.269432},
  {0.251231, 0.274286, 0.048053},
  {0.267598, 0.281678, 0.129887},
  {0.349184, 0.318523, 0.537817},
  {0.499577, 0.386442, 1.289782},
  {0.315149, 0.410112, 0.076588},
  {0.331516, 0.417504, 0.158422},
  {0.413102, 0.454349, 0.566352},
  {0.563495, 0.522269, 1.318317},
  {0.394687, 0.579129, 0.112096},
  {0.411054, 0.586521, 0.193930},
  {0.492640, 0.623366, 0.601860},
  {0.643033, 0.691286, 1.353825},
  {0.484444, 0.769864, 0.152166},
  {0.500811, 0.777256, 0.234000},
  {0.582397, 0.814101, 0.641930},
  {0.732790, 0.882020, 1.393895},
  {0.260208, 0.141553, 0.014572},
  {0.276575, 0.148945, 0.096405},
  {0.358161, 0.185790, 0.504336},
  {0.508554, 0.253710, 1.256301},
  {0.262292, 0.145980, 0.015502},
  {0.278658, 0.153372, 0.097335},
  {0.360244, 0.190217, 0.505266},
  {0.510637, 0.258136, 1.257231},
  {0.274319, 0.171537, 0.020871},
  {0.290685, 0.178929, 0.102704},
  {0.372271, 0.215774, 0.510635},
  {0.522664, 0.283694, 1.262600},
  {0.301668, 0.229655, 0.033080},
  {0.318035, 0.237046, 0.114914},
  {0.399621, 0.273892, 0.522844},
  {0.550014, 0.341811, 1.274809},
  {0.347252, 0.326522, 0.053431},
  {0.363619, 0.333913, 0.135264},
  {0.445205, 0.370758, 0.543195},
  {0.595598, 0.438678, 1.295159},
  {0.411171, 0.462348, 0.081965},
  {0.427537, 0.469739, 0.163799},
  {0.509123, 0.506585, 0.571729},
  {0.659516, 0.574504, 1.323694},
  {0.490708, 0.631365, 0.117473},
  {0.507075, 0.638756, 0.199307},
  {0.588661, 0.675602, 0.607237},
  {0.739054, 0.743521, 1.359202},
  {0.580466, 0.822100, 0.157544},
  {0.596832, 0.829491, 0.239377},
  {0.678418, 0.866337, 0.647308},
  {0.828811, 0.934256, 1.399273},
  {0.381061, 0.207297, 0.021339},
  {0.397428, 0.214689, 0.103173},
  {0.479014, 0.251534, 0.511103},
  {0.629407, 0.319454, 1.263068},
  {0.383145, 0.211724, 0.022269},
  {0.399511, 0.219116, 0.104103},
  {0.481098, 0.255961, 0.512033},
  {0.631490, 0.323880, 1.263998},
  {0.395172, 0.237281, 0.027639},
  {0.411538, 0.244673, 0.109472},
  {0.493124, 0.281518, 0.517403},
  {0.643517, 0.349438, 1.269368},
  {0.422521, 0.295399, 0.039848},
  {0.438888, 0.302790, 0.121682},
  {0.520474, 0.339636, 0.529612},
  {0.670867, 0.407555, 1.281577},
  {0.468105, 0.392266, 0.060198},
  {0.484472, 0.399657, 0.142032},
  {0.566058, 0.436502, 0.549962},
  {0.716451, 0.504422, 1.301927},
  {0.532024, 0.528092, 0.088733},
  {0.548390, 0.535483, 0.170567},
  {0.629976, 0.572329, 0.578497},
  {0.780369, 0.640248, 1.330462},
  {0.611561, 0.697109, 0.124241},
  {0.627928, 0.704500, 0.206075},
  {0.709514, 0.741346, 0.614005},
  {0.859907, 0.809265, 1.365970},
  {0.701319, 0.887844, 0.164311},
  {0.717685, 0.895235, 0.246145},
  {0.799271, 0.932081, 0.654075},
  {0.949664, 1.000000, 1.406040}
};

/* The rasters used to emulate colors on black-and-white displays */

static char rast0_bits[] = {
   0x00, 0x00, 0x00, 0x00};
static char rast1_bits[] = {
   0x00, 0x02, 0x00, 0x00};
static char rast2_bits[] = {
   0x00, 0x02, 0x00, 0x08};
static char rast3_bits[] = {
   0x00, 0x0a, 0x00, 0x08};
static char rast4_bits[] = {
   0x00, 0x0a, 0x00, 0x0a};
static char rast5_bits[] = {
   0x04, 0x0a, 0x00, 0x0a};
static char rast6_bits[] = {
   0x04, 0x0a, 0x01, 0x0a};
static char rast7_bits[] = {
   0x05, 0x0a, 0x01, 0x0a};
static char rast8_bits[] = {
   0x05, 0x0a, 0x05, 0x0a};
static char rast9_bits[] = {
   0x0d, 0x0a, 0x05, 0x0a};
static char rast10_bits[] = {
   0x0d, 0x0a, 0x07, 0x0a};
static char rast11_bits[] = {
   0x0f, 0x0a, 0x07, 0x0a};
static char rast12_bits[] = {
   0x0f, 0x0a, 0x0f, 0x0a};
static char rast13_bits[] = {
   0x0f, 0x0a, 0x0f, 0x0e};
static char rast14_bits[] = {
   0x0f, 0x0b, 0x0f, 0x0e};
static char rast15_bits[] = {
   0x0f, 0x0b, 0x0f, 0x0f};
static char rast16_bits[] = {
   0x0f, 0x0f, 0x0f, 0x0f};

char *rastptr[MAXRAST] = {
  rast0_bits, rast1_bits, rast2_bits, rast3_bits,
  rast4_bits, rast5_bits, rast6_bits, rast7_bits,
  rast8_bits, rast9_bits, rast10_bits, rast11_bits,
  rast12_bits, rast13_bits, rast14_bits, rast15_bits,
  rast16_bits
};

/*
 * Opens new virtual colormap and tries not to mess up
 * the rest of the screen too much
 */

void installPalette (struct cw_status *gp, Widget widget, Colormap virt_cmap)
{
  register float bestbadness, colbadness;
  int i, j, bestind, n;
  float  cx, cy, cz;
  XcmsColor def_col[MAXCOLS], exact_def[MAXCOLS];
  char usedCol[MAXCOLS];
  unsigned char *src = defpicdata, *dst;
  unsigned long pixels[MAXCOLS];
  
  /* Fill default-colors with pixel-values */
  for(i=0; i<MAXCOLS; i++) {
    awe_system.color[i].color.pixel = def_col[i].pixel = exact_def[i].pixel = i;
    exact_def[i].format = XcmsCIEXYZFormat;
    usedCol[i] = FALSE;
  }
  /* Fill def_col-array with color values of default colormap */
  if (gp->system.depth == 8) {
    /* Perfect or adapted, get current colors in order to mess up
       as little as possible */
    XcmsQueryColors (XtDisplay (widget),
		     DefaultColormapOfScreen (XtScreen (widget)),
		     def_col, MAXCOLS, XcmsCIEXYZFormat);
    
    if (awe_system.colorModel == PSEUDO8ADAPTED) {
      /* Allocate as many as possible into pixels */
      for (n = 0; n < MAXCOLS
	     && XAllocColorCells (XtDisplay (widget),
				  DefaultColormapOfScreen (XtScreen (widget)),
				  FALSE, NULL, 0, &pixels[n], 1); n++);
      if (n == MAXCOLS) { /* Got all */
	awe_system.colorModel = PSEUDO8PERFECT;
	goto P8P;
      }
      /* Find the allocated pixels */
      for (i = 0; i < MAXCOLS; i++) {
	for (j = 0; j < n && pixels[j] != i; j++);
	if (j < n) continue;
	  /* i is an allocated pixel */
	
	cx = def_col[i].spec.CIEXYZ.X;
	cy = def_col[i].spec.CIEXYZ.Y;
	cz = def_col[i].spec.CIEXYZ.Z;
	/* Find the best fitting color in the CIE-palette for this color */
	bestbadness = MAXFLOAT;
	for (j = 0; j < MAXCOLS; j++)
	    if (!usedCol[j]
		&& (colbadness = (CIEpal[j][0]-cx)*(CIEpal[j][0]-cx)
		    +(CIEpal[j][1]-cy)*(CIEpal[j][1]-cy)
		    +(CIEpal[j][2]-cz)*(CIEpal[j][2]-cz))
	    < bestbadness) {
	      bestbadness = colbadness; bestind = j;
	    }
	usedCol[bestind] = TRUE;
	
	awe_system.color[i].color.spec.CIEXYZ.X = cx;
	  awe_system.color[i].color.spec.CIEXYZ.Y = cy;
	  awe_system.color[i].color.spec.CIEXYZ.Z = cz;
	  awe_system.color[i].color.pixel = i;
      }
      /* Matched the used colors to the best ones in CIEpal,
	 now write the rest to the previously allocated entries */
      j = 0;
      for (i = 0; i < MAXCOLS; i++) {
	if (!usedCol[i]) {
	  assert ((usedCol[i] = TRUE));
	  awe_system.color[pixels[j]].color.spec.CIEXYZ.X = CIEpal[i][0];
	  awe_system.color[pixels[j]].color.spec.CIEXYZ.Y = CIEpal[i][1];
	  awe_system.color[pixels[j]].color.spec.CIEXYZ.Z = CIEpal[i][2];
	  awe_system.color[pixels[j]].color.format = XcmsCIEXYZFormat;
	  /* And, just in case */
	  awe_system.color[pixels[j]].color.pixel = pixels[j];

	  XcmsStoreColor (XtDisplay (widget), 
			  DefaultColormapOfScreen (XtScreen (widget)),
			  &awe_system.color[pixels[j]].color);
	  j++; /* Index of next candle-allocated pixel */
	}
      }
      /* Query again in order to get the result, this is a little
	 redundant, but for now, it's more important that it works.
	 The reason for doing this is to remove X calls from colorOfRGB () */
      XcmsQueryColors (XtDisplay (widget),
		       DefaultColormapOfScreen (XtScreen (widget)),
		       def_col, MAXCOLS, XcmsRGBFormat);
      /* Now copy the result to awe_system.color, which is actually used */
      for (i = 0; i < MAXCOLS; i++) {

	assert (usedCol[i]);
	awe_system.color[i].color = def_col[i];
	/*	printf ("%4X : %5X%5X%5X - %d\n", i,
		awe_system.color[i].color.spec.RGB.red,
		awe_system.color[i].color.spec.RGB.green,
		awe_system.color[i].color.spec.RGB.blue,
		awe_system.color[i].color.format); */
	assert (awe_system.color[i].color.pixel == i);
      }
    }
    else if (awe_system.colorModel == PSEUDO8PERFECT) {
    P8P:
      for (i = 0; i < MAXCOLS; i++) {
	cx = def_col[i].spec.CIEXYZ.X;
	cy = def_col[i].spec.CIEXYZ.Y;
	cz = def_col[i].spec.CIEXYZ.Z;
	/* Find the best fitting color in the CIE-palette for this color */
	bestbadness = MAXFLOAT;
	for (j = 0; j < MAXCOLS; j++)
	  if (!usedCol[j]
	      && (colbadness = (CIEpal[j][0]-cx)*(CIEpal[j][0]-cx)
		  +(CIEpal[j][1]-cy)*(CIEpal[j][1]-cy)
		  +(CIEpal[j][2]-cz)*(CIEpal[j][2]-cz))
	  < bestbadness) {
	    bestbadness = colbadness; bestind = j;
	  }
	usedCol[bestind] = TRUE;
	
	awe_system.color[i].color.spec.CIEXYZ.X =
	  exact_def[i].spec.CIEXYZ.X = CIEpal[bestind][0];
	awe_system.color[i].color.spec.CIEXYZ.Y =
	  exact_def[i].spec.CIEXYZ.Y = CIEpal[bestind][1];
	awe_system.color[i].color.spec.CIEXYZ.Z =
	  exact_def[i].spec.CIEXYZ.Z = CIEpal[bestind][2];

	awe_system.color[bestind].color.pixel = i;
      }
      XcmsStoreColors (XtDisplay (widget), virt_cmap, exact_def,
		       MAXCOLS, NULL);
    }
    /* Initialize sortindex so that they can be used in bin-search
    for (i = 0; i < MAXCOLS; i++) {
      bestind = -1;
      for (j = 0; j < MAXCOLS; j++)
	if (!usedCol[j]
	    && (bestind == -1
		|| (awe_system.color[j].color.spec.CIEXYZ.X
		    < awe_system.color[bestind].color.spec.CIEXYZ.X)
		|| (awe_system.color[j].color.spec.CIEXYZ.X
		    == awe_system.color[bestind].color.spec.CIEXYZ.X
		    && awe_system.color[j].color.spec.CIEXYZ.Y
		    < awe_system.color[bestind].color.spec.CIEXYZ.Y)
		|| (awe_system.color[j].color.spec.CIEXYZ.X
		    == awe_system.color[bestind].color.spec.CIEXYZ.X
		    && awe_system.color[j].color.spec.CIEXYZ.Y
		    == awe_system.color[bestind].color.spec.CIEXYZ.Y
		    && awe_system.color[j].color.spec.CIEXYZ.Z
		    < awe_system.color[bestind].color.spec.CIEXYZ.Z)))
	  bestind = j;
      usedCol[bestind] = TRUE;
      awe_system.color[i].sortindex = bestind;
    } */
  }
  
  if (!gp->defpic
      && !(gp->defpic =
	   (unsigned char *)CalCalloc (1, DEFPICWIDTH*DEFPICHEIGHT))) {
    fprintf (stderr, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  dst = gp->defpic;
  for (i = 0; i < DEFPICHEIGHT; i++)
    for (j = 0; j < DEFPICWIDTH; j++) {
      n = *(src++);
      *(dst++) =
	colorOfRGB (((n&0xE0)<<16)|((n&0x1C)<<11)|((n&0x03)<<6))&0xFF;
    }
}

void makeColors (struct cw_status *gp)
{
  int r, g, b, i;

  gp->system.colormap =
    DefaultColormapOfScreen (XtScreen (gp->system.topLevel));

  switch (awe_system.colorModel) {
  case GRAY1:
/*
 * Black-and-white :
 * awe_system.color is filled with 17 values used as an index into rastptr
 */
    i = 0;
    for (r = 0; r < 8; r++)
      for (g = 0; g < 8; g++)
	for (b = 0; b < 4; b++)
	  awe_system.color[i++].color.pixel = (int)
	    ((((float)r*YIQ11/7.0+(float)g*YIQ12/7.0+(float)b*YIQ13/3.0)
	      *MAXRAST)+.5);
    awe_system.color[MAXCOLS-1].color.pixel = MAXRAST-1;
    break;
  case GRAY4:
/* 16 color Grayscale */
    i = 0;
    for (r = 0; r < 8; r++)
      for (g = 0; g < 8; g++)
	for (b = 0; b < 4; b++)
	  awe_system.color[i++].color.pixel = (int)
	    ((((float)r*YIQ11/7.0+(float)g*YIQ12/7.0+(float)b*YIQ13/3.0)*16)+.5);
    awe_system.color[MAXCOLS-1].color.pixel = 15;
    break;
  case PSEUDO8PERFECT:
/* 256 colors pseudocolor */
    gp->system.colormap =
      XCreateColormap (XtDisplay (gp->system.topLevel),
		       RootWindowOfScreen (XtScreen (gp->system.topLevel)),
		       gp->system.visual, AllocAll);
    /* Fall through */
  case PSEUDO8ADAPTED:
    installPalette (gp, gp->system.topLevel,gp->system.colormap);
    break;
  }
}

/* Just to improve performance, here's 2^3, 2^2, 2^1, 2^0 */
static int power2bits[] = {8, 4, 2, 1};

struct imgnode *initDefImg (struct cw_status *gp)
{
  struct imgnode *in;
  char *maskarray, *ip;
  int i, j, inv, transind = 0, value;
  XImage *image, *mask;
  unsigned long palette24[MAXCOLS]; char used24[MAXCOLS];

  if (!(in = (struct imgnode *)CalCalloc (1, sizeof (struct imgnode)))) {
    fprintf (stderr, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  in->key = strdup ("");
  in->img.width = DEFPICWIDTH;
  in->img.height = DEFPICHEIGHT;
  in->img.color = colorOfRGB (0x000000);
  in->lunit = NULL;
  in->img.in_cache = TRUE;

/* For depth 8, this was fixed in installPalette */
  if (gp->system.depth == 8)
    transind = colorOfRGB (0x000000);
  else {
    if ((gp->defpic =
	 (char *)CalCalloc (1, gp->system.depth == 24 ?
			    in->img.width*in->img.height*4
			    : gp->system.depth == 16 ?
			    in->img.width*in->img.height*2
			    : in->img.width*in->img.height)) == NULL) {
      fprintf (stderr, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }
  }

  if ((maskarray =
       (char *)CalCalloc (1, in->img.width*in->img.height)) == NULL) {
    fprintf (stderr, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }

  mask = XCreateImage (gp->system.dpy, gp->system.visual,
		       1, ZPixmap, 0, maskarray,
		       in->img.width, in->img.height, 8, 0);

  for (j = 0; j < in->img.height; j++)
    for (i = 0; i < in->img.width; i++)
      XPutPixel (mask, i, j,
		 (unsigned long)(*(defpicdata+j*in->img.width+i) != transind));

  imgToServer (gp, &in->img, mask, FALSE);

  image = XCreateImage (gp->system.dpy, gp->system.visual,
			gp->system.depth, ZPixmap, 0, gp->defpic,
			in->img.width, in->img.height, 8, 0);

  switch (gp->system.depth) {
  case 1:
/*
 * For depth 8, this is done in installColormap because def. colormap
 * may change.
 * For 1 and 4 we have to calculate the real mask first, thus it's done here
 */

/* I like to view it like this:
   If BlackPixelOfScreen == 1, the display is inverted */

    /* Must be rewritten to work with pseudo8 colormixing */

    inv = BlackPixelOfScreen (XtScreen (gp->system.topLevel)); 
    for (j = 0; j < in->img.height; j++)
      for (i = 0; i < in->img.width; i++)
	XPutPixel (image, i, j, (unsigned long)
		   (((rastptr[colorOfRGB ((int)*(defpicdata
						     +j*in->img.width+i))][j%4]
		      &power2bits[i%4]) != 0)^inv));
    break;
  case 4:
    ip = gp->defpic;
    for (j = 0; j < in->img.height; j++)
      for (i = 0; i < in->img.width; i++)
	XPutPixel (image, i, j, (unsigned long)
		   colorOfRGB ((int)*(defpicdata+j*in->img.width+i)));
    break;
  case 24:
    for (i = 0; i < MAXCOLS; i++) used24[i] = FALSE;
    for (j = 0; j < in->img.height; j++)
      for (i = 0; i < in->img.width; i++) {
	value = *(defpicdata+j*in->img.width+i);
	if (!used24[value]) {
	  palette24[value] = (unsigned long)/*colorOfRGB*/
	    (((value&0xE0)<<16)|((value&0x1C)<<11)|((value&0x03)<<6));
	  used24[value] = TRUE;
	}
	XPutPixel (image, i, j, palette24[value]);
      }
    break;
  }
  imgToServer (gp, &in->img, image, TRUE);

  return in;
}

long displayPixel (long RGB)
{
  long p;


  if (awe_system.depth != 1) {
    p = colorOfRGB (RGB);
    if (awe_system.depth != 24) p &= 0xFF;
    return p;
  }
  else
    return ((float)((RGB>>(2*8+5))&0x7)*YIQ11/7.0
	    +(float)((RGB>>(8+5))&0x7)*YIQ12/7.0
	    +(float)((RGB>>6)&0x3)*YIQ13/3.0 >= .5) ?
      WhitePixelOfScreen (XtScreen (awe_system.topShell))
      : BlackPixelOfScreen (XtScreen (awe_system.topShell));
}

/* Must be rewritten with color-mixing! */

long defColor (struct cw_status *gp, long color)
{
  int R, G, B, bits;

  for (bits = 0;
       bits < 256 && awe_system.color[bits].color.pixel != color;
       bits++);
  if (bits >= 256) {
    fprintf(stderr, ErrCOLNIPAL);
    c_exit (gp, NOT_OK);
  }
  R = ((((bits >> (2+3))&0x7) << 5)+0x80)&0xFF; /* Wrong! Only for now */
  G = ((((bits >> 2) & 0x7) << 5)+0x80)&0xFF;
  B = (((bits & 0x3) << 6)+0x80)&0xFF;

  return R*256*256+G*256+B;
}

void changeColEntry (struct cw_status *gp, int i, COLORREF v)
{
/*
  XColor col;

  if (i >= 256) {
    fprintf (stderr, WarnIND256);
    return;
  }
  if (gp->system.depth != 8) {
    fprintf (stderr,
	     "Warning: This display does not yet support colortable operations\n");
    return;
  }
  col.pixel = awe_system.color[i];
  col.red = (v >> 8) & 0xFF00;
  col.green = v & 0xFF00;
  col.blue = (v << 8) & 0xFF00;
  col.flags = DoRed | DoGreen | DoBlue;

  XStoreColor (gp->system.dpy, gp->system.colormap, &col);
*/
}

/*
 * Draws object into win with offset winx/yoffset and clips at subwin
 * iff statlevel is non-NULL, the static objects are set up and statlevel
 * points to the level being set up. This function behaves differently on
 * different displays.
 */

void drawGraph (struct cw_status *gp,
		struct graphobj *object, struct levlist *statlevel,
		CalRect subwin, int winxoffs, int winyoffs)
{
  int i, lw, fill, sa, ea, type, minx, maxx, miny, maxy, severity = Complex,
    colval;
  int imh, imw;
  struct levlist *level;
  XPoint *point;
  Pixmap mask = None;
  CalRect defrect;
  CalText *textnode;
  struct txtlist *tl;
  Pixmap win;
  long *dval = object->pointlist->directval.ptarr;

  win = statlevel ?
    gp->curlunit->picstatic->picture : gp->curlunit->win->picture;

  if (subwin.width == 0 || subwin.height == 0 || !win)
    return;
  assert (subwin.width > 0 && subwin.height > 0);

/* Get ACTIVE-attribute */
  if (object->active && !object->active->directval.typint) {
    object->visible = FALSE;
    return;
  }

  type = object->type;
  if (!(point =
	(XPoint *)CalCalloc (1, (object->ptcount+1)*sizeof (XPoint)))) {
    fprintf (stderr, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  defrect = screenRect (gp);

  if (winxoffs != 0 || winyoffs != 0
      || subwin.x != object->boundingbox.x
      || subwin.y != object->boundingbox.y
      || subwin.width != object->boundingbox.width
      || subwin.height != object->boundingbox.height) {
    mask = XCreatePixmap (gp->system.dpy,
			  XtWindow (gp->system.workArea),
			  subwin.width, subwin.height, 1);
    XCopyGC (gp->system.dpy, gp->system.omskGc,
	     GCFunction | GCLineStyle | GCFillStyle | GCFont,
	     gp->system.maskGc);
  }

/* Map points and lines to polygon, since they are equivalent */
  if (type == S_POINT || type == S_LINE) type = S_POLYGON;

/* (part of) object must be drawn, get original values for Gc */
  XCopyGC (gp->system.dpy, gp->system.origGc,
	   GCFunction
	   | GCForeground | GCLineStyle
	   | GCFillStyle | GCTile | GCStipple
	   | GCFont | GCClipXOrigin | GCClipYOrigin | GCClipMask
	   | GCDashList,
	   gp->system.drawGc);
  XSetTSOrigin (gp->system.dpy, gp->system.drawGc,
		-winxoffs, -winyoffs);

/* Put all the points into an array */
  for (i = 0; i < object->ptcount; i++) {
    point[i].x =  dval[2*i] + winxoffs;
    point[i].y = dval[2*i+1] + winyoffs;
  }
  point[i].x = point[0].x;
  point[i].y = point[0].y;

  /* The following will be used to draw some primitives whose size must
     be positive */
  minx = min (point[0].x, point[1].x);
  maxx = max (point[0].x, point[1].x);
  miny = min (point[0].y, point[1].y);
  maxy = max (point[0].y, point[1].y);

  if (type != S_IMAGE) {
 /* Get the COLOR-attribute */
    switch (gp->system.depth) {
    case 1:
      if ((colval = object->color->directval.typint) > 0
	  && colval < MAXRAST-1) {
	XSetBackground (gp->system.dpy, gp->system.drawGc,
			BlackPixelOfScreen (XtScreen (gp->system.topLevel)));
	XSetForeground (gp->system.dpy, gp->system.drawGc,
			WhitePixelOfScreen (XtScreen (gp->system.topLevel)));
	XSetFillStyle (gp->system.dpy, gp->system.drawGc,
		       FillOpaqueStippled);
	XSetStipple (gp->system.dpy, gp->system.drawGc,
		     gp->raster[colval].picture);
      }
      else {
	XSetFillStyle (gp->system.dpy, gp->system.drawGc, FillSolid);
	XSetForeground (gp->system.dpy, gp->system.drawGc,
			colval == 0 ?
			BlackPixelOfScreen (XtScreen (gp->system.topLevel))
			: WhitePixelOfScreen (XtScreen (gp->system.topLevel)));
      }
      break;
    case 8:
      if ((colval = object->color->directval.typint)>>16 > 0
	  && (colval>>16) < MAXRAST-1
	  && ((colval>>8)&0xFF) != (colval&0xFF)) {
	XSetBackground (gp->system.dpy, gp->system.drawGc,
			(colval>>8)&0xFF);
	XSetForeground (gp->system.dpy, gp->system.drawGc,
			colval&0xFF);
	XSetFillStyle (gp->system.dpy, gp->system.drawGc,
		       FillOpaqueStippled);
	XSetStipple (gp->system.dpy, gp->system.drawGc,
		     gp->raster[colval>>16].picture);
      }
      else {
	XSetFillStyle (gp->system.dpy, gp->system.drawGc, FillSolid);
	XSetForeground (gp->system.dpy, gp->system.drawGc,
			(colval>>16) == 0 ? colval&0xFF : (colval>>8)&0xFF);
      }
      break;
    case 24:
    case 16:
    case 4:
      XSetForeground (gp->system.dpy, gp->system.drawGc,
		      object->color->directval.typint);
      break;
    }
  }

/* Get LEVEL-attribute (all objects have this, at least lev0).
   The actual mask to use resides one level above the attributes level-value,
   as the static objects on the object's level are not to be part of this
   level's mask. It's the object's mask we want here, so if the object is
   (part of) a window, use the parent-window's level */

  level = object->parentwin ?
    (struct levlist *)next(object->parentwin->level->directval.typlev) :
      (struct levlist *)next(object->level->directval.typlev);

/* Get FILL-attribute (textobj is always filled) */  
  fill = (type == S_TEXTOBJECT) ?
    TRUE : object->fill ? object->fill->directval.typint : DEF_FILL;

/* Get TEXTURE-attribute */
  if (object->texture
      && object->texture->curval.typint > 0
      && object->texture->curval.typint < MAXSTIPS) {
    if (mask == None) { /* Need mask anyway */
      mask = XCreatePixmap (gp->system.dpy,
			    XtWindow (gp->system.workArea),
			    subwin.width, subwin. height, 1);
      XCopyGC (gp->system.dpy, gp->system.omskGc,
	       GCFunction | GCLineStyle | GCFont,
	       gp->system.maskGc);
    }
    XSetFillStyle (gp->system.dpy, gp->system.maskGc,
		   FillOpaqueStippled);
    XSetStipple (gp->system.dpy, gp->system.maskGc,
		 object->texture->directval.typimage->picture);
    XFillRectangle (gp->system.dpy, mask, gp->system.maskGc,
		    0, 0, subwin.width, subwin.height);
  }
  else {/* In case it will be needed later */
    XSetFillStyle (gp->system.dpy, gp->system.maskGc, FillSolid);
    if (mask != None)
      XFillRectangle (gp->system.dpy, mask, gp->system.maskGc,
		      0, 0, subwin.width, subwin.height);
  }

/* Get object-specific attributes, and draw if possible */
  switch (type) {
  case S_ARC:
    if (object->startangle)
      sa = object->startangle->directval.typint;
    else sa = DEF_STARTANGLE;
    if (object->endangle)
      ea = object->endangle->directval.typint;
    else ea = DEF_ENDANGLE;
    break;
  case S_IMAGE:
/* If the image is in the cache, don't kill */
    imh = point[1].y-point[0].y;
    imw = point[1].x-point[0].x;
    if(object->ptcount == 1){
      imw = object->image->directval.typimage->width; 
      imh = object->image->directval.typimage->height; 
    }
      
    if(object->ptcount > 1 && imw && imh)
      if (object->image->directval.typimage->in_cache)
	object->image->directval.typimage =
	  scaleImage (gp, object->image->directval.typimage, imw, imh);
      else if (object->image->directval.typimage->width != imw ||
	       object->image->directval.typimage->height != imh) {
	killImage (gp, object->image->directval.typimage);
	object->image->directval.typimage =
	  scaleImage (gp, getImage (gp, object->image->curval.typtxt),
		      imw, imh);
      }

    /* From this point, we're just really interested in positive sizes */
    if (imw < 0) imw = -imw; if (imh < 0) imh = -imh;
  
    if (object->image->directval.typimage->mask) {
/* Copy the subwin-part of the picture's original mask
   into the temporary mask */
      if (mask == None) { /* Need mask anyway */
	mask = XCreatePixmap (gp->system.dpy,
			      XtWindow (gp->system.workArea),
			      subwin.width, subwin. height, 1);
	XCopyGC (gp->system.dpy, gp->system.omskGc,
		 GCLineStyle | GCFillStyle | GCFont, gp->system.maskGc);
	XSetFunction (gp->system.dpy, gp->system.maskGc, GXcopy);
      }
      else
	XSetFunction (gp->system.dpy, gp->system.maskGc, GXand);

      XCopyArea (gp->system.dpy,
		 object->image->directval.typimage->mask, mask,
		 gp->system.maskGc,
		 subwin.x-minx/*point[0].x*/,
		 subwin.y-miny/*point[0].y*/,
		 subwin.width, subwin.height, 0, 0);
    }
    if (level && level->maskstatic) {
      if (mask == None) { /* Need mask anyway */
	mask = XCreatePixmap (gp->system.dpy,
			      XtWindow (gp->system.workArea),
			      subwin.width, subwin. height, 1);
	XCopyGC (gp->system.dpy, gp->system.omskGc,
		 GCLineStyle | GCFillStyle | GCFont,
		 gp->system.maskGc);
	XSetFunction (gp->system.dpy, gp->system.maskGc,
		      GXcopyInverted);
      }
      else
	XSetFunction (gp->system.dpy, gp->system.maskGc,
		      GXandInverted);
      XCopyArea (gp->system.dpy, level->maskstatic->mask, mask,
		 gp->system.maskGc,
		 subwin.x, subwin.y, subwin.width, subwin.height, 0, 0);
    }

/* Now draw image with this mask */
    if (mask != None) {
      XSetClipOrigin (gp->system.dpy, gp->system.drawGc,
		      subwin.x, subwin.y);
      XSetClipMask (gp->system.dpy, gp->system.drawGc, mask);
    }
    XCopyArea (gp->system.dpy, 
	       object->image->directval.typimage->picture, win,
	       gp->system.drawGc,
	       0, 0, imw, imh, minx, miny);

    if (statlevel) {
/* Mask must be drawn into maskstatic on the level */
/* INVAR: When static object are setup, they are always drawn entirely */
      if (!statlevel->maskstatic) newLevMask (gp, statlevel);
      XSetFunction (gp->system.dpy, gp->system.maskGc, GXor);
      if (object->image->directval.typimage->mask)
	XCopyArea (gp->system.dpy, 
		   object->image->directval.typimage->mask,
		   statlevel->maskstatic->mask,
		   gp->system.maskGc,
		   0, 0, imw, imh, minx, miny);
      else
	XFillRectangle (gp->system.dpy, statlevel->maskstatic->mask,
			gp->system.maskGc,
			minx, miny, imw, imh);
    }
    goto objdrawn;
  }

/* For the other objects (not IMAGE) */
  if (level && level->maskstatic) {
    if (mask == None) { /* Need mask anyway */
      mask = XCreatePixmap (gp->system.dpy,
			    XtWindow (gp->system.workArea),
			    subwin.width, subwin. height, 1);
      XCopyGC (gp->system.dpy, gp->system.omskGc,
	       GCLineStyle | GCFillStyle | GCFont,
	       gp->system.maskGc);
      XSetFunction (gp->system.dpy, gp->system.maskGc,
		    GXcopyInverted);
    }
    else
      XSetFunction (gp->system.dpy, gp->system.maskGc, GXandInverted);
    XCopyArea (gp->system.dpy, level->maskstatic->mask, mask,
	       gp->system.maskGc,
	       subwin.x, subwin.y, subwin.width, subwin.height, 0, 0);
  }
  if (mask != None) {
    XSetClipOrigin (gp->system.dpy, gp->system.drawGc,
		    subwin.x, subwin.y);
    XSetClipMask (gp->system.dpy, gp->system.drawGc, mask);
  }

  if (statlevel)
    XSetFunction (gp->system.dpy, gp->system.maskGc, GXor);

  if (!fill) {
/* Get LINEWIDTH-attribute */
    lw = (object->linewidth && object->linewidth->directval.typint != 1) ?
      object->linewidth->directval.typint : 0;

/* Get DASHES-attribute */
    if (object->dashes
	&& object->dashes->curval.typint > 0
	&& object->dashes->curval.typint < MAXDASHES) {
      XSetDashes (gp->system.dpy, gp->system.drawGc, 0,
		  object->dashes->directval.typdash->list,
		  object->dashes->directval.typdash->length);
      XSetLineAttributes (gp->system.dpy, gp->system.drawGc,
			  lw, LineOnOffDash, CapButt, JoinMiter);
    }
    else
      XSetLineAttributes (gp->system.dpy, gp->system.drawGc,
			  lw, LineSolid, CapButt, JoinMiter);
    if (statlevel) {
      if (object->dashes && object->dashes->curval.typint) {
	XSetDashes (gp->system.dpy, gp->system.maskGc, 0,
		    object->dashes->directval.typdash->list,
		    object->dashes->directval.typdash->length);
	XSetLineAttributes (gp->system.dpy, gp->system.maskGc,
			    lw, LineOnOffDash, CapButt, JoinMiter);
      }
      else {
	XSetLineAttributes (gp->system.dpy, gp->system.maskGc,
			    lw, LineSolid, CapButt, JoinMiter);
      }
    }
/* Draw unfilled object */
    switch (type) {
    case S_ARC:
      if (minx != maxx && miny != maxy) { /* Due to X bug */
	XDrawArc (gp->system.dpy, win,
		  gp->system.drawGc,
		  minx, miny, maxx-minx-1, maxy-miny-1, sa*64, (ea-sa)*64);
	if (statlevel) {
	  if (!statlevel->maskstatic) newLevMask (gp, statlevel);
	  XDrawArc (gp->system.dpy, statlevel->maskstatic->mask,
		    gp->system.maskGc,
		    minx, miny, maxx-minx-1, maxy-miny-1, sa*64, (ea-sa)*64);
	}
      }
      goto objdrawn;
    case S_BOX:
      XDrawRectangle (gp->system.dpy, win,
		      gp->system.drawGc,
		      minx, miny, maxx-minx-1, maxy-miny-1);
      if (statlevel) {
	if (!statlevel->maskstatic) newLevMask (gp, statlevel);
	XDrawRectangle (gp->system.dpy, statlevel->maskstatic->mask,
			gp->system.maskGc,
			minx, miny, maxx-minx-1, maxy-miny-1);
      }
      goto objdrawn;
    case S_POLYGON:
      if (i == 1) {
	XDrawPoint (gp->system.dpy, win,
		    gp->system.drawGc,
		    point[0].x, point[0].y);
      }
      else {
	XDrawLines (gp->system.dpy, win,
		    gp->system.drawGc,
		    point, i == 2 ? 2 : i+1, CoordModeOrigin);
      }
      if (statlevel) {
	if (!statlevel->maskstatic) newLevMask (gp, statlevel);
	if (i == 1) {
	  XDrawPoint (gp->system.dpy, statlevel->maskstatic->mask,
		      gp->system.maskGc,
		      point[0].x, point[0].y);
	}
	else {
	  XDrawLines (gp->system.dpy, statlevel->maskstatic->mask,
		      gp->system.maskGc,
		      point, i == 2 ? 2 : i+1, CoordModeOrigin);
	}
      }
      goto objdrawn;
    }
  }
  else {                      /* FILL-attribute == TRUE */

/* Get IMAGE-attribute */
    if (object->image && strcmp (object->image->curval.typtxt, "") != 0) {
      XSetFillStyle (gp->system.dpy, gp->system.drawGc,
		     FillTiled);
      XSetTile (gp->system.dpy, gp->system.drawGc,
		object->image->directval.typimage->picture);
    }

/* Draw filled object */
    switch (type) {
    case S_ARC:
      XFillArc (gp->system.dpy, win,
		gp->system.drawGc,
		minx, miny, maxx-minx, maxy-miny, sa*64, (ea-sa)*64);
      if (statlevel) {
	if (!statlevel->maskstatic) newLevMask (gp, statlevel);
	XFillArc (gp->system.dpy, statlevel->maskstatic->mask,
		  gp->system.maskGc,
		  minx, miny, maxx-minx, maxy-miny, sa*64, (ea-sa)*64);
      }
      goto objdrawn;
    case S_BOX:
      XFillRectangle (gp->system.dpy, win,
		      gp->system.drawGc,
		      minx, miny, maxx-minx, maxy-miny);
      if (statlevel) {
	if (!statlevel->maskstatic) newLevMask (gp, statlevel);
	XFillRectangle (gp->system.dpy, statlevel->maskstatic->mask,
			gp->system.maskGc,
			minx, miny, maxx-minx, maxy-miny);
      }
      goto objdrawn;
    case S_POLYGON:
      /* IMPORTANT: Determine here characteristics of the polygon,
	 this should be in the client, not the -program! */
      if (i < 4) severity = Convex;
      if (i == 1) {
	XDrawPoint (gp->system.dpy, win,
		    gp->system.drawGc,
		    point[0].x, point[0].y);
      }
      else {
	XFillPolygon (gp->system.dpy, win,
		      gp->system.drawGc,
		      point, i, severity, CoordModeOrigin);
      }
      if (statlevel) {
	if (!statlevel->maskstatic) newLevMask (gp, statlevel);
	if (i == 1) {
	  XDrawPoint (gp->system.dpy, statlevel->maskstatic->mask,
		      gp->system.maskGc,
		      point[0].x, point[0].y);
	}
	else {
	  XFillPolygon (gp->system.dpy, statlevel->maskstatic->mask,
			gp->system.maskGc,
			point, i, severity, CoordModeOrigin);
	}
      }
      goto objdrawn;
    case S_TEXTOBJECT:
      textnode = object->outtext ? object->outtext->directval.typtxtnode
	: object->outint ? object->outint->directval.typtxtnode
	  : object->outfloat->directval.typtxtnode;

      if (!textnode->defaultfont) {
	XSetFont (gp->system.dpy, gp->system.drawGc,
		  textnode->fontinfo->fontid);
	if (statlevel)
	  XSetFont (gp->system.dpy, gp->system.maskGc,
		    textnode->fontinfo->fontid);
      }
      if (statlevel && !statlevel->maskstatic) newLevMask (gp, statlevel);

      for (tl = textnode->linlist; tl; tl = (struct txtlist *)next(tl)) {
	XDrawString (gp->system.dpy, win,
		     gp->system.drawGc,
		     point[0].x, point[0].y+textnode->fontinfo->ascent,
		     tl->line, tl->length);
	if (statlevel)
	  XDrawString (gp->system.dpy, statlevel->maskstatic->mask,
		       gp->system.maskGc,
		       point[0].x, point[0].y+textnode->fontinfo->ascent,
		       tl->line, tl->length);
	point[0].y += (textnode->fontinfo->ascent+textnode->fontinfo->descent);
      }
      goto objdrawn;

    }
  }
  objdrawn:
  CalFree (point);
  if (mask != None) XFreePixmap (gp->system.dpy, mask);
}

void drawInitial (struct cw_status *gp, Window win)
{
#ifndef VERSIONNAME
#define VERSIONNAME "- C A N D L E W E B -"
#endif

  int i, j;

  switch (gp->system.depth) {
  case 1:
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    WhitePixelOfScreen (XtScreen (gp->system.topLevel)));
    XSetBackground (gp->system.dpy, gp->system.drawGc,
		    BlackPixelOfScreen (XtScreen (gp->system.topLevel)));
    XSetFillStyle (gp->system.dpy, gp->system.drawGc,
		   FillOpaqueStippled);

    XSetStipple (gp->system.dpy, gp->system.drawGc,
		 gp->raster[colorOfRGB (0x888888)].picture);

    j = 340; i = 10;
    while (j > 0) {
      XDrawLine (gp->system.dpy, win, gp->system.drawGc, 0, j, i, 0);
      j-=35; i+=64;
    }
    
    XSetStipple (gp->system.dpy, gp->system.drawGc,
		 gp->raster[colorOfRGB (0x000000)].picture);
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 19, 30, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 21, 30, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 29, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 31, VERSIONNAME, strlen (VERSIONNAME));
    XSetStipple (gp->system.dpy, gp->system.drawGc,
		 gp->raster[colorOfRGB (0xFFFFCC)].picture);
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 30, VERSIONNAME, strlen (VERSIONNAME));
    break;
  default:
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    colorOfRGB (0x888888));
    j = 340; i = 10;
    while (j > 0) {
      XDrawLine (gp->system.dpy, win, gp->system.drawGc, 0, j, i, 0);
      j-=35; i+=64;
    }
    
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    colorOfRGB (0x000000));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 19, 30, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 21, 30, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 29, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 31, VERSIONNAME, strlen (VERSIONNAME));
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    colorOfRGB (0xFFFFCC));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 30, VERSIONNAME, strlen (VERSIONNAME));
    break;
    /*  case 24:
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    colorOfRGB (0x888888));
    j = 340; i = 10;
    while (j > 0) {
      XDrawLine (gp->system.dpy, win,
		 gp->system.drawGc,
		 0, j, i, 0);
      j-=35; i+=64;
    }
    
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    colorOfRGB (0x000000));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 19, 30, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 21, 30, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 29, VERSIONNAME, strlen (VERSIONNAME));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 31, VERSIONNAME, strlen (VERSIONNAME));
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    colorOfRGB (0xFFFFCC));
    XDrawString (gp->system.dpy, win,
		 gp->system.drawGc,
		 20, 30, VERSIONNAME, strlen (VERSIONNAME));
    break; */
  }
}

CalImage *blankImage (struct cw_status *gp, int width, int height, int color)
{
  CalImage *img = (CalImage *)CalCalloc (1, sizeof (CalImage));
  img->width = width; img->height = height;
  img->color = color;
  img->picture =
    XCreatePixmap (gp->system.dpy, XtWindow (gp->system.workArea),
		   width, height, gp->system.depth);

  XCopyGC (gp->system.dpy, gp->system.origGc,
	   GCFunction
	   | GCForeground | GCLineWidth | GCLineStyle | GCCapStyle
	   | GCJoinStyle | GCFillStyle | GCTile | GCStipple
	   | GCFont | GCClipXOrigin | GCClipYOrigin | GCClipMask
	   | GCDashList,
	   gp->system.drawGc);

  if (gp->system.depth == 1) {
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    WhitePixelOfScreen (XtScreen (gp->system.topLevel)));
    XSetBackground (gp->system.dpy, gp->system.drawGc,
		    BlackPixelOfScreen (XtScreen (gp->system.topLevel)));
    XSetFillStyle (gp->system.dpy, gp->system.drawGc, FillOpaqueStippled);
    XSetStipple (gp->system.dpy, gp->system.drawGc,
		 gp->raster[color].picture);
  }
  else
    XSetForeground (gp->system.dpy, gp->system.drawGc, color);

  XFillRectangle (gp->system.dpy, img->picture, gp->system.drawGc,
		  0, 0, width, height);
  return img;
}

void clearImage (struct cw_status *gp, Pixmap pic, int width, int height)
{
/*
  if (gp->system.depth == 1) {
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    WhitePixelOfScreen (XtScreen (gp->system.topLevel)));
    XSetBackground (gp->system.dpy, gp->system.drawGc,
		    BlackPixelOfScreen (XtScreen (gp->system.topLevel)));
    XSetFillStyle (gp->system.dpy, gp->system.drawGc, FillOpaqueStippled);
    XSetStipple (gp->system.dpy, gp->system.drawGc,
		 gp->raster[gp->curlunit->win->color].picture);
  }
  else
    XSetForeground (gp->system.dpy, gp->system.drawGc,
		    gp->curlunit->win->color); */

  XFillRectangle (gp->system.dpy, pic, gp->system.copyGc,
		  0, 0, width, height);
}

void gifSendData (struct cw_status *gp, AweStream *fd, unsigned char count,
		  long *cmap, CalImage *cip, int interlace)
{

/* Reads data from an image and send them to server */

  long pix;
  int xpos = 0, ypos = 0, pass = 0,
    inv = BlackPixelOfScreen (XtScreen (gp->system.topLevel));
  char *imagedata, *maskdata;
  XImage *image, *mask;

/* Allocate XImages */
  if ((imagedata =
       (char *)CalMalloc (gp->system.depth == 24 ?
			  cip->width*cip->height*4 :
			  gp->system.depth == 16 ?
			  cip->width*cip->height*2
			  : cip->width*cip->height)) == NULL) {
    fprintf (stderr, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  image = XCreateImage (gp->system.dpy, gp->system.visual,
			gp->system.depth, ZPixmap, 0, imagedata,
			cip->width, cip->height, 8, 0);

  if (cip->color != -1) {
    if ((maskdata = (char *)CalMalloc (cip->width*cip->height)) == NULL) {
      fprintf (stderr, ErrNOMOREMEM);
      c_exit (gp, NOT_OK);
    }
    mask = XCreateImage (gp->system.dpy, gp->system.visual,
			 1, ZPixmap, 0, maskdata,
			 cip->width, cip->height, 8, 0);
  }

  while ((pix = LWZReadByte(gp, fd,FALSE,count)) >= 0 ) {
    pix = cmap[pix];
    if (cip->color == -1) {
      if (gp->system.depth == 8) pix = (pix>>16) == 8 ?
	(xpos&1)^(ypos&1) ? (pix>>8)&0xFF : pix&0xFF : pix&0xFF;
      if (gp->system.depth == 1)
	pix = ((rastptr[pix][ypos%4]&power2bits[xpos%4]) != 0)^inv;
      XPutPixel (image, xpos, ypos, (unsigned long)pix);
    }
    else {
      if (pix != cip->color) {
	XPutPixel (mask, xpos, ypos, (unsigned long)1);
	if (gp->system.depth == 8) pix = (pix>>16) == 8 ?
	  (xpos&1)^(ypos&1) ? (pix>>8)&0xFF : pix&0xFF : pix&0xFF;
	if (gp->system.depth == 1)
	  pix = ((rastptr[pix][ypos%4]&power2bits[xpos%4]) != 0)^inv;
	XPutPixel (image, xpos, ypos, (unsigned long)pix);
      }
      else {
	XPutPixel (mask, xpos, ypos, (unsigned long)0);
	XPutPixel (image, xpos, ypos, (unsigned long)0);
      }
    }
    ++xpos;
    if (xpos == cip->width) {
      xpos = 0;
      if (interlace) {
        switch (pass) {
        case 0:
        case 1:
          ypos += 8; break;
        case 2:
          ypos += 4; break;
        case 3:
          ypos += 2; break;
        }
	
        if (ypos >= cip->height) {
          ++pass;
          switch (pass) {
          case 1:
            ypos = 4; break;
          case 2:
            ypos = 2; break;
          case 3:
            ypos = 1; break;
          }
        }
      } else {
	++ypos;
      }
    }
  }
  if (cip->color != -1) imgToServer (gp, cip, mask, FALSE);
  imgToServer (gp, cip, image, TRUE);
}

void jpgSendData (struct cw_status *gp, CalImage *cip,
		  struct jpeg_decompress_struct *cinfo)
{
  int lc, i, inv = BlackPixelOfScreen (XtScreen (gp->system.topLevel));
  char *imagedata;
  XImage *image;
  JSAMPARRAY buffer = (*cinfo->mem->alloc_sarray)
    ((j_common_ptr) cinfo, JPOOL_IMAGE,
     cinfo->output_width * cinfo->output_components, 1);

  cip->width = cinfo->output_width;
  cip->height = cinfo->output_height;
  cip->color = -1; /* Indicates no mask, mask-ptr set to zero in imgToServer */

  if ((imagedata =
       (char *)CalMalloc (gp->system.depth == 24 ?
			  cip->width*cip->height*4
			  : gp->system.depth == 16 ?
			  cip->width*cip->height*2
			  : cip->width*cip->height)) == NULL) {
    fprintf (stderr, ErrNOMOREMEM);
    c_exit (gp, NOT_OK);
  }
  image = XCreateImage (gp->system.dpy, gp->system.visual,
			gp->system.depth, ZPixmap, 0, imagedata,
			cip->width, cip->height, 8, 0);

  for (lc = 0; cinfo->output_scanline < cinfo->output_height; lc++) {
    jpeg_read_scanlines(cinfo, buffer, 1);
    switch (gp->system.depth) {
    case 1:
      for (i = 0; i < cinfo->output_width; i++)
	XPutPixel (image, i, lc,
		   (unsigned long)
		   ((rastptr[(int)(buffer[0][i])][lc%4]
		     &power2bits[i%4]) != 0)^inv);
      break;
    case 24:
      for (i = 0; i < cinfo->output_width; i++)
	XPutPixel (image, i, lc, (unsigned long)((buffer[0][i*3]<<16)
						 |(buffer[0][i*3+1]<<8)
						 |buffer[0][i*3+2]));
      break;
    case 16:
      for (i = 0; i < cinfo->output_width; i++)
	XPutPixel (image, i, lc,
		   (unsigned long)((((buffer[0][i*3])&0xF8)<<8)
				   |(((buffer[0][i*3+1])&0xFC)<<3)
				   |((buffer[0][i*3+2])&0xF8)>>3));
      break;
    default:
      for (i = 0; i < cinfo->output_width; i++)
	XPutPixel (image, i, lc, (unsigned long)(buffer[0][i]));
      break;
    }
  }
  imgToServer (gp, cip, image, TRUE);
}

int colsInCMap (struct cw_status *gp)
{
  return
    awe_system.depth == 1 ? MAXRAST
      : awe_system.depth == 4 ? 16
	: awe_system.depth == 8 ? MAXCOLS
	  : -1;
}

/* Fill colormap with standard values and returns their actual number */
void fillStdCMap (JSAMPLE *rcmap, JSAMPLE *gcmap, JSAMPLE *bcmap, int colcount)
{
  int i, idx;
  unsigned long colval;

  /* Not called for 24-bit!!! */
  for (i = 0; i < colcount; i++) {
    colval = awe_system.colorModel == PSEUDO8ADAPTED ?
      ((awe_system.color[i].color.spec.RGB.red<<8)&0xFF0000)
      |(awe_system.color[i].color.spec.RGB.green&0x00FF00)
      |((awe_system.color[i].color.spec.RGB.blue>>8)&0x0000FF)
      : ((i&0xE0)<<16)|((i&0x1C)<<11)|((i&0x03)<<6);
    
    idx = awe_system.colorModel == PSEUDO8PERFECT ?
      awe_system.color[i].color.pixel : i;

    rcmap[idx] = (JSAMPLE)(colval>>16&0xFF);
    gcmap[idx] = (JSAMPLE)(colval>>8&0xFF);
    bcmap[idx] = (JSAMPLE)(colval&0xFF);
  }
}


#ifdef MESSAGE
void message (char *m)
{
  FILE *fp = fopen ("awelog", "a");
  if (fp) {
    fprintf (fp, m);
    fclose (fp);
  }
}
#endif
