/* spin_write_buffer object */

#include "spin_py_mod.h"
#include "Python.h"
#include "write_buffer.h"
#include "str_obj.h"

#include "buffer/misc.h"
#include "buffer/blk1.h"
#include "buffer/buffer1.h"
 

/*
	spin_write_buffer is a buffer to reduce the number of syscalls.

 Python usage:
	buff = Buffer( downstream, [lo], [hi] ) # create buffer
	buff.write( str ) # add str to buffer
	buff.flush()	  # calls downstream,write() and free() if..
	buff.flush2()	# buff.flush() downstream.flush2()

	Also need a derived class (type) which keeps line info
	such as indentation, line_is_empty, even wrapping text
	see (TBA) spin_line_buffer
*/

extern PyTypeObject spin_write_buffer_Type; // forward

static PyMethodDef spin_write_buffer_methods[]; // forward without problem!

#define SPIN_write_buffer_Check(v)	((v)->ob_type == &spin_write_buffer_Type)

extern SPIN_write_buffer *
as_SPIN_write_buffer( PyObject * buff )
{
	/*
		Check that buff is really a write_buffer, else error
	*/
	if (buff && SPIN_write_buffer_Check( buff )) {
		return (SPIN_write_buffer *) buff;
	} else {
		PyErr_SetString(PyExc_TypeError,"SPIN_write_buffer expected");
		return (SPIN_write_buffer *) NULL;
	}
}

/* EXPORTED */
SPIN_write_buffer * new_SPIN_write_buffer(	// create or edit an write_buffer 
	PyObject *downstream_,
	int lo_,
	int hi_
) {
	if( !downstream_ ) {
		PyErr_SetString(PyExc_TypeError,"new_SPIN_write_buffer: downstream is NULL");
		return NULL;
	}
	// create the basic object
	SPIN_write_buffer *self;
	self = PyObject_NEW( SPIN_write_buffer, &spin_write_buffer_Type );
	self->buff.raw_init(); // didnt use C++ new, used malloc
	self->downstream = downstream_;
	Py_INCREF( downstream_ );
	self->lo = lo_;
	self->hi = hi_;
	self->var_pool = NULL;
	self->buff.resize( lo_ );
	return self;
}

/*
	new_SPIN_write_buffer( ... ) is the only thing exprted from here,
	once the object exists, it can find its hidden static functions
*/

/* --------------------------------------------------------------------- */

static void
spin_write_buffer_dealloc( SPIN_write_buffer *self )
{
	self->buff.resize(0); // MUST do full C++ closedown (its C mostly)
	Py_XDECREF(self->downstream);
	PyMem_DEL(self);
}

static PyObject *
spin_write_buffer_getattr(
	SPIN_write_buffer *self,
	char *name )
{
	if(0==strcmp( name, "downstream" ))
	{
		Py_INCREF( self->downstream ); // can't be null!
		return self->downstream ; 
	}
	if(0==strcmp( name, "lo" ))
	{
		return PyInt_FromLong( self->lo );
	}
	if(0==strcmp( name, "hi" ))
	{
		return PyInt_FromLong( self->hi );
	}
	if(0==strcmp( name, "atts" )) // rename "var_pool" ?
	{
		if(! self->var_pool ) {
			Py_INCREF( Py_None );
			return Py_None;
		}
		Py_INCREF( self->var_pool ); // you have to know about is_pre
		return self->var_pool ; 
	}
	// printf(" spin.getattr(%s)\n",name);
	if (self->var_pool != NULL) {
		PyObject *ret = PyDict_GetItemString(self->var_pool, name);
		if (ret) {
			Py_INCREF(ret);
			return ret;
		}
	}
	return Py_FindMethod(spin_write_buffer_methods, (PyObject *)self, name);
}

static int
spin_write_buffer_setattr(
	SPIN_write_buffer *self,
	char *name,
	PyObject *v )
{
	if(0==strcmp( name, "downstream" ))
	{
		int breaks_word = PyInt_AsLong( v ); // no error detected
		if( !v ) {
			PyErr_SetString(PyExc_AttributeError,
			        "delete non-existing spin_write_buffer attribute");
			return -1;
		}
		self->downstream = v;
		Py_INCREF( v );
		return 0;
	}
	if(0==strcmp( name, "hi" ))
	{
		int val = PyInt_AsLong( v ); // no error detected
		self->hi = val;
		return 0;
	}
	if(0==strcmp( name, "lo" ))
	{
		int val = PyInt_AsLong( v ); // no error detected
		self->lo = val;
		return 0;
	}
	// printf("SPIN.setattr(%s)\n",name);
	if (self->var_pool == NULL) {
		self->var_pool = PyDict_New();
		if (self->var_pool == NULL)
			return -1;
	}
	if (v == NULL) {
		int ret = PyDict_DelItemString(self->var_pool, name);
		if (ret < 0)
			PyErr_SetString(PyExc_AttributeError,
			        "delete non-existing spin_write_buffer attribute");
		return ret;
	}
	else
		return PyDict_SetItemString(self->var_pool, name, v);
}
/* --------------------------------------------------------------------- */
//	SPIN  write_buffer  Type -- 
/* --------------------------------------------------------------------- */

// static // warning due to previous declaration 
// which HAD to be external for g++ (but not gcc)

PyTypeObject spin_write_buffer_Type = {

	 PyObject_HEAD_INIT(&PyType_Type)
		0,				/*ob_size*/
	 (char *) "spin_write_buffer",		/*tp_name*/
		sizeof(	SPIN_write_buffer ),	/*tp_basicsize*/
		0,				/*tp_itemsize*/
	 /* methods */
	 (destructor)	spin_write_buffer_dealloc, /*tp_dealloc*/
	 (printfunc)	0,	/*tp_print*/
	 (getattrfunc)	spin_write_buffer_getattr, /*tp_getattr*/
	 (setattrfunc)	spin_write_buffer_setattr, /*tp_setattr*/
		0,				/*tp_compare*/
		0,				/*tp_repr*/
		0,				/*tp_as_number*/
		0,				/*tp_as_sequence*/
		0,				/*tp_as_mapping*/
		0,				/*tp_hash*/
};


/* --------------------------------------------------------------------- */


static PyObject *
spin_write_buffer_write(
	SPIN_write_buffer * self,
	PyObject *args )
{
	char * str; // binary block though!
	int len;
	if (!PyArg_ParseTuple(args, (char *)"s#:spin_write_buffer_write",
		&str, &len ))
	{
		// 	"Python should gen own error");
		return NULL;
	}
	/* 
		// buffer can still exceed hi, esp single request, but OK
	*/
	if(( self->buff.nbytes_used + len ) >= self->hi )  {
		// call self.flush()
		if( ! PyObject_CallMethod(
			(PyObject *)self,
			(char *) "flush",
			(char *) "()" ) )
		{
			return NULL;
		}
	}

	self->buff.put_nn_bytes( len, str );
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
spin_write_buffer_flush(
	SPIN_write_buffer * self,
	PyObject *args )
{
	char * str; // binary block though!
	int len;
	if (!PyArg_ParseTuple(args, (char *)":spin_write_buffer_flush"))
		return NULL;
	/* 
		// buffer can still exceed hi, esp single request, but OK
		if(( self->buff.nbytes_used + len ) > self.hi ) flush() ;
	*/
	len = self->buff.nbytes_used;
	if( ! len ) { // OK
		Py_INCREF(Py_None);
		return Py_None;
	}

	// create the String for the next (Python) device
	PyObject * string;
	string = PyString_FromStringAndSize(
		self->buff.buff,
		self->buff.nbytes_used);
	if(!string) {
		return NULL;
	}
	// clear the buffer, reclaim some space ?
	self->buff.nbytes_used = 0; //leave putc_prev
	self->buff.resize( self->lo );

	if( ! PyObject_CallMethod(
		self->downstream,
		(char *) "write",
		(char *) "(O)",
		(PyObject *) string ) )
	{
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
spin_write_buffer_flush2(
	SPIN_write_buffer * self,
	PyObject *args )
{
	/*
		flush2 = self.flush() and downstream.flush2() ...

		self.flush()
		try:
			self.downstream.flush2()
			return
		except:
			# noone else uses flush2, try flush, even then OK
			try:
				self.downstream.flush()
			except:
				pass
	*/
	if (!PyArg_ParseTuple(args, (char *)":spin_write_buffer_flush2"))
		return NULL;

	// call self.flush()
	if( ! PyObject_CallMethod(
		(PyObject *)self,
		(char *) "flush",
		(char *) "()" ) )
	{
		return NULL;
	}

	// call downstream.flush2()
	if( ! PyObject_CallMethod(
		self->downstream,
		(char *) "flush2",
		(char *) "" ) )
	{
		// FAILED: ok if attribute error "flush2()", try flush()
		PyErr_Clear();
		// call downstream.flush()
		if( ! PyObject_CallMethod(
			self->downstream,
			(char *) "flush",
			(char *) "" ) )
		{
			// ok if attribute error "flush()", pass
			PyErr_Clear();
		}
	}

	Py_INCREF(Py_None);
	return Py_None;
}

static PyMethodDef spin_write_buffer_methods[] = {
	{(char *)"write",	(PyCFunction)spin_write_buffer_write,	1},
	{(char *)"flush",	(PyCFunction)spin_write_buffer_flush,	1},
	{(char *)"flush2",	(PyCFunction)spin_write_buffer_flush2,	1},
	{NULL,		NULL}		/* sentinel */
};
