/* 
 * commandsAM.cc --
 *
 *      This file contains the top-level command routines for most of
 *      the PAPyRUS built-in commands whose names begin with the letters
 *      A to M.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

extern "C" {
#include <stdio.h>
#include <string.h>

#ifdef HAVE_GUI
  #include <X11/Intrinsic.h>
#else
  #include <X11/Xlib.h>
#endif

#include "../misc/shortcuts.h"
}


#include "../kernel/container.h"
#include "../kernel/wordSegment.h"
#include "../kernel/image.h"
#include "../kernel/word.h"
#include "../kernel/line.h"
#include "../kernel/page.h"
#include "../kernel/document.h"
#include "../kernel/papyrus.h"
#include "../kernel/isfuncs.h"
#include "../kernel/events.h"
#include "../kernel/util.h"
#include "tclGet.h"
#include "commands.h"


#ifdef HAVE_GUI
extern "C" {
extern void GUI_Add_OptionMenuEntry(char *, char *, char *);
extern void GUI_Set_DocumentTitle(char *);
}
#endif

Tcl_Channel channel;



/*
 *----------------------------------------------------------------------
 *
 * CloseDocument_Cmd --
 *
 *      This procedure is invoked to process the "CloseDocument" or 
 *      "cd" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
CloseDocument_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  if( argc != 1 ) {
    interp->result = "Usage: CloseDocument";
    return TCL_ERROR;
  }
  
  if( current.doc == NULL ) {
    interp->result = "there's no opened document";
    return TCL_ERROR;
  }
  delete current.doc;
  current.doc = NULL;
  Tcl_SetVar(interp, "alive", "0", TCL_GLOBAL_ONLY);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorLeft_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorLeft" or 
 *      "mcl" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorLeft_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Shape *shape;
  Page *page;


  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorLeft";
    return TCL_ERROR;
  }

  page = current.shape->get_page_parent();

  if( current.pos > 0 ) {
    current.pos--;
  } else {
    shape = (Shape *)current.shape->get_previous_same_container();
    
    if( shape == NULL ) {
      Tcl_AppendResult(interp, "Can't move left, beginning of the text", (char *)NULL);
      return TCL_ERROR;
    }
    if( shape->has_same_parent( current.shape ) == TRUE )
      current.pos = shape->get_children_num()-1;
    else
      current.pos = shape->get_children_num();
    current.shape = shape;
  }

  /*
   * The cursor is perhaps not in the same page as before the moved
   */
  if( page != current.shape->get_page_parent() ) {
    papyrus->set_win( current.shape->get_page_parent()->get_win() );
  }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorRight_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorRight" or 
 *      "mcr" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorRight_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Shape *shape;
  Page *page;

  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorRight";
    return TCL_ERROR;
  }

  page = current.shape->get_page_parent();

  if( current.pos < current.shape->get_children_num() ) {
    current.pos++;
  } else {
    shape = (Shape *)current.shape->get_next_same_container();
    
    if( shape == NULL ) {
      Tcl_AppendResult(interp, "Can't move right, end of the text", (char *)NULL);
      return TCL_ERROR;
    }
    if( shape->has_same_parent( current.shape ) == TRUE 
       && shape->get_children_num() > 0 )
      current.pos = 1;
    else
      current.pos = 0;
    current.shape = shape;
  }

  /*
   * The cursor is perhaps not in the same page as before the moved
   */
  if( page != current.shape->get_page_parent() ) {
    papyrus->set_win( current.shape->get_page_parent()->get_win() );
  }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorUp_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorUp" or 
 *      "mcu" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorUp_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Frame *line, *pline;
  int xpos, ypos, tmp;
  Page *page, *ppage;

  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorUp";
    return TCL_ERROR;
  }

  line  = current.shape->get_line_parent();
  pline = (Frame *)line->get_previous_same_container();
  page  = (Page *)line->get_parent();
  
  if( pline == NULL ) {
    Tcl_AppendResult(interp, "Can't move up, first line", (char *)NULL);
    return TCL_ERROR;
  }

  ppage = (Page *)pline->get_parent();

  /* Takes the x position of the current object */
  current.shape->frame_to_xy(xpos, tmp, current.pos);

  /* Takes the y position of the current object */
  pline->frame_to_xy(tmp, ypos, 0);

  /*
   * Tells the object of the previous line which has the
   * same coordinates than the current object
   */
  current.shape  = (Shape *)ppage->xy_to_frame(xpos, ypos, current.pos);
  
  /*
   * The cursor is perhaps not in the same page as before the moved
   */
  if( page != ppage )
    papyrus->set_win( ppage->get_win() );

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorDown_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorDown" or 
 *      "mcd" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorDown_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Frame *line, *nline;
  int xpos, ypos, tmp;
  Page *page, *npage;


  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorDown";
    return TCL_ERROR;
  }

  line = current.shape->get_line_parent();
  nline = (Frame *)line->get_next_same_container();
  page = (Page *)line->get_parent();
  
  if( nline == NULL ) {
    Tcl_AppendResult(interp, "Can't move down, last line", (char *)NULL);
    return TCL_ERROR;
  }

  npage = (Page *)nline->get_parent();

  /* Takes the x position of the current object */
  current.shape->frame_to_xy(xpos, tmp, current.pos);
  

  /* Takes the x position of the current object */
  nline->frame_to_xy(tmp, ypos, 0);


  /*
   * Tells the object of the next line which has the
   * same coordinates than the current object
   */
  current.shape = (Shape *)npage->xy_to_frame(xpos, ypos, current.pos);
  
  /*
   * The cursor is perhaps not in the same page as before the moved
   */
  if( page != npage )
    papyrus->set_win( npage->get_win() );

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorNextWord_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorNextWord" or 
 *      "mcnw" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorNextWord_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Container *next_word;


  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorNextWord";
    return TCL_ERROR;
  }

  next_word = current.shape->get_parent()->get_next_same_container();
  if( next_word == NULL ) {
    Tcl_AppendResult(interp, "Can't move next word, end of text", (char *)NULL);
    return TCL_ERROR;
  }
  current.shape = (Shape *)next_word->get_child(0);
  current.pos = 0;

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorPreviousWord_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorPreviousWord" or 
 *      "mcpw" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorPreviousWord_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Container *prev_word;

  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorPreviousWord";
    return TCL_ERROR;
  }

  prev_word = current.shape->get_parent()->get_previous_same_container();
  if( prev_word == NULL ) {
    Tcl_AppendResult(interp, "Can't move previous word, beginning of text", (char *)NULL);
    return TCL_ERROR;
  }
  current.shape = (Shape *)prev_word->get_child(0);
  current.pos = 0;

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorBeginLine_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorBeginLine" or 
 *      "mcbl" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorBeginLine_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Container *line;


  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorBeginLine";
    return TCL_ERROR;
  }

  if( is_first_of_line(current) == TRUE ) {
    Tcl_AppendResult(interp, "Already at the beginning of line", (char *)NULL);
    return TCL_ERROR;
  }

  line = current.shape->get_parent()->get_parent();

  current.shape = (Shape *)line->get_child(0)->get_child(0);
  current.pos = 0;

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorEndLine_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorEndLine" or 
 *      "mcel" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int MoveCursorEndLine_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Line *line;
  Container *con;


  if( argc != 1 ) {
    interp->result = "Usage: MoveCursorEndLine";
    return TCL_ERROR;
  }

  if( is_last_of_line(current) == TRUE ) {
    Tcl_AppendResult(interp, "Already at the end of line", (char *)NULL);
    return TCL_ERROR;
  }

  line = current.shape->get_line_parent();

  con = line->get_child( line->get_children_num()-1 );
  current.shape = (Shape *)con->get_child( con->get_children_num() - 1);
  current.pos = current.shape->get_children_num();

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * MoveCursorXY_Cmd --
 *
 *      This procedure is invoked to process the "MoveCursorXY" or 
 *      "mcxy" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
MoveCursorXY_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  
  int x, y;
  int curX, curY;
  Page *pagePtr;

  if( argc != 3 ) {
    interp->result = "Usage: MoveCursorXY";
    return TCL_ERROR;
  }

  if( Tcl_GetInt(interp, argv[1], &curX) != TCL_OK )
    return TCL_ERROR;

  if( Tcl_GetInt(interp, argv[2], &curY) != TCL_OK )
    return TCL_ERROR;

  pagePtr = current.shape->get_page_parent();
  current.shape = (Shape *)pagePtr->xy_to_frame(curX, curY, current.pos);
  current.shape->frame_to_xy(x, y, current.pos);  

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * AddEntry_Cmd --
 *
 *      This procedure is invoked to process the "AddEntry" or 
 *      "ae" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
AddEntry_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

#ifdef HAVE_GUI
  int i;
#endif

  if( argc < 4 && argc&1 ) {
    Tcl_AppendResult(interp,
		     "wrong # args: should be \"", argv[0],
		     " type entry ?entry? ...?\"",
		     (char *)NULL);
    return TCL_ERROR;    
  }

  if( argv[1][0] == 'z' && strcmp(argv[1], "zoom") == 0 );
  else if( argv[1][0] == 's' && strcmp(argv[1], "style") == 0 );
  else if( argv[1][0] == 'f' && strcmp(argv[1], "family") == 0 );
  else if( argv[1][0] == 's' && strcmp(argv[1], "size") == 0 );
  else {
    Tcl_AppendResult(interp, "Error: unknown entry \"", argv[1],
		     "\": must be zoom, family, style or size", (char *)NULL);
    return TCL_ERROR;
  }
#ifdef HAVE_GUI
  for(i=3; i<argc; i+=2)
    GUI_Add_OptionMenuEntry(argv[1], argv[i-1], argv[i]);
#endif
  
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * AddFont_Cmd --
 *
 *      This procedure is invoked to process the "AddFont" or 
 *      "af" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
AddFont_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int fa, st, x11, ps;
  int i;
  char message[64];
  FontStyle style;

  if( argc != 9 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0],
		     " -family familyname -style style -x11 x11name -ps psname\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  fa = st = x11 = ps = 0;

  for(i=1; i<argc; i += 2) {
    if( argv[i][0] != '-' )
      goto error;
    if( argv[i][1] == 'f' && strcmp(argv[i], "-family") == 0 )     fa = i+1;
    else if( argv[i][1] == 's' && strcmp(argv[i], "-style") == 0 ) st = i+1;
    else if( argv[i][1] == 'x' && strcmp(argv[i], "-x11") == 0 )   x11= i+1;
    else if( argv[i][1] == 'p' && strcmp(argv[i], "-ps") == 0 )    ps = i+1;
    else {
    error:
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be -family, -style, -x11 or -ps",
		       (char *)NULL);
      return TCL_ERROR;
    }
  }

  if( fa == 0 || st == 0 || x11 == 0 || ps == 0 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0],
		     " family style x11_name ps_name\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  if( Tcl_GetFontStyle(interp, argv[st], style) != TCL_OK )
      return TCL_ERROR;

  if( papyrus->add_font(argv[fa], style, argv[x11], argv[ps]) == NULL ) {
    sprintf(message, "Error: font \"%s-%s\" already exists\n",
	    argv[fa], FontStyle_to_String(style));
    Tcl_AppendResult(interp, message, (char *)NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * AddStyle_Cmd --
 *
 *      This procedure is invoked to process the "AddStyle" or 
 *      "as" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
AddStyle_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i;
  int name, base, font, align, margins, flmargin;
  int tag, next;
  StyleItem *si, *bsi;
  StyleAlignType *sat;


  name = base = font = align = margins = flmargin = tag = next = 0;
  bsi = NULL;

  if( (argc & 1) == 0 ) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		     " -name stylename ?-font fontname? ?-align alignment?",
		     " ?-margins distances? ?-tag tagchar? ?-next stylename?\"",
		     (char *)NULL);
    return TCL_ERROR;
  }
  
  for(i=1; i<argc; i+=2) {
    if( argv[i][0] != '-' )
      goto error;
    if( argv[i][1] == 'n' && strcmp(argv[i], "-name") == 0 )         name = i+1;
    else if( argv[i][1] == 'f' && strcmp(argv[i], "-font") == 0 )    font = i+1;
    else if( argv[i][1] == 'a' && strcmp(argv[i], "-align") == 0 )   align = i+1;
    else if( argv[i][1] == 'm' && strcmp(argv[i], "-margins") == 0 ) margins = i+1;
    else if( argv[i][1] == 't' && strcmp(argv[i], "-tag") == 0 )     tag = i+1;
    else if( argv[i][1] == 'f' && strcmp(argv[i], "-fline") == 0 )   flmargin = i+1;
    else if( argv[i][1] == 'n' && strcmp(argv[i], "-next") == 0 )    next = i+1;
    else if( argv[i][1] == 'b' && strcmp(argv[i], "-base") == 0 )    base = i+1;
    else {
    error:
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be -name, -font, -align, -margins, -tag, -fline,\
-next or -base", (char *)NULL);
      return TCL_ERROR;
    }
  }

  if( name == 0 ) {  //   || font == 0 || align == 0 || margins == 0 ) {
    Tcl_AppendResult(interp, "wrong # args: must have at least \"-name\" option",
		     (char *)NULL);
    return TCL_ERROR;
  }
  
  if( base != 0 ) {
    bsi = papyrus->query_style(argv[base]);
    if( bsi == NULL ) {
      Tcl_AppendResult(interp, "Error: base style \"", argv[base],
		       " \" for style \"", argv[name],
		       " \" is unknown", (char *)NULL);
      return TCL_ERROR;
    }
  }

  si = papyrus->add_style(argv[name], bsi);

  if( si == NULL ) {
    Tcl_AppendResult(interp, "Error: style name \"", argv[name],
		     "\" already exists", (char *)NULL);
    return TCL_ERROR;
  }
  
  if( align != 0 ) {
    if( Tcl_GetAlignment(interp, argv[align], sat) != TCL_OK )
      return TCL_ERROR;
    si->set_attr(STYLE_ALIGNMENT, (void *)sat);
  }

  if( font != 0 ) {
    FontItem *fi;

    if( Tcl_GetFont(interp, argv[font], fi) != TCL_OK )
      return TCL_ERROR;
    si->set_attr(STYLE_FONT, (void *)fi);
  }

  if( margins != 0 ) {
    int m[4];

    if( Tcl_GetMargins(interp, argv[margins], m) != TCL_OK )
      return TCL_ERROR;
    si->set_attr(STYLE_TOP_MARGIN,    (void *)m[0]);
    si->set_attr(STYLE_BOTTOM_MARGIN, (void *)m[1]);
    si->set_attr(STYLE_LEFT_MARGIN,   (void *)m[2]);
    si->set_attr(STYLE_RIGHT_MARGIN,  (void *)m[3]);
  }

  if( flmargin != 0 ) {
    if( Tcl_GetPixels(interp, argv[flmargin], &flmargin) != TCL_OK )
      return TCL_ERROR;
    si->set_attr(STYLE_FLINE_MARGIN,  (void *)flmargin);
  }

  if( tag != 0 ) {
    if( Tcl_GetInt(interp, argv[tag], &tag) != TCL_OK )
      return TCL_ERROR;
    si->set_attr(STYLE_TAG, (void *)tag);
  }
  
  if( next != 0 )
    si->set_attr(STYLE_NEXT_STYLE, (void *)argv[next]);

  /*
   * If the current style is not based on another style, we have
   * to verify that the required 'font' field has been set. This
   * field is the only one which hasn't a default value (except
   * of course, the 'name' field).
   *
   * The field 'name' is also required, but it is not tested,
   * because it has been yet requested (for StyleItem creation)
   */

  if( base == 0 && font == 0 ) {
    Tcl_AppendResult(interp, "Error: must specify a font", (char *)NULL);
    return TCL_ERROR;
  }

#ifdef HAVE_GUI
  Tcl_VarEval(interp, 
	      "AddEntry style ", argv[2], " {ParaConfigure -style ", argv[2], "}",
	      (char *)NULL);
#endif
  
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Debug_Cmd --
 *
 *      This procedure is invoked to process the "Debug" or 
 *      "dbg" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Debug_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int keys, tree, num;

  if( argc != 2 ) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " option\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  tree = keys = 0;

  if( strcmp(argv[1], "tree") == 0 )
    tree = 1;
  else if( strcmp(argv[1], "keys") == 0 )
    keys = 1;
  else if( strcmp(argv[1], "all") == 0 )
    keys = tree = 1;
  else {
    Tcl_AppendResult(interp, "Error: unknown option \"", argv[1],
		     "\": must be tree, keys or all", (char *)NULL);
    return TCL_ERROR;
  }

  if( tree == 1 ) {
    current.doc->debug();
    fprintf(stderr, "con = %p \t pos = %d \t attr = %p \t a_mask = %d \t a_attr = %p\n",
	    current.shape, current.pos, current.attr,
 	    current.attr->has_mark(), current.attr->get_attributes());

    Attribute *attr = current.attr->get_attributes();
    int i = 0;
    while( attr != NULL ) {
      i++;
      attr = attr->next;
    }
    fprintf(stderr, "Nbr d'attributs = %d\n", i);
  }

  if( keys == 1 ) {
    num = display_all_shortcuts();
    fprintf(stderr, "There's %d shortcuts\n", num);
  }
    
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * DeleteCharacterBefore_Cmd --
 *
 *      This procedure is invoked to process the "DeleteCharacterBefore" or 
 *      "dcb" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
DeleteCharacterBefore_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Shape *shape, *c;
  Line *line, *linePtr;
  BOOL flag = FALSE;
  Page *page, *pagePtr;


  if( argc != 1 ) {
    interp->result = "Usage: DeleteCharacterBefore";
    return TCL_ERROR;
  }

  PositionSave();
  c = current.shape;
  line = current.shape->get_line_parent();

  if( Tcl_Eval(interp, "MoveCursorLeft") == TCL_ERROR ) {
    return TCL_ERROR;
  }

  c->get_line_parent()->clear_frame();

  /*
   * The current Paragraph has changed
   */

  if( c->get_para() != current.shape->get_para() ) {
    shape = current.shape;
    
    /*
     * Look if either the current line, or the start line (current line
     * before the 'MoveCursorLeft' are empty. If true we just delete it
     * and redisplay all the page
     */

    if( c->get_line_parent()->get_width() == 0 ) {
      c->get_parent()->delete_children(1, 0);      
      goto recompute_label;
    }

    if( shape->get_line_parent()->get_width() == 0 ) {
      shape->delete_children(1, 0);
      PositionRestore();
      shape = current.shape;

    recompute_label:
      shape->get_line_parent()->set_to_redraw(REDRAW_PAGE);
      page = shape->get_page_parent();
      page->can_fit(current);

      pagePtr = (Page *)page->get_previous_same_container();
      if( pagePtr != NULL )
	page = pagePtr;

      page->format_frame();
      current.doc->draw_frame(0,0);

      return TCL_OK;
    } 

    /*
     * Else we have to merge the two paragraphs (the current, and the
     * previous one) and redisplay all the page
     */
    
    MergeParagraphs(c->get_line_parent(), shape->get_para());
    shape->get_line_parent()->set_to_redraw(REDRAW_PAGE);

    /*
     * Look if current 'word' is empty, if true we have to delete
     * it, to prevent a nil WordSegment in the middle of the line
     */

    if( ((Frame *)current.shape->get_parent())->get_width() == 0 )
      c = current.shape;

  }

  shape = current.shape;

  /*
   * The current word has changed
   */

  if( c->has_same_parent(current.shape) == FALSE ) {
    /*
     * Look if the current word is not in the save line
     * as the start word (current word before cursor move)
     * if true, we have to redisplay the line.
     */
    if( shape->get_line_parent() != c->get_line_parent() )
      shape->get_line_parent()->set_to_redraw(REDRAW_ME);
    
    if( current.shape->can_append() == FALSE || c->can_append() == FALSE ) {
      Tcl_Eval(interp, "MoveCursorRight");
      return TCL_ERROR;
    }

    current.shape->get_parent()->merge_container(c->get_parent());
  } else {

    /*
     * Look if the current wordSegment will be completly erased
     * ie if it contains anly one character.
     */

    if( current.shape->get_children_num() <= 1 )
      flag = PositionFind_And_Replace(current);
    
    linePtr = shape->get_line_parent();
    shape->delete_children(1, current.pos);

    if( flag == TRUE )
      PositionRestore();

    line = current.shape->get_line_parent();

    if( line->is_in_same_para(linePtr) == FALSE )
      line->set_to_redraw(REDRAW_PAGE);
  }
  
  shape = current.shape;
  linePtr = shape->get_line_parent();


  if( linePtr->get_child(0) == shape->get_parent() ) {
    linePtr = (Line *)linePtr->get_previous_same_container();
    if( linePtr != NULL ) {
      if( shape->get_line_parent()->is_in_same_para( linePtr ) == TRUE ) {
	linePtr->set_to_redraw(REDRAW_ME);
	linePtr->format_frame();
      }
    }
  }
  shape->get_line_parent()->can_fit(current, TRUE);
  ((Word *)shape->get_parent())->format_frame(current);

  /*
   * Now recompute the current line, if it's the last line, and if there's
   * a bottom margin (ie distance between the last line of the paragraph and
   * the first line of the next paragraph).
   */

  if( shape->get_line_parent()->is_last_of_para() == TRUE )
    if( shape->get_para()->query(STYLE_BOTTOM_MARGIN) != 0 )
      shape->get_line_parent()->recompute_size();
  
  current.doc->draw_frame(0,0);  
  
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * DeleteCharacterAfter_Cmd --
 *
 *      This procedure is invoked to process the "DeleteCharacterAfter" or 
 *      "dca" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
DeleteCharacterAfter_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  if( argc != 1 ) {
    interp->result = "Usage: DeleteCharacterAfter";
    return TCL_ERROR;
  }

  if( Tcl_Eval(interp, "MoveCursorRight") == TCL_ERROR )
    return TCL_ERROR;

  return Tcl_Eval(interp, "DeleteCharacterBefore");
}



/*
 *----------------------------------------------------------------------
 *
 * DocumentConfigure_Cmd --
 *
 *      This procedure is invoked to process the "DocConfigure" or 
 *      "dc" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
DocumentConfigure_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i;
  int ma, wi, he, ti, cr;
  BOOL redraw;

  if( argc < 3 || (argc&1) == 0 ) {
    Tcl_AppendResult(interp,
		     "wrong # args: should be \"", argv[0], " options values\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  ti = cr = ma = wi = he = 0;
  redraw = FALSE;

  for(i=1; i<argc; i+=2) {
    if( argv[i][0] != '-' )
      goto error;

    if( argv[i][1] == 'm' && strcmp(argv[i], "-margins") == 0 )      ma = i+1;
    else if( argv[i][1] == 'w' && strcmp(argv[i], "-width") == 0 )   wi = i+1;
    else if( argv[i][1] == 'h' && strcmp(argv[i], "-height") == 0 )  he = i+1;
    else if( argv[i][1] == 't' && strcmp(argv[i], "-title") == 0 )   ti = i+1;
    else if( argv[i][1] == 'o' && strcmp(argv[i], "-owner") == 0 ) cr = i+1;    
    else {
    error:
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be -margins, -width, -height, -title or -owner",
		       (char *)NULL);
      return TCL_ERROR;
    }
  }
  if( ma != 0 ) {
    int m[4];

    if( Tcl_GetMargins(interp, argv[ma], m) != TCL_OK )
      return TCL_ERROR;
    
    current.doc->set_tmargin(m[0]);
    current.doc->set_bmargin(m[1]);
    current.doc->set_lmargin(m[2]);
    current.doc->set_rmargin(m[3]);
  }
  if( wi != 0 ) {
    if( Tcl_GetPixels(interp, argv[wi], &wi) != TCL_OK )
      return TCL_ERROR;
    current.doc->set_width(wi);
    redraw = TRUE;
  }
  if( he != 0 ) {
    if( Tcl_GetPixels(interp, argv[he], &he) != TCL_OK )
      return TCL_ERROR;
    current.doc->set_ascent(he);
    redraw = TRUE;
  }
  if( ti != 0 ) {
    current.doc->set_title(argv[ti]);
#ifdef HAVE_GUI
    GUI_Set_DocumentTitle(argv[ti]);
#else
    Page *page;
    page = (Page *)current.doc->get_child(0);
    XStoreName(papyrus->get_disp(), page->get_win(), argv[ti]);
#endif
  }
  if( cr != 0 ) {
    current.doc->set_owner(argv[cr]);
  }

  if( redraw == TRUE && current.shape != NULL ) {
    Page *page;

    page = current.shape->get_page_parent();
    page->set_to_redraw(REDRAW_ME);

    ((Line *)page->get_child(0))->set_to_redraw(REDRAW_PAGE);
    current.doc->recompute(current);
    current.doc->draw_frame(0,0);
  }
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * DocumentQuery_Cmd --
 *
 *      This procedure is invoked to process the "DocumentQuery" or 
 *      "dq" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
DocumentQuery_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i;
  char result[128];


  if( argc < 2 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], " options\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  for(i=1; i<argc; i++) {
    if( argv[i][0] == 'm' && strcmp(argv[i], "margins") == 0 ) {
      sprintf(result, "%d %d %d %d",
	      (int)current.doc->get_tmargin(),
	      (int)current.doc->get_bmargin(),
	      (int)current.doc->get_lmargin(),
	      (int)current.doc->get_rmargin());
      Tcl_AppendElement(interp, result);

    } else if( argv[i][0] == 'w' && strcmp(argv[i], "width") == 0 ) {
      sprintf(result, "%d ", current.doc->get_width());
      Tcl_AppendResult(interp, " ", result, (char *)NULL);

    } else if( argv[i][0] == 'h' && strcmp(argv[i], "height") == 0 ) {
      sprintf(result, "%d ", current.doc->get_height());
      Tcl_AppendResult(interp, " ", result, (char *)NULL);

    } else if( argv[i][0] == 't' && strcmp(argv[i], "title") == 0 ) {
      Tcl_AppendElement(interp, current.doc->get_title());

    } else if( argv[i][0] == 'o' && strcmp(argv[i], "owner") == 0 ) {
      Tcl_AppendElement(interp, current.doc->get_owner());

    } else {
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be margins, width, height, title or owner",
		       (char *)NULL);
      return TCL_ERROR;
    }
  }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * DocumentOutput_Cmd --
 *
 *      This procedure is invoked to process the "DocOutput" or 
 *      "do" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
DocumentOutput_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int mode;

  if( argc != 3 ) {
    Tcl_AppendResult(interp,
		     "wrong # args: should be \"", argv[0], " format channelId\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  channel = Tcl_GetChannel(interp, argv[1], &mode);
  if( channel == (Tcl_Channel) NULL ) {
    return TCL_ERROR;
  }
  if( (mode & TCL_WRITABLE) == 0 ) {
    Tcl_AppendResult(interp, "channel \"", argv[1],
		     "\" wasn't opened for writing", (char *) NULL);
    return TCL_ERROR;
  }
  
  if( argv[2][0] == 'a' && strcmp(argv[2], "ascii") == 0 )
    return current.doc->output_ascii();
  
  else if( argv[2][0] == 'p' && strcmp(argv[2], "papyrus") == 0 )
    return current.doc->output_papyrus();
  
  else if( argv[2][0] == 'p' && strcmp(argv[2], "postscript") == 0 )
    return current.doc->output_ps();
  
  Tcl_AppendResult(interp,"Error: bad format \"",
		   argv[2], "\": must be papyrus, postscript or ascii", (char *)NULL);

  return TCL_ERROR;
}



/*
 *----------------------------------------------------------------------
 *
 * InsertCharacter_Cmd --
 *
 *      This procedure is invoked to process the "InsertCharacter" or 
 *      "ic" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
InsertCharacter_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  if( argc != 2 ) {
    interp->result = "Usage: InsertCharacter char";
    return TCL_ERROR;
  }

  if( current.shape->can_append() == FALSE )
    Add_Empty_WordSegment();    

  current.shape->get_line_parent()->clear_frame();
  current.shape->insert_children((Container **)&argv[1][0], 1, current.pos);
  current.shape->can_fit(current);
  
  Tcl_Eval(interp, "MoveCursorRight");
  
  if( interactive_flag == TRUE ) {
    ((Word *)current.shape->get_parent())->format_frame(current);
    current.doc->draw_frame(0,0);
  }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * InsertString_Cmd --
 *
 *      This procedure is invoked to process the "InsertString" or 
 *      "is" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
InsertString_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i, cpt;

  if( argc != 2 ) {
    interp->result = "Usage: InsertString string";
    return TCL_ERROR;
  }

  current.shape->get_line_parent()->clear_frame();

  CALL(
       for(i=0, cpt=0; i<strlen(argv[1]); i++)
         if( argv[1][i] == '\n' )
           argv[1][i] = ' ';
         else
	   if( argv[1][i] == ' ' ) {
	     
	     if( current.shape->can_append() == FALSE )
	       Add_Empty_WordSegment();
	     
	     if( current.attr->has_mark() )
	       Update_Shape_Attributes(current);
	     
	     current.shape->insert_children((Container **)(argv[1]+cpt), i-cpt, current.pos);
	     current.pos += (i-cpt);
	     current.shape->can_fit(current);
	     
	     /*
	      * Look if 'can_fit' have changed the current position
	      * If true, it seems that the inserted string has some
	      * attributes
	      */
	     if( current.pos == 0 )
	       Tcl_Eval(interp, "MoveCursorNextWord");
	     
	     cpt = i+1;
	     Tcl_Eval(interp, "NewWord");
	   }
       );
  
  if( current.shape->can_append() == FALSE )
    Add_Empty_WordSegment();

  if( current.attr->has_mark() )
    Update_Shape_Attributes(current);
  
  current.shape->insert_children((Container **)(argv[1]+cpt), i-cpt, current.pos);
  current.shape->can_fit(current);
  current.pos += (i-cpt);

  current.doc->draw_frame(0,0);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * InsertImage_Cmd --
 *
 *      This procedure is invoked to process the "InsertImage" or 
 *      "ii" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
InsertImage_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Image *image;
  int offset;

  if( argc != 2 ) {
    interp->result = "Usage: InsertImage image";
    return TCL_ERROR;
  }

  image = new Image;
  CALL( image->insert_children((Container **)argv[1], 1, current.pos) );
  
  if( image->get_children_num() != 1 ) {
    delete image;
    return TCL_ERROR;
  }


  current.shape->get_line_parent()->clear_frame();
  CALL( Tcl_Eval(interp, "NewWord") );

  offset = current.shape->get_offset();
  current.shape->get_parent()->insert_children((Container **)&image, 1, offset);
  current.pos = 0;
  current.shape = image;
  current.pos++;

  image->recompute_size();
  image->can_fit(current);

  if( ((Word *)current.shape->get_parent())->get_width() == current.shape->get_width() ) {
    current.shape->get_parent()->delete_children(1, offset+1);
    current.doc->draw_frame(0,0);
  } else
    if( is_last_of_word(current) == FALSE )
      Tcl_Eval(interp, "NewWord");
  
  return TCL_OK;
}
