/* 
 * ps_output.cc --
 *
 *      This file contains all the methods used to translate the
 *      document into PostScript, and several utilities.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>

extern "C" {
#include "../misc/parseAFM.h"
#include "../tcl/mytcl.h"
}

#include "frame.h"
#include "document.h"
#include "page.h"
#include "line.h"
#include "wordSegment.h"
#include "image.h"
#include "util.h"
#include "output.h"
#include "papyrus.h"


static Tcl_DString outputString;
static int pageheight;

static int fontsize;
static char *psname;
static int bottommargin;
static int x;

static AFMFontInfo *fi;




/*
 *----------------------------------------------------------------------
 *
 * Load_AFM_File --
 *
 *      This function parses an Adobe Font Metrics file.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Load_AFM_File(char *filename) {

  FILE *fp;
  char name[128];  

  sprintf(name, "/usr/local/lib/Papyrus/%s", filename);

  fp = fopen(name, "r");
  if( fp == NULL ) {
    Tcl_AppendResult(theInterp, "Load_AFM_File: can't find: %s\n", name, (char *)NULL);
    return TCL_ERROR;
  }

  switch( parseFile(fp, &fi, P_W|P_P) ) {
  case parseError:
    Tcl_AppendResult(theInterp, "Load_AFM_File: problem in parsing the AFM File");
    break;

  case ok:
    fclose(fp);
    break;

  case earlyEOF:
    Tcl_AppendResult(theInterp, "Load_AFM_File: the AFM File ended prematurely");
    break;

  case storageProblem:
    Tcl_AppendResult(theInterp, "Load_AFM_FILE: problem allocating storage");
    break;

  default:
    break;
  }
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * AFM_GetWidth --
 *
 *      Given a string and its length, this function return its width
 *      in dpi.
 *
 * Results:
 *      Returns the length in dpi.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AFM_GetWidth(char *string, int num) {
  
  int i, len;

  len = 0;
  for(i=0; i<num; i++)
    len += fi->cwi[(unsigned char)string[i]];
  len = (int)(((float)len) / 1000.00 * ((float)fontsize));
  return len;
}



/*
 *----------------------------------------------------------------------
 *
 * PS_Insert_File --
 *
 *      This function inserts a PostScript file in the current file
 *      output.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
PS_Insert_File(char *filename) {
  FILE *f;
  char string[256];
  char name[128];
  
  sprintf(name, "/usr/local/lib/Papyrus/%s", filename);

  f = fopen(name, "r");
  if( f == NULL ) {
    Tcl_AppendResult(theInterp, "can't open: %s\n", name, (char *)NULL);
    return TCL_ERROR;
  }
  
  while( fgets(string, 255, f) != NULL )
    OUTS(string);
  fclose(f);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * PS_Document_Header --
 *
 *      This function generates the document header of the PostScript
 *      file, conforming to Adobe-2.0 specifications.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
PS_Document_Header(Document *doc) {

  int i, len;
  FontFamily *ffPtr;


  OUTS("%%!PS-Adobe-2.0\n");
  OUTS("%%%%Title: ");
  if( doc->get_title() != NULL )
    OUTS(doc->get_title());
  else
    OUTS("(no title)");
  OUTS("\n%%%%Creator: Papyrus\n");
  OUTS("%%%%CreationDate: ");
  OUTS(Time_to_String());
  OUTS("%%%%For: ");
  if( doc->get_owner() != NULL )
    OUTS(doc->get_owner());
  else
    OUTS(getenv("LOGNAME"));
  OUTS("\n%%%%Orientation: Portrait\n");
  OUTS("%%%%DocumentMedia: A4 596 842\n");
  OUTS("%%%%BoundingBox: 0 0 596 842\n");
  OUTS("%%%%Pages: ");
  OUTI(doc->get_children_num());
  OUTS("\n%%%%PageOrder: Ascend\n");
  OUTS("%%%%DocumentFonts:");


  len = strlen("%%%%DocumentFonts:");
  for(i=0; i<4; i++) {
    ffPtr = papyrus->get_first((FontStyle)i);

    while( ffPtr != NULL ) {
      len += strlen(ffPtr->get_psName())+1;
      if( len > 80 ) {
	OUTS("\n%%%%+");
	len = strlen("%%%%+") + strlen(ffPtr->get_psName())+1;
      }
      OUTS(" ");
      OUTS(ffPtr->get_psName());
      ffPtr = papyrus->get_next();
    }
  }

  OUTS("\n%%%%EndComments\n");
  OUTS("%%%%BeginProcSet: \n");
  OUTS("/s  { show } bind def\n");
  OUTS("/hr { 0 rmoveto } bind def\n");
  OUTS("/vr { 0 exch rmoveto } bind def\n");  
  OUTS("/m  { moveto } bind def\n");
  OUTS("/sp { showpage } bind def\n");
  OUTS("/ff { findfont } bind def\n");
  OUTS("/sf { scalefont setfont } bind def\n");
  OUTS("%%%%EndProcSet\n");

  OUTS("%%%%BeginProcSet: iso_encode.ps\n");
  if( PS_Insert_File("iso_encode.ps") != TCL_OK )
    return TCL_ERROR;

  for(i=0; i<4; i++) {
    ffPtr = papyrus->get_first((FontStyle)i);

    while( ffPtr != NULL ) {
      OUTS("/");
      OUTS(ffPtr->get_psName());
      OUTS(" /");
      OUTS(ffPtr->get_psName());
      OUTS(" isovec ReEncode\n");
      ffPtr = papyrus->get_next();
    }
  }

  OUTS("%%%%EndProcSet\n");
  OUTS("%%%%EndProlog\n");
  OUTS("%%%%BeginSetup\n");
  OUTS("%%%%Feature: *Resolution 300dpi\n");
  OUTS("%%%%PaperSize: a4\n");
  OUTS("a4\n");
  OUTS("%%%%EndPaperSize\n");  
  OUTS("%%%%EndSetup\n");

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * PS_Document_Footer --
 *
 *      This function generates the document footer of the PostScript
 *      file, conforming to Adobe-2.0 specifications.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
PS_Document_Footer(void) {

  OUTS("\n%%%%Trailer\n");
  OUTS("%%end\n");
  OUTS("%%%%EOF\n");
}



/*
 *----------------------------------------------------------------------
 *
 * PS_Page_Header --
 *
 *      This function generates the page header of the PostScript
 *      file, conforming to Adobe-2.0 specifications.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
PS_Page_Header(int num) {

  OUTS("%%%%Page: ");
  OUTI(num); OUTS(" "); OUTI(num);
  OUTS("\n%%%%PageFonts:\n");
  OUTS("%%%%PageBoundingBox: 0 0 596 842\n");
  OUTS("%%%%BeginPageSetup\n");
  OUTS("%%%%EndPageSetup\n");
}



/*
 *----------------------------------------------------------------------
 *
 * PS_Page_Footer --
 *
 *      This function generates the page footer of the PostScript
 *      file, conforming to Adobe-2.0 specifications.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
PS_Page_Footer(void) {

  OUTS("\nsp\n");
  OUTS("%%%%PageTrailer\n");
}



/*
 *----------------------------------------------------------------------
 *
 * WordSegment::output_ps --
 *
 *      This function generates the contents of the 'WordSegment'
 *      object into PostScript format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      The x variable is updated. x is used to compute the length
 *      of the current line.
 *
 *----------------------------------------------------------------------
 */

int
WordSegment::output_ps(void) {

  int i, j;
  FontItem *fi;
  char string[32];

  if( get_children_num() == 0 )
    return TCL_OK;

  fi = get_ws_font();

  OUTS2(outputString, " (");
  
  j = 0;
  string[31] = '\0';

  for(i=0; i<_children_num; i++) {
    if( j == 31 ) {
      OUTS2(outputString, string);
      j = 0;
    }
    if( ((char *)_children)[i] == ')' || ((char *)_children)[i] == '(' )
      string[j++] = '\\';
    string[j++] = ((char *)_children)[i];
  }
  if( j > 0 ) {
    string[j] = '\0';
    OUTS2(outputString, string);
  }
  OUTS2(outputString, ")s");

  x += AFM_GetWidth((char *)_children, _children_num);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Image::output_ps --
 *
 *      This function generates the contents of the 'Image'
 *      object into PostScript format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Image::output_ps(void) {
 
  FLUSH(outputString, "InsertString \"", "\"\n");
  OUTS("InsertImage ");
  OUTS(_filename);
  OUTS("\n");
  
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Line::output_ps --
 *
 *      This function generates the contents of the 'Line'
 *      object into PostScript format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Line::output_ps(void) {
  
  int i;
  int spacing, xoffset;
  FontItem *fi;
  char string[32];

  fi = (FontItem *)get_para()->query(STYLE_FONT);

  spacing = papyrus->xpixels_to_points(fi->get_spacing());
  x = 0;
  sprintf(string, " %d hr", spacing);

  for(i=0; i<get_children_num()-1; i++) {
    ((Frame *)get_child(i))->output_ps();
    OUTS2(outputString, string);
  }

  ((Frame *)get_child(i))->output_ps();
  x += (spacing * i);


  switch( (StyleAlignType)_para->query(STYLE_ALIGNMENT) ) {
    
  case STYLE_ALIGN_RIGHT:
    xoffset = papyrus->xpixels_to_points(((Frame *)get_parent())->get_width());
    xoffset -= x;
    sprintf(string, " %d hr", xoffset);
    break;

  case STYLE_ALIGN_CENTER:
    xoffset = get_max_width() - get_real_width();
    xoffset = papyrus->xpixels_to_points(get_max_width()) - x;
    xoffset /= 2;
    sprintf(string, " %d hr", xoffset);
    break;

  case STYLE_ALIGN_FULL:
    break;

  default:
    string[0] = '\0';
    break;

  }
  FLUSH(outputString, string, "\n");
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Page::output_ps --
 *
 *      This function generates the contents of the 'Page'
 *      object into PostScript format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Page::output_ps(void) {

  int i, len, x, y;
  Line *line;
  Paragraph *para;


  PS_Page_Header(get_offset()+1);

  x = papyrus->xpixels_to_points( get_lmargin() );
  y = papyrus->ypixels_to_points( get_tmargin() );

  /*
   * Lets force Paragraph to load & scale font
   */
  psname = NULL;
  fontsize = -1;
  bottommargin = 0;


  OUTI(x); OUTS(" "); OUTI(pageheight-y);
  OUTS(" m\n");

  para = NULL;

  for(i=0; i<get_children_num(); i++) {
    line = (Line *)get_child(i);
    
    if( line->get_para() != para ) {
      para = line->get_para();
      para->output_ps();
    }

    line->output_ps();

    len = line->get_height() + 2; // INTERLINE
    len = papyrus->xpixels_to_points(len);
    y += len;

    OUTS(" "); OUTI(x); OUTS(" "); OUTI(pageheight-y);
    OUTS(" m\n");
  }

  PS_Page_Footer();
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Document::output_ps --
 *
 *      This function generates the contents of the 'Document'
 *      object into PostScript format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Document::output_ps(void) {

  Tcl_DStringInit(&outputString);

  pageheight = papyrus->ypixels_to_points(get_height());
  if( Load_AFM_File("Courier.afm") != TCL_OK )
    return TCL_ERROR;

  if( PS_Document_Header(this) != TCL_OK )
    return TCL_ERROR;

  Frame::output_ps();
  PS_Document_Footer();

  freeStorage(fi);
  Tcl_DStringFree(&outputString);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Frame::output_ps --
 *
 *      This function generates the contents of the 'Frame'
 *      object into PostScript format. This function is used
 *      by all the object which have no output_ps method.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Frame::output_ps(void) {
  
  int i;

  for(i=0; i<_children_num; i++)
    ((Frame *)_children[i])->output_ps();
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Paragraph::output_ps --
 *
 *      This function generates the contents of the 'Paragraph'
 *      object into PostScript format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Paragraph::output_ps(void) {

  FontItem *fPtr;
  BOOL psn, fs;
  int margin;


  psn = fs = FALSE;
  fPtr = (FontItem *)query(STYLE_FONT);

  if( psname == NULL || strcmp(fPtr->get_psName(), psname) ) {
    psname = fPtr->get_psName();
    psn = TRUE;
  }
  if( fPtr->get_size() != fontsize ) {
    fontsize = fPtr->get_size();
    fs = TRUE;
  }

  if( psn == TRUE || fs == TRUE ) {
    OUTS("/"); OUTS(psname); OUTS(" ff ");
    OUTI(fontsize); OUTS(" sf\n");
  }
  margin = (int)query(STYLE_FLINE_MARGIN);
  if( margin > 0 ) {
    OUTS(" ");
    OUTI(papyrus->xpixels_to_points(margin));
    OUTS(" hr");
  }

  if( has_mark(STYLE_MARGINS) ) {
    if( has_mark(STYLE_TOP_MARGIN) ) {
      margin = (int)query(STYLE_TOP_MARGIN);
      OUTS(" ");
      OUTI(papyrus->ypixels_to_points(margin));
      OUTS(" vr");
    }

    if( bottommargin > 0 ) {
      OUTS(" ");
      OUTI(bottommargin);
      OUTS(" vr");
    }

    if( has_mark(STYLE_BOTTOM_MARGIN) ) {
      margin = (int)query(STYLE_BOTTOM_MARGIN);
      bottommargin = papyrus->ypixels_to_points(margin);
    }

    if( has_mark(STYLE_LEFT_MARGIN) ) {
      margin = (int)query(STYLE_LEFT_MARGIN);
      OUTS(" ");
      OUTI(papyrus->xpixels_to_points(margin));
      OUTS(" hr");
    }
  }
  return TCL_OK;
}


