/* 
 * page.cc --
 *
 *      This file contains the definitions of the 'Page' class
 *      methods.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifdef HAVE_GUI
extern "C" {
  
#include <X11/Intrinsic.h>
  extern void keyPress_handler(Widget, XtPointer, XEvent *, Boolean *);
  extern Widget GUI_CreateNewPage(int, Dimension, Dimension);
}
extern void buttonPress_handler(Widget, XtPointer, XEvent *, Boolean *);
extern void expose_handler(Widget, XtPointer, XEvent *, Boolean *);

static pageNumber = 0;
#endif


#include "page.h"
#include "word.h"
#include "wordSegment.h"
#include "papyrus.h"




/*
 *----------------------------------------------------------------------
 *
 * Page --
 *
 *      This procedure is invoked every time a Page is created
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized, and a window is created.
 *
 *----------------------------------------------------------------------
 */

Page::Page() {

#ifdef HAVE_GUI

  Widget page;

  page = GUI_CreateNewPage(pageNumber++,
			   (Dimension)papyrus->xmm_to_pixels(210),
			   (Dimension)papyrus->xmm_to_pixels(297));
  _win = XtWindow(page);
  papyrus->set_win(_win);

  XtAddEventHandler(page, ExposureMask,    False, expose_handler, (XtPointer)this);
  XtAddEventHandler(page, KeyPressMask,    False, keyPress_handler,    0);
  XtAddEventHandler(page, ButtonPressMask, False, buttonPress_handler, 0);

#else

  s = DefaultScreen(papyrus->get_disp());

  _win = XCreateSimpleWindow(papyrus->get_disp(),
			     RootWindow(papyrus->get_disp(), s),
			     0, 0, 
			     papyrus->xmm_to_pixels(210),
			     papyrus->xmm_to_pixels(297),
			     2,
			     BlackPixel(papyrus->get_disp(), s),
			     WhitePixel(papyrus->get_disp(), s));
  
  if( _win == 0 ) {
    fprintf(stderr, "Page: Can't open window\n");
    exit(1);
  }

  XMapWindow(papyrus->get_disp(), _win);
  papyrus->set_win(_win);

  XEvent evt;
  
  XSelectInput(papyrus->get_disp(), papyrus->get_win(), ExposureMask );
  XNextEvent(papyrus->get_disp(), &evt);

  XSelectInput(papyrus->get_disp(), papyrus->get_win(),
               ButtonPressMask | KeyPressMask);
#endif

}



/*
 *----------------------------------------------------------------------
 *
 * ~Page --
 *
 *      This procedure is invoked every time a Page is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The window of the Page is destroyed.
 *
 *----------------------------------------------------------------------
 */
Page::~Page() {

#ifdef HAVE_GUI
  XtUnmanageChild(XtWindowToWidget(papyrus->get_disp(), _win));
  XtDestroyWidget(XtWindowToWidget(papyrus->get_disp(), _win));
  pageNumber--;
#else
  XDestroyWindow(papyrus->get_disp(), _win);
#endif
  XFlush(papyrus->get_disp());
}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate a
 *      Page without known anything about it.
 *
 * Results:
 *      A new Page is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Page::new_of_same_type(void) {

  return new Page;
}



/*
 *----------------------------------------------------------------------
 *
 * delete_children --
 *
 *      This procedure deletes children from position 'where' to
 *      'where'+'howmany'.
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if "where" is not equal to "_children_num".
 *
 *----------------------------------------------------------------------
 */

void
Page::delete_children(int howmany, int where) {

  int i, offset;

  for(i=where; i<_children_num-howmany; i++)
    _children[ i ] = _children[ i+howmany ];

  _children_num -= howmany;

  /*
   * Build a document tree
   */
  if( _children_num <= 0 ) {
    offset = get_offset();

    if( offset == 0 ) {
      Line *line;
      Word *word;
      WordSegment *ws;
      
      line = new Line( new Paragraph );
      insert_children((Container **)&line, 1, 0);
      
      word = new Word;
      line->insert_children((Container **)&word, 1, 0);
      
      ws = new WordSegment();
      word->insert_children((Container **)&ws, 1, 0);
      
      current.shape = (Shape *)ws;
      current.pos = 0;
    } else {
      _parent->delete_children(1, offset);
      delete this;
      return;
    }
  } 
  recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * recompute_size --
 *
 *      This function recomputes the Page dimensions (height & width).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The dimensions are updated.
 *
 *----------------------------------------------------------------------
 */

void
Page::recompute_size(BOOL) {

  unsigned int h;
  int i;

  h = 0;

  for(i=0; i<_children_num; i++) {
    h += ((Line *)_children[i])->get_height();
    h += INTERLIGNE;
  }
  set_width( ((Line *)_children[0])->get_max_width() );
  set_ascent( h );
}



/*
 *----------------------------------------------------------------------
 *
 * debug --
 *
 *      This function gives some informations about the Page.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Page::debug(int depth) {

  int i;

  for(i=0; i<depth; i++)
    fprintf(stderr, "    ");

  fprintf(stderr, "%s (%d) (%d : %d)  %c\n", type(),
	  _children_num, UNZOOM(get_height()), UNZOOM(get_max_height()),
	  (has_to_redraw() == REDRAW_ME) ? 'X' : ' ');

  for(i=0; i<_children_num; i++)
    _children[ i ]->debug( depth+1 );
}



/*
 *----------------------------------------------------------------------
 *
 * can_fit --
 *
 *      This function looks if the Page is not too big.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      If the Page is too big then the last Lines are passed to
 *      the next Page.
 *
 *----------------------------------------------------------------------
 */

void
Page::can_fit(ThePosition &cur) {

  int height, i;
  Page *next, *new_page = NULL;
  Line *child;


  height = get_height();
  i = get_children_num();

  while( height > get_max_height() && i > 0 )
    height -= (((Line *)_children[--i])->get_height() + INTERLIGNE);
  
  if( i == get_children_num() )
    return;

  child = (Line *)_children[i-1];
  child->set_to_redraw(REDRAW_PAGE);

  next = (Page *)get_next_same_container();
  if( next == NULL ) {
    new_page = (Page *)split_container(i);
  } else {
    move_children(get_children_num()-i, i, next, 0);
    recompute_size();
  }

  if( next == NULL )
    next = new_page;

  next->can_fit(cur);
  next->set_to_redraw(REDRAW_ME);
}



/*
 *----------------------------------------------------------------------
 *
 * format_frame --
 *
 *      This function looks if the Page is big enough.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      If the current Page can accept more Lines from the next
 *      Page, they are moved.
 *
 *----------------------------------------------------------------------
 */

void
Page::format_frame() {

  Page *next;
  int height;
  int i;

  next = (Page *)get_next_same_container();
  if( next == NULL )
    return;

  height = get_height();
  i = 0;

  while( height <= get_max_height() ) {
    if( i == next->get_children_num() ) {
      merge_container(next);
      next = (Page *)get_next_same_container();

      i = 0;
      if( next == NULL )
	break;
      next->set_to_redraw(REDRAW_PAGE);
    }
    height += (((Page *)next->get_child(i++))->get_height() + INTERLIGNE);
  }
  
  if( i > 1 ) {
    next->move_children(i-1, 0, this, get_children_num());
    ((Frame *)next->get_child(0))->set_to_redraw(REDRAW_PAGE);
  }
  
  next = (Page *)get_next_same_container();
  
  if( i > 0 && next != NULL ) {
    next->set_to_redraw(REDRAW_ME);
    next->format_frame();
  }
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the Page.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Page::draw_frame(int x, int y) {

  int i;
  Line *line;

  if( interactive_flag == FALSE )
    return;

  set_to_redraw(REDRAW_NONE);
  papyrus->set_win( _win );

  x = get_lmargin();
  y = get_tmargin();
  
  
  for(i=0; i<_children_num; i++) {
      
    line = (Line *)_children[i];
    
    switch( line->has_to_redraw() ) {

    case REDRAW_PARA:
      redraw_para(i, y);
      goto end;
      break;

    case REDRAW_PAGE:
      redraw_page(i, y);
      goto end;
      break;

    case REDRAW_ME:
      line->draw_frame(x, y, line->is_last_of_para());
      break;

    default:
      break;
    }
    y += line->get_height();
    y += INTERLIGNE;
  }

 end:
  if( papyrus->mask_doc() == TRUE )
    draw_margins();
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the Page, but only Line which are in the
 *      same Paragraph than 'child(i)'.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Page::redraw_para(int i, int &y) {
  
  Paragraph *cur_para;
  BOOL end_of_para;
  Line *line;

  cur_para = ((Line *)get_child(i))->get_para();
  end_of_para = FALSE;   /* To prevent warnings compilation */
  
  while( i<_children_num ) {
    line = (Line *)_children[i];

    if( line->get_para() != cur_para )
      break;

    if( papyrus->mask_doc() == TRUE )
      end_of_para = line->is_last_of_para();

    line->draw_frame(get_lmargin(), y, end_of_para);
    
    y += line->get_height();
    y += INTERLIGNE;
    i++;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * draw_page --
 *
 *      This function draws all the Page, without tests.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Page::redraw_page(int i, int &y) {

  BOOL end_of_para;
  Line *line;

  while( i<_children_num ) {
    line = (Line *)_children[i];
    
    end_of_para = line->is_last_of_para();    
    line->draw_frame(get_lmargin(), y, end_of_para);
    
    y += line->get_height();
    y += INTERLIGNE;
    i++;
  }
 
  /*
   * If the updated page is smaller (vertically) then clear
   * the rest of the page (from here to the bottom margin).
   */
  papyrus->clear_area(get_lmargin(), y,
		    ((Frame *)_parent)->get_width(),
		    get_max_height() - y + get_tmargin());
}



/*
 *----------------------------------------------------------------------
 *
 * mark_to_redraw --
 *
 *      This function marks this Page to be redrawed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Page::mark_to_redraw() {

  set_to_redraw(REDRAW_ME);
}



/*
 *----------------------------------------------------------------------
 *
 * frame_to_xy --
 *
 *      This function returns the coordinates of the 'offset' child
 *      of the Page.
 *
 * Results:
 *      Returns the coordinates of the 'offset'.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Page::frame_to_xy(int &x, int &y, int offset) {
  
  int i;

  y = get_tmargin();
  x = get_lmargin();

  for(i=0; i<offset; i++) {
    y += ((Frame *)get_child(i))->get_height();
    y += INTERLIGNE;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * xy_to_frame --
 *
 *      This function returns the 'offset' which has the correct
 *      coordinates.
 *
 * Results:
 *      Returns the 'offset' child specified.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Frame *
Page::xy_to_frame(int x, int y, int &offset) {
  
  int yy, i;

  yy = get_tmargin();
  x -= get_lmargin();


  if( x < 0 )
    x = 0;

  for(i=0; i<_children_num; i++)
    if( (yy + INTERLIGNE/2 + ((Frame *)_children[i])->get_height()) <= y ) {
      yy += ((Frame *)_children[i])->get_height();
      yy += INTERLIGNE;
    } else {
      y -= yy;
      y -= INTERLIGNE;
      return ((Frame *)_children[i])->xy_to_frame(x, y, offset);
    }
  return ((Frame *)_children[i-1])->xy_to_frame(x, y, offset);
}



/*
 *----------------------------------------------------------------------
 *
 * clear_frame_para --
 *
 *      This function clears (in the screen) all the lines which are
 *      in the same Paragraph than 'first_line'.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
Page::clear_frame_para(Line *first_line) {

  Line *next, *next_tmp;
  int xpos, ypos, xpos2, ypos2;

  next = first_line;
  
  for(;;) {
    next_tmp = (Line *)next->get_next_same_container();
    
    if( first_line->has_same_parent(next_tmp) == FALSE )
      break;
    if( next_tmp->get_para() != first_line->get_para() )
      break;

    next = next_tmp;
  }

  /*
   * Clear the lines between 'first_line' and 'next'
   * 'first_line' and 'next' are also cleared.
   */
    
  frame_to_xy(xpos, ypos, get_child_offset(first_line));
  frame_to_xy(xpos2, ypos2, get_child_offset(next));
  
  first_line->set_to_redraw(REDRAW_PARA);

  return;
}



/*
 *----------------------------------------------------------------------
 *
 * draw_margins --
 *
 *      This function draws the margins of the Page.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
Page::draw_margins(void) {
  
  unsigned short x, y, h;
  GC gc;

  /*
   * 'papyrus->mask_doc()' not positionned ( ie == FALSE)
   * means that margins have to be cleared (this is invoked
   * when 'papyrus->mask_doc()' is changed.
   */

  gc = (papyrus->mask_doc() == TRUE) ? 
    papyrus->get_dashed_gc() : papyrus->get_clear_gc();

  x = get_lmargin();
  y = get_tmargin();
  h = ((Frame *)get_parent())->get_height();

  XDrawLine(papyrus->get_disp(), papyrus->get_win(), gc,
	    x-1, 0,
	    x-1, h);
  
  XDrawLine(papyrus->get_disp(), papyrus->get_win(), gc,
	    get_width() + x+1, 0,
	    get_width() + x+1, h);
  
  XDrawLine(papyrus->get_disp(), papyrus->get_win(), gc,
	    0, y-1,
	    get_width() + x + get_rmargin(), y-1);
  
  XDrawLine(papyrus->get_disp(), papyrus->get_win(), gc,
	    0,  get_max_height() + get_bmargin()+1,
	    get_width() + x + get_rmargin(), get_max_height()+y+1);
}
