/* 
 * image.cc --
 *
 *      This file contains the definitions of the 'Image' class
 *      methods.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


extern "C" {
#include <stdio.h>
#include <string.h>
}

#include "image.h"
#include "papyrus.h"



/*
 *----------------------------------------------------------------------
 *
 * Image --
 *
 *      This procedure is invoked every time an Image is created.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Image::Image() {
}



/*
 *----------------------------------------------------------------------
 *
 * ~Image --
 *
 *      This procedure is invoked every time an Image is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Private datas are deleted.
 *
 *----------------------------------------------------------------------
 */

Image::~Image() {

  delete (char *)_filename;
  _children_num = 0;
}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate an
 *      Image without known anything about it.
 *
 * Results:
 *      A new Image is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Image::new_of_same_type(void) {

  return new Image;
}



/*
 *----------------------------------------------------------------------
 *
 * alloc_offset --
 *
 *      This procedure is invoked everytime a child is inserted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::alloc_offset(int) {
}



/*
 *----------------------------------------------------------------------
 *
 * move_children --
 *
 *      This function moves 'homany' children from position 'isrc'
 *      to container 'dst' at position 'idst'.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::move_children(int, int, Container *, int) {
}



/*
 *----------------------------------------------------------------------
 *
 * delete_children --
 *
 *      This procedure deletes the only children of the Image.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Child and the Image are deleted.
 *
 *----------------------------------------------------------------------
 */

void
Image::delete_children(int, int) {

  _children_num = 0;
  _parent->delete_children(1, get_offset());
  delete this;
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * insert_children --
 *
 *      This procedure load an image file and puts it on the object.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The image file is loaded.
 *
 *----------------------------------------------------------------------
 */

void
Image::insert_children(Container **name, int, int) {

  unsigned short w, h;

  w = 0;

  _filename = strdup((char *)name);
  _children = (Container **)papyrus->get_xpm_file((char *)name, w, h);

  /*
   * Don't make the test on _children (==NULL) because
   * if the 'hide_images' flag is set (on XDisplay) the
   * pixmap is not load.
   */

  if( w == 0 )
    return;

  set_width( w );
  set_ascent( h );
  set_descent( 0 );

  _children_num = 1;

  if( interactive_flag == TRUE )
    recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * debug --
 *
 *      This function gives some informations about the Container.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::debug(int depth) {
  
  int i;

  for(i=0; i<depth; i++)
    fprintf(stderr, "    ");

  fprintf(stderr, 
	  "%s (%d) (%d %d) %s : <<image>>\n", type(), _children_num,
	  UNZOOM(get_width()), UNZOOM(get_height()), 
	  (_attr == NULL) ? "" : "X");
}



/*
 *----------------------------------------------------------------------
 *
 * can_fit --
 *
 *      This function looks if the Image is correct.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::can_fit(ThePosition &cur) {

  if( _parent != NULL )
    get_line_parent()->can_fit(cur, TRUE);
}



/*
 *----------------------------------------------------------------------
 *
 * recompute_size --
 *
 *      This function recomputes the Image dimensions.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::recompute_size(BOOL propagate) {

  if( (propagate == TRUE) && (_parent != NULL) )
    ((Frame *)_parent)->recompute_size();  
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the Image.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::draw_frame(int x, int y) {

  papyrus->draw_pixmap((Pixmap *)_children, x, y, get_width(), get_ascent());
}



/*
 *----------------------------------------------------------------------
 *
 * frame_to_xy --
 *
 *      This function returns the coordinates of the child (alone)
 *      of the Line.
 *
 * Results:
 *      Returns the coordinates of the 'child'.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Image::frame_to_xy(int &x, int &y, int child) {
  
  Container *parent;

  x = y = 0;

  parent = get_parent();
  if( parent == NULL )
    return;

  ((Frame *)parent)->frame_to_xy(x, y, get_offset());
  
  if( child == 1 )
    x += get_width();
}



/*
 *----------------------------------------------------------------------
 *
 * xy_to_frame --
 *
 *      This function returns the 'offset' which has the correct
 *      coordinates.
 *
 * Results:
 *      Returns 'this'.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Frame *
Image::xy_to_frame(int x, int y, int &offset) {

  offset = 0;

  return this;
}



/*
 *----------------------------------------------------------------------
 *
 * merge_container --
 *
 *      This is a convenience function to concatenate two Images.
 *      In fact, this function puts all the 'con' children at the end
 *      of the Image 'this'. But the Image object can't has more than
 *      one child.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The result is : this + con -> this
 *
 *----------------------------------------------------------------------
 */

void
Image::merge_container(Container *con) {

  con->move_children(con->get_children_num(), 0, this, get_children_num());
}



/*
 *----------------------------------------------------------------------
 *
 * get_absolute_attr --
 *
 *      This function returns the value of the AttrType 't' of the
 *      current WordSegment which has the 'attr' attributes.
 *      But none is done because the Image object doesn't have
 *      attributes.
 *
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None..
 *
 *----------------------------------------------------------------------
 */

void *
Image::get_absolute_attr(Attributes *attr, AttrType t) const {
  return NULL;
}
