/* 
 * frame.cc --
 *
 *      This file contains the definitions of the 'Frame' class
 *      methods.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include "frame.h"
#include "papyrus.h"



/*
 *----------------------------------------------------------------------
 *
 * Frame --
 *
 *      This procedure is invoked every time a Frame is created
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

Frame::Frame() {

  _size.width = 0;
  _size.ascent = 0;
  _size.descent = 0;
  _redraw = REDRAW_NONE;
}



/*
 *----------------------------------------------------------------------
 *
 * ~Frame --
 *
 *      This procedure is invoked every time a Frame is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Frame::~Frame() {
}



/*
 *----------------------------------------------------------------------
 *
 * insert_children --
 *
 *      This procedure inserts 'howmany' children ('what') at 'where'
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if 'where' is not equal to '_children_num'.
 *
 *----------------------------------------------------------------------
 */

void
Frame::insert_children(Container **what, int howmany, int where) {

  Container::insert_children(what, howmany, where);

  if( interactive_flag == TRUE )
    recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * mark_to_redraw --
 *
 *      This function marks this Frame to be redrawed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Frame::mark_to_redraw() {

  if( has_to_redraw() == REDRAW_NONE )
    set_to_redraw(REDRAW_ME);
  
  if( _parent != NULL )
    ((Frame *)_parent)->mark_to_redraw();
}



/*
 *----------------------------------------------------------------------
 *
 * delete_children --
 *
 *      This procedure deletes children from position 'where' to
 *      'where'+'howmany'.
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if "where" is not equal to "_children_num".
 *
 *----------------------------------------------------------------------
 */

void
Frame::delete_children(int howmany, int where) {

  BOOL del_flag = FALSE;

  if( get_children_num() - howmany <= 0 )
    del_flag = TRUE;
  
  Container::delete_children(howmany, where);
  
  if( del_flag == FALSE )
    if( interactive_flag == TRUE )
      recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * split_container --
 *
 *      This function splits the Frame, at 'offset' position.
 *
 * Results:
 *      The new Frame created (second part of the current).
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Frame::split_container(int offset) {

  Container *result;

  result = Container::split_container(offset);
  recompute_size();
  ((Frame *)result)->recompute_size();

  return result;
}


/*
 *----------------------------------------------------------------------
 *
 * recompute_size --
 *
 *      This function recomputes the Frame dimensions. But it does
 *      nothing because each derivated class redefines it.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Frame::recompute_size(BOOL) {
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the Frame. But it does nothing because
 *      each derivated class redefines it.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Frame::draw_frame(int, int) {
}



/*
 *----------------------------------------------------------------------
 *
 * frame_to_xy --
 *
 *      This function returns the coordinates of the 'child' child
 *      of the Line. But it does nothing because each derivated class
 *      redefines it.
 *
 * Results:
 *      Returns the coordinates of the 'child'.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Frame::frame_to_xy(int &x, int &y, int child) {
}



/*
 *----------------------------------------------------------------------
 *
 * xy_to_frame --
 *
 *      This function returns the 'child' which has the correct
 *      coordinates.
 *
 * Results:
 *      Returns the 'child' specified.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Frame *
Frame::xy_to_frame(int x, int y, int &offset) {

  int xx, i;
  
  xx = 0;

  
  for(i=0; i<_children_num; i++) {
    if( (xx + ((Frame *)_children[i])->get_width()) <= x )
      xx += ((Frame *)_children[i])->get_width();
    else {
      x -= xx;
      return ((Frame *)_children[i])->xy_to_frame(x, y, offset);
    }
  }
  return ((Frame *)_children[i-1])->xy_to_frame(x, y, offset);
}



/*
 *----------------------------------------------------------------------
 *
 * clear_frame --
 *
 *      This function clears the Container in the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Frame::clear_frame(void) {

  int x, y;

  if( has_to_redraw() != REDRAW_NONE )
    return;

  frame_to_xy(x, y, 0);
  papyrus->clear_area(x, y, get_width(), get_height()+2);
  set_to_redraw(REDRAW_ME);
}
