/* 
 * display.cc --
 *
 *      This file contains the definitions of the 'XDisplay'
 *      class methods.
 *
 * Copyright (C) 1996  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */


#ifdef HAVE_GUI
extern "C" {
 #include <X11/Intrinsic.h>

void GUI_Add_ErrorMessage(char *);
};
extern Widget toplevel;
#endif

extern "C" {
#include <stdio.h>
#include <X11/xpm.h>
}

#include "display.h"


#define MAX(x, y) ( (x > y) ? x : y )
#define X(x) (x-_destX)
#define Y(y) (y-_destY)


float zoom;



/*
 *----------------------------------------------------------------------
 *
 * XDisplay --
 *
 *      This method is invoked every time an XDisplay object is
 *      created
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Initialize some object datas, and some X11 ressources.
 *
 *----------------------------------------------------------------------
 */

XDisplay::XDisplay(Display *disp) {

  XGCValues gcv;
  Screen *s;

  zoom = 1;

  _disp = disp;
  _win = DefaultRootWindow(_disp);

  _screen = DefaultScreen(_disp);
  _root   = RootWindow(_disp, _screen);

  s = DefaultScreenOfDisplay(_disp);
  _xppm = (float)WidthOfScreen(s) / (float)WidthMMOfScreen(s);
  _yppm = (float)HeightOfScreen(s) / (float)HeightMMOfScreen(s);

  gcv.function = GXcopy;
  gcv.foreground = WhitePixel(_disp, _screen);
  gcv.graphics_exposures = 0;
  _clear_gc = XCreateGC(_disp, _win,
                        GCFunction|GCForeground|GCGraphicsExposures, &gcv);

  _pixMaxWidth  = xmm_to_pixels(210);
  _pixMaxHeight = ymm_to_pixels(20);

  _pix = XCreatePixmap(_disp, _win,
		       _pixMaxWidth, _pixMaxHeight,
		       DefaultDepth(_disp, 0));
  XFillRectangle(_disp, _pix, _clear_gc, 0, 0, _pixMaxWidth, _pixMaxHeight);
  gcv.function = GXcopy;
  gcv.foreground = BlackPixel(_disp, _screen);
  _gc = XCreateGC(_disp, _win,
		  GCFunction|GCForeground|GCGraphicsExposures, &gcv);

  gcv.line_style = LineOnOffDash;
  _dashed_gc = XCreateGC(_disp, _win,
			 GCFunction|GCForeground|GCGraphicsExposures|GCLineStyle, &gcv);
 
  gcv.function = GXxor;
  gcv.foreground = BlackPixel(_disp, _screen) ^ WhitePixel(_disp, _screen);
  _xor_gc = XCreateGC(_disp, _win,
                      GCFunction|GCForeground|GCGraphicsExposures, &gcv);

  XFlush( _disp );
}



/*
 *----------------------------------------------------------------------
 *
 * ~XDisplay --
 *
 *      This method is invoked every time an XDisplay object is
 *      deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Frees some X11 ressources.
 *
 *----------------------------------------------------------------------
 */

XDisplay::~XDisplay() {

  XFreeGC(_disp, _gc);
  XFreeGC(_disp, _xor_gc);
}



/*
 *----------------------------------------------------------------------
 *
 * draw_string --
 *
 *      This method draws a string in the object Pixmap.
 *      deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::draw_string(int x, int y, char *string, int len) {

  XDrawString(_disp, _pix, _gc, x-_destX, y-_destY, string, len);
}



/*
 *----------------------------------------------------------------------
 *
 * clear_area --
 *
 *      This method clears a region of the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::clear_area(int x, int y, unsigned int w, unsigned int h) {

  XClearArea(_disp, _win, x, y, w, h, 0);
}


/*
 *----------------------------------------------------------------------
 *
 * reverse_area --
 *
 *      This method reverses a region of the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::reverse_area(int x, int y, unsigned int w, unsigned int h) {

  XFillRectangle(_disp, _win, _xor_gc, x, y, w, h);
  XFlush(_disp);
}



/*
 *----------------------------------------------------------------------
 *
 * set_current_font --
 *
 *      This method sets the current font to use for drawing string.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::set_current_font(FontItem *fi) {

  if( _fi == fi )
    return;

  _fi = fi;
  if( _fi == NULL )
    return;

  XSetFont(_disp, _gc, _fi->get_fid());
}



/*
 *----------------------------------------------------------------------
 *
 * resize_window --
 *
 *      This method resize the window passed as an argument. The
 *      method is used when zoom changes.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::resize_window(Window win, float wf, float hf) {
  
  Window drawable;
  int x, y;
  unsigned int width, height, border, depth;

  XGetGeometry(_disp, win, &drawable, &x, &y,
	       &width, &height, &border, &depth);

#ifdef HAVE_GUI

  Dimension w_ret, h_ret;

  XtMakeResizeRequest(XtNameToWidget(toplevel, "*PageWidget"), (Dimension)(width*wf),
		      (Dimension)(height*hf), &w_ret, &h_ret);
#else
    XResizeWindow(_disp, win, (unsigned int)(width*wf), (unsigned int)(height*hf));
#endif
  XFlush(_disp);
}



/*
 *----------------------------------------------------------------------
 *
 * flush_pixmap --
 *
 *      This method maps the object pixmap on the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::flush_pixmap(void) {

  XCopyArea(_disp, _pix, _win, _gc,
	    0, 0, _pixWidth, _pixHeight, _destX, _destY); 
}



/*
 *----------------------------------------------------------------------
 *
 * alloc_pixmap --
 *
 *      This method reallocates the object pixmap if it's not big
 *      enough.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::alloc_pixmap(int x, int y, unsigned w, unsigned h) {

  _destX = x;
  _destY = y;
  _pixWidth = w;
  _pixHeight = h;
  
  if( w > _pixMaxWidth || h > _pixMaxHeight ) {
    _pixMaxWidth  = MAX(w, _pixMaxWidth);
    _pixMaxHeight = MAX(h, _pixMaxHeight);
    
    XFreePixmap(_disp, _pix);
    
    _pix = XCreatePixmap(_disp, _win,
			 _pixMaxWidth, _pixMaxHeight,
			 DefaultDepth(_disp, 0));
  }
  
  XFillRectangle(_disp, _pix, _clear_gc, 0, 0, _pixWidth, _pixHeight);
}



/*
 *----------------------------------------------------------------------
 *
 * draw_box --
 *
 *      This method draws a plain box in the object pixmap.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::draw_box(int x, int y, int w, int h) {

  XFillRectangle(_disp, _pix, _gc, x-_destX, y-_destY, w, h);
}



/*
 *----------------------------------------------------------------------
 *
 * draw_pixmap --
 *
 *      This method is invoked to draw the contents of the object pixmap
 *      on the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The screen is updated.
 *
 *----------------------------------------------------------------------
 */

void
XDisplay::draw_pixmap(Pixmap *pixmap, int x, int y, unsigned short w, unsigned short h) {

  if( hide_images() == TRUE ) {
    XDrawRectangle(_disp, _pix, _gc, X(x), Y(y), w, h);
    XDrawLine(_disp, _pix, _gc, X(x), Y(y), X(x+w), Y(y+h));
    XDrawLine(_disp, _pix, _gc, X(x), Y(y+h), X(x+w), Y(y));
  } else
    XCopyArea(_disp, *pixmap, _pix, _gc, 0, 0, w, h, x-_destX, y-_destY);
}



/*
 *----------------------------------------------------------------------
 *
 * draw_pixmap --
 *
 *      This method reads a Xpm format file.
 *
 * Results:
 *      Returns a Pixmap pointer to the image..
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Pixmap *
XDisplay::get_xpm_file(char *filename, unsigned short &width, unsigned short &height) {

  XWindowAttributes root_attr;
  XpmAttributes xpm_attr;
  Pixmap *pixmap, pixmapMask;
  int result;
  char msg[128];
  

  XGetWindowAttributes(_disp, _root, &root_attr);

  xpm_attr.colormap = root_attr.colormap;             /* mother window's colormap */
  xpm_attr.color_key = XPM_COLOR;                     /* xpm file has colors      */ 
  xpm_attr.closeness = 40000;                         /* color distance           */
  xpm_attr.visual = DefaultVisual(_disp, _screen);    /* default visual  */

  /* if there's no enough colors, choose the more similars */
  xpm_attr.exactColors = False;     

  xpm_attr.valuemask= XpmExactColors | XpmCloseness | XpmVisual | XpmColormap
                    | XpmSize | XpmColorKey | XpmReturnPixels;

  pixmap = new Pixmap;

  result = XpmReadFileToPixmap(_disp, _root, filename,
			       pixmap,
			       &pixmapMask,
			       &xpm_attr);
  switch( result ) {

  case XpmSuccess:
    width  = (unsigned short)xpm_attr.width;
    height = (unsigned short)xpm_attr.height;

    if( hide_images() == TRUE )
      XFreePixmap(_disp, *pixmap);
    return pixmap;
    break;

  case XpmOpenFailed:
    sprintf(msg, "Can't open pixmap file: %s\n", filename);
    GUI_Add_ErrorMessage(msg);
    break;

  case XpmColorFailed:
    sprintf(msg, "Can't allocate colormap for pixmap file: %s\n", filename);
    GUI_Add_ErrorMessage(msg);
    break;

  case XpmFileInvalid:
    sprintf(msg, "Pixmap file: %s has wrong format\n", filename);
    GUI_Add_ErrorMessage(msg);
    break;

  case XpmColorError:
    sprintf(msg, "Invalid color in pixmap file: %s\n", filename);
    GUI_Add_ErrorMessage(msg);
    break;

  case XpmNoMemory:
    sprintf(msg, "Not enough memory for pixmap file: %s\n", filename);
    GUI_Add_ErrorMessage(msg);
    break;
  }
  return NULL;
}
