/* 
 * My_Tcl_Main.c --
 *
 *	Main program for Tcl shells and other Tcl-based applications.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 * Copyright (c) 1988-1994 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#include "mytcl.h"

extern void Scale_Increment();
extern void GUI_Add_ErrorMessage(char *); 
extern void Fatal_Intro_Message(char *);

/*
 * Declarations for various library procedures and variables (don't want
 * to include tclPort.h here, because people might copy this file out of
 * the Tcl source directory to make their own modified versions).
 * Note:  "exit" should really be declared here, but there's no way to
 * declare it without causing conflicts with other definitions elsewher
 * on some systems, so it's better just to leave it out.
 */

extern int		errno;
extern int		isatty _ANSI_ARGS_((int fd));
extern char *		strcpy _ANSI_ARGS_((char *dst, CONST char *src));

Tcl_Interp *theInterp;	        /* Interpreter for application. */
Tcl_DString command;	        /* Used to buffer incomplete commands being
				 * read from stdin. */

int docOpen;

/*
 * Forward references for procedures defined later in this file:
 */


/*
 *----------------------------------------------------------------------
 *
 * My_Tcl_Main --
 *
 *	Main program for tclsh and most other Tcl-based applications.
 *
 * Results:
 *	None. This procedure never returns (it exits the process when
 *	it's done.
 *
 * Side effects:
 *	This procedure initializes the Tk world and then starts
 *	interpreting commands;  almost anything could happen, depending
 *	on the script being interpreted.
 *
 *----------------------------------------------------------------------
 */

void
My_Tcl_Main(argc, argv)
    int argc;				/* Number of arguments. */
    char **argv;			/* Array of argument strings. */
{
    char buffer[1000], *args, *fileName;
    int code, tty;
    int exitCode = 0;

    theInterp = Tcl_CreateInterp();

    /*
     * Make command-line arguments available in the Tcl variables "argc"
     * and "argv".  If the first argument doesn't start with a "-" then
     * strip it off and use it as the name of a script file to process.
     */

    fileName = NULL;
    if ((argc > 1) && (argv[1][0] != '-')) {
	fileName = argv[1];
	argc--;
	argv++;
    }
    args = Tcl_Merge(argc-1, argv+1);
    Tcl_SetVar(theInterp, "argv", args, TCL_GLOBAL_ONLY);
    ckfree(args);
    sprintf(buffer, "%d", argc-1);
    Tcl_SetVar(theInterp, "argc", buffer, TCL_GLOBAL_ONLY);
    Tcl_SetVar(theInterp, "argv0", (fileName != NULL) ? fileName : argv[0],
	    TCL_GLOBAL_ONLY);

    /*
     * Set the "tcl_interactive" variable.
     */

    tty = isatty(0);
    Tcl_SetVar(theInterp, "tcl_interactive",
	    ((fileName == NULL) && tty) ? "1" : "0", TCL_GLOBAL_ONLY);

    /*
     * Invoke application-specific initialization.
     */

    if ( Tcl_AppInit(theInterp) != TCL_OK) {
      Fatal_Intro_Message(theInterp->result);
      /* fprintf(stderr, "application-specific initialization failed: %s\n",
	      theInterp->result); */
    }

    /*
     * If a script file was specified then just source that file
     * and quit.
     */

    if (fileName != NULL) {
	code = Tcl_EvalFile(theInterp, fileName);
	if (code != TCL_OK) {
	  Fatal_Intro_Message(theInterp->result);
	  exitCode = 1;
	}
	return;
    }

    /*
     * We're running interactively.  Source a user-specific startup
     * file if the application specified one and if the file exists.
     */

    fileName = Tcl_GetVar(theInterp, "filename", TCL_GLOBAL_ONLY);

    if (fileName != NULL) {
	Tcl_DString buffer;
	char *fullName;
	FILE *f;

	fullName = Tcl_TildeSubst(theInterp, fileName, &buffer);
	if (fullName == NULL) {
	    fprintf(stderr, "%s\n", theInterp->result);
	} else {
	    f = fopen(fullName, "r");
	    if (f != NULL) {
		code = Tcl_EvalFile(theInterp, fullName);
		if (code != TCL_OK) {
		  Fatal_Intro_Message(theInterp->result);
		}
		fclose(f);
	    }
	}
	Tcl_DStringFree(&buffer);
    }
    Tcl_LinkVar(theInterp, "alive", (char *)&docOpen, TCL_LINK_BOOLEAN);
    docOpen = 0;
    return;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_MyEval --
 *
 *      This function is invoked to eval Tcl commands in PAPyRUS. So
 *      some updating code was added.
 *
 * Results:
 *      The result of the execution.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_MyEval(char *cmd) {
  int result;

  /*
   * Copy the strings one after the other into a single larger
   * string.  Use stack-allocated space for small commands, but if
   * the command gets too large than call ckalloc to create the
   * space.
   */
  Tcl_DStringInit(&command);
  Tcl_DStringAppend(&command, cmd, -1);
  Tcl_CommandComplete(cmd);
  
  result = Tcl_RecordAndEval(theInterp, Tcl_DStringValue(&command), 0);
  Tcl_DStringFree(&command);
  
  if( result != TCL_OK )
    GUI_Add_ErrorMessage(theInterp->result); 

  /*
   * Update the toolbar only if there's an open document.
   */
  if( docOpen != 0 )
    Tcl_Eval(theInterp, "Get_State");
  
  return result;
}

