/* 
 * tclInit.c --
 *
 *      This file contains functions to intialize the Tcl
 *      interpret.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

extern "C" {
#include <tcl.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>

#include "../gui/actions.h"
#include "../gui/mainWin.h"
#include "../misc/env.h"

extern void GUI_SetOMEntry(char *, char *);
}

#include "commands.h"


/*
 * ClientData data type of trace procedures
 */
typedef enum {PSTYLE, PALIGN, WFAMILY, WSIZE, WSTYLE} TraceCD;

/*
 * Structure of a Tcl command
 */
typedef struct Papyrus_Cmd {
  char         *cmdName;   /* Command name     */
  char        *cmdAlias;   /* Command alias    */
  Tcl_CmdProc  *cmdProc;   /* Function to call */
};


/*
 * List of the new functions.
 */

Papyrus_Cmd baseCommands[] = {

  { "AddEntry",               "ae",   AddEntry_Cmd               },
  { "AddFont",                "af",   AddFont_Cmd                },
  { "AddStyle",               "as",   AddStyle_Cmd               },

  { "DeleteStyle",            "ds",   DeleteStyle_Cmd            },

  { "PapyConfigure",         "ppc",   PapyrusConfigure_Cmd       },
  { "DocConfigure",           "dc",   DocumentConfigure_Cmd      },
  { "ParaConfigure",          "pc",   ParagraphConfigure_Cmd     },
  { "WordConfigure",          "wc",   WordConfigure_Cmd          },
  
  { "PapyQuery",             "ppq",   PapyrusQuery_Cmd           },
  { "DocQuery",               "dq",   DocumentQuery_Cmd          },
  { "ParaQuery",              "pq",   ParagraphQuery_Cmd         },
  { "WordQuery",              "wq",   WordQuery_Cmd              },

  { "NewDocument",            "nd",   NewDocument_Cmd            },
  { "NewParagraph",           "np",   NewParagraph_Cmd           },
  { "NewWord",                "nw",   NewWord_Cmd                },

  { "CloseDocument",          "cld",   CloseDocument_Cmd          },

  { "InsertCharacter",        "ic",   InsertCharacter_Cmd        },
  { "InsertImage",            "ii",   InsertImage_Cmd            },
  { "InsertString",           "is",   InsertString_Cmd           },
  { "DeleteCharacterBefore",  "dcb",  DeleteCharacterBefore_Cmd  },
  { "DeleteCharacterAfter",   "dca",  DeleteCharacterAfter_Cmd   },

  { "MoveCursorLeft",         "mcl",  MoveCursorLeft_Cmd         },
  { "MoveCursorRight",        "mcr",  MoveCursorRight_Cmd        },
  { "MoveCursorUp",           "mcu",  MoveCursorUp_Cmd           },
  { "MoveCursorDown",         "mcd",  MoveCursorDown_Cmd         },
  { "MoveCursorNextWord",     "mcnw", MoveCursorNextWord_Cmd     },
  { "MoveCursorPreviousWord", "mcpw", MoveCursorPreviousWord_Cmd },
  { "MoveCursorBeginLine",    "mcbl", MoveCursorBeginLine_Cmd    },
  { "MoveCursorEndLine",      "mcel", MoveCursorEndLine_Cmd      },
  { "MoveCursorXY",           "mcxy", MoveCursorXY_Cmd           },

  { "SelectRegion",           "sr",   SelectRegion_Cmd           },

  { "SetZoom",                "sz",   SetZoom_Cmd                },
  { "SetKey",                 "sk",   SetKey_Cmd                 },
  { "SetCursor",              "sc",   SetCursor_Cmd              },

  { "DocOutput",              "do",   DocumentOutput_Cmd         },
  { "WordOutput",             "wo",   WordOutput_Cmd             },

  { "RefreshScreen",          "rs",   RefreshScreen_Cmd          },
  { "Debug",                  "dbg",  Debug_Cmd                  },
  { NULL, NULL, NULL }
};



/*
 *----------------------------------------------------------------------
 *
 * toolBar_TraceProc --
 *
 *      This function is invoked each time a button on the toolbar
 *      have to be armed or disarmed. This function is a trace 
 *      function on the variables state(..).
 *
 * Results:
 *      Returns NULL.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

char *
toolBar_TraceProc(ClientData clientData, Tcl_Interp *interp,
		 char *name1, char *name2, int) {
  char *string;
  extern Widget toplevel;

  string = Tcl_GetVar(interp, "interactive_flag", TCL_GLOBAL_ONLY);
  if( string[0] == '0' )
    return NULL;

  string = Tcl_GetVar2(interp, name1, name2, TCL_GLOBAL_ONLY);

  switch( (TraceCD)clientData ) {
  case PSTYLE:
    GUI_SetOMEntry("style", string);
    break;
  case PALIGN:
    if( string[0] == 'l' )
      XtCallActionProc(XtNameToWidget(toplevel,"*button_b17"), "one-of-many", NULL, NULL, 0);
    else if( string[0] == 'f' )
      XtCallActionProc(XtNameToWidget(toplevel,"*button_b20"), "one-of-many", NULL, NULL, 0);
    else if( string[0] == 'c' )
      XtCallActionProc(XtNameToWidget(toplevel,"*button_b19"), "one-of-many", NULL, NULL, 0);
    else { /* right alignment */
      XtCallActionProc(XtNameToWidget(toplevel,"*button_b18"), "one-of-many", NULL, NULL, 0);
    }
    break;
  case WFAMILY:    
    GUI_SetOMEntry("family", string);
    break;
  case WSTYLE:
    if( string[0] == 'b' ) {
      Arm_ToggleButton("*button_b14");
      if( string[4] == '_' )
	Arm_ToggleButton("*button_b15");
      else
	Disarm_ToggleButton("*button_b15");
      
    } else if( string[0] == 'i' ) {
      Arm_ToggleButton("*button_b15");
      Disarm_ToggleButton("*button_b14");
    } else {
      Disarm_ToggleButton("*button_b14");
      Disarm_ToggleButton("*button_b15");
    }
    break;
  case WSIZE:
    GUI_SetOMEntry("size", string);
    break;
  default:
    break;
  }
  return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * alive_TraceProc --
 *
 *      This function is invoked when the document is created or
 *      deleted. THe function updates the buttons and menus
 *      sensitives.
 *
 * Results:
 *      Returns NULL.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

char *
alive_TraceProc(ClientData, Tcl_Interp *interp,
		 char *, char *, int) {

  char *string;
  int value;
  extern Widget toplevel;


  string = Tcl_GetVar(interp, "alive", TCL_GLOBAL_ONLY);
  
  if( Tcl_GetBoolean(interp, string, &value) != TCL_OK )
    return NULL;
  
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b0"), !value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b1"), !value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b2"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b3"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b4"), value);
/*
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b5"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b6"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b7"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b8"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b9"), value);
  */
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b10"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b11"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b12"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b13"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b14"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b15"), value);
/*  XtSetSensitive(XtNameToWidget(toplevel, "*button_b16"), value); */
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b17"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b18"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b19"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*button_b20"), value);

  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p0_b0"), !value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p0_b1"), !value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p0_b2"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p0_b4"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p0_b5"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p0_b7"), value);

  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p2_b0"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p2_b1"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p2_b2"), value);

  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p3_b0"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p3_b1"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*menu_p3_b3"), value);

  XtSetSensitive(XtNameToWidget(toplevel, "*styleOM_rc"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*familyOM_rc"), value);
  XtSetSensitive(XtNameToWidget(toplevel, "*sizeOM_rc"), value);

  XmUpdateDisplay(toplevel);

  return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_AppInit --
 *
 *      This function is invoked to create the PAPyRUS specifie
 *      command of the application. Some trace procedures are
 *      also registered.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_AppInit(Tcl_Interp *interp) {

  int i;
  char filename[128];

  if( Tcl_Init(interp) == TCL_ERROR ) {
    return TCL_ERROR;
  }

  for(i=0; baseCommands[i].cmdProc != NULL; i++) {
    Tcl_CreateCommand(interp, baseCommands[i].cmdName, baseCommands[i].cmdProc,
                    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateCommand(interp, baseCommands[i].cmdAlias, baseCommands[i].cmdProc,
		      (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  }
  
  Tcl_SetVar(interp, "tcl_rcFileName", "~/ess.tcl", TCL_GLOBAL_ONLY);

  sprintf(filename, "%s/scripts/papyrus-init.tcl", papyrus_dir);
  
  if( Tcl_EvalFile(interp, filename) != TCL_OK )
    return TCL_ERROR;

  Tcl_TraceVar(interp, "alive", TCL_TRACE_WRITES|TCL_GLOBAL_ONLY,
	       alive_TraceProc, (ClientData)PSTYLE);
  
  Tcl_TraceVar2(interp, "state", "pstyle", TCL_TRACE_WRITES|TCL_GLOBAL_ONLY,
		toolBar_TraceProc, (ClientData)PSTYLE);
  Tcl_TraceVar2(interp, "state", "palign", TCL_TRACE_WRITES|TCL_GLOBAL_ONLY,
		toolBar_TraceProc, (ClientData)PALIGN);
  Tcl_TraceVar2(interp, "state", "wfamily", TCL_TRACE_WRITES|TCL_GLOBAL_ONLY,
		toolBar_TraceProc, (ClientData)WFAMILY);
  Tcl_TraceVar2(interp, "state", "wsize", TCL_TRACE_WRITES|TCL_GLOBAL_ONLY,
		toolBar_TraceProc, (ClientData)WSIZE);
  Tcl_TraceVar2(interp, "state", "wstyle", TCL_TRACE_WRITES|TCL_GLOBAL_ONLY,
		toolBar_TraceProc, (ClientData)WSTYLE);

  return TCL_OK;
}
